/**
 * Growl/Meow/message service.
 *
 * Usage -
 * 1. Include this file
 * 2. Inject cla.core.growl into the module.
 * 3. Inject growl.service into the controller.
 * 4. Use growl.showSuccess(<title>, <message>) and/or growl.showError(<title>, <message>)
 *
 * Just wraps cla.showMessage for now but opens up the possibility of testing messages in our Angular apps
 * by replacing this service with something that checks if it was called or not.
 */

define(['cla_angular'], function (angular) {
	var moduleName = 'cla.core.growl';
	try {
		module = angular.module(moduleName);
	} catch(err) {
		// named module does not exist, so create one
		module = angular.module(moduleName, []);
	}

	angular.module(moduleName)
	.service('growl.service', growl_service).run(["growl.service", angular.noop]);

	function growl_service() {
		var service = {
			showSuccess: showSuccess,
			showError: showError,
			meow: meow,
			flash: flash
		};

		var pvt = {
			setCookie: setCookie,
			getCookie: getCookie
		};

		init();

		return service;
		//////////////////


		/** service initialisation */
		function init() {
			// check for any flash messages to display
			service.flash();
		}

		/** show a success message */
		function showSuccess(title, message) {
			service.meow(title, message, false)
		}

		/** show an error message */
		function showError(title, message) {
			service.meow(title, message, true)
		}

		/** flash messages */
		function flash(title, msg, is_error, url) {
			var cookie = pvt.getCookie("flash_message");

			if (url) {
				// create a message object
				var msg_cookie_obj = {
					msg: msg ? escape(msg) : null,
					title: title ? escape(title) : null,
					is_error: is_error
				};

				// set message as cookie and redirect
				pvt.setCookie("flash_message", JSON.stringify(msg_cookie_obj), 1);
				window.location.href = url;
				return;

			} else if (cookie) {
				msg_cookie_obj = JSON.parse(cookie);

				msg = msg_cookie_obj.msg ? unescape(msg_cookie_obj.msg) : null;
				title = msg_cookie_obj.title ? unescape(msg_cookie_obj.title) : null;
				is_error = msg_cookie_obj.is_error;
			}

			// clear cookie
			pvt.setCookie("flash_message", null, -1);

			if (msg || title) {
				service.meow(title, msg, is_error);
			}
		}


		/** execute the meow (growl-like) notification */
		function meow(title, message, is_error) {

			var m_class = ( is_error === 'info' )
				? 'inner-info'
				: ( is_error ? 'inner-error' : 'inner-success' );

			$.meow({
				'title': title,
				'message': message,
				'duration' : (m_class === 'inner-error' ? Infinity : 5000),
				'sticky' : (m_class === 'inner-error'),
				'className' : m_class
			});
		}

		/** set cookie data */
		function setCookie(cname, cvalue, exdays) {
			var d = new Date();
			d.setTime(d.getTime() + (exdays*24*60*60*1000));
			var expires = "expires="+d.toUTCString();
			var path = "path=/";
			document.cookie = cname + "=" + cvalue + "; " + expires + "; " + path;
		}

		/** get cookie data */
		function getCookie(cname) {
			var name = cname + "=";
			var ca = document.cookie.split(';');
			for(var i=0; i<ca.length; i++) {
				var c = ca[i];
				while (c.charAt(0)==' ') c = c.substring(1);
				if (c.indexOf(name) == 0) return c.substring(name.length, c.length);
			}
			return "";
		}
	}
});