define([], function () {
    /**
     * Form field controller
     * @class
     * @constructor
     * @property {Object} model - Field.Model object
     * @property {Object} view - Field.View object
     */
    var Field = function () {
        'use strict';
        /**
         * Field.Model object
         * @public
         * @type {Object}
         */
        this.model = new Field.Model();

        /**
         * Field.View object
         * @public
         * @type {Object}
         */
        this.view = new Field.View();

        this.model.setView(this.view);
        this.view.setModel(this.model);
    };

    /**
     * Form field model
     * @class
     * @constructor
     * @property {Object} view - Field.View object
     * @property {String} label - field label
     * @property {String} fieldSymName - field symbolic name
     */
    Field.Model = function () {
        'use strict';
        /**
         * View object
         * @public
         * @type {Object}
         */
        this.view = null;

        /**
         * Field label
         * @public
         * @type {String}
         */
        this.label = null;

        /**
         * Field symbolic name
         * @public
         * @type {String}
         */
        this.fieldSymName = null;
    };

    Field.Model.prototype = (function () {
        'use strict';
        /**
         * Set the view object
         * @private
         * @memberOf Field.Model
         * @param {Object} view - Field.View object
         */
        var setView = function (view) {
            this.view = view;
        };

        /**
         * Get the view object
         * @private
         * @memberOf Field.Model
         * @returns {Object}
         */
        var getView = function () {
            return this.view;
        };

        var setActive = function(active) {
            this.active = active
        }

        /**
         * Set the field data
         * @private
         * @memberOf Field.Model
         * @param {Object} data - JSON object
         */
        var setData = function (data) {
            setFieldSymName.call(this, data.field_sym_name);
            setLabel.call(this, data.label);
            setActive.call(this, data.active);
        };

        /**
         * Set field symbolic name
         * @private
         * @memberOf Field.Model
         * @param {String} symName - field symbolic name
         */
        var setFieldSymName = function (symName) {
            this.fieldSymName = symName;
        };

        /**
         * Get the field symbolic name
         * @private
         * @memberOf Field.Model
         * @returns {String}
         */
        var getFieldSymName = function () {
            if (_.isString(this.fieldSymName)) {
                return this.fieldSymName;
            } else {
                return '';
            }
        };

        /**
         * Set field label
         * @private
         * @memberOf Field.Model
         * @param {String} label - field label
         */
        var setLabel = function (label) {
            this.label = label;
        };

        /**
         * Get the field label
         * @private
         * @memberOf Field.Model
         * @returns {String}
         */
        var getLabel = function () {
            if (_.isString(this.fieldSymName)) {
                return this.label;
            } else {
                return '';
            }
        };

        return {
            /**
             * Set the view object
             * @public
             * @memberOf Field.Model
             */
            setView: setView,

            /**
             * Initalise object
             * @public
             * @memberOf Field.Model
             * @returns {Array}
             */
            setData: setData,

            /**
             * Get the field symbolic name
             * @public
             * @memberOf Field.Model
             * @returns {String}
             */
            fieldSymName: getFieldSymName,

            /**
             * Get the field label
             * @public
             * @memberOf Field.Model
             * @returns {String}
             */
            label: getLabel
        };
    })();

    /**
     * Form field view class
     * @class
     * @constructor
     * @property {Object} model - Field.Model object
     */
    Field.View = function () {
        'use strict';
        /**
         * Field.Model object
         * @public
         * @type {Object}
         */
        this.model = null;
    };

    Field.View.prototype = (function () {
        'use strict';
        /**
         * Set the model object
         * @private
         * @memberOf Field.View
         */
        var setModel = function (model) {
            this.model = model;
        };

        /**
         * Get the model object
         * @private
         * @memberOf Field.View
         */
        var getModel = function () {
            return this.model;
        };

        /**
         * Print the field label
         * @private
         * @memberOf Field.View
         * @returns {Object}
         */
        var render = function () {
            var fieldExists = (this.model.fieldSymName !== null || this.model.label !== null);

            var div_class = "alert alert-info";
            if (this.model.active === false)
            {
                div_class = "alert alert-warning";
            }

            if (fieldExists) {
                return _.template('<div class="' + div_class + '">{{- label }} <small>{{- fieldSymName }}</small></div>')(this.model);
            }
            return '';
        };

        return {
            /**
             * Set the model object
             * @public
             * @memberOf Field.View
             */
            setModel: setModel,

            /**
             * Print the field label
             * @public
             * @memberOf Field.View
             * @returns {Object}
             */
            render: render
        };
    })();

    return Field;
});
