/**
 * Abstract view for the documents list. Used in main docs list, bookmarks, search results.
 * Displays documents collection stored inside, which can be assigned by calling setCollection method
 */
var DocsListViewBase = Backbone.View.extend({
	collection: null,
	document_template: _.template($('#document_item_template').html()),
	folder_template: _.template($('#folder_item_template').html()),
	is_visible: false,
	filter_model: null,
	selected_docs: null,
	selected_actions_view: null,

	events: {
		'showMenu': "onShowMenu"
	},

	initialize: function ()
	{
		this.filter_model = new SearchFormModel();
		this.filter_model.bind('change', this.onFilterChange, this);
		this.selected_docs = new SelectedDocsCollection();
		this.selected_docs.bind('reset', this.updateCheckboxes, this);
		this.selected_actions_view = new DashboardActionsView();
		this.info_menu_view = new InfoMenuActionsView();
	},

	setVisible: function (visible)
	{
		this.is_visible = visible;
		if (visible)
		{
			search_form.setFilterModel(this.filter_model);
			aside_view.setSelectedCollection(this.selected_docs, this.selected_actions_view);
			this.selected_docs.reset();
		}

		//this.render();
	},

	setViewMode: function (view_mode)
	{
		switch (view_mode)
		{
			case 'thumb':
				this.document_template = _.template($('#document_item_template_thumb').html());
				this.folder_template = _.template($('#folder_item_template_thumb').html());
				break;
			case 'icon':
			case 'details':
			case 'min':
			default:
				this.document_template = _.template($('#document_item_template').html());
				this.folder_template = _.template($('#folder_item_template').html());
		}
		this.render();
	},

	onFilterChange: function ()
	{
		this.render();
	},

	renderLoading: function ()
	{
		$(".main-content-block").hide();
		this.$('.documentLayout').empty();
		this.$el.show();
	},

	render: function ()
	{
		if (!this.is_visible)
			return this;

		var result = '';
		this.collection.each(function (doc, index) {
			result = result + this.getNodeHtml(doc, index);
		}, this);

		$("#parentFolder .js-select-all").hide();
		$(".main-content-block").hide();
		this.$('.documentLayout').html(result);
		this.$el.show();

		this.updateCheckboxes();

		if(result != '')
		{
			$("#parentFolder .js-select-all").show();
		}

		return this;
	},

	onCheckbox: function (cb_value, checked)
	{
		var parts = cb_value.split(' ');
		var obj = this.collection.getDocObject(parts[0], parts[1]);

		if (checked)
		{
			if (!this.selected_docs.get(obj.cid))
				this.selected_docs.add(obj);
		}
		else
			this.selected_docs.remove(obj);
	},

	updateCheckboxes: function ()
	{
		this.$('input[name=selected_doc]').prop('checked', false);
		this.selected_docs.each(function (doc, index) {
			var val = doc.get('type') + ' ' + doc.getObjId();
			this.$('input[name=selected_doc][value="' + val + '"]').prop('checked', true);
		}, this);
	},

	// add document or folder into display
	appendNode: function (doc)
	{
		if (!this.is_visible)
			return;
		this.$('.documentLayout').append(this.getNodeHtml(doc));
		this.$('#load_next_trigger').hide();
		if(this.getNodeHtml(doc) != '')
		{
			$("#parentFolder .js-select-all").show();
		}
	},

	getNodeHtml: function(doc)
	{
		if (!this.filter_model || this.filter_model.get('keywords').length == 0 || doc.matchKeywords(this.filter_model.get('keywords')))
		{
			var data = this.toTemplateData(doc);
			if (data != null)
			{
				if (dms_options.interfaceDebug && doc.has('permissions'))
				{
					var p = doc.get('permissions');
					var pstr =  (p.view ? 'v' : '') +
					            (p.edit ? 'e' : '') +
					            (p.edit_meta ? 'm' : '') +
					            (p['delete'] ? 'd' : '');
					data.perm_string = '(<font color="green">'+pstr+'</font>)';
				}
				else
					data.perm_string = '';

				if (data.obj_id == 0)
					return _.template($('#empty_item_template').html())(data);
				else if (data.type == 'folder')
					return this.folder_template(data);
				else if (data.type == 'document')
				{
					if (data.doc_type == 'marker')
						data.icon = 'glyphicons glyphicons-map';
					else if (data.doc_type == 'link')
                        data.icon = 'glyphicons glyphicons-link';
					else if (data.doc_type == 'gdoc_link')
						data.icon = 'glyphicons glyphicons-file-drive';
					else
						data.icon = claIcon(data.title);

					if (data.fsize != null)
						data.fsize = fileSizeConvert(data.fsize);
					else
						data.fsize = null;

					return this.document_template(data);
				}
			}
		}
		return '';
	},

	setCollection: function (collection)
	{
		if (this.collection)
		{
			this.collection.unbind('change', this.render, this);
			this.collection.unbind('reset', this.onCollectionReset, this);
			this.collection.unbind('remove', this.onItemRemove, this);
			this.collection.unbind('add', this.appendNode, this);
			this.collection.unbind('loaded', this.showLoaderNode, this);
		}

		this.collection = collection;
		this.collection.bind('change', this.render, this);
		this.collection.bind('reset', this.onCollectionReset, this);
		this.collection.bind('remove', this.onItemRemove, this);
		this.collection.bind('add', this.appendNode, this);
		this.collection.bind('loaded', this.showLoaderNode, this);
	},

	onItemRemove: function ()
	{
		this.selected_docs.reset();
		this.render();
	},

	onCollectionReset: function ()
	{
		this.selected_docs.reset();
		this.render();
	},

	onShowMenu: function (event)
	{
		var menu = $('.dropdown-menu', event.target);

		var parts = $('input[name=selected_doc]', event.target).val().split(' ');
		var object = this.collection.getDocObject(parts[0], parts[1]);

		if (object)
		{
			this.doShowMenu(object, menu);
		}
	},

	doShowMenu: function (object, menu_el)
	{
		if (this.info_menu_view)
		{
			this.info_menu_view.renderMenu(object, menu_el);
		}
	}


});

