<?php
/**
 * @obfuscate_disable
 */
namespace Claromentis\Setup\Task;

use Claromentis\Core\Config\Config;
use Claromentis\Core\DAL;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

/**
 * Database verification task.
 *
 * Verifies database permissions required to install and upgrade applications.
 */
class DatabaseCheck
{
	/**
	 * @var Config
	 */
	protected $config;

	/**
	 * @var DAL\Interfaces\DbInterface
	 */
	protected $database;

	/**
	 * @var LoggerInterface
	 */
	protected $logger;

	/**
	 * The last error that occurred during the database checks.
	 *
	 * @var array
	 */
	protected $errors = array();

	/**
	 * An array of methods that run database checks.
	 *
	 * @var array
	 */
	protected $checks = array(
		'CheckPermissions'
	);

	/**
	 * Create a new database check task.
	 *
	 * @param Config $config
	 * @param DAL\Interfaces\DbInterface $database
	 */
	public function __construct(Config $config, DAL\Interfaces\DbInterface $database)
	{
		$this->config = $config;
		$this->database = $database;
		$this->setLogger(new NullLogger());
	}

	/**
	 * Set the logger.
	 *
	 * @param LoggerInterface $logger
	 */
	public function setLogger(LoggerInterface $logger)
	{
		$this->logger = $logger;
	}

	/**
	 * An error handler that logs any database errors.
	 *
	 * @param int $number
	 * @param string $string
	 * @param string $file
	 * @param int $line
	 */
	public function HandleError($number, $string, $file, $line)
	{
		// Error was suppressed with the @ operator
		if (error_reporting() === 0) {
			return;
		}

		$error = "$string in file $file line $line";
		$this->errors[] = $error;
		$this->logger->error($error);
	}

	/**
	 * Retrieve all errors logged during database checks.
	 *
	 * @return string[]
	 */
	public function GetErrors()
	{
		return $this->errors;
	}

	/**
	 * Run the database checks.
	 *
	 * @return bool
	 */
	public function Check()
	{
		$this->logger->info('Checking database permissions...');

		// Set a temporary custom error handler to track any errors that occur
		set_error_handler([$this, 'HandleError']);

		$this->database->DisableTokenCheck();

		// Run each check
		foreach ($this->checks as $check)
		{
			// TODO: Wrap in a try-catch, log exceptions
			if (!call_user_func([$this, $check]) || !empty($this->errors))
			{
				$this->database->EnableTokenCheck();
				restore_error_handler();

				$this->logger->error('Database checks failed - see errors above');

				return false;
			}
		}

		$this->logger->info('Database checks passed');

		$this->database->EnableTokenCheck();
		restore_error_handler();

		return true;
	}

	/**
	 * Check database permissions for the configured user.
	 *
	 * @return bool
	 */
	protected function CheckPermissions()
	{
		$database = $this->config->Get('cfg_db_name');
		$username = $this->config->Get('cfg_db_user');

		$result = true;

		switch ($this->database->type())
		{
			case 'mysql':
				$result = $this->CheckMySqlGrants($database, $username);
				break;
			case 'mssql':
				$result = $this->CheckSqlServerPermissions($database, $username);
				break;
			default:
				$this->logger->warning("No permission checks implemented for database type '{$this->database->type()}'");
		}

		return $result;
	}

	/**
	 * Check for adequate MySQL grants for the given database and username.
	 *
	 * @param string $database
	 * @param string $username
	 * @return bool
	 */
	protected function CheckMySqlGrants($database, $username)
	{
		static $minimum = array(
			'SELECT', 'INSERT', 'UPDATE', 'DELETE', 'CREATE', 'ALTER',
			'DROP', 'INDEX'
		);

		$result = $this->database->query('SHOW GRANTS');

		// Bail if we haven't found any grants
		if (!$result->hasData())
		{
			$this->logger->error("PROBLEM: No MySQL grants set for the configured database user '$username'");

			return false;
		}

		$all_databases = false;
		$all_tables = false;
		$matching_grants = 0;
		$missing_grants = 0;

		// Let's spin through the results and see what we've got
		while ($row = $result->fetchRow())
		{
			// If the grant string doesn't contain the configured database
			// and user, it's not relevant to our checks
			if (!((strpos($row[0], "`$database`") || strpos($row[0], '*.*')) && strpos($row[0], "'$username'")))
				continue;

			$matching_grants++;

			// We have a match; let's use a regex to read the grants
			preg_match("/GRANT\s+(.*)\s+ON/", $row[0], $matches);

			if (!$matches[1])
			{
				$this->logger->warning('WARNING: No MySQL grants found in matching row: ' . $row[0]);
				continue;
			}

			// We can skip the USAGE grant, it just denotes that the user
			// exists without any global privileges
			if ($matches[1] === 'USAGE')
			{
				continue;
			}

			// Let's also use another regex to work out the scope of
			// the grants; wildcard, database-specific or table-specific
			preg_match('/((?:\`'. preg_quote("$database", '/') . ")\`|(?:\*))\.((?:\`\w+\`)|(?:\*))/", $row[0], $scope_matches);

			$scope = $scope_matches[0];
			$scope_database = $scope_matches[1];
			$scope_table = $scope_matches[2];

			// Split the grants up into an array
			$grants = explode(', ', $matches[1]);

			// Determine whether the grants are adequate
			$diff = array_diff($minimum, $grants);

			// If the grants are adequate we can move on to the next row
			if (empty($diff) || $matches[1] === 'ALL PRIVILEGES')
			{
				if ($scope_database === '*')
					$all_databases = true;

				if ($scope_table === '*')
					$all_tables = true;

				$this->logger->debug("MySQL user '$username' has the following grants for scope $scope: " . implode(', ', $grants));

				continue;
			}

			// Otherwise we have an inadequate set of grants
			$message = "PROBLEM: MySQL user '$username' lacks grants for scope $scope: " . implode(', ', $diff);

			$this->logger->error($message);

			$missing_grants++;
		}

		if (!$matching_grants)
		{
			$this->logger->error("PROBLEM: MySQL user '$username' has no matching grants for the '$database' database");

			return false;
		}

		if (!$all_tables)
		{
			$this->logger->warning("WARNING: MySQL user '$username' may not have adequate grants for all tables of the '$database' database");
		}

		$adequate = ($all_databases || $all_tables || $matching_grants) && !$missing_grants;

		if ($adequate)
		{
			$this->logger->info("MySQL user '$username' has sufficient grants for database '$database'");
		}

		return $adequate;
	}

	/**
	 * Check for adequate SQL Server permissions for the given database and
	 * username.
	 *
	 * @param string $database
	 * @param string $username
	 * @return bool
	 */
	protected function CheckSqlServerPermissions($database, $username)
	{
		static $minimum_server = array(
			'CONNECT SQL'
		);

		static $minimum_database = array(
			'CONNECT', 'CREATE TABLE', 'CREATE SCHEMA',
			'SELECT', 'INSERT', 'UPDATE', 'DELETE', 'EXECUTE', 'ALTER',
			'ALTER ANY SCHEMA'
		);

		$server_permissions = array();
		$database_permissions = array();

		// Grab the user's server permissions
		$result = $this->database->query("SELECT * FROM fn_my_permissions(NULL, 'SERVER')");

		while ($row = $result->fetchRow())
			$server_permissions[] = $row[2];

		// Grab the user's database permissions
		$result = $this->database->query("SELECT * FROM fn_my_permissions(NULL, 'DATABASE')");

		while ($row = $result->fetchRow())
			$database_permissions[] = $row[2];

		// Log the permissions found for debugging
		$this->logger->debug("MSSQL user '$username' has SERVER permissions: " . (implode(', ', $server_permissions) ?: 'None'));
		$this->logger->debug("MSSQL user '$username' has DATABASE permissions for database '$database': " . (implode(', ', $database_permissions) ?: 'None'));

		// Diff the permissions with the minimum set we're after
		$server_diff = array_diff($minimum_server, $server_permissions);
		$database_diff = array_diff($minimum_database, $database_permissions);

		// Log an error if we lack some server permissions
		if (!empty($server_diff))
		{
			$server_message = "PROBLEM: MSSQL user '$username' lacks SERVER permissions: " . implode(', ', $server_diff);

			$this->logger->error($server_message);
		}

		// Log an error if we lack some database permissions
		if (!empty($database_diff))
		{
			$database_message = "PROBLEM: MSSQL user '$username' lacks DATABASE permissions for database '$database': " . implode(', ', $database_diff);

			$this->logger->error($database_message);
		}

		$adequate = empty($server_diff) && empty($database_diff);

		if ($adequate)
		{
			$this->logger->info("MSSQL user '$username' has sufficient SERVER and DATABASE permissions for database '$database'");
		}

		return $adequate;
	}
}
