<?php
use Claromentis\Core\DAL;
/**
 * Storage driver for fetching mail queue data from a Claromentis database
 *
 * @package  Mail_Queue
 */
require_once 'Mail/Queue/Container.php';

/**
 * Mail_Queue_Container_cla
 */
class Mail_Queue_Container_cla extends Mail_Queue_Container
{
    /**
     * Reference to the current database connection.
     * @var object DB instance
     */
    public $db;

    /**
     * Table for sql database
     * @var  string
     */
    public $mail_table = 'mail_queue';

    /**
     * @var string  the name of the sequence for this table
     */
    public $sequence = null;

    /**
     * Contructor
     *
     * @param mixed $options    An associative array of connection option.
     *
     * @public
     */
    public function __construct($options)
    {
        if (!is_array($options))
        {
            return new Mail_Queue_Error(MAILQUEUE_ERROR_NO_OPTIONS,
                $this->pearErrorMode, E_USER_ERROR, __FILE__, __LINE__,
                'No dns specified!');
        }

        if (isset($options['mail_table']))
        {
            $this->mail_table = $options['mail_table'];
        }

        $this->sequence = (isset($options['sequence']) ? $options['sequence'] : $this->mail_table);

        if (!empty($options['pearErrorMode']))
        {
            $this->pearErrorMode = $options['pearErrorMode'];
        }

        try
        {
            global $g_application;
	        $this->db = $g_application->db;
        }
        catch(Exception $e)
        {
            return new Mail_Queue_Error(MAILQUEUE_ERROR_CANNOT_CONNECT,
                $this->pearErrorMode, E_USER_ERROR, __FILE__, __LINE__,
                'DB::connect failed: '. $e->getMessage());
        }
        $this->setOption();
    }

    /**
     * Preload mail to queue.
     *
     * @return mixed  True on success else Mail_Queue_Error object.
     * @protected
     */
    protected function _preload()
    {
        $query_str = 'SELECT * FROM ' . $this->mail_table .
                ' WHERE sent_time = 0 AND try_sent < int:try' .
                ' AND time_to_send <= int:date '.
                ' ORDER BY time_to_send';

        $query = new DAL\Query($query_str, $this->try, time());
        $query->setLimit($this->limit, $this->offset);
        $result = $this->db->query($query);

        if (!$result)
        {
            return new Mail_Queue_Error(MAILQUEUE_ERROR_QUERY_FAILED,
                $this->pearErrorMode, E_USER_ERROR, __FILE__, __LINE__,
                'DB::query failed - "'.$query_str);
        }

        $this->_last_item = 0;
        $this->queue_data = array(); //reset buffer

        while ($row = $result->fetchArray())
        {
            $this->queue_data[$this->_last_item] = new Mail_Queue_Body(
                $row['id'],
                $row['create_time'],
                $row['time_to_send'],
                $row['sent_time'],
                $row['id_user'],
                $row['ip'],
                $row['sender'],
                $this->_isSerialized($row['recipient']) ? unserialize($row['recipient']) : $row['recipient'],
                unserialize($row['headers']),
                unserialize($row['body']),
                $row['delete_after_send'],
                $row['try_sent']
            );
            $this->_last_item++;
        }

        return true;
    }

    /**
     * Put new mail in queue and save in database.
     *
     * Mail_Queue_Container::put()
     *
     * @param string $time_to_send  When mail have to be send
     * @param integer $id_user  Sender id
     * @param string $ip  Sender ip
     * @param string $from  Sender e-mail
     * @param string $to  Reciepient e-mail
     * @param string $hdrs  Mail headers (in RFC)
     * @param string $body  Mail body (in RFC)
     * @param bool $delete_after_send  Delete or not mail from db after send
     *
     * @return mixed  ID of the record where this mail has been put
     *                or Mail_Queue_Error on error
     * @public
     **/
    public function put($time_to_send, $id_user, $ip, $sender,
                $recipient, $headers, $body, $delete_after_send=true)
    {

        $time_to_send_obj = new Date();
        $time_to_send_obj->setDateTime($time_to_send, DATE_FORMAT_ISO);

		$info = array(
			'int:create_time'  => time(),
			'int:time_to_send' => $time_to_send_obj->getDate(DATE_FORMAT_UNIXTIME),
			'int:id_user'      => $id_user,
			'str:ip'           => $ip,
			'str:sender'       => $sender,
			'str:recipient'   => $recipient,
			'clob:headers'     => $headers,
			'clob:body'        => $body,
			'int:delete_after_send' => (int) $delete_after_send,
		);

        $q = new DAL\QueryInsert($this->mail_table, $info);
        $this->db->unsafe_query = true;
        $res = $this->db->query($q);
        $id = $this->db->insertId();

        if (!$res)
        {
            return new Mail_Queue_Error(MAILQUEUE_ERROR_QUERY_FAILED,
                $this->pearErrorMode, E_USER_ERROR, __FILE__, __LINE__,
                'DB::query failed');
        }

        return $id;
    }

    /**
     * Check how many times mail was sent.
     *
     * @param object   Mail_Queue_Body
     * @return mixed  Integer or Mail_Queue_Error class if error.
     * @public
     */
    public function countSend($mail)
    {
        if (!is_object($mail) || !is_a($mail, 'mail_queue_body')) {
            return new Mail_Queue_Error(MAILQUEUE_ERROR_UNEXPECTED, __FILE__, __LINE__);
        }
        $count = $mail->_try();
        $query_str = 'UPDATE ' . $this->mail_table
                .' SET try_sent = int:count '
                .' WHERE id = int:id';

        $this->db->unsafe_query = true;
        $query = new DAL\Query($query_str, $count, $mail->getId());
        $res = $this->db->query($query);

        if (!$res)
        {
            return new Mail_Queue_Error(MAILQUEUE_ERROR_QUERY_FAILED,
                $this->pearErrorMode, E_USER_ERROR, __FILE__, __LINE__,
                'DB::query failed - "'.$query_str);
        }
        return $count;
    }

    /**
     * Set mail as already sent.
     *
     * @param object Mail_Queue_Body object
     * @return bool
     * @public
     */
    public function setAsSent($mail)
    {
        if (!is_object($mail) || !is_a($mail, 'mail_queue_body')) {
            return new Mail_Queue_Error(MAILQUEUE_ERROR_UNEXPECTED,
                $this->pearErrorMode, E_USER_ERROR, __FILE__, __LINE__,
               'Expected: Mail_Queue_Body class');
        }

        $query_str = 'UPDATE ' . $this->mail_table
                .' SET sent_time = int:date'
                .' WHERE id = int:id';

        $query = new DAL\Query($query_str, time(), $mail->getId());

        $this->db->unsafe_query = true;
        $res = $this->db->query($query);

        if (!$res)
        {
            return new Mail_Queue_Error(MAILQUEUE_ERROR_QUERY_FAILED,
                $this->pearErrorMode, E_USER_ERROR, __FILE__, __LINE__,
                'DB::query failed - "'.$query_str);
        }
        return true;
    }

    /**
     * Return mail by id $id (bypass mail_queue)
     *
     * @param integer $id  Mail ID
     * @return mixed  Mail object or false on error.
     * @public
     */
    public function getMailById($id)
    {
        $query_str = 'SELECT * FROM ' . $this->mail_table
            .' WHERE id = int:id';
        $query = new DAL\Query($query_str, $id);
        $result = $this->db->query($query);

        $row = $result->fetchArray();

        if ($row || !is_array($row))
        {
            return new Mail_Queue_Error(MAILQUEUE_ERROR_QUERY_FAILED,
                $this->pearErrorMode, E_USER_ERROR, __FILE__, __LINE__,
                'DB: query failed - "'.$query_str);
        }

        return new Mail_Queue_Body(
            $row['id'],
            $row['create_time'],
            $row['time_to_send'],
            $row['sent_time'],
            $row['id_user'],
            $row['ip'],
            $row['sender'],
            $this->_isSerialized($row['recipient']) ? unserialize($row['recipient']) : $row['recipient'],
            unserialize($row['headers']),
            unserialize($row['body']),
            $row['delete_after_send'],
            $row['try_sent']
        );
    }

    /**
     * Remove from queue mail with $id identifier.
     *
     * @param integer $id  Mail ID
     * @return bool  True on success else Mail_Queue_Error class
     * @public
     */
    public function deleteMail($id)
    {
        $query_str = 'DELETE FROM ' . $this->mail_table .' WHERE id = int:id';

        $this->db->unsafe_query = true;
        $res = $this->db->query($query_str, $id);

        if (!$res)
        {
            return new Mail_Queue_Error(MAILQUEUE_ERROR_QUERY_FAILED,
                $this->pearErrorMode, E_USER_ERROR, __FILE__, __LINE__,
                'MDB2::query failed - "'.$query_str);
        }
        return true;
    }
}
