/*
* For individual sections, search for:
* KNOWLEDGEBASE - initial creation of window.KB object.
* MODEL
* COLLECTION
* VIEW
* ROUTER
* INITIALIZE
* */
define(['jquery', 'backbone', './scroll_observer.js'], function($, Backbone, ScrollObserver){
	return function(){

		// KNOWLEDGEBASE

		window.KB = {};
		window.KB.home = {};
		window.KB.params = {};

		window.KB.params.questions_enabled = kb_params.questions_enabled;

		// MODEL

		window.KB.home.Category = Backbone.Model.extend({
			urlRoot : '/intranet/rest/knowledgebase/categories',

			fetchCategory: function(mixed) {
				this.set({id: mixed});
				this.fetch();

				// trigger navigation menu update
				var category = window.KB.home.categoryNavigationCollection.get(mixed);
				this.trigger("select_category", category);

				// update URL
				window.KB.home.router.navigate("category/" + this.get('id'));
			}
		});

		window.KB.home.CategoryNavigation = Backbone.Model.extend({
			select: function() {
				this.collection.invoke('set', {'selected': false});
				this.set({selected: true});
			}
		});

		window.KB.home.Article = Backbone.Model.extend({
            //the API returns an array with a single element when we get an article directly
            parse: function(data){
                if(Array.isArray(data))
                    return data[0];
                else
                    return data;
            }
		});

		function toggleQuestionFunctionality($el) {
			var questionFunctionalityDiv = $el.find('.questionNavItem');

			if (kb_params.questions_enabled === true || kb_params.questions_enabled === 1) {
				questionFunctionalityDiv.removeClass('hidden');
			}
		}

		window.KB.home.Question = Backbone.Model.extend({
		});

		window.KB.home.MixedArticleQuestionSearch = Backbone.Model.extend({
			defaults: {
				"articles": true,
				"questions": false,
				"sort_id": null,
				"filter": false,
				"questions_enabled": false
			},

			filters: [
				'subscriptions'
			],

			initialize: function(){
				this.set('questions_enabled', window.KB.params.questions_enabled);
				this.set('sort_id', getUserSortId());
			},

			setSort: function(selected_sort_id, silent) {
				if (selected_sort_id == undefined)
					selected_sort_id = getUserSortId();

				var sort_id = 0;

				if (!isNaN(parseFloat(selected_sort_id)) && isFinite(selected_sort_id)) {
					sort_id = selected_sort_id;
				} else {
					sort_id = selected_sort_id.match(/sort-(\d+)/);
					sort_id = +sort_id[1]; // make sure is int
				}

				this.set('sort_id', sort_id, {silent: silent});
				setUserSortFromId(sort_id);
				updateObjSortSettings();
			},

			setArticles: function(val) {
				if (this.get('articles') == true)
					return;

				this.set('questions', false, {silent: true});
				this.set('questions_enabled', window.KB.params.questions_enabled, {silent: true});
				toggleQuestionFunctionality(this.$el);
				this.set('articles', true);
			},

			setQuestions: function(val) {
				if (this.get('questions') == true)
					return;

				this.set('articles', false, {silent: true});
				this.set('questions', true);
				this.set('questions_enabled', window.KB.params.questions_enabled);
			},

			setFilter: function(filter) {
				if (this.isFilter(filter)) {
					this.set('filter', filter);
				} else {
					this.set('filter', false);
				}
			},

			isFilter: function(filter) {
				return _.contains(this.filters, filter);
			}
		});

		window.KB.home.CategoryArticleQuestionSearch = Backbone.Model.extend({
			defaults: {
				"articles": false,
				"questions": false,
				"sort_id": null,
				"category": 0
			},

			initialize: function(){
				this.set('sort_id', getUserSortId());
			},

			setSort: function(selected_sort_id) {
				if (selected_sort_id == undefined)
					selected_sort_id = getUserSortId();

				var sort_id = 0;

				if (!isNaN(parseFloat(selected_sort_id)) && isFinite(selected_sort_id)) {
					sort_id = selected_sort_id;
				} else {
					sort_id = selected_sort_id.match(/sort-(\d+)/);
					sort_id = +sort_id[1]; // make sure is int
				}

				this.set('sort_id', sort_id);
				setUserSortFromId(sort_id);
				updateObjSortSettings();
			},

			/**
			 * Set search filter for articles list
			 *
			 * @param category_id
			 * @param search_category_ids
			 */
			setListAllArticles: function(category_id, search_category_ids) {
				this.resetFilterType();
				this.set('sort_id', getUserSortId());

				if (category_id)
					this.attributes['category'] = category_id;

				if (search_category_ids)
					this.attributes['search_category_ids'] = search_category_ids;

				this.set('articles', true);
			},

			/**
			 * Set search filter for questions list
			 *
			 * @param category_id
			 * @param search_category_ids
			 */
			setListAllQuestions: function(category_id, search_category_ids) {
				this.resetFilterType();

				if (category_id)
					this.attributes['category'] = category_id;

				if (search_category_ids)
					this.attributes['search_category_ids'] = search_category_ids;

				this.set('questions', true);
				this.set('questions_enabled', window.KB.params.questions_enabled);
			},

			resetFilterType: function() {
				this.attributes['articles'] = false;
				this.attributes['questions'] = false;
				this.attributes['questions_enabled'] = window.KB.params.questions_enabled;
			}
		});

		window.KB.home.SearchBox = Backbone.Model.extend({
		});


		// COLLECTION

		window.KB.home.CategoryNavigationCollection = Backbone.Collection.extend({

			model: window.KB.home.CategoryNavigation,
			url: "/intranet/rest/knowledgebase/categories",

			fetchCategories: function() {
				return this.fetch({reset: true});
			},

			getTopCategories: function() {
				return this.where({parent_id: 0});
			},

			getSubcategories: function(parent_id) {
				return this.where({parent_id: parent_id});
			},
			getCategoryList: function(parent_id){
				var categories = [parent_id];
				var subcategories = this.getSubcategories(parent_id);
				for (var i in subcategories){
					var subId = subcategories[i].id;
                    var subsubcategories = this.getCategoryList(subId);
					categories = categories.concat(subsubcategories);
				}
				return categories;
			},
			getCompoundCategory: function(parent_id){
				return this.getCategoryList(parent_id).join(',');
			}
		});

		window.KB.home.CategoryCollection = Backbone.Collection.extend({
			model: window.KB.home.Category,
			url: function() {
				return '/intranet/rest/knowledgebase/categories/' + this.id;
			},

			fetchCategories: function(category_model) {
				this.id = category_model.get('id');

				var options = {};
				options.reset = 1;

				return this.fetch(options);
			}
		});

		window.KB.home.ArticleCollection = Backbone.Collection.extend({
			model: window.KB.home.Article,
			url: "/intranet/rest/knowledgebase/articles",

			initialize: function() {
				this.resetPagination();
			},

			resetPagination: function() {
				this.start = 0;
				this.limit = 10;
				this.has_next = true;
				this.sort = getUserSortType();
				this.subscriptions = false;
				this.category = 0;
				this.categories = [];

				return this;
			},

			setSortMethod: function(search_model) {
				switch(search_model.get('sort_id')) {
					case 1:
						this.sort = 'created_date';
						break;
					case 2:
						this.sort = 'title';
						break;
					case 3:
						this.sort = 'like_count';
						break;
					default:
						break;
				}

				return this;
			},

			setSubscriptions: function(search_model) {
				this.subscriptions = search_model.get('filter') === 'subscriptions';
			},

			fetchArticles: function(search_model) {
				this.resetPagination();
				this.setSortMethod(search_model);
				this.setSubscriptions(search_model);

				var options = {};
				options.data = {
					start: this.start,
					limit: this.limit,
					sort: this.sort,
					subscriptions: this.subscriptions
				};

				//if category
				if (search_model.get('category') > 0) {
                    this.category = search_model.get('category');
                    this.categories = search_model.get('search_category_ids');
                    options.data.category = this.categories;
				}

				options.reset = 1;

				return this.fetch(options);
			},

			fetchScroll: function(options)
			{
				return this.fetch(options);
			},

			hasNext: function() {
				var v = 0;

				if (this.start > 0)
					v = this.start + this.limit;
				else
					v = this.limit;

				return v <= this.length;
			}
		});

		window.KB.home.QuestionCollection = Backbone.Collection.extend({
			model: window.KB.home.Question,
			url: "/intranet/rest/knowledgebase/questions",

			initialize: function() {
				this.resetPagination();
			},

			resetPagination: function() {
				this.start = 0;
				this.limit = 10;
				this.has_next = true;
				this.sort = getUserSortType();
				this.category = 0;
				this.categories = [];

				return this;
			},

			setSortMethod: function(search_model) {
				switch(search_model.get('sort_id')) {
					case 1:
						this.sort = 'created_date';
						break;
					case 2:
						this.sort = 'title';
						break;
					case 3:
						this.sort = 'like_count';
						break;
					default:
						break;
				}

				return this;
			},

			setSubscriptions: function(search_model) {
				this.subscriptions = search_model.get('filter') === 'subscriptions';
			},

			fetchQuestions: function(search_model){
				this.resetPagination();
				this.setSortMethod(search_model);
				this.setSubscriptions(search_model);

				var options = {};
				options.data = {
					start: this.start,
					limit: this.limit,
					sort: this.sort,
					subscriptions: this.subscriptions
				};

				//if category
				if(search_model.get('category') > 0)
				{
                    this.category = search_model.get('category');
                    this.categories = search_model.get('search_category_ids');
                    options.data.category = this.categories;
				}

				options.reset = 1;

				return this.fetch(options);
			},

			fetchScroll: function(options)
			{
				return this.fetch(options);
			},

			hasNext: function() {

				var v = 0;

				if(this.start > 0)
					v = this.start + this.limit;
				else
					v = this.limit;

				return v <= this.length;
			}
		});

        /**
		 * Most recent collection: automatically fetches a list of article objects from /articles
		 * has a category property to filter by category, and an optional parent collection
		 * if the parent collection is set, this collection listens to its sync events and updates
		 * itself when they fire. This will automatically trigger its MostRecentListView to update
		 * (if it is contained in one)
         */
        window.KB.home.MostRecentCollection = Backbone.Collection.extend({
			name: 'most recent articles collection',
			url:'/intranet/rest/knowledgebase/articles',
			model: window.KB.home.Article,
			parentCollection: null,
			initialize: function() {
				this.params = this.params || {};
				this.params.limit = this.params.limit || 5;
				this.setCategory(this.params.category || 0, true);
			},
			fetch: function(options){
				options = options || {};
				options.data = options.data || {};
				options.data.sort = options.data.sort || "created_date";

				for (option in this.params)
                {
					if (this.params.hasOwnProperty(option) && options.data[option] === undefined)
						options.data[option] = this.params[option];
				}

				if (options.data.category === 0)
					delete options.data.category;

				return Backbone.Collection.prototype.fetch.call(this, options);
			},
			setCategory: function(category, forceFetch){
				category = parseInt(category) || 0;
				var oldCategory = this.params.category;
				this.params.category = category;

				if (forceFetch || category !== oldCategory)
					this.fetch();
			},
			setLimit: function(val){
				this.params.limit = val;
			},
			setCategoryCollection: function(collection){
                this.stopListening();
                this.listenTo(collection, "sync change", this.updateFromCategory);
			},
			updateFromCategory: function(category){
				if (typeof category === 'number' || typeof category === 'string')
					categoryId = parseInt(category);
				else if (typeof category === 'object' && category !== null && category.id !== undefined)
					categoryId = parseInt(category.id);
				else
					return null;

				this.setCategory(categoryId);
			}
		});

        window.KB.home.MostRecentQuestionCollection = window.KB.home.MostRecentCollection.extend({
            url:'/intranet/rest/knowledgebase/questions',
            model: window.KB.home.Question,
			name: "most recent question collection"
		});

        window.KB.home.UnpublishedCollection = window.KB.home.MostRecentCollection.extend({
            params: {
            	status: "draft",
				limit: -1,
				subcategories: 1
			},
			name: "draft articles collection"
        });

		// VIEW

		// Generate category menu
		window.KB.home.CategoryNavigationView = Backbone.View.extend({
			el: "#categories-navigation",
			parent_ids: [], // used to keep track of parent IDs of the selected category

			initialize: function(options) {
				_.bindAll(this, "update_category_list");

				//listeners
				this.collection.on("reset", this.update_category_list);
				this.listenTo(window.KB.home.category, 'select_category', this.update_category_list);
			},

			getParentIds: function (selected) {
				var ids = [];
				ids.push(selected.get('id'));

				var category = selected;

				while (category && category.get('parent_id')) {
					ids.push(category.get('parent_id'));
					category = this.collection.get(category.get('parent_id'));
				}

				return ids;
			},

			update_category_list: function(selected) {
				this.$("#categories-list").empty();

				var init = false;

				if(selected instanceof Backbone.Collection)
				{
					init = true;

					// Add 'All categories' item into the collection
					if(this.collection.get(0) == undefined)
						this.collection.add({id: 0, name: lmsg('knowledgebase.list.all'), parent_id: 0},{at: 0});

					this.collection.get(0).set({selected: true});
				}

				// Store parent IDs of the selected category - these need their subcategories rendered
				this.parent_ids = this.getParentIds(selected);

				// Show categories with subcategories as selected
				_.each(this.collection.getTopCategories(), function(category) {
					this.$("#categories-list").append(new window.KB.home.CategoryNavigationItem({model: category}).render().el);

					// Draw subcategories if this category is a parent of the selected category
					if (selected.get('id') && this.parent_ids.indexOf(category.get('id')) !== -1) {
						this.renderSubcategories(category);
					}
				}, this);
			},

			renderSubcategories: function(category, depth) {
				depth = depth || 1;

				_.each(this.collection.getSubcategories(category.get('id')), function(subcategory) {
					this.$("#categories-list").append(new window.KB.home.CategoryNavigationItem({
						model: subcategory,
						depth: depth
					}).render().el);

					if (this.parent_ids.indexOf(subcategory.get('id')) !== -1) {
						this.renderSubcategories(subcategory, depth + 1);
					}
				}, this);
			}
		});

//Draw category menu item
		window.KB.home.CategoryNavigationItem = Backbone.View.extend({
			tagName: 'li',
			tmpl: require("../tmpl/category_navigation_item.tmpl"),
			className: 'nav-item',

			events: {
				'click a': 'fetch_category'
			},

			initialize: function(options) {
				this.depth = options['depth'] || 0;
			},

			render: function() {
				if (this.model == undefined)
					return false;

				var depth = this.depth;
				var model = this.model.toJSON();

				model.indent = function() {
					var indent = '';

					for (var i = 0; i < depth; i++) {
						indent += '-';
					}

					return indent;
				};
				this.$el.html(this.tmpl(model));

				$('#main_breadcrumb').html('<li class="breadcrumb-item"><a href="/knowledgebase/">'+lmsg('knowledgebase.index.app_name')+'</a></li>');

				if (this.model.get('selected')) {
					this.$el.attr('class', 'nav-item').attr('class', 'active nav-item');
				}

				return this;
			},

			fetch_category: function(e) {
				e.preventDefault();
				$("#main-content").unbind();

				window.KB.home.categoryNavigationCollection.get(this.model.get('id')).select();

				if(this.model.get('id') == 0) //All
				{
					window.KB.home.mixedArticleQuestionView.render();
					window.KB.home.categoryNavigationView.update_category_list(this.model);
					//update most recent views if no category
                    updatePanels(0);
					//update URL
					window.KB.home.router.navigate("/");
				}
				else
				{
					window.KB.home.category.fetchCategory(this.model.get('id'));
				}
			}
		});

//Draw Category view content
		window.KB.home.CategoryView = Backbone.View.extend({

			el: "#main-content",
			header_tmpl: require("../tmpl/category_header.tmpl"),
			subcategories_tmpl: "#subcategories_list-tmpl",

			initialize: function(options) {
				_.bindAll(this, "render");
				_.bindAll(this, "get_category_content");
				_.bindAll(this, "get_subcategories_content");
				_.bindAll(this, "get_category_list_content");

				this.collection.sort = getUserSortType();

				this.model.on("sync", this.render);
			},

			render: function(category) {
				this.$el.empty();
				// $("#list_content").empty();

				//Show main category content
				this.get_category_content(category);
				category.set('is_current', true);

				//Show breadcrumbs
				this.get_breadcrumbs_navigation(category);

				//Show subcategories content
				this.get_subcategories_content(category);

				//Show articles and questions list
				this.get_category_list_content(category);

				$("#main-content").unbind();

				window.KB.home.mostRecentListView.hide();
				window.KB.home.mostRecentQuestionView.hide();
				window.KB.home.unpublishedListView.show();
			},

			get_category_content: function(category) {
				data = category.toJSON();
				data['csrf_token'] = document.csrf_token;
				this.$el.append(this.header_tmpl(data));
			},

			get_breadcrumbs_navigation: function(category) {
				var list = '<li class="breadcrumb-item"><a href="/knowledgebase/">'+lmsg('knowledgebase.index.app_name')+'</a></li>',
					crumbs = [];

				while (category) {
					crumbs.unshift(new window.KB.home.CategoryBreadcrumbItemView({model: category}).render().el);

					if (category.get('parent_id'))
						category = window.KB.home.categoryNavigationCollection.get(category.get('parent_id'));
					else
						category = null;
				}

				_.each(crumbs, function(crumb) {
					list += '<li class="breadcrumb-item active">{0}</li>'.replace('{0}', $(crumb).html());
				});

				$('#main_breadcrumb').empty().prepend(list);
			},

			get_subcategories_content: function(category) {
				//if category has no subcategories
				if (!category.get('subcategories'))
					return false;

				var list = $('<ul id="subcategories_content"></ul>');
				list.addClass("subcategories_list list-unstyled");

				_.each(category.get('subcategories'), function(subcategory) {
					list.append(new window.KB.home.SubcategoryItemView({model: subcategory}).render().el);
				}, this);

				this.$el.append(list);
			},

			get_category_list_content: function(category) {
				this.$el.append(window.KB.home.categoryArticlesQuestionsListView.render().el);
			},

			removeSubcategories: function() {
				$("#category_breadcrumbs").remove();
				$("#subcategories_content").remove();
			},

			removeArticleQuestionList: function() {
				$("#category_article_question_list").remove();
			}
		});

		window.KB.home.SubcategoryItemView = Backbone.View.extend({
			tagName: 'li',
			tmpl: require("../tmpl/category_item.tmpl"),
			className: 'nav-item',

			events: {
				'click a': 'fetch_category'
			},

			render: function() {
				var plain_text = this.model.plain_text;

				if (plain_text.length > 120)
					plain_text = plain_text.substr(0,120) + "…";

				this.$el.html(this.tmpl({
					name: this.model.name,
					plain_text: plain_text,
					user_is_subscribed: this.model.user_is_subscribed
				}));

				return this;
			},

			fetch_category: function(e) {
				e.preventDefault();

				window.KB.home.categoryNavigationCollection.get(this.model.id).select();
				window.KB.home.category.fetchCategory(this.model.id);
			}
		});

		window.KB.home.CategoryBreadcrumbItemView = Backbone.View.extend({
			tagName: 'li',
			className: 'breadcrumb-item',

			events: {
				'click a': 'fetch_category'
			},

			initialize: function(options) {
				_.bindAll(this, 'render');
				_.bindAll(this, 'fetch_category');
			},

			render: function() {
				if (this.model.get('is_current') == true) {
					this.$el.text(this.model.get('name')).addClass('active');
				} else {
					this.$el.html($('<a href="#category/'+ this.model.get('id') + '"></a>').text(this.model.get('name')));
				}

				return this;
			},

			renderCustomSelect: function(name, is_selected) {
				if (is_selected == true) {
					this.$el.text(name).addClass('active');
				} else {
					this.$el.html($('<a href="#"></a>').text(name));
				}

				return this;
			},

			fetch_category: function(e) {
				e.preventDefault();
				//set category as selected - done in category colelction
				window.KB.home.categoryNavigationCollection.get(this.model.get('id')).select();
				//fetch selected category
				window.KB.home.category.fetchCategory(this.model.get('id'));
			}
		});

		window.KB.home.MixedArticleQuestionView = Backbone.View.extend({

			el: "#main-content",

			initialize: function() {
				_.bindAll(this, "render");
			},

			render: function() {
				this.$el.empty();
				this.$el.append(window.KB.home.mixedArticleQuestionSearchView.render().el);

				toggleQuestionFunctionality(this.$el);
			}
		});

		window.KB.home.MixedArticleQuestionSearchView = Backbone.View.extend({

			tagName: "div",
			tmpl: require("../tmpl/home_filters_sorting_header.tmpl"),
			className: 'mb-3',

			events: {
				"click #articleLabel": "toggleArticles",
				"click #questionLabel": "toggleQuestions",
				"click #sort a": "updateSortURL",
				"click .js-subscriptions-checkbox": "toggleSubscriptions"
			},

			initialize: function() {
				_.bindAll(this, "render");
				this.listenTo(this.model, 'change', this.render);
			},

			render: function() {
				this.$el.html(this.tmpl(this.model.toJSON()));
				this.delegateEvents(this.events); //make sure we bind the events when page is reloaded

				if (this.model.get('articles') == true) {
                    window.KB.home.articleCollection.fetchArticles(this.model);
                    window.KB.home.mostRecentListView.hide();
					window.KB.home.unpublishedListView.show();
                }else{
                    window.KB.home.mostRecentListView.show();
				}

				if (this.model.get('questions') == true) {
                    window.KB.home.questionCollection.fetchQuestions(this.model);
                    window.KB.home.mostRecentQuestionView.hide();
                    window.KB.home.unpublishedListView.hide();
                }else{
                    window.KB.home.mostRecentQuestionView.show();
				}

				return this;
			},

			getUrl: function() {
				var url = this.model.get('articles') ? 'articles' : 'questions';

				if (this.model.get('sort_id') > 1)
					url += '/sort/' + this.model.get('sort_id');

				if (this.model.get('filter'))
					url += '/filter/' + this.model.get('filter');

				return url;
			},

			updateUrl: function(url) {
				window.KB.home.router.navigate(url || this.getUrl());
			},

			toggleArticles: function() {
				$("#main-content").unbind();

				this.model.setArticles(true); //trigger render()
				this.updateUrl();

				return false;
			},

			toggleQuestions: function() {
				$("#main-content").unbind();

				this.model.setQuestions(true); //trigger render()
				this.updateUrl();

				return false;
			},

			updateSortURL: function(e) {
				var obj = $(e.currentTarget);
				this.updateSort(obj.attr('id'));

				return false;
			},

			updateSort: function(sort_id) {
				// update search object
				this.model.setSort(sort_id); //set sort and trigger render()

				this.updateUrl();

				return false;
			},

			toggleSubscriptions: function(e) {
				if (e.currentTarget.checked)
					this.model.setFilter('subscriptions');
				else
					this.model.setFilter(false);

				this.updateUrl();

				return false;
			}
		});

		window.KB.home.ArticlesListView = Backbone.View.extend({

			tagName: "div",
			tmpl: require("../tmpl/article_list.tmpl"),

			initialize: function() {
				_.bindAll(this, "render");
				this.collection.on("reset", this.render, this);
				this.collection.on("add", this.render, this);
				this.collection.sort = getUserSortType();

				this.isLoading = false;
			},

			render: function() {
				$(".article-question-list").remove();
				this.$el.addClass('article-question-list js-scroll-container');

				var col = this.collection.toJSON();
				_.each(col, function(o) {
					if(o.plain_text.length > 210)
					{
						var plain_text = o.plain_text;
						o.plain_text = plain_text.substr(0,210) + "...";
					}
				}, this);

				this.$el.html(this.tmpl({
					articles: col
				}));

				$("#main-content").append(this.el);

				this.$el.off("onScrollListEnd");
				this.$el.on("onScrollListEnd", this.loadMore.bind(this));
			},

			getThumbnail: function(img) {
				//default image
				var defualt_img = '<img class="media-object" src="/intranet/knowledgebase/html/images/article-placeholder.gif" alt="" width="150">';
				if(!img) return defualt_img;

				return '<img class="media-object" src="'+img+'" alt="" width="150">';
			},

			loadResults: function() {
				this.isLoading = true;

				var self = this;
				var options = {};
				options.data = {
					start: this.collection.start,
					limit: this.collection.limit,
					sort: this.collection.sort,
					subscriptions: this.collection.subscriptions
				};

				if(this.collection.category > 0)
                    options.data.category = this.collection.categories;

				options.remove = false;
				options.success = function(){
					self.isLoading = false;
					$('.ajax-loader').hide();
				};

				this.collection.fetchScroll(options);
			},

			loadMore: function() {
				if(!this.isLoading && this.collection.hasNext())
				{
					$('.ajax-loader').show();
					this.collection.start += 10;
					this.loadResults();
				}
			}
		});

		window.KB.home.QuestionsListView = Backbone.View.extend({
			tagName: "div",
			tmpl: require("../tmpl/question_list.tmpl"),

			initialize: function() {
				_.bindAll(this, "render");
				this.collection.on("reset", this.render, this);
				this.collection.on("add", this.render, this);
				this.collection.sort = getUserSortType();

				this.isLoading = false;
			},

			render: function() {
				$(".article-question-list").remove();
				this.$el.addClass('article-question-list js-scroll-container');

				var col = this.collection.toJSON();
				_.each(col, function(o) {
					if(o.plain_text.length > 210)
					{
						var plain_text = o.plain_text;
						o.plain_text = plain_text.substr(0,210) + "...";
					}
				}, this);
				this.$el.html(this.tmpl({
					questions: col
				}));

				$("#main-content").append(this.el);

				this.$el.off("onScrollListEnd");
				this.$el.on("onScrollListEnd", this.loadMore.bind(this));
			},

			loadResults: function() {
				this.isLoading = true;
				var self = this;
				var options = {};
				options.data = {start: this.collection.start, limit: this.collection.limit, sort: this.collection.sort };

				if(this.collection.category > 0)
					options.data.category = this.collection.category;

				options.remove = false;
				options.success = function(){
					$('.ajax-loader').hide();
					self.isLoading = false;
				};

				this.collection.fetchScroll(options);
			},

			loadMore: function() {
				if(!this.isLoading && this.collection.hasNext())
				{
					$('.ajax-loader').show();
					this.collection.start += 10;
					this.loadResults();
				}
			}
		});

//Draw Category - All Articles / Questions View
		window.KB.home.CategoryArticleQuestionSearchView = Backbone.View.extend({

			tagName: "div",
			className: "category_all_list",
			tmpl: require("../tmpl/list_sort_options.tmpl"),

			events: {
				"click .sort a": "updateSort"
			},

			initialize: function() {
				_.bindAll(this, "render");
				this.listenTo(this.model, 'change', this.render);
			},

			render: function() {
				if (!$(".category_all_list").length)
					$("#main-content").append(this.el);

				this.$el.empty();
				this.$el.html(this.tmpl(this.model.toJSON()));

				//this.renderBreadcrumbs();

				if (this.model.get('articles') == true)
					window.KB.home.articleCollection.fetchArticles(this.model);

				if (this.model.get('questions') == true)
					window.KB.home.questionCollection.fetchQuestions(this.model);

				this.delegateEvents(this.events);
			},

			renderBreadcrumbs: function() {
				var category = window.KB.home.category;

				var $breadcrumb = $(".category_all_list .breadcrumb");

				if(this.model.get('articles') == true)
					$breadcrumb.prepend('<li class="active">All articles</li>');
				if(this.model.get('questions') == true)
					$breadcrumb.prepend('<li class="active">All questions</li>');

				$breadcrumb.prepend(new window.KB.home.CategoryBreadcrumbItemView({ model: category }).renderCustomSelect(category.get('name'), false).el);

				if(!_.isEmpty(category.get('parent')))
				{
					var parent_object = new window.KB.home.Category(category.get('parent'));
					$breadcrumb.prepend(new window.KB.home.CategoryBreadcrumbItemView({ model: parent_object }).renderCustomSelect(parent_object.get('name'), false).el);
				}
			},

			updateSort: function(e) {
				e.preventDefault();
				var obj = $(e.currentTarget);
				this.model.setSort(obj.attr('id'));
			}
		});

//Draw Category - Articles & List View Overview limited by 5 most recent entries
		window.KB.home.CategoryArticlesQuestionsListView = Backbone.View.extend({
			tagName: 'div',
			tmpl: require("../tmpl/category_articles_questions.tmpl"),

			events: {
				"click #view_all_articles" : "showAllCategoryArticles",
				"click #view_all_questions" : "showAllCategoryQuestions"
			},

			initialize: function() {
				_.bindAll(this, "render");
			},

			render: function() {
				if (!window.KB.params.questions_enabled)
				{
                    // IE has it's own special way of creating custom events. Typically.
                    var evt;
                    try {
                        evt = new Event("click");
                    }
                    catch (error) {
                        // IE
                        evt = document.createEvent("Event");
                        evt.initEvent("click", false, false);
                    }
                    // if questions are disabled we want to show the list of all articles instead, fake the click
                    this.showAllCategoryArticles(evt);
					return this;
				}

				this.delegateEvents(this.events); //make sure we bind the events when page is reloaded

				$("#category_article_question_list").remove();
				this.$el.attr('id', 'category_article_question_list').addClass("row");

				var articles = new window.KB.home.ArticleCollection();
				var questions = new window.KB.home.QuestionCollection();


				var self = this;
				var options = {};
				options.silent = true;
				//options.reset = 1;

				var categories = this.getCategoryWithSubcategories(this.model); //Get current category and its subcategories if any

				options.data = { start: 0, limit: 5, category: categories.join(','), sort: getUserSortType() }; //prepare search options

				//wait until articles and questions finish fetch
				$.when( articles.fetch(options), questions.fetch(options) ).done(function(){
                    // if here are no questions to show we want to redirect to the articles listing instead.
                    if (questions.length === 0)
                    {
                        // IE has it's own special way of creating custom events. Typically.
                        var evt;
                        try {
                            evt = new Event("click");
                        }
                        catch (error) {
                            // IE
                            evt = document.createEvent("Event");
                            evt.initEvent("click", false, false);
                        }

                        self.showAllCategoryArticles(evt);
                    } else
                    {
                    	self.$el.html(self.tmpl({
							articles: articles.toJSON(),
							questions: questions.toJSON()
                    	}));
                    }
				});

				return this;
			},

			showAllCategoryArticles: function(e) {
				e.preventDefault();
				window.KB.home.mostRecentListView.hide();
				window.KB.home.mostRecentQuestionView.show();
				window.KB.home.unpublishedListView.show();

				window.KB.home.categoryView.removeSubcategories();
				window.KB.home.categoryView.removeArticleQuestionList();

				var categories = this.getCategoryWithSubcategories(this.model);
				window.KB.home.categoryArticleQuestionSearch.setListAllArticles(this.model.get('id'), categories);
			},

			showAllCategoryQuestions: function(e) {
				e.preventDefault();
                window.KB.home.mostRecentQuestionView.hide();
                window.KB.home.mostRecentListView.show();
				window.KB.home.unpublishedListView.hide();

				window.KB.home.categoryView.removeSubcategories();
				window.KB.home.categoryView.removeArticleQuestionList();

				var categories = this.getCategoryWithSubcategories(this.model);
				window.KB.home.categoryArticleQuestionSearch.setListAllQuestions(this.model.get('id'), categories);
			},

			// Get the IDs of the given category and all of its subcategories, if any
			getCategoryWithSubcategories: function(category, ids) {
				ids = ids || [];

				ids.push(category.get('id'));

				_.each(category.get('subcategories'), function(subcategory) {
					subcategory_model = window.KB.home.categoryNavigationCollection.get(subcategory.id);

					if (typeof subcategory_model !== 'undefined') {
						ids.concat(this.getCategoryWithSubcategories(subcategory_model, ids));
					}
				}, this);

				return ids;
			}
		});

		window.KB.home.SearchboxView = Backbone.View.extend({
			el: '#search-box',

			events: {
				'click .btn_submit': 'search',
				'keypress #keywords': 'showKey'
			},

			updateSearch: function () {
				this.$('#keywords').val(this.model.get('keywords'));
				this.$('#category').select2("val", this.model.get('category'));
				this.$('#language').select2("val", this.model.get('language'));
			},

			search: function(e){
				// $('#home_list').hide();

				this.model.set({
					keywords: this.$('#keywords').val(),
					category: this.$('#category').val(),
					language: this.$('#language').val()
				});

				// Display image here..
				window.KB.home.articleCollection.fetchArticles(this.model);
				return false;
			},

			showKey: function(e){
				if(e.keyCode === 13)
				{
					window.location.href = '/knowledgebase/#search/';
				}
			}
		});

        /**
		 * Most recent list view: renders the most recent list inside #most-recent-box
		 * Contains a most recent collection which syncs with /articles/recent
         */
        window.KB.home.MostRecentListView = Backbone.View.extend({
			el:'#most-recent-box',
			template: require('../tmpl/most_recent.tmpl'),
			collection: new window.KB.home.MostRecentCollection(),
			initialize: function(){
                this.listenTo(this.collection, "sync change", this.render);
			},
			render: function(event){
                var rendered_content = this.template({items: this.collection.models});
                this.$el.html(rendered_content);
			},
			hide: function(){
				$(this.el).hide();
			},
            show: function(){
                $(this.el).show();
            },
			hideElements: function ()	{
				$('div[name="question_functionality"]').addClass('hidden');
				$('a[id="questionLabel"]').addClass('hidden');
			},
		});

        window.KB.home.UnpublishedListView = window.KB.home.MostRecentListView.extend({
            el:'#draft-article-box',
            collection: new window.KB.home.UnpublishedCollection(),
            initialize: function () {
                this.constructor.__super__.initialize.apply(this, arguments);
            },
            template: require('../tmpl/draft_articles.tmpl'),
        });

        window.KB.home.MostRecentQuestionView = window.KB.home.MostRecentListView.extend({
            el:'#most-recent-questions',
            template: require('../tmpl/recent_questions.tmpl'),
            collection: new window.KB.home.MostRecentQuestionCollection(),
            render: function(event){
                if(window.kb_params['questions_enabled'])
					this.constructor.__super__.render.apply(this, arguments);
                else
					this.$el.html('');
            }
        });

		// ROUTER

		var Router = Backbone.Router.extend({
			routes: {
				"articles": "view_articles",
				"articles/filter/:filter": "view_articles",
				"articles/sort/:sort_id": "view_articles",
				"articles/sort/:sort_id/filter/:filter": "view_articles",
				"questions": "view_questions",
				"questions/filter/:filter": "view_questions",
				"questions/sort/:sort_id": "view_questions",
				"questions/sort/:sort_id/filter/:filter": "view_questions",
				"category/:cat_id_or_name": "category",
				"*path": "start"
			},

			observer: null,

			start: function() {
				this.view_articles();
			},

			view_articles: function(sort_id, filter) {
				window.KB.home.categoryNavigationCollection.fetchCategories();

				window.KB.home.mixedArticleQuestionSearchView.model.attributes['articles'] = true;
				window.KB.home.mixedArticleQuestionSearchView.model.attributes['questions'] = false;

				sort_id = parseInt(sort_id) || 0;

				if (sort_id) {
					if (window.KB.home.mixedArticleQuestionSearchView.model.isFilter(sort_id)) {
						window.KB.home.mixedArticleQuestionSearchView.model.setFilter(sort_id);
					} else {
						window.KB.home.mixedArticleQuestionSearchView.model.attributes['sort_id'] = sort_id;
					}
				} else
				{
					var user_sort_id = getUserSortId();
					if (window.KB.home.mixedArticleQuestionSearchView.model.isFilter(user_sort_id)) {
						window.KB.home.mixedArticleQuestionSearchView.model.setFilter(user_sort_id);
					} else {
						window.KB.home.mixedArticleQuestionSearchView.model.attributes['sort_id'] = user_sort_id;
					}
				}

				if (filter)
					window.KB.home.mixedArticleQuestionSearchView.model.setFilter(filter);

				window.KB.home.mixedArticleQuestionView.render();
			},

			view_questions: function(sort_id, filter) {
				if (window.KB.params.questions_enabled === false || window.KB.params.questions_enabled === 0) {
					this.navigate('articles', {trigger: true, replace: true});
					return;
				}

				window.KB.home.categoryNavigationCollection.fetchCategories();

				window.KB.home.mixedArticleQuestionSearchView.model.attributes['articles'] = false;
				window.KB.home.mixedArticleQuestionSearchView.model.attributes['questions'] = true;

				sort_id = parseInt(sort_id) || 0;

				if (sort_id) {
					if (window.KB.home.mixedArticleQuestionSearchView.model.isFilter(sort_id)) {
						window.KB.home.mixedArticleQuestionSearchView.model.setFilter(sort_id);
					} else {
						window.KB.home.mixedArticleQuestionSearchView.model.attributes['sort_id'] = sort_id;
					}
				} else
				{
					var user_sort_id = getUserSortId();
					if (window.KB.home.mixedArticleQuestionSearchView.model.isFilter(user_sort_id)) {
						window.KB.home.mixedArticleQuestionSearchView.model.setFilter(user_sort_id);
					} else {
						window.KB.home.mixedArticleQuestionSearchView.model.attributes['sort_id'] = user_sort_id;
					}
				}

				if (filter)
					window.KB.home.mixedArticleQuestionSearchView.model.setFilter(filter);

				window.KB.home.mixedArticleQuestionView.render();
			},

			category: function(cat_id_or_name) {
				if (cat_id_or_name == 0) {
					this.view_articles();
				} else {
					window.KB.home.categoryNavigationCollection.fetchCategories().done(function(){
						var category = window.KB.home.categoryNavigationCollection.get(cat_id_or_name);
						if (category) {
                            category.select();
                            window.KB.home.category.fetchCategory(cat_id_or_name);
                        } else {
                            cla.showMessage("Inaccessible or wrong category", '', 'info');
                        }
					});
				}
			},

			initialize: function(options) {
				//Custom events
				var evnt = _.extend({}, Backbone.Events);

				if (typeof kb_params.default_sort == 'undefined' || kb_params.default_sort.length < 1){
					setUserSortFromType('created_date');
				} else
				{
					setUserSortFromType(kb_params.default_sort);
				}

				if (typeof kb_params.questions_enabled == 'undefined'){
					window.KB.params.questions_enabled = 1;
				} else
				{
					window.KB.params.questions_enabled = kb_params.questions_enabled;
				}

				if (kb_params.questions_enabled === false || kb_params.questions_enabled === 0) {
					$('a[id="questionLabel"]').addClass('hidden');
				}

				//Collections
				window.KB.home.categoryNavigationCollection = new window.KB.home.CategoryNavigationCollection();
				window.KB.home.categoryCollection = new window.KB.home.CategoryCollection();
				window.KB.home.articleCollection = new window.KB.home.ArticleCollection();
				window.KB.home.questionCollection = new window.KB.home.QuestionCollection();
				//window.KB.home.mostRecentCollection = new window.KB.home.MostRecentCollection();

				//Models
				window.KB.home.categoryNavigation = new window.KB.home.CategoryNavigation({evnt: evnt});
				window.KB.home.category = new window.KB.home.Category();
				window.KB.home.article = new window.KB.home.Article();
				window.KB.home.question = new window.KB.home.Question();

				window.KB.home.mixedArticleQuestionSearch = new window.KB.home.MixedArticleQuestionSearch();
				window.KB.home.categoryArticleQuestionSearch = new window.KB.home.CategoryArticleQuestionSearch();

				//View
				window.KB.home.categoryView = new window.KB.home.CategoryView({ model: window.KB.home.category, collection: window.KB.home.categoryNavigationCollection });
				window.KB.home.categoryNavigationView = new window.KB.home.CategoryNavigationView({ collection: window.KB.home.categoryNavigationCollection });
				window.KB.home.articlesListView = new window.KB.home.ArticlesListView({ collection: window.KB.home.articleCollection });
				window.KB.home.questionsListView = new window.KB.home.QuestionsListView({ collection: window.KB.home.questionCollection });
				window.KB.home.categoryArticlesQuestionsListView = new window.KB.home.CategoryArticlesQuestionsListView({ model: window.KB.home.category});
				window.KB.home.categoryArticleQuestionSearchView = new window.KB.home.CategoryArticleQuestionSearchView({ model: window.KB.home.categoryArticleQuestionSearch });

				//Mixed Article Question View
				window.KB.home.mixedArticleQuestionView = new window.KB.home.MixedArticleQuestionView();
				window.KB.home.mixedArticleQuestionSearchView = new window.KB.home.MixedArticleQuestionSearchView({ model: window.KB.home.mixedArticleQuestionSearch });

				//Most recent list view
				window.KB.home.mostRecentListView = new window.KB.home.MostRecentListView();
				window.KB.home.mostRecentListView.collection.setCategoryCollection(window.KB.home.category);
                //window.KB.home.mostRecentListView.collection.setParentCollection(window.KB.home.articleCollection);
				if (kb_params.questions_enabled === false || kb_params.questions_enabled === 0) {
					window.KB.home.mostRecentListView.hideElements();
				}


				//unpublished articles
                window.KB.home.unpublishedListView = new window.KB.home.UnpublishedListView();
                window.KB.home.unpublishedListView.collection.setCategoryCollection(window.KB.home.category);

                //Most recent questions
                window.KB.home.mostRecentQuestionView = new window.KB.home.MostRecentQuestionView();
                window.KB.home.mostRecentQuestionView.collection.setCategoryCollection(window.KB.home.category);
                //window.KB.home.mostRecentQuestionView.collection.setParentCollection(window.KB.home.questionCollection);

				this.observer = new ScrollObserver();
				this.observer.registerScrollListener()
			}
		});


		// INITIALIZE

		window.KB.home.router = new Router();
		Backbone.history.start();

		// Load comments in when we hit the bottom of the page
		$(window).scroll(function(){
			var cvPos = $(window).scrollTop();
			// If we've hit the bottom of the page...
			if($(window).scrollTop() >= ($(document).height() - ($(window).height())) - 150) {
				$("#main-content").trigger("endOfScroll");
			}
		});
	};
});

var updatePanels = function(id) {
	id = parseInt(id) || 0;
	KB.home.mostRecentListView.collection.updateFromCategory(id);
	KB.home.mostRecentQuestionView.collection.updateFromCategory(id);
	KB.home.unpublishedListView.collection.updateFromCategory(id);
};

// get user's preferred sorting
var getUserSortId = function(){
	return parseInt(window.KB.params.user_sort_id) || 0;
};

var getUserSortType = function(){
	return window.KB.params.user_sort_type;
};


// locally update the user's preferred sorting
var setUserSortFromId = function(sort_id)
{
	sort_id = parseInt(sort_id) || 0;

	window.KB.params.user_sort_id = sort_id;

	var sort_type = '';

	switch (sort_id){
		case 1:
			sort_type = 'created_date';
			break;
		case 2:
			sort_type = 'title';
			break;
		case 3:
			sort_type = 'like_count';
			break;
		default:
			sort_type = 'id';
			break;
	}
	window.KB.params.user_sort_type = sort_type;
};

var setUserSortFromType = function(sort_type)
{
	switch (sort_type){
		case 'created_date':
			sort_id = 1;
			break;
		case 'title':
			sort_id = 2;
			break;
		case 'like_count':
			sort_id = 3;
			break;
		default:
			sort_id = 0;
			break;
	}
	setUserSortFromId(sort_id);
};

// sort id/type is initialized on page load in a few different places in the application.
// When it changes, we need to update them, otherwise a page refresh is needed for the change to take full effect.
var updateObjSortSettings = function()
{
	window.KB.home.articleCollection.sort = getUserSortType();
	window.KB.home.questionCollection.sort = getUserSortType();

	window.KB.home.categoryArticleQuestionSearch.sort_id = getUserSortId();
	window.KB.home.mixedArticleQuestionSearch.sort_id = getUserSortId();

	if (kb_params.questions_enabled === false || kb_params.questions_enabled === 0) {
		$('div[name="question_functionality"]').addClass('hidden');
	}
};
