define(['jquery'], function ($) {

	function extend(){
		for(var i=1; i<arguments.length; i++)
			for(var key in arguments[i])
				if(arguments[i].hasOwnProperty(key))
					arguments[0][key] = arguments[i][key];
		return arguments[0];
	}

	// base object
	var api_base = {
		version : '',
		model: '',
		endpoint: '',

		/**
		 * Model key value pairs, as populated by the SCORM package.
		 */
		storage: {},

		/**
		 * Current number of interactions.
		 */
		question: 0,

		/**
		 * Associative array of default error codes. overwritten in adapters.
		 */
		error_codes: {
			0:   {short: 'No error', full: 'No error occurred, the previous API call was successful.'},
			101: {short: 'General Exception', full: 'No specific error code exists to describe the error. Use LMSGetDiagnostic for more information.'}
		},

		/**
		 * List of errors occurring during SCORM execution.
		 */
		errors: [],

		/**
		 * Developer debugging
		 */
		trace: false,

		/**
		 * SCORM initialized
		 */
		initialized: false,

		timer: 0,

        triggerFinish: false,

        /**
		 * Custom initialisation function.
		 * @param {string} version
		 * @param {object} model
		 * @param {array} error_codes
		 */
		init: function (version, model, error_codes)
		{
			this.version = version;
			this.model = model;
			this.endpoint = '/intranet/rest/lms/scorm/' + LMS.module_id + '/' + LMS.user_id;
			this.error_codes = extend(this.error_codes, error_codes);
		},

		/**
		 * For debugging.
		 * @param {string} msg
		 */
		log: function (msg) {
			if (this.trace) {
				console.log(msg);
			}
		},

		/**
		 * Send collected SCORM information to LMS.
		 */
		ajax: function () {
			var api = this;
			$.ajax({
				method: 'post',
				url: this.endpoint,
				data: JSON.stringify({ storage: this.storage, version: this.version }),
				contentType: 'application/json',
				error: function (data) {
					api.log(data);
				}
			});
		},

		/**
		 * Clear up working variables
		 */
		clear: function () {
			this.initialized = false;
			this.errors = [];
			this.storage = {};
			this.question = 0;
			this.timer = 0;

			this.log('cla cleared');
		},

		/**
		 * ---------------
		 * SCORM signature (all versions)
		 */

		/**
		 * Begins a communication session with the LMS.
		 */
		initialize: function (last_question) {

			last_question = parseInt(last_question || 0);

			this.errors = [];
			this.storage = {};
			this.storage['cla.last_question'] = last_question;
			this.question = last_question;
			this.initialized = true;
			this.timer = new Date().getTime(); // time tracker

			this.log('api init');

            const self = this;

            // trigger LmsFinish() once user close the module
            window.addEventListener('beforeunload', function (e) {
                e.preventDefault();
                self.log('triggered finish');
                self.finish()
            });

			return 'true';
		},

		/**
		 * Ends a communication session with the LMS.
		 */
		finish: function () {
			if (!this.errors.length) {
				var end = new Date().getTime();
				this.storage['cla.time_taken'] = (end - this.timer);
				this.storage['cla.last_question'] = this.question;
				this.ajax();
			} else {
				var code = this.errorCode();
				this.log(this.diagnose(code));
			}
			this.clear();

			this.log('finish');
			return 'true';
		},

		/**
		 * Retrieves a value from the LMS.
		 * @param {CMIElement} value
		 */
		get: function (element) {
			if (!this.initialized) {
				this.errors.unshift(301);
				return 'false';
			}

			this.log('get : ' + element + ' : ' + (this.storage[element] || ''));
			return this.storage[element] || '';
		},

		/**
		 * Saves a value to the LMS.
		 * @param {element}
		 * @param {value}
		 */
		set: function (element, value) {
			if (!this.initialized) {
				this.errors.unshift(301);
				return 'false';
			}

            if (element === 'cmi.core.lesson_status') {
                const storableStatuses = ['passed', 'failed', 'completed'];
                if (storableStatuses.includes(value) && value !== this.storage['cmi.core.lesson_status']) {
                    this.triggerFinish = true;
                }
            }

			if (value) {
				this.storage[element] = value;
			}

			this.log('set : ' + element + ' : ' + value);

            return 'true';
		},

		/**
		 * Indicates to the LMS that all data should be persisted (not required).
		 */
		commit: function () {
            // check if we can finish the module
            if (this.triggerFinish === true) {
                this.log('triggered finish');
                this.finish();

                this.triggerFinish = false;
            }

			this.log('commit');
			return 'true';
		},

		/**
		 * Returns the error code that resulted from the last API call.
		 */
		errorCode: function () {
			if (this.errors.length > 0) {
				return this.errors[this.errors.length - 1];
			}

			return 0;
		},

		/**
		 * Returns a short string describing the specified error code.
		 * @param {CMIErrorCode} code
		 */
		errorString: function (code) {
			for (var i in this.errors) {
				if (code === i) {
					return this.error_codes[i].short;
				}
			}
		},

		/**
		 * Returns detailed information about the last error that occurred.
		 * @param {CMIErrorCode} code
		 */
		diagnose: function (code) {
			for (var i in this.errors) {
				if (code === i) {
					return this.error_codes[i].full;
				}
			}
		}
	};

	window.cla_scorm_api = api_base;

});
