define(['backbone', 'jquery'], function (Backbone, $) {
	return function (mBootstrap) {

		/**
		 * Course module (Model)
		 */
		LMS.CourseModule = Backbone.Model.extend({
			defaults: {
				'edit_label': lmsg('lms.module.module_edit'),
				'remove_module_label': lmsg('lms.course.remove_module'),
				'move_module': lmsg('lms.course.move_module')
			},
			url: function () {
				return "/intranet/rest/lms/course_modules/" + parseInt(LMS.course_id, 10) + '/' + this.get("assign_id");
			}
		});

		/**
		 * Course modules (Collection)
		 */
		LMS.CourseModulesListCollection = Backbone.Collection.extend({
			model: LMS.CourseModule,
			manage_mode: false,
			url: function () {
				return '/intranet/rest/lms/course_modules/' + parseInt(LMS.course_id, 10) + (this.manage_mode === true ? '/manage' : '');
			}
		});

		/**
		 * Course modules main container (View)
		 */
		LMS.CourseModulesContainerView = Backbone.View.extend({
			el: ".module_list",
			manage_mode: false,

			initialize: function () {
				_.bindAll(this, "appendModule", "reset");
				this.collection.bind("add", this.appendModule);
				this.collection.bind("reset", this.reset);
			},

			appendModule: function (moduleModel) {
				this.$el.append(new LMS.CourseModuleView({model: moduleModel}).render().el);
			},

			reset: function () {
				this.$el.empty();
				this.collection.each(function (moduleModel) {
					if (this.manage_mode) moduleModel.set('manage_view', true);
					this.appendModule(moduleModel);
				}, this);
			}
		});

		/**
		 * Course module single container (View)
		 */
		LMS.CourseModuleView = Backbone.View.extend({
			templateID: "#courseModule",
            eventListenerInit: false,
            userListChecked: {},

			events: {
				"click .js-detach-module":            "removeModule",
                "click .js-complete-module-add":      "modalStep2",
                "click .js-complete-module-cancel":   "modalCancel",
                "click .js-complete-module-checkbox": "modalNextEnable",
                "click .js-btn-modal-show":           "showModal",
                "click .js-close-modal":              "closeModal",
			},

			initialize: function () {
				_.bindAll(this, "render", "_remove", "removeModule", "modalStep2", "modalNextEnable", "modalCancel", "updateAuditUsersStatusesTable", "showModal", "closeModal");
				this.model.bind("destroy", this._remove);

                // This is a small hack to solve the issue with remaining backdrop shadow for the modal window
                if (!this.eventListenerInit) {
                    this.eventListenerInit = true;

                    $(document).on('hide.bs.modal', '.modal', function () {
                        $('body').removeClass('modal-open');
                        $('.modal-backdrop').remove();
                    });
                }

                $('.js-step-number').text(lmsg('lms.course.steps', 1, 2));
            },

			_remove: function () {
				this.remove();
				return false;
			},

			removeModule: function (e) {
				this.$('.js-detach-module').tooltip('hide');

				if (confirm(lmsg('lms.course.confirm_module_remove'))) {
					this.model.destroy({
						error: function (model, response) {
							cla.showMessage(response.responseText, '', 'info');
						}, success: function (model, response) {
							cla.showMessage(response, '', false);
						}
					});
				}
				return false;
			},

			render: function () {
				this.$el.html($(this.templateID).tmpl(this.model.toJSON()));
                return this;
			},

            /**
             * Enable/disable btn "Next" in modal window
             */
            modalNextEnable: function (event) {
                if (event.currentTarget.value) {
                    var user_id = parseInt(event.currentTarget.value);

                    var clickedUser = this.userCollection.models
                        .filter(function(user) {
                            if (user.attributes.id === user_id && event.currentTarget.checked) {
                                user.attributes.checked = true;
                                return user;
                            }
                        });

                    // check if not already in the array and add model to the array
                    if (event.currentTarget.checked)
                        this.userListChecked[user_id] = clickedUser[0];

                    // remove from array if unchecked
                    if (!event.currentTarget.checked && this.userListChecked.hasOwnProperty(user_id))
                        delete this.userListChecked[user_id];

                    // enable button next
                    this.$('.js-complete-module-add').prop(
                        'disabled',
                        Object.keys(this.userListChecked).length === 0
                    );
                }
            },

            /**
             * Display confirm view in the confirmation modal window
             */
            modalStep2: function () {
                var id = this.model.get('id');
                var templateId = $('#modalStep2');

                var btnNext = this.$('.js-complete-module-add');
                var modalContainerId = this.$('#markAsCompleted_' + id);
                var modalContainerBody = modalContainerId.find('.modal-body');

                // last step collect users and complete the module
                if (btnNext.hasClass('js-complete-module-step-2')) {
                    LMS.completeModuleModel = new LMS.CompleteModule([], {id: id});

                    // update the table with enrolled user for module
                    LMS.completeModuleModel.on('updateModelEvent', this.updateAuditUsersStatusesTable, this);

                    LMS.completeModuleModel.save(
                        {
                            id: id,
                            users: this.userListChecked,
                        },
                        {
                            type: 'POST',
                            success: function (model, response) {
                                var usersListText = '';

                                if (response.success === true && Array.isArray(response.users)) {
                                    if (response.users.length === 1)
                                        usersListText += response.users[0]['username'];
                                    else {
                                        response.users.map(function (user) {
                                            usersListText += user.username + ', ';
                                        });

                                        usersListText = usersListText.slice(0, -2);
                                    }
                                }

                                modalContainerId.modal('hide');

                                // trigger event to update table with enrolled user in module
                                model.trigger('updateModelEvent', response.users);
                                cla.showMessage('', lmsg('lms.module.complete.success_message', usersListText));
                            },
                            error: function (model, response) {
                                console.error(response);
                                cla.showMessage(response.responseText, '', true);
                            }
                        }
                    );

                    return;
                }

                // hide step 1
                modalContainerBody.find('#modal-step-1').addClass('d-none');
                btnNext.text(lmsg('lms.course.confirm'));
                btnNext.addClass('js-complete-module-step-2');
                this.$('.js-step-number').text(lmsg('lms.course.steps', 2, 2));
                this.$('.js-complete-module-cancel').text(lmsg('lms.course.previous'));

                // add table to step 2
                modalContainerBody.find('#modal-step-2').empty();
                modalContainerBody.find('#modal-step-2').append($(templateId).tmpl({users: this.userListChecked}));
                modalContainerBody.find('#modal-step-2').removeClass('d-none');
            },

            /**
             * Action on click cancel button
             */
            modalCancel: function () {
                var modalContainerId = this.$('#markAsCompleted_' + this.model.get('id'));
                var modalContainerBody = this.$(modalContainerId).find('.modal-body');
                var btnConfirm = this.$('.js-complete-module-add');

                // Move view back to step 1
                if (btnConfirm.hasClass('js-complete-module-step-2')) {
                    modalContainerBody.find('#modal-step-1').removeClass('d-none');
                    modalContainerBody.find('#modal-step-2').addClass('d-none');
                    btnConfirm.text(lmsg('lms.common.next')).removeClass('js-complete-module-step-2');
                    this.$('.js-complete-module-cancel').text(lmsg('lms.common.cancel'));
                    this.$('.js-step-number').text(lmsg('lms.course.steps', 1, 2));

                    return this;
                }

                modalContainerId.modal('hide');
                this.$(document).trigger('hidden.bs.modal');

                // clean up checkboxes
                this.userListChecked = {};

                this.render();
            },

            /**
             * Re-render table when user close the modal window
             */
            closeModal: function () {
                this.userListChecked = {};
                this.render();
            },

            /**
             * Show and render the modal window for the module.
             * @param event
             */
            showModal: function (event) {
                // Initialize collection for paginated table
                this.userCollection = new LMS.PaginatedCollection([], { id: this.model.get('id') });
                this.userCollection.fetchPage(1);
                var view = new LMS.PaginatedView({ collection: this.userCollection });
                var self = this;

                // remember the state of checked users on different pages
                view.on('finishedUserTableRenderEvent', function () {
                    Object.keys(self.userListChecked).map(function (id) {
                        $('.js-complete-module-checkbox[value="' + id + '"]').prop('checked', true);
                    });
                });

                view.render();
            },

            /**
             * Update table after request send to the server
             * @param users
             */
            updateAuditUsersStatusesTable: function (users) {
                this.userListChecked = {};

                var id = this.model.get('id');
                this.userCollection.fetchPage(1);

                // hide button "Mark as completed"
                this.userCollection.on('reset', function() {
                    if (this.toJSON().length === 0)
                        $('#btnModalShow_' + id).hide();
                });

                this.render();
            }
		});

		/**
		 * Main controller (Router)
		 */
		LMS.Router = Backbone.Router.extend({

			routes: {
				"": "start",
				"manage": "manage",
				"done": "done",
				"audit": "audit",
				"browse": "browseModules"
			},

			initialize: function () {
				LMS.manageButtons = new LMS.ManageModules();
				LMS.courseModulesListCollection = new LMS.CourseModulesListCollection();
				LMS.courseModulesContainerView = new LMS.CourseModulesContainerView({collection: LMS.courseModulesListCollection});
				LMS.browseModules = new LMS.BrowseModules();
			},

			bootstrap: function () {
				LMS.courseModulesListCollection.reset(mBootstrap);
			},

			start: function () {
				this.bootstrap();
				this.setActiveTab('Done');
			},

			manage: function () {
                LMS.courseModulesContainerView.manage_mode = true;
                LMS.courseModulesListCollection.manage_mode = true;
				LMS.courseModulesListCollection.fetch({reset:true});
				$('#browse-module-table, #course-audit').hide();
				$('#manage-module-list, #course-manage-modules').show();
				this.setActiveTab('Manage');
            },

			done: function () {
				LMS.courseModulesContainerView.manage_mode = false;
				LMS.courseModulesListCollection.manage_mode = false;
				LMS.courseModulesListCollection.fetch({reset:true});
				$('#browse-module-table, #course-manage-modules, #course-audit').hide();
				$('#manage-module-list').show();
				this.setActiveTab('Done');
			},

			audit: function () {
				$('#browse-module-table, #manage-module-list, #course-manage-modules').hide();
				$('#course-audit').show();
				LMS.auditListCollection = new LMS.AuditListCollection();
				LMS.auditContainerView = new LMS.AuditContainerView({collection: LMS.auditListCollection});
				LMS.auditListCollection.fetch({reset:true});
				this.setActiveTab('Audit');
			},

			browseModules: function () {
				LMS.selectedModulesCollection = new LMS.SelectedModulesListCollection();
				LMS.modulesListCollection = new LMS.ModulesListCollection();
				LMS.modulesContainerView = new LMS.ModulesContainerView({collection: LMS.modulesListCollection});
				LMS.modulesListCollection.fetch({reset:true});

				$('#browse-module-table').show();
				$('#manage-module-list').hide();
				$('#course-manage-modules').hide();
				$('#course-audit').hide();
				this.setActiveTab('Manage');
			},
			setActiveTab: function (tab) {
				var target = '#courseManageTab' + tab;
				$('#course-manage-buttons').find('.nav-item .nav-link').removeClass('active');
				$(target).find('.nav-link').addClass('active');
			}
		});

		/**
		 * Selected module (Model)
		 */
		LMS.SelectedModule = Backbone.Model.extend({
			url: function () {
				return this.isNew() ? "/intranet/rest/lms/course_modules/" : "/intranet/rest/lms/course_modules/" + parseInt(LMS.course_id, 10) + '/' + this.get('id');
			}
		});

		/**
		 * Selected modules (Collection)
		 */
		LMS.SelectedModulesListCollection = Backbone.Collection.extend({
			model: LMS.SelectedModule,
			url: '/intranet/rest/lms/course_modules/' + parseInt(LMS.course_id, 10)
		});

		/**
		 * Module (Model)
		 */
		LMS.Module = Backbone.Model.extend({
			defaults: {
				assign_module: 'Add Module',
				remove_module: 'Remove Module'
			},
			url: function () {
				return "/intranet/rest/lms/course_modules/" + parseInt(LMS.course_id, 10);
			}
		});

		/**
		 * Modules (Collection)
		 */
		LMS.ModulesListCollection = Backbone.Collection.extend({
			model: LMS.Module,
			url: '/intranet/rest/lms/browse_modules/' + parseInt(LMS.course_id, 10)
		});

        /**
         * Complete module
         */
        LMS.CompleteModule = Backbone.Model.extend({
            defaults: {
                users: [],
                id: 0,
            },
            url: function () {
                var id = this.get('id');

                return '/intranet/rest/lms/module_users/' + id + '/';
            }
        });

        /**
         * Get uncompleted module list with pagination
         */
        LMS.PaginatedCollection = Backbone.Collection.extend({
            page: 1,
            limit: 10, // Number of items per page

            initialize: function(models, options) {
                // Store the ID passed during initialization
                this.id = options.id || null;
            },

            // Dynamically construct the URL using the moduleId
            url: function() {
                if (!this.id) {
                    console.error("Module ID is not set!");
                    return '/intranet/rest/lms/';
                }

                return '/intranet/rest/lms/module_users_uncompleted/' + this.id + '/';
            },

            // Fetch data for a specific page
            fetchPage: function (page) {
                this.page = 1;
                this.page = page;
                return this.fetch({
                    data: {
                        limit: this.limit,
                        offset: (this.page - 1) * this.limit
                    },
                    reset: true, // Use reset to replace the collection instead of adding to it
                });
            },

            parse: function (response) {
                this.totalPages = Math.ceil(response.paging.total / this.limit);

                return response.data;
            }
        });

        LMS.PaginatedView = Backbone.View.extend({
            el: function() {
                return '#course-modal-table-nav_' + this.collection.id;
            },

            events: {
                'click .page-link': 'changePage'
            },

            initialize: function () {
                $('.js-step-number').text(lmsg('lms.course.steps', 1, 2));
                this.listenTo(this.collection, 'reset', this.render); // Re-render on collection reset
            },

            render: function () {
                if (this.collection.length <= 0)
                    return this;

                var modalSelector = $('#markAsCompleted_' + this.collection.id);
                var rowSelector = modalSelector.find(".js-mark-as-completed-row");

                // hide spinner and show table content
                $('#js-course-modal-table_spinner_' + this.collection.id).hide();
                modalSelector.find('.js-course-modal-table').removeClass('hidden');

                // here is the content page loaded we can use our table
                rowSelector.html('');

                // Render table for modal window
                this.collection.map(function(model) {
                    var row = new LMS.ManageModulesModalRow({model: model.attributes}).render();
                    rowSelector.append(row.el);
                });

                // skip pagination if numbers of the pages are >= 2
                if (this.collection.totalPages === 1)
                    return this;

                const paginationHtml = this.renderPagination();

                var modalContainerId = $('#js-course-modal-table-nav_' + this.collection.id);
                modalContainerId.html('<nav><ul class="pagination">' + paginationHtml + '</ul></nav>');

                this.trigger('finishedUserTableRenderEvent', {});

                // Add events to the pagination buttons as they are dynamic loaded
                modalContainerId
                    .off('click', '.page-link')
                    .on('click', '.page-link', this.changePage.bind(this));

                return this;
            },

            renderPagination: function () {
                const currentPage = this.collection.page;
                const totalPages = this.collection.totalPages || 1;

                var html = '';
                for (var i = 1; i <= totalPages; i++) {
                    html += '<li class="page-item' + (i === currentPage ? ' active' : '') + '"><button class="page-link" data-page="' + i + '">' + i + '</button></li>';
                }

                return html;
            },

            changePage: function (e) {
                const page = parseInt($(e.currentTarget).data('page'), 10);
                this.collection.fetchPage(page);
            }
        });

        /**
		 * Modules main container (View)
		 */
		LMS.ModulesContainerView = Backbone.View.extend({
			el: ".js-module-row",

			initialize: function () {
				_.bindAll(this, "appendModule", "reset");

				this.collection.bind("add", this.appendModule);
				this.collection.bind("reset", this.reset);
			},

			appendModule: function (moduleModel) {
				this.$el.append(new LMS.ModuleView({model: moduleModel}).render().el);
			},

			reset: function () {
				this.$el.empty();
				this.collection.each(function (moduleModel) {
					if (parseInt(moduleModel.get('assign_id'), 10) > 0) {
						LMS.selectedModulesCollection.push({id: moduleModel.get('assign_id')});
					}
					this.appendModule(moduleModel);
				}, this);
				LMS.browseModules.toggleCheckbox();
			}
		});

		/**
		 * Module single container (View)
		 */
		LMS.ModuleView = Backbone.View.extend({
			templateID: "#module",
			tagName: "tr",

			events: {
				"click .js-browse-module-add-remove": "addOrRemoveModule"
			},

			initialize: function () {
				_.bindAll(this, "render", "toggleCheckbox", "removeModule", "addModule");
				this.model.bind("change", this.toggleCheckbox);
			},

			toggleCheckbox: function () {
				var self = this;
				this.$el.find('.js-browse-module-add-remove')
					.prop('checked', this.model.get('assign_id') ? true : false)
					.attr('data-original-title', this.model.get('assign_id') ? self.model.get('remove_module') : self.model.get('assign_module'));
			},

			addOrRemoveModule: function () {
				var assign_id = parseInt(this.model.get('assign_id'), 10);
				if (assign_id > 0)
					this.removeModule(assign_id);
				else
					this.addModule();
				this.$el.find('.js-browse-module-add-remove').tooltip('hide').prop('disabled', true);
				return true;
			},

			removeModule: function (id) {
				var self = this;
				LMS.selectedModulesCollection.where({id: id})[0].destroy(
					{
						wait: true,
						success: function (model, response) {
							self.$el.find('.js-browse-module-add-remove').prop('disabled', false);
							self.model.set('assign_id', 0);
						},
						error: function (model, response) {
							self.$el.find('.js-browse-module-add-remove').prop('disabled', false);
							cla.showMessage(response.responseText, '', 'info');
						}
					});
				return false;
			},

			addModule: function () {
				var self = this;
				LMS.selectedModulesCollection.create({
					course_id: LMS.course_id,
					module_id: this.model.get('id')
				}, {
					wait: true,
					success: function (model, response) {
						self.$el.find('.js-browse-module-add-remove').prop('disabled', false);
						self.model.set('assign_id', model.id);
					},
					error: function (model, response) {
						self.$el.find('.js-browse-module-add-remove').prop('disabled', false);
						cla.showMessage(response.responseText, '', 'info');
					}
				});
				return false;
			},

			render: function () {
				this.$el.html($(this.templateID).tmpl(this.model.toJSON()));
				return this;
			}
		});

		/**
		 * Manage modules actions (View)
		 */
		LMS.ManageModules = Backbone.View.extend({
			el: "#course-manage-buttons",

			events: {
				"click #add-new-category": "toggleManageButtons"
			},

			initialize: function () {
				_.bindAll(this, "toggleManageButtons");
			},

			toggleManageButtons: function () {
				$('#course-manage-modules').toggle();
			}
		});

		/**
		 * Browse modules actions (View)
		 */
		LMS.BrowseModules = Backbone.View.extend({
			el: "#browse-module-table",
			select_all: lmsg('lms.course.add_all_modules'),
			unselect_all: lmsg('lms.course.add_remove_modules'),

			events: {
				"click #browse-select-all": "toggleSelectAll"
			},

			initialize: function () {
				_.bindAll(this, "toggleSelectAll");
			},

			toggleCheckbox: function () {
				var checked = false;
				var self = this;
				_.each($('.js-browse-module-add-remove'), function (e) {
					var checkbox = $(e);
					if (checkbox.prop('checked')) checked = true;
				});
				$('#browse-select-all').prop('checked', checked)
					.attr('data-original-title', checked ? self.unselect_all : self.select_all);
			},

			toggleSelectAll: function () {
				var self = this;
				var checked = $('#browse-select-all').prop('checked') ? true : false;
				_.each($('.js-browse-module-add-remove'), function (e) {
					var checkbox = $(e);
					if (!checkbox.prop('checked') && checked) checkbox.trigger('click');
					if (checkbox.prop('checked') && !checked) checkbox.trigger('click');
				});
				$('#browse-select-all').attr('data-original-title', checked ? self.unselect_all : self.select_all);
			}
		});

		/**
		 * Audit (Model)
		 */
		LMS.Audit = Backbone.Model.extend({
			url: function () {
				return "/intranet/rest/lms/course_audit/" + parseInt(LMS.course_id, 10);
			}
		});

		/**
		 * Audits (Collection)
		 */
		LMS.AuditListCollection = Backbone.Collection.extend({
			model: LMS.Audit,
			url: '/intranet/rest/lms/course_audit/' + parseInt(LMS.course_id, 10)
		});

		/**
		 * Audits main container (View)
		 */
		LMS.AuditContainerView = Backbone.View.extend({
			el: ".js-audit-row",

			initialize: function () {
				_.bindAll(this, "appendAudit", "reset");
				this.collection.bind("add", this.appendAudit);
				this.collection.bind("reset", this.reset);
			},

			appendAudit: function (auditModel) {
                this.$el.append(new LMS.AuditView({model: auditModel}).render().el);
			},

			reset: function () {
				this.$el.empty();
				if (this.collection.length > 0) $('#course-audit-table').show();
				this.collection.each(function (auditModel) {
					this.appendAudit(auditModel);
				}, this);
			}
		});

		/**
		 * Audit single container (View)
		 */
        LMS.AuditView = Backbone.View.extend({
			templateID: "#audit",
			tagName: "tr",

			initialize: function () {
				_.bindAll(this, "render");
			},

			render: function () {
				this.$el.html($(this.templateID).tmpl(this.model.toJSON()));
				return this;
			}
		});

        /**
         * Generate a single row for the table in modal window for the enrolled modules.
         */
        LMS.ManageModulesModalRow = Backbone.View.extend({
            templateID: "#modalStep1",
            tagName: "tr",

            initialize: function () {
                _.bindAll(this, "render");
            },

            render: function () {
                this.$el.html($(this.templateID).tmpl(this.model));
                return this;
            }
        });

		LMS.updateSortable = function () {
			$('.js-course-manage-module-row').each(function (index) {
				var assign_id = parseInt(this.id.replace('module-id-', ''), 10);
				LMS.courseModulesListCollection.where({assign_id: assign_id})[0].set('position', index + 1).save();
			});
		};

		LMS.sendReminder = function() {
			var choice = $("input[type='radio'][name='reminder-choice']:checked").val();
			var course_id = parseInt($(this).data('course-id'));

			$.ajax({
				type: "post",
				url: "/intranet/rest/lms/send_reminder/",
				data: JSON.stringify({ choice: choice, course_id: course_id }),
				contentType: 'application/json',
				success: function (d) {
					if (d.success) {
						cla.showMessage(lmsg('lms.course.reminder_sent'));
					}
				}
			})
		}
	}
});
