define(['jquery', 'cla_highcharts', 'moment', 'cla_locale'], function($, Highcharts, moment, locale)
{
    var FinanceChart = function(target, data) {
        this.target = target;
        this.data = data;
    }

    FinanceChart.prototype.Init = function() {
        var self = this;

        self.Show(this.target, this.data);
    }

    FinanceChart.prototype.Show = function(target, orgData) {
        var targetEl = document.querySelector('#' + this.target);

        if (!targetEl)
            return;

        // group organizations by currency so we can generate one chart per
        // currency. If the the currency is not defined set it as 'N/A'
        var groupedByCcy = {};
        for (var i = 0; i < orgData.length; i++) {
            var cur = orgData[i].currency;
            cur = cur.length > 0 ? cur.toUpperCase() : 'N/A';
            if (!groupedByCcy[cur]) groupedByCcy[cur] = [];
            groupedByCcy[cur].push(orgData[i]);
        }


        var metrics = [
            { key: 'net_revenue', name: lmsg('organizations.dashboard.net_sales'), color: 'var(--blue-500)' },
            { key: 'gross_revenue', name: lmsg('organizations.dashboard.gross_sales'), color: 'var(--indigo-500)' },
            { key: 'net_profit', name: lmsg('organizations.dashboard.net_profit'), color: 'var(--pink-500)' }
        ];

        //create a Chart for each currency used by organizations
        for (var currency in groupedByCcy) {
            if (!groupedByCcy.hasOwnProperty(currency)) continue;

            var orgData = groupedByCcy[currency];

            // Create a new container div per chart
            var containerId = 'chart-' + currency;
            var div= document.createElement('div');
            div.id = containerId;
            div.className = 'finance-chart';
            document.getElementById(target).appendChild(div);

            // Build series array
            var series = [];
            for (var j = 0; j < metrics.length; j++) {
                var metric    = metrics[j];
                var dataArray = [];
                for (var k = 0; k < orgData.length; k++) {
                    var value = {
                        y: orgData[k][metric.key] || 0,
                        symbol: orgData[k]['currencySymbol']
                    };
                    dataArray.push(value);
                }
                series.push({
                    name: metric.name,
                    color: metric.color,
                    data: dataArray
                });
            }


            if(currency === "NOT_SET")
                currency = lmsg('organizations.currencies.not_set');

            //Calculate the chart's height based upon the number of Organizations
            var categories = orgData.map(function (d) { return d.name; });
            var barHeight = 100;
            var chartHeight = Math.max((categories.length * barHeight) + 140, 250);

            // Render Highcharts
            var chart = new Highcharts.Chart({
                chart: {
                    renderTo: containerId,
                    type: 'bar',
                    height: chartHeight,
                    style: {
                        fontFamily: 'var(--theme-body-font-family)',
                        fontSize: '12px',
                    }
                },
                accessibility: {
                    enabled: true,
                    keyboardNavigation: { enabled: true }
                },
                title: { text:  currency },
                xAxis: {
                    categories: (function () {
                        var cats = [];
                        for (var i = 0; i < orgData.length; i++) {
                            cats.push({
                                id: orgData[i].id,
                                name: orgData[i].name,
                                as_of_date: orgData[i].as_of_date
                            });
                        }
                        return cats;
                    }()),
                    gridLineWidth: 1,
                    labels: {
                        style: {
                            fontSize: '.875rem',
                        },
                        useHTML: true,
                        step: 1,
                        formatter: function () {
                            var id            = this.value.id;
                            var rawDate       = this.value.as_of_date && this.value.as_of_date.date;
                            var date          = moment(rawDate);
                            var formattedDate = date.isValid() ? date.format('L') : 'N/A';

                            return '<a class="organization-graph-link mb-2 text-dark" href="/organizations/organization/' + id + '">' + this.value.name + '</a>' +
                                '<br><div>As of date ' + formattedDate + '</div>';
                        }
                    },
                    title: { text: null },
                },
                yAxis: {
                    min: 0,
                    title: {
                        text: 'Amount ('+ currency +')',
                        align: 'middle',
                    },
                },
                plotOptions: {
                    series: {
                        grouping: true,
                        pointPadding: 0.0,
                        groupPadding: 0.0,
                        pointWidth: 25,
                        dataLabels: {
                            enabled: false,
                            formatter: function () {
                                return currency + Highcharts.numberFormat(this.y, 0, '.', ',');
                            }
                        },
                    }
                },
                tooltip: {
                    headerFormat: "{point.key.name}<br/>",
                    pointFormat: "<strong>{series.name}</strong>: {point.symbol}{point.y}<br/>",
                    positioner: function (labelWidth, labelHeight, point) {
                        var chart = this.chart;
                        var xOffset = chart.plotLeft; // move right so it clears labels
                        var yOffset = point.plotY - labelHeight / 2;
                        return { x: xOffset, y: yOffset };
                    },
                },
                credits: { enabled: false },
                series: series
            });
        }
    }

    return FinanceChart;
})
