/**
 * Controls the taking of a survey
 *
 */
(function () {
    'use strict';

    angular.module('cla.survey').service('surveyResources', surveyResources);

    surveyResources.$inject = ['$resource', 'surveyInterceptor'];

    /**
     * @param $resource
     * @param surveyInterceptor
     * @constructor
     */
    function surveyResources($resource, surveyInterceptor) {
        const version = 2;
        let service = {
            version: version,
            endpoint: `/api/quiz/v${version}/`,
            resource: null,

            getSurvey: getSurvey,
            getAttempts: getAttempts,
            startSurvey: startSurvey,
            submitQuestion: submitQuestion,
            finishSurvey: finishSurvey,
            getResults: getResults,
            enableInterceptor: enableInterceptor,
            disableInterceptor: disableInterceptor
        };

        service.resource = resource();

        return service;

        /**
         * enable resource interceptor drop-in
         */
        function enableInterceptor() {
            service.server = service.resource;
            service.resource = surveyInterceptor;
        }

        /**
         * disable resource interceptor drop-in
         */
        function disableInterceptor() {
            service.resource = service.server || service.resource;
            service.server = null;
        }

        /**
         * gets json for the current survey only
         * @param survey_id
         * @returns {*}
         */
        function getSurvey(survey_id) {
            return service.resource.survey({id: survey_id}, {});
        }


        /***
         * starts the survey by creating or updating the users attempt, and returns json for all questions in current survey.
         * @param survey_id
         * @param anonymous
         * @returns {*}
         */
        function startSurvey(survey_id, anonymous = false) {
            let req = {}

            if (anonymous) {
                req = {anonymous: 1}
            }

            return service.resource.start({id: survey_id}, req);
        }

        /**
         * submits the current question and users chosen answer to the attempt endpoint
         * @param survey_id
         * @param question_id
         * @param {object} post_args
         */
        function submitQuestion(attempt_id, post_args) {
            return service.resource.submit({id: attempt_id}, post_args);
        }

        /**
         * submits the survey for completion and returns a results object.
         */
        function finishSurvey(attempt_id, anonymous = false) {
            let req = {complete: true};

            if (anonymous) {
                Object.assign(req, {anonymous: 1})
            }

            return service.resource.finish({attempt_id: attempt_id}, req);
        }

        /**
         * gets the results for a given survey id
         * @param survey_id
         * @param attempt_id
         * @returns {*}
         */
        function getResults(survey_id, attempt_id) {
            return service.resource.results({id: survey_id, attempt_id: attempt_id});
        }

        /**
         * get survey's attempts
         * @param survey_id
         * @returns {*}
         */
        function getAttempts(survey_id) {
            return service.resource.attempts({id: survey_id})
        }

        /**
         * returns the resource object
         */
        function resource() {
            return $resource('', {}, {
                survey: {method: 'get', url: service.endpoint + 'surveys/:id'},

                // get survey attempts
                attempts: {method: 'get', url: service.endpoint + 'surveys/:id/attempts'},

                // attempt to the survey
                start: {method: 'post', url: service.endpoint + 'surveys/:id/attempts'},

                // add answer to attempt
                submit: {method: 'post', url: service.endpoint + 'attempts/:id/answers'},
                finish: {method: 'put', url: service.endpoint + 'attempts/:attempt_id'},
                results: {method: 'get', url: service.endpoint + 'attempts/:id/results/:attempt_id'}
            });
        }
    }
}());
