/**
 * Controls the taking of a survey
 */
(function () {
    'use strict';

    angular.module('cla.survey').service('surveyTracker', surveyTracker);

    /**
     *
     * @constructor
     */
    function surveyTracker() {
        let service = {
            options: {                // tracker options
                showHelpText: false,  // show single or multiple answer help text
                shuffleOptions: false // shuffle question options
            },

            sections: {},          // all sections in this survey
            attempt: {},
            optionIndexes: {},      // dictionary of question option indexes
            question: 0,		    // current question index
            currentSection: {},    // current question object
            count: 0,

            timer: 0,			    // question timer
            chosen: [],			    // what the user has chosen
            loaded: false,		    // tracker initialised?
            highlightedIndex: null, // highlighted option index (for keyboard navigation and selection)

            tracker: [],            // track array for question tracking (question indices)
            trackpos: 0,            // current position of tracker index

            results: null,          // results objects
            can_retake: null,       // whether the survey can be attempted again
            attempt_id: null,       // the current attempt id

            anonymous: false,       // when survey is anonymous
            preview: false,       // when survey is in preview mode

            init: init,
            prev: prev,
            next: next,
            goto: goto,
            update: update,
            setOption: setOption,
            setTracker: setTracker,
            getComplete: getComplete,
            toggleOption: toggleOption,
            resetTracker: resetTracker,
            getIncomplete: getIncomplete,
            isOptionChosen: isOptionChosen,
            getChosenCount: getChosenCount,
            getOptionsCount: getOptionsCount,
            buildOptionIndexes: buildOptionIndexes,
            getCorrectOptionsCount: getCorrectOptionsCount,
            firstUnansweredQuestion: firstUnansweredQuestion,
            setAnonymous: setAnonymous
        };

        return service;

        /**
         * initialise tracker
         * @param {object[]} sections
         * @param {object[]} answers
         * @param {object}   [options]
         */
        function init(sections, answers, options, attempt) {
            // skip initialisation if we have already loaded the data
            if (service.loaded) {
                return;
            }

            reset();

            service.sections = angular.extend(service.sections, sections);
            service.options = Object.values(options);
            service.attempt = Object.assign(service.attempt, attempt);
            if (!service.preview)
            {
                service.answers = Object.values(answers);

            }


            // remove key length if exist
            if (service.sections.hasOwnProperty('length')) {
                delete service.sections.length
            }

            // set tracker
            resetTracker();

            // set prefilled choices if choices are given
            setAnswers(service.answers);

            // survey started, timer set time
            service.timer = Date.now();
            service.loaded = true;

            // set up question option indexes
            service.buildOptionIndexes();
        }

        /**
         * Set any preloaded user answers.
         *
         * Usually the result of continuing a suspended survey.
         *
         * @param {object} answers
         */
        function setAnswers(answers) {
            if (service.preview)
                return false;

            // from service.answers
            if (!answers && answers.prototype === Array) {
                return false;
            }

            let arrayWithAnswersIds = [];

            /**
             * create array with answer ids only for checkboxes and select fields,
             * strings are passed directly to model
             */
            answers.map(answer => {
                if (typeof answer.properties.response === 'object' && answer.properties.response !== null) {
                    Object.keys(answer.properties.response).map(response_id => arrayWithAnswersIds.push(response_id))
                }
            })

            // assign answer to the question object
            for (let section in service.sections) {
                for (let question_id in service.sections[section].questions) {
                    let id = service.sections[section].questions[question_id].id
                    let answer = answers.filter(answer => answer.question_id === id)

                    Object.assign(service.sections[section].questions[question_id], {answer: answer[0]})

                    for (let option_id in service.sections[section].questions[question_id].options) {
                        /**
                         * check is answer is correct one - and add answer property to option model
                         */
                        if (arrayWithAnswersIds.includes(option_id.toString())) {
                            Object.assign(service.sections[section].questions[question_id].options[option_id], {answer: true})
                        }
                    }
                }
            }

            service.question = 0;
        }

        /**
         * Count the number of sections a user has answered.
         *
         * @returns {number}
         */
        function getChosenCount() {
            let count = 0;

            for (let i in service.chosen) {
                if (service.chosen[i].submitted === true) {
                    count++;
                }
            }

            return count;
        }

        /**
         * Get the options count for a question.
         *
         * @returns {number}
         */
        function getOptionsCount(question_index) {
            var count = 0;
            var question = service.sections[question_index];

            for (let i in question.options) {
                if (question.options.hasOwnProperty(i)) {
                    count++;
                }
            }

            return count;
        }

        /**
         * Count the number of correct options for this question.
         *
         * @returns {number}
         */
        function getCorrectOptionsCount(question_index) {
            var question = service.sections[question_index];

            return question.correct_options_count;
        }

        /**
         * Get the first unanswered question for the current survey attempt.
         *
         * @returns {number}
         */
        function firstUnansweredQuestion() {
            for (let i = 0; i < service.sections.length; i++) {
                if (!service.chosen[i]) {
                    return i;
                }
            }
            return (service.sections.length - 1);
        }

        /**
         * Update the current question view.
         */
        function update() {
            // angular extend used to update the existing object rather than overwrite entirely
            service.currentSection = angular.extend(service.currentSection, service.sections[service.question]);

            // reset timer whenever there is an object update (change in question)
            service.timer = Date.now();
            service.highlightedIndex = null;
        }

        /**
         * Build the option index arrays for each question.
         *
         * This facilitates shuffling answer order, if enabled, and is used in
         * the template to iterate over the sections in that order.
         */
        function buildOptionIndexes() {
            var i, j, question, sections = service.sections;

            for (i = 0; i < sections.length; i++) {
                question = sections[i];

                // Build the dictionary of options per question
                if (!service.optionIndexes[question.id]) {
                    service.optionIndexes[question.id] = [];
                }

                for (j in question.options) {
                    if (question.options.hasOwnProperty(j)) {
                        service.optionIndexes[question.id].push(parseInt(j));
                    }
                }

                // Shuffle the options if configured
                if (service.options.shuffleOptions) {
                    service.optionIndexes[question.id] = _.shuffle(service.optionIndexes[question.id]);
                }
            }
        }

        /**
         * Set the selected option for the current question.
         *
         * @param {int[]} option_ids
         * @param {int} selected_time
         */
        function setOption(option_ids, selected_time) {
            service.chosen[question_id] = {
                options: [option_ids],
                time_taken: selected_time,
                submitted: false
            };
        }

        /**
         * Toggle the selection of an option for the current question.
         *
         * @param {int} question_id
         * @param {int} option_id
         * @param {int} selected_time
         */
        function toggleOption(question_id, option_id, selected_time) {
            if (!service.chosen[question_id]) {
                service.chosen[question_id] = {
                    options: {option_id},
                    question_id: question_id,
                    time_taken: selected_time,
                    submitted: true
                };
            }

            // add answer prop to chosen option
            Object.keys(service.currentSection.questions).map(key_question => {
                if (service.currentSection.questions[key_question].id === question_id) {
                    Object.keys(service.currentSection.questions[key_question].options).map(key_option => {

                        // prevent to select more than one option in radio btn, while user is moving between sections
                        if (service.currentSection.questions[key_question].options[key_option].hasOwnProperty('answer') && (service.currentSection.questions[key_question].type === '0' || service.currentSection.questions[key_question].type === '4')) {
                            delete service.currentSection.questions[key_question].options[key_option].answer
                        }

                        if (service.currentSection.questions[key_question].options[key_option].id === option_id) {

                            // add/remove answer prop to option object
                            if (!service.currentSection.questions[key_question].options[key_option].hasOwnProperty('answer')) {
                                Object.assign(service.currentSection.questions[key_question].options[key_option], {'answer': true})
                            } else {
                                delete service.currentSection.questions[key_question].options[key_option].answer
                            }

                            Object.keys(service.sections).map(sectionKey => {
                                if (service.sections[sectionKey].id === service.currentSection.id) {
                                    service.sections[sectionKey].id = service.currentSection.id
                                }
                            })

                        }
                    })
                }
            })
        }

        /**
         * Check whether the given option ID is chosen for a question index.
         *
         * @param {int} question_index - Index of the question to check
         * @param {int} option_id - Option ID to check
         * @returns {boolean}
         */
        function isOptionChosen(question_index, option_id) {
            var i, answer = service.chosen[question_index]

            if (answer) {
                for (i in answer.options) {
                    if (answer.options[i] == option_id) {
                        return true;
                    }
                }
            }

            return false;
        }

        /**
         * returns a list of sections that have been answered
         * @returns {Array}
         */
        function getComplete() {
            var complete = [];

            // get the keys of our chosen sections
            for (let i in service.chosen) {
                complete.push(i);
            }

            return complete;
        }

        /**
         * returns a list of sections not yet answered
         * @returns {Array}
         */
        function getIncomplete() {
            var incomplete = [];

            for (var i in service.sections) {
                if (!service.chosen[i]) {
                    incomplete.push(i);
                }
            }

            return incomplete;
        }

        /**
         * move on to the next question
         */
        function next() {
            track(++service.trackpos);
            service.update();
        }

        /**
         * move back to the previous question
         */
        function prev() {
            track(--service.trackpos);
            service.update();
        }

        /**
         * go to a specific question
         */
        function goto(question) {
            track(question);
            service.update();
        }

        /**
         * reset any initial vars back to their default states without breaking references.
         */
        function reset() {
            service.sections.length = 0;

            service.highlightedIndex = null;
            service.currentSection = {};
            service.chosen = [];

            // reset current question to 0
            service.question = 0;
        }

        /**
         * ensure question index cannot go outside of the bounds of available sections.
         */
        function clamp() {
            // clamp
            if (service.trackpos >= service.count - 1) {
                service.trackpos = service.count - 1;
            } else if (service.trackpos < 0) {
                service.trackpos = 0;
            }
        }

        /**
         * next and previous should use a track to navigate
         */
        function track(pos) {
            service.trackpos = pos;
            if (service.tracker.length > 0) {
                clamp();
                service.question = service.tracker[service.trackpos];
            }
        }

        /**
         * fill a track with indexes
         */
        function setTracker(array) {
            service.tracker = [];
            service.trackpos = 0;

            for (let i = 0; i < array.length; i++) {
                service.tracker.push(parseInt(array[i]));
            }

            service.count = service.tracker.length;
        }

        /**
         * reset tracker to question list
         */
        function resetTracker() {
            // set tracking array
            let tracking = Object.keys(service.sections);
            service.setTracker(tracking);
            service.question = 0;
            service.update();
        }

        function setAnonymous(is_anonymous) {
            service.anonymous = !!parseInt(is_anonymous)
        }
    }
}());
