/**
 * Navigation controls within a quiz
 *
 * @author Simon Willan <simon.willan@claromentis.com>
 */
(function () {
    'use strict';

    angular.module('cla.quiz')
        .directive('claQuizNav', quizNav);

    quizNav.$inject = ['quizTracker', 'quizActions'];

    /**
     *
     * @param quizTracker
     * @param quizActions
     * @constructor
     */
    function quizNav(quizTracker, quizActions) {
        return {
            bindToController: true,
            controller: controller,
            controllerAs: 'nav',
            replace: true,
            restrict: 'E',
            scope: {
                quizId: '@'
            },
            templateUrl: '/intranet/quiz/html/templates/quiz/nav.html'
        };

        function controller($scope, $location) {

            var $ctrl = this;

            this.test = 1;
            this.next = next;
            this.previous = previous;
            this.isPrevHidden = isPrevHidden;
            this.finish = finish;
            this.review = review;

            this.onNext = onNext;
            this.onPrevious = onPrevious;
            this.onFinish = onFinish;
            this.onSkip = onSkip;
            this.onGoto = onGoto;
            this.onReview = onReview;

            init();

            function init() {
                $scope.$on('quiz.finished', $ctrl.onFinish);
                $scope.$on('quiz.next', $ctrl.onNext);
                $scope.$on('quiz.previous', $ctrl.onPrevious);
                $scope.$on('quiz.skip', $ctrl.onSkip);
                $scope.$on('quiz.goto', $ctrl.onGoto);
                $scope.$on('quiz.review', $ctrl.onReview);
            }

            /**
             * Moves the question on to the next. Optionally saves the current state.
             *
             * @param {boolean} save Whether to save
             * @param {int} [skip_to] Question number to skip to
             */
            function next(save, skip_to) {
                var question = quizTracker.question;
                var submitted = quizTracker.chosen[question] ? !quizTracker.chosen[question].submitted : false;

                // sidetrack through the review function if we are on the final question.
                if (skip_to == null && quizTracker.trackpos === quizTracker.count - 1) {
                    review();
                    return;
                }

                if (save) {
                    quizActions.next($ctrl.quizId);
                }

                // we're skipping, lets remove the chosen answer
                if (!save && submitted) {
                    delete quizTracker.chosen[quizTracker.question];
                }

                // proceed to the next, or a given question
                if (skip_to == null) {
                    quizTracker.next();
                } else {
                    quizTracker.goto(skip_to);
                }

                // defocus all buttons on next to stop custom highlights showing via CSS
                $("button").blur();
            }

            /**
             * Moves the question back by one
             */
            function previous() {
                // Submit answer to this question
                quizActions.next($ctrl.quizId);

                // Navigate to previous question
                quizTracker.prev();

                // Remove focus for all buttons on next to stop custom highlights showing via CSS
                $("button").blur();
            }

            /**
             * Checks if the previous button should be hidden or not
             * @returns {boolean}
             */
            function isPrevHidden() {
                return (quizTracker.question == 0);
            }

            /**
             * finishes the current quiz and submits the last answered question to the endpoint
             */
            function finish() {
                quizActions.finish($ctrl.quizId, function () {
                    $location.path('/finish');
                });
            }

            /**
             * switch to the review page. save current progress
             */
            function review() {
                // move the question on, or stay where we are?
                var skip_to = quizTracker.question;
                if (quizTracker.chosen[quizTracker.question]) {
                    skip_to = quizTracker.firstUnansweredQuestion();
                }

                // save progress before going to review page
                $ctrl.next(true, skip_to);
                $location.path('/review');
            }

            /**
             * on next event
             */
            function onNext() {
                $ctrl.next(true);
            }

            /**
             * on previous event
             */
            function onPrevious() {
                $ctrl.previous();
            }

            /**
             * on finished event
             */
            function onFinish() {
                $ctrl.finish();
            }

            /**
             * on skip question event
             */
            function onSkip() {
                $ctrl.next();
            }

            /**
             * on skip to specific question event
             * @param e
             * @param args
             */
            function onGoto(e, args) {
                for (var i = 0; i < quizTracker.tracker.length; i++) {
                    if (quizTracker.tracker[i] == args.num) {
                        $ctrl.next(true, i);
                        break;
                    }
                }
            }

            /**
             * on review event. nav to review page
             */
            function onReview() {
                $ctrl.review();
            }
        }
    }

}());
