/**
 * Navigation controls within a survey
 *
 */
'use strict';

(function () {
    angular.module('cla.survey').directive('claSurveyNav', surveyNav);

    surveyNav.$inject = ['surveyTracker', 'surveyActions', '$window'];

    /**
     *
     * @param surveyTracker
     * @param surveyActions
     * @constructor
     */
    function surveyNav(surveyTracker, surveyActions, $window) {
        return {
            bindToController: true,
            controller: controller,
            controllerAs: 'nav',
            replace: true,
            restrict: 'E',
            scope: {
                surveyId: '@'
            },
            templateUrl: '/intranet/quiz/html/templates/survey/nav.html'
        };

        function controller($scope, $location) {
            let $ctrl = this;

            this.test = 1;
            this.next = next;
            this.previous = previous;
            this.isPrevHidden = isPrevHidden;
            this.finish = finish;
            this.review = review;

            this.onNext = onNext;
            this.onPrevious = onPrevious;
            this.onFinish = onFinish;
            this.onSkip = onSkip;
            this.onGoto = onGoto;
            this.onReview = onReview;

            init();

            function init() {
                $scope.$on('survey.finished', $ctrl.onFinish);
                $scope.$on('survey.next', $ctrl.onNext);
                $scope.$on('survey.previous', $ctrl.onPrevious);
                $scope.$on('survey.skip', $ctrl.onSkip);
                $scope.$on('survey.goto', $ctrl.onGoto);
                $scope.$on('survey.review', $ctrl.onReview);
                $scope.$on('survey.finish', $ctrl.finish);
            }

            /**
             * Moves the section on to the next. Optionally saves the current state.
             *
             * @param {boolean} save Whether to save
             * @param {int} [skip_to] Question number to skip to
             */
            function next(save, skip_to) {
                let question = surveyTracker.question;
                let submitted = surveyTracker.chosen[question] ? !surveyTracker.chosen[question].submitted : false;
                // sidetrack through the review function if we are on the final question.

                if (skip_to == null && surveyTracker.trackpos === surveyTracker.count - 1) {
                    review();

                    return;
                }

                // pass to next section
                if (save) {
                    surveyActions.next($ctrl.surveyId);
                }

                // we're skipping, lets remove the chosen answer
                if (!save && submitted) {
                    delete surveyTracker.chosen[surveyTracker.question];
                }

                // proceed to the next, or a given question
                if (skip_to == null) {
                    surveyTracker.next();
                } else {
                    surveyTracker.goto(skip_to);
                }

                // defocus all buttons on next to stop custom highlights showing via CSS
                $("button").blur();
                $window.scrollTo(0, 0);
            }

            /**
             * Moves the question back by one
             */
            function previous() {
                // Submit answer to this question
                surveyActions.next($ctrl.surveyId);

                // Navigate to previous question
                surveyTracker.prev();

                // Remove focus for all buttons on next to stop custom highlights showing via CSS
                $('button').blur();
                $window.scrollTo(0, 0);
            }

            /**
             * Checks if the previous button should be hidden or not
             * @returns {boolean}
             */
            function isPrevHidden() {
                return (surveyTracker.question == 0);
            }

            /**
             * finishes the current survey and submits the last answered question to the endpoint
             */
            function finish(event, callback) {
                var result = surveyActions.finish($ctrl.surveyId, function () {
                    $location.path('/finish');
                });

                callback(result);
            }

            /**
             * switch to the review page. save current progress
             */
            function review() {
                // move the question on, or stay where we are?
                var skip_to = surveyTracker.question;
                if (surveyTracker.chosen[surveyTracker.question]) {
                    skip_to = surveyTracker.firstUnansweredQuestion();
                }

                // save progress before going to review page
                $ctrl.next(true, skip_to);
                $location.path('/review');
            }

            /**
             * on next event
             */
            function onNext() {
                $ctrl.next(true);
            }

            /**
             * on previous event
             */
            function onPrevious() {
                $ctrl.previous();
            }

            /**
             * on finished event
             */
            function onFinish() {
                $ctrl.finish();
            }

            /**
             * on skip question event
             */
            function onSkip() {
                $ctrl.next();
            }

            /**
             * on skip to specific question event
             * @param e
             * @param args
             */
            function onGoto(e, args) {
                for (var i = 0; i < surveyTracker.tracker.length; i++) {
                    if (surveyTracker.tracker[i] == args.num) {
                        $ctrl.next(true, i);
                        break;
                    }
                }
            }

            /**
             * on review event. nav to review page
             */
            function onReview() {
                $ctrl.review();
            }
        }
    }
}());
