(function () {
    'use strict';

    angular.module('cla.poll')
        .service('pollResources', pollResources);

    pollResources.$inject = ['$resource', 'pollInterceptor'];

    /**
     * @param $resource
     * @param pollInterceptor
     * @constructor
     */
    function pollResources($resource, pollInterceptor) {
        var version = 2;
        var service = {
            version: version,
            endpoint: '/api/quiz/v' + version + '/polls/',
            attempt_endpoint: '/api/quiz/v' + version + '/attempts/',
            resource: null,

            getPoll: getPoll,
            startPoll: startPoll,
            submitQuestion: submitQuestion,
            finishPoll: finishPoll,
            getResults: getResults,

            enableInterceptor: enableInterceptor,
            disableInterceptor: disableInterceptor
        };

        service.resource = resource();
        return service;

        /**
         * enable resource interceptor drop-in
         */
        function enableInterceptor() {
            service.server = service.resource;
            service.resource = pollInterceptor;
        }

        /**
         * disable resource interceptor drop-in
         */
        function disableInterceptor() {
            service.resource = service.server || service.resource;
            service.server = null;
        }

        /**
         * gets json for the current poll only
         * @param poll_id
         * @returns {*}
         */
        function getPoll(poll_id) {
            return service.resource.poll({id: poll_id}, {});
        }

        /**
         * starts the poll by creating or updating the users attempt,
         * and returns json for all questions in current poll.
         * @param poll_id
         * @param post_args
         * @returns {*}
         */
        function startPoll(poll_id, post_args) {
            return service.resource.start({id: poll_id}, post_args);
        }

        /**
         * submits the current question and users chosen answer to the attempt endpoint
         * @param attempt_id
         * @param {object} post_args
         */
        function submitQuestion(attempt_id, post_args) {
            return service.resource.submit({id: attempt_id}, post_args);
        }

        /**
         * submits the poll for completion and returns a results object.
         * @param {int} attempt_id
         * @param {object} post_args
         * @returns {*}
         */
        function finishPoll(attempt_id, post_args) {
            return service.resource.finish({id: attempt_id}, post_args);
        }

        /**
         * gets the results for a given poll id
         * @param poll_id
         * @returns {*}
         */
        function getResults(poll_id) {
            return service.resource.results({id: poll_id}, {});
        }

        /**
         * returns the resource object
         */
        function resource() {
            return $resource('', {}, {
                poll: {method: 'get', url: service.endpoint + ':id'},
                start: {method: 'post', url: service.endpoint + ':id/attempts'},
                submit: {method: 'post', url: service.attempt_endpoint + ':id/answers'},
                finish: {method: 'put', url: service.attempt_endpoint + ':id'},
                results: {method: 'get', url: service.endpoint + ':id/results'}
            });
        }
    }

}());
