/**
 * Controls the taking of a quiz
 *
 * @author Simon Willan <simon.willan@claromentis.com>
 */
(function () {
    'use strict';

    angular.module('cla.quiz')
        .service('quizResources', quizResources);

    quizResources.$inject = ['$resource', 'quizInterceptor'];

    /**
     * @param $resource
     * @param quizInterceptor
     * @constructor
     */
    function quizResources($resource, quizInterceptor) {
        var version = 1;
        var service = {
            version: version,
            endpoint: '/api/quiz/v' + version + '/',
            resource: null,

            getQuiz: getQuiz,
            startQuiz: startQuiz,
            submitQuestion: submitQuestion,
            finishQuiz: finishQuiz,
            getResults: getResults,

            enableInterceptor: enableInterceptor,
            disableInterceptor: disableInterceptor
        };

        service.resource = resource();
        return service;

        /**
         * enable resource interceptor drop-in
         */
        function enableInterceptor() {
            service.server = service.resource;
            service.resource = quizInterceptor;
        }

        /**
         * disable resource interceptor drop-in
         */
        function disableInterceptor() {
            service.resource = service.server || service.resource;
            service.server = null;
        }

        /**
         * gets json for the current quiz only
         * @param quiz_id
         * @returns {*}
         */
        function getQuiz(quiz_id) {
            return service.resource.quiz({id: quiz_id}, {});
        }

        /**
         * starts the quiz by creating or updating the users attempt,
         * and returns json for all questions in current quiz.
         * @param quiz_id
         * @returns {*}
         */
        function startQuiz(quiz_id) {
            return service.resource.start({id: quiz_id}, {});
        }

        /**
         * submits the current question and users chosen answer to the attempt endpoint
         * @param quiz_id
         * @param question_id
         * @param {object} post_args
         */
        function submitQuestion(quiz_id, question_id, post_args) {
            return service.resource.submit({id: quiz_id, qid: question_id}, post_args);
        }

        /**
         * submits the quiz for completion and returns a results object.
         * @param {int} quiz_id
         * @param {object} post_args
         * @returns {*}
         */
        function finishQuiz(quiz_id, post_args) {
            return service.resource.finish({id: quiz_id}, post_args);
        }

        /**
         * gets the results for a given quiz id
         * @param quiz_id
         * @param attempt_id
         * @returns {*}
         */
        function getResults(quiz_id, attempt_id) {
            return service.resource.results({id: quiz_id, attempt: attempt_id});
        }

        /**
         * returns the resource object
         */
        function resource() {
            return $resource('', {}, {
                quiz: {method: 'get', url: service.endpoint + ':id'},
                start: {method: 'post', url: service.endpoint + ':id/start'},
                submit: {method: 'post', url: service.endpoint + ':id/questions/:qid/answer'},
                finish: {method: 'post', url: service.endpoint + ':id/finish'},
                results: {method: 'get', url: service.endpoint + ':id/results/:attempt'}
            });
        }
    }

}());
