<?php
namespace Claromentis\RssReader\UI;

use Claromentis\Core\Application;
use Claromentis\Core\Component\ComponentInterface;
use Claromentis\Core\Component\Exception\ComponentRuntimeException;
use Claromentis\Core\Component\OptionsInterface;
use Claromentis\Core\Component\TemplaterTrait;
use Claromentis\RssReader\Exception\SubscriptionsException;
use Claromentis\RssReader\UserSubscriptions;
use Claromentis\RssReader\View\CustomizableView;
use Claromentis\RssReader\View\ReadonlyView;
use ClaText;

/**
 * RssReader component for Pages
 */
class PagesRssReader implements ComponentInterface
{
	use TemplaterTrait;

	/**
	 * Returns information about supported options for this component as array
	 *
	 * array(
	 *   'option_name' => ['type' => ...,
	 *                     'default' => ...,
	 *                     'title' => ...,
	 *                    ],
	 *   'other_option' => ...
	 * )
	 *
	 * @return array
	 */
	public function GetOptions()
	{
		return [
			'title' => [
				'title' => lmsg('rssreader.component.options.title'),
				'type' => 'string',
				'default' => 'RSS Reader',
			],
			'metakey' => [
				'type' => 'string',
				'input' => 'select_callback',
				'title' => lmsg('rssreader.component.options.metadata_key'),
				'callback' => [$this, 'MetakeySelectCallback']
			],
			'feed' => [
				'title' => lmsg('rssreader.component.options.url'),
				'type' => 'string',
				'input' => 'textarea',
				'width' => 'medium'
			],
			'display' => [
				'type' => 'int',
				'title' => lmsg('rssreader.component.options.num_items'),
				'default' => 5,
				'min' => 1, 'max' => 100
			],
			'summary_length' => [
				'type' => 'int',
				'title' => lmsg('rssreader.component.options.summary_length'),
				'default' => 0,
				'min' => 0,
			],
		];
	}

	public function MetakeySelectCallback()
	{
		$result = ['' => __('-- None --')];

		$mds = new \MetaDataSet();
		$mds->LoadByType(METADATA_SET_USER);
		$keys = $mds->GetMetaKeys();

		foreach ($keys as $key)
		{
			$field = $mds->GetMetaFieldByName($key);

			// Find any repeatable string fields
			if (($field->IsRepeatable()) &&
				(in_array($field->GetMetaType(), [META_TYPE_DEFAULT, META_TYPE_TEXT])))
			{
				$result[$field->name] = $field->title;
			}
		}

		return $result;
	}

	/**
	 * Render this component with the specified options
	 *
	 * @param string $id_string
	 * @param OptionsInterface $options
	 * @param Application $app
	 *
	 * @return string
	 */
	public function ShowBody($id_string, OptionsInterface $options, Application $app)
	{
		$attr_urls = [];
		$feeds_str = $options->Get('feed');
		if ($feeds_str != '')
		{
			$attr_urls = explode("\n", $feeds_str);
		}

		$user_urls = [];
		$metakey = $options->Get('metakey');
		if ($metakey)
		{
			try
			{
				$user_feeds = new UserSubscriptions($metakey);
				$user_urls = $user_feeds->GetUrls();
			} catch (SubscriptionsException $e)
			{
				throw new ComponentRuntimeException($e->getMessage(), $e->getCode(), $e);
			}
		}

		if (!$metakey)
		{
			$view = new ReadonlyView();
		} else
		{
			$attributes = $options->GetAll();
			$view = new CustomizableView($attr_urls, $user_urls, $metakey, $attributes);
			//if ($options->Get('title') == '')
			//	$view->SetDisplayEditLink(true);
		}
		$view->SetSummaryLength($options->Get('summary_length'));

		/** @var \SimplePie $feed_reader */
		$feed_reader = $app['rssreader.feed_reader'];
		$feed_reader->set_feed_url(array_merge($attr_urls, $user_urls));
		$feed_reader->init();

		$max_items = $options->Get('display');
		$feed_items = $feed_reader->get_items(0, $max_items);

		$view->SetFeedItems($feed_items);

		return $view->Show();
	}

	/**
	 * Render component header with the specified options.
	 * If null or empty string is returned, header is not displayed.
	 *
	 * @param string           $id_string
	 * @param OptionsInterface $options
	 * @param Application      $app
	 *
	 * @return string
	 */
	public function ShowHeader($id_string, OptionsInterface $options, Application $app)
	{
		$title = cla_htmlsafe(ClaText::ProcessAvailableLocalisation((string) $options->Get('title')));
		$edit_link = ($options->Get('metakey') ?
			'<span class="float-right"><a href="#" class="js-nf-edit-rss">' . cla_htmlsafe(lmsg('common.edit')) . '</a></span>' :
			'');

		$ret = '';
		$ret .= !empty($edit_link) ? $edit_link : '';
		$ret .= !empty($title)     ? $title     : '';

		return $ret;
	}

	/**
	 * Define any minimum or maximum size constraints that this component has.
	 * Widths are measured in 12ths of the page as with Bootstrap.
	 * Heights are measured in multiples of the grid row height (around 47 pixels currently?)
	 *
	 * @return array should contain any combination of min_width, max_width, min_height and max_height.
	 */
	public function GetSizeConstraints()
	{
		return [
			'min_width' => 3,
			'min_height' =>  2
		];
	}

	/**
	 * Returns CSS class name to be set on component tile when it's displayed.
	 * This class then can be used to change the display style.
	 *
	 * Recommended class name is 'tile-' followed by full component code.
	 *
	 * It also can be empty.
	 *
	 * @return string
	 */
	public function GetCssClass()
	{
		return 'tile-rss-reader-component';
	}

	/**
	 * Returns associative array with description of this component to be displayed for users in the
	 * components list.
	 *
	 * Result array has these keys:
	 *   title       - Localized component title, up to 40 characters
	 *   description - A paragraph-size plain text description of the component, without linebreaks or HTML
	 *   image       - Image URL
	 *   application - One-word lowercase application CODE (same as folder name and admin panel code)
	 *
	 * Some values may be missing - reasonable defaults will be used.
	 *
	 * @return array
	 */
	public function GetCoverInfo()
	{
		return [
			'title'       => lmsg('rssreader.component.cover_info.title'),
			'description' => lmsg('rssreader.component.cover_info.desc'),
			'application' => 'rssreader',
			'icon_svg'    => file_get_contents('../rssreader/icons/rss-icon.svg'),
			'categories'  => ['news_announcements']
		];
	}
}
