<?php
namespace Claromentis\RssReader;

use Claromentis\Core\Application;
use Claromentis\Core\ControllerCollection;
use Claromentis\Core\REST\RestServiceInterface;
use Pimple\Container;
use Pimple\ServiceProviderInterface;

/**
 *
 * @author Alexander Polyanskikh
 */
class Plugin implements ServiceProviderInterface, RestServiceInterface
{
	const BASE_URL = '/rssreader';

	/**
	 * Registers services on the given container.
	 *
	 * This method should only be used to configure services and parameters.
	 * It should not get services.
	 *
	 * @param Container $app A container instance
	 */
	public function register(Container $app)
	{
		$app['localization.domain.rssreader'] = function ($app)
		{
			return $app['localization.domain_from_files_factory']('rssreader');
		};

		$app['templater.ui.rssreader.feed'] = function ()
		{
			return new UI\Feed();
		};

		$app['pages.component.rssreader'] = function()
		{
			return new UI\PagesRssReader();
		};

		$app['rssreader.feed_reader'] = $app->factory(function ($app)
		{
			$rss = new \SimplePie();
			$rss->set_curl_options([CURLOPT_SSL_VERIFYPEER => false]);
			$rss->set_useragent('ClaromentisRSSReader/v3.1');
			$rss->set_cache_location($app['rssreader.cache_dir']);
			return $rss;
		});

		$app['rssreader.cache_dir'] = function ($app)
		{
			$dir = $app->config['DATA_DIR'] . '/rss_cache'; // . $this->data['feed_id'];

			if (!is_dir($dir))
			{
				mkdir($dir, 0777);
			}

			return $dir;
		};
	}

	/**
	 * Returns REST routes for the application.
	 *
	 * This method should be fast and should only return new static routes, preferably without
	 * using any other services, especially database.
	 *
	 * It should return an array in form of array($prefix => $closure), where $prefix is a string starting
	 * from slash, without trailing slash and $closure is a closure that takes an instance of
	 * ControllerCollection and registers all route handlers into it.
	 *
	 * Each route handler should be defined as a string - controller class and method name, such as
	 * $routes->get('/home', "MyApp\MyAppMainControler::OnHome")
	 *
	 * Make sure to secure all routes either by defining default security such as $collection->secure('rest', 'user');
	 * or for each route $collection->get(....)->secure('rest', 'guest');
	 * Note, default security works only for routes defined _after_ it's set, so put it to the top of the closure
	 *
	 * Example:
	 *  return array(
	 *     '/core/v0' => '/core/v1', // defining default version
	 *     '/core/v1' => function (\Claromentis\Core\ControllerCollection $routes)
	 *     {
	 *          $routes->get('/login', 'RestCore::OnLogin')->secure('rest', 'user');
	 *          $routes->get('/aggregation/{code}', 'RestCore::GetAggregation')->secure('rest', 'none');
	 *     }
	 *  );
	 *
	 * @param \Claromentis\Core\Application $app An Application instance
	 *
	 * @return array
	 */
	public function GetRestRoutes(Application $app)
	{
		return array(
			self::BASE_URL . '/v0' => self::BASE_URL . '/v1',
			self::BASE_URL . '/v1' => function (ControllerCollection $routes)
			{
				$routes->secure('rest', 'user');
				$routes->post('/', Rest\RestV1::class . '::PostAddFeed');
				$routes->put('/{id}/', Rest\RestV1::class . '::PutEditFeed');
				$routes->delete('/{id}/', Rest\RestV1::class . '::DeleteFeed');
			}
		);
	}
}
