<?php
namespace Claromentis\RssReader\Rest;

use Claromentis\Core\Application;
use Claromentis\Core\Component\ValidatingOptions;
use Claromentis\Core\Http\JsonPrettyResponse;
use Claromentis\Core\Util\ArrayUtil;
use Claromentis\RssReader\Exception\AlreadySubscribedException;
use Claromentis\RssReader\Exception\InvalidUrlException;
use Claromentis\RssReader\UI\PagesRssReader;
use Claromentis\RssReader\UserSubscriptions;
use Psr\Http\Message\ServerRequestInterface;

/**
 * REST handler for user-editable RSS reader
 */
class RestV1
{
	/**
	 * @param $attributes
	 *
	 * @return string Feed rendered in HTML
	 */
	protected function GetFeedHtml($attributes)
	{
		if (!is_array($attributes))
		{
			$attributes = array();
		}

		require_once '../common/templater.php';
		global $g_application;
		/** @var PagesRssReader $component */
		$component = $g_application['pages.component.rssreader'];

		$id_string = isset($attributes['id']) ? $attributes['id'] : uniqid();
		$options_info = $component->GetOptions();
		$options = ValidatingOptions::CreateFromSerialized($options_info, $attributes);

		return $component->ShowBody($id_string, $options, $g_application);

		//$view = new CustomizableView($id_string, $attr_urls, $user_urls, $metakey);
		//$feed = new Feed();
		//return $feed->ShowFeeds($attributes, $g_application);
	}

	/**
	 * Responds to an rssReader POST request to add a feed url
	 *
	 * @param Application $app
	 * @param ServerRequestInterface $request
	 *
	 * @return JsonPrettyResponse
	 */
	public function PostAddFeed(Application $app, ServerRequestInterface $request)
	{
		/** @var \RestFormat $formatter */
		$formatter = $app['rest.formatter'];

		$request_data = $formatter->GetJson($request);

		$url_value = trim(ArrayUtil::getScalarValue($request_data, 'rssfeed_url', ''));
		$metakey = ArrayUtil::getScalarValue($request_data, 'metakey', '');

		$user_subscriptions = new UserSubscriptions($metakey);

		$user_subscriptions->ValidateUrl($url_value);

		$this->TryLoadFeed($url_value);

		$user_subscriptions->AddUrl($url_value);
		$values = $user_subscriptions->GetUrls();
		$id = array_search($url_value, $values);

		$attributes = ArrayUtil::getArrayValue($request_data, 'attributes');

		$result = array
		(
			'error' => false,
			'url' => $url_value,
			'id' => (int)$id,
			'feeds' => $this->GetFeedHtml($attributes)
		);

		return new JsonPrettyResponse($result);
	}

	/**
	 * Responds to an rssReader PUT request to edit a feed url
	 *
	 * @param Application $app
	 * @param ServerRequestInterface $request
	 *
	 * @return JsonPrettyResponse
	 * @throws \RestExNotFound
	 */
	public function PutEditFeed(Application $app, ServerRequestInterface $request)
	{
		/** @var \RestFormat $formatter */
		$formatter = $app['rest.formatter'];

		$request_data = $formatter->GetJson($request);

		$url_value = trim(ArrayUtil::getScalarValue($request_data, 'rssfeed_url', ''));
		$metakey = ArrayUtil::getScalarValue($request_data, 'metakey', '');
		$edit_id = $request->getAttribute('id', 0);
		if (!$edit_id)
			throw new \RestExNotFound("Empty element id");

		$user_subscriptions = new UserSubscriptions($metakey);

		try {
			$user_subscriptions->ValidateUrl($url_value);
		} catch (AlreadySubscribedException $e)
		{
			return new JsonPrettyResponse(['error' => false]);
		}

		$this->TryLoadFeed($url_value);

		$user_subscriptions->ReplaceUrl($edit_id, $url_value);

		$attributes = ArrayUtil::getArrayValue($request_data, 'attributes');

		$result = array
		(
			'error' => false,
			'url' => $url_value,
			'id' => (int)$edit_id,
			'feeds' => $this->GetFeedHtml($attributes)
		);

		return new JsonPrettyResponse($result);
	}

	/**
	 * Responds to an rssReader DELETE request
	 *
	 * @param Application $app
	 * @param ServerRequestInterface $request
	 *
	 * @return JsonPrettyResponse
	 * @throws \RestExNotFound
	 */
	public function DeleteFeed(Application $app, ServerRequestInterface $request)
	{
		$edit_id = $request->getAttribute('id', 0);

		/** @var \RestFormat $formatter */
		$formatter = $app['rest.formatter'];

		$request_data = $formatter->GetJson($request);
		$metakey = ArrayUtil::getScalarValue($request_data, 'metakey', '');

		$user_subscriptions = new UserSubscriptions($metakey);
		$urls = $user_subscriptions->GetUrls();

		if (!isset($urls[$edit_id]))
			throw new \RestExNotFound("Feed URL with such id doesn't exist");

		$user_subscriptions->DeleteUrl($urls[$edit_id]);

		$attributes = ArrayUtil::getArrayValue($request_data, 'attributes');

		$data = array
		(
			'error' => false,
			'id' => (int)$edit_id,
			'feeds' => $this->GetFeedHtml($attributes)
		);

		return new JsonPrettyResponse($data);
	}


	/**
	 * Attempt to load and parse the URL
	 *
	 * @param string $url
	 * @throws InvalidUrlException
	 */
	private function TryLoadFeed($url)
	{
		//Check the RSS feed itself is valid - Magpie will quite effectively cache the information for this.
		$rss = new \SimplePie();
		$rss->set_curl_options([CURLOPT_SSL_VERIFYPEER => false]);
		$rss->enable_cache(false);
		$rss->set_feed_url($url);
		$valid = $rss->init();

		unset($rss);
		if (!$valid)
			throw new InvalidUrlException(lmsg('rssreader.error_rss_feed'));
	}

}