<?php
namespace Claromentis\RssReader;

use Claromentis\RssReader\Exception\AlreadySubscribedException;
use Claromentis\RssReader\Exception\InvalidUrlException;
use Claromentis\RssReader\Exception\MetadataTypeException;

/**
 * Stores URLs for individual user preferences
 */
class UserSubscriptions
{
	protected $metadata;
	protected $metadata_key;

	public function __construct($metadata_key)
	{
		$this->metadata_key = $metadata_key;
		$this->CheckMetakey($this->metadata_key);
		$user = \AuthUser::I()->GetUser();
		$this->metadata = $user->GetMetaDataObj();
	}

	/**
	 * Returns array of strings - URLs for RSS feeds
	 */
	public function GetUrls()
	{
		$urls = $this->metadata->GetValue($this->metadata_key);
		// $default = $this->metadata->GetPrototype($this->metadata_key)->GetDefaultValue();
		return $urls;
	}

	/**
	 * Check URL format and ensure it's a new one - not already subscribed
	 *
	 * @param string $url
	 */
	public function ValidateUrl($url)
	{
		$url = trim($url);

		if ($url == '')
			throw new InvalidUrlException(lmsg('rssreader.error_empty_url'));

		//Is this a valid URL?
		if (!preg_match('|^http(s)?://[a-z0-9-]+(.[a-z0-9-]+)*(:[0-9]+)?(/.*)?$|i', $url))
			throw new InvalidUrlException(__("The specified URL doesn't seem to have the correct format"));

		$urls = $this->GetUrls();

		if (in_array($url, $urls))
			throw new AlreadySubscribedException(lmsg('rssreader.already_subscribed'));
	}

	public function AddUrl($url)
	{
		$this->ValidateUrl($url);

		$this->metadata->SetValue($this->metadata_key, $url); // for repeatable metadata, this ADDS an element
		$this->metadata->Save();
	}

	public function DeleteUrl($url)
	{
		$urls = $this->metadata->GetValue($this->metadata_key);
		$md_value_id = array_search($url, $urls);
		if ($md_value_id)
		{
			$this->metadata->DeleteValue($this->metadata_key, $md_value_id);
			$this->metadata->Save();
		}
	}

	public function ReplaceUrl($item_id, $url)
	{
		$this->ValidateUrl($url);

		$this->metadata->SetValue($this->metadata_key, $url, $item_id);
		$this->metadata->Save();
	}

	protected function CheckMetakey($metakey)
	{
		$meta_field_factory = \MetaFieldFactory::I();
		$md_prototype = $meta_field_factory->GetPrototypeByKey($metakey);

		if (!$md_prototype) //If object isnt instantiated loading failed and thus key does not exist
			throw new MetadataTypeException($meta_field_factory->GetErrMsg());

		if (!$md_prototype->IsRepeatable())
			throw new MetadataTypeException(lmsg('rssreader.metadata_not_repeatable'));
		if ($md_prototype->GetMetaType() != META_TYPE_DEFAULT)
			throw new MetadataTypeException(lmsg('rssreader.metakey_incorrect_type', $metakey));

		$metadata_set = new \MetaDataSet();
		$metadata_set->LoadByType(METADATA_SET_USER);
		if (!in_array($metakey, $metadata_set->GetMetaKeys(), true))
			throw new MetadataTypeException(lmsg('rssreader.metakey_incorrect_set', $metakey));
	}
}