<?php
namespace Claromentis\SocialConnect;

use Claromentis\Core\DAL;
use Claromentis\SocialConnect\Exception\AccountNotFoundException;
use Claromentis\SocialConnect\Exception\LinkedOtherUserException;
use Claromentis\SocialConnect\Exception\LogicException;

/**
 *
 * @author Alexander Polyanskikh
 */
class AccountsRepository
{
	/**
	 * @var \ObjectsStorage
	 */
	protected $objects_storage;

	public function __construct($objects_storage)
	{
		$this->objects_storage = $objects_storage;
	}

	/**
	 * Find a social account or throw an exception if account is not found
	 *
	 * @param $provider
	 * @param $remote_uid
	 *
	 * @return SocialAccount
	 *
	 * @throws AccountNotFoundException
	 * @throws LogicException if more than one account exists
	 */
	public function Find($provider, $remote_uid)
	{
		$accounts = $this->objects_storage->GetMultiple(
			new SocialAccount(),
			new DAL\QueryPart('provider eq:str:prov AND remote_uid eq:str:uid', $provider, $remote_uid)
		);
		if (count($accounts) == 0)
			throw new AccountNotFoundException("Account not found");
		if (count($accounts) > 1)
			throw new LogicException(__("Multiple accounts exist with the same Provider and ID"));

		reset($accounts);
		return current($accounts);
	}

	/**
	 * Add a new social account. Throws a LogicException on error
	 *
	 * @param SocialAccount $account
	 *
	 * @throws LogicException
	 */
	public function Add(SocialAccount $account)
	{
		// check for unique provider+id
		try {
			$acc = $this->Find($account->provider, $account->remote_uid);
			if ($acc->user_id == $account->user_id)
				$this->Delete($acc);
			else
				throw new LinkedOtherUserException(__("This account is associated with another user"));
		} catch (AccountNotFoundException $e)
		{
			// it's ok, we're expecting this
		}

		$account->Save();
	}

	public function Delete(SocialAccount $account)
	{
		$account->Delete();
	}

	/**
	 * Returns list of social account for the specified user.
	 * If $provider is given, only returns account(s) with that provider.
	 *
	 * @param int $user_id
	 * @param string|null $provider
	 *
	 * @return SocialAccount[]
	 */
	public function GetForUser($user_id, $provider = null)
	{
		$accounts = $this->objects_storage->GetMultiple(
			new SocialAccount(),
			new DAL\QueryPart('user_id=int:user_id' . ($provider !== null ? ' AND provider eq:str:prov' : ''), $user_id, $provider)
		);
		/** @noinspection PhpIncompatibleReturnTypeInspection */
		return $accounts;
	}

    /**
     * Creates and returns a new social account using the users email address as opposed to the user's id.
     * Null is returned if there isn't a matching email in Claromentis.
     *
     * @param String $email
     * @param String $provider
     * @param String $user_id_str
     *
     * @return SocialAccount | null
     */
    public function CreateFromEmail($email, $provider, $user_id_str)
    {
        require_once("../common/user_functions.php");

        $social_account = null;

        $user_id = get_user_id_by_email($email);
        $user_id = reset($user_id);

        if ($user_id)
        {
            $social_account = SocialAccount::CreateNew($user_id, $provider, $user_id_str);

            try
            {
                \Claromentis\Core\Services::I()->GetDb()->DisableTokenCheck();
                $this->Add($social_account);
                \Claromentis\Core\Services::I()->GetDb()->EnableTokenCheck();
            } catch (\Exception $e)
            {
                /* Just returning null should be fine. */
            }
        }

        return $social_account;
    }
}
