<?php
namespace Claromentis\SocialConnect\Controller;

use Claromentis\Core\Application;
use Claromentis\Core\Http\gpc;
use Claromentis\Core\Http\RedirectResponse;
use Claromentis\Core\Http\TemplaterCallResponse;
use Opauth\Opauth\Opauth;
use Opauth\Opauth\OpauthException;
use Psr\Http\Message\ServerRequestInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Social connect controller that performs authentication against remote service, saves the
 * result into session and redirects to other URL that can read the session and perform the required
 * steps
 *
 * @author Alexander Polyanskikh
 */
class AuthenticateController
{
	public function Start(Application $app, ServerRequestInterface $request)
	{
		$config = $app['socialconnect.opauth_config'];

		if (gpc::getvar($request, 'popup'))
		{
			if (isset($config['Strategy']['Facebook']))
				$config['Strategy']['Facebook']['display'] = 'popup';
		}

		$opauth = new Opauth($config);
		try {
			\ErrorReporter::SetReportingLow();
			$response = $opauth->run();
			\ErrorReporter::SetReportingStandard();

			if ($response == null)
			{
				$headers = [];
				$status = Response::HTTP_OK;

				foreach (headers_list() as $h)
				{
					list($name, $value) = explode(':', $h, 2);
					// If OpAuth has already sent the location header, don't include that in the response object (BT-4892)
					if (strtolower($name) === 'location')
						$status = Response::HTTP_SEE_OTHER;
					else
						$headers[$name] = trim($value);
				}

				return new Response('Redirect', $status, $headers);
			}
			//echo "Authed as " . $response->name . " with uid" . $response->uid;
			//var_dump($response);

			$session = $app->session;
			$s_data = $session->get('socialconnect', []);
			$s_data['result'] = [
				'success' => true,
				'provider' => $response->provider,
				'uid' => $response->uid,
				'info' => $response->info,
				'credentials' => $response->credentials,
				'raw' => $response->raw
			];
			$session->set('socialconnect', $s_data);

			$provider = $response->provider;
		} catch (OpauthException $e)
		{
			//echo "Authentication error: " . $e->getMessage();
			//var_dump($e);
			$path = explode('/', gpc::getRequestPath($request));
			$url_part = isset($path[2]) ? $path[2] : 'google';
			$provider = ucfirst($url_part);
			$providers_mapping = $app['socialconnect.providers_mapping'];
			foreach ($providers_mapping as $provider_name => $info)
			{
				if ($info['url'] === $url_part)
				{
					$provider = $provider_name;
					break;
				}
			}

			if (strpos($e->getMessage(), 'Unsupported or undefined Opauth strategy') !== false) // AAAARGH!
			{
				throw new NotFoundHttpException();
			}

			$session = $app->session;
			$s_data = $session->get('socialconnect', []);
			$s_data['result'] = [
				'success' => false,
				'message' => $e->getMessage(),
				'provider' => $provider,
			];
			$session->set('socialconnect', $s_data);
		}

		$url = isset($s_data['continue']) ? $s_data['continue'] : '/login?social=' . strtolower($provider);
		return RedirectResponse::httpRedirect($url);
	}

	public function ShowIndex(Application $app)
	{
		$config = $app['socialconnect.opauth_config'];

		$providers = $config['Strategy'];

		$args = [];
		$args['providers.datasrc'] = [];
		foreach ($providers as $provider_name => $provider_info)
		{
			if (!isset($provider_info['enabled']) || $provider_info['enabled'])
			{
				$args['providers.datasrc'][] = [
					'provider_link.href' => '/socialconnect/' . strtolower($provider_name),
					'provider_link.body' => $provider_name,
				];
			}
		}

		return new TemplaterCallResponse('socialconnect/index.html', $args);
	}

}
