<?php
namespace Claromentis\SocialConnect;

use Claromentis\Core\Application;
use Claromentis\Core\Config\Config;
use Claromentis\Core\Event\LazyResolver;
use Claromentis\Core\RouteProviderInterface;
use Claromentis\Core\Templater\Plugin\TemplaterComponent;
use Claromentis\Core\Icon;
use Claromentis\Core\Localization\Lmsg;
use Pimple\Container;
use Pimple\ServiceProviderInterface;
use Silex\Api\EventListenerProviderInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 *
 *
 * @author Alexander Polyanskikh
 */
class SocialConnectApplication implements ServiceProviderInterface, RouteProviderInterface, TemplaterComponent, EventListenerProviderInterface
{
	public const APPLICATION_SVG_URL = '/intranet/socialconnect/assets/images/social-connect-icon.svg';
	/**
	 * Registers services on the given container.
	 *
	 * This method should only be used to configure services and parameters.
	 * It should not get services.
	 *
	 * @param Container $app A container instance
	 */
	public function register(Container $app)
	{
		$app['security.login_handlers'] = $app->extend('security.login_handlers', function ($login_handlers, $app) {
			$login_handlers[] = SocialLoginHandler::class;
			return $login_handlers;
		});

		$app['admin.panels'] = $app->extend('admin.panels', function ($panels) use ($app) {

			$lmsg = $app[Lmsg::class];

			$panels['socialconnect'] = new \Claromentis\Core\Admin\AdminPanelStandard('socialconnect', [
				'name' => ($lmsg)('socialconnect.menu.app.name'),
				'url' => '/socialconnect/admin',
				'icon' => [
					'type' => Icon\Types::SVG,
					'url'  => self::APPLICATION_SVG_URL
				]
			]);
			return $panels;
		});

		$app['localization.domain.socialconnect'] = function ($app) {
			return $app['localization.domain_from_files_factory']('socialconnect');
		};

		$app[AccountsRepository::class] = function () {
			return new AccountsRepository(\ObjectsStorage::I());
		};
		$app['socialconnect.accounts']  = AccountsRepository::class;

		$app['socialconnect.providers_mapping'] = function () {
			return [
				'Google' => [
					'url' => 'google',
					'callback' => 'callback',
					'client_id' => 'client_id',
					'client_secret' => 'client_secret',
                    'login_with_email' => 'login_with_email',
				],
				'Facebook' => [
					'url' => 'facebook',
					'callback' => 'callback',
					'client_id' => 'app_id',
					'client_secret' => 'app_secret',
				],
				'Live' => [
					'url' => 'live',
					'callback' => 'callback',
					'client_id' => 'client_id',
					'client_secret' => 'client_secret',
				],
				'Twitter' => [
					'url' => 'twitter',
					'callback' => 'callback',
					'client_id' => 'key',
					'client_secret' => 'secret',
				]
			];
		};

		$app['socialconnect.config'] = function ($app) {
			return $app['config.factory']('socialconnect');
		};

		$app['socialconnect.opauth_config'] = function($app) {
			$config = $app['socialconnect.config'];
			/** @var Config $config */

			$defaults = array(
				/**
				 * URL where Opauth is accessed.
				 *
				 * Can be either path starting and ending with / or full url starting from http(s):// and ending with /
				 * eg. if Opauth is reached via http://example.org/auth/, path is '/auth/' or 'http://example.org/auth/'
				 * if Opauth is reached via https://auth.example.org/, path is '/' or 'https://auth.example.org/'
				 */
				'base_url' => create_internal_link('/socialconnect/'),

				/**
				 * Uncomment if you would like to view debug messages
				 */
				//'debug' => true,

				'Strategy' => $config->Get('providers') ?: [],

				'http_client' => "Opauth\\Opauth\\HttpClient\\Guzzle",
			);


			$custom_config = $config->Get('opauth_config');
			if (is_array($custom_config))
				$opauth_config = array_merge_recursive($defaults, $custom_config);
			else
				$opauth_config = $defaults;

			return $opauth_config;
		};
	}

	/**
	 * Returns routes for the application.
	 *
	 * This method should be fast and should only return new static routes, preferably without
	 * using any other services, especially database.
	 *
	 * It should return an array in form of array($prefix => $closure), where $prefix is a string starting
	 * from slash, without trailing slash and $closure is a closure that takes an instance of
	 * \Claromentis\Core\ControllerCollection and registers all route handlers into it.
	 *
	 * Each route handler should be defined as a string - controller class and method name, such as
	 * $routes->get('/home', "MyApp\MyAppMainControler::OnHome")
	 *
	 * Make sure to secure all routes either by defining default security such as $collection->secure('html', 'user');
	 * or for each route $collection->get(....)->secure('html', 'user');
	 * Note, default security works only for routes defined _after_ it's set, so put it to the top of the closure
	 *
	 * Example:
	 *  return array(
	 *     '/main' => function (\Claromentis\Core\ControllerCollection $routes)
	 *     {
	 *          $routes->secure('html', 'user'); // default security rule
	 *          $routes->get('/', '\Claromentis\Main\Controller\HomePageController::Show');
	 *          $routes->get('/whats_new', '\Claromentis\Main\Controller\WhatsNewController::Show')->secure('ajax');
	 *          $routes->get('/{item_id}', '\Claromentis\Main\Controller\ItemController::Show')->assert('item_id', '\d+');
	 *     }
	 *  );
	 *
	 * @param \Claromentis\Core\Application $app An Application instance
	 *
	 * @return array
	 */
	public function GetRoutes(Application $app)
	{
		return array(
			'/socialconnect' => function (\Claromentis\Core\ControllerCollection $routes)
			{
				$routes->get('/_connect/{provider}', '\Claromentis\SocialConnect\Controller\ConnectController::Connect')->secure('html', 'user');
				$routes->post('/_disconnect/{provider}', '\Claromentis\SocialConnect\Controller\ConnectController::Disconnect')->secure('ajax', 'user');

				$routes->secure('html', 'admin', ['panel_code' => 'socialconnect']);
				$routes->get('/admin', '\Claromentis\SocialConnect\Controller\AdminController::Show');
				$routes->post('/admin', '\Claromentis\SocialConnect\Controller\AdminController::Save');

				$routes->secure('html', 'none');
				$routes->get('/', '\Claromentis\SocialConnect\Controller\AuthenticateController::ShowIndex');
				$routes->get('/{slug}', '\Claromentis\SocialConnect\Controller\AuthenticateController::Start')->assert('slug', '.+');
				/*$routes->get('/{provider}', '\Claromentis\SocialConnect\Controller\LoginController::Start');
				$routes->get('/{provider}/', '\Claromentis\SocialConnect\Controller\LoginController::Start');
				$routes->get('/{provider}/{action}', '\Claromentis\SocialConnect\Controller\LoginController::Callback');*/
			}
		);
	}

	public function Show($attributes, Application $app)
	{
		switch ($attributes['page'])
		{
			case 'mysettings.password_settings':
				$tc = new UI\ConnectedAccounts();
				return $tc->Show($attributes, $app);
			case 'login_page.login_method_link':
				$tc = new UI\LoginButtons();
				return $tc->Show($attributes, $app);
		}
		return '';
	}

	public function subscribe(Container $app, EventDispatcherInterface $dispatcher)
	{
		(new LazyResolver(
			$app,
			EventSubscriber::class,
			EventSubscriber::getSubscribedEvents())
		)->subscribe($dispatcher);
	}
}
