<?php
namespace Claromentis\SocialConnect;

use Claromentis\SocialConnect\Exception\AccountNotFoundException;
use Claromentis\SocialConnect\Exception\SocialConnectException;
use LoginHandler;

/**
 * Login handler for social connections
 *
 * @author Alexander Polyanskikh
 */
class SocialLoginHandler extends LoginHandler
{
	/**
	 * Reads data from GPC or any other available source and checks if it's correct
	 * Returns status value:
	 *   NO_DATA
	 *   FAILURE
	 *   SUCCESS
	 *
	 * Note, this method does not need to care about frozen accounts - this will be checked outside
	 *
	 * @return int
	 */
	public function TryLogin()
	{
		global $g_application;
		$session = $g_application->session;
		// create a session if one doesn't exist otherwise we end up in an infinite redirect loop
		if ($session->isStarted())
			$session->migrate();
		else
			$session->start();

		$socialconnect_session = $session->get('socialconnect');
		if ((getvar('google') || getvar('social')) && ($socialconnect_session === null || !isset($socialconnect_session['result'])))
		{
			$session->set('socialconnect', [
				'page' => getvar('page'),
				'continue' => '/login?page=' . urlencode(getvar('page'))
			]);

			if (!($provider = getvar('social')))
			{
				$provider = 'google';
			}
			httpRedirect('/socialconnect/' . strtolower($provider));
		}

		if (!$socialconnect_session || !isset($socialconnect_session['result']))
			return LoginHandler::NO_DATA;

		$result = $socialconnect_session['result'];
		$session->remove('socialconnect');

		if (!$result['success'])
		{
			$this->audit_comment = $result['message'];
			$this->error_message = $result['message'];
			return LoginHandler::FAILURE;
		}

		$provider = $result['provider'];
		$user_id_str = $result['uid'];

		/** @var AccountsRepository $repo */
		$repo = $g_application['socialconnect.accounts'];
		try
		{
			$account = $repo->Find($provider, $user_id_str);
			$this->last_user_id = $account->user_id;
		} catch (AccountNotFoundException $e)
		{
			$providers = array_change_key_case($g_application['socialconnect.config']['providers']);
			$login_with_email = isset($providers[strtolower($provider)], $providers[strtolower($provider)]['login_with_email']) ? true : false;

			$email = !empty($result['info']['email']) ? $result['info']['email'] : null;

			if ($login_with_email && $email)
			{
				$account = $repo->CreateFromEmail($email, $provider, $user_id_str);

				if ($account)
				{
					$this->last_user_id = $account->user_id;
					return LoginHandler::SUCCESS;
				}
			}

			$this->audit_comment = "user $provider:$user_id_str doesn't exist";
			$this->error_message = "Please log-in and connect your account in My settings";
			return LoginHandler::FAILURE;
		} catch (SocialConnectException $e)
		{
			$this->audit_comment = "$provider:$user_id_str : " . $e->getMessage();
			$this->error_message = "Internal error, details are in the the log file";
			return LoginHandler::FAILURE;
		}
		$this->audit_comment = "through the social connect ($provider)";
		return LoginHandler::SUCCESS;
	}
}
