function [MusFkts] = Bsp_FormFilt(anz, N, fa)
%
% Aufruf: [MusFkts] = Bsp_FormFilt(anz, N, fa)
%
% Aufrufbeispiel: [MusFkts] = Bsp_FormFilt(20, 1024, 100)
%
% Auf der Grundlage des Wiener-Lee-Theorems sollen ber eine
% spektrale Filterung anz Musterfunktionen eines Prozesses 
% erzeugt werden, der das Leistungsdichtespektrum
%
% S_{XX}(j*omega) = rect_{40*pi}(omega+20*pi)*cos(omega/40)
%
% besitzt.
%
% Eingabeparameter:    anz         Anzahl der Musterfunktionen
%                      N           verwendete FFT-Lnge (= Lnge der
%                                  Musterfunktionen)
%                      fa          BEZUGS-Abtastrate um S_{XX}(j*omega)
%                                  auf H(Omgega) umzurechnen. Diese
%                                  Bezugsabtastrate mus grer als 
%                                  2*10 Hz gewhlt werden (10 Hz ist
%                                  Bandbreite von S_{XX}(j*omega))
%                          
% Ausgabeparameter:    MusFkts     Musterfunktionen (zeilenweise
%                                  organisiert)
%
% Es werden KEINE Fehleingaben abgefangen !!
%
% Autor: Prof. Dr. Ottmar Beucher
%        HS Karlsruhe
% Version: 2.2
% Datum: 25.11.2014/20.05.2018

% Vorinitialisierung
MusFkts = [];

for k=1:anz
    % Zufallsgenerator initialisieren
    rng('shuffle', 'twister')

    % Gleichverteiltes mittelwertfreies
    % weies Rauschen mit RauschleistungsDICHTE 1 (Leistung) 
    % erzeugen
    xn = (rand(1,N)-0.5);
    xn = xn/std(xn);        % hat Varianz (=Leistung 1)
    xn = sqrt(fa)*xn;       % hat Varianz (=Leistung fa)
                            % und damit die Leistungsdichte 1
                            % im Nyquist-band [-fa/2,fa/2]
    
    % FFT des Signals berechnen und shiften
    spxn = fft(xn,N);
    spxn = fftshift(spxn);
    
    % Mit der Kennlinie des Filters multiplizieren
    dOmega = 2*pi/N; domega = dOmega*fa;
    omegas = (-pi*fa:domega:pi*fa-domega);  % FFT-Frequenzen in rad/s
    % Kennlinie an diesen Stellen berechnen
    Sxx = cos(omegas/40).*(omegas>=-20*pi & omegas<=20*pi);
 
    % und mit den Werten des Spektrums von xn multiplizieren
    spyn = spxn.*Sxx;
    
    % mit IFFT in Zeitwerte zurckverwandeln
    spyn = fftshift(spyn);
    yn = ifft(spyn,N);
    
    % und in Ausgabematrix speichern
    % (damit der "Imaginrteil" 0 nicht mitgespeichert wird,
    % wird nur der Realteil gespeichert)
    MusFkts = [MusFkts;real(yn)];
    
 
end
