% Script  Bsp_SysIdent.m
%
% Aufruf:  Bsp_SysIdent
%
% Simuliert das Simulink-System s_Rausch2.slx mit 
% den Simulationsparametern
%       simudauer = 10;     % Simulationsdauer
%       bw = 5;             % Bandbreite in Hz
%       frequenz = 200;     % Mittenfrequenz in Hz
%       fixstep = 0.0005;   % Schrittweite / 1/Abtastrate = 1/2000 
%       dichte = fixstep;   % Rauschleistungsdichte (Rauschleistung auf 1)
%
% 50 Mal durch und berechnet eine Systemidentifikation des Bandpasses auf
% der Grundlage des Wiener-Lee-Theorems.
%
% Es werden KEINE Fehleingaben abgefangen !!
%
% Autor: Prof. Dr. Ottmar Beucher
%        HS Karlsruhe
% Version: 4.01
% Datum: 25.11.2014/20.05.2018

% Simulink-System ffnen
open_system('s_Rausch2')

% Initialisierung der Simulationsparameter
simudauer = 10;     % Simulationsdauer
bw = 5;             % Bandbreite in Hz
frequenz = 200;     % Mittenfrequenz in Hz
fixstep = 0.0005;   % Schrittweite / 1/Abtastrate = 1/2000 
dichte = fixstep;   % Rauschleistungsdichte (Rauschleistung auf 1)

set_param('s_Rausch2/Filter', 'Wlo', num2str((frequenz-bw)*2*pi));
set_param('s_Rausch2/Filter', 'Whi', num2str((frequenz+bw)*2*pi));

% Mehrfacher Aufruf des Systems und Mittelung der
% berechneten Korrelationen mit der Funktion
% IterRausch2 (hier 50 Mittelungen)
[ct,corrNoise, corrBP] = IterRausch2(50, simudauer, fixstep, 0);

% Leistungsdichte des Eingangssignals mit Hilfe der FFT
N = length(corrNoise); M = floor(N/2);
K = 4*1024;  % 4*1024 Werte aus der Mitte herausschneiden
corrNoisemitte = corrNoise(M-K+1:M+K);   
Phixx = fft(corrNoisemitte, 2*K);
Phixx = fftshift(Phixx);

% Leistungsdichte des Ausgangssignals mit Hilfe der FFT
corrBPmitte = corrBP(M-K+1:M+K);   
Phiyy = fft(corrBPmitte, 2*K);
Phiyy = fftshift(Phiyy);

% Frequenzvektor definieren
fa = 1/fixstep;          % Abtastrate
df = fa/(2*K);           % Frequenzabstand
frq = (-fa/2:df:fa/2-df);

% Schtzung der bertragungsfunktion
Hquad = Phiyy./Phixx;

% Darstellung des Amplitudengangs
plot(frq, 10*log10(abs(Hquad)), 'b', 'LineWidth', 2);
grid
xlabel('Frequenz / Hz')
ylabel('|H|/ dB')
axis([-1000,1000,-45,5])

% Berechnung der gegebenen Bandpasscharakteristik
Wn = [(frequenz-bw)/(fa/2),(frequenz+bw)/(fa/2)];
[B,A] = butter(4,Wn);
frq2 = (160:1:240);
fa = 1/fixstep;
Hth = freqz(B,A,frq2,fa);

% Ausschnittsvergrerung des Amplitudengangs
figure
plot(frq, 10*log10(abs(Hquad)), 'b', ...
    frq2,20*log10(abs(Hth)),'r--','LineWidth', 2);
grid
xlabel('Frequenz / Hz')
ylabel('|H|/ dB')
title('Bandpass vs. Identifikation')
axis([160,240,-45,5])

% Beide Plots nebeneinander
figure
subplot(121)
plot(frq, 10*log10(abs(Hquad)), 'k', 'LineWidth', 2);
grid
xlabel('Frequenz / Hz')
ylabel('|H|/ dB')
axis([-1000,1000,-45,5])

subplot(122)
frqx = frq(frq>=160&frq<=240);
Hquadx = Hquad(frq>=160&frq<=240);
plot(frqx, 10*log10(abs(Hquadx)), 'b', ...
    frq2,20*log10(abs(Hth)),'r--','LineWidth', 2);
grid
xlabel('Frequenz / Hz')
ylabel('|H|/ dB')
title('Bandpass vs. Identifikation')
axis([160,240,-45,5])





