% Script  Bsp_SysIdentWelch.m
%
% Aufruf:  Bsp_SysIdentWelch
%
% Simuliert das Simulink-System s_Rausch2.slx mit 
% den Simulationsparametern
%       simudauer = 65.536; % Simulationsdauer (fr 64-K Daten)
%       bw = 5;             % Bandbreite in Hz
%       frequenz = 200;     % Mittenfrequenz in Hz
%       fixstep = 0.0005;   % Schrittweite / 1/Abtastrate = 1/2000 
%       dichte = fixstep;   % Rauschleistungsdichte (Rauschleistung auf 1)
%
% durch und berechnet eine Systemidentifikation des Bandpasses auf
% der Grundlage des Wiener-Lee-Theorems und mit Hilfe von
% Welchs Periodogrammmethode mit 2-K Daten-Blcken fr die Leistungsdichten
%
% Es werden KEINE Fehleingaben abgefangen !!
%
% Autor: Prof. Dr. Ottmar Beucher
%        HS Karlsruhe
% Version: 2.2
% Datum: 26.11.2014/20.05.2018

% Initialisierung 
bw = 5;             % Bandbreite in Hz
frequenz = 200;     % Mittenfrequenz in Hz
fixstep = 0.0005;   % Schrittweite einer Fixed-Step Simulation
simudauer = 65.536; % Simulationsdauer (fr 64-K Daten)
dichte = 1;         % Rauschleistungsdichte

% Simulink-System ffen
open_system('s_Rausch2')

% Initialisierung des Filters
set_param('s_Rausch2/Filter', 'Wlo', num2str((frequenz-bw)*2*pi));
set_param('s_Rausch2/Filter', 'Whi', num2str((frequenz+bw)*2*pi));

% Simulink-System ffen und Parameter setzen
set_param('s_Rausch2', 'FixedStep', num2str(fixstep));
set_param('s_Rausch2', 'Solver', 'ode4');
set_param('s_Rausch2', 'StopTime', num2str(simudauer));
set_param('s_Rausch2/Rauschquelle', 'Ts', num2str(fixstep));
set_param('s_Rausch2/Rauschquelle', 'Cov', num2str(fixstep));

% Simulink-System starten
[t,x,rsig] = sim('s_Rausch2');

% Eingangssignal definieren
sigin = rsig(:,1);

% Ausgangssignal definieren
sigout = rsig(:,2);

% Parmeter der Spektralschtzung
blocklaenge = 2048;        % 2-K-FFTs werden berechnet
ueberlapp = blocklaenge/2; % berlappungsgrad 50%

% Parmeter der Spektralschtzung
% blocklaenge = length(sigin);  % nur EINE FFT wird berechnet
% ueberlapp = 0;                % keine berlappung
abtastrate = 1/fixstep;    % Abtastrate

% Schtzung des Leistungsdichtespektrums des 
% Eingangssignals mit der Welch-Methode
% (es wird das ZWEISEITIGE LDS berechnet)
[Sxx,~] = cpsd(sigin,sigin,rectwin(blocklaenge),...
    ueberlapp,blocklaenge,abtastrate,'twosided');
 
% Schtzung des Leistungsdichtespektrums 
% des Ausgangssignals 
% mit der Welch-Methode
% (es wird das ZWEISEITIGE LDS berechnet)
[Syy,F] = cpsd(sigout,sigout,rectwin(blocklaenge),...
    ueberlapp,blocklaenge,abtastrate,'twosided');

% Schtzung der bertragungsfunktion
UebHquad = Syy./Sxx;

% Berechnung der gegebenen Bandpasscharakteristik
Wn = [(frequenz-bw)/(abtastrate/2),(frequenz+bw)/(abtastrate/2)];
[B,A] = butter(4,Wn);
frq2 = (160:1:240);
fa = 1/fixstep;
Hth = freqz(B,A,frq2,abtastrate);

figure
% Darstellung des Amplitudengangs
plot(F, 10*log10(abs(UebHquad)), 'b', 'LineWidth', 2)
grid
xlabel('Frequenz / Hz')
ylabel('|H|/ dB')

% Ausschnittsvergrerung des Amplitudengangs
figure
plot(F, 10*log10(abs(UebHquad)), 'b', ...
    frq2,20*log10(abs(Hth)),'r--','LineWidth', 2);
grid
xlabel('Frequenz / Hz')
ylabel('|H|/ dB')
title('Bandpass vs. Identifikation')
axis([160,240,-50,5])
 
% Beide Plots nebeneinander
figure
subplot(121)
% nur bis Abtastfrequenz/2 plotten
F2 = F(1:end/2);
UebHquad2 = UebHquad(1:end/2);

plot(F2, 10*log10(abs(UebHquad2)), 'k', 'LineWidth', 2);
grid
xlabel('Frequenz / Hz')
ylabel('|H|/ dB')
axis([0,1000,-60,5])

subplot(122)
frqx = F(F>=160&F<=240);
Hquadx = UebHquad2(F>=160&F<=240);
plot(frqx, 10*log10(abs(Hquadx)), 'b', ...
    frq2,20*log10(abs(Hth)),'r--','LineWidth', 2);
grid
xlabel('Frequenz / Hz')
ylabel('|H|/ dB')
title('Bandpass vs. Identifikation')
axis([160,240,-50,5])