function [] = DigiSimuRCTP(fa,T)
%
% Digitale Simulation des RC-Tiefpass mit Digitalfiltern,
% die durch Approximation 0-ter und 1-ter Ordnung der
% Integralgleichung gewonnen wurden.
%
% Aufruf:          DigiSimuRCTP(fa,T);
% Aufrufbeispiel:  DigiSimuRCTP(100,1);
%
% Eingangsparameter:   fa      Abtastrate
%                       T      Zeitkonstante des RC-Tiefpass
%
% Ausgangsparameter:   keine
%                      (Es werden bertragungsfunktionen geplottet)
%
% Autor: Prof. Dr. Ottmar Beucher
%        HS Karlsruhe
% Version: 1.2
% Datum: 15.11.2014/23.04.2018

% Vorinitialisierungen
Ta = 1/fa;
z= tf('z',Ta);

% bertragungsfunktionen definieren
% Digitale Approximation 0-ter Ordnung
H0 = (Ta/T)*z/(z-exp(-Ta/T)); 

% Digitale Approximation 1-ter Ordnung
H1 = (Ta/(2*T))*(z+exp(-Ta/T))/(z-exp(-Ta/T));

% Bode-Diagramm plotten
omega = (0:0.001:pi)*fa;   % Frequenzen in rad/s

% Bode-Diagramme. Auszug aus der Hilfe:
% For discrete-time models with sample time Ts, 
% BODE uses the transformation Z = exp(j*W*Ts) 
% to map the unit circle to the real frequency axis.
% The frequency response is only plotted 
% for frequencies smaller than the Nyquist frequency pi/Ts
[AH0,PH0] = bode(H0,omega);
[AH1,PH1] = bode(H1,omega);

% Plot vorbereiten (nur Amplitudengang)
AH0 = AH0(:); AH1 = AH1(:);
PH0 = PH0(:); PH1 = PH1(:);

semilogx(omega, 20*log10(abs(AH0)),'k', 'Linewidth', 2)
hold
semilogx(omega, 20*log10(abs(AH1)),'b', 'Linewidth', 2)
grid
xlabel('Frequenz/rad/s')
ylabel('|H(\Omega)| / dB')

figure
semilogx(omega, PH0*pi/180,'k', 'Linewidth', 2)
hold
semilogx(omega, PH1*pi/180,'b', 'Linewidth', 2)
grid
xlabel('Frequenz/rad/s')
ylabel('arg H(\Omega) / rad')


