function [EW, VAR, corrSig, t, ct] = HarmonProzess(anz, simudauer, fa, Amp, Frq)
%
% Funktion  HarmonProzess
%
% Aufruf:  [EW, VAR, corrSig, t, ct] = ...
%                HarmonProzess(anz, simudauer, fa, Amp, Frq)
%          HarmonProzess(anz, simudauer, fa, Amp, Frq)
%
% Aufrufbeispiel: HarmonProzess(200, 2, 1000, sqrt(2), 50);
%         [EW, VAR, corrSig, t, ct] = HarmonProzess(200, 2, 1000, sqrt(2), 50);
%
% Mit Hilfe dieser Funktion wird der so genannte Harmonischen
% Prozess simuliert. Dabei wird MEHRFACH (anz) ein Sinussignal mit
% stochatischer Nullphase erzeugt.
%
% Am Ende der Simulation werden Erwartungswert, Varianz
% und Autokorrelationsfunktion des Prozesses durch SCHARMITTELUNG
% geschtzt!
%
% Die Autokorrelationsfunktion sowie Mittelwert und
% Varianzfunktion werden darber hinaus geplottet.
% 
% Eingabeparameter:    anz         Anzahl der Simulationen
%                      simudauer   Simulationsdauer
%                      fa          Abtastrate
%                      Amp         Amplitude des Sinus
%                      Frq         Frequenz des Sinus
%                          
% Ausgabeparameter:    EW          Erwartungswert (in Abhngigkeit von t)
%                      VAR         Varianz (in Abhngigkeit von t)
%                      corrSig     Autokorrelationsfunktion
%                      t           Zeitvektor
%                      ct          Korrelationszeitvektor
%
% Es werden KEINE Fehleingaben abgefangen !!
%
% Autor: Prof. Dr. Ottmar Beucher
%        HS Karlsruhe
% Version: 4.2
% Datum: 12.05.2018/25.04.2018

% Initialisierung der Ausgabematrizen

T = simudauer;                        % Simulationszeit
dt = 1/fa;                            % Abtastintervall
N = floor(T/dt)+1;                    % Zahl der Werte
t = (0:dt:(N-1)*dt);                  % Abtastzeitpunkte
ct = (-T:dt:T);                       % Korrelationszeitvektor
CorSigM = zeros(N,2*N-1);             % Vorinitialisierte Korrelationsfolge
EW = zeros(N,1);                      % Vorinitialisierte EW un VAR und Signale
VAR = zeros(N,1); 
Sigs = zeros(anz,N); 

% Aufruf der Iterationsschleife fr die Simulation
% Signale mit Hilfe des Zufallsgenerators generieren
for i=1:anz 
   % Zufallsgenerator initialisieren
   rng('shuffle', 'twister');
   
   % Stochastische Nullphase erzeugen und setzen
   stochPhase = 2*pi*rand(1,1);  % Zufallsphase in [0,2*pi]
   
   % Mustersignal des harmonischen Prozesses erzeugen
   signal = Amp*sin(2*pi*Frq*t+stochPhase);
   
   % Signale sammeln; jedes Signal ist Zeilenvektor von Sigs
   Sigs(i,:) = signal;
end;

% Autokorrelationssignal ber die SCHAR-Mittelwerte berechnen
CorrSigs = zeros(anz,2*N-1);
for k=1:anz
    CorrSigs(k,:) = xcorr(Sigs(k,:), Sigs(k,:), 'biased');
end
% Scharmittelwert bilden
corrSig = sum(CorrSigs,1)/anz;

% Erwartungswert und Varianz berechnen
EW = mean(Sigs);
VAR = var(Sigs);

% Autokorrelationsfunktion plotten
% Periodendauer des Sinus
Per = 1/Frq;
% Zahl der Werte fr eine Periode
M = floor(Per/dt);
% drei Perioden links und rechts um tau = 0 darstellen
subplot(122)
plot(ct(N-3*M+1:N+3*M+1), corrSig(N-3*M+1:N+3*M+1), 'b', 'LineWidth',2);
grid
xlabel('Zeit \tau/s')
ylabel('r_{XX}(\tau)')
axis([ct(N-3*M+1), ct(N+3*M+1), min(corrSig), max(corrSig)])

% Mittelwertfunktion und Varianzfunktion plotten
subplot(121)
plot(t, EW, 'k', t, VAR, 'b', 'LineWidth', 2);
axis([0,T,-Amp^2, Amp^2])
grid
xlabel('Zeit t/s')
ylabel('Kennwerte')




