function [ct,corrNoise, corrBP] = IterRausch2(anz, simudauer, fixstep, plt)
%
% Funktion  IterRausch2
%
% Aufruf:  [ct,corrNoise, corrBP] = IterRausch2(anz, simudauer, fixstep, plt);
%          IterRausch2(anz, simudauer, fixstep, plt);
%
% Aufrufbeispiel: [ct,corrNoise, corrBP] = IterRausch2(10,10,0.0005,1);
%
% Mit Hilfe dieser Funktion wird das Simulink-System
% s_Rausch2.slx (mit den durch die Initialisierung
% festgelegten Parametern) MEHRFACH (anz) aufgerufen.
%
% Nach jeder Simulation werden die Autokorrelationsfolgen
% des Rauschsignals und des Bandpassausgangssignals
% berechnet. Diese werden anschlieend gemittelt.
%
% Die gelittelten Korrelationsfolgen werden darber 
% hinaus geplottet.
% 
% Eingabeparameter:    anz         Anzahl der Simulationen
%                      simudauer   die im Initialisierungsfile
%                                  festgelegte simulationsdauer
%                      fixstep     die im Initialisierungsfile
%                                  festgelegte Schrittweite der
%                                  fixstep-Simulation
%                      plt         0, falls Ergebnis NICHT geplottet 
%                                  werden soll, sonsz erfolgt grafische
%                                  Darstellung (bei plot!=0)
%                          
% Ausgabeparameter:    ct          Korrelations-Zeitvektor
%                      corrNoise   Gemittelte Autokorrelation
%                                  des Rauschsignals
%                      corrBP      Gemittelte Autokorrelation
%                                  des Bandpassausgangssignals
%
% Es werden KEINE Fehleingaben abgefangen !!
%
% Autor: Prof. Dr. Ottmar Beucher
%        HS Karlsruhe
% Version: 4.2
% Datum: 21.11.2014/25.05.2018


% ffnung und Initialisierung des Systems
open_system('s_Rausch2');
set_param('s_Rausch2', 'FixedStep', num2str(fixstep));
set_param('s_Rausch2', 'StopTime', num2str(simudauer));
set_param('s_Rausch2/Rauschquelle', 'Ts', num2str(fixstep));
set_param('s_Rausch2/Rauschquelle', 'Cov', num2str(fixstep));
set_param('s_Rausch2', 'MaxStep', num2str(fixstep*5));
set_param('s_Rausch2', 'MinStep', num2str(fixstep/10));
set_param('s_Rausch2', 'OutputTimes', ['0:',num2str(fixstep),':',num2str(simudauer)]);


bw = 5;             % Bandbreite des Bandpass in Hz
frequenz = 200;     % Mittenfrequenz in Hz
set_param('s_Rausch2/Filter', 'Wlo', num2str((frequenz-bw)*2*pi));
set_param('s_Rausch2/Filter', 'Whi', num2str((frequenz+bw)*2*pi));


% Initialisierung der Ausgabematrizen

T = simudauer;                        % Simulationszeit
dt = fixstep;                         % Abtastintervall
N = floor(T/dt)+1;                    % Zahl der Werte
corrNoise = zeros(2*N-1,1);           % Vorinitialisierte
corrBP = zeros(2*N-1,1);              % Korrelationsfolgen

% Aufruf der Iterationsschleife fr die Simulation

for i=1:anz 
   % Zufallsgenerator fr den Rauschblock neu 
   % initialisierenen
   rng('shuffle', 'twister');
   zufall = floor(rand(1)*32000)+1;  
   set_param('s_Rausch2/Rauschquelle', 'seed', num2str(zufall));
   
   % Simulink-System aufrufen und Signale
   [t,x,signale] = sim('s_Rausch2');
   rsig0 = signale(:,1);
   rsigbp = signale(:,2);
   
   % Korrelationssignale berechnen
   crsig0 = xcorr(rsig0,'unbiased');
   crsigbp = xcorr(rsigbp,'unbiased');
   
   % Korrekationssignale akkumulieren
   corrNoise = corrNoise + crsig0;
   corrBP = corrBP + crsigbp;
end;

% Korrelationen mitteln
corrNoise = corrNoise/anz;
corrBP = corrBP/anz;

% Korrelations-Zeitvektor festlegen
ct = (-T:dt:T);

if plt
% Autokorrelationsfunktion plotten
figure
plot(ct, corrNoise, 'b', 'LineWidth',2);
grid
xlabel('Zeit/s')
ylabel('r_{NN}(k)')
axis([-T,T,min(corrNoise), max(corrNoise)])

figure
plot(ct, corrBP, 'r', 'LineWidth',2);
grid
xlabel('Zeit/s')
ylabel('r_{BP}(k)')
axis([-T,T,min(corrBP), max(corrBP)])
end
