function [F, PSDsBP] = VRausch3(anz, M, N, fixstep)
%
% Funktion  VRausch3
%
% Aufruf:  [F, PSDsBP] = VRausch3(anz, M, N, fixstep)
%
% Aufrufbeispiel: [F, PSDsBP] = VRausch3(100, 16, 1, 0.0005)
%
% Mit Hilfe dieser Funktion wird das Simulink-System
% s_Rausch3.slx (mit den durch die Initialisierung unten
% festgelegten Parametern) anz Mal aufgerufen.
%
% Nach jeder Simulation wird das Leistungsdichtespektrum
% des Bandpass-Ausgangssignals mit M NICHT-berlappenden 
% N-K-Periodogrammen berechnet  und als Spalte in  PSDsBP 
% gespeichert (zum Beispiel fr N = 1 werden 1-K-Periodogramme
% berechnet).
%
% Die Simulation dient zur Untersuchung der Varianz
% der Schtzungen fr nach der Periodogramm-Methode
% berechneten Leistungsdichtespektren.
% 
% Eingabeparameter:    anz         Zahl der Simulationen
%                      M           Zahl der Periodogramme
%                                  fixstep     festgelegte Schrittweite 
%                                  (1/Abtastrate)
%                      N           Anzahl der K Daten fr die FFT
%                                  (N sollte nicht grer als 64 sein)
%                          
% Ausgabeparameter:    F           Frequenzvektor
%                      PSDsBP      Matrix mit den geschtzten
%                                  Leistungsdichtespektren des
%                                  Bandpass-Ausgangssignals
%
% Es werden KEINE Fehleingaben abgefangen !!
%
% Autor: Prof. Dr. Ottmar Beucher
%        FH Karlsruhe
% Version: 2.2
% Datum: 27.11.2014/20.05.2018

% N berprfen

if N>64
    error('Bitte nicht mehr als 64 K angeben.');
end

% Gesamtdauer der Simulation berechnen

simudauer = (M*N*1024-1)*fixstep;

% ffnung und Initialisierung des Systems
open_system('s_Rausch3');
set_param('s_Rausch3', 'FixedStep', num2str(fixstep));
set_param('s_Rausch3', 'MaxStep', num2str(fixstep*5));
set_param('s_Rausch3', 'MinStep', num2str(fixstep/10));
set_param('s_Rausch3', 'StopTime', num2str(simudauer));
set_param('s_Rausch3', 'OutputTimes', ['0:',num2str(fixstep),':',num2str(simudauer)]);

set_param('s_Rausch3/Rauschquelle', 'Ts', num2str(fixstep));

% Blockparameter entsprechend der Standard-Vorinitialisierung
% mit init_Rausch

bw = 5;             % Bandbreite in Hz
frequenz = 200;     % Mittenfrequenz in Hz
dichte = fixstep;   % Rauschleistungsdichte (Rauschleistung auf 1)
set_param('s_Rausch3/Filter', 'Wlo', num2str((frequenz-bw)*2*pi));
set_param('s_Rausch3/Filter', 'Whi', num2str((frequenz+bw)*2*pi));
set_param('s_Rausch3/Rauschquelle', 'Cov', num2str(dichte));

% Initialisierung der Ausgabematrix

PSDsBP = [];

% Aufruf der Iterationsschleife fr die Simulation

for i=1:anz
   % Zufallsgenerator fr den Rauschblock neu 
   % initialisierenen
   rng('shuffle', 'twister')
   zufall = floor(rand(1)*32000)+1;     
   zufall = floor(rand(1)*32000)+1;  
   set_param('s_Rausch3/Rauschquelle', 'seed', num2str(zufall));
   
   % Simulink-System aufrufen und Bandpassignal speichern
   [t,x,signale] = sim('s_Rausch3');
   bpsig = signale(:,2);
   
   % Leistungsdichtespektrum berechnen
   [Pyy,F] = cpsd(bpsig,bpsig,rectwin(N*1024),0,N*1024,1/fixstep, 'onesided');
   
   % und in der Ausgabematrix aufsammeln
   PSDsBP = [PSDsBP,Pyy];
end;

