/**
 * @module Lens Scripting
 * @version 5.0.17
 * For Snapchat Version: 12.96
*/
interface ComponentNameMap {
    "Animation": Animation;
    "AnimationMixer": AnimationMixer;
    "AnimationPlayer": AnimationPlayer;
    "AudioChainComponent": AudioEffectComponent;
    "AudioComponent": AudioComponent;
    "AudioEffectComponent": AudioEffectComponent;
    "AudioListenerComponent": AudioListenerComponent;
    "AudioPlayer": AudioComponent;
    "BaseMeshVisual": BaseMeshVisual;
    "BlurNoiseEstimation": BlurNoiseEstimation;
    "Camera": Camera;
    "Canvas": Canvas;
    "ClearDepth": ClearDepth;
    "ClearScreen": ClearDepth;
    "ClothVisual": ClothVisual;
    "ColliderComponent": ColliderComponent;
    "ColocatedTrackingComponent": ColocatedTrackingComponent;
    "Component.Animation": Animation;
    "Component.AnimationMixer": AnimationMixer;
    "Component.AnimationPlayer": AnimationPlayer;
    "Component.AudioChainComponent": AudioEffectComponent;
    "Component.AudioComponent": AudioComponent;
    "Component.AudioEffectComponent": AudioEffectComponent;
    "Component.AudioListenerComponent": AudioListenerComponent;
    "Component.AudioPlayer": AudioComponent;
    "Component.BaseMeshVisual": BaseMeshVisual;
    "Component.BlurNoiseEstimation": BlurNoiseEstimation;
    "Component.Camera": Camera;
    "Component.Canvas": Canvas;
    "Component.ClearDepth": ClearDepth;
    "Component.ClearScreen": ClearDepth;
    "Component.ClothVisual": ClothVisual;
    "Component.ColliderComponent": ColliderComponent;
    "Component.ColocatedTrackingComponent": ColocatedTrackingComponent;
    "Component.DepthSetter": DepthSetter;
    "Component.DeviceLocationTrackingComponent": DeviceLocationTrackingComponent;
    "Component.DeviceTracking": DeviceTracking;
    "Component.EyeColorVisual": EyeColorVisual;
    "Component.FaceInsetVisual": FaceInsetVisual;
    "Component.FaceMaskVisual": FaceMaskVisual;
    "Component.FaceStretchVisual": FaceStretchVisual;
    "Component.FaceSubVisual": FaceMaskVisual;
    "Component.GaussianSplattingVisual": GaussianSplattingVisual;
    "Component.HairSimulationColliderComponent": ColliderComponent;
    "Component.HairVisual": HairVisual;
    "Component.Head": Head;
    "Component.Hints": HintsComponent;
    "Component.HintsComponent": HintsComponent;
    "Component.Image": Image;
    "Component.InteractionComponent": InteractionComponent;
    "Component.LightSource": LightSource;
    "Component.LiquifyVisual": LiquifyVisual;
    "Component.LocatedAtComponent": LocatedAtComponent;
    "Component.LookAtComponent": LookAtComponent;
    "Component.MLComponent": MLComponent;
    "Component.ManipulateComponent": ManipulateComponent;
    "Component.MarkerTrackingComponent": MarkerTrackingComponent;
    "Component.MaskingComponent": MaskingComponent;
    "Component.MaterialMeshVisual": MaterialMeshVisual;
    "Component.MeshVisual": RenderMeshVisual;
    "Component.ObjectTracker2D": ObjectTracking;
    "Component.ObjectTracking": ObjectTracking;
    "Component.ObjectTracking3D": ObjectTracking3D;
    "Component.PinToMeshComponent": PinToMeshComponent;
    "Component.PostEffectVisual": PostEffectVisual;
    "Component.RectangleSetter": RectangleSetter;
    "Component.RenderMeshVisual": RenderMeshVisual;
    "Component.RetouchVisual": RetouchVisual;
    "Component.ScreenRegionComponent": ScreenRegionComponent;
    "Component.ScreenTransform": ScreenTransform;
    "Component.Script": ScriptComponent;
    "Component.ScriptComponent": ScriptComponent;
    "Component.Skin": Skin;
    "Component.SplineComponent": SplineComponent;
    "Component.SpriteAligner": SpriteAligner;
    "Component.SpriteVisualV2": Image;
    "Component.Text": Text;
    "Component.Text3D": Text3D;
    "Component.Touch": InteractionComponent;
    "Component.TouchComponent": InteractionComponent;
    "Component.TrackedPointComponent": TrackedPointComponent;
    "Component.VFXComponent": VFXComponent;
    "Component.VertexCache": VertexCache;
    "Component.Visual": Visual;
    "Component.WorldTracking": DeviceTracking;
    "DepthSetter": DepthSetter;
    "DeviceLocationTrackingComponent": DeviceLocationTrackingComponent;
    "DeviceTracking": DeviceTracking;
    "EyeColorVisual": EyeColorVisual;
    "FaceInsetVisual": FaceInsetVisual;
    "FaceMaskVisual": FaceMaskVisual;
    "FaceStretchVisual": FaceStretchVisual;
    "FaceSubVisual": FaceMaskVisual;
    "GaussianSplattingVisual": GaussianSplattingVisual;
    "HairSimulationColliderComponent": ColliderComponent;
    "HairVisual": HairVisual;
    "Head": Head;
    "Hints": HintsComponent;
    "HintsComponent": HintsComponent;
    "Image": Image;
    "InteractionComponent": InteractionComponent;
    "LightSource": LightSource;
    "LiquifyVisual": LiquifyVisual;
    "LocatedAtComponent": LocatedAtComponent;
    "LookAtComponent": LookAtComponent;
    "MLComponent": MLComponent;
    "ManipulateComponent": ManipulateComponent;
    "MarkerTrackingComponent": MarkerTrackingComponent;
    "MaskingComponent": MaskingComponent;
    "MaterialMeshVisual": MaterialMeshVisual;
    "MeshVisual": RenderMeshVisual;
    "ObjectTracker2D": ObjectTracking;
    "ObjectTracking": ObjectTracking;
    "ObjectTracking3D": ObjectTracking3D;
    "Physics.BodyComponent": BodyComponent;
    "Physics.ColliderComponent": ColliderComponent;
    "Physics.ConstraintComponent": ConstraintComponent;
    "Physics.WorldComponent": WorldComponent;
    "PinToMeshComponent": PinToMeshComponent;
    "PostEffectVisual": PostEffectVisual;
    "RectangleSetter": RectangleSetter;
    "RenderMeshVisual": RenderMeshVisual;
    "RetouchVisual": RetouchVisual;
    "ScreenRegionComponent": ScreenRegionComponent;
    "ScreenTransform": ScreenTransform;
    "Script": ScriptComponent;
    "ScriptComponent": ScriptComponent;
    "Skin": Skin;
    "SplineComponent": SplineComponent;
    "SpriteAligner": SpriteAligner;
    "SpriteVisualV2": Image;
    "Text": Text;
    "Text3D": Text3D;
    "Touch": InteractionComponent;
    "TouchComponent": InteractionComponent;
    "TrackedPointComponent": TrackedPointComponent;
    "VFXComponent": VFXComponent;
    "VertexCache": VertexCache;
    "Visual": Visual;
    "WorldTracking": DeviceTracking;
}

interface EventNameMap {
    "BrowsLoweredEvent": BrowsLoweredEvent;
    "BrowsRaisedEvent": BrowsRaisedEvent;
    "BrowsReturnedToNormalEvent": BrowsReturnedToNormalEvent;
    "CameraBackEvent": CameraBackEvent;
    "CameraFrontEvent": CameraFrontEvent;
    "ClientInterfacePlayButtonTriggerEvent": ConnectedLensEnteredEvent;
    "ConnectedLensEnteredEvent": ConnectedLensEnteredEvent;
    "DelayedCallbackEvent": DelayedCallbackEvent;
    "FaceFoundEvent": FaceFoundEvent;
    "FaceLostEvent": FaceLostEvent;
    "FaceTrackingEvent": FaceTrackingEvent;
    "KissFinishedEvent": KissFinishedEvent;
    "KissStartedEvent": KissStartedEvent;
    "LateUpdateEvent": LateUpdateEvent;
    "ManipulateEndEvent": ManipulateEndEvent;
    "ManipulateStartEvent": ManipulateStartEvent;
    "MouthClosedEvent": MouthClosedEvent;
    "MouthOpenedEvent": MouthOpenedEvent;
    "OnAwakeEvent": OnAwakeEvent;
    "OnDestroyEvent": OnDestroyEvent;
    "OnDisableEvent": OnDisableEvent;
    "OnEnableEvent": OnEnableEvent;
    "OnStartEvent": OnStartEvent;
    "SceneEvent.BrowsWereJustFrownedEvent": BrowsLoweredEvent;
    "SceneEvent.BrowsWereJustRaisedEvent": BrowsRaisedEvent;
    "SceneEvent.BrowsWereJustReturnedToNormalEvent": BrowsReturnedToNormalEvent;
    "SceneEvent.ClientInterfacePlayButtonTriggerEvent": ConnectedLensEnteredEvent;
    "SceneEvent.KissJustFinishedEvent": KissFinishedEvent;
    "SceneEvent.KissJustStartedEvent": KissStartedEvent;
    "SceneEvent.MouthWasJustClosedEvent": MouthClosedEvent;
    "SceneEvent.MouthWasJustOpenedEvent": MouthOpenedEvent;
    "SceneEvent.SmileJustFinishedEvent": SmileFinishedEvent;
    "SceneEvent.SmileJustStartedEvent": SmileStartedEvent;
    "SceneEvent.SurfaceTrackingResetEvent": SurfaceTrackingResetEvent;
    "SceneObjectEvent": SceneObjectEvent;
    "SinglePlaneTrackingUpdatedEvent": SinglePlaneTrackingUpdatedEvent;
    "SmileFinishedEvent": SmileFinishedEvent;
    "SmileStartedEvent": SmileStartedEvent;
    "SnapImageCaptureEvent": SnapImageCaptureEvent;
    "SnapRecordStartEvent": SnapRecordStartEvent;
    "SnapRecordStopEvent": SnapRecordStopEvent;
    "TapEvent": TapEvent;
    "TouchEndEvent": TouchEndEvent;
    "TouchMoveEvent": TouchMoveEvent;
    "TouchStartEvent": TouchStartEvent;
    "TriggerPrimaryEvent": TriggerPrimaryEvent;
    "TurnOffEvent": TurnOffEvent;
    "UpdateEvent": UpdateEvent;
    "WorldTrackingMeshesAddedEvent": WorldTrackingMeshesAddedEvent;
    "WorldTrackingMeshesRemovedEvent": WorldTrackingMeshesRemovedEvent;
    "WorldTrackingMeshesUpdatedEvent": WorldTrackingMeshesUpdatedEvent;
    "WorldTrackingPlanesAddedEvent": WorldTrackingPlanesAddedEvent;
    "WorldTrackingPlanesRemovedEvent": WorldTrackingPlanesRemovedEvent;
    "WorldTrackingPlanesUpdatedEvent": WorldTrackingPlanesUpdatedEvent;
    "WorldTrackingResetEvent": SurfaceTrackingResetEvent;
}

/**
* @description Returns the time in seconds since the lens was started.
*/
declare function getTime(): number

/**
* @description Get current time in Nanoseconds. Useful when optimizing a Lens to understand its performance.
*/
declare function getRealTimeNanos(): number

/**
* @description Returns the time difference in seconds between the current frame and previous frame.
*/
declare function getDeltaTime(): number

/**
* @description Load a JavaScript module. Used for importing another JavaScript file found in the Resources panel to be used in the current script. Similar to `require` found in CommonJS specification. You can access Lens Studio specific modules (like `RemoteServiceModule`) using the `LensStudio` prefix, such as: `require("LensStudio:RemoteServiceModule)`. 

* You can use the modules name or path, meaning you can have two version of the same module in your project if needed. Both by name or by path will be relative to the script which is calling require, but by name will check the same folder as the script, then the parent folder, and so on. In addition, any require can be relative to your library folder. Take a look at the [Scripting guide](https://docs.snap.com/lens-studio/references/guides/lens-features/adding-interactivity/scripting-overview) to learn more.

*/
declare function require(moduleName: (ScriptAsset|string)): unknown

/**
* @description Load a script by name or path in order to use that type to create or get a component at runtime. 

* You can use the modules name or path. Both by name or by path will be relative to the script which is calling require, but by name will check the same folder as the script, then the parent folder, and so on. In addition, any require can be relative to your library folder. Take a look at the [Scripting guide](https://docs.snap.com/lens-studio/references/guides/lens-features/adding-interactivity/scripting-overview) to learn more.
*/
declare function requireType(name: string): string

/**
* @description Load an asset like a `Texture`, `Material`, or `MLAsset` directly from script. 

* You can use the modules name or path. Both by name or by path will be relative to the script which is calling require, but by name will check the same folder as the script, then the parent folder, and so on. In addition, any require can be relative to your library folder. Take a look at the [Scripting guide](https://docs.snap.com/lens-studio/references/guides/lens-features/adding-interactivity/scripting-overview) to learn more.
*/
declare function requireAsset(name: string): Asset

/**
* @description Prints out a message to the Logger window.
*/
declare function print(message: unknown): void

/**
* @description Returns true if the passed in object is null or destroyed. Useful as a safe way to check if a SceneObject or Component has been destroyed.
*/
declare function isNull(reference: unknown): boolean

declare namespace global {
    /**
    * @description Returns the global [ScriptScene](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#ScriptScene) object, which offers information and controls for the current scene.
    */
    let scene: ScriptScene
    
    /**
    * @description Returns the global [TouchDataProvider](https://lensstudio.snapchat.com/api/lens-studio/Classes/Providers#TouchDataProvider), which controls how the Lens handles touch events.
    */
    let touchSystem: TouchDataProvider
    
    /**
    * @description Returns the global [PersistentStorageSystem](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#PersistentStorageSystem), which allows data to persist between Lens sessions.
    */
    let persistentStorageSystem: PersistentStorageSystem
    
    /**
    * @description Returns the global `GeneralDataStore` for Launch Params, which provides any special data passed in when the Lens is launched.
    */
    let launchParams: GeneralDataStore
    
    /**
    * @description Returns the global DeviceInfoSystem, which provides information about the device running the Lens.
    */
    let deviceInfoSystem: DeviceInfoSystem
    
    /**
    * @description Returns the global [LocalizationSystem](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#LocalizationSystem), which helps convert times, dates, and other units to user friendly strings.
    */
    let localizationSystem: LocalizationSystem
    
    let hapticFeedbackSystem: HapticFeedbackSystem
    
    let textInputSystem: TextInputSystem
    
    /**
    * @description Returns the global [UserContextSystem](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#UserContextSystem), which provides information about the user such as display name, birthday, and even current weather.
    */
    let userContextSystem: UserContextSystem
    
}

/**
* @description An object containing the position of an object relative to a LocationAsset. 
*/
interface Anchor extends ScriptObject {
    /**
    * @readonly
    
    * @description The LocationAsset which this anchor is relative to.
    */
    location: LocationAsset
    
    /**
    * @readonly
    
    * @description The position of this anchor relative to the LocationAsset.
    */
    position: vec3
    
}

/**
* @description Controls an animated texture resource. Can be accessed from [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#texture--control) on an animated texture.
* See also: [2D Animation Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/adding-content/2d/2d-animation).
*/
interface AnimatedTextureFileProvider extends TextureProvider {
    /**
    * @description Returns whether the animation is currently playing.
    */
    isPlaying(): boolean
    
    /**
    * @description Returns whether the animation is currently paused.
    */
    isPaused(): boolean
    
    /**
    * @description Returns whether the animation is finished playing.
    */
    isFinished(): boolean
    
    /**
    * @description Plays the animation `loops` times, starting with an offset of `offset` seconds.
    */
    play(loops: number, offset: number): void
    
    /**
    * @description Stops the animation.
    */
    stop(): void
    
    /**
    * @description Pauses the animation.
    */
    pause(): void
    
    /**
    * @description Resumes a paused animation from the frame that was last played.
    */
    resume(): void
    
    /**
    * @description Start playing the animation from frame `frameIndex`, `loops` times.
    */
    playFromFrame(frameIndex: number, loops: number): void
    
    /**
    * @description Pauses the animation at frame `frameIndex`.
    */
    pauseAtFrame(frameIndex: number): void
    
    /**
    * @description Returns the number of frames in the animation.
    */
    getFramesCount(): number
    
    /**
    * @description Returns the index of the frame that is currently playing.
    */
    getCurrentPlayingFrame(): number
    
    /**
    * @description Returns how long the animation is in seconds.
    */
    getDuration(): number
    
    /**
    * @description Sets the callback function to be called whenever the animation stops playing.
    */
    setOnFinish(eventCallback: (animatedTexture: AnimatedTextureFileProvider) => void): void
    
    /**
    * @description Duplicates the AnimatedTextureFileProvider and returns the new copy. Can be used for playing the same animation at different offsets.
    */
    clone(): AnimatedTextureFileProvider
    
    /**
    * @description Whether the animation plays in reverse.
    */
    isReversed: boolean
    
    /**
    * @description If enabled, the animation will alternate between normal and reverse each time it loops.
    */
    isPingPong: boolean
    
    /**
    * @description Returns whether the animation was set to automatically play and loop.
    */
    isAutoplay: boolean
    
    /**
    * @description The animation track used to control the frame animation.
    */
    track: IntStepAnimationTrackKeyFramed
    
    /**
    * @description Length of the animation in seconds.
    */
    duration: number
    
}

/**
* @description Used by [AnimationMixer](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AnimationMixer) to animate a single object in the hierarchy.
* These are automatically added to SceneObjects when importing animated FBX files.
* See also: [Playing 3D Animation Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/additional-examples/playing-3d-animation), [AnimationMixer](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AnimationMixer), [AnimationLayer](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#AnimationLayer).
*/
interface Animation extends Component {
    /**
    * @description Returns the AnimationLayer under the name `layerName`.
    */
    getAnimationLayerByName(layerName: string): AnimationLayer
    
    /**
    * @description Adds an AnimationLayer under the name `layerName`.
    */
    setAnimationLayerByName(layerName: string, animationLayer: AnimationLayer): void
    
    /**
    * @description Removes the AnimationLayer under the name `layerName`.
    */
    removeAnimationLayerByName(layerName: string): void
    
}

/**
* @description Asset that contains multiple animation layers. Animation assets themselves do not handle playing or orchestrating animations. This is left to the animation player component to handle.
*/
interface AnimationAsset extends Asset {
    deleteEvent(layerName: AnimationPropertyEventRegistration): void
    
    createEvent(eventName: string, time: number): AnimationPropertyEventRegistration
    
    /**
    * @description Adds AnimationPropertyLayer to Animation asset.
    */
    addLayer(layerName: string, layer: AnimationPropertyLayer): void
    
    /**
    * @description Delete the AnimationPropertyLayer named `layerName`.
    */
    deleteLayer(layerName: string): void
    
    /**
    * @description Retrieves an animation associated with a particular object.
    */
    getLayer(layerName: string): AnimationPropertyLayer
    
    /**
    * @description Delete all the AnimationPropertyLayer in this AnimationAsset.
    */
    clearLayers(): void
    
    /**
    * @readonly
    
    * @description Animation duration in seconds.
    */
    duration: number
    
    /**
    * @readonly
    
    * @description Denotes how many key frames this animation was sampled at.
    */
    fps: number
    
}

/**
* @description Animation Clip is what an Animation Player uses to manage playback for a specific animation. It defines that animation by referencing an Animation Asset and providing start and end points, playback speed and direction, and blending information. 
*/
interface AnimationClip extends ScriptObject {
    /**
    * @description Clones the existing clip with a new name.
    */
    clone(clipName: string): AnimationClip
    
    /**
    * @readonly
    
    * @description Name of the clip.
    */
    name: string
    
    /**
    * @description Points to the animation asset to be played by the Animation Player.
    */
    animation: AnimationAsset
    
    /**
    * @description Strength of animation clip contribution. Lies between [0.0, 1.0] inclusive. For default blending a 1.0 weight indicates this clip will override all earlier clips, a less than 1.0 weight indicates it will blend onto the calculated pose using a weighted average.
    */
    weight: number
    
    /**
    * @description Scalar value to represent playback speed percentage. 1.0 is 100% playback speed.
    */
    playbackSpeed: number
    
    /**
    * @description Choose whether to play animation clip once, loop the clip, or ping pong it.
    */
    playbackMode: PlaybackMode
    
    /**
    * @description Returns begin time of clip.
    */
    begin: number
    
    /**
    * @description Returns end time of clip.
    */
    end: number
    
    /**
    * @description Specifies if the clip should be played reversed.
    */
    reversed: boolean
    
    /**
    * @description Whether the animation clip is disabled.
    */
    disabled: boolean
    
    /**
    * @readonly
    
    * @description Returns the duration of the clip which is calculated based on the begin and end times.
    */
    duration: number
    
    /**
    * @description The blend mode for this particular clip.
    */
    blendMode: AnimationLayerBlendMode
    
    /**
    * @description How scale is accumulated. Usually does not need to be changed after import.
    */
    scaleMode: AnimationLayerScaleMode
    
}
declare namespace AnimationClip {
    /**
    * @description Creates a clip.
    */
    export function create(clipName: string): AnimationClip
    
    export function createFromAnimation(clipName: string, animation: AnimationAsset): AnimationClip
    

}

declare namespace AnimationClip {
    /**
    * @description Used by [AnimationMixerLayer](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#AnimationMixerLayer) for setting animation looping behavior.
    */
    enum PostInfinity {
        /**
        * @description The animation will restart from the beginning each time it loops.
        */
        Cycle,
        /**
        * @description The animation will switch between normal and reverse playback each time it loops.
        */
        Oscillate
    }

}

declare namespace AnimationClip {
    /**
    * @description Used by [AnimationMixerLayer](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#AnimationMixerLayer) for setting animation clip range type.
    */
    enum RangeType {
        /**
        * @description Range is specified by start and end time, in seconds
        */
        Time,
        /**
        * @description Range is specified by start and end frame numbers
        */
        Frames
    }

}

interface AnimationCurve extends ScriptObject {
    addKeyframe(frame: AnimationKeyFrame): void
    
    removeKeyFrame(t: number): void
    
    getKeyFrame(index: number): AnimationKeyFrame
    
    evaluate(time: number): number
    
    /**
    * @readonly
    */
    keyFrameCount: number
    
}
declare namespace AnimationCurve {
    export function create(): AnimationCurve
    
    export function createKeyFrame(): AnimationKeyFrame
    
    export function createEasingCurve(startValue: number, endValue: number, x1: number, y1: number, x2: number, y2: number): AnimationCurve
    

}

interface AnimationCurveTrack extends AnimationTrack {
    getPropertyKeys(): string[]
    
    setProperty(key: string, MISSING: any, UNKNOWNNAME: AnimationCurve): void
    
    getProperty(key: string): AnimationCurve
    
    evaluateNumber(time: number): number
    
    evaluateVec2(time: number): vec2
    
    evaluateVec3(time: number): vec3
    
    evaluateVec4(time: number): vec4
    
    evaluateRotation(time: number): quat
    
}

interface AnimationKeyFrame extends ScriptObject {
    value: number
    
    inWeightPoint: vec2
    
    outWeightPoint: vec2
    
    rightTangentType: TangentType
    
    leftTangentType: TangentType
    
    weightedMode: WeightedMode
    
    time: number
    
}

/**
* @description Configures an animation layer for a single [SceneObject](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject).
* Gives access to position, rotation, scale and blend shape animation tracks.
* See also: [Playing 3D Animation Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/additional-examples/playing-3d-animation), [AnimationMixer](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AnimationMixer), [Animation](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Animation).
*/
interface AnimationLayer extends AnimationTrack {
    /**
    * @description Returns a [FloatAnimationTrack](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#FloatAnimationTrack) from this AnimationLayer's blend shapes.
    */
    getBlendShapeTrack(shapeName: string): FloatAnimationTrack
    
    /**
    * @description Sets or adds a [FloatAnimationTrack](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#FloatAnimationTrack) to this AnimationLayer's blend shapes.
    */
    setBlendShapeTrack(shapeName: string, track: FloatAnimationTrack): void
    
    /**
    * @description The [Vec3AnimationTrack](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Vec3AnimationTrack) controlling position in this AnimationLayer.
    */
    position: Vec3AnimationTrack
    
    /**
    * @description The [QuaternionAnimationTrack](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#QuaternionAnimationTrack) controlling rotation in this AnimationLayer.
    */
    rotation: QuaternionAnimationTrack
    
    /**
    * @description The [Vec3AnimationTrack](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Vec3AnimationTrack) controlling scale in this AnimationLayer.
    */
    scale: Vec3AnimationTrack
    
    /**
    * @description The [IntAnimationTrack](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#IntAnimationTrack) controlling visibility in this AnimationLayer.
    */
    visibility: IntAnimationTrack
    
}

declare enum AnimationLayerBlendMode {
    Default,
    Additive
}

declare enum AnimationLayerScaleMode {
    Multiply,
    Additive
}

/**
* @description Controls playback of animations on the attached [SceneObject](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject) and its child objects.
* Please refer to the [Playing 3D Animation Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/additional-examples/playing-3d-animation) for setting up and playing animations.
*/
interface AnimationMixer extends Component {
    /**
    * @description Starts playing animation layers named `name`, or all layers if `name` is empty. The animation will start with an offset of `offset` seconds. The animation will play `cycles` times, or loop forever if `cycles` is -1.
    */
    start(name: string, offset: number, cycles: number): void
    
    /**
    * @description Starts playing animation layers named `name`, or all layers if `name` is empty. The animation will start with an offset of `offset` seconds. The animation will play `cycles` times, or loop forever if `cycles` is -1. `eventCallback` will be called after any animation layer finishes playing.
    */
    startWithCallback(name: string, offset: number, cycles: number, eventCallback: (name: string, animationMixer: AnimationMixer) => void): void
    
    /**
    * @description Stops any animation layer with name `name`, or all layers if `name` is empty.
    */
    stop(name: string): void
    
    /**
    * @description Pauses animation layers named `name`, or all layers if `name` is empty.
    */
    pause(name: string): void
    
    /**
    * @description Resumes any paused animation layer with name `name`, or all layers if `name` is empty.
    */
    resume(name: string): void
    
    /**
    * @description Makes a copy of the layer `name` and stores it as `newName`.
    */
    cloneLayer(name: string, newName: string): AnimationMixerLayer
    
    /**
    * @description Adds a new AnimationMixerLayer to this AnimationMixer.
    */
    createClip(name: string): AnimationMixerLayer
    
    /**
    * @description Returns the AnimationMixerLayer with the name `name`.
    */
    getLayer(name: string): AnimationMixerLayer
    
    /**
    * @description Returns a list of all AnimationMixerLayers controlled by the AnimationMixer.
    */
    getLayers(): AnimationMixerLayer[]
    
    /**
    * @description Rebuild the animation hierarchy by finding all [Animation](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Animation) components in the SceneObject and its children.
    */
    resetAnimations(): void
    
    /**
    * @description Returns the current time (in seconds) of the layer named `name`.
    */
    getLayerTime(name: string): number
    
    /**
    * @description Returns a list of names of AnimationLayers in this AnimationMixer.
    */
    getAnimationLayerNames(): string[]
    
    /**
    * @description Whether this AnimationMixer is set to automatically play animations on start.
    */
    autoplay: boolean
    
    /**
    * @description A multiplying value for the speed of all animations being controlled by the AnimationMixer.
    * For example, a value of 2.0 will double animation speed, while a value of 0.5 will cut the speed in half.
    */
    speedRatio: number
    
}

/**
* @description Controls animation playback for a single animation layer.
* See also: [AnimationMixer](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#AnimationMixer).
*/
interface AnimationMixerLayer extends ScriptObject {
    /**
    * @description Starts playing the animation with an offset of `offsetArg` seconds.
    * The animation will play `cycles` times, or loop forever if `cycles` is -1.
    */
    start(offset: number, cycles: number): void
    
    /**
    * @description Starts the animation with an offset of `offsetArg` seconds.
    * The animation will play `cycles` times, or loop forever if `cycles` is -1.
    * `eventCallback` will be called after the animation finishes.
    */
    startWithCallback(offset: number, cycles: number, eventCallback: (name: string, animationMixer: AnimationMixer) => void): void
    
    /**
    * @description Pauses the animation.
    */
    pause(): void
    
    /**
    * @description Resumes the animation if it has been paused.
    */
    resume(): void
    
    /**
    * @description Stops the animation from playing and jumps to the animation's end.
    */
    stop(): void
    
    /**
    * @description Returns the length of the animation in seconds.
    */
    getDuration(): number
    
    /**
    * @description Returns whether the animation is currently playing.
    */
    isPlaying(): boolean
    
    /**
    * @description Returns the current playback position of the animation in seconds.
    */
    getTime(): number
    
    /**
    * @description Returns a copy of this AnimationMixerLayer, with the name changed to `newName`.
    */
    clone(newName: string): AnimationMixerLayer
    
    /**
    * @description The name of the AnimationMixerLayer.
    */
    name: string
    
    /**
    * @description The name of the animation layer being used for this animation.
    */
    animationLayerName: string
    
    /**
    * @description The weight of this animation layer. Range is from [0-1], 0 being no animation strength and 1 being full animation strength.
    */
    weight: number
    
    /**
    * @description A multiplying value for the speed of this animation.
    * For example, a value of 2.0 will double animation speed, while a value of 0.5 will cut the speed in half.
    */
    speedRatio: number
    
    /**
    * @description The starting point for this animation clip.
    * If `rangeType` is set to `Time`, this is the point to start at in seconds.
    * If `rangeType` is set to `Frames`, this is the frame number to start at.
    */
    from: number
    
    /**
    * @description The ending point for this animation clip.
    * If `rangeType` is set to `Time`, this is the point to end at in seconds.
    * If `rangeType` is set to `Frames`, this is the frame number to end at.
    */
    to: number
    
    /**
    * @description The framerate (frames per second) of the animation.
    */
    fps: number
    
    /**
    * @description If true, the animation will play play in reverse.
    */
    reversed: boolean
    
    /**
    * @description The number of times this animation will play. If -1, the animation will loop forever.
    */
    cycles: number
    
    /**
    * @description If true, the animation will stop having an effect.
    */
    disabled: boolean
    
    blendMode: AnimationLayerBlendMode
    
    scaleMode: AnimationLayerScaleMode
    
    /**
    * @description The range type used for defining the animation clip.
    * If set to `AnimationClip.RangeType.Time`, `to` and `from` represent times in seconds.
    * If set to `AnimationClip.RangeType.Frames`, `to` and `from` represent frame numbers.
    */
    rangeType: AnimationClip.RangeType
    
    /**
    * @description Defines the animation's looping behavior.
    * If set to `AnimationClip.PostInfinity.Cycle`, the animation will restart from the beginning each time it loops.
    * If set to `AnimationClip.PostInfinity.Oscillate`, the animation will switch between normal and reverse playback each time it loops.
    * This is set to `Cycle` by default.
    */
    postInfinity: AnimationClip.PostInfinity
    
}

/**
* @description Component that handles playing animation clips as well as binding callbacks to user defined events.
*/
interface AnimationPlayer extends Component {
    /**
    * @description Plays all clips.
    */
    playAll(): void
    
    /**
    * @description Pauses all clips.
    */
    pauseAll(): void
    
    /**
    * @description Stops all clips and resets time to t = 0.
    */
    stopAll(): void
    
    /**
    * @description Resumes all clips.
    */
    resumeAll(): void
    
    /**
    * @description Tries to get a clip from the player, returns null if it does not exist.
    */
    getClip(name: string): AnimationClip
    
    /**
    * @description Plays clip with the given name and starting from the given time.
    */
    playClipAt(name: string, time: number): void
    
    /**
    * @description Pause the clip with name.
    */
    pauseClip(name: string): void
    
    /**
    * @description Stops the clip and resets time to t = 0.
    */
    stopClip(name: string): void
    
    /**
    * @description Resumes clip with name.
    */
    resumeClip(name: string): void
    
    /**
    * @description Adds a clip to the player. If one exists, replace existing clip.
    */
    addClip(clip: AnimationClip): void
    
    /**
    * @description Removes a clip from the player.
    */
    removeClip(name: string): void
    
    /**
    * @description Sets the clip to be enabled.
    */
    setClipEnabled(name: string, enabled: boolean): void
    
    /**
    * @description Get currently playing clips.
    */
    getActiveClips(): string[]
    
    /**
    * @description Get currently inactive clips.
    */
    getInactiveClips(): string[]
    
    /**
    * @description Returns if a clip is enabled for playback.
    */
    getClipEnabled(name: string): boolean
    
    /**
    * @description Returns if a clip is playing.
    */
    getClipIsPlaying(name: string): boolean
    
    /**
    * @description Returns the current time for a clip.
    */
    getClipCurrentTime(name: string): number
    
    /**
    * @description Updates the animation player forcing sampling, resulting in the setting of transforms and firing of animation events.
    */
    forceUpdate(deltaTime: number): void
    
    /**
    * @readonly
    
    * @description Array of animation clips
    */
    clips: AnimationClip[]
    
    /**
    * @readonly
    */
    onEvent: event1<any, void>
    
}

interface AnimationPropertyEventRegistration extends ScriptObject {
}

/**
* @description A layer containing different properties. Examples include position, rotation, scale or any other arbitrary properties a user would like to add and sample from.
*/
interface AnimationPropertyLayer extends ScriptObject {
}

/**
* @description The base class for animation tracks.
*/
interface AnimationTrack extends Asset {
}

/**
* @description Base class for all assets used in the engine.
*/
interface Asset extends SerializableWithUID {
    /**
    * @description The name of the Asset in Lens Studio.
    */
    name: string
    
}

/**
* @description Used by [Head.setAttachmentPointType()](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Head) to specify the type of attachment used with a Head binding.
*/
declare enum AttachmentPointType {
    HeadCenter,
    CandideCenter,
    TriangleBarycentric,
    LeftEyeballCenter,
    RightEyeballCenter,
    MouthCenter,
    Chin,
    Forehead,
    LeftForehead,
    RightForehead,
    LeftCheek,
    RightCheek
}

declare namespace Audio {
    /**
    * @description The curve that specifies how sound fades with the distance from Audio Component to the Audio Listener.****
    */
    enum DistanceCurveType {
        /**
        * @description Linear Curve ~ y(x) = ax+b
        */
        Linear,
        /**
        * @description Inverse Curve ~ y(x) = (a/x)+b
        */
        Inverse,
        /**
        * @description Logarithmic Curve ~ -log(x)
        */
        Logarithm,
        /**
        * @description Inverse Logarithmic Curve ~ +log(x)
        */
        InverseLogarithm
    }

}

/**
* @description Used to play audio in a Lens.
* You can assign an [AudioTrackAsset](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#AudioTrackAsset) to play through script or through the AudioComponent's inspector in Lens Studio.
* See the [Playing Audio](https://lensstudio.snapchat.com/guides/audio) guide for more information.
*/
interface AudioComponent extends Component {
    /**
    * @description Sets the callback function to be called whenever this sound stops playing.
    */
    setOnFinish(eventCallback: (audioComponent: AudioComponent) => void): void
    
    /**
    * @description Plays the current sound `loops` number of times.  If `loops` is -1, the sound will repeat forever.
    */
    play(loops: number): void
    
    /**
    * @description Stops the current sound if already playing.
    */
    stop(fade: boolean): void
    
    /**
    * @description Pauses the sound.
    */
    pause(): boolean
    
    /**
    * @description Resumes a paused sound.
    */
    resume(): boolean
    
    /**
    * @description Returns whether the AudioComponent is currently playing sound.
    */
    isPlaying(): boolean
    
    /**
    * @description Returns whether the sound is currently paused.
    */
    isPaused(): boolean
    
    /**
    * @description Length (in seconds) of a volume fade in applied to the beginning of sound playback.
    */
    fadeInTime: number
    
    /**
    * @description Length (in seconds) of a volume fade out applied to the end of sound playback.
    */
    fadeOutTime: number
    
    /**
    * @description When true, records sound directly into the snap. This mode works only when all Audio Components in the scene are using mix to snap. In this case input from microphone will be ignored.
    */
    mixToSnap: boolean
    
    /**
    * @description The volume of audio recorded to the snap, from 0 to 1.
    */
    recordingVolume: number
    
    /**
    * @readonly
    
    * @description Spatial Audio settings.
    */
    spatialAudio: SpatialAudio
    
    /**
    * @description The audio asset currently assigned to play.
    */
    audioTrack: AudioTrackAsset
    
    /**
    * @description A volume multiplier for any sounds played by this AudioComponent.
    */
    volume: number
    
    /**
    * @readonly
    
    * @description The length (in seconds) of the current sound assigned to play.
    */
    duration: number
    
    /**
    * @description The current playback time in seconds
    */
    position: number
    
}

/**
* @description Configures an audio effect for [AudioEffectComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AudioEffectComponent).
*/
interface AudioEffectAsset extends Asset {
}

/**
* @description Used to add an audio effect to a Lens.
* When present in the scene, it will automatically apply the selected audio effect to recordings made with the Lens.
* See the [Audio Effect](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/audio/audio-effect) guide for more information.

*/
interface AudioEffectComponent extends Component {
}

/**
* @description Provider for [AudioEffectAsset](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#AudioEffectAsset).
*/
interface AudioEffectProvider extends Provider {
}

/**
* @description A component that receives input from Audio Components that have Spatial Audio enabled. Calculates their positions relative to the scene object it is attached to,and properly mixes them.
*/
interface AudioListenerComponent extends Component {
}

/**
* @description Provider of the Audio Output Audio Track asset. 
*/
interface AudioOutputProvider extends AudioTrackProvider {
    /**
    * @description Preferred size for audio output at the current frame update.
    */
    getPreferredFrameSize(): number
    
    /**
    * @description Enqueue audio data into an audio playback system.
    
    * `shape.x` - buffer size, must be less or equal to `audioFrame` length.
    
    */
    enqueueAudioFrame(audioFrame: Float32Array, inShape: vec3): void
    
}

/**
* @description Represents an audio file asset.
* See also: [AudioComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AudioComponent).
*/
interface AudioTrackAsset extends Asset {
    /**
    * @description The provider for this audio track asset.
    */
    control: AudioTrackProvider
    
}

/**
* @description Base class for Audio Track providers.
*/
interface AudioTrackProvider extends Provider {
    /**
    * @description Sample rate (samples per second) of the audio track asset.
    */
    sampleRate: number
    
    /**
    * @readonly
    
    * @description The maximum frame size of the audio track asset.
    */
    maxFrameSize: number
    
}

/**
* @description Cardinal axis enumeration.
*/
declare enum Axis {
    /**
    * @description The X axis.
    */
    X,
    /**
    * @description The Y axis.
    */
    Y,
    /**
    * @description The Z axis.
    */
    Z
}

/**
* @description Settings for rendering the background on a [Text](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) component.
* Accessible through the [Text](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) component's `backgroundSettings` property.
*/
interface BackgroundSettings extends ScriptObject {
    /**
    * @description If enabled, the background will be rendered.
    */
    enabled: boolean
    
    /**
    * @description Settings for how the inside of the background is drawn.
    */
    fill: TextFill
    
    /**
    * @description Controls how far in each direction the background should extend away from the text.
    */
    margins: Rect
    
    /**
    * @description Controls how rounded the corner of the background should be.
    */
    cornerRadius: number
    
}

/**
* @description The base class for all mesh rendering components.
* Comparable to the former class "MeshVisual", which was split into the classes:
* [BaseMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#BaseMeshVisual),
* [MaterialMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MaterialMeshVisual),
* and [RenderMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#RenderMeshVisual).
*/
interface BaseMeshVisual extends Visual {
    /**
    * @description Projects screen positions from `camera`'s view onto the mesh's UVs.
    * If the MeshVisual's material uses the same texture as the camera input, the MeshVisual will look identical to the part of the screen it covers.
    */
    snap(camera: Camera): void
    
    /**
    * @description Range minimum of the world-space axis-aligned bounding box (AABB) of the visual.
    */
    worldAabbMin(): vec3
    
    /**
    * @description Range maximum of the world-space axis-aligned bounding box (AABB) of the visual.
    */
    worldAabbMax(): vec3
    
    /**
    * @description Range minimum of the local-space axis-aligned bounding box (AABB) of the visual.
    */
    localAabbMin(): vec3
    
    /**
    * @description Range maximum of the local-space axis-aligned bounding box (AABB) of the visual.
    */
    localAabbMax(): vec3
    
    /**
    * @description When a [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) is present on this SceneObject,
    * and `extentsTarget` is a child of this SceneObject, `extentsTarget` will be repositioned to match the exact
    * area this MeshVisual is being rendered. Very useful for [Image](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Image) and [Text](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) components.
    */
    extentsTarget: ScreenTransform
    
    /**
    * @description None = 0, Caster = 1, Receiver = 2
    */
    meshShadowMode: MeshShadowMode
    
    /**
    * @description Affects the color of shadows being cast by this MeshVisual. The color of the cast shadow is a mix between shadowColor and the material's base texture color. The alpha value of shadowColor controls the mixing of these two colors, with 0 = shadowColor and 1 = shadowColor * textureColor.
    */
    shadowColor: vec4
    
    /**
    * @description Density of shadows cast by this MeshVisual.
    */
    shadowDensity: number
    
    /**
    * @description When a [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) is attached to the same SceneObject, this controls how the mesh will be stretched relative to the ScreenTransform's boundaries.
    */
    stretchMode: StretchMode
    
    /**
    * @description When a [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) is attached to the same SceneObject, this controls how the mesh will be positioned vertically depending on `stretchMode`.
    */
    verticalAlignment: VerticalAlignment
    
    /**
    * @description When a [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) is attached to the same SceneObject, this controls how the mesh will be positioned horizontally depending on `stretchMode`.
    */
    horizontalAlignment: HorizontalAlignment
    
}

/**
* @description Base class for MultiplayerSession options. This class is not used directly - use ConnectedLensSessionOptions instead.
*/
interface BaseMultiplayerSessionOptions extends ScriptObject {
    /**
    * @description Function to be called when a connection to the realtime backend is established. All realtime requests can be called after this callback. When you invite others to join, a new session will be created, hence you should update your session handler with the argument passed in this callback.
    */
    onConnected: (session: MultiplayerSession, connectionInfo: ConnectedLensModule.ConnectionInfo) => void
    
    /**
    * @description Function to be called when the connection to the realtime backend is lost, either via successful disconnect or passive disconnect due to error.
    */
    onDisconnected: (session: MultiplayerSession, disconnectInfo: string) => void
    
    /**
    * @description Function to be called when a string-based message sent by sendMessage() is received from another user via the realtime backend.
    */
    onMessageReceived: (session: MultiplayerSession, userId: string, message: string, senderInfo: ConnectedLensModule.UserInfo) => void
    
    /**
    * @description Function to be called when another user joins the session. When joining a session, the current user will get a callback for each of the existing active users in the current session. This way you can build a list of existing players in game.
    */
    onUserJoinedSession: (session: MultiplayerSession, userInfo: ConnectedLensModule.UserInfo) => void
    
    /**
    * @description Function to be called when another user leaves the session, either deliberately or via passive disconnect due to error.
    */
    onUserLeftSession: (session: MultiplayerSession, userInfo: ConnectedLensModule.UserInfo) => void
    
    /**
    * @description Callback function that will be executed when a realtime store is created.
    */
    onRealtimeStoreCreated: (session: MultiplayerSession, store: GeneralDataStore, ownerInfo: ConnectedLensModule.UserInfo, creationInfo: ConnectedLensModule.RealtimeStoreCreationInfo) => void
    
    /**
    * @description Callback function that will be executed when a realtime store is updated.
    */
    onRealtimeStoreUpdated: (session: MultiplayerSession, store: GeneralDataStore, key: string, updateInfo: ConnectedLensModule.RealtimeStoreUpdateInfo) => void
    
    /**
    * @description Callback function that will be executed when a realtime store is deleted.
    */
    onRealtimeStoreDeleted: (session: MultiplayerSession, store: GeneralDataStore, deleteInfo: ConnectedLensModule.RealtimeStoreDeleteInfo) => void
    
    /**
    * @description Callback function that will be executed when ownership of a realtime store is updated.
    */
    onRealtimeStoreOwnershipUpdated: (session: MultiplayerSession, store: GeneralDataStore, ownerInfo: ConnectedLensModule.UserInfo, ownershipUpdateInfo: ConnectedLensModule.RealtimeStoreOwnershipUpdateInfo) => void
    
    /**
    * @description Function to be called when a key is removed from a RealtimeStore.
    */
    onRealtimeStoreKeyRemoved: (session: MultiplayerSession, removalInfo: ConnectedLensModule.RealtimeStoreKeyRemovalInfo) => void
    
    /**
    * @description Function to be called when an error occurs in the session life cycle.
    */
    onError: (session: MultiplayerSession, code: string, description: string) => void
    
    onHostUpdated: (session: MultiplayerSession, removalInfo: ConnectedLensModule.HostUpdateInfo) => void
    
    /**
    * @description Provides a single host for every session. Useful when an experience has a single authority. This should not be used in latency sensitive situations. The creator of the session will by default be the host. If the host leaves, thee server will determine a new host and transfer all ownership of entities owned by the original host. Use with `onHostUpdated` API to get a callback.
    */
    hostManagementEnabled: boolean
    
}

/**
* @description Base class for Input and Output Placeholders used by MLComponent.
*/
interface BasePlaceholder extends ScriptObject {
    /**
    * @readonly
    
    * @description The name of the Placeholder.
    */
    name: string
    
    /**
    * @readonly
    
    * @description The shape of the Placeholder's data.
    */
    shape: vec3
    
    /**
    * @readonly
    
    * @description Transformer object for applying transformations on the PlaceHolder's data.
    */
    transformer: Transformer
    
}

/**
* @description Provides basic information about a transformation.
* See also: [DeviceTracking](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#DeviceTracking)
*/
interface BasicTransform extends ScriptObject {
    /**
    * @description Returns the world matrix of the BasicTransform.
    */
    getMatrix(): mat4
    
    /**
    * @description Returns the inverted world matrix of the BasicTransform.
    */
    getInvertedMatrix(): mat4
    
    /**
    * @description Returns the world position of the BasicTransform.
    */
    getPosition(): vec3
    
    /**
    * @description Returns the world rotation of the BasicTransform.
    */
    getRotation(): quat
    
    /**
    * @description Returns the world scale of the BasicTransform.
    */
    getScale(): vec3
    
}

/**
* @description File based asset.
*/
interface BinAsset extends Asset {
}

interface Bitmoji2DOptions extends ScriptObject {
    user: SnapchatUser
    
    /**
    * @description The pose id for the 2D Bitmoji.
    */
    poseId: string
    
}
declare namespace Bitmoji2DOptions {
    export function create(): Bitmoji2DOptions
    

}

interface Bitmoji2DResource extends DynamicResource {
}

interface Bitmoji3DOptions extends ScriptObject {
    user: SnapchatUser
    
}
declare namespace Bitmoji3DOptions {
    export function create(): Bitmoji3DOptions
    

}

/**
* @description Provides information about the current user's 3D Bitmoji avatar to be downloaded via the RemoteMediaModule.
*/
interface Bitmoji3DResource extends DynamicResource {
}

/**
* @description Provides access to getting information about the current user's Bitmoji.
*/
interface BitmojiModule extends Asset {
    /**
    * @description Request the current user's 3D Bitmoji avatar.
    */
    requestBitmoji3DResource(callback: (resource: Bitmoji3DResource) => void): void
    
    requestBitmoji3DResourceWithOptions(options: Bitmoji3DOptions, callback: (resource: Bitmoji3DResource) => void): void
    
    /**
    * @description Returns a `DynamicResource` via the provided `callback`, which can be resolved into a texture using `RemoteMediaModule`. 
    */
    requestBitmoji2DResource(options: Bitmoji2DOptions, callback: (resource: Bitmoji2DResource) => void): void
    
}

declare enum BlendMode {
    Normal,
    Screen,
    PremultipliedAlpha,
    AlphaToCoverage,
    Disabled,
    Add,
    AlphaTest,
    ColoredGlass,
    Multiply,
    Min,
    Max,
    PremultipliedAlphaHardware,
    PremultipliedAlphaAuto
}

/**
* @description Used to analyze the camera input and apply similar image artifacts to your AR objects in order to allow it to blend and match the real world better.
*/
interface BlurNoiseEstimation extends Component {
}

/**
* @description Derived from ColliderComponent, attaching this to a SceneObject turns it into a dynamic rigid-body that is automatically moved by the physics simulation in response to gravity, collisions, and other forces.
*/
interface BodyComponent extends ColliderComponent {
    /**
    * @description Apply linear force at the object's center-of-mass.
    */
    addForce(force: vec3, mode: Physics.ForceMode): void
    
    /**
    * @description Apply force at a point offset from the object's origin, effectively generating torque.
    */
    addForceAt(force: vec3, offset: vec3, mode: Physics.ForceMode): void
    
    /**
    * @description Apply torque (angular force).
    */
    addTorque(torque: vec3, mode: Physics.ForceMode): void
    
    /**
    * @description Relative to local rotation, apply linear force at the object's center-of-mass.
    */
    addRelativeForce(force: vec3, mode: Physics.ForceMode): void
    
    /**
    * @description Relative to local rotation, apply force at a point offset from the object's origin, effectively generating torque.
    */
    addRelativeForceAt(force: vec3, position: vec3, mode: Physics.ForceMode): void
    
    /**
    * @description Relative to local rotation, apply torque (angular force).
    */
    addRelativeTorque(torque: vec3, mode: Physics.ForceMode): void
    
    /**
    * @description Add a point constraint between this body and the given collider, at the given position. `target` is optional. If null, it is statically constrained to the world.   This is a convenience function that creates a child SceneObject with a ConstraintComponent, and sets its type, target, and constraint parameters. To fully remove the constraint, its SceneObject must be removed - not the component. Call `removeConstraint()` to do this.
    */
    addPointConstraint(target: ColliderComponent, position: vec3): ConstraintComponent
    
    /**
    * @description Removes a constraint that was added with one of the `add*Constraint()` functions.
    */
    removeConstraint(constraint: ConstraintComponent): void
    
    /**
    * @description If enabled, the body is dynamically simulated, such that it responds to forces and collisions. Otherwise, it acts as a static collider, functionally equivalent to Physics.ColliderComponent.
    */
    dynamic: boolean
    
    /**
    * @description Mass (kg) of the object. If modified from this field, mass is constant irrespective of shape and scale.
    */
    mass: number
    
    /**
    * @description Density (kg/L) of the object, used to derive mass from volume. If modified, mass changes according to shape and scale.
    */
    density: number
    
    /**
    * @description Damping applied to linear velocity, in the range 0.0 (no damping) to 1.0 (maximum damping). This produces an effect similar to drag in that it causes the object to slow down over time. It is not however physically accurate, and it doesn't take into account surface area or mass.
    */
    damping: number
    
    /**
    * @description Damping applied to angular velocity, in the range 0.0 (no damping) to 1.0 (maximum damping).
    */
    angularDamping: number
    
}

/**
* @description Provides depth values of the tracked body encoded as D24_UNORM_S8_UINT in screen resolution. Depth is normalized between 0-1, and the stencil component is ignored. The values may be remapped from normalized units (0 to 1) to negative centimeters from the camera (-near in cm to -far in cm) using the `Depth Map` mode of the `Texture 2D Sample` node in Material Editor.

*/
interface BodyDepthTextureProvider extends TextureProvider {
    /**
    * @description The index of the body to track. The first body detected is `0`.
    */
    bodyIndex: number
    
    /**
    * @description A background depth (where confidence < `minimumConfidence`) is represented by 1.0 (zFar). A value from 0.0 to 1.0 (default value 0.5). 
    
    */
    minimumConfidence: number
    
    /**
    * @readonly
    
    * @description Near plane value in cm, Read only, always outputs 1.0.
    
    */
    zNear: number
    
    /**
    * @readonly
    
    * @description Far plane value in cm, Read only, always outputs 1000.0.
    
    */
    zFar: number
    
}

/**
* @description Provides surface normal values of the tracked body encoded as RGBA (x, y, z, confidence) in `R8G8B8A8_UNORM` format in screen resolution. The XYZ values may need to be remapped from [0 -> 1] to [-1 -> 1] using the "Normal Map" mode of the `Texture 2D Sample` node in Material Editor.
*/
interface BodyNormalsTextureProvider extends TextureProvider {
    /**
    * @description The index of the body to track. The first body detected is `0`.
    */
    bodyIndex: number
    
}

/**
* @description Provider for full Body Mesh render object.
*/
interface BodyRenderObjectProvider extends RenderObjectProvider {
    /**
    * @description Index of body in scene to track.
    */
    bodyIndex: number
    
    /**
    * @description Enable main Body Mesh geometry.
    */
    bodyGeometryEnabled: boolean
    
    /**
    * @description Enable left hand mesh.
    */
    leftHandGeometryEnabled: boolean
    
    /**
    * @description Enable right hand mesh.
    */
    rightHandGeometryEnabled: boolean
    
    /**
    * @description Enable head mesh.
    */
    headGeometryEnabled: boolean
    
}

/**
* @description Asset used to configure Body Tracking for the [ObjectTracking3D](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ObjectTracking3D) component.
*/
interface BodyTrackingAsset extends Object3DAsset {
    /**
    * @description When true, hand tracking will be enabled.
    */
    handTrackingEnabled: boolean
    
}
declare namespace BodyTrackingAsset {
    /**
    * @description Key for Hips attachment point.
    */
    let Hips: string
    
    /**
    * @description Key for Spine attachment point.
    */
    let Spine: string
    
    /**
    * @description Key for Spine1 attachment point.
    */
    let Spine1: string
    
    /**
    * @description Key for Spine2 attachment point.
    */
    let Spine2: string
    
    /**
    * @description Key for Neck attachment point.
    */
    let Neck: string
    
    /**
    * @description Key for Head attachment point.
    */
    let Head: string
    
    /**
    * @description Key for Left Shoulder attachment point.
    */
    let LeftShoulder: string
    
    /**
    * @description Key for Left Arm attachment point.
    */
    let LeftArm: string
    
    /**
    * @description Key for Left Forearm attachment point.
    */
    let LeftForeArm: string
    
    /**
    * @description Key for Left Hand attachment point.
    */
    let LeftHand: string
    
    /**
    * @description Key for Right Shoulder attachment point.
    */
    let RightShoulder: string
    
    /**
    * @description Key for Right Arm attachment point.
    */
    let RightArm: string
    
    /**
    * @description Key for Right Forearm attachment point.
    */
    let RightForeArm: string
    
    /**
    * @description Key for Right Hand attachment point.
    */
    let RightHand: string
    
    /**
    * @description Key for Upper Left Leg attachment point.
    */
    let LeftUpLeg: string
    
    /**
    * @description Key for Left Leg attachment point.
    */
    let LeftLeg: string
    
    /**
    * @description Key for Left Foot attachment point.
    */
    let LeftFoot: string
    
    /**
    * @description Key for Left Toe Base attachment point.
    */
    let LeftToeBase: string
    
    /**
    * @description Key for Upper Right Leg attachment point.
    */
    let RightUpLeg: string
    
    /**
    * @description Key for Right Leg attachment point.
    */
    let RightLeg: string
    
    /**
    * @description Key for Right Foot attachment point.
    */
    let RightFoot: string
    
    /**
    * @description Key for Right Toe Base attachment point.
    */
    let RightToeBase: string
    
    /**
    * @description Key for Left Hand Thumb 1 attachment point.
    */
    let LeftHandThumb1: string
    
    /**
    * @description Key for Left Hand Thumb 2 attachment point.
    */
    let LeftHandThumb2: string
    
    /**
    * @description Key for Left Hand Thumb 3 attachment point.
    */
    let LeftHandThumb3: string
    
    /**
    * @description Key for Left Hand Index 1 attachment point.
    */
    let LeftHandIndex1: string
    
    /**
    * @description Key for Left Hand Index 2 attachment point.
    */
    let LeftHandIndex2: string
    
    /**
    * @description Key for Left Hand Index 3 attachment point.
    */
    let LeftHandIndex3: string
    
    /**
    * @description Key for Left Hand Middle 1 attachment point.
    */
    let LeftHandMiddle1: string
    
    /**
    * @description Key for Left Hand Middle 2 attachment point.
    */
    let LeftHandMiddle2: string
    
    /**
    * @description Key for Left Hand Middle 3 attachment point.
    */
    let LeftHandMiddle3: string
    
    /**
    * @description Key for Left Hand Ring 1 attachment point.
    */
    let LeftHandRing1: string
    
    /**
    * @description Key for Left Hand Ring 2 attachment point.
    */
    let LeftHandRing2: string
    
    /**
    * @description Key for Left Hand Ring 3 attachment point.
    */
    let LeftHandRing3: string
    
    /**
    * @description Key for Left Hand Pinky 1 attachment point.
    */
    let LeftHandPinky1: string
    
    /**
    * @description Key for Left Hand Pinky 2 attachment point.
    */
    let LeftHandPinky2: string
    
    /**
    * @description Key for Left Hand Pinky 3 attachment point.
    */
    let LeftHandPinky3: string
    
    /**
    * @description Key for Right Hand Thumb 1 attachment point.
    */
    let RightHandThumb1: string
    
    /**
    * @description Key for Right Hand Thumb 2 attachment point.
    */
    let RightHandThumb2: string
    
    /**
    * @description Key for Right Hand Thumb 3 attachment point.
    */
    let RightHandThumb3: string
    
    /**
    * @description Key for Right Hand Index 1 attachment point.
    */
    let RightHandIndex1: string
    
    /**
    * @description Key for Right Hand Index 2 attachment point.
    */
    let RightHandIndex2: string
    
    /**
    * @description Key for Right Hand Index 3 attachment point.
    */
    let RightHandIndex3: string
    
    /**
    * @description Key for Right Hand Middle 1 attachment point.
    */
    let RightHandMiddle1: string
    
    /**
    * @description Key for Right Hand Middle 2 attachment point.
    */
    let RightHandMiddle2: string
    
    /**
    * @description Key for Right Hand Middle 3 attachment point.
    */
    let RightHandMiddle3: string
    
    /**
    * @description Key for Right Hand Ring 1 attachment point.
    */
    let RightHandRing1: string
    
    /**
    * @description Key for Right Hand Ring 2 attachment point.
    */
    let RightHandRing2: string
    
    /**
    * @description Key for Right Hand Ring 3 attachment point.
    */
    let RightHandRing3: string
    
    /**
    * @description Key for Right Hand Pinky 1 attachment point.
    */
    let RightHandPinky1: string
    
    /**
    * @description Key for Right Hand Pinky 2 attachment point.
    */
    let RightHandPinky2: string
    
    /**
    * @description Key for Right Hand Pinky 3 attachment point.
    */
    let RightHandPinky3: string
    

}

/**
* @description A box collision shape.
*/
interface BoxShape extends Shape {
    /**
    * @description The size of the box on each local axis.
    */
    size: vec3
    
}

/**
* @description Triggered when eyebrows are lowered on the tracked face.
*/
interface BrowsLoweredEvent extends FaceTrackingEvent {
}

/**
* @description Triggered when eyebrows are raised on the tracked face.
*/
interface BrowsRaisedEvent extends FaceTrackingEvent {
}

/**
* @description Triggered when eyebrows are returned to normal on the tracked face.
*/
interface BrowsReturnedToNormalEvent extends FaceTrackingEvent {
}

/**
* @description Renders the scene to a Render Target texture.
* A Camera will only render a SceneObject if the SceneObject's render layer is enabled on the Camera.
* For more information, see the [Camera and Layers](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/scene-set-up/camera) guide.
*/
interface Camera extends Component {
    /**
    * @description For orthographic cameras, returns the camera size as (width, height).
    */
    getOrthographicSize(): vec2
    
    /**
    * @description Converts a screen space position to a world space position, given an absolute depth.
    * The screen space position should be provided as a `vec2` in the range ([0-1], [0-1]),
    * (0,0) being the top-left of the screen and (1,1) being the bottom-right.
    * The returned world space position will be the point `absoluteDepth` units away from the Camera's
    * `near` plane at the point specified in screen space.
    */
    screenSpaceToWorldSpace(normalizedScreenSpacePoint: vec2, absoluteDepth: number): vec3
    
    /**
    * @description Converts the world space position `worldSpacePoint` to a screen space position.
    * Screen positions are represented in the range ([0-1], [0-1]), (0,0) being the top-left of the screen and (1,1) being the bottom-right.
    */
    worldSpaceToScreenSpace(worldSpacePoint: vec3): vec2
    
    /**
    * @description Converts a world space position to a raw screen space position.
    * The screen space position will be returned as a `vec3` with `x`,`y` representing normalized screen space,
    * and `z` representing a raw depth value not directly convertible to world units.
    * This returned value will mostly be useful for passing into `unproject()`.
    */
    project(worldSpacePoint: vec3): vec3
    
    /**
    * @description Converts a raw screen space position to a world space position.
    * `clipSpacePoint` should be a `vec3` returned from a previous `project()` call, since the
    * `z` value represents a raw depth value not directly convertible to world units.
    */
    unproject(clipSpacePoint: vec3): vec3
    
    /**
    * @description Returns true if a sphere with the specified world space center position and radius is visible within the camera frustum, false otherwise.
    
    */
    isSphereVisible(center: vec3, radius: number): boolean
    
    /**
    * @description The distance of the near clipping plane.
    */
    near: number
    
    /**
    * @description The distance of the far clipping plane.
    */
    far: number
    
    /**
    * @description The Camera's field of view in radians.
    */
    fov: number
    
    /**
    * @description The aspect ratio of the camera (width/height).
    */
    aspect: number
    
    /**
    * @description The orthographic size of the camera.
    */
    size: number
    
    /**
    * @description Controls which type of rendering the camera uses.
    
    */
    type: Camera.Type
    
    /**
    * @description Determines the way depth is handled on this Camera. Changing this can help sort objects at different distance ranges.
    */
    depthBufferMode: Camera.DepthBufferMode
    
    /**
    * @description Controls the set of layers this Camera will render.
    */
    renderLayer: LayerSet
    
    /**
    * @description Controls which Camera settings will be overridden by physical device properties.
    * For example, this can be used to override the `fov` property to match the device camera's actual field of view.
    */
    devicePropertyUsage: Camera.DeviceProperty
    
    /**
    * @description Returns an array of Color Render Targets. The first color render target is always available.
    */
    colorRenderTargets: Camera.ColorRenderTarget[]
    
    /**
    * @description Descriptor of depth/stencil textures and clear options.
    */
    depthStencilRenderTarget: Camera.DepthStencilRenderTarget
    
    /**
    * @description The sorting order the Camera renders in. Every frame, Cameras render in ascending order determined by their `renderOrder` properties.
    */
    renderOrder: number
    
    /**
    * @description A texture controlling which parts of the output texture the camera will draw to.
    * The "red" value of each pixel determines how strongly the camera will draw to that part of the image.
    * For example, a completely black section will cause the camera to not draw there at all. A completely
    * white (or red) section will cause the camera to draw normally. Colors in between, like gray, will be semitransparent.
    */
    maskTexture: Texture
    
    /**
    * @description When `enableClearColor` is true, this texture is used to clear this Camera's `renderTarget` before drawing.
    * If this texture is null, `clearColor` will be used instead.
    */
    inputTexture: Texture
    
    /**
    * @description The RenderTarget this Camera will draw to.
    */
    renderTarget: Texture
    
    /**
    * @description If enabled, this Camera will clear the depth buffer on its `renderTarget` before drawing to it.
    */
    enableClearDepth: boolean
    
    /**
    * @description When `enableClearColor` is true and `inputTexture` is null, this color is used to clear this Camera's `renderTarget` before drawing to it.
    */
    clearColor: vec4
    
    /**
    * @description Sets which face of the cubemap this camera will render to.
    */
    renderTargetCubemapFace: Camera.CubemapFace
    
    /**
    * @description If enabled, this Camera will clear the color on its `renderTarget` before drawing to it.
    * `inputTexture` will be used to clear it unless it is null, in which case `clearColor` is used instead.
    */
    enableClearColor: boolean
    
    /**
    * @description Toggles ray tracing for the camera. When true, ray tracing is enabled.
    */
    rayTracing: boolean
    
}
declare namespace Camera {
    /**
    * @description Return true if the device supports stencil operations and render to depth texture.
    */
    export function depthStencilRenderTargetSupported(): boolean
    
    /**
    * @description Returns the number of possible render target bindings to the camera.
    */
    export function getSupportedColorRenderTargetCount(): number
    
    /**
    * @description Creates and returns a new Color Render Target.
    */
    export function createColorRenderTarget(): Camera.ColorRenderTarget
    
    /**
    * @description Create a depth/stencil render target descriptor for the camera.
    */
    export function createDepthStencilRenderTarget(): Camera.DepthStencilRenderTarget
    

}

declare namespace Camera {
    /**
    * @description The base class from which ColorRenderTarget and DepthStencilRenderTarget are derived from
    */
    interface BaseRenderTarget extends ScriptObject {
        /**
        * @description Render target texture. Camera will render scene color values to this texture.
        */
        targetTexture: Texture
        
        /**
        * @description Color texture used in clear color operation in "CustomTexture" mode.
        */
        inputTexture: Texture
        
        /**
        * @description Texture used like a color mask for target texture.
        */
        maskTexture: Texture
        
    }

}

declare namespace Camera {
    /**
    * @description Color based RenderTarget.
    */
    interface ColorRenderTarget extends Camera.BaseRenderTarget {
        /**
        * @description Sets how the RenderTarget's color will be cleared before rendering during each frame.
        */
        clearColorOption: ClearColorOption
        
        /**
        * @description Sets the clear color of the camera when its `clearColorOption` is set to `ClearColorOption.CustomColor`
        */
        clearColor: vec4
        
    }

}

declare namespace Camera {
    /**
    * @description Different faces of the cubemap that a camera can render into.
    */
    enum CubemapFace {
        /**
        * @description The positive X cubemap face.
        */
        PositiveX,
        /**
        * @description The negative X cubemap face.
        */
        NegativeX,
        /**
        * @description The positive Y cubemap face.
        */
        PositiveY,
        /**
        * @description The negative Y cubemap face.
        */
        NegativeY,
        /**
        * @description The positive Z cubemap face.
        */
        PositiveZ,
        /**
        * @description The negative Z cubemap face.
        */
        NegativeZ,
        /**
        * @description The left cubemap face, same as NegativeX.
        */
        Left,
        /**
        * @description The right cubemap face, same as PositiveX.
        */
        Right,
        /**
        * @description The top cubemap face, same as PositiveY.
        */
        Top,
        /**
        * @description The bottom cubemap face, same as NegativeY.
        */
        Bottom,
        /**
        * @description The front cubemap face, same as PositiveZ.
        */
        Front,
        /**
        * @description The back cubemap face, same as NegativeZ.
        */
        Back
    }

}

declare namespace Camera {
    /**
    * @description Used in [Camera's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Camera) `depthBufferMode` property.
    * Each mode is suited for handling objects at a certain distance range.
    * For more information on depth modes, see the [Camera and Layers](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/scene-set-up/camera) guide.
    */
    enum DepthBufferMode {
        /**
        * @description Gives higher depth precision on nearby objects, so is better suited for scenes near to the camera.
        */
        Regular,
        /**
        * @description Gives higher depth precision on far away objects, so is better suited for scenes far away from the camera.
        */
        Logarithmic
    }

}

declare namespace Camera {
    /**
    * @description This class inherits from the BaseRenderTarget class. BaseRenderTarget class is not available for creation and is used like the base class for DepthStencilRenderTarget class to provide access to targetTexture, inputTexture and maskTexture properties.
    */
    interface DepthStencilRenderTarget extends Camera.BaseRenderTarget {
        /**
        * @description The same as "depthClearOption" property of DepthStencilRenderTargetProvider. But if Camera's depth clear option property is set to "CustomValue" or "CustomTexture" then this has priority over depth/stencil provider settings.
        */
        depthClearOption: DepthClearOption
        
        /**
        * @description The same as "stencilClearOption" property of DepthStencilRenderTargetProvider. But if Camera's stencil clear option property is set to "CustomValue" or "CustomTexture" then this has priority over depth/stencil provider settings.
        */
        stencilClearOption: StencilClearOption
        
        /**
        * @description Float value in range [0.0..1.0] used in depth buffer clear operation in "CustomValue" mode. The initial value is 1.0.
        */
        clearDepthValue: number
        
        /**
        * @description Unsigned int value in range [0..0xFF] used in stencil buffer clear operation in "CustomValue" mode. The initial value is 0.
        */
        clearStencilValue: number
        
    }

}

declare namespace Camera {
    /**
    * @description Used in [Camera's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Camera) `devicePropertyUsage` property.
    * Specifies which camera properties should be overridden by device properties.
    */
    enum DeviceProperty {
        /**
        * @description No Camera properties are overridden with device properties.
        */
        None,
        /**
        * @description Overrides the Camera's `aspect` property to use the device's aspect ratio instead.
        */
        Aspect,
        /**
        * @description Overrides the Camera's `fov` property to use the device's field of view instead.
        */
        Fov,
        /**
        * @description Overrides both `aspect` and `fov` with the device's properties.
        */
        All
    }

}

declare namespace Camera {
    /**
    * @description Returned from [Camera's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Camera) `type` property.
    */
    enum Type {
        /**
        * @description Simulates how perspective and depth perception work in the real world. Useful for rendering objects in 3D space.
        */
        Perspective,
        /**
        * @description Does not simulate perspective distortion. Useful for 2D effects like Images and Text.
        */
        Orthographic
    }

}

/**
* @description Triggered when the device's back facing camera becomes active.
*/
interface CameraBackEvent extends SceneEvent {
}

/**
* @description Triggered when the device's front facing camera becomes active.
*/
interface CameraFrontEvent extends SceneEvent {
}

interface CameraTextureProvider extends TextureProvider {
}

/**
* @description A 2D canvas anchored in 3D space that acts as the root of the ScreenTransform hierarchy. ScreenTransform SceneObjects can be placed on the Canvas, and the Canvas can be sized and placed anywhere in 3D space. It is like a painter’s canvas for ScreenTransforms.
*/
interface Canvas extends Component {
    /**
    * @description Get size of rectangle as (width, height)
    */
    getSize(): vec2
    
    /**
    * @description Set size of rectangle as (width, height)
    */
    setSize(value: vec2): void
    
    /**
    * @description World Space Rectangle that defines the Canvas as offsets in world units from the SceneObject's position
    */
    worldSpaceRect: Rect
    
    /**
    * @description The point about which the Canvas Rectangle will rotate. Defined as fractional coordinates of the Canvas's dimensions. e.g. (1 , 1) brings the pivot to the top right corner of the canvas. Or (0.5, 0) moves the pivot to the right by half the canvas width.
    */
    pivot: vec2
    
    unitType: Canvas.UnitType
    
    sortingType: Canvas.SortingType
    
}

declare namespace Canvas {
    enum SortingType {
        Hierarchy,
        Depth
    }

}

declare namespace Canvas {
    enum UnitType {
        World,
        Pixels,
        Points
    }

}

/**
* @description Changes the capitalization of the text component. Useful when using dynamic texts.
*/
declare enum CapitilizationOverride {
    /**
    * @description Display the capitalization of the displayed text as provided.
    */
    None,
    /**
    * @description Sets the capitalization of the displayed text to lowercase.
    */
    AllLower,
    /**
    * @description Sets the capitalization of the displayed text to uppercase.
    */
    AllUpper
}

/**
* @description A capsule collision shape. Also known as a capped cylinder.
*/
interface CapsuleShape extends Shape {
    /**
    * @description Lengthwise local axis along which the capsule is oriented.
    */
    axis: Axis
    
    /**
    * @description Length of the capsule along its local axis. This is the distance between the two end-cap centers.
    */
    length: number
    
    /**
    * @description Radius of the capsule cylinder, and its end-cap spheres.
    */
    radius: number
    
}

/**
* @description Settings for how color will be cleared before rendering.
*/
declare enum ClearColorOption {
    /**
    * @description The frame will not be cleared before being rendered to (draws over the previous frame).
    */
    None,
    /**
    * @description Use the Device Texture for the color color
    */
    Background,
    /**
    * @description The frame will be cleared with a color before being rendered to.
    */
    CustomColor,
    /**
    * @description The frame will be cleared with a texture before being rendered to.
    */
    CustomTexture
}

/**
* @description Clears depth in the drawing order.
*/
interface ClearDepth extends Visual {
}

/**
* @description Handles the mesh data of cloth and prepares it for cloth simulation. Also controls all the parameters of the cloth simulator and colliders.
*/
interface ClothVisual extends MaterialMeshVisual {
    /**
    * @description Returns true if the Cloth Simulation feature is supported by the current device.
    */
    isHardwareSupported(): boolean
    
    /**
    * @description Returns true if the Cloth Simulation and resources are initialized. Always return false if device is not supported.
    */
    isInitialized(): boolean
    
    /**
    * @description Resets the cloth simulation.
    */
    resetSimulation(): void
    
    /**
    * @description Clears the colliders list.
    */
    clearColliders(): void
    
    /**
    * @description Adds a collider to the list of colliders.
    */
    addCollider(colliderComponent: ColliderComponent): void
    
    /**
    * @description Removes collider by its index and returns the removed collider.
    */
    removeColliderByIndex(index: number): ColliderComponent
    
    /**
    * @description Returns all the indices of vertices that are labeled by this color.
    */
    getPointIndicesByColor(color: vec4, colorMask: vec4b): number[]
    
    /**
    * @description Returns all the indices on the cloth mesh that are matching the color mask. Match means that the color has value on the channels which in colorMask is true.
    */
    getPointIndicesByMask(colorMask: vec4b): number[]
    
    /**
    * @description Returns the vertex color by vertex index.
    */
    getPointColorByIndex(index: number): vec4
    
    /**
    * @description Returns all available vertex colors on cloth mesh.
    */
    getAllColors(): vec4[]
    
    /**
    * @description Sets the binding SceneObject for the vertex.
    */
    setVertexBinding(index: number, bindingObj: SceneObject): void
    
    /**
    * @description Gets binding SceneObject of the vertex.
    */
    getVertexBinding(index: number): SceneObject
    
    /**
    * @description Gets simulation settings of the vertex.
    */
    getVertexSettings(index: number): VertexSimulationSettings
    
    /**
    * @description Sets simulation settings of the vertex.
    */
    setVertexSettings(index: number, vertexSettings: VertexSimulationSettings): void
    
    /**
    * @description Function called when the ClothVisual is initialized.
    */
    onInitialized: (clothVisual: ClothVisual) => void
    
    /**
    * @description List of colliders assigned.
    */
    colliders: ColliderComponent[]
    
    /**
    * @description Attached mesh.
    */
    mesh: RenderMesh
    
    /**
    * @readonly
    
    * @description Returns the modified simulated mesh which can be used in another [RenderMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#RenderMeshVisual) if the same simulated mesh is needed. Useful when creating effects that might require the same mesh to be rendered twice, such as with a mirror effect. Prevents the need to run a simulation twice. 
    
    */
    simulatedMesh: RenderMesh
    
    /**
    * @description Select bend mode to use (Isometric bending/Linear bending).
    */
    bendMode: ClothVisual.BendMode
    
    /**
    * @description Gravity force vector.
    */
    gravity: vec3
    
    /**
    * @description Number of simulation iterations to perform per each step. Higher number contributes to higher quality, but can be difficult for device performance.
    */
    iterationsPerStep: number
    
    /**
    * @description Mass value.
    */
    mass: number
    
    /**
    * @description Mass value weight on all the VertexSettings.
    */
    massVertexWeight: number
    
    /**
    * @description StretchStiffness Value.
    */
    stretchStiffness: number
    
    /**
    * @description StretchStiffness Value weight on all the VertexSettings.
    */
    stretchStiffnessVertexWeight: number
    
    /**
    * @description BendStiffness value.
    */
    bendStiffness: number
    
    /**
    * @description BendStiffness value weight on all the VertexSettings.
    */
    bendStiffnessVertexWeight: number
    
    /**
    * @description Friction value.
    */
    friction: number
    
    /**
    * @description Friction value weight on all the VertexSettings.
    */
    frictionVertexWeight: number
    
    /**
    * @description Indicates whether we will enable collision repulsion with collider models.
    */
    repulsionEnabled: boolean
    
    /**
    * @description Indicates the offset we set when the cloth mesh is too close to the colliders.
    */
    repulsionOffset: number
    
    /**
    * @description Indicates the stiffness of repulsion when collision.
    */
    repulsionStiffness: number
    
    /**
    * @description Collision friction to dampen relative motion.
    */
    repulsionFriction: number
    
    /**
    * @description Control for the acceleration on motion of vertices in cloth simulation, by default set to 2000.
    */
    maxAcceleration: number
    
    /**
    * @description Whether to update normals for the cloth mesh each frame in order to get reflection update.
    */
    updateNormalsEnabled: boolean
    
    /**
    * @description Whether to merge close vertices (Lens Studio might split vertices when loading FBX mesh). Changing this resets the simulation.
    */
    mergeCloseVerticesEnabled: boolean
    
    /**
    * @description Threshold of close vertices. Changing this value resets the simulation.
    */
    mergeCloseVerticesThreshold: number
    
    /**
    * @description The influence of the external body mesh acting on the cloth visual.
    */
    externalBodyMeshWeight: number
    
}
declare namespace ClothVisual {
    /**
    * @description Creates a new instance of vertex simulation settings.
    */
    export function createVertexSettings(): VertexSimulationSettings
    

}

declare namespace ClothVisual {
    /**
    * @description Cloth bend mode to use for cloth simulation.
    */
    enum BendMode {
        Isometric,
        Linear
    }

}

/**
* @description Options associated with the listValues method call.
*/
interface CloudStorageListOptions extends ScriptObject {
    /**
    * @description The scope of the listValues request. Required
    */
    scope: StorageScope
    
    /**
    * @description The position in the resulting list. Each time a list is requested, 10 entities are returned. If no cursor is provided, first 10 entities will be returned.
    */
    cursor: string
    
}
declare namespace CloudStorageListOptions {
    /**
    * @description Creates the listValues options object for Cloud Storage
    */
    export function create(): CloudStorageListOptions
    

}

/**
* @description Provides access to Cloud Storage.
*/
interface CloudStorageModule extends Asset {
    /**
    * @description Get the Cloud Store. Future calls to this method will return the same cloud store, even if the options change.
    */
    getCloudStore(options: CloudStorageOptions, onCloudStoreReady: (store: CloudStore) => void, onError: (code: string, description: string) => void): void
    
}

/**
* @description Used to configure `Cloud Storage Module` with various options. Note: if `session` scoped storage is required, this option must be provided.

*/
interface CloudStorageOptions extends ScriptObject {
    /**
    * @description If session scoped storage is required, set this property with the session object from [Connected Lenses Module](https://lensstudio.snapchat.com/api/classes/ConnectedLensModule).
    
    */
    session: MultiplayerSession
    
}
declare namespace CloudStorageOptions {
    /**
    * @description Create options for use with Cloud Storage.
    */
    export function create(): CloudStorageOptions
    

}

/**
* @description Options associated with the getValue/deleteValue methods for Cloud Storage.
*/
interface CloudStorageReadOptions extends ScriptObject {
    /**
    * @description The scope of the request.
    */
    scope: StorageScope
    
}
declare namespace CloudStorageReadOptions {
    /**
    * @description Options associated with the getValue/deleteValue methods for Cloud Storage.
    */
    export function create(): CloudStorageReadOptions
    

}

/**
* @description Options associated with the setValue method for Cloud Storage.
*/
interface CloudStorageWriteOptions extends ScriptObject {
    /**
    * @description The scope of the write option.
    */
    scope: StorageScope
    
}
declare namespace CloudStorageWriteOptions {
    /**
    * @description Creates the setValue options object for Cloud Storage.
    */
    export function create(): CloudStorageWriteOptions
    

}

/**
* @description An instance of Cloud Storage that can store data in a multiplayer experience.
*/
interface CloudStore extends ScriptObject {
    /**
    * @description Gets a value from the persistence backend. Note that scope must match that which was used when the value was originally saved.
    */
    getValue(key: string, readOptions: CloudStorageReadOptions, onRetrieved: (key: string, value: (mat4|mat3|mat2|vec4|vec2|vec3|boolean|quat|number|string)) => void, onError: (code: string, description: string) => void): void
    
    /**
    * @description Lists values from the persistence backend. Note that the scope must match that which was used when the values were originally saved.
    */
    listValues(listOptions: CloudStorageListOptions, onRetrieved: (values: (mat4|mat3|mat2|vec4|vec2|vec3|boolean|quat|number|string)[][], cursor: string) => void, onError: (code: string, description: string) => void): void
    
    /**
    * @description Sets a value in the persistence backend.
    */
    setValue(key: string, value: (mat4|mat3|mat2|vec4|vec2|vec3|boolean|quat|number|string), writeOptions: CloudStorageWriteOptions, onSaved: () => void, onError: (code: string, description: string) => void): void
    
    /**
    * @description Deletes a value from the persistence backend. Note that the scope must match that which was used when the value was originally saved.
    */
    deleteValue(key: string, readOptions: CloudStorageReadOptions, onDeleted: () => void, onError: (code: string, description: string) => void): void
    
}

/**
* @description Collider used by the [Hair Visual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#HairVisual) for its simulation.
*/
interface ColliderComponent extends Component {
    /**
    * @description Clears all velocities and forces on a collider.
    */
    clearMotion(): void
    
    /**
    * @description The Shape object used for collision.
    */
    shape: Shape
    
    /**
    * @description The matter used by the collider to define its physical substance, such as friction and bounciness.
    */
    matter: Matter
    
    /**
    * @description Collision filter to use for this collider.
    */
    filter: Filter
    
    /**
    * @description The WorldSettingsAsset used by the collider to define the physics simulation settings.
    */
    worldSettings: Physics.WorldSettingsAsset
    
    /**
    * @description Toggle collider wire rendering for visualizing collider geometry shape and where it is placed.
    */
    debugDrawEnabled: boolean
    
    /**
    * @description If enabled, the collider shape extends to fit the visual mesh, if any. Only applicable for Box and Sphere shapes.
    */
    fitVisual: boolean
    
    /**
    * @description Nested collider/body components may be merged into compound shapes. By default, this occurs only for dynamic bodies and not for static colliders. If `forceCompound` is set, this occurs for static colliders as well.
    */
    forceCompound: boolean
    
    /**
    * @description If enabled, the collider doesn't interact with the world but may still be detected with ray casts and intersection events.
    */
    intangible: boolean
    
    /**
    * @description Smooth transform changes using a dampened spring. Useful to reduce motion noise. This only applies to the simulation and changes from outside the simulation. The scene object's transform is not affected. It has no effect for dynamic bodies. 
    */
    smooth: boolean
    
    /**
    * @description Smoothing spring factor, for translation. This controls the restitution strength of the spring, so low values are smoother but lag more.
    */
    translateSmoothFactor: number
    
    /**
    * @description Smoothing spring factor, for rotation.
    */
    rotateSmoothFactor: number
    
    /**
    * @description Linear velocity (cm/s).
    */
    velocity: vec3
    
    /**
    * @description Angular velocity, expressed as an axis of rotation scaled by angular speed (radians/s).
    */
    angularVelocity: vec3
    
    /**
    * @description Collision filter used for overlap events.
    */
    overlapFilter: Filter
    
    /**
    * @readonly
    
    * @description Signals when objects initially collide.
    */
    onCollisionEnter: event1<CollisionEnterEventArgs, void>
    
    /**
    * @readonly
    
    * @description Signals every frame while objects continue to collide.
    */
    onCollisionStay: event1<CollisionStayEventArgs, void>
    
    /**
    * @readonly
    
    * @description Signals when objects stop colliding.
    */
    onCollisionExit: event1<CollisionExitEventArgs, void>
    
    /**
    * @readonly
    
    * @description Signals when colliders first overlap.
    */
    onOverlapEnter: event1<OverlapEnterEventArgs, void>
    
    /**
    * @readonly
    
    * @description Signals every frame while colliders continue to overlap.
    */
    onOverlapStay: event1<OverlapStayEventArgs, void>
    
    /**
    * @readonly
    
    * @description Signals when colliders stop overlapping.
    */
    onOverlapExit: event1<OverlapExitEventArgs, void>
    
}

/**
* @description A state generated for ColliderComponent collision events.
*/
interface Collision extends ScriptObject {
    /**
    * @readonly
    
    * @description Primitive shapes that physics objects interact with, such as spheres, boxes, and meshes, or compound shapes formed from multiple shapes. Used with Physics, Hair Simulation and Cloth Simulation.
    */
    collider: ColliderComponent
    
    /**
    * @readonly
    
    * @description ID of the collision, unique for this collider.
    */
    id: number
    
    /**
    * @readonly
    
    * @description The number of contact points in the collision.
    */
    contactCount: number
    
    /**
    * @readonly
    
    * @description Array of contacts in the collision.
    */
    contacts: Contact[]
    
}

/**
* @description Args used for [ColliderComponent.onCollisionEnter](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), which is triggered when a collision begins.
*/
interface CollisionEnterEventArgs extends ScriptObject {
    /**
    * @readonly
    
    * @description Structure containing information about the current collision.
    */
    collision: Collision
    
}

/**
* @description Args used for [ColliderComponent.onCollisionExit](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), which is triggered when a collision ends.
*/
interface CollisionExitEventArgs extends ScriptObject {
    /**
    * @readonly
    
    * @description Structure containing information about the current collision.
    */
    collision: Collision
    
}

interface CollisionMesh extends Asset {
}

/**
* @description Args used for [ColliderComponent.onCollisionStay](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), which is triggered every frame while a collision continues.
*/
interface CollisionStayEventArgs extends ScriptObject {
    /**
    * @readonly
    
    * @description Structure containing information about the current collision.
    */
    collision: Collision
    
}

/**
* @description Not usable from JS. Accessed via the Colocated Landmarks 2D Mesh resource. Contains the 2D keypoints when creating a Colocated map. Expanded by the material provided with template.
*/
interface ColocatedLandmarks2DRenderObjectProvider extends ColocatedLandmarksRenderObjectProviderBase {
}

/**
* @description Not usable from JS. Accessed via the Colocated Landmarks 3D Mesh resource. Contains the 3D landmarks when creating a Colocated map. Expanded by the material provided with template.
*/
interface ColocatedLandmarks3DRenderObjectProvider extends ColocatedLandmarksRenderObjectProviderBase {
}

interface ColocatedLandmarksRenderObjectProviderBase extends RenderObjectProvider {
}

/**
* @description Creates Colocated Connected Lenses experiences by enabling the creation and tracking of a shared space which can be used to place several users in the same coordinate frame. This shared space will be made available and can be tracked by any friend you invite to join your session via Snapcode. Users are expected to be located in the same room when using the colocated feature. This component needs to be attached to the camera. 
*/
interface ColocatedTrackingComponent extends Component {
    /**
    * @description Joins an existing session, retrieving the shared space that the colocated session initiator just created. Throws an exception if a join or build operation is in progress or if a shared space is already present.
    */
    join(session: MultiplayerSession): void
    
    /**
    * @description Starts the shared space building process locally in the session initiator's device.
    
    * If a session is provided, the shared space is placed into the session after building has completed, which is required for Colocated experiences running on mobile devices. When running the experience in Lens Studio Preview or solo mode the session is not required.
    
    * Throws an exception if a join or build operation is in progress or if a map is already present.
    */
    startBuilding(session: MultiplayerSession): void
    
    /**
    * @readonly
    
    * @description Indicates whether a shared space can be built on this device. Colocated Tracking is available on devices with ARKit/ARCore in the rear camera. This flag should be checked before attempting to build or join a session. If it is false, you should inform users: "Sorry, your device does not support shared AR experiences."
    */
    canBuild: boolean
    
    /**
    * @readonly
    
    * @description Indicates whether a shared space is present and ready to be tracked.
    */
    canTrack: boolean
    
    /**
    * @readonly
    
    * @description Indicates whether a shared space building operation is in progress. Once the flag is false, the shared space still needs to be shared. Use `onTrackingAvailable` to determine when your space has been shared.
    */
    isBuilding: boolean
    
    /**
    * @readonly
    
    * @description Indicates whether a session joining operation is in progress.
    */
    isJoining: boolean
    
    /**
    * @readonly
    
    * @description Indicates whether a shared space is actively being tracked. This value will be true while tracking is active, either with or without the shared space in view. This property matches the equivalent one in `MarkerTrackingComponent`.
    */
    isTracking: boolean
    
    /**
    * @readonly
    
    * @description Event fired when the building operation fails (for example, sharing your space failed). Once `onTrackingAvailable` event is triggered, this event will not be triggered anymore. 
    */
    onBuildFailed: event0<void>
    
    /**
    * @readonly
    
    * @description Event fired when a shared space starts being actively tracked. This property is an analogue of the `onMarkerFound` property in `MarkerTrackingComponent`.
    */
    onFound: event0<void>
    
    /**
    * @readonly
    
    * @description Event fired when a join operation completes but no shared space was found in the session.
    */
    onJoinFailed: event0<void>
    
    /**
    * @readonly
    
    * @description Event fired when a shared space stops being actively tracked. This property is an analogue of the `onMarkerLost` property in `MarkerTrackingComponent`.
    */
    onLost: event0<void>
    
    /**
    * @readonly
    
    * @description Event fired when a shared space has been shared or received and the device can attempt to start tracking.  If this event is not being triggered for a long period, the process will time out and `onBuildFailed` event will be triggered.
    */
    onTrackingAvailable: event0<void>
    
    /**
    * @readonly
    
    * @description Provides the shared space building progress expressed in values 0-1. These values can be used to populate a progress bar. Once this value has reached 1, the colocated tracking component attempts to share the space which the user created.
    */
    buildingProgress: number
    
}

/**
* @description Data type used for color values.
*/
declare enum Colorspace {
    /**
    * @description Color data has one value: Red
    */
    R,
    /**
    * @description Color data has 2 values: Red, Green
    */
    RG,
    /**
    * @description Color data has 4 values: Red, Green, Blue, Alpha
    */
    RGBA
}

/**
* @description The base class for all components.  Components are attached to [SceneObjects](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject).
*/
interface Component extends SerializableWithUID {
    /**
    * @description Destroys the component.
    */
    destroy(): void
    
    /**
    * @description Returns the SceneObject the component is attached to.
    */
    getSceneObject(): SceneObject
    
    /**
    * @description Returns the Transform this component is attached to.
    */
    getTransform(): Transform
    
    /**
    * @description If disabled, the Component will stop enacting its behavior.
    */
    enabled: boolean
    
    /**
    * @readonly
    */
    sceneObject: SceneObject
    
}

/**
* @description A cone collision shape.
*/
interface ConeShape extends Shape {
    /**
    * @description Lengthwise local axis along which the cone is oriented.
    */
    axis: Axis
    
    /**
    * @description The length of the cone.
    */
    length: number
    
    /**
    * @description The radius of the cone.
    */
    radius: number
    
}

/**
* @description Event fired when the "Launch connected lens" button was pressed. Wait for this event to be triggered before creating a session, as having this event gaurantees the user has accepted the necessary disclosures to use a connected lens experience.
*/
interface ConnectedLensEnteredEvent extends SceneEvent {
}

/**
* @description Connected Lenses Module allows use of networked Lens communication capabilities (real-time communication, co-located session creation and joining, and shared persistent storage). It's recommended to only use one ConnectedLensModule per Lens.
*/
interface ConnectedLensModule extends Asset {
    /**
    * @description Create session with the provided options. Will also check if there is a session that can be created from a session sharetype received from other users.
    */
    createSession(sessionOptions: ConnectedLensSessionOptions): void
    
    /**
    * @description Share the session with other users, specified by the SessionShareType. Note that if shared via Invitation, a new session will be created. Expect a new onConnected callback with the new session being passed in.
    */
    shareSession(sessionShareType: ConnectedLensModule.SessionShareType, onSessionShared: (session: MultiplayerSession, snapcode: Texture) => void): void
    
}

declare namespace ConnectedLensModule {
    /**
    * @description Information that is bootstrapped to the user who just connected to the session.
    */
    interface ConnectionInfo extends ScriptObject {
        /**
        * @readonly
        
        * @description Get the information about the local user.
        */
        localUserInfo: ConnectedLensModule.UserInfo
        
        /**
        * @readonly
        */
        hostUserInfo: ConnectedLensModule.UserInfo
        
        /**
        * @readonly
        
        * @description Get the information about all the other users connected to the same session. 
        */
        externalUsersInfo: ConnectedLensModule.UserInfo[]
        
        /**
        * @readonly
        
        * @description Get all the Realtime Stores that are being used in the session.
        */
        realtimeStores: GeneralDataStore[]
        
        /**
        * @readonly
        
        * @description Provides creation info about every existing RealtimeStore.
        */
        realtimeStoresCreationInfos: ConnectedLensModule.RealtimeStoreCreationInfo[]
        
    }

}

declare namespace ConnectedLensModule {
    /**
    * @description Information about the host update.
    */
    interface HostUpdateInfo extends ScriptObject {
        /**
        * @readonly
        
        * @description Information about the new host user. 
        */
        userInfo: ConnectedLensModule.UserInfo
        
        /**
        * @readonly
        
        * @description Provides the server timestamp (in milliseconds) of when the host was updated.
        */
        sentServerTimeMilliseconds: number
        
    }

}

declare namespace ConnectedLensModule {
    /**
    * @description Provides extra context about a RealtimeStore's creation.
    */
    interface RealtimeStoreCreationInfo extends ScriptObject {
        /**
        * @readonly
        
        * @description A string that can be used to identify the RealtimeStore.
        */
        storeId: string
        
        /**
        * @readonly
        
        * @description The persistence setting that the store was created with.
        */
        persistence: RealtimeStoreCreateOptions.Persistence
        
        /**
        * @readonly
        
        * @description If true, ownership of the store can be claimed even if the store is already owned.
        */
        allowOwnershipTakeOver: boolean
        
        /**
        * @readonly
        
        * @description Provides the server timestamp (in milliseconds) of the last time the store was updated.
        */
        lastUpdatedServerTimestamp: number
        
        /**
        * @readonly
        
        * @description Provides the server timestamp (in milliseconds) of when the store was created.
        */
        sentServerTimeMilliseconds: number
        
        /**
        * @readonly
        
        * @description Provides the UserInfo of the current owner of the RealtimeStore. If the store is unowned, a UserInfo object with null fields will be returned.
        
        */
        ownerInfo: ConnectedLensModule.UserInfo
        
    }

}

declare namespace ConnectedLensModule {
    /**
    * @description Gives information about the Realtime Store delete operation. 
    */
    interface RealtimeStoreDeleteInfo extends ScriptObject {
        /**
        * @readonly
        
        * @description Get the information of the user who deleted the Store.
        */
        deleterInfo: ConnectedLensModule.UserInfo
        
        /**
        * @readonly
        
        * @description Provides the server timestamp (in milliseconds) of when the store was deleted.
        */
        sentServerTimeMilliseconds: number
        
    }

}

declare namespace ConnectedLensModule {
    /**
    * @description Provides information about a key being removed from a RealtimeStore.
    */
    interface RealtimeStoreKeyRemovalInfo extends ScriptObject {
        /**
        * @readonly
        
        * @description The RealtimeStore that the key was removed from.
        */
        store: GeneralDataStore
        
        /**
        * @readonly
        
        * @description Key of the property that was removed.
        */
        key: string
        
        /**
        * @readonly
        
        * @description User that removed the key.
        */
        removerInfo: ConnectedLensModule.UserInfo
        
        /**
        * @readonly
        
        * @description Provides the server timestamp (in milliseconds) of when the key was removed.
        */
        sentServerTimeMilliseconds: number
        
    }

}

declare namespace ConnectedLensModule {
    /**
    * @description Provides information about a RealtimeStore's ownership being updated.
    */
    interface RealtimeStoreOwnershipUpdateInfo extends ScriptObject {
        /**
        * @readonly
        
        * @description Provides the server timestamp (in milliseconds) of when the store ownership was updated.
        */
        sentServerTimeMilliseconds: number
        
    }

}

declare namespace ConnectedLensModule {
    /**
    * @description Gives information about the Realtime Store update operation.
    */
    interface RealtimeStoreUpdateInfo extends ScriptObject {
        /**
        * @readonly
        
        * @description Get the information of the user who updated the store.
        */
        updaterInfo: ConnectedLensModule.UserInfo
        
        /**
        * @readonly
        
        * @description Provides the server timestamp (in milliseconds) of when the store was updated.
        */
        sentServerTimeMilliseconds: number
        
    }

}

declare namespace ConnectedLensModule {
    /**
    * @description Used by ConnectedLensesModule to specify the session share type.
    */
    enum SessionShareType {
        /**
        * @description Share a session by inviting your friends, launching the "send to" screen.
        */
        Invitation,
        /**
        * @description Share session via Snapcode that your friends can scan.
        */
        Snapcode
    }

}

declare namespace ConnectedLensModule {
    /**
    * @description Provides information about a user in a Connected Lens session.
    */
    interface UserInfo extends ScriptObject {
        /**
        * @readonly
        
        * @description A unique identifier for each participant of connected lens experience. It is unique per lens for each user.
        */
        userId: string
        
        /**
        * @readonly
        */
        connectionId: string
        
        /**
        * @readonly
        
        * @description The current display name of the user.
        */
        displayName: string
        
        /**
        * @readonly
        
        * @description Provides the server timestamp (in milliseconds) that the user joined the session.
        */
        joinServerTimeMilliseconds: number
        
    }

}

/**
* @description Settings for configuring a Connected Lens session.
*/
interface ConnectedLensSessionOptions extends DirectMultiplayerSessionOptions {
    /**
    * @description Defines maximum number of receipients that a user of connected lens can select while sharing a connected lens session via Invitation flow.
    */
    maxNumberOfInvitations: number
    
    /**
    * @description Function called when the session is created. The session creation type in this callback can be used to tell if the session is being created from scratch, or is being received from another user.
    */
    onSessionCreated: (session: MultiplayerSession, sessionCreationType: ConnectedLensSessionOptions.SessionCreationType) => void
    
}
declare namespace ConnectedLensSessionOptions {
    /**
    * @description Create a new ConnectedLensesOptions object.
    */
    export function create(): ConnectedLensSessionOptions
    

}

declare namespace ConnectedLensSessionOptions {
    /**
    * @description Type of the created Connected Lens session.
    */
    enum SessionCreationType {
        /**
        * @description New Session was created.
        */
        New,
        /**
        * @description Session was joined via invite.
        */
        MultiplayerReceiver,
        /**
        * @description A Connected Lens session with only the current user.
        */
        NewSoloMode
    }

}

/**
* @description Constraints body motion in configurable ways, for simulating physical objects such as joints and hinges.
*/
interface Constraint extends ScriptObject {
    /**
    * @readonly
    
    * @description The type of constraint that is applied.
    */
    constraintType: Physics.ConstraintType
    
}

/**
* @description Used to apply a constraint to an object.
*/
interface ConstraintComponent extends Component {
    /**
    * @description The target is attached to the constraint by a fixed local-space matrix, calculated from the difference between the target's and the constraint's world-space transforms. This local-space matrix is generated on-load, or whenever the target is changed. Call this function to explicitly recalculate it for the current world-space transforms.
    */
    reanchorTarget(): void
    
    /**
    * @description Dictates constraint type and settings. Note, when setting this field it creates a copy of the constraint, rather than referencing it. So if you intend to modify the constraint after assigning it, you must do so on the component's constraint field, rather than the source constraint object.
    */
    constraint: Constraint
    
    /**
    * @description Reference to connected target collider. If null, constraint target is attached to a fixed world transform.
    */
    target: ColliderComponent
    
    /**
    * @description Show the constraint with debug-draw.
    */
    debugDrawEnabled: boolean
    
}

/**
* @description Contact point between two colliding objects.  See also: [ColliderComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), [CollisionEnterEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#CollisionEnterEventArgs), [CollisionExitEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#CollisionExitEventArgs), [CollisionStayEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#CollisionStayEventArgs).
*/
interface Contact extends ScriptObject {
    /**
    * @readonly
    
    * @description Distance along the normal between the hit collider and this collider.
    */
    distance: number
    
    /**
    * @readonly
    
    * @description Impulse (kg*cm/s) applied along the normal in response to the collision.
    */
    impulse: number
    
    /**
    * @readonly
    
    * @description Position on the hit collider.
    */
    position: vec3
    
    /**
    * @readonly
    
    * @description Normal on the hit collider.
    */
    normal: vec3
    
}

/**
* @description Base class for Texture Providers that crop an input texture.
*/
interface CropTextureProvider extends TextureProvider {
    /**
    * @description Input texture to crop.
    */
    inputTexture: Texture
    
}

/**
* @description Used with [Pass's](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Pass) `cullMode` property.
* Determines which faces of a surface are culled (not rendered).
*/
declare enum CullMode {
    /**
    * @description Front facing surfaces are not rendered.
    */
    Front,
    /**
    * @description Back facing surfaces are not rendered.
    */
    Back,
    /**
    * @description Neither front facing nor back facing surfaces are rendered.
    */
    FrontAndBack
}

/**
* @description A cylinder collision shape.
*/
interface CylinderShape extends Shape {
    /**
    * @description Lengthwise local axis along which the cylinder is oriented.
    */
    axis: Axis
    
    /**
    * @description The length of the cylinder.
    */
    length: number
    
    /**
    * @description The radius of the cylinder.
    */
    radius: number
    
}

interface DeformingCollisionMesh extends CollisionMesh {
}

/**
* @description Creates a buffer for the audio data.
*/
interface Delay extends ScriptObject {
    /**
    * @description Process current frame data passed in `inTensor` of shape `inShape`, writes the result (delayed frame) to the `outTensor` and returns the shape of `outTensor`.
    */
    process(inTensor: Float32Array, inShape: vec3, outTensor: Float32Array): vec3
    
    /**
    * @readonly
    
    * @description Maximum amount of features (channels).
    */
    maxTensorSize: number
    
}

/**
* @description Builder class for the Delay.
*/
interface DelayBuilder extends ScriptObject {
    /**
    * @description Creates new Delay object.
    */
    build(): Delay
    
    /**
    * @description Specify number of channels. Currently, only mono is supported, so should be set to `1`.
    */
    setNumFeatures(numFeatures: number): DelayBuilder
    
    /**
    * @description Set the delay of the Delay object in samples. Delay is equal to `sampleRate * delayInSeconds`.
    */
    setDelay(delay: number): DelayBuilder
    
}

/**
* @description An event that gets triggered after a delay.
*/
interface DelayedCallbackEvent extends SceneEvent {
    /**
    * @description Returns the total delay time in seconds set on the event.
    */
    getDelayTime(): number
    
    /**
    * @description Returns the current time in seconds left in the event's delay.
    */
    getTimeLeft(): number
    
    /**
    * @description Calling this will cause the event to trigger in `time` seconds.
    */
    reset(time: number): void
    
    /**
    * @description Cancel the callback in progress.
    */
    cancel(): void
    
}

/**
* @description Local estimate of the derivative of the input data along the selected axis. Outputs the derivative of the input features along the window.
*/
interface Delta extends ScriptObject {
    /**
    * @description Process `inTensor` with shape `inShape` and writes the result to the `outTensor` and returns the shape of `outTensor`.
    */
    process(inTensor: Float32Array, inShape: vec3, outTensor: Float32Array): vec3
    
    /**
    * @readonly
    
    * @description Maximum amount of features (channels).
    */
    maxTensorSize: number
    
}

/**
* @description Builder class for Delta.
*/
interface DeltaBuilder extends ScriptObject {
    /**
    * @description Create a new Delta object.
    */
    build(): Delta
    
    /**
    * @description Specify number of channels. Currently, only mono is supported, so should be set to `1`.
    */
    setNumFeatures(numFeatures: number): DeltaBuilder
    
    /**
    * @description Set the size of the window.
    */
    setWindowSize(winSize: number): DeltaBuilder
    
}

/**
* @description Settings for the depth clear option modes on a DepthStencilRenderTargetProvider.
*/
declare enum DepthClearOption {
    /**
    * @description Do not clear the depth buffer, just keep values. Equal to disabled clear depth checkbox in Camera in previous Studio version.
    */
    None,
    /**
    * @description Clear depth buffer by specific value. Equal to enabled clear depth checkbox in Camera in previous Studio version.
    */
    CustomValue,
    /**
    * @description Clear depth buffer by values from depth part of input texture. Will use custom value in case of unavailable input texture.
    */
    CustomTexture
}

/**
* @description Writes video feed depth information to the depth buffer, which automatically sets up depth occlusion for 3D visuals.
* Only works in some cases, such as in Lenses for Spectacles 3.
* See the [Lenses for Spectacles guide](https://lensstudio.snapchat.com/lens-studio/references/guides/distributing/spectacles/spectacles-3/lenses-for-spectacles-3) for more information.
*/
interface DepthSetter extends PostEffectVisual {
}

/**
* @description Access to a Depth Stencil Render Target that can output depth and stencil values from a Camera in Depth24/Stencil8 format.
*/
interface DepthStencilRenderTargetProvider extends TextureProvider {
    /**
    * @description Custom render target resolution, this property will use if the "outputResolution" property is a "Custom".
    */
    resolution: vec2
    
    /**
    * @description Depth buffer clear option.  "None" - depth buffer clear operation will be skipped.  "CustomValue" - depth buffer will be cleared by "clearDepthValue" property value. "CustomTexture" - depth buffer will be cleared by texture from "inputTexture" property, if "inputTexture" is null then depth clear option will fallback to "CustomValue" mode.
    */
    depthClearOption: DepthClearOption
    
    /**
    * @description Float value in range [0.0..1.0] used in depth buffer clear operation in "CustomValue" mode. The initial value is 1.0.
    */
    clearDepthValue: number
    
    /**
    * @description Stencil buffer clear option.  "None" - stencil buffer clear operation will be skipped.  "CustomValue" - stencil buffer will be cleared by "clearStencilValue" property value. "CustomTexture" - stencil buffer will be cleared by texture from "maskTexture" property, if "maskTexture" is null then the stencil clear option will fallback to "CustomValue" mode.
    */
    stencilClearOption: StencilClearOption
    
    /**
    * @description Unsigned int value in range [0..0xFF] used in stencil buffer clear operation in "CustomValue" mode. The initial value is 0.
    */
    clearStencilValue: number
    
    /**
    * @description Texture with Depth24_Stencil8 format. Depth24 part used in depth clear operation in "CustomTexture" mode.
    */
    inputTexture: Texture
    
    /**
    * @description Texture with Depth24_Stencil8 format. Stencil8 part used in stencil clear operation in "CustomTexture" mode.
    */
    maskTexture: Texture
    
}

/**
* @description Provides depth information of the video feed that the Lens is being applied to when available, such as in Lenses for Spectacles 3.
* Can be accessed from `mainPass.baseTex.control` of a Spectacles Depth material.

* See the [Lenses for Spectacles guide](https://lensstudio.snapchat.com/lens-studio/references/guides/distributing/spectacles/spectacles-2021/lenses-for-spectacles) for more information.
*/
interface DepthTextureProvider extends TextureProvider {
    /**
    * @exposesUserData
    
    * @description Returns the depth at the screen space position "point".
    * The value returned is between 0 and 6,550, which corresponds to the distance the point is from the camera in centimeters, or world-space units.
    * If depth data is not available, -1 will be returned. Note that depth data isn't available during the very first Initialize event, before TurnOn event fires.
    */
    getDepth(point: vec2): number
    
}

/**
* @description Provides information about the device's camera.
*/
interface DeviceCamera extends ScriptObject {
    /**
    * @description Unprojects the normalized screen space point `normalizedScreenSpacePoint` based on some distance `absoluteDepth`.
    */
    unproject(normalizedScreenSpacePoint: vec2, absoluteDepth: number): vec3
    
    /**
    * @readonly
    
    * @description Provides the focal length of the device's camera.
    */
    focalLength: vec2
    
    /**
    * @readonly
    
    * @description Provides the principal point, or center of image, of the device's camera.
    */
    principalPoint: vec2
    
    /**
    * @readonly
    
    * @description Provides the resolution of the image returned by the device's camera.
    */
    resolution: vec2
    
    /**
    * @readonly
    
    * @description Provides the transform of the camera.
    */
    pose: mat4
    
}

/**
* @description Provides information about the device running the Lens. Accessible through `global.deviceInfoSystem`.
*/
interface DeviceInfoSystem extends ScriptObject {
    /**
    * @exposesUserData
    
    * @description Returns the operating system type of the device.
    */
    getOS(): OS
    
    /**
    * @description Provides the tracking camera's [DeviceCamera](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#DeviceCamera).
    */
    getTrackingCamera(): DeviceCamera
    
    /**
    * @description Returns whether the current Lens is running in a desktop computer.
    */
    isDesktop(): boolean
    
    /**
    * @description Returns whether the current Lens is running in a mobile device.
    */
    isMobile(): boolean
    
    /**
    * @description Returns whether the current Lens is running in a Spectacles device.
    */
    isSpectacles(): boolean
    
    /**
    * @description Returns whether the current Lens is running in Lens Studio.
    */
    isEditor(): boolean
    
}

/**
* @description Used to track a landmarker in the camera. Moves the SceneObject's transform to match the detected landmarker scene.
* See the [Landmarker guide](https://lensstudio.snapchat.com/templates/landmarker) for more information.
*/
interface DeviceLocationTrackingComponent extends Component {
    /**
    * @description Returns whether the location landmarker is currently being tracked.
    */
    isTracking(): boolean
    
    /**
    * @description The location that the tracker is tracking. Useful for dynamically controlling the target location being tracked.
    */
    location: LocationAsset
    
    /**
    * @description A function that gets called when location data is downloaded.
    */
    onLocationDataDownloaded: () => void
    
    /**
    * @description A function that gets called when location data fails to download.
    */
    onLocationDataDownloadFailed: () => void
    
    /**
    * @description A function that gets called when location is found.
    */
    onLocationFound: () => void
    
    /**
    * @description A function that gets called when location is lost. Note this will also happen when the user flips the camera.
    */
    onLocationLost: () => void
    
    /**
    * @readonly
    
    * @description Returns the user's current LocationProximityStatus. Useful for telling if a user is close enough to the location to track it.
    */
    locationProximityStatus: LocationProximityStatus
    
    /**
    * @readonly
    
    * @description Returns the distance, in meters, to the location. If the distance is unavailable, -1 is returned.
    */
    distanceToLocation: number
    
}

/**
* @description Moves or rotates the SceneObject to match device orientation.

* If using "Surface" tracking mode, adding this to a SceneObject enables surface tracking for the scene, and moves the
* object to a position and rotation that matches the physical camera's pose in the world. Surface tracking can also be enhanced
* with native AR by enabling the "Use Native AR" option in the Inspector panel, or through script by setting the
* component's [`surfaceOptions.enhanceWithNativeAR`](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SurfaceOptions) property.

* If using "Rotation" tracking mode, adding this to a SceneObject will apply the device's real world rotation to the object.

* If using "World" tracking mode, adding this to a SceneObject enables native AR tracking for the scene, and moves the
* object to a position and rotation that matches the physical camera's pose in the world.

* See the [Tracking Modes](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/world/tracking-modes) guide for more information.

**Note:** This component was named "WorldTracking" in previous versions of Lens Studio.
*/
interface DeviceTracking extends Component {
    /**
    * @description Offsets the default position of the World Tracking surface origin by `offset`.
    * Avoid using a `y` value of zero in `offset`, because it may cause problems with tracking.
    * If used outside of `Initialized` or `TurnOnEvent`, you will need to call `resetTracking()` to apply the offset.
    * Note: calling `resetTracking()` will overwrite the `x` and `z` components of the offset.
    */
    setWorldOriginOffset(offset: vec3): void
    
    /**
    * @description Resets the World Tracking origin to the point on the surface plane aligned with the screen position `position`.
    * Screen positions are represented in the range ([0-1], [0-1]), (0,0) being the top-left of the screen and (1,1) being the bottom-right.
    */
    resetTracking(position: vec2): void
    
    /**
    * @description Returns an array of TrackedMeshHitTestResult that intersect with a ray cast from screen position screenPos. Only available when world mesh tracking is supported and enabled.
    */
    hitTestWorldMesh(screenPos: vec2): TrackedMeshHitTestResult[]
    
    /**
    * @description Returns an array of TrackedMeshHitTestResult that intersect with a ray cast from the world position `from` and continuing through the world position `to`. Only available when world mesh tracking is supported and enabled.
    */
    raycastWorldMesh(from: vec3, to: vec3): TrackedMeshHitTestResult[]
    
    /**
    * @description Calculates a histogram of world mesh surfaces within a sphere at the given world position and radius. Only available when world mesh tracking is supported and enabled.
    */
    calculateWorldMeshHistogram(center: vec3, radius: number): TrackedMeshHistogramResult
    
    /**
    * @description Creates a TrackedPoint at world position `worldPos` and world rotation `worldRot`.
    */
    createTrackedWorldPoint(worldPos: vec3, worldRot: quat): TrackedPoint
    
    /**
    * @description Returns whether the DeviceTrackingMode is supported.
    */
    isDeviceTrackingModeSupported(mode: DeviceTrackingMode): boolean
    
    /**
    * @description Requests that a DeviceTrackingMode be used. This requested change may not happen immediately.
    */
    requestDeviceTrackingMode(val: DeviceTrackingMode): void
    
    /**
    * @description Returns the DeviceTrackingMode currently requested to be used.
    * This may be different from the actual DeviceTrackingMode being used.
    */
    getRequestedDeviceTrackingMode(): DeviceTrackingMode
    
    /**
    * @description Returns the actual DeviceTrackingMode being used. This may be different from the requested DeviceTrackingMode.
    */
    getActualDeviceTrackingMode(): DeviceTrackingMode
    
    /**
    * @description Returns an array of BasicTransform objects describing each point that the camera travels through.
    * Each item in the array matches the camera's basic transform in the corresponding frame of the video feed that the Lens is applied to.
    * Only available in some cases, such as in Lenses for Spectacles 3. When not available, it will return an empty array.
    * See the [Lenses for Spectacles guide](https://lensstudio.snapchat.com/lens-studio/references/guides/distributing/spectacles/spectacles-2021/lenses-for-spectacles) for more information.
    */
    getDevicePath(): BasicTransform[]
    
    /**
    * @description Returns the current frame index of the video feed that the Lens is being applied to.
    * This can be used as an index to access the current BasicTransform in `getDevicePath()`.
    * Only available in some cases, such as in Lenses for Spectacles 3. When not available, it will return -1.
    * See the [Lenses for Spectacles guide](https://lensstudio.snapchat.com/lens-studio/references/guides/distributing/spectacles/spectacles-2021/lenses-for-spectacles) for more information.
    */
    getDevicePathIndex(): number
    
    /**
    * @exposesUserData
    
    * @description Returns the 3D point cloud representing important features visible by the camera.
    */
    getPointCloud(): PointCloud
    
    /**
    * @description Helps to improve surface tracking accuracy while the target `SceneObject` is being moved.
    */
    surfaceTrackingTarget: SceneObject
    
    /**
    * @description Used to access rotation tracking settings.
    */
    rotationOptions: RotationOptions
    
    /**
    * @description Used to access surface tracking settings.
    */
    surfaceOptions: SurfaceOptions
    
    /**
    * @description Returns the WorldOptions object of this component, which can be used to control World Tracking settings.
    */
    worldOptions: WorldOptions
    
    /**
    * @readonly
    
    * @description Returns the World Tracking Capabilities of the current device.
    */
    worldTrackingCapabilities: WorldTrackingCapabilities
    
}

/**
* @description Tracking modes used by the [DeviceTracking](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#DeviceTracking) component to specify what type of tracking to use.
*/
declare enum DeviceTrackingMode {
    /**
    * @description Use gyroscope tracking (rotation only)
    */
    Rotation,
    /**
    * @description Use surface tracking (position and rotation)
    */
    Surface,
    /**
    * @description Use native tracking (position and rotation)
    */
    World
}

declare namespace Dialog {
    /**
    * @description Provides answer information in response to `DialogModule.askQuestions()`.
    */
    interface Answer extends ScriptObject {
        /**
        * @readonly
        
        * @description The index of the question this is answering in the questions array.
        */
        questionId: number
        
        /**
        * @readonly
        
        * @description The answer to the question.
        */
        answer: string
        
        /**
        * @readonly
        
        * @description Status of the question response.
        */
        status: number
        
    }

}

interface DialogModule extends Asset {
    /**
    * @description Sends a request to ask questions using the DialogModule. `context` is the text the model will use as context for answering the question. `questions` is a list of questions to ask. When answers are ready, `onQuestionsAnswerComplete` will be called with a list of `Dialog.Answer` objects corresponding to the questions in the `questions` list. `onQuestionsAnswerError` will be called if any error occurs.
    */
    askQuestions(context: string, questions: string[], onQuestionsAnswerComplete: (answers: Dialog.Answer[]) => void, onQuestionsAnswerError: (error: number, description: string) => void): void
    
}

/**
* @description An Audio Component effect that simulates sound attenuation based on the orientation of the transform relative to the [AudioListenerComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AudioListenerComponent). 
*/
interface DirectivityEffect extends ScriptObject {
    /**
    * @description Whether the audio directivity effect is applied or not.
    */
    enabled: boolean
    
    /**
    * @description A ratio that specifies the shape of pattern directivity from omnidirectional(0) to cardioid (1.0).
    */
    shapeFactor: number
    
    /**
    * @description The width of main lobe.
    */
    shapeDecay: number
    
}

interface DirectMultiplayerSessionOptions extends BaseMultiplayerSessionOptions {
}

/**
* @description An audio effect that simulates sound attenuation based on the distance between the Audio and the [AudioListenerComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AudioListenerComponent). 
*/
interface DistanceEffect extends ScriptObject {
    /**
    * @description If enabled, the distance effect will be applied.
    */
    enabled: boolean
    
    /**
    * @description Curve type that describes how volume attenuates with distance.
    */
    type: Audio.DistanceCurveType
    
    /**
    * @description If the distance is less than this value, the distance effect is not applied at all. 
    */
    minDistance: number
    
    /**
    * @description If the distance is higher than this value, the sound cannot be heard at all.
    */
    maxDistance: number
    
}

/**
* @description Used in [Text's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) `dropShadowSettings` property.
* Configures how dropshadow will appear on a Text component.
*/
interface DropshadowSettings extends ScriptObject {
    /**
    * @description Whether dropshadow is enabled on the Text.
    */
    enabled: boolean
    
    /**
    * @description Settings for how the inside of the dropshadow is drawn.
    */
    fill: TextFill
    
    /**
    * @description An (x, y) offset controlling where the dropshadow is drawn relative to the Text.
    */
    offset: vec2
    
}

/**
* @description A resource that is resolved at runtime.
*/
interface DynamicResource extends ScriptObject {
}

interface EventRegistration extends ScriptObject {
}

/**
* @description Expression names used with `FaceRenderObjectProvider.getExpressionWeightByName()`
* and returned by `FaceRenderObjectProvider.getExpressionNames()`.
*/
interface Expressions {
}
declare namespace Expressions {
    /**
    * @description Left eyelids closing
    */
    let EyeBlinkLeft: string
    
    /**
    * @description Right eyelids closing
    */
    let EyeBlinkRight: string
    
    /**
    * @description Left eye squinting
    */
    let EyeSquintLeft: string
    
    /**
    * @description Right eye squinting
    */
    let EyeSquintRight: string
    
    /**
    * @description Left eyelids downward look
    */
    let EyeDownLeft: string
    
    /**
    * @description Right eyelids downward look
    */
    let EyeDownRight: string
    
    /**
    * @description Left eyelids looking towards center
    */
    let EyeInLeft: string
    
    /**
    * @description Right eyelids looking towards center
    */
    let EyeInRight: string
    
    /**
    * @description Left eyelids opening
    */
    let EyeOpenLeft: string
    
    /**
    * @description Right eyelids opening
    */
    let EyeOpenRight: string
    
    /**
    * @description Left eyelids looking away from center
    */
    let EyeOutLeft: string
    
    /**
    * @description Right eyelids looking away from center
    */
    let EyeOutRight: string
    
    /**
    * @description Left eyelids upward look
    */
    let EyeUpLeft: string
    
    /**
    * @description Right eyelids upward look
    */
    let EyeUpRight: string
    
    /**
    * @description Left eyebrow downward movement
    */
    let BrowsDownLeft: string
    
    /**
    * @description Right eyebrow downward movement
    */
    let BrowsDownRight: string
    
    /**
    * @description Between eyebrows upward movement
    */
    let BrowsUpCenter: string
    
    /**
    * @description Left eyebrow upward movement
    */
    let BrowsUpLeft: string
    
    /**
    * @description Right eyebrow upward movement
    */
    let BrowsUpRight: string
    
    /**
    * @description Jaw forward movement
    */
    let JawForward: string
    
    /**
    * @description Jaw leftward movement
    */
    let JawLeft: string
    
    /**
    * @description Jaw opening
    */
    let JawOpen: string
    
    /**
    * @description Jaw rightward movement
    */
    let JawRight: string
    
    /**
    * @description Both lips leftward movement
    */
    let MouthLeft: string
    
    /**
    * @description Both lips rightward movement
    */
    let MouthRight: string
    
    /**
    * @description Left mouth corner downward movement
    */
    let MouthFrownLeft: string
    
    /**
    * @description Right mouth corner downward movement
    */
    let MouthFrownRight: string
    
    /**
    * @description Left mouth corner upward movement
    */
    let MouthSmileLeft: string
    
    /**
    * @description Right mouth corner upward movement
    */
    let MouthSmileRight: string
    
    /**
    * @description Left mouth corner back and leftward movement
    */
    let MouthDimpleLeft: string
    
    /**
    * @description Right mouth corner back and rightward movement
    */
    let MouthDimpleRight: string
    
    /**
    * @description Left side of mouth upward movement
    */
    let MouthUpLeft: string
    
    /**
    * @description Right side of mouth upward movement
    */
    let MouthUpRight: string
    
    /**
    * @description Upper lip moving towards and behind lower lip
    */
    let UpperLipClose: string
    
    /**
    * @description Lower lip moving towards and behind upper lip
    */
    let LowerLipClose: string
    
    /**
    * @description Left upper lip upward movement
    */
    let UpperLipUpLeft: string
    
    /**
    * @description Right upper lip upward movement
    */
    let UpperLipUpRight: string
    
    /**
    * @description Left lower lip downward movement
    */
    let LowerLipDownLeft: string
    
    /**
    * @description Right lower lip downward movement
    */
    let LowerLipDownRight: string
    
    /**
    * @description Lips forming open circular shape together
    */
    let LipsFunnel: string
    
    /**
    * @description Lips compressing together while closed
    */
    let LipsPucker: string
    
    /**
    * @description Lower lip upward movement
    */
    let LowerLipRaise: string
    
    /**
    * @description Upper lip upward movement
    */
    let UpperLipRaise: string
    
    /**
    * @description Both cheeks puffing outward movement
    */
    let Puff: string
    
    /**
    * @description Left cheek and below left eye upward movement
    */
    let CheekSquintLeft: string
    
    /**
    * @description Right cheek and below right eye upward movement
    */
    let CheekSquintRight: string
    
    /**
    * @description Left nostril raising
    */
    let SneerLeft: string
    
    /**
    * @description Right nostril raising
    */
    let SneerRight: string
    
    /**
    * @description Left side of mouth leftward movement
    */
    let MouthStretchLeft: string
    
    /**
    * @description Right side of mouth rightward movement
    */
    let MouthStretchRight: string
    
    /**
    * @description Lips moving together
    */
    let MouthClose: string
    

}

/**
* @description Applies an eye color effect to a face.
*/
interface EyeColorVisual extends MaterialMeshVisual {
    /**
    * @description The index of the face this EyeColorVisual is attached to.
    */
    faceIndex: number
    
}

/**
* @description Texture Provider giving a cropped region of the input texture, calculated based on face position.
* Can be accessed using `Texture.control` on a FaceCropTexture asset.
* For more information, see the [Crop Textures](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/crop-textures#crop-textures) guide.
*/
interface FaceCropTextureProvider extends CropTextureProvider {
    /**
    * @description Ratio of the mouth position on the cropped texture. Value ranges from 0 to 1, with 0 having no effect and 1 centering the image on the mouth.
    */
    faceCenterMouthWeight: number
    
    /**
    * @description Scaling of the cropped texture.
    */
    textureScale: vec2
    
    /**
    * @description Index of the face being tracked.
    */
    faceIndex: number
    
}

/**
* @description Triggered when a new face is detected and starts being tracked.
*/
interface FaceFoundEvent extends FaceTrackingEvent {
}

/**
* @description Controls the face image picker texture resource.
* Can be accessed through [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#texture--control) on a Face Image Picker texture.
* For more information, see the [Face Image Picker Texture guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/face/face-effects/face-image-picker-texture).
*/
interface FaceImagePickerTextureProvider extends ImagePickerTextureProvider {
    /**
    * @description If enabled, the selected image will be cropped to only show the face region.
    */
    cropToFace: boolean
    
    /**
    * @description The FaceTextureProvider used to provide the face texture.
    */
    faceControl: FaceTextureProvider
    
}

/**
* @description Used with [FaceInsetVisual.faceRegion](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#FaceInsetVisual) for setting the face region to draw.
*/
declare enum FaceInsetRegion {
    /**
    * @description Targets the left eye
    */
    LeftEye,
    /**
    * @description Targets the right eye
    */
    RightEye,
    /**
    * @description Targets the mouth
    */
    Mouth,
    /**
    * @description Targets the nose
    */
    Nose,
    /**
    * @description Targets the entire face
    */
    Face
}

/**
* @description Draws a section of a tracked face.
*/
interface FaceInsetVisual extends MaterialMeshVisual {
    /**
    * @description The index of the face this FaceInsetVisual uses.
    */
    faceIndex: number
    
    /**
    * @description The region of the face this FaceInsetVisual draws.
    */
    faceRegion: FaceInsetRegion
    
    /**
    * @description The x and y scaling used to draw the face region.
    * Think of scaling as meaning how many times the face region could fit into the drawing area.
    * Higher values will zoom away from the face region, and lower values will zoom into it.
    * The normal, unzoomed scaling value is (1,1).
    */
    sourceScale: vec2
    
    /**
    * @description The amount of alpha fading applied from the border of the face inset inward.
    * This value must be in the range 0-1.
    */
    innerBorderRadius: number
    
    /**
    * @description The amount of alpha fading applied from the border of the face inset outward.
    * This value must be in the range 0-1.
    */
    outerBorderRadius: number
    
    /**
    * @description Determines the quality of the face inset's borders.
    * A higher value means better looking borders but lower performance.
    * This value must be greater than 10 and less than 100.
    */
    subdivisionsCount: number
    
    /**
    * @description Flips the drawn face region horizontally if enabled.
    */
    flipX: boolean
    
    /**
    * @description Flips the drawn face region vertically if enabled.
    */
    flipY: boolean
    
}

/**
* @description Triggered when a face can no longer be tracked.  For example, if a face gets blocked from the camera's view, or gets too far away.
*/
interface FaceLostEvent extends FaceTrackingEvent {
}

/**
* @description Applies a face mask effect. See the [Face Mask Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/face/face-effects/face-mask) for more information.
*/
interface FaceMaskVisual extends MaterialMeshVisual {
    hidesMaskOnMouthClosed: boolean
    
    swapsMaskOnMouthClosed: boolean
    
    customMaskOnMouthClosed: Texture
    
    /**
    * @description The index of the face this effect is attached to.
    */
    faceIndex: number
    
    /**
    * @description If "Use Orig. Face" is enabled for this FaceMaskVisual in the Inspector panel, this property
    * specifies the face index to use for drawing the mask.
    */
    originalFaceIndex: number
    
    teethAlpha: number
    
    useOriginalTexCoords: boolean
    
}

/**
* @description Mesh provider for a Face Mesh. Accessible through the `control` property on a Face Mesh `RenderMesh`.
*/
interface FaceRenderObjectProvider extends RenderObjectProvider {
    /**
    * @description Returns a list of all expression names being tracked.
    */
    getExpressionNames(): string[]
    
    /**
    * @description Returns the weight of the expression with the passed in name. See [Expressions](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#Expressions) for valid expression names.
    */
    getExpressionWeightByName(expressionName: string): number
    
    /**
    * @description Returns a Float32Array of all expression weights being tracked.
    */
    getExpressionWeights(): Float32Array
    
    /**
    * @description Index of the face this FaceRenderObjectProvider mirrors.
    */
    faceIndex: number
    
    /**
    * @description When true, the general face (not including eyes and mouth) will be included in the Face Mesh geometry.
    */
    faceGeometryEnabled: boolean
    
    /**
    * @description When true, eyes will be included in the Face Mesh geometry.
    */
    eyeGeometryEnabled: boolean
    
    /**
    * @description When true, a small area in the corners of the eyes will be included in the Face Mesh geometry.
    */
    eyeCornerGeometryEnabled: boolean
    
    /**
    * @description When true, the mouth will be included in the Face Mesh geometry.
    */
    mouthGeometryEnabled: boolean
    
    /**
    * @description When true, the skull will be included in the Face Mesh geometry.
    */
    skullGeometryEnabled: boolean
    
    /**
    * @description When true, ears will be included in the Face Mesh geometry.
    */
    earGeometryEnabled: boolean
    
}

/**
* @description Applies a face stretch effect.
* Face stretch features can be added to a FaceStretchVisual through the Inspector panel in Lens Studio.
* See the [Face Stretch Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/face/face-effects/face-stretch) for more information.
*/
interface FaceStretchVisual extends BaseMeshVisual {
    /**
    * @description Sets the weight of the face stretch feature named `feature` to `intensity`.
    * The intensity must be greater than -0.5 and less than 2.
    */
    setFeatureWeight(feature: string, intensity: number): void
    
    /**
    * @description Returns the weight of the face stretch feature named `feature`.
    */
    getFeatureWeight(feature: string): number
    
    /**
    * @description The index of the face the stretch will be applied to.
    */
    faceIndex: number
    
}

/**
* @description TextureProvider for face textures.
* See the [Face Texture Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/face/face-effects/face-texture) for more information.
* Can be accessed using [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Texture#control-textureprovider) on a face texture asset.
*/
interface FaceTextureProvider extends TextureProvider {
    /**
    * @description The source texture being drawn.
    * This is useful for controlling which effects are visible on the face texture, based on which camera output texture is being used.
    */
    inputTexture: Texture
    
    /**
    * @description The x and y scale used to draw the face within the texture region.
    * A lower scale will be more zoomed in on the face, and a higher scale will be more zoomed out.
    * The default scale is (1, 1).
    */
    scale: vec2
    
    /**
    * @description Index of the face to track.
    */
    faceIndex: number
    
}

/**
* @description This is the base class for all face tracking events. This event won't actually get triggered itself, so use one of the child classes instead.
*/
interface FaceTrackingEvent extends SceneEvent {
    /**
    * @description The index of the face this event is tracking. Change this value to control which face the event tracks.
    */
    faceIndex: number
    
}

/**
* @description Provider for file based Audio Tracks.
*/
interface FileAudioTrackProvider extends AudioTrackProvider {
    /**
    * @description Writes current audio frame to the passed in `Float32Array` and returns the frame shape.
    */
    getAudioFrame(audioFrame: Float32Array): vec3
    
    /**
    * @description Writes readSize samples into the passed in audioBuffer Float32Array.
    */
    getAudioBuffer(audioBuffer: Float32Array, readSize: number): vec3
    
    /**
    * @readonly
    
    * @description The duration of the AudioTrackAsset in seconds.
    */
    duration: number
    
    /**
    * @description The current position of the AudioTrackAsset in seconds.
    */
    position: number
    
    /**
    * @description Loop count, if `-1` is provided, the audio track will loop forever.
    */
    loops: number
    
}

/**
* @description A file track provider of the Licensed Sounds from Asset LIbrary.
*/
interface FileLicensedSoundProvider extends AudioTrackProvider {
    /**
    * @description Writes current audio frame to the passed in Float32Array and returns the frame shape.
    */
    getAudioFrame(audioFrame: Float32Array): vec3
    
    /**
    * @description Writes readSize samples into the passed in audioBuffer Float32Array.
    */
    getAudioBuffer(audioBuffer: Float32Array, readSize: number): vec3
    
    /**
    * @readonly
    
    * @description The duration of the AudioTrackAsset in seconds.
    */
    duration: number
    
    /**
    * @description Loop count, if -1 is provided, the audio track will loop forever.
    */
    loops: number
    
}

/**
* @description A [TextureProvider](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#TextureProvider) for textures originating from files.
*/
interface FileTextureProvider extends TextureProvider {
}

/**
* @description Intersection filter settings. Unifies settings for world probes and collider overlap tests.   See also: [ColliderComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), [Physics](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#Physics).
*/
interface Filter extends ScriptObject {
    /**
    * @description Include static objects in intersection tests.
    */
    includeStatic: boolean
    
    /**
    * @description Include dynamic objects in intersection tests.
    */
    includeDynamic: boolean
    
    /**
    * @description Include intangible objects in intersection tests.
    */
    includeIntangible: boolean
    
    /**
    * @description If non-empty, only perform collision with colliders in these layers. In other words: the set of layers to include when performing collision tests, excluding all others.  If empty, this setting is disabled (effectively including all layers, minus skipLayers).
    */
    onlyLayers: LayerSet
    
    /**
    * @description Skip collision with colliders in these layers. In other words: the set of layers to exclude when performing collision tests. This takes precedence over onlyLayers, so a layer that is in both is skipped.
    */
    skipLayers: LayerSet
    
    /**
    * @description If non-empty, only perform collision with these colliders. In other words: the set of colliders to include when performing collision tests, excluding all others.  If empty, this setting is disabled (effectively including all colliders, minus skipColliders).
    */
    onlyColliders: ColliderComponent[]
    
    /**
    * @description Skip collision with these colliders. In other words: the set of colliders to exclude when performing collision tests. This takes precedence over onlyColliders, so a collider that is in both is skipped.
    */
    skipColliders: ColliderComponent[]
    
}

declare enum FilteringMode {
    Nearest,
    Bilinear,
    Trilinear
}

interface FixedCollisionMesh extends CollisionMesh {
}

/**
* @description Fully constrain rotation and translation.  See also: [ConstraintComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ConstraintComponent).
*/
interface FixedConstraint extends Constraint {
}

/**
* @description The base class for animation tracks using float values.
*/
interface FloatAnimationTrack extends AnimationTrack {
}

/**
* @description Represents an animation track using float value keyframes.
*/
interface FloatAnimationTrackKeyFramed extends FloatAnimationTrack {
    /**
    * @description Removes all keys.
    */
    removeAllKeys(): void
    
    /**
    * @description Removes key at index `index`.
    */
    removeKeyAt(index: number): void
    
    /**
    * @description Adds a key with value `value` at time `time`.
    */
    addKey(time: number, value: number): void
    
}

/**
* @description Represents an animation track using vec3 value keyframes for a bezier curve.
*/
interface FloatBezierAnimationTrackKeyFramed extends FloatAnimationTrack {
    /**
    * @description Removes all keys.
    */
    removeAllKeys(): void
    
    /**
    * @description Removes key at index `index`.
    */
    removeKeyAt(index: number): void
    
    /**
    * @description Adds a key with value `value` at time `time`.
    */
    addKey(time: number, value: vec3): void
    
}

/**
* @description Arguments used with the `InteractionComponent.onFocusEnd` event.
*/
interface FocusEndEventArgs extends ScriptObject {
}

/**
* @description Arguments used with the `InteractionComponent.onFocusStart` event.
*/
interface FocusStartEventArgs extends ScriptObject {
}

/**
* @description A font asset used for rendering text.
* Used by [Text](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text).
* For more information, see the [Text guide](https://lensstudio.snapchat.com/lens-studio/references/guides/adding-content/text/2d-text).
*/
interface Font extends Asset {
}

/**
* @description Mode for setting frustum culling on Pass
*/
declare enum FrustumCullMode {
    /**
    * @description Based on shader property, frustum culling will either be disabled or test with render object's aabb
    */
    Auto,
    /**
    * @description Enable frustum culling and extend render object's aabb to (1 + value)
    */
    Extend,
    /**
    * @description Users define the specific AABB which is used for culling test. Assumes frustumCullMin and frustumCullMax are calculated in local space of the object. frustumCullMin is the bottom-left-back and frustumCullMax is the top-right-front of the user defined AABB.
    
    */
    UserDefinedAABB
}

interface GaussianSplattingAsset extends Asset {
}

interface GaussianSplattingVisual extends MaterialMeshVisual {
    asset: GaussianSplattingAsset
    
}

/**
* @description Class for storing and retrieving data based on keys.
* Used by [PersistentStorageSystem](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#PersistentStorageSystem).
* For more information, see the [Persistent Storage guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/persistent-storage).
*/
interface GeneralDataStore extends ScriptObject {
    /**
    * @description Returns an integer number stored under the given key, or 0 if none exists.
    */
    getInt(key: string): number
    
    /**
    * @description Returns a double precision floating point number stored under the given key, or 0 if none exists.
    */
    getDouble(key: string): number
    
    /**
    * @description Returns a boolean value stored under the given key, or false if none exists.
    */
    getBool(key: string): boolean
    
    /**
    * @description Returns a string value stored under the given key, or empty string if none exists.
    */
    getString(key: string): string
    
    /**
    * @description Stores an integer number value under the given key.
    */
    putInt(key: string, value: number): void
    
    /**
    * @description Stores a double precision floating point number under the given key.
    */
    putDouble(key: string, value: number): void
    
    /**
    * @description Stores a boolean value under the given key.
    */
    putBool(key: string, value: boolean): void
    
    /**
    * @description Stores a string value under the given key.
    */
    putString(key: string, value: string): void
    
    /**
    * @description Returns a floating point value stored under the given key, or 0 if none exists.
    */
    getFloat(key: string): number
    
    /**
    * @description Stores a floating point value under the given key.
    */
    putFloat(key: string, value: number): void
    
    /**
    * @description Clears all data stored in the General Data Store.
    */
    clear(): void
    
    /**
    * @description Returns the maximum total size allowed, in bytes, of all data stored in this General Data Store.
    */
    getMaxSizeInBytes(): number
    
    /**
    * @description If `onStoreFull` has been set, this method returns the current total size, in bytes, of all data stored in this General Data Store. Otherwise, `0` is returned.
    */
    getSizeInBytes(): number
    
    /**
    * @description Returns true if a value is being stored under the given key.
    */
    has(key: string): boolean
    
    /**
    * @description Removes the value being stored under the given key. If no value exists under the key, nothing will happen.
    */
    remove(key: string): void
    
    /**
    * @description Returns a vec2 value stored under the given key, or a default vec2 if none exists.
    */
    getVec2(key: string): vec2
    
    /**
    * @description Stores a vec2 value under the given key.
    */
    putVec2(key: string, value: vec2): void
    
    /**
    * @description Returns a vec3 value stored under the given key, or a default vec3 if none exists.
    */
    getVec3(key: string): vec3
    
    /**
    * @description Stores a vec3 value under the given key.
    */
    putVec3(key: string, value: vec3): void
    
    /**
    * @description Returns a vec4 value stored under the given key, or a default vec4 if none exists.
    */
    getVec4(key: string): vec4
    
    /**
    * @description Stores a vec4 value under the given key.
    */
    putVec4(key: string, value: vec4): void
    
    /**
    * @description Returns a mat2 value stored under the given key, or a default mat2 if none exists.
    */
    getMat2(key: string): mat2
    
    /**
    * @description Stores a mat2 value under the given key.
    */
    putMat2(key: string, value: mat2): void
    
    /**
    * @description Stores a mat3 value under the given key.
    */
    getMat3(key: string): mat3
    
    /**
    * @description Stores a mat3 value under the given key.
    */
    putMat3(key: string, value: mat3): void
    
    /**
    * @description Returns a mat4 value stored under the given key, or a default mat4 if none exists.
    */
    getMat4(key: string): mat4
    
    /**
    * @description Stores a mat4 value under the given key.
    */
    putMat4(key: string, value: mat4): void
    
    /**
    * @description Returns a quat value stored under the given key, or a default quat if none exists.
    */
    getQuat(key: string): quat
    
    /**
    * @description Stores a quat value under the given key.
    */
    putQuat(key: string, value: quat): void
    
    /**
    * @description Stores an integer array under the given key.
    */
    putIntArray(key: string, value: number[]): void
    
    /**
    * @description Returns an integer array being stored under the given key, or an empty array if none exists.
    */
    getIntArray(key: string): number[]
    
    /**
    * @description Stores a floating point array under the given key.
    */
    putFloatArray(key: string, value: number[]): void
    
    /**
    * @description Returns a floating point array being stored under the given key, or an empty array if none exists.
    */
    getFloatArray(key: string): number[]
    
    /**
    * @description Stores a boolean array under the given key.
    */
    putBoolArray(key: string, value: boolean[]): void
    
    /**
    * @description Returns a boolean array being stored under the given key, or an empty array if none exists.
    */
    getBoolArray(key: string): boolean[]
    
    /**
    * @description Stores a string array under the given key.
    */
    putStringArray(key: string, value: string[]): void
    
    /**
    * @description Returns a string array being stored under the given key, or an empty array if none exists.
    */
    getStringArray(key: string): string[]
    
    /**
    * @description Stores a vec2 array under the given key.
    */
    putVec2Array(key: string, value: vec2[]): void
    
    /**
    * @description Returns a vec2 array being stored under the given key, or an empty array if none exists.
    */
    getVec2Array(key: string): vec2[]
    
    /**
    * @description Stores a vec3 array under the given key.
    */
    putVec3Array(key: string, value: vec3[]): void
    
    /**
    * @description Returns a vec3 array being stored under the given key, or an empty array if none exists.
    */
    getVec3Array(key: string): vec3[]
    
    /**
    * @description Stores a vec4 array under the given key.
    */
    putVec4Array(key: string, value: vec4[]): void
    
    /**
    * @description Returns a vec4 array being stored under the given key, or an empty array if none exists.
    */
    getVec4Array(key: string): vec4[]
    
    /**
    * @description Stores a mat2 array under the given key.
    */
    putMat2Array(key: string, value: mat2[]): void
    
    /**
    * @description Returns a mat2 array being stored under the given key, or an empty array if none exists.
    */
    getMat2Array(key: string): mat2[]
    
    /**
    * @description Stores a mat3 array under the given key.
    */
    putMat3Array(key: string, value: mat3[]): void
    
    /**
    * @description Returns a mat3 array being stored under the given key, or an empty array if none exists.
    */
    getMat3Array(key: string): mat3[]
    
    /**
    * @description Stores a mat4 array under the given key.
    */
    putMat4Array(key: string, value: mat4[]): void
    
    /**
    * @description Returns a mat4 array being stored under the given key, or an empty array if none exists.
    */
    getMat4Array(key: string): mat4[]
    
    /**
    * @description Stores a quat array under the given key.
    */
    putQuatArray(key: string, value: quat[]): void
    
    /**
    * @description Returns a quat array being stored under the given key, or an empty array if none exists.
    */
    getQuatArray(key: string): quat[]
    
    putUint8Array(key: string, value: Uint8Array): void
    
    getUint8Array(key: string): Uint8Array
    
    putInt8Array(key: string, value: any): void
    
    getInt8Array(key: string): any
    
    putUint16Array(key: string, value: Uint16Array): void
    
    getUint16Array(key: string): Uint16Array
    
    putInt16Array(key: string, value: any): void
    
    getInt16Array(key: string): any
    
    putUint32Array(key: string, value: Uint32Array): void
    
    getUint32Array(key: string): Uint32Array
    
    putInt32Array(key: string, value: any): void
    
    getInt32Array(key: string): any
    
    putFloat32Array(key: string, value: Float32Array): void
    
    getFloat32Array(key: string): Float32Array
    
    putFloat64Array(key: string, value: any): void
    
    getFloat64Array(key: string): any
    
    /**
    * @description Returns an array with all the keys in the store.
    */
    getAllKeys(): string[]
    
    /**
    * @description Callback function that gets called when the allowed storage limit has been passed.
    * The store won't be saved if it is full, so if this is called make sure to remove data until back under the limit.
    */
    onStoreFull: () => void
    
}
declare namespace GeneralDataStore {
    /**
    * @description Creates a General Data Store.
    */
    export function create(): GeneralDataStore
    

}

/**
* @description Namespace for location functionality.
*/
interface GeoLocation {
}
declare namespace GeoLocation {
    /**
    * @description Creates a new `LocationService`.
    */
    export function createLocationService(): LocationService
    
    /**
    * @description Calculates heading based on north aligned device orientation.
    */
    export function getNorthAlignedHeading(northAlignedOrientation: quat): number
    

}

/**
* @description Enumeration of supported GPS location accuracy.
*/
declare enum GeoLocationAccuracy {
    /**
    * @description Used for guiding the user. Generally accurate up to 5 meters.
    */
    Navigation,
    /**
    * @description Best possible accuracy without navigation requirement. Generally accurate up to 5 meters.
    */
    High,
    /**
    * @description Generally accurate up to 10 meters. Power efficient option.
    */
    Medium,
    /**
    * @description Generally accurate up to 100 meters. The most power efficient option.
    */
    Low
}

/**
* @description The location of the device.
*/
interface GeoPosition extends ScriptObject {
    /**
    * @description The position's latitude in decimal degrees.
    */
    latitude: number
    
    /**
    * @description The position's longitude in decimal degrees.
    */
    longitude: number
    
    /**
    * @readonly
    
    * @description Represents the direction towards which the device is facing. This value, specified in degrees, indicates how far off from heading true north the device is. 0 degrees represents true north, and the direction is determined clockwise.
    */
    heading: number
    
    /**
    * @readonly
    
    * @description Indicates whether the device is able to provide heading information.
    */
    isHeadingAvailable: boolean
    
    /**
    * @description The accuracy of the latitude and longitude properties, expressed in meters.
    */
    horizontalAccuracy: number
    
    /**
    * @description The accuracy of the altitude property, expressed in meters.
    */
    verticalAccuracy: number
    
}
declare namespace GeoPosition {
    export function create(): GeoPosition
    

}

/**
* @description Represents a GLTF 3D Model.
*/
interface GltfAsset extends Asset {
    /**
    * @description Try instantiating an object from the GLTF asset.
    */
    tryInstantiate(parent: SceneObject, material: Material): SceneObject
    
    /**
    * @description Try instantiating an object from the GLTF asset with supplied GltfSetting
    */
    tryInstantiateWithSetting(parent: SceneObject, material: Material, gltfSettings: GltfSettings): SceneObject
    
    /**
    * @readonly
    */
    extras: string
    
}

/**
* @description Settings for importing a glTF Asset.
* Use this with [GltfAsset](https://lensstudio.snapchat.com/api/classes/GltfAsset) component's `tryInstantiateWithSetting` method.
*/
interface GltfSettings extends ScriptObject {
    /**
    * @description Units for all linear distances in glTF are meters and in LensStudio are centimeters, enable this option if you want to automatically scale from meters into centimeters when importing the glTF file.
    */
    convertMetersToCentimeters: boolean
    
    /**
    * @description Whether the GLB Loader should optimize geometry. 
    */
    optimizeGeometry: boolean
    
}
declare namespace GltfSettings {
    export function create(): GltfSettings
    

}

/**
* @description Hair asset converted from an FBX containing splines to be used with [Hair Visual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#HairVisual).
*/
interface HairDataAsset extends Asset {
}

/**
* @description Component that renders hair simulation.
*/
interface HairVisual extends BaseMeshVisual {
    /**
    * @description Returns `true` if hair simulation is supported by the device.
    */
    isHardwareSupported(): boolean
    
    /**
    * @description Returns `true` if the hair resources are initialized.
    */
    isInitialized(): boolean
    
    /**
    * @description Resets the simulation. May be called if the object with the hair simulation is toggled between enabled and disabled.
    */
    resetSimulation(): void
    
    /**
    * @description Removes all hair colliders from the hair simulation.
    */
    clearColliders(): void
    
    /**
    * @description Adds a collider to the simulation.
    */
    addCollider(colliderComponent: ColliderComponent): void
    
    /**
    * @description Removes a collider from the simulation by index and returns it.
    */
    removeColliderByIndex(index: number): ColliderComponent
    
    /**
    * @description List of all colliders assigned to the hair simulation of the current hair visual.
    */
    colliders: ColliderComponent[]
    
    /**
    * @description Determines the amount of vertices on each strand. `0` means using original number of the strands. `1` is the smallest resolution for hair. Keep in mind when you change this property, the hair geometry will be regenerated.
    */
    hairResolution: number
    
    /**
    * @description Material used by this simulation to render hair.
    */
    hairMaterial: Material
    
    /**
    * @description The Width of each hair strand.
    */
    strandWidth: number
    
    /**
    * @description The taper size of each hair towards the tip.
    */
    strandTaper: number
    
    /**
    * @description Density of each clump, higher number would result in "messier" looking hair.
    */
    clumpDensity: number
    
    /**
    * @description Offset amount of hair roots.
    */
    clumpRadius: number
    
    /**
    * @description Offset amount of hair tips.
    */
    clumpTipScale: number
    
    /**
    * @description Amount of density created to thicken hair.
    */
    density: number
    
    /**
    * @description Offset of hair density.
    */
    noise: number
    
    /**
    * @description Higher value of stiffness will make hair move less.
    */
    stiffness: number
    
    /**
    * @description Higher value of damp will cause hair to have less free movement.
    */
    damp: number
    
    /**
    * @description Amount of friction for hair strands.
    */
    friction: number
    
    /**
    * @description Gravity force that gets added for hair's physics simulation.
    */
    gravity: vec3
    
    /**
    * @description Determines if collision is enabled for hair.
    */
    collisionEnabled: boolean
    
    /**
    * @description Determines how stiff the collision is.
    */
    collisionStiffness: number
    
    /**
    * @description Determines how much position offset collision will cause.
    */
    collisionOffset: number
    
    /**
    * @description Determines amount of friction for collision between hair strands.
    */
    collisionFriction: number
    
    /**
    * @description If enabled, hair strands will collide with themselves.
    */
    selfCollisionEnabled: boolean
    
    /**
    * @description Determines how stiff collision between hair strands are.
    */
    selfCollisionStiffness: number
    
    /**
    * @description Radius limit of collisions between hair strands.
    */
    selfCollisionRadius: number
    
    /**
    * @description Friction of collisions between hair strands.
    */
    selfCollisionFriction: number
    
    /**
    * @description If checked will add wind force to hair simulation.
    */
    windEnabled: boolean
    
    /**
    * @description Amount of wind force added to hair simulation.
    */
    windForce: vec3
    
    /**
    * @description This enables fallback mode which turns off physics simulations; it is recommended to enable fallback mode on lower-end devices. It will be enabled automatically if device doesn't support hair simulation feature.
    */
    fallbackModeEnabled: boolean
    
    /**
    * @description This function gets called when the hair visual is initialized in the Lens.
    */
    onInitialized: (hairVisual: HairVisual) => void
    
}

/**
* @description Provides additional data for the tracked hand. You can figure out whether the tracked hand is the left hand by accessing the *isLeft* property [true/false], as well as the probability of this data through the  *isLeftProbability*  property [0-1].

*/
interface HandSpecificData extends ObjectSpecificData {
}

/**
* @description     
*/
interface HandTracking3DAsset extends Object3DAsset {
}

/**
* @description Triggers haptic feedback on the device. (iOS Devices only)

*/
interface HapticFeedbackSystem extends ScriptObject {
    hapticFeedback(type: HapticFeedbackType): void
    
}

/**
* @description The method in which haptic feedback is provided. Use with the `HapticFeedbackSystem`.
*/
declare enum HapticFeedbackType {
    /**
    * @description Taptic engine based haptic feedback. Available only on some devices.
    */
    TapticEngine,
    /**
    * @description Vibration based haptic feedback.
    */
    Vibration
}

/**
* @description Binds the SceneObject to a tracked face.
* See the [Head Attached 3D Objects Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/face/face-effects/head-attached-3d-objects) for more information.
*/
interface Head extends Component {
    /**
    * @description Changes the attachment point type used to anchor this object to a face.
    */
    setAttachmentPointType(attachmentPointType: AttachmentPointType): void
    
    /**
    * @description Returns the total number of faces currently being tracked.
    */
    getFacesCount(): number
    
    /**
    * @description Returns the screen position of the face landmark at the passed in index.
    */
    getLandmark(index: number): vec2
    
    /**
    * @description Returns the number of face landmarks being tracked.
    */
    getLandmarkCount(): number
    
    /**
    * @description Returns a list of screen positions of all tracked landmarks.
    */
    getLandmarks(): vec2[]
    
    /**
    * @description The index of the face this head is attached to.
    */
    faceIndex: number
    
}

/**
* @description Constraints translation and a single axis of rotation.  See also: [ConstraintComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ConstraintComponent).
*/
interface HingeConstraint extends Constraint {
}

/**
* @description Used to show and hide hints to the user.
* For more information and useful helper scripts, see the [Scripting Hints Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/additional-examples/scripting-hints).

* <table cellspacing=0 cellpadding=0><thead><tr><th>Hint ID</th><th>Hint Message</th></tr></thead><tbody><tr><td>&#8220;lens_hint_blow_a_kiss&#8221;</td><td>&#8220;Blow A Kiss&#8221;</td></tr><tr><td>&#8220;lens_hint_come_closer&#8221;</td><td>&#8220;Come Closer&#8221;</td></tr><tr><td>&#8220;lens_hint_do_not_smile&#8221;</td><td>&#8220;Do Not Smile&#8221;</td></tr><tr><td>&#8220;lens_hint_do_not_try_with_a_friend&#8221;</td><td>&#8220;Do Not Try With A Friend&#8221;</td></tr><tr><td>&#8220;lens_hint_find_face&#8221;</td><td>&#8220;Find Face&#8221;</td></tr><tr><td>&#8220;lens_hint_keep_raising_your_eyebrows&#8221;</td><td>&#8220;Keep Raising Your Eyebrows&#8221;</td></tr><tr><td>&#8220;lens_hint_kiss&#8221;</td><td>&#8220;Kiss&#8221;</td></tr><tr><td>&#8220;lens_hint_kiss_again&#8221;</td><td>&#8220;Kiss Again&#8221;</td></tr><tr><td>&#8220;lens_hint_look_around&#8221;</td><td>&#8220;Look Around&#8221;</td></tr><tr><td>&#8220;lens_hint_look_down&#8221;</td><td>&#8220;Look Down&#8221;</td></tr><tr><td>&#8220;lens_hint_look_left&#8221;</td><td>&#8220;Look Left&#8221;</td></tr><tr><td>&#8220;lens_hint_look_right&#8221;</td><td>&#8220;Look Right&#8221;</td></tr><tr><td>&#8220;lens_hint_look_up&#8221;</td><td>&#8220;Look Up&#8221;</td></tr><tr><td>&#8220;lens_hint_make_some_noise&#8221;</td><td>&#8220;Make Some Noise!&#8221;</td></tr><tr><td>&#8220;lens_hint_nod_your_head&#8221;</td><td>&#8220;Nod Your Head&#8221;</td></tr><tr><td>&#8220;lens_hint_now_kiss&#8221;</td><td>&#8220;Now Kiss&#8221;</td></tr><tr><td>&#8220;lens_hint_now_open_your_mouth&#8221;</td><td>&#8220;Now Open Your Mouth&#8221;</td></tr><tr><td>&#8220;lens_hint_now_raise_your_eyebrows&#8221;</td><td>&#8220;Now Raise Your Eyebrows&#8221;</td></tr><tr><td>&#8220;lens_hint_now_smile&#8221;</td><td>&#8220;Now Smile&#8221;</td></tr><tr><td>&#8220;lens_hint_open_your_mouth&#8221;</td><td>&#8220;Open Your Mouth&#8221;</td></tr><tr><td>&#8220;lens_hint_open_your_mouth_again&#8221;</td><td>&#8220;Open Your Mouth Again&#8221;</td></tr><tr><td>&#8220;lens_hint_raise_eyebrows_or_open_mouth&#8221;</td><td>&#8220;Raise Your Eyebrows / Or / Open Your Mouth&#8221;</td></tr><tr><td>&#8220;lens_hint_raise_your_eyebrows&#8221;</td><td>&#8220;Raise Your Eyebrows&#8221;</td></tr><tr><td>&#8220;lens_hint_raise_your_eyebrows_again&#8221;</td><td>&#8220;Raise Your Eyebrows Again&#8221;</td></tr><tr><td>&#8220;lens_hint_smile&#8221;</td><td>&#8220;Smile&#8221;</td></tr><tr><td>&#8220;lens_hint_smile_again&#8221;</td><td>&#8220;Smile Again&#8221;</td></tr><tr><td>&#8220;lens_hint_swap_camera&#8221;</td><td>&#8220;Swap Camera&#8221;</td></tr><tr><td>&#8220;lens_hint_tap&#8221;</td><td>&#8220;Tap!&#8221;</td></tr><tr><td>&#8220;lens_hint_tap_a_surface&#8221;</td><td>&#8220;Tap A Surface&#8221;</td></tr><tr><td>&#8220;lens_hint_tap_ground&#8221;</td><td>&#8220;Tap The Ground&#8221;</td></tr><tr><td>&#8220;lens_hint_tap_ground_to_place&#8221;</td><td>&#8220;Tap Ground To Place&#8221;</td></tr><tr><td>&#8220;lens_hint_tap_surface_to_place&#8221;</td><td>&#8220;Tap Surface To Place&#8221;</td></tr><tr><td>&#8220;lens_hint_try_friend&#8221;</td><td>&#8220;Try It With A Friend&#8221;</td></tr><tr><td>&#8220;lens_hint_try_rear_camera&#8221;</td><td>&#8220;Try It With Your Rear Camera&#8221;</td></tr><tr><td>&#8220;lens_hint_turn_around&#8221;</td><td>&#8220;Turn Around&#8221;</td></tr><tr><td>&#8220;lens_hint_walk_through_the_door&#8221;</td><td>&#8220;Walk Through The Door&#8221;</td></tr></tbody></table>
*/
interface HintsComponent extends Component {
    /**
    * @description Shows the hint with id `hintID` for a duration of `duration` seconds. Use a duration of -1 to keep the hint onscreen forever.
    */
    showHint(hintID: string, duration: number): boolean
    
    /**
    * @description Hides the hint with id `hintID`.
    */
    hideHint(hintID: string): boolean
    
}

/**
* @description Used by the `horizontalAlignment` property in [MeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MeshVisual).
* When a [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) is attached to the same SceneObject, this determines how the mesh will be positioned horizontally.
*/
declare enum HorizontalAlignment {
    /**
    * @description The mesh will be aligned to the left side.
    */
    Left,
    /**
    * @description The mesh will be centered.
    */
    Center,
    /**
    * @description The mesh will be aligned to the right side.
    */
    Right
}

/**
* @description Options for wrapping text horizontally.
* Used by [Text component's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) `horizontalOverflow` property.
*/
declare enum HorizontalOverflow {
    /**
    * @description Text will continue drawing past horizontal boundaries.
    */
    Overflow,
    /**
    * @description Text is clipped to the width of horizontal boundaries.
    */
    Truncate,
    /**
    * @description Text wraps when reaching horizontal boundaries and continues on the next line.
    */
    Wrap,
    /**
    * @description Text will shrink to fit within the horizontal boundaries.
    */
    Shrink
}

/**
* @description The base class for parameter objects passed into event callbacks.
*/
interface IEventParameters extends ScriptObject {
}

/**
* @description A 2D visual used for drawing texture assets.
* Commonly used with [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) for drawing images on the screen.

* See the [Image guide](https://lensstudio.snapchat.com/lens-studio/references/guides/adding-content/2d/image) for more information.
*/
interface Image extends MaterialMeshVisual {
    /**
    * @description If enabled, the drawn image will be flipped horizontally.
    */
    flipX: boolean
    
    /**
    * @description If enabled, the drawn image will be flipped vertically.
    */
    flipY: boolean
    
    /**
    * @description The location of the Image's pivot point relative to its boundaries.
    * Where (-1, -1) is the bottom left corner, (0, 0) is the center, and (1, 1) is the top right corner of the Image.
    */
    pivot: vec2
    
}

/**
* @description Controls an image picker texture and UI.
* Can be accessed through [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Texture#control-textureprovider) on an Image Picker texture.
* For more information, see the [Image Picker Texture](https://lensstudio.snapchat.com/lens-studio/references/guides/adding-content/2d/media-picker-texture) guide.
*/
interface ImagePickerTextureProvider extends MediaPickerTextureProvider {
    /**
    * @description Binds a callback function for when the user selects or changes an image from the picker.
    */
    setImageChangedCallback(callback: () => void): void
    
    /**
    * @description Shows the image picker UI.
    */
    showImagePicker(): void
    
    /**
    * @description Hides the image picker UI.
    */
    hideImagePicker(): void
    
    /**
    * @description If enabled, the image picker UI will be shown automatically when the Lens starts.
    */
    autoShowImagePicker: boolean
    
}

/**
* @description Builds InputPlaceHolders for MLComponent.
*/
interface InputBuilder extends ScriptObject {
    /**
    * @description Sets the name of the InputPlaceholder to be built.
    */
    setName(name: string): InputBuilder
    
    /**
    * @description Sets the shape of the InputPlaceholder to be built.
    */
    setShape(shape: vec3): InputBuilder
    
    /**
    * @description Sets the input texture of the InputPlaceholder to be built.
    */
    setInputTexture(texture: Texture): InputBuilder
    
    /**
    * @description Builds and returns a new InputPlaceholder.
    */
    build(): InputPlaceholder
    
    /**
    * @description Sets the Transformer of the InputPlaceholder to be built.
    */
    setTransformer(transformer: Transformer): InputBuilder
    
    /**
    * @description Sets sampler for input placeholder builder.
    */
    setSampler(sampler: Sampler): InputBuilder
    
}

/**
* @description Controls input data for a neural network used by an MLComponent.
* For more information, see the [MLComponent Scripting](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/scripting-ml-component) guide.
*/
interface InputPlaceholder extends BasePlaceholder {
    /**
    * @description Texture used as input.
    */
    texture: Texture
    
    /**
    * @readonly
    
    * @description Data used as input.
    */
    data: Float32Array
    
}

/**
* @description The base class for animation tracks using integer values.
*/
interface IntAnimationTrack extends AnimationTrack {
}

/**
* @description Allows the `MeshVisual` provided to this component to handle touches on the screen (blocking Snapchat from receiving the touches), and optionally let certain touch types to pass through (let Snapchat handle the touch).

* Possible `TouchType` values:

* "TouchTypeNone"
* "TouchTypeTouch"
* "TouchTypeTap"
* "TouchTypeDoubleTap"
* "TouchTypeScale"
* "TouchTypePan"
* "TouchTypeSwipe"
*/
interface InteractionComponent extends Component {
    /**
    * @description Adds a touch type that this component will ignore.
    */
    addTouchBlockingException(exception: string): void
    
    /**
    * @description Sets the camera that will be used for interaction detection.
    */
    setCamera(camera: Camera): void
    
    /**
    * @description Adds a MeshVisual as a target for interaction detection.
    */
    addMeshVisual(meshVisual: BaseMeshVisual): void
    
    /**
    * @description Removes a MeshVisual as a target for interaction detection.
    */
    removeMeshVisual(meshVisual: BaseMeshVisual): void
    
    /**
    * @description Returns the minimum bounding box size used for detecting touches. Value range is from [0-1], relative to screen width.
    */
    getMinimumTouchSize(): number
    
    /**
    * @description Sets the minimum bounding box size used for detecting touches. Value range is from [0-1], relative to screen width.
    */
    setMinimumTouchSize(value: number): void
    
    /**
    * @readonly
    
    * @description Returns whether the user is currently gazing at the object on wearable devices, and when the user is touching on mobile devices.
    */
    isFocused: boolean
    
    /**
    * @readonly
    
    * @description Returns whether the user is currently selecting and gazing at an object on wearable devices, or touching it on mobile devices.
    */
    isSelected: boolean
    
    /**
    * @description When enabled, interaction events will be invoked only on the"closest" object, where order is defined by distance from camera and camera render order.
    */
    isFilteredByDepth: boolean
    
    /**
    * @readonly
    
    * @description Triggered when a touch event starts--either on the screen or on a touchpad. On wearables with touchpad, the coordinate returned are based on the touchpad and not the screen.
    */
    onTouchStart: event1<TouchStartEventArgs, void>
    
    /**
    * @readonly
    
    * @description Triggered when a touch position on the screen is moved.
    */
    onTouchMove: event1<TouchMoveEventArgs, void>
    
    /**
    * @readonly
    
    * @description Triggered when a touch event ends.
    */
    onTouchEnd: event1<TouchEndEventArgs, void>
    
    /**
    * @readonly
    
    * @description Triggered when the user taps on the screen.
    */
    onTap: event1<TapEventArgs, void>
    
    /**
    * @readonly
    
    * @description On mobile devices, this is triggered when the user starts touching. On wearable devices, this is triggered when the user gazes on the current object. 
    */
    onFocusStart: event1<FocusStartEventArgs, void>
    
    /**
    * @readonly
    
    * @description On mobile devices, this is triggered when the user releases their touch. On wearable devices, this is triggered when the user looks away from the current object. 
    */
    onFocusEnd: event1<FocusEndEventArgs, void>
    
    /**
    * @readonly
    
    * @description On mobile devices, this is triggered when the user starts touching the current object. On wearable devices, this is triggered when the user touches the touchpad while focusing on an object. 
    */
    onSelectStart: event1<SelectStartEventArgs, void>
    
    /**
    * @readonly
    
    * @description On mobile devices, this is triggered when the user releases their touch. On wearable devices, this is triggered when the user stops touching the current object. 
    */
    onSelectEnd: event1<SelectEndEventArgs, void>
    
    /**
    * @readonly
    
    * @description Gets called when the user triggers the primary input on their device. For example touch on touch screens.
    */
    onTriggerPrimary: event1<TriggerPrimaryEventArgs, void>
    
}

/**
* @description Represents an animation track using stepped integer value keyframes.
*/
interface IntStepAnimationTrackKeyFramed extends IntAnimationTrack {
    /**
    * @description Removes all keys.
    */
    removeAllKeys(): void
    
    /**
    * @description Removes key at index `index`.
    */
    removeKeyAt(index: number): void
    
    /**
    * @description Adds a key with value `value` at time `time`.
    */
    addKey(time: number, value: number): void
    
}

/**
* @description Represents an animation track using stepped integer value keyframes.
*/
interface IntStepNoLerpAnimationTrackKeyFramed extends IntAnimationTrack {
    /**
    * @description Removes all keys.
    */
    removeAllKeys(): void
    
    /**
    * @description Removes key at index `index`.
    */
    removeKeyAt(index: number): void
    
    /**
    * @description Adds a key with value `value` at time `time`.
    */
    addKey(time: number, value: number): void
    
}

/**
* @description Triggered when the tracked face ends a kiss.
*/
interface KissFinishedEvent extends FaceTrackingEvent {
}

/**
* @description Triggered when the tracked face starts a kiss.
*/
interface KissStartedEvent extends FaceTrackingEvent {
}

/**
* @description This event is triggered at the end of every frame, after normal [UpdateEvents](https://lensstudio.snapchat.com/api/lens-studio/Classes/Events#UpdateEvent) trigger
* but before rendering occurs.
*/
interface LateUpdateEvent extends SceneEvent {
    /**
    * @description Returns the time elapsed (in seconds) between the current frame and previous frame.
    */
    getDeltaTime(): number
    
}

/**
* @description Used to describe a set of layers that an object belongs to or interacts with.

* See [SceneObject's](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject) `layer` property, [Camera's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Camera) `renderLayer` property, and [LightSource's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#LightSource) `renderLayer` property.
*/
interface LayerSet {
    /**
    * @description Returns true if all layers in the `other` LayerSet are also present in this one.
    */
    contains(other: LayerSet): boolean
    
    /**
    * @description Returns true if this LayerSet contains no layers.
    */
    isEmpty(): boolean
    
    /**
    * @description Returns a new LayerSet combining this LayerSet and `other`.
    */
    union(other: LayerSet): LayerSet
    
    /**
    * @description Returns a new LayerSet that only contains layers present in both this LayerSet and `other`.
    */
    intersect(other: LayerSet): LayerSet
    
    /**
    * @description Returns a new LayerSet that contains layers present in this LayerSet but not present in `other`.
    */
    except(other: LayerSet): LayerSet
    
    /**
    * @description Returns a string representation of this LayerSet.
    */
    toString(): string
    
    numbers: number[]
    
}
declare namespace LayerSet {
    /**
    * @description Returns a new LayerSet based on the passed in number.
    */
    export function fromNumber(layerId: number): LayerSet
    
    /**
    * @description Creates a new LayerSet that is guaranteed to be empty and not intersect existing layers. This bypasses the normal 32 layer limit in Studio, allowing for around 65,000 unique layers.
    */
    export function makeUnique(): LayerSet
    

}

/**
* @description Collider asset generated from a mesh to be used with the [Hair Visual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#HairVisual) as part of the hair simulation.
*/
interface LevelsetColliderAsset extends BinAsset {
}

/**
* @description A levelset collision shape. A levelset is an asset that is generated to approximate a mesh.
*/
interface LevelsetShape extends Shape {
    /**
    * @description Reference to the levelset asset.
    */
    asset: LevelsetColliderAsset
    
}

/**
* @description Acts as a source of light in the scene.
* See the [Light and Shadows](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/graphics/light-and-shadow) guide for more information about lighting.
*/
interface LightSource extends Component {
    /**
    * @description The color of the light.
    */
    color: vec3
    
    /**
    * @description The strength of the light on a scale of 0.0  1.0.
    */
    intensity: number
    
    /**
    * @description If enabled, the LightSource will be able to cast shadows.
    */
    castsShadows: boolean
    
    /**
    * @description Controls the color used when casting shadows from this LightSource.
    */
    shadowColor: vec4
    
    /**
    * @description The lightness and darkness value of the shadow cast by objects from this light source.
    */
    shadowDensity: number
    
    /**
    * @description Controls the blurring size used when casting shadows from this LightSource.
    */
    shadowBlurRadius: number
    
    /**
    * @description If enabled, the LightSource will be automatically positioned based on its orientation relative to any shadow casting meshes in the scene.
    */
    autoLightSourcePosition: boolean
    
    /**
    * @description If enabled, `shadowFrustumSize` will be automatically updated based on its orientation relative to any shadow casting meshes in the scene.
    */
    autoShadowFrustumSize: boolean
    
    /**
    * @description The simulated distance of the light source from objects to calculate the softness of the shadow.
    */
    shadowFrustumSize: number
    
    /**
    * @description The minimum distance at which shadows will be calculated for this LightSource.
    */
    shadowFrustumNearClipPlane: number
    
    /**
    * @description The maximum distance at which shadows will be calculated for this LightSource.
    */
    shadowFrustumFarClipPlane: number
    
    /**
    * @description Enable if you would like the LightSource to use information from the `diffuseEnvmapTexture` for light color information.
    */
    useEnvmap: boolean
    
    /**
    * @description A value used to increase the intensity of light information derived from the `diffuseEnvmapTexture` exponentially.
    */
    envmapExposure: number
    
    /**
    * @description Controls the amount of rotation applied to the `diffuseEnvmapTexture`.
    */
    envmapRotation: number
    
    /**
    * @description A color image applied to an imaginary skybox the LightSource will use for color information.
    */
    diffuseEnvmapTexture: Texture
    
    /**
    * @description A color image applied to an imaginary skybox the light source will use for specular and reflection information.
    */
    specularEnvmapTexture: Texture
    
    /**
    * @description The set of layers this LightSource will affect.
    */
    renderLayer: LayerSet
    
}

/**
* @description Applies a liquify effect to anything rendered behind it.
*/
interface LiquifyVisual extends BaseMeshVisual {
    /**
    * @description The radius of the liquify effect circle.
    */
    radius: number
    
    /**
    * @description How strong the liquify effect is.
    */
    intensity: number
    
}

declare enum LoadStatus {
    Idle,
    Loading,
    Loaded
}

/**
* @description Asset used with the Localizations system to support custom localization strings.
*/
interface LocalizationsAsset extends Asset {
}

/**
* @description Helps convert data types to localized string representations. Accessible through `global.localizationSystem`.

* Note that formatted or localized strings may appear differently to users depending on their region.
* The example results given here are representative of a user in the United States, but may appear differently for users in other regions.
*/
interface LocalizationSystem extends ScriptObject {
    /**
    * @exposesUserData
    
    * @description Returns the language code of the language being used on the device.
    
    * Example: "en" (for English)
    */
    getLanguage(): string
    
    /**
    * @exposesUserData
    
    * @description The method takes a [localization key](https://lensstudio.snapchat.com/lens-studio/references/guides/adding-content/text/localization) and returns the localized string according to device language. Useful for localizing strings before formatting them and assigning them to Text.
    */
    localize(key: string): string
    
    /**
    * @description Returns a localized string for the date of the passed in `Date` object.
    
    * Example: "Jan 1, 2019"
    */
    getDateFormatted(date: Date): string
    
    /**
    * @description Returns a short, localized string for the date of the passed in `Date` object.
    
    * Example: "1/1/19"
    */
    getDateShortFormatted(date: Date): string
    
    /**
    * @description Returns a localized string for the time of the passed in `Date` object.
    
    * Example: "12:34 AM"
    */
    getTimeFormatted(date: Date): string
    
    /**
    * @description Returns a localized string for the date and time of the passed in `Date` object.
    
    * Example: "Jan 1, 2019 at 12:34 AM"
    */
    getDateAndTimeFormatted(date: Date): string
    
    /**
    * @description Returns a localized string for the month of the passed in `Date` object.
    
    * Example: "January"
    */
    getMonth(date: Date): string
    
    /**
    * @description Returns a localized string for the day of the week of the passed in `Date` object.
    
    * Example: "Tuesday"
    */
    getDayOfWeek(date: Date): string
    
    /**
    * @description Returns a localized, formatted string representing the number of seconds passed in.
    
    * Example: "2:06" (from 126 passed in)
    */
    getFormattedSeconds(seconds: number): string
    
    /**
    * @description Returns a localized, formatted string representation of the number passed in.
    
    * Example: "1,234" (from 1234 passed in)
    */
    getFormattedNumber(number: number): string
    
    /**
    * @description Returns a localized, formatted string representation of the celsius temperature passed in.
    
    * Example: "32F" (from 0 passed in)
    */
    getFormattedTemperatureFromCelsius(temperature: number): string
    
    /**
    * @description Returns a localized, formatted string representation of the fahrenheit temperature passed in.
    
    * Example: "32F" (from 32 passed in)
    */
    getFormattedTemperatureFromFahrenheit(temperature: number): string
    
    /**
    * @description Returns a localized, formatted string representation of the distance in meters passed in.
    
    * Example: "39.4 in" (from 1 passed in)
    */
    getFormattedDistanceFromMeters(meters: number): string
    
    /**
    * @exposesUserData
    
    * @description Use this property to set the language. Intended for debugging.
    */
    language: string
    
}

/**
* @description A component which modifies the [Transform](https://lensstudio.snapchat.com/api/classes/Transform) of the object it is on to a position in the real world, based on a `LocationAsset` and a `position`.

*/
interface LocatedAtComponent extends Component {
    /**
    * @description The geo anchored position within a LocationAsset that this object should be anchored to.
    */
    position: vec3
    
    /**
    * @description The LocationAsset which contains the position this object should be anchored to.
    
    */
    location: LocationAsset
    
    /**
    * @readonly
    */
    onCanTrack: event0<void>
    
    /**
    * @readonly
    */
    onCannotTrack: event0<void>
    
    /**
    * @readonly
    */
    onFound: event0<void>
    
    /**
    * @readonly
    */
    onLost: event0<void>
    
    /**
    * @readonly
    */
    onReady: event0<void>
    
    /**
    * @readonly
    */
    onError: event0<void>
    
    /**
    * @readonly
    */
    proximityStatus: LocationProximityStatus
    
    /**
    * @readonly
    */
    distanceToLocation: number
    
}
declare namespace LocatedAtComponent {
    /**
    * @description Creates an option object for the `LocatedAtComponent`.
    */
    export function createMappingOptions(): MappingOptions
    
    /**
    * @description Creates a mapping session.
    */
    export function createMappingSession(options: MappingOptions): MappingSession
    

}

/**
* @description Provides a frame of reference in which to localize objects to the real world. Use with [LocatedAtComponent](https://lensstudio.snapchat.com/api/classes/LocatedAtComponent).
*/
interface LocationAsset extends Asset {
    /**
    * @description Returns the `LocationAsset` Tile that is  specified by the containing tile and the specified `xOffset`, `yOffset` and `zoomOffset`. When zooming in, index offset is relative to the center. When zooming out or not zooming, index offset is relative to the containing tile.
    */
    adjacentTile(xOffset: number, yOffset: number, zoomOffset: number): LocationAsset
    
    /**
    * @description Gets the Proxy LocationID if the LocationAsset is a proxyLocation; returns null otherwise. This is useful in instances where a callback may return a mix of different location assets, and the creator of the proxy asset has specific handling in mind for particular proxy locations.
    */
    getProxyId(): string | undefined
    
}
declare namespace LocationAsset {
    /**
    * @description Get an [Anchor](https://lensstudio.snapchat.com/api/classes/anchor) based on the given GPS coordinate.
    */
    export function getGeoAnchoredPosition(longitude: number, latitude: number): Anchor
    
    /**
    * @description Gets nearby LocationAsset tiles, where each argument represents the x,y,z position relative to the current LocationAsset.
    */
    export function getNearby(xOffset: number, yOffset: number, zoomOffset: number): LocationAsset
    
    /**
    * @description Get a Proxy LocationAsset with the proxyId embedded in the locationId. proxyId must not be null, must start with an alpha char, and consist only of alphanumerics or ".".
    */
    export function getProxy(proxyId: string): LocationAsset
    
    /**
    * @description Gets the origin position of the `LocationAsset`.
    */
    export function getAROrigin(): LocationAsset
    

}

/**
* @description Provides access to location cloud storage depending upon the LocationCloudStorageOptions.
*/
interface LocationCloudStorageModule extends Asset {
    /**
    * @description Request discovery of LocationCloudStores based on the user location.
    */
    getNearbyLocationStores(options: LocationCloudStorageOptions): void
    
    /**
    * @description Stores the specified `LocationAsset`.
    */
    storeLocation(location: LocationAsset, onStoredLocation: (persistedLocationId: string) => void, onError: (error: string) => void): void
    
    /**
    * @description Retrieves the specified `LocationAsset`.
    */
    retrieveLocation(persistedLocationId: string, onRetrievedLocation: (location: LocationAsset) => void, onError: (error: string) => void): void
    
}

/**
* @description Used to configure LocationCloudStorage module with various options. 
*/
interface LocationCloudStorageOptions extends ScriptObject {
    /**
    * @description Provides a frame of reference in which to localize objects to the real world. Could be any location type.
    */
    location: LocationAsset
    
    /**
    * @description Tag to represent a collection of objects/entities
    */
    collection: string
    
    /**
    * @readonly
    
    * @description Event raised on discovery of nearby cloud stores.
    
    */
    onDiscoveredNearby: event2<LocationAsset, LocationCloudStore, void>
    
    /**
    * @readonly
    
    * @description Error event raised when failing to instantiate a location cloud store associated with that location asset.
    */
    onError: event3<LocationAsset, string, string, void>
    
}
declare namespace LocationCloudStorageOptions {
    /**
    * @description Create options to provide to LocationCloudStorageModule  
    */
    export function create(): LocationCloudStorageOptions
    

}

/**
* @description Instance of location cloud store which has a similar interface as cloud store.
*/
interface LocationCloudStore extends ScriptObject {
    /**
    * @description Gets a value on the location cloud store. Note: when updating a value, the scope and collection must match that of the original value.
    */
    getValue(key: string, readOptions: CloudStorageReadOptions, onRetrieved: (key: string, value: (mat4|mat3|mat2|vec4|vec2|vec3|boolean|quat|number|string), collection: string) => void, onError: (code: string, description: string) => void): void
    
    /**
    * @description Create/update a value on the location cloud store. Note: when updating a value, the scope and collection must match that of the original value.
    */
    setValue(key: string, value: (mat4|mat3|mat2|vec4|vec2|vec3|boolean|quat|number|string), writeOptions: CloudStorageWriteOptions, onSaved: () => void, onError: (code: string, description: string) => void): void
    
    /**
    * @description Deletes a value on the location cloud store. Note: when updating a value, the scope and collection must match that of the original value.
    */
    deleteValue(key: string, readOptions: CloudStorageReadOptions, onDeleted: () => void, onError: (code: string, description: string) => void): void
    
    /**
    * @description List values on the location cloud store. Note: when updating a value, the scope and collection must match that of the original value.
    */
    listValues(listOptions: CloudStorageListOptions, onRetrieved: (values: (mat4|mat3|mat2|vec4|vec2|vec3|boolean|quat|number|string)[][], cursor: string, collection: string) => void, onError: (code: string, description: string) => void): void
    
}

/**
* @description Used by [DeviceLocationTrackingComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#DeviceLocationTrackingComponent) to indicate the user's distance from the landmarker location.
* See the [Landmarker guide](https://lensstudio.snapchat.com/templates/landmarker) for more information.
*/
declare enum LocationProximityStatus {
    /**
    * @description User's distance cannot be determined or has not been determined yet.
    */
    Unknown,
    /**
    * @description User is close enough to the landmarker location to begin tracking.
    */
    WithinRange,
    /**
    * @description User is too far away from the landmarker location to track it.
    */
    OutOfRange
}

/**
* @description Provides access to a location's Mesh--such as when working with City Scale AR. Usually used in conjunction with `LocationTextureProvider`.
*/
interface LocationRenderObjectProvider extends RenderObjectProvider {
    /**
    * @description The Render Object's `LocationAsset` nearby tile.
    */
    location: LocationAsset
    
    /**
    * @readonly
    */
    onLocationDataDownloaded: event0<void>
    
    /**
    * @readonly
    */
    onLocationDataDownloadFailed: event0<void>
    
}
declare namespace LocationRenderObjectProvider {
    /**
    * @description Creates the location mesh.
    */
    export function create(): RenderMesh
    

}

/**
* @description The LocationService allows the user to provide their location to lens applications if they so desire. For privacy reasons, the user is asked for permission to report location information.
*/
interface LocationService extends ScriptObject {
    /**
    * @exposesUserData
    
    * @description Retrieves the device's current location.
    * `onSuccess`: a callback function that takes a GeoPosition object as its sole input parameter.
    * `onError`: a callback function that takes a string error message as its sole input parameter.
    */
    getCurrentPosition(onSucess: (geoPosition: GeoPosition) => void, onError: (error: string) => void): void
    
    /**
    * @description The accuracy of the provided position.
    */
    accuracy: GeoLocationAccuracy
    
    /**
    * @readonly
    
    * @description Event to notify when north aligned orientation data is available to use.
    */
    onNorthAlignedOrientationUpdate: event1<quat, void>
    
}

/**
* @description Provides access to a location's texture--such as when working with City Scale AR.
*/
interface LocationTextureProvider extends TextureProvider {
    /**
    * @description The location texture's `LocationAsset` nearby tile.
    */
    location: LocationAsset
    
}
declare namespace LocationTextureProvider {
    /**
    * @description Create location texture.
    */
    export function create(): Texture
    

}

/**
* @description Every frame, LookAtComponent rotates its SceneObject to face towards a target SceneObject.
*/
interface LookAtComponent extends Component {
    /**
    * @description Controls the method of rotation being used.
    */
    lookAtMode: LookAtComponent.LookAtMode
    
    /**
    * @description The "aim" and "up" vectors used when determining rotation.
    * LookAtComponent will try to point the `Aim` axis of the SceneObject towards the target,
    * while keeping the `Up` axis of the SceneObject pointing towards `worldUpVector`.
    */
    aimVectors: LookAtComponent.AimVectors
    
    /**
    * @description The vector to be considered the "up" vector when determining rotation.
    */
    worldUpVector: LookAtComponent.WorldUpVector
    
    /**
    * @description The SceneObject this LookAtComponent targets.
    */
    target: SceneObject
    
    /**
    * @description Adds an additional rotation offset.
    */
    offsetRotation: quat
    
}

declare namespace LookAtComponent {
    /**
    * @description The "aim" and "up" vectors used with [LookAtComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#LookAtComponent) when determining rotation.
    * LookAtComponent will try to point the `Aim` axis of the SceneObject towards the target,
    * while keeping the `Up` axis of the SceneObject pointing towards `worldUpVector`.
    */
    enum AimVectors {
        /**
        * @description X Aim, Y Up
        */
        XAimYUp,
        /**
        * @description X Aim, Z Up
        */
        XAimZUp,
        /**
        * @description Y Aim, X Up
        */
        YAimXUp,
        /**
        * @description Y Aim, Z Up
        */
        YAimZUp,
        /**
        * @description Z Aim, X Up
        */
        ZAimXUp,
        /**
        * @description Z Aim, Y Up
        */
        ZAimYUp,
        /**
        * @description X Aim, -Y Up
        */
        XAimNegativeYUp,
        /**
        * @description X Aim, -Z Up
        */
        XAimNegativeZUp,
        /**
        * @description Y Aim, -X Up
        */
        YAimNegativeXUp,
        /**
        * @description Y Aim, -Z Up
        */
        YAimNegativeZUp,
        /**
        * @description Z Aim, -X Up
        */
        ZAimNegativeXUp,
        /**
        * @description Z Aim, -Y Up
        */
        ZAimNegativeYUp,
        /**
        * @description -X Aim, Y Up
        */
        NegativeXAimYUp,
        /**
        * @description -X Aim, Z Up
        */
        NegativeXAimZUp,
        /**
        * @description -Y Aim, X Up
        */
        NegativeYAimXUp,
        /**
        * @description -Y Aim, Z Up
        */
        NegativeYAimZUp,
        /**
        * @description -Z Aim, X Up
        */
        NegativeZAimXUp,
        /**
        * @description -Z Aim, Y Up
        */
        NegativeZAimYUp,
        /**
        * @description -X Aim, -Y Up
        */
        NegativeXAimNegativeYUp,
        /**
        * @description -X Aim, -Z Up
        */
        NegativeXAimNegativeZUp,
        /**
        * @description -Y Aim, -X Up
        */
        NegativeYAimNegativeXUp,
        /**
        * @description -Y Aim, -Z Up
        */
        NegativeYAimNegativeZUp,
        /**
        * @description -Z Aim, -X Up
        */
        NegativeZAimNegativeXUp,
        /**
        * @description -Z Aim, -Y Up
        */
        NegativeZAimNegativeYUp
    }

}

declare namespace LookAtComponent {
    /**
    * @description Modes used in `LookAtComponent.lookAtMode` to determine the rotation method being used.
    */
    enum LookAtMode {
        /**
        * @description Rotation is based on the target object's position
        */
        LookAtPoint,
        /**
        * @description Rotation is based on the target object's rotation
        */
        LookAtDirection
    }

}

declare namespace LookAtComponent {
    /**
    * @description Used with [LookAtComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#LookAtComponent) to set the "up" vector when determining rotation.
    */
    enum WorldUpVector {
        /**
        * @description Scene's X vector
        */
        SceneX,
        /**
        * @description Scene's Y vector
        */
        SceneY,
        /**
        * @description Scene's Z vector
        */
        SceneZ,
        /**
        * @description Target object's X vector
        */
        TargetX,
        /**
        * @description Target object's Y vector
        */
        TargetY,
        /**
        * @description Target object's Z vector
        */
        TargetZ,
        /**
        * @description Current object's X vector
        */
        ObjectX,
        /**
        * @description Current object's Y vector
        */
        ObjectY,
        /**
        * @description Current object's Z vector
        */
        ObjectZ
    }

}

/**
* @description Namespace for Machine Learning related classes and methods.
* For more information, see the [Machine Learning Overview](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-overview).
*/
interface MachineLearning {
}
declare namespace MachineLearning {
    /**
    * @description Creates a new InputBuilder object.
    */
    export function createInputBuilder(): InputBuilder
    
    /**
    * @description Creates a new OutputBuilder object.
    */
    export function createOutputBuilder(): OutputBuilder
    
    /**
    * @description Creates a new TransformerBuilder object.
    */
    export function createTransformerBuilder(): TransformerBuilder
    
    /**
    * @description Creates SamplerBuilder.
    */
    export function createSamplerBuilder(): SamplerBuilder
    
    /**
    * @description Creates a new SpectrogramBuilder object.
    */
    export function createSpectrogramBuilder(): SpectrogramBuilder
    
    /**
    * @description Creates a new MelSpectrogramBuilder object.
    */
    export function createMelSpectrogramBuilder(): MelSpectrogramBuilder
    
    /**
    * @description Creates a new MFCCBuilder object.
    */
    export function createMFCCBuilder(): MFCCBuilder
    
    /**
    * @description Creates a new DeltaBuilder object.
    */
    export function createDeltaBuilder(): DeltaBuilder
    
    /**
    * @description Creates a new DelayBuilder object.
    */
    export function createDelayBuilder(): DelayBuilder
    
    export function createPitchShifterBuilder(): PitchShifterBuilder
    
    export function createNoiseReductionBuilder(): NoiseReductionBuilder
    

}

declare namespace MachineLearning {
    /**
    * @description Timing options for when MLComponent should start or stop running. Used with `MLComponent.runScheduled()`.
    * For more information, see the [MLComponent Scripting](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/scripting-ml-component) guide.
    */
    enum FrameTiming {
        /**
        * @description Only valid as an end timing. There is no exact time specified when MLComponent should finish its run.
        */
        None,
        /**
        * @description Run during MLComponent update, before script update.
        */
        Update,
        /**
        * @description Run in MLComponent LateUpdate, after all scripts update, but before they get LateUpdate.
        */
        LateUpdate,
        /**
        * @description Run at a specific point during frame rendering.
        */
        OnRender
    }

}

declare namespace MachineLearning {
    /**
    * @description Inference modes used by `MLComponent.inferenceMode`. Each mode describes how the neural network will be run.
    */
    enum InferenceMode {
        /**
        * @description MLComponent will run the neural network on CPU. Available on all devices.
        */
        CPU,
        /**
        * @description MLComponent will attempt to run the neural network on GPU. If the device doesn't support it, CPU mode will be used instead.
        */
        GPU,
        /**
        * @description MLComponent will attempt to use a dedicated hardware accelerator to run the neural network. If the device doesn't support it, GPU mode will be used instead.
        */
        Accelerator,
        /**
        * @description MLComponent will automatically decide how to run the neural network based on what is supported. It will start with Accelerator, then fall back to GPU, then CPU.
        */
        Auto
    }

}

declare namespace MachineLearning {
    /**
    * @description Describes the current state of the MLComponent model.
    * For more information, see the [MLComponent Scripting](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/scripting-ml-component) guide.
    */
    enum ModelState {
        /**
        * @description Model is running
        */
        Running,
        /**
        * @description Model is loading
        */
        Loading,
        /**
        * @description Model is built and ready to run
        */
        Idle,
        /**
        * @description Model is not ready to run
        */
        NotReady
    }

}

declare namespace MachineLearning {
    /**
    * @description Types of output used by OutputPlaceholder.
    */
    enum OutputMode {
        /**
        * @description The output will be in the form of a Texture.
        */
        Texture,
        /**
        * @description The output will be in the form of a Float32Array.
        */
        Data
    }

}

/**
* @description Handles input information from user touch input via the [TouchComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#TouchComponent) to control Scale, Rotation, and Translation of objects.
*/
interface ManipulateComponent extends Component {
    /**
    * @description Repositions the object to be within the bounds of `minDistance`, `maxDistance`.
    */
    clampWorldPosition(): void
    
    /**
    * @description Checks for an intersection point between the manipulation plane and a line extending
    * from the camera through the specified screen space point. The screen point is passed in as (x, y) with both values
    * ranging from ([0-1], [0-1]), (0,0) being left-top and (1,1) being right-bottom. The result is returned as a
    * [ManipulateFrameIntersectResult](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#ManipulateFrameIntersectResult) object.
    */
    intersectManipulateFrame(screenSpacePoint: vec2): ManipulateFrameIntersectResult
    
    /**
    * @description Enables or disables the specified ManipulateType for this ManipulateComponent.
    */
    enableManipulateType(type: ManipulateType, enable: boolean): void
    
    /**
    * @description Returns whether the specified ManipulateType is enabled for this ManipulateComponent.
    */
    isManipulateTypeEnabled(type: ManipulateType): boolean
    
    /**
    * @readonly
    
    * @description Event fired when manipulation starts.
    */
    onManipulateStart: event1<ManipulateStartEventArgs, void>
    
    /**
    * @readonly
    
    * @description Event fired when manipulation ends.
    */
    onManipulateEnd: event1<ManipulateEndEventArgs, void>
    
    /**
    * @description Changes swivel behavior based on the object's height relative to the camera.
    */
    isContextualSwivel: boolean
    
    /**
    * @description The minimum height of the object.
    */
    minHeight: number
    
    /**
    * @description The maximum height of the object.
    */
    maxHeight: number
    
    /**
    * @description The minimum size the object can shrink to.
    */
    minScale: number
    
    /**
    * @description The maximum size the object can scale to.
    */
    maxScale: number
    
    /**
    * @description The minimum distance the object can be from the user.
    */
    minDistance: number
    
    /**
    * @description The maximum distance the object can travel from the user.
    */
    maxDistance: number
    
    /**
    * @description Multiplier for swivel rotation speed.
    * For example, a value of 0.5 will cut rotation speed in half,
    * and a value of 2.0 will double rotation speed.
    */
    rotationScale: number
    
    /**
    * @readonly
    
    * @description Returns whether the object that the Manipulate Component is on is currently being manipulated
    */
    isManipulating: boolean
    
}

/**
* @description This event is triggered when manipulation on the object ends.
*/
interface ManipulateEndEvent extends SceneObjectEvent {
}

/**
* @description Arguments used with the `ManipulateComponent.onManipulateEnd` event.
*/
interface ManipulateEndEventArgs extends ScriptObject {
}

/**
* @description Result object returned from [ManipulateComponent.intersectManipulateFrame()](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ManipulateComponent).
*/
interface ManipulateFrameIntersectResult {
    /**
    * @description If there was a valid intersection, returns the intersection point in world space.
    */
    getIntersectionPoint(): vec3
    
    /**
    * @description Returns whether there was a valid intersection.
    */
    isValid(): boolean
    
}

/**
* @description This event is triggered when manipulation on the object begins.
*/
interface ManipulateStartEvent extends SceneObjectEvent {
}

/**
* @description Arguments used with the `ManipulateComponent.onManipulateStart` event.
*/
interface ManipulateStartEventArgs extends ScriptObject {
}

/**
* @description Enum values specifying each type of manipulation. See [ManipulateComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ManipulateComponent).
*/
declare enum ManipulateType {
    /**
    * @description The object can be scaled by pinching with two fingers.
    */
    Scale,
    /**
    * @description The object can be rotated by swiveling with two fingers.
    */
    Swivel,
    /**
    * @description The object can be moved by touching and dragging.
    */
    Drag
}

/**
* @description Module for providing Map utils. 
*/
interface MapModule extends Asset {
    /**
    * @description Creates a new texture which holds a `MapTextureProvider`.
    */
    createMapTextureProvider(): Texture
    
    /**
    * @description Returns the position of the marker (relative to the associated map tile) based on the specified `longitude`, `latitude` and `location`. The top left corner of the provided `location` has a value of `[0,0]` and the right down corner has a value of `[1.1]`.
    */
    longLatToImageRatio(longitude: number, latitude: number, location: LocationAsset): vec2
    
}

interface MappingOptions extends ScriptObject {
    location: LocationAsset
    
    policy: string
    
    locationCloudStorageModule: LocationCloudStorageModule
    
}

interface MappingSession extends ScriptObject {
    cancel(): void
    
    checkpoint(): void
    
    throttling: MappingSession.MappingThrottling
    
    /**
    * @readonly
    */
    canCheckpoint: boolean
    
    /**
    * @readonly
    */
    capacityUsed: number
    
    /**
    * @readonly
    */
    quality: number
    
    /**
    * @readonly
    */
    onMapped: event1<LocationAsset, void>
    
    handheldMinimumSize: number
    
    handheldMaximumSize: number
    
    wearableMinimumSize: number
    
    wearableMaximumSize: number
    
}

declare namespace MappingSession {
    enum MappingThrottling {
        Auto,
        Foreground,
        Background,
        Off
    }

}

interface MapTextureProvider extends TextureProvider {
    /**
    * @readonly
    
    * @description A function that gets called when location data is downloaded.
    */
    onReady: event0<void>
    
    /**
    * @readonly
    
    * @description A function that gets called when location data fails to download.
    */
    onFailed: event0<void>
    
    /**
    * @description The location asset associated with the `MapTextureProvider`.
    */
    location: LocationAsset
    
}

/**
* @description Defines a marker to use for Marker Tracking with [MarkerTrackingComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MarkerTrackingComponent).
* For more information, see the [Marker Tracking](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/world/marker-tracking) guide.
*/
interface MarkerAsset extends Asset {
    /**
    * @description Returns the aspect ratio (width / height) of the texture used by the marker asset.
    */
    getAspectRatio(): number
    
    /**
    * @description The height of the marker asset in real-life centimeters. Used to provide accurate transformation.
    * A [MarkerTrackingComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MarkerTrackingComponent) using this MarkerAsset will be scaled so that
    * one unit in the SceneObject's local space is equal to one centimeter in real life.
    */
    height: number
    
}

/**
* @description Base class for marker providers.
* For more information, see the [Marker Tracking](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/world/marker-tracking) guide.
*/
interface MarkerProvider extends Provider {
}

/**
* @description Used to track images in the camera. Moves the containing object's transform to match the detected image.
* For more information, see the [Marker Tracking](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/world/marker-tracking) guide.
*/
interface MarkerTrackingComponent extends Component {
    /**
    * @description Returns whether the marker image is currently being tracked in camera.
    */
    isTracking(): boolean
    
    /**
    * @description The marker asset describing the tracking target.
    */
    marker: MarkerAsset
    
    /**
    * @description If true, child objects of this Component's [SceneObject](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject) will be disabled when the marker image is not being tracked.
    */
    autoEnableWhenTracking: boolean
    
    /**
    * @description A function that gets called when marker tracking begins.
    */
    onMarkerFound: () => void
    
    /**
    * @description A function that gets called when marker tracking is lost.
    */
    onMarkerLost: () => void
    
}

/**
* @description Clips visuals and Interaction Component events within a tree hierarchy. Any Visual or Interaction Components will be clipped user defined 2D bounds. These 2D bounds are defined by a Screen Transform. Useful for clipping some screen transforms--for example a scroll view. 
*/
interface MaskingComponent extends BaseMeshVisual {
    /**
    * @description The radius of the mask's corner.
    */
    cornerRadius: number
    
}

/**
* @description A 2x2 matrix.
*/
declare class mat2 {
    /**
    * @description Creates a new mat2, defaulting to identity values.
    */
    constructor()
    
    /**
    * @description Returns the result of adding the two matrices together.
    */
    add(mat: mat2): mat2
    
    /**
    * @description Returns the result of subtracting the two matrices.
    */
    sub(mat: mat2): mat2
    
    /**
    * @description Returns the result of multiplying the two matrices.
    */
    mult(mat: mat2): mat2
    
    /**
    * @description Returns the result of dividing the two matrices.
    */
    div(mat: mat2): mat2
    
    /**
    * @description Returns the determinant of the matrix.
    */
    determinant(): number
    
    /**
    * @description Returns the inverse of the matrix.
    */
    inverse(): mat2
    
    /**
    * @description Returns the transpose of this matrix.
    */
    transpose(): mat2
    
    /**
    * @description Returns whether the two matrices are equal.
    */
    equal(mat: mat2): boolean
    
    /**
    * @description Returns the result of scalar multiplying the matrix.
    */
    multiplyScalar(scalar: number): mat2
    
    /**
    * @description Returns a string representation of the matrix.
    */
    toString(): string
    
    /**
    * @description Returns a string representation of the matrix.
    */
    description: string
    
    /**
    * @description The first column of the matrix.
    */
    column0: vec2
    
    /**
    * @description The second column of the matrix.
    */
    column1: vec2
    
}
declare namespace mat2 {
    /**
    * @description Returns the identity matrix.
    */
    export function identity(): mat2
    
    /**
    * @description Returns a matrix with all zero values.
    */
    export function zero(): mat2
    

}

/**
* @description A 3x3 matrix.
*/
declare class mat3 {
    /**
    * @description Creates a new mat3, defaulting to identity values.
    */
    constructor()
    
    /**
    * @description Returns the result of adding the two matrices together.
    */
    add(mat: mat3): mat3
    
    /**
    * @description Returns the result of subtracting the two matrices.
    */
    sub(mat: mat3): mat3
    
    /**
    * @description Returns the result of multiplying the two matrices.
    */
    mult(mat: mat3): mat3
    
    /**
    * @description Returns the result of dividing the two matrices.
    */
    div(mat: mat3): mat3
    
    /**
    * @description Returns the determinant of the matrix.
    */
    determinant(): number
    
    /**
    * @description Returns the inverse of the matrix.
    */
    inverse(): mat3
    
    /**
    * @description Returns the transpose of this matrix.
    */
    transpose(): mat3
    
    /**
    * @description Returns whether the two matrices are equal.
    */
    equal(mat: mat3): boolean
    
    /**
    * @description Returns the result of scalar multiplying the matrix.
    */
    multiplyScalar(scalar: number): mat3
    
    /**
    * @description Returns a string representation of the matrix.
    */
    toString(): string
    
    /**
    * @description Returns a string representation of the matrix.
    */
    description: string
    
    /**
    * @description The first column of the matrix.
    */
    column0: vec3
    
    /**
    * @description The second column of the matrix.
    */
    column1: vec3
    
    /**
    * @description The third column of the matrix.
    */
    column2: vec3
    
}
declare namespace mat3 {
    /**
    * @description Returns the identity matrix.
    */
    export function identity(): mat3
    
    /**
    * @description Returns a matrix with all zero values.
    */
    export function zero(): mat3
    
    /**
    * @description Returns a matrix representing the specified rotation.
    */
    export function makeFromRotation(arg1: quat): mat3
    

}

/**
* @description A 4x4 matrix.
*/
declare class mat4 {
    /**
    * @description Creates a new mat4, defaulting to identity values.
    */
    constructor()
    
    /**
    * @description Returns the result of adding the two matrices together.
    */
    add(mat: mat4): mat4
    
    /**
    * @description Returns the result of subtracting the two matrices.
    */
    sub(mat: mat4): mat4
    
    /**
    * @description Returns the result of multiplying the two matrices.
    */
    mult(mat: mat4): mat4
    
    /**
    * @description Returns the result of dividing the two matrices.
    */
    div(mat: mat4): mat4
    
    /**
    * @description Returns the determinant of the matrix.
    */
    determinant(): number
    
    /**
    * @description Returns the inverse of the matrix.
    */
    inverse(): mat4
    
    /**
    * @description Returns the transpose of this matrix.
    */
    transpose(): mat4
    
    /**
    * @description Returns whether the two matrices are equal.
    */
    equal(mat: mat4): boolean
    
    /**
    * @description Returns the point `point` multiplied by this matrix.
    */
    multiplyPoint(point: vec3): vec3
    
    /**
    * @description Returns the direction vector multiplied by this matrix.
    */
    multiplyDirection(direction: vec3): vec3
    
    /**
    * @description Returns the vector multiplied by this matrix.
    */
    multiplyVector(vector: vec4): vec4
    
    /**
    * @description Returns the result of scalar multiplying the matrix.
    */
    multiplyScalar(scalar: number): mat4
    
    /**
    * @description Returns an euler angle representation of this matrix's rotation, in radians.
    */
    extractEulerAngles(): vec3
    
    /**
    * @description Returns a string representation of the matrix.
    */
    toString(): string
    
    /**
    * @description Returns a string representation of the matrix.
    */
    description: string
    
    /**
    * @description The first column of the matrix.
    */
    column0: vec4
    
    /**
    * @description The second column of the matrix.
    */
    column1: vec4
    
    /**
    * @description The third column of the matrix.
    */
    column2: vec4
    
    /**
    * @description The fourth column of the matrix.
    */
    column3: vec4
    
}
declare namespace mat4 {
    /**
    * @description Returns the identity matrix.
    */
    export function identity(): mat4
    
    /**
    * @description Returns a matrix with all zero values.
    */
    export function zero(): mat4
    
    /**
    * @description Returns a new matrix with translation `translation`, rotation `rotation`, and scale `scale`.
    */
    export function compose(translation: vec3, rotation: quat, scale: vec3): mat4
    
    /**
    * @description Returns a new matrix using the provided vectors.
    */
    export function makeBasis(x: vec3, y: vec3, z: vec3): mat4
    
    /**
    * @description Returns a new matrix generated using the provided arguments.
    */
    export function lookAt(eye: vec3, center: vec3, up: vec3): mat4
    
    /**
    * @description Returns a new matrix generated using the provided arguments.
    */
    export function perspective(fovY: number, aspect: number, zNear: number, zFar: number): mat4
    
    /**
    * @description Returns a new matrix generated using the provided arguments.
    */
    export function orthographic(left: number, right: number, bottom: number, top: number, zNear: number, zFar: number): mat4
    
    /**
    * @description Returns a new matrix with rotation `rotation`.
    */
    export function fromRotation(rotation: quat): mat4
    
    /**
    * @description Returns a new matrix with scale `scale`.
    */
    export function fromScale(scale: vec3): mat4
    
    /**
    * @description Returns a new matrix with the translation `translation`.
    */
    export function fromTranslation(translation: vec3): mat4
    
    /**
    * @description Returns a new matrix with x euler angle `xAngle` (in radians).
    */
    export function fromEulerX(xAngle: number): mat4
    
    /**
    * @description Returns a new matrix with y euler angle `yAngle` (in radians).
    */
    export function fromEulerY(yAngle: number): mat4
    
    /**
    * @description Returns a new matrix with z euler angle `zAngle` (in radians).
    */
    export function fromEulerZ(zAngle: number): mat4
    
    /**
    * @description Returns a new matrix with the specified euler angles (in radians).
    */
    export function fromEulerAngles(euler: vec3): mat4
    
    /**
    * @description Returns the two matrices multiplied component-wise.
    */
    export function compMult(arg1: mat4, arg2: mat4): mat4
    
    /**
    * @description Returns the outer product of the two matrices.
    */
    export function outerProduct(arg1: vec4, arg2: vec4): mat4
    

}

/**
* @description An asset that describes how visual objects should appear.
* Each Material is a collection of [Passes](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Pass) which define the actual rendering passes.
* Materials are used by [MeshVisuals](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MeshVisual) for drawing meshes in the scene.
*/
interface Material extends Asset {
    /**
    * @description Returns a copy of the Material.
    */
    clone(): Material
    
    /**
    * @description Returns the number of [Passes](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Pass) for the Material.
    */
    getPassCount(): number
    
    /**
    * @description Returns the [Pass](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Pass) of the Material at index `index`.
    */
    getPass(index: number): Pass
    
    /**
    * @description The first Pass of the Material.
    */
    mainPass: Pass
    
}

/**
* @description Base class for all MeshVisual components using [Materials](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#Material) to render.
* Comparable to the former class "MeshVisual", which was split into the classes:
* [BaseMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#BaseMeshVisual),
* [MaterialMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#MaterialMeshVisual),
* and [RenderMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#RenderMeshVisual).
*/
interface MaterialMeshVisual extends BaseMeshVisual {
    /**
    * @description Returns the number of Materials used for rendering.
    */
    getMaterialsCount(): number
    
    /**
    * @description Returns the Material at index `index`.
    */
    getMaterial(index: number): Material
    
    /**
    * @description Adds a Material to use for rendering.
    */
    addMaterial(material: Material): void
    
    /**
    * @description Clears all Materials.
    */
    clearMaterials(): void
    
    /**
    * @description Get the array of materials used by the MaterialMeshVisual.
    */
    materials: Material[]
    
    /**
    * @description Returns the first Material.
    */
    mainMaterial: Material
    
    /**
    * @description Returns the `mainPass` of the `mainMaterial`.
    */
    mainPass: Pass
    
}

/**
* @description Provides useful math utility functions.
*/
interface MathUtils {
}
declare namespace MathUtils {
    /**
    * @description Generate a random floating-point value in the range `[lo, hi)`. Note, the range is inclusive at the lower end, and exclusive at the upper.
    */
    export function randomRange(lo: number, hi: number): number
    
    /**
    * @description Linearly interpolate from floating-point `a` to `b`, based on fraction `t` (where t=0.0 returns a, and t=1.0 returns b). This is equivalent to the vec2 and vec3 lerp() functions, but operates on scalar floating-point values.
    */
    export function lerp(a: number, b: number, time: number): number
    
    /**
    * @description Clamp floating-point value `v` in the range `[lo, hi]` (inclusive).
    */
    export function clamp(v: number, lo: number, hi: number): number
    
    /**
    * @description Map floating-point `v` from range [inMin, inMax] to [outMin, outMax].
    */
    export function remap(v: number, inMin: number, inMax: number, outMin: number, outMax: number): number
    
    /**
    * @description Constant mapping degrees to radians. This equals pi/180.
    */
    let DegToRad: number
    
    /**
    * @description Constant mapping radians to degrees. This equals 180/pi.
    */
    let RadToDeg: number
    

}

/**
* @description Settings for the physical substance, such as friction and bounciness, of a collider. If unset, uses the default matter from the world settings.
*/
interface Matter extends Asset {
    /**
    * @description Coefficient of friction. 
    * The effective friction between two objects is the product of both objects' friction. So for example, the default between two objects is 0.5*0.5=0.25. 
    * Typical ranges are between 0 and 1, but larger values (up to 10) are allowed. This may be used to increase the effective friction when colliding against another object with low friction. 
    * This is a very simplified approximation of realistic friction, since it does not correctly take into account the combined surface characteristics of both objects. It also doesn't distinguish between static and dynamic friction.
    */
    friction: number
    
    /**
    * @description Friction applied to rolling objects. This isn't physically realistic, but helps prevent objects from rolling indefinitely.
    */
    rollingFriction: number
    
    /**
    * @description Friction applied to spinning objects. This isn't physically realistic, but helps prevent objects from spinning indefinitely.
    */
    spinningFriction: number
    
    /**
    * @description Bounciness (AKA restitution), for static colliders. This is the proportion of energy preserved after a collision, in the range 0 to 1. 
    * This defaults to 1.0. Typically we use high bounciness for static colliders because they are unaffected by collision, and thus maximally preserve energy. 
    * The effective bounciness for a collision is the product of both colliding objects. So if your bounciness setting appears to have no effect, it probably means the object it's colliding with has 0 bounciness.
    */
    staticBounciness: number
    
    /**
    * @description Bounciness, or restitution, for dynamic bodies. This is the proportion of energy preserved after a collision, in the range 0 to 1. 
    * This defaults to 0.0. Dynamic bounciness should usually be less than 1.0, to prevent energy from accumulating excessively (there is a certain amount of energy gained due to collision response). 
    * The effective bounciness for a collision is the product of both colliding objects. So if your bounciness setting appears to have no effect, it probably means the object it's colliding with has 0 bounciness.
    */
    dynamicBounciness: number
    
}

/**
* @description Base class for Texture Providers based on selectable media.
*/
interface MediaPickerTextureProvider extends TextureProvider {
    /**
    * @description Sets a callback to fire when a file is picked through media picking UI.
    */
    setFilePickedCallback(callback: () => void): void
    
    /**
    * @description Requests SDK to show media picking UI.
    */
    showMediaPicker(): void
    
    /**
    * @description Requests SDK to hide media picking UI.
    */
    hideMediaPicker(): void
    
    /**
    * @description Requests face mesh and sets a callback to fire when it's ready.
    */
    setFaceMeshReadyCallback(callback: () => void): void
    
    /**
    * @description If set to true, MediaPickerTextureProvider will request media picker UI automatically during loading.
    */
    autoShowMediaPicker: boolean
    
    /**
    * @description If set to true, MediaPickerTextureProvider will provide a proper texture transform for image with face(s), that will "zoom" UVs to the first found face on the image.
    */
    cropFace: boolean
    
    /**
    * @readonly
    
    * @description Returns underlying VideoTextureProvider for the last selected media file. If the last media file was not video, null is returned. 
    */
    videoControl: VideoTextureProvider
    
    /**
    * @readonly
    
    * @description Returns underlying TextureProvider for the last selected media file. If the last media file was not image, null is returned.
    */
    imageControl: TextureProvider
    
    /**
    * @readonly
    
    * @description Returns true if an image is selected, or a video file has been loaded and is ready for decoding, false otherwise.
    */
    isContentReady: boolean
    
    /**
    * @description MediaPickerTextureProvider will allow users to select video files through media picker UI.
    */
    isVideoPickingEnabled: boolean
    
    /**
    * @description MediaPickerTextureProvider will allow users to select all images files through media picker UI.
    */
    isImagePickingEnabled: boolean
    
    /**
    * @description MediaPickerTextureProvider will allow users to select only images with detected faces through media picker UI. See also "cropFace" option.
    */
    isFaceImagePickingEnabled: boolean
    
    /**
    * @description Returns underlying TextureProvider for the last selected media file. If the last media file was not image with at least one face, null is returned.
    */
    faceImageControl: FaceTextureProvider
    
}

/**
* @description Computes a mel scale spectrogram - a spectrogram where the frequencies are converted to the mel scale.
*/
interface MelSpectrogram extends ScriptObject {
    /**
    * @description Process in tensor with shape, write result to the outTensor and return the shape of outTensor.
    */
    process(inTensor: Float32Array, inShape: vec3, outTensor: Float32Array): vec3
    
}

/**
* @description A builder class for MelSpectrogram.
*/
interface MelSpectrogramBuilder extends ScriptObject {
    /**
    * @description Create new MelSpectrogram object.
    */
    build(): MelSpectrogram
    
    /**
    * @description Length of the window, which is the length of frameSize and then padded with zeros to match fftSize.
    */
    setFrameSize(frameSize: number): MelSpectrogramBuilder
    
    /**
    * @description Set number of samples between successive fft segments.
    */
    setHopSize(hopSize: number): MelSpectrogramBuilder
    
    /**
    * @description Set length of the fft window.
    */
    setFFTSize(fftSize: number): MelSpectrogramBuilder
    
    /**
    * @description Set number of mel bins.
    */
    setNumMel(numMel: number): MelSpectrogramBuilder
    
    /**
    * @description Set min frequency.
    */
    setMinFreq(minFreq: number): MelSpectrogramBuilder
    
    /**
    * @description Set max frequency.
    */
    setMaxFreq(maxFreq: number): MelSpectrogramBuilder
    
    /**
    * @description Set number of samples per second.
    */
    setSampleRate(sampleRate: number): MelSpectrogramBuilder
    
}

/**
* @description A class for generating meshes at runtime.
*/
declare class MeshBuilder {
    /**
    * @description Creates a new MeshBuilder with the specified vertex layout.
    
    * Layout is given as a list of "attribute" objects with the following properties:
    
    * __name__ - Attribute name
    * __components__ - Size of the attribute (how many float values it uses)
    * __normalized__ (optional) - Optional property declaring that the attribute should be normalized
    
    * ```js
    * var builder = new MeshBuilder([
    *     // vertex position (x,y,z)
    *     { name: "position", components: 3 },
    *     // normal vector (x,y,z)
    *     { name: "normal", components: 3, normalized: true },
    *     // texture UV (u,v)
    *     { name: "texture0", components: 2 },
    * ]);
    * ```
    */
    constructor(layout: unknown[])
    
    /**
    * @description Returns a RenderMesh asset that can be applied to a MeshVisual's `mesh` property. This asset stays linked to the MeshBuilder that provided it, so making changes to the mesh data and calling `updateMesh()` will update the RenderMesh as well.
    */
    getMesh(): RenderMesh
    
    /**
    * @description Rebuilds the MeshAsset controlled by this MeshBuilder using the current mesh data.
    */
    updateMesh(): void
    
    /**
    * @description Checks whether the current data entered will create a valid mesh.
    */
    isValid(): boolean
    
    /**
    * @description Takes a list of list of vertex values according to the layout.
    */
    appendVertices(verts: number[][]): void
    
    /**
    * @description Similar to `appendVertices`, but takes all values in one large array.
    */
    appendVerticesInterleaved(verts: number[]): void
    
    /**
    * @description Sets data for a single vertex at vertex index `index`.
    */
    setVertexInterleaved(index: number, verts: number[]): void
    
    /**
    * @description Appends indices to the index list.
    */
    appendIndices(indices: number[]): void
    
    /**
    * @description Removes all indices starting at index `from` and ending before index `to`.
    */
    eraseIndices(from: number, to: number): void
    
    /**
    * @description Removes all vertex data starting at vertex index `from` and ending before vertex index `to`.
    */
    eraseVertices(from: number, to: number): void
    
    /**
    * @description Returns the number of vertices in the vertex list.
    */
    getVerticesCount(): number
    
    /**
    * @description Returns the number of indices in the index list.
    */
    getIndicesCount(): number
    
    /**
    * @description The topology type used for the mesh.
    */
    topology: MeshTopology
    
    /**
    * @description The index data type used by this MeshBuilder. `MeshIndexType.UInt16` is the value normally used for this.
    */
    indexType: MeshIndexType
    
}

/**
* @description Formats of mesh classification used by WorldRenderObjectProvider.
*/
declare enum MeshClassificationFormat {
    /**
    * @description Do not bake classifications to mesh
    */
    None,
    /**
    * @description Classifications are baked per vertex - vertices with multiple classes will use the value from the last face
    */
    PerVertexFast
}

/**
* @description Possible index data types used by [MeshBuilder](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#MeshBuilder). `MeshIndexType.UInt16` is the value normally used.
*/
declare enum MeshIndexType {
    /**
    * @description No index data type
    */
    None,
    /**
    * @description Unsigned integer, this is the value normally used
    */
    UInt16
}

interface MeshRenderObjectProvider extends RenderObjectProvider {
}

declare enum MeshShadowMode {
    None,
    Caster,
    Receiver
}

/**
* @description Allows meshes to be used as collision shapes, for ColliderComponent and BodyComponent.
*/
interface MeshShape extends Shape {
    /**
    * @description Sets geometry from a render mesh. The render mesh is automatically converted to a collision mesh.
    */
    mesh: RenderMesh
    
    /**
    * @description Specifies skinning component used to animate skinned meshes.
    */
    skin: Skin
    
    /**
    * @description Bake mesh as a convex hull, generated from mesh colliders. This allows for irregular shapes to be used as dynamic bodies.
    
    **Limitations:**
    
    * They are non-deforming. A hull may be generated from a deforming render mesh, but it will not deform with it. Because of this, intrinsically deforming mesh types will not work as convex hulls. For instance world and face meshes. In these cases the convex hull will exist, but be empty and wont simulate.
    
    * It produces an approximation of the source mesh, reducing triangle count and eliminating concave areas.
    * Concave shapes may be composed of convex hulls in the scene graph. Splitting a concave mesh into convex hulls is known as "convex decomposition". It is an expensive process not supported by Studio, but there are many standalone tools available for this purpose (including plugins for 3D modeling apps). The resulting split mesh can be imported into Studio as a prefab.
    
    * Original triangle data is lost, so it is not available to script in ray casts.
    */
    convex: boolean
    
}

/**
* @description Mesh topology types used by [MeshBuilder](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#MeshBuilder).
*/
declare enum MeshTopology {
    /**
    * @description Draws unconnected line segments. Each group of two vertices specifies a new line segment.
    */
    Lines,
    /**
    * @description Draws connected line segments. Starting with the second vertex, a line is drawn between each vertex and the preceding one.
    */
    LineStrip,
    /**
    * @description Draws individual points. Each vertex specifies a new point to draw.
    */
    Points,
    /**
    * @description Draws unconnected triangles. Each group of three vertices specifies a new triangle.
    */
    Triangles,
    /**
    * @description Draws connected triangles sharing one central vertex. The first vertex is the shared one, or "hub" vertex. Starting with the third vertex, each vertex forms a triangle connecting with the previous vertex and hub vertex.
    */
    TriangleFan,
    /**
    * @description Draws connected triangles in a strip. After the first two vertices, each vertex defines the third point on a new triangle extending from the previous one.
    */
    TriangleStrip
}

/**
* @description This class has been DEPRECATED starting in Lens Studio 2.3.
* The `Component.MeshVisual` typename is now an alias for [BaseMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#BaseMeshVisual).
* When upgrading a project to Lens Studio 2.3 or higher, any instances of the MeshVisual component will be upgraded to [RenderMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#RenderMeshVisual).

* This class was split into the following three classes, to better distinguish the behaviors of child classes.

* [BaseMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#BaseMeshVisual): Base class for all visual classes using meshes to render

* [MaterialMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#MaterialMeshVisual): Child class of BaseMeshVisual, gives access to the [Materials](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#Material) used to render

* [RenderMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#RenderMeshVisual): Child class of MaterialMeshVisual, gives access to the [RenderMesh](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#RenderMesh) used to render
*/
interface MeshVisual extends Component {
}

/**
* @description Mel-frequency cepstral coefficients.
*/
interface MFCC extends ScriptObject {
    /**
    * @description Process in tensor with `shape`, write result to the `outTensor`, and returns the shape of `outTensor`.
    */
    process(inTensor: Float32Array, inShape: vec3, outTensor: Float32Array): vec3
    
    /**
    * @readonly
    
    * @description Max tensor size.
    */
    maxTensorSize: number
    
}

/**
* @description Builder class for the MFCC (Mel Frequency Cepstral Co-efficients).
*/
interface MFCCBuilder extends ScriptObject {
    /**
    * @description Create a new MFCC object.
    */
    build(): MFCC
    
    /**
    * @description Length of the window, the window will be the length of frameSize and then padded with zeros to mach FFTSize.
    */
    setFrameSize(frameSize: number): MFCCBuilder
    
    /**
    * @description Sets the number of samples between successive FFT segments.
    */
    setHopSize(hopSize: number): MFCCBuilder
    
    /**
    * @description Sets the length of the FFT window.
    */
    setFFTSize(fftSize: number): MFCCBuilder
    
    /**
    * @description Sets the number of MFCCs to return.
    */
    setNumMFCC(numMFCC: number): MFCCBuilder
    
    /**
    * @description Sets number of mel bins.
    */
    setNumMel(numMel: number): MFCCBuilder
    
    /**
    * @description If lifter > 0, apply liftering to the MFCCs.
    */
    setLifter(lifter: number): MFCCBuilder
    
    /**
    * @description Sets the min frequency.
    */
    setMinFreq(minFreq: number): MFCCBuilder
    
    /**
    * @description Sets the max frequency.
    */
    setMaxFreq(maxFreq: number): MFCCBuilder
    
    /**
    * @description Sets the number of samples per second.
    */
    setSampleRate(sampleRate: number): MFCCBuilder
    
}

/**
* @description The Audio Track Provider of the audio from microphone.
*/
interface MicrophoneAudioProvider extends AudioTrackProvider {
    /**
    * @exposesUserData
    
    * @description Writes current frame audio data to the passed in `Float32Array` and returns its shape. The length of the array can't be more than `maxFrameSize`.
    */
    getAudioFrame(audioFrame: Float32Array): vec3
    
    /**
    * @description Start processing audio from microphone. Useful to avoid redundant processing.
    */
    start(): void
    
    /**
    * @description Stop processing audio from microphone.
    */
    stop(): void
    
}

/**
* @description Binary ML model supplied by the user.
*/
interface MLAsset extends BinAsset {
    /**
    * @description Returns model metadata as JSON object. 
    
    */
    getMetadata(): object
    
}

/**
* @description Transforms inputs (Textures or Float32Array) into outputs (Textures or Float32Array) using a neural network.
* The neural network is represented by an MLAsset, which is set as the `model` property.
* For more information, see the [MLComponent Overview](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/ml-component-overview).
*/
interface MLComponent extends Component {
    /**
    * @description Returns the InputPlaceholder with the matching name.
    */
    getInput(name: string): InputPlaceholder
    
    /**
    * @description Returns the OutputPlaceholder with the matching name.
    */
    getOutput(name: string): OutputPlaceholder
    
    /**
    * @description Returns an array of InputPlaceholders of the MLComponent's model with default settings. Returns empty array if the model asset is not set.
    
    */
    getInputs(): InputPlaceholder[]
    
    /**
    * @description Returns an array of OutputPlaceholders of MLComponent's model with default settings. Returns empty array if the model is not set.
    
    */
    getOutputs(): OutputPlaceholder[]
    
    /**
    * @description Runs the MLComponent once.
    */
    runImmediate(sync: boolean): void
    
    /**
    * @description Schedules the MLComponent to run at the start timing and terminate at the end timing. The scheduled running will recur if `recurring` is true.
    */
    runScheduled(recurring: boolean, startTiming: MachineLearning.FrameTiming, endTiming: MachineLearning.FrameTiming): void
    
    /**
    * @description Returns the start time of the scheduled MLComponent run.
    */
    getScheduledStart(): MachineLearning.FrameTiming
    
    /**
    * @description Returns the end time of the scheduled MLComponent run.
    */
    getScheduledEnd(): MachineLearning.FrameTiming
    
    /**
    * @description If running asynchronously, makes the entire system wait until the last run is finished.
    */
    waitOnRunning(): void
    
    /**
    * @description If loading asynchronously, makes the entire system wait until loading is finished.
    */
    waitOnLoading(): void
    
    /**
    * @description Stops running the MLComponent. The `onRunningFinished` callback will not be executed.
    */
    cancel(): void
    
    /**
    * @description Stops running the MLComponent.
    */
    stop(): void
    
    /**
    * @description Returns true if running is requested on each frame.
    */
    isRecurring(): boolean
    
    /**
    * @description Builds the MLComponent model when all placeholders are determined. Config is an array of Input and Output placeholders.
    */
    build(placeholders: BasePlaceholder[]): void
    
    /**
    * @description Binary ML model supplied by the user.
    */
    model: MLAsset
    
    /**
    * @readonly
    
    * @description Returns the current status of the neural network model.
    */
    state: MachineLearning.ModelState
    
    /**
    * @description Controls the inference mode that MLComponent will run in. For example, GPU, CPU, etc.
    */
    inferenceMode: MachineLearning.InferenceMode
    
    /**
    * @description Function that gets called when model loading is finished.
    */
    onLoadingFinished: () => void
    
    /**
    * @description Function that gets called when the model stops running.
    */
    onRunningFinished: () => void
    
    /**
    * @description Render order of the MLComponent.
    */
    renderOrder: number
    
    /**
    * @description Runs the MLComponent automatically when the object or component it's on is enabled.
    */
    autoRun: boolean
    
}

/**
* @description Triggered when the tracked face's mouth closes.
*/
interface MouthClosedEvent extends FaceTrackingEvent {
}

/**
* @description Triggered when the tracked face's mouth opens.
*/
interface MouthOpenedEvent extends FaceTrackingEvent {
}

/**
* @description An instance of a Connected Lens session among a group of participants who were successfully invited into the experience. 
*/
interface MultiplayerSession extends ScriptObject {
    getLocalUserId(localUserIdCallback: (userId: string) => void): void
    
    /**
    * @description Gets information about the current user, which includes their display name and unique `userID` which can be used to identify them in session.
    
    */
    getLocalUserInfo(localUserInfoCallback: (userInfo: ConnectedLensModule.UserInfo) => void): void
    
    /**
    * @description Returns a unix timestamp in milliseconds of the current time according to the server. Useful for synchronizing time-based game events across devices.
    * `-1` will be returned if session is not connected to the server.
    */
    getServerTimestamp(): number
    
    /**
    * @description Sends a string message via the realtime backend.
    */
    sendMessage(message: string): void
    
    /**
    * @description Sends a string message with a timeout in milliseconds. Allows messages to be dropped from the outgoing message queue when they become too old. Useful to ensure the quick delivery of subsequent messages.
    */
    sendMessageWithTimeout(message: string, timeoutMs: number): void
    
    /**
    * @description Creates a realtime store to share data within a multiplayer session. The realtime store will allow you to pass data between multiple connected users in the same experience, such as: syncing the position of objects in the scene, syncing the current game score, etc.
    
    */
    createRealtimeStore(options: RealtimeStoreCreateOptions, onSuccess: (store: GeneralDataStore) => void, onError: (message: string) => void): void
    
    /**
    * @description Deletes a realtime store. 
    */
    deleteRealtimeStore(store: GeneralDataStore, onSuccess: (store: GeneralDataStore) => void, onError: (message: string) => void): void
    
    /**
    * @description Requests for ownership of the realtime store.
    */
    requestRealtimeStoreOwnership(store: GeneralDataStore, onSuccess: (store: GeneralDataStore) => void, onError: (message: string) => void): void
    
    /**
    * @description Sets the realtime store to unowned.
    */
    clearRealtimeStoreOwnership(store: GeneralDataStore, onSuccess: (store: GeneralDataStore) => void, onError: (message: string) => void): void
    
    /**
    * @description Returns information about the passed in RealtimeStore.
    */
    getRealtimeStoreInfo(store: GeneralDataStore): ConnectedLensModule.RealtimeStoreCreationInfo
    
    /**
    * @readonly
    
    * @description Get the number of active users in the Session.
    
    */
    activeUserCount: number
    
    /**
    * @readonly
    
    * @description Get all the Realtime Stores in the current session.
    */
    allRealtimeStores: GeneralDataStore[]
    
    /**
    * @readonly
    */
    activeUsersInfo: ConnectedLensModule.UserInfo[]
    
}

/**
* @description Tracking type used by the [Device Tracking](https://lensstudio.snapchat.com/api/classes/devicetracking) component to specify what type of plane to detect. 
*/
declare enum NativePlaneTrackingType {
    /**
    * @description No planes will be detected.
    */
    None,
    /**
    * @description Only horizontal planes will be detected.
    */
    Horizontal,
    /**
    * @description Only vertical planes will be detected.
    */
    Vertical,
    /**
    * @description Both horizontal and vertical planes will be detected. 
    */
    Both
}

interface NoiseReduction extends ScriptObject {
    process(inTensor: Float32Array, inShape: vec3, outTensor: Float32Array): vec3
    
    /**
    * @readonly
    */
    maxTensorSize: number
    
    amount: number
    
}

interface NoiseReductionBuilder extends ScriptObject {
    build(): NoiseReduction
    
    setSampleRate(sampleRate: number): NoiseReductionBuilder
    
}

/**
* @description Base class for configuring object tracking in the [ObjectTracking3D](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ObjectTracking3D) component.
*/
interface Object3DAsset extends Asset {
}

/**
* @description A reusable object hierarchy stored as a resource.
* Can be instantiated through script or brought into the scene through Lens Studio.
* For more information, see the [Prefabs](https://lensstudio.snapchat.com/lens-studio/references/guides/general/prefabs) guide.
*/
interface ObjectPrefab extends Asset {
    /**
    * @description Creates and returns a new instance of this object hierarchy underneath the specified parent object.
    * If parent is null, the object will be created with no parent.
    */
    instantiate(parent: SceneObject): SceneObject
    
}

/**
* @description Provides additional data for the tracked object. For example, with hand tracking, you can figure out whether the tracked hand is the left hand by accessing the `isLeft` property [true/false], as well as the probability of this data through the `isLeftProbability` property [0-1]. 
*/
interface ObjectSpecificData extends ScriptObject {
}

/**
* @description Used to track objects in the camera. Moves the local [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) to match the detected image.

* See the [Object Tracking guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/world/object-tracking) and the [Hand Gestures Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/body/hand-gestures) for more information.
*/
interface ObjectTracking extends Component {
    /**
    * @description Returns true if the object is currently being tracked on camera.
    */
    isTracking(): boolean
    
    /**
    * @description Registers a callback to be executed when the passed in descriptor starts for this tracked object. For example, the possible descriptors for hand tracking are: "victory", "open", "index_finger", "horns", "close", "thumb".
    */
    registerDescriptorStart(descriptor: string, callback: (descriptor: string) => void): void
    
    /**
    * @description Registers a callback to be executed when the passed in descriptor ends for this tracked object. For example, the possible descriptors for hand tracking are: "victory", "open", "index_finger", "horns", "close", "thumb". 
    */
    registerDescriptorEnd(descriptor: string, callback: (descriptor: string) => void): void
    
    /**
    * @description The index of the object being tracked.
    */
    objectIndex: number
    
    /**
    * @description If true, child objects of this Component's [SceneObject](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject) will be disabled when the object is not being tracked.
    */
    autoEnableWhenTracking: boolean
    
    /**
    * @description Function that gets called when the tracked object is found.
    */
    onObjectFound: () => void
    
    /**
    * @description Function that gets called when the tracked object is lost.
    */
    onObjectLost: () => void
    
    /**
    * @readonly
    
    * @description Gets additional data of the current object being tracked.
    */
    objectSpecificData: ObjectSpecificData
    
}

/**
* @description Component used for tracking objects in 3D space.
*/
interface ObjectTracking3D extends Component {
    /**
    * @description Returns whether the specified attachment point is being tracked.
    */
    isAttachmentPointTracking(name: string): boolean
    
    /**
    * @description Returns whether this object is currently being tracked.
    */
    isTracking(): boolean
    
    /**
    * @description Creates a SceneObject which is attached to the specified attachment point.
    */
    createAttachmentPoint(name: string): SceneObject
    
    /**
    * @description Attaches the SceneObject to the specified attachment point.
    */
    addAttachmentPoint(name: string, object: SceneObject): void
    
    removeAttachmentPoint(object: SceneObject): void
    
    /**
    * @description Returns all SceneObjects currently attached to the specified point.
    */
    getAttachedObjects(name: string): SceneObject[]
    
    /**
    * @description Asset containing tracking parameters, such as the tracking model and specific options.
    */
    trackingAsset: Object3DAsset
    
    /**
    * @description Function called when tracking begins.
    */
    onTrackingStarted: () => void
    
    /**
    * @description Function called when tracking is lost.
    */
    onTrackingLost: () => void
    
    /**
    * @description Index of the object to track, starting at 0. Useful when tracking multiple instances of the same type of object.
    */
    objectIndex: number
    
    /**
    * @description Strategy for updating attached SceneObjects.
    */
    trackingMode: ObjectTracking3D.TrackingMode
    
    /**
    * @description When true, the attached root SceneObject's world position will be updated to match the tracked object's world position.
    */
    trackPosition: boolean
    
    /**
    * @description Whether world scale is applied or not.
    */
    attachmentModeInheritScale: boolean
    
    /**
    * @description Whether world rotation is applied or not.
    */
    attachmentModeInheritRotation: boolean
    
}

declare namespace ObjectTracking3D {
    /**
    * @description Strategies for updating attached objects. Used by the [ObjectTracking3D](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ObjectTracking3D) component.
    */
    enum TrackingMode {
        /**
        * @description Update the local transform's rotation and scale
        */
        ProportionsAndPose,
        /**
        * @description Update the local transform's rotation only
        */
        PoseOnly,
        /**
        * @description Applies world position and world rotation by default, and optionally world scale. Additionally, the world rotation can be opt-out via the `attachmentModeInheritRotation` property
        */
        Attachment
    }

}

interface ObjectTrackingMaskedTextureProvider extends CropTextureProvider {
    objectIndex: number
    
}

interface ObjectTrackingNormalsTextureProvider extends ObjectTrackingMaskedTextureProvider {
}

/**
* @description Controls a segmentation texture and its placement using information provided by Object tracking.
*/
interface ObjectTrackingTextureProvider extends ObjectTrackingMaskedTextureProvider {
}

/**
* @description Triggered when the Lens starts, earlier than all OnStart events. Also fires immediately on a newly instantiated or copied object.
*/
interface OnAwakeEvent extends SceneEvent {
}

/**
* @description Triggered when the associated ScriptComponent is destroyed while the lens is running.
*/
interface OnDestroyEvent extends SceneEvent {
}

/**
* @description Triggered when the `ScriptComponent` this event is bound to is disabled.

*/
interface OnDisableEvent extends SceneEvent {
}

/**
* @description Triggered when the `ScriptComponent` this event is bound to is enabled.

*/
interface OnEnableEvent extends SceneEvent {
}

/**
* @description Triggered when the Lens starts, after all OnAwakeEvents have triggered. Also triggers later on newly instantiated or copied objects.
*/
interface OnStartEvent extends SceneEvent {
}

/**
* @description Types of operating system that may be running on the device.
*/
declare enum OS {
    /**
    * @description Android device
    */
    Android,
    /**
    * @description iOS device
    */
    iOS,
    /**
    * @description MacOS device
    */
    MacOS,
    /**
    * @description Windows device
    */
    Windows
}

/**
* @description Used in [Text's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) `outlineSettings` property.
* Configures how text outlining will appear on a Text component.
*/
interface OutlineSettings extends ScriptObject {
    /**
    * @description Whether outline is enabled on the Text.
    */
    enabled: boolean
    
    /**
    * @description Settings for how the outline is drawn.
    */
    fill: TextFill
    
    /**
    * @description The strength of the outline effect, ranging from 0.0 (no outline) to 1.0 (very strong outline).
    */
    size: number
    
}

/**
* @description Builds OutputPlaceholders for MLComponent.
*/
interface OutputBuilder extends ScriptObject {
    /**
    * @description Sets the name of the OutputPlaceholder to be built.
    */
    setName(name: string): OutputBuilder
    
    /**
    * @description Sets the OutputMode of the OutputPlaceholder to be built.
    */
    setOutputMode(outputMode: MachineLearning.OutputMode): OutputBuilder
    
    /**
    * @description Sets the Transformer of the OutputPlaceholder to be built.
    */
    setTransformer(transformer: Transformer): OutputBuilder
    
    /**
    * @description Builds and returns a new OutputPlaceholder.
    */
    build(): OutputPlaceholder
    
}

/**
* @description Provides output data from the neural network used by an MLComponent.
* For more information, see the [MLComponent Scripting](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/scripting-ml-component) guide.
*/
interface OutputPlaceholder extends BasePlaceholder {
    /**
    * @readonly
    
    * @description Output as a Texture. Usable when `mode` is set to `MachineLearning.OutputMode.Texture`.
    */
    texture: Texture
    
    /**
    * @readonly
    
    * @exposesUserData
    
    * @description Output as a Float32Array. Usable when `mode` is set to `MachineLearning.OutputMode.Data`.
    */
    data: Float32Array
    
    /**
    * @description Which type of data the output is provided as. For example, Texture or Data.
    */
    mode: MachineLearning.OutputMode
    
}

/**
* @description Exposes state generated for ColliderComponent overlap events.  See also: [ColliderComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), [OverlapEnterEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#OverlapEnterEventArgs), [OverlapExitEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#OverlapExitEventArgs), [OverlapStayEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#OverlapStayEventArgs).
*/
interface Overlap extends ScriptObject {
    /**
    * @readonly
    
    * @description The collider which is overlapping.
    */
    collider: ColliderComponent
    
    /**
    * @readonly
    
    * @description ID of the overlap, unique for this overlap.
    */
    id: number
    
}

/**
* @description Args used for [ColliderComponent.onOverlapEnter](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), which is triggered when the collider begins overlapping another object. Typically used for volume triggers.
*/
interface OverlapEnterEventArgs extends ScriptObject {
    /**
    * @readonly
    
    * @description Structure containing information about the current overlap.
    */
    overlap: Overlap
    
    /**
    * @readonly
    
    * @description Number of objects currently overlapping.
    */
    currentOverlapCount: number
    
    /**
    * @readonly
    
    * @description Array of all current overlaps.
    */
    currentOverlaps: Overlap[]
    
}

/**
* @description Args used for [ColliderComponent.onOverlapEnd](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), which is triggered when the collider stops overlapping another object. Typically used for volume triggers.
*/
interface OverlapExitEventArgs extends ScriptObject {
    /**
    * @readonly
    
    * @description Structure containing information about the current overlap.
    */
    overlap: Overlap
    
    /**
    * @readonly
    
    * @description Number of objects currently overlapping.
    */
    currentOverlapCount: number
    
    /**
    * @readonly
    
    * @description Array of all current overlaps.
    */
    currentOverlaps: Overlap[]
    
}

/**
* @description Args used for [ColliderComponent.onOverlapStay](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), which is triggered every frame while the collider continues overlapping another object. Typically used for volume triggers.
*/
interface OverlapStayEventArgs extends ScriptObject {
    /**
    * @readonly
    
    * @description Structure containing information about the current overlap.
    */
    overlap: Overlap
    
    /**
    * @readonly
    
    * @description Number of objects currently overlapping.
    */
    currentOverlapCount: number
    
    /**
    * @readonly
    
    * @description Array of all current overlaps.
    */
    currentOverlaps: Overlap[]
    
}

/**
* @description Controls how a mesh will get rendered. Each Pass acts as an interface for the shader it's associated with.
* Any properties on a Pass's shader will automatically become properties on that Pass.
* For example, if the shader defines a variable named `baseColor`, a script would be able to access that property as `material.mainPass.baseColor`.
*/
interface Pass extends ScriptObject {
    [index:string]: any
    
    baseTex: Texture
    
    baseColor: vec4
    
    /**
    * @description The name of the Pass.
    */
    name: string
    
    /**
    * @description Whether the material renders on both sides of a mesh face.
    */
    twoSided: boolean
    
    /**
    * @description Enables depth-sorting.
    */
    depthTest: boolean
    
    /**
    * @description Enables writing pixels to the depth buffer.
    */
    depthWrite: boolean
    
    /**
    * @description Controls the masking of color channels with a vec4b representing each channel with a boolean.
    */
    colorMask: vec4b
    
    /**
    * @description Line width used for rendering.
    */
    lineWidth: number
    
    /**
    * @description Extend render object's aabb to (1 + value). Only applys when user select FrustumCullMode.Extend
    */
    frustumCullPad: number
    
    /**
    * @description Set min corner of aabb. It only applys when user selects FrustumCullMode.UserDefinedAABB
    */
    frustumCullMin: vec3
    
    /**
    * @description Set max corner of aabb. It only applys when user selects FrustumCullMode.UserDefinedAABB
    */
    frustumCullMax: vec3
    
    /**
    * @description The cull mode used for rendering.
    */
    cullMode: CullMode
    
    /**
    * @description Mode for setting frustum culling on Pass
    */
    frustumCullMode: FrustumCullMode
    
    /**
    * @description Changes the position that each polygon gets drawn.
    */
    polygonOffset: vec2
    
    /**
    * @description Number of times the pass will be rendered. Useful with the Instance ID node in Material Editor.
    */
    instanceCount: number
    
    /**
    * @description The blend mode used for rendering. Possible values:
    
    * | BlendMode                     | Value | Expression                    |
    * | ----------------------------- | ----- | ----------------------------- |
    * | Normal                        | 0     | SrcAlpha, OneMinusSrcAlpha    |
    * | MultiplyLegacy [DEPRECATED]	| 1	    | DstColor, OneMinusSrcAlpha    |
    * | AddLegacy [DEPRECATED]        | 2     | One, One                      |
    * | Screen	                    | 3	    | One, OneMinusSrcColor         |
    * | PremultipliedAlpha            | 4	    | One, OneMinusSrcAlpha         |
    * | AlphaToCoverage               | 5	    | Blend Disabled                |
    * | Disabled                      | 6	    | Blend Disabled                |
    * | Add                           | 7	    | SrcAlpha, One                 |
    * | AlphaTest                     | 8	    | Blend Disabled                |
    * | ColoredGlass                  | 9	    | Blend Disabled                |
    * | Multiply                      | 10    | DstColor, Zero                |
    * | Min                           | 11    | One, One                      |
    * | Max                           | 12    | One, One                      |
    
    * ```
    * // Sets the blend mode of the main pass for a material to Screen
    * //@input Asset.Material material
    
    * script.material.mainPass.blendMode = 3;
    * ```
    */
    blendMode: BlendMode
    
    /**
    * @readonly
    
    * @description Returns texture samplers used by this pass to set filtering and wrap modes.
    */
    samplers: SamplerWrappers
    
    /**
    * @description The stencil test state for Pass.
    */
    stencilState: StencilState
    
}

/**
* @description Similar to [Pass](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Pass), except used by [VFX Assets](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#VFXAsset).
*/
interface PassWrapper extends ScriptObject {
    /**
    * @description The name of the pass wrapper.
    */
    name: string
    
    /**
    * @description Whether the material renders on both sides of a mesh face.
    */
    twoSided: boolean
    
    /**
    * @description Enables depth-sorting.
    */
    depthTest: boolean
    
    /**
    * @description Enables writing pixels to the depth buffer.
    */
    depthWrite: boolean
    
    /**
    * @description Controls the masking of color channels with a vec4b representing each channel with a boolean.
    */
    colorMask: vec4b
    
    /**
    * @description Line width used for rendering.
    */
    lineWidth: number
    
    /**
    * @description The cull mode used for rendering.
    */
    cullMode: CullMode
    
    /**
    * @description Changes the position that each polygon gets drawn.
    */
    polygonOffset: vec2
    
    /**
    * @description The blend mode used for rendering. 
    */
    blendMode: BlendMode
    
    /**
    * @description Number of times the pass will be rendered. Useful with the Instance ID node in Material Editor.
    */
    instanceCount: number
    
    /**
    * @readonly
    
    * @description A proxy class that provides the access to the properties of the passes under the hood of [VFX Asset](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#VFXAsset/). PassWrapper.samplers property which is tied to the Pass.samplers one. For example:  
    * ```js
    * pass.samplers.texture_name.Fitering = FilteringMode.Nearest
    * ```
    */
    samplers: SamplerWrappers
    
}

/**
* @description Allows for retrieval of a collection of Pass objects used by VFXAsset
*/
interface PassWrappers extends ScriptObject {
    /**
    * @readonly
    
    * @description Returns an array of Pass objects
    */
    allPasses: PassWrapper[]
    
}

/**
* @description Allows data to be stored and retrieved between Lens sessions.
* In other words, data can be saved on device and loaded back in the next time the Lens is opened.
* Can be accessed with `global.persistentStorageSystem`.

* See the [Persistent Storage guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/persistent-storage) for more information.
*/
interface PersistentStorageSystem extends ScriptObject {
    /**
    * @description The `GeneralDataStore` object used to store and retrieve data.
    */
    store: GeneralDataStore
    
}

/**
* @description Namespace containing physics classes and static physics methods.
*/
interface Physics {
}
declare namespace Physics {
    /**
    * @description Get settings that apply to the implicit root world.
    */
    export function getRootWorldSettings(): Physics.WorldSettingsAsset
    
    /**
    * @description Create an intersection probe that spans all worlds.
    */
    export function createGlobalProbe(): Probe
    
    /**
    * @description Create an intersection probe for the implicit root world.
    */
    export function createRootProbe(): Probe
    

}

declare namespace Physics {
    /**
    * @description Namespace containing static helper methods for [Constraints](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Constraint).
    */
    interface Constraint {
    }

}
declare namespace Physics {
    namespace Constraint {
        /**
        * @description Create a constraint of the given type, with default settings.
        */
        export function create(type: Physics.ConstraintType): Constraint
        
    
    }

}

declare namespace Physics {
    /**
    * @description Constraint type used by a [Constraint](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Contraint).   See also: [Physics.Constraint.create()](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#PhysicsConstraint), [ConstraintComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ConstraintComponent).
    */
    enum ConstraintType {
        /**
        * @description Constrains colliders to fixed rotation and position. See [FixedConstraint](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#FixedConstraint).
        */
        Fixed,
        /**
        * @description Constrains colliders to rotate around a point. See [PointConstraint](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#PointConstraint).
        */
        Point,
        /**
        * @description Constrains colliders to rotate around a single axis. See [HingeConstraint](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#HingeConstraint).
        */
        Hinge
    }

}

declare namespace Physics {
    /**
    * @description Script interface for applying collision filtering to colliders and ray/shape-casts.
    */
    interface Filter {
    }

}
declare namespace Physics {
    namespace Filter {
        /**
        * @description Create an instance with default settings.
        */
        export function create(): Filter
        
    
    }

}

declare namespace Physics {
    /**
    * @description Type of force to use when applying force or torque to a [BodyComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#BodyComponent).
    */
    enum ForceMode {
        /**
        * @description Continuous force (kg*cm/s^2), used for cases where force is applied over multiple frames.
        */
        Force,
        /**
        * @description Instantaneous force impulse (kg*cm/s).
        
        */
        Impulse,
        /**
        * @description Continuous acceleration (cm/s^2), applied without respect to mass, used for cases where force is applied over multiple frames.
        
        */
        Acceleration,
        /**
        * @description Instantaneous change in velocity (cm/s), applied without respect to mass.
        */
        VelocityChange
    }

}

declare namespace Physics {
    /**
    * @description Stores reusable settings uniform for a world (such as gravity magnitude and direction). See also: [WorldComponent.worldSettings](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#WorldComponent).
    */
    interface WorldSettingsAsset extends Asset {
        /**
        * @description Given 2 layer numbers A and B, returns true if colliders in A collide with colliders in B, and vice-versa. The layer numbers correspond to those used to form a `LayerSet` with `LayerSet.fromNumber()`. The relationship is symmetric, so if collision is disabled for (A, B), it is also disabled for (B, A). This accesses a flag in the "Layer Collision Matrix", as viewable in Studio. Note however that layer numbers are not the same as layer IDs. To get the number of a Studio-created layer, use `LayerSet.numbers`.
        */
        getLayersCollidable(layerNumberA: number, layerNumberB: number): boolean
        
        /**
        * @description Given 2 layer numbers A and B, enables or disables collision between colliders in those layers. The layer numbers correspond to those used to form a `LayerSet` with `LayerSet.fromNumber()`. The relationship is symmetric, so if collision is disabled for (A, B), it is also disabled for (B, A). This accesses a flag in the "Layer Collision Matrix", as viewable in Studio. Note however that layer numbers are not the same as layer IDs. To get the number of a Studio-created layer, use LayerSet.numbers.
        
        */
        setLayersCollidable(layerNumberA: number, layerNumberB: number, enable: boolean): void
        
        /**
        * @description Resets the layer collision matrix such that all layer pairs are enabled.
        */
        resetLayerCollisionMatrix(): void
        
        /**
        * @description Gravity acceleration vector (cm/s^2). Defaults to real-world gravity of 980 cm/s^2, downward.
        */
        gravity: vec3
        
        /**
        * @description Simulation rate, in steps per second. Larger values improve simulation accuracy at the cost of performance. This is limited to intervals of 30hz, in the range 30-240hz, with 60hz as the default.
        */
        simulationRate: number
        
        /**
        * @description Slow down simulation step frequency (higher values are slower). Limited to >=1.0. This achieves a slow-motion effect by reducing the number of simulation steps each frame. Useful for debugging as large values will cause noticeably discrete steps.
        */
        slowDownStep: number
        
        /**
        * @description Slow down simulation time (higher values are slower). This achieves a slow-motion effect by scaling simulation time. Unlike slowDownStep, it will maintain smooth motion, but has accuracy problems at large scales.
        */
        slowDownTime: number
        
        /**
        * @description Speed limit, in world space cm/s. Set to 0 to disable this.
        */
        absoluteSpeedLimit: number
        
        /**
        * @description Speed limit, relative to shape size. Set to 0 to disable this. The effective world space speed limit is scaled by simulation rate, so this is the maximum distance the object can move in a single step. The default of 0.5 only allows the object to move by half its size in a single step, which should prevent tunneling.
        */
        relativeSpeedLimit: number
        
        /**
        * @description Default Matter used for colliders in the world. This is used for a collider when its matter field is unset.
        */
        defaultMatter: Matter
        
        /**
        * @description Default Filter used for colliders in the world.
        */
        defaultFilter: Filter
        
    }

}
declare namespace Physics {
    namespace WorldSettingsAsset {
        /**
        * @description Create an instance with default settings.
        */
        export function create(): Physics.WorldSettingsAsset
        
    
    }

}

/**
* @description Attaches the SceneObject to the mesh surface of a different SceneObject.
* See the [Pin To Mesh](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/scene-set-up/3d/pin-to-mesh#adding-a-pin-to-mesh-component) guide for more information.
*/
interface PinToMeshComponent extends Component {
    /**
    * @description The UV coordinates on the target mesh to attach to.
    */
    pinUV: vec2
    
    /**
    * @description Index of the UV coordinate set to use for pinning.
    */
    preferredUVLayerIndex: number
    
    /**
    * @description The orientation type to use.
    */
    orientation: PinToMeshComponent.Orientation
    
    /**
    * @description If enabled, interpolated vertex normals will be used when calculating the attachment position.
    */
    useInterpolatedVertexNormal: boolean
    
    /**
    * @description The position offset to apply.
    */
    offsetPosition: vec3
    
    /**
    * @description The euler angle offset to apply. Only has an effect when `orientation` is set to `PositionAndDirection`.
    */
    offsetRotation: vec3
    
    /**
    * @description The target mesh to attach to.
    */
    target: BaseMeshVisual
    
    /**
    * @description The preferred triangle index to attach to when multiple triangles contain the desired UV coordinate.
    */
    preferredTriangle: number
    
}

declare namespace PinToMeshComponent {
    /**
    * @description Used with [PinToMeshComponent.orientation](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#PinToMeshComponent).
    */
    enum Orientation {
        /**
        * @description Pins only the position. Rotation is independent from the target mesh.
        */
        OnlyPosition,
        /**
        * @description Pins both the position and direction. The normal of the target mesh is the Y axis.
        * The U texture coordinate of the target mesh's UV is the X axis.
        */
        PositionAndDirection
    }

}

interface PitchShifter extends ScriptObject {
    process(inTensor: Float32Array, inShape: vec3, outTensor: Float32Array): vec3
    
    /**
    * @readonly
    */
    maxTensorSize: number
    
    pitch: number
    
}

interface PitchShifterBuilder extends ScriptObject {
    build(): PitchShifter
    
    setSampleRate(sampleRate: number): PitchShifterBuilder
    
}

declare enum PlaybackMode {
    Single,
    Loop,
    PingPong
}

/**
* @description Represents 3D locations of stationary features in the environment. The resulting cloud of points provides a sparse description of the 3D environment.

*/
interface PointCloud extends ScriptObject {
    /**
    * @readonly
    
    * @description A list of vec3 objects representing the 3D position of every point.
    */
    positions: vec3[]
    
    /**
    * @readonly
    
    * @description A list of numeric identifiers for each unique 3D point in the cloud.
    */
    ids: number[]
    
}

/**
* @description A type of constraint that only allows rotation.  See also: [ConstraintComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ConstraintComponent).
*/
interface PointConstraint extends Constraint {
}

/**
* @description An Audio Component effect that allows the Lens to simulate sound based on the direction of the Audio Listener relative to the Audio Component.
*/
interface PositionEffect extends ScriptObject {
    /**
    * @description If enabled, the position effect will be applied.
    */
    enabled: boolean
    
}

/**
* @description Uses an input color lookup table image to adjust the coloring of the Lens.
* See the [Color Correction Post Effect guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/graphics/materials/post-effects#color-correction) for more information.
*/
interface PostEffectVisual extends MaterialMeshVisual {
}

/**
* @description Performs collision tests (such as ray casts) in one or more world. See [Physics.createGlobalProbe()](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#Physics), [Physics.createRootProbe()](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#Physics), [WorldComponent.createProbe()](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#WorldComponent).
*/
interface Probe extends ScriptObject {
    /**
    * @description Returns (via callback) the nearest intersection in any world. If there is no hit, the callback is called with a null hit argument.  Ray casts are performed after simulation update, which occurs after script Update but prior to LateUpdate.
    */
    rayCast(start: vec3, end: vec3, hitCB: (hit: RayCastHit) => void): void
    
    /**
    * @description Returns (via callback) all intersections in every world. The callback receives an array of hits, in order of nearest to farthest. If there were no hits, the array length is 0.  Ray casts are performed after simulation update, which occurs after script Update but prior to LateUpdate.
    */
    rayCastAll(start: vec3, end: vec3, hitCB: (hit: RayCastHit[]) => void): void
    
    /**
    * @description Like rayCast(), but sweeps a sphere from start to end positions.  This is shorthand for calling shapeCast() with a SphereShape.  Returns (via callback) the nearest intersection.  If there is no hit, the callback is called with a null hit argument.  Ray casts are performed after simulation update, which occurs after script Update but prior to LateUpdate.
    */
    sphereCast(radius: number, start: vec3, end: vec3, hitCB: (hit: RayCastHit) => void): void
    
    /**
    * @description Like rayCastAll(), but sweeps a sphere from start to end positions. This is shorthand for calling shapeCastAll() with a SphereShape. Returns (via callback) all intersections. The callback receives an array of hits, in order of nearest to farthest. If there were no hits, the array length is 0.  Ray casts are performed after simulation update, which occurs after script Update but prior to LateUpdate.
    */
    sphereCastAll(radius: number, start: vec3, end: vec3, hitCB: (hit: RayCastHit[]) => void): void
    
    /**
    * @description Like rayCast(), but sweeps a sphere from start to end positions. This is shorthand for calling shapeCast() with a SphereShape. Returns (via callback) the nearest intersection. If there is no hit, the callback is called with a null hit argument. Ray casts are performed after simulation update, which occurs after script Update but prior to LateUpdate.
    */
    shapeCast(shape: Shape, start: vec3, startRot: quat, end: vec3, endRot: quat, hitCB: (hit: RayCastHit) => void): void
    
    /**
    * @description Like rayCastAll(), but sweeps a shape from start to end transforms (expressed as position and rotation).  The provided shape can be created with one of the Shape.create*() functions, or referenced from a Physics.ColliderComponent.  Does not support MeshShape, and will throw an exception if attempted.  Returns (via callback) all intersections.  The callback receives an array of hits, in order of nearest to farthest. If there were no hits, the array length is 0. Ray casts are performed after simulation update, which occurs after script Update but prior to LateUpdate.
    */
    shapeCastAll(shape: Shape, start: vec3, startRot: quat, end: vec3, endRot: quat, hitCB: (hit: RayCastHit[]) => void): void
    
    /**
    * @description Show intersection tests with debug-draw.
    */
    debugDrawEnabled: boolean
    
    /**
    * @description Filter settings applied to intersection tests.
    */
    filter: Filter
    
}

/**
* @description RenderObjectProvider for mesh objects generated procedurally.
*/
interface ProceduralMeshRenderObjectProvider extends MeshRenderObjectProvider {
}

/**
* @description Provides a texture that can be written to or read from. Can be accessed using Texture.control on a Procedural Texture.
*/
interface ProceduralTextureProvider extends TextureProvider {
    /**
    * @description Sets a region of pixels on the texture. The region starts at the pixel coordinates x, y, and extends rightward by width and upward by height. Uses the values of the passed in Uint8Array data, which should be integer values ranging from 0 to 255.
    */
    setPixels(x: number, y: number, width: number, height: number, data: Uint8Array): void
    
    /**
    * @exposesUserData
    
    * @description Returns a Uint8 array containing the pixel values in a region of the texture. The region starts at the pixel coordinates x, y, and extends rightward by width and upward by height. Values returned are integers ranging from 0 to 255.
    */
    getPixels(x: number, y: number, width: number, height: number, data: Uint8Array): void
    
}
declare namespace ProceduralTextureProvider {
    /**
    * @description Creates a new, blank Texture Provider using the passed in dimensions and Colorspace. The ProceduralTextureProvider can be accessed through the control property on the returned texture.
    */
    export function create(width: number, height: number, colorspace: Colorspace): Texture
    
    /**
    * @description Creates a new Procedural Texture based on the passed in texture. The ProceduralTextureProvider can be accessed through the control property on the returned texture.
    */
    export function createFromTexture(texture: Texture): Texture
    

}

interface Properties extends ScriptObject {
}

interface PropertyOnEventArgs extends ScriptObject {
}

/**
* @description Base class for all resource providers.
*/
interface Provider extends ScriptObject {
    getLoadStatus(): LoadStatus
    
}

/**
* @description A quaternion, used to represent rotation.
*/
declare class quat {
    /**
    * @description Creates a new quat.
    */
    constructor(w: number, x: number, y: number, z: number)
    
    /**
    * @description Returns an inverted version of the quat.
    */
    invert(): quat
    
    /**
    * @description Normalizes the quat.
    */
    normalize(): void
    
    /**
    * @description Returns a string representation of the quat.
    */
    toString(): string
    
    /**
    * @description Returns an euler angle representation of the quat, in radians.
    */
    toEulerAngles(): vec3
    
    /**
    * @description Returns the rotation angle of the quat.
    */
    getAngle(): number
    
    /**
    * @description Returns the rotation axis of the quat.
    */
    getAxis(): vec3
    
    /**
    * @description Returns the dot product of the two quats.
    */
    dot(quat: quat): number
    
    /**
    * @description Returns the product of this quat and `b`.
    */
    multiply(b: quat): quat
    
    /**
    * @description Returns the result of rotating direction vector `vec3` by this quat.
    */
    multiplyVec3(vec3: vec3): vec3
    
    /**
    * @description Returns whether this quat and `b` are equal.
    */
    equal(b: quat): boolean
    
    /**
    * @description x component of the quat.
    */
    x: number
    
    /**
    * @description y component of the quat.
    */
    y: number
    
    /**
    * @description z component of the quat.
    */
    z: number
    
    /**
    * @description w component of the quat.
    */
    w: number
    
}
declare namespace quat {
    /**
    * @description Returns the angle between `a` and `b`.
    */
    export function angleBetween(a: quat, b: quat): number
    
    /**
    * @description Returns a new quat with angle `angle` and axis `axis`.
    */
    export function angleAxis(angle: number, axis: vec3): quat
    
    /**
    * @description Returns a new quat using the euler angles `x`, `y`, `z` (in radians).
    */
    export function fromEulerAngles(x: number, y: number, z: number): quat
    
    /**
    * @description Returns a new quat using the euler angle `eulerVec` (in radians).
    */
    export function fromEulerVec(eulerVec: vec3): quat
    
    /**
    * @description Returns a rotation quat between direction vectors `from` and `to`.
    */
    export function rotationFromTo(from: vec3, to: vec3): quat
    
    /**
    * @description Returns a new quat with a forward vector `forward` and up vector `up`.
    */
    export function lookAt(forward: vec3, up: vec3): quat
    
    /**
    * @description Returns a new quat linearly interpolated between `a` and `b`.
    */
    export function lerp(a: quat, b: quat, t: number): quat
    
    /**
    * @description Returns a new quat spherically linearly interpolated between `a` and `b`.
    */
    export function slerp(a: quat, b: quat, t: number): quat
    
    /**
    * @description Returns the identity quaternion.
    */
    export function quatIdentity(): quat
    
    /**
    * @description Creates a quaternion from a matrix.
    */
    export function fromRotationMat(rotationMat: mat3): quat
    

}

/**
* @description The base class for animation tracks using quaternion values.
*/
interface QuaternionAnimationTrack extends AnimationTrack {
}

/**
* @description Represents an animation track using quaternion value keyframes.
*/
interface QuaternionAnimationTrackKeyFramed extends QuaternionAnimationTrack {
    /**
    * @description Removes all keys.
    */
    removeAllKeys(): void
    
    /**
    * @description Removes key at index `index`.
    */
    removeKeyAt(index: number): void
    
    /**
    * @description Adds a key with value `value` at time `time`.
    */
    addKey(time: number, value: quat): void
    
}

/**
* @description Represents a rotation animation track using euler angles.
*/
interface QuaternionAnimationTrackXYZEuler extends QuaternionAnimationTrack {
    /**
    * @description Returns child track at index `index`.
    */
    getChildTrackByIndex(index: number): AnimationTrack
    
    /**
    * @description Sets child track at index `index` to track `track`.
    */
    setChildTrackByIndex(index: number, track: AnimationTrack): void
    
}

/**
* @description Hit results of a ray-cast provided to script. See [Probe](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Probe).
*/
interface RayCastHit extends ScriptObject {
    /**
    * @description Set to `true` to skip remaining results, if any.
    */
    skipRemaining: boolean
    
    /**
    * @readonly
    
    * @description Distance from the ray origin to the point of intersection.
    */
    distance: number
    
    /**
    * @readonly
    
    * @description Ray interpolant at the point of intersection, in the range [0, 1].
    */
    t: number
    
    /**
    * @readonly
    
    * @description Position at the point of intersection.
    */
    position: vec3
    
    /**
    * @readonly
    
    * @description Surface normal on the collider at the point of intersection.
    */
    normal: vec3
    
    /**
    * @readonly
    
    * @description The collider that was hit.
    */
    collider: ColliderComponent
    
    /**
    * @readonly
    
    * @description The hit triangle, if it exists. This is set on intersection with mesh-based colliders, and null otherwise.
    */
    triangle: TriangleHit
    
}

/**
* @description The options for the realtime store.
*/
interface RealtimeStoreCreateOptions extends ScriptObject {
    /**
    * @description The ownership model for the realtime store.
    */
    ownership: RealtimeStoreCreateOptions.Ownership
    
    /**
    * @description The persistence model for the realtime store.
    */
    persistence: RealtimeStoreCreateOptions.Persistence
    
    /**
    * @description An existing store to be used as the initial values for the real time store.
    
    */
    initialStore: GeneralDataStore
    
    allowOwnershipTakeOver: boolean
    
    /**
    * @description Writes an id string to the store that can be used to identify it later.
    */
    storeId: string
    
}
declare namespace RealtimeStoreCreateOptions {
    /**
    * @description Creates the realtime store options object.
    
    */
    export function create(): RealtimeStoreCreateOptions
    

}

declare namespace RealtimeStoreCreateOptions {
    /**
    * @description The ownership model of a realtime store.
    */
    enum Ownership {
        /**
        * @description Indicates ownership of an entity. Only the owner can edit a store.
        */
        Owned,
        /**
        * @description Indicates ownership of an entity. Any user in the session can edit an unowned store.
        */
        Unowned
    }

}

declare namespace RealtimeStoreCreateOptions {
    /**
    * @description The persistence model for a realtime store.
    */
    enum Persistence {
        /**
        * @description Indicates that the entity will be deleted after it is broadcast.
        */
        Ephemeral,
        /**
        * @description Indicates that the entity will be deleted when the owner leaves.
        */
        Owner,
        /**
        * @description Indicates that the entity will be marked as unowned when the owner leaves. The Entity will be deleted when all clients leave.
        */
        Session,
        /**
        * @description Indicates that the entity will be marked as unowned when the owner leaves. The Entity will be created as unowned when any client rejoins.
        */
        Persist
    }

}

/**
* @description An axis aligned rectangle.
* Used by `anchors` and `offsets` in [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) to represent screen boundaries.
* Rect can only store finite numbers in the range Number.MIN_VALUE to Number.MAX_VALUE.

*/
interface Rect extends ScriptObject {
    /**
    * @description Returns a string representation of the Rect.
    */
    toString(): string
    
    /**
    * @description Returns the size of the rectangle as (width, height).
    */
    getSize(): vec2
    
    /**
    * @description Sets the rectangle's size while maintaining its center position.
    */
    setSize(value: vec2): void
    
    /**
    * @description Returns the rectangle's center position as (x, y).
    */
    getCenter(): vec2
    
    /**
    * @description Sets the rectangle's center position while maintaining its size.
    */
    setCenter(value: vec2): void
    
    /**
    * @description The x position of the rectangle's left side.
    */
    left: number
    
    /**
    * @description The x position of the rectangle's right side.
    */
    right: number
    
    /**
    * @description The y position of the rectangle's top side.
    */
    top: number
    
    /**
    * @description The y position of the rectangle's bottom side.
    */
    bottom: number
    
}
declare namespace Rect {
    /**
    * @description Creates a new Rect with the given properties.
    */
    export function create(left: number, right: number, bottom: number, top: number): Rect
    

}

/**
* @description Applies ScreenTransform positioning to match the cropped region of a texture.
* For more information, see the [Crop Textures](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/crop-textures) guide.
*/
interface RectangleSetter extends Component {
    /**
    * @description Cropped texture to match the screen region of. Should be a texture using a RectCropTextureProvider, such as a Screen Crop Texture or Face Crop Texture.
    */
    cropTexture: Texture
    
}

/**
* @description Texture Provider providing a cropped region of the input texture. The region is specified by the cropRect in local space and rotation.
* Can be accessed using Texture.control on a RectCropTexture asset, such as a Screen Crop Texture.
* For more information, see the [Crop Textures](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/crop-textures) guide.
*/
interface RectCropTextureProvider extends CropTextureProvider {
    /**
    * @description The cropped region to draw.
    */
    cropRect: Rect
    
    /**
    * @description Angle, in radians, the cropped region is rotated by.
    */
    rotation: number
    
}

interface RemoteApiRequest extends ScriptObject {
    endpoint: string
    
    body: (number[]|string)
    
    parameters: object
    
}
declare namespace RemoteApiRequest {
    export function create(): RemoteApiRequest
    

}

interface RemoteApiResponse extends ScriptObject {
    /**
    * @description Converts the response into a DynamicResource object, which can be used by RemoteMediaModule to load the media content in the response body.
    */
    asResource(): DynamicResource
    
    /**
    * @readonly
    
    * @description The integer status code of the response.
    
    * The meaning of possible status code values are defined as follows:
    * - 1: Success. This code corresponds to the 2XX HTTP response status codes.
    * - 2: Redirected. This code corresponds to the 3XX HTTP response status codes.
    * - 3: Bad request. This code corresponds to the 4XX HTTP response status codes other than 401, 403, 404, 408, 413, 414, and 431 which are mapped separately below.
    * - 4: Access denied. This code corresponds to the HTTP response status codes 401 and 403.
    * - 5: Not found. This code corresponds to the HTTP response status code 404. It is also returned when the API spec is not found by the remote API service.
    * - 6: Timeout. This code corresponds to the HTTP response status codes 408 and 504.
    * - 7: Request too large. This code corresponds to the HTTP response status codes 413, 414, and 431.
    * - 8: Server error. This code corresponds to the 5XX HTTP response status codes other than 504 (timeout).
    * - 9: Request cancelled by the caller.
    * - 10: Internal error happened inside the remote API framework (i.e., not from the remote service being called).
    
    * All other values have undefined meaning and should be treated as internal error (code 10).
    
    */
    statusCode: number
    
    /**
    * @readonly
    */
    body: string
    
    /**
    * @readonly
    */
    metadata: object
    
}

/**
* @description Provides access to a remote media.
*/
interface RemoteMediaModule extends Asset {
    /**
    * @description Downloads the remote image resource from the [`DynamicResource`](https://lensstudio.snapchat.com/api/classes/dynamicresource) object and loads the resource as an [Image](https://lensstudio.snapchat.com/api/classes/Image) Texture.
    */
    loadResourceAsImageTexture(resource: DynamicResource, onSuccess: (texture: Texture) => void, onFailure: (errorMessage: string) => void): void
    
    /**
    * @description Downloads the remote image resource from the [`DynamicResource`](https://lensstudio.snapchat.com/api/classes/dynamicresource) object and loads the resource as a [VideoTextureProvider](https://lensstudio.snapchat.com/api/classes/VideoTextureProvider).
    */
    loadResourceAsVideoTexture(resource: DynamicResource, onSuccess: (texture: Texture) => void, onFailure: (errorMessage: string) => void): void
    
    /**
    * @description Downloads a remote 3D GLTF file given a [`DynamicResource`](https://lensstudio.snapchat.com/api/classes/dynamicresource).
    
    */
    loadResourceAsGltfAsset(resource: DynamicResource, onSuccess: (glTFAsset: GltfAsset) => void, onFailure: (errorMessage: string) => void): void
    
    /**
    * @description Downloads the remote image resource from the [`DynamicResource`](https://lensstudio.snapchat.com/api/classes/dynamicresource) object and loads the resource as [AudioTrackAsset](https://lensstudio.snapchat.com/api/classes/AudioTrackAsset).
    */
    loadResourceAsAudioTrackAsset(resource: DynamicResource, onSuccess: (glTFAsset: AudioTrackAsset) => void, onFailure: (errorMessage: string) => void): void
    
}

/**
* @description Provides a reference to a remote asset (i.e. assets outside of the Lens size limit) that can be downloaded at runtime using script.
*/
interface RemoteReferenceAsset extends Asset {
    /**
    * @description Request to download the Remote Reference Asset.
    */
    downloadAsset(onDownloaded: (asset: Asset) => void, onFailed: () => void): void
    
}

interface RemoteServiceModule extends Asset {
    performApiRequest(request: RemoteApiRequest, onApiResponse: (response: RemoteApiResponse) => void): void
    
}

/**
* @description Represents a mesh asset.
* See also: [RenderMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#RenderMeshVisual).
*/
interface RenderMesh extends Asset {
    /**
    * @description Returns a list of values of each vertices in the RenderMesh for the specified attribute.
    */
    extractVerticesForAttribute(attributeName: string): number[]
    
    /**
    * @description Returns a list of indices of each vertices in the RenderMesh.
    */
    extractIndices(): number[]
    
    /**
    * @readonly
    
    * @description The index data type used by this mesh.
    */
    indexType: MeshIndexType
    
    /**
    * @readonly
    
    * @description The topology type used by this mesh.
    */
    topology: MeshTopology
    
    /**
    * @readonly
    
    * @description Returns the minimum value in each dimension of the axis-aligned bounding box containing this mesh.
    */
    aabbMin: vec3
    
    /**
    * @readonly
    
    * @description Returns the maximum value in each dimension of the axis-aligned bounding box containing this mesh.
    */
    aabbMax: vec3
    
    /**
    * @description The RenderObjectProvider for this RenderMesh, which can provide more controls depending on the mesh type.
    * See also: [FaceRenderObjectProvider](https://lensstudio.snapchat.com/api/lens-studio/Classes/Providers#FaceRenderObjectProvider)
    */
    control: RenderObjectProvider
    
}

/**
* @description Renders a [RenderMesh](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#RenderMesh) asset in the scene.
* Comparable to the former class "MeshVisual", which was split into the classes:
* [BaseMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#BaseMeshVisual),
* [MaterialMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#MaterialMeshVisual),
* and [RenderMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#RenderMeshVisual).
*/
interface RenderMeshVisual extends MaterialMeshVisual {
    /**
    * @description Sets the weight of the blend shape with the matching name.
    */
    setBlendShapeWeight(name: string, weight: number): void
    
    /**
    * @description Returns the weight of the blend shape with the matching name.
    */
    getBlendShapeWeight(name: string): number
    
    /**
    * @description  Clears the blend shape with the matching name.
    */
    unsetBlendShapeWeight(name: string): void
    
    /**
    * @description Returns whether this component has a weight for the blend shape with matching name.
    */
    hasBlendShapeWeight(name: string): boolean
    
    /**
    * @description Resets all blend shape weights on the component.
    */
    clearBlendShapeWeights(): void
    
    /**
    * @description Returns the names of the blend shapes on this RenderMeshVisual.
    */
    getBlendShapeNames(): string[]
    
    /**
    * @description Sets the [Skin](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Skin) to use for rendering this mesh.
    */
    setSkin(value: Skin): void
    
    /**
    * @description The [RenderMesh](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#RenderMesh) asset to render.
    */
    mesh: RenderMesh
    
    /**
    * @description If enabled, normal directions are also blended by blend shapes.
    */
    blendNormals: boolean
    
    /**
    * @description If enabled, blend shapes will affect the component.
    */
    blendShapesEnabled: boolean
    
    /**
    * @description If enabled, rays are generated from this object during raytracing so that it can receive ray-traced reflections.
    */
    emitter: boolean
    
    /**
    * @description If enabled, rays can hit this object during raytracing so that it may be visible in ray-traced reflections.
    */
    receiver: boolean
    
}

/**
* @description Provider for RenderMesh data.
*/
interface RenderObjectProvider extends Provider {
}

/**
* @description Controls a camera texture resource.
* Can be accessed through [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Texture#control-textureprovider) on a Camera texture.
* For more information, see the [Camera and Layers](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/scene-set-up/camera) guide.
*/
interface RenderTargetProvider extends TextureProvider {
    /**
    * @description The texture type of the render target.
    */
    textureType: RenderTargetProvider.TextureType
    
    /**
    * @description When `useScreenResolution` is false, controls the horizontal and vertical resolution of the Render Target.
    */
    resolution: vec2
    
    /**
    * @description If true, the Render Target's resolution will match the device's screen resolution.
    */
    useScreenResolution: boolean
    
    /**
    * @description When `clearColorEnabled` is true, this texture is used to clear this RenderTarget the first time it is drawn to each frame.
    * If this texture is null, `clearColor` will be used instead.
    */
    inputTexture: Texture
    
    /**
    * @description Sets the clear color option.
    */
    clearColorOption: ClearColorOption
    
    /**
    * @description When `clearColorEnabled` is true and `inputTexture` is null, this color is used to clear this RenderTarget the first time it is drawn to each frame.
    */
    clearColor: vec4
    
    /**
    * @description When Use Screen Resolution is enabled, this scales the render target resolution relative to the device resolution.
    */
    resolutionScale: number
    
    /**
    * @description If true, the depth buffer will be cleared on this RenderTarget the first time it is drawn to each frame.
    */
    clearDepthEnabled: boolean
    
    /**
    * @description If true, the color on this RenderTarget will be cleared the first time it is drawn to each frame.
    * `inputTexture` will be used to clear it unless it is null, in which case `clearColor` is used instead.
    */
    clearColorEnabled: boolean
    
}

declare namespace RenderTargetProvider {
    /**
    * @description Specifies what kind of render target it is. Some texture types, for example TextureCubemap, need additional properties set on the rendering camera to work correctly.
    */
    enum TextureType {
        /**
        * @description A 2d texture. Render targets are this type by default.
        */
        Texture2D,
        /**
        * @description A cubemap render target.
        */
        TextureCubemap
    }

}

/**
* @description Visual effect used to add subtle retouching effects to detected faces (soft skin, teeth whitening, etc.).
* To learn more, visit the [Retouch Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/face/face-effects/face-retouch).
*/
interface RetouchVisual extends MaterialMeshVisual {
    isAuto(): boolean
    
    /**
    * @description The index of the face the effect is being applied to.
    */
    faceIndex: number
    
    /**
    * @description The strength of the soft-skin effect.
    */
    softSkinIntensity: number
    
    /**
    * @readonly
    */
    softSkinEnabled: boolean
    
    /**
    * @description The strength of the eye sharpening effect.
    */
    sharpenEyeIntensity: number
    
    /**
    * @readonly
    */
    eyeSharpeningEnabled: boolean
    
    /**
    * @description The strength of the teeth whitening effect.
    */
    teethWhiteningIntensity: number
    
    /**
    * @readonly
    */
    teethWhiteningEnabled: boolean
    
    /**
    * @description The strength of the eye whitening effect.
    */
    eyeWhiteningIntensity: number
    
    /**
    * @readonly
    */
    eyeWhiteningEnabled: boolean
    
}

interface RotatedRect extends ScriptObject {
    /**
    * @description Returns the description of RotatedRect.
    */
    toString(): string
    
    /**
    * @description The center point (mass center) of RotatedRect.
    */
    center: vec2
    
    /**
    * @description The width and height of RotatedRect.
    */
    size: vec2
    
    /**
    * @description The rotation angle in degrees.
    */
    angle: number
    
}
declare namespace RotatedRect {
    /**
    * @description Creates a RotatedRect object.
    */
    export function create(center: vec2, size: vec2, angle: number): RotatedRect
    

}

/**
* @description Used with [DeviceTracking.rotationOptions](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#DeviceTracking) to change settings for Rotation tracking mode.
*/
interface RotationOptions extends ScriptObject {
    /**
    * @description If enabled, rotation will be inverted.
    */
    invertRotation: boolean
    
}

interface Sampler extends ScriptObject {
}

/**
* @description Class for building Sampler.
*/
interface SamplerBuilder extends ScriptObject {
    /**
    * @description Sets the filtering mode used for textures sampling.
    */
    setFilteringMode(filteringMode: FilteringMode): SamplerBuilder
    
    /**
    * @description Sets the same wrap mode for all U, V and W axes.
    */
    setWrapMode(wrapMode: WrapMode): SamplerBuilder
    
    /**
    * @description Sets the wrap mode U axis.
    */
    setWrapUMode(wrapMode: WrapMode): SamplerBuilder
    
    /**
    * @description Sets the wrap mode V axis.
    */
    setWrapVMode(wrapMode: WrapMode): SamplerBuilder
    
    /**
    * @description Sets the wrap mode W axis.
    */
    setWrapWMode(wrapMode: WrapMode): SamplerBuilder
    
    /**
    * @description Sets border color that will be used for `WrapMode.ClampToBorder` case.
    */
    setBorderColor(borderColor: vec4): SamplerBuilder
    
    /**
    * @description Sets the flag to use/generate texture mipmaps if they exist/requested.
    */
    setUseMipmaps(value: boolean): SamplerBuilder
    
    /**
    * @description Builds Sampler.
    */
    build(): Sampler
    
}

/**
* @description An accessor for Pass.samplers when using PassWrappers
*/
interface SamplerWrapper extends ScriptObject {
    /**
    * @description Whether the texture should wrap.
    */
    wrap: WrapMode
    
    /**
    * @description Whether the texture should wrap in the x-axis.
    */
    wrapU: WrapMode
    
    /**
    * @description Whether the texture should wrap in the y-axis.
    */
    wrapV: WrapMode
    
    /**
    * @description Whether the texture should wrap in the z-axis.
    */
    wrapW: WrapMode
    
    /**
    * @description How the texture will be filtered by the sampler.
    */
    filtering: FilteringMode
    
    /**
    * @description The texture used by the sampler.
    */
    texture: Texture
    
}

interface SamplerWrappers extends ScriptObject {
}

/**
* @exposes user data

* @description Asset for detecting an object through the Scan system.
*/
interface ScanModule extends Asset {
    /**
    * @description Starts a single Scan call using the provided list of ScanModule.Contexts. On success it will invoke `scanComplete` providing a JSON string. On failure it will invoke `onFailure` with a failure message passed in as an argument.
    */
    scan(contexts: string[], scanComplete: (resultJson: string) => void, scanFailed: (failureMessage: string) => void): void
    
    /**
    * @description Optional property to pass in a texture for Scan to use.
    */
    scanTarget: Texture
    
}

declare namespace ScanModule {
    /**
    * @description Contexts used in `ScanModule.scan()`.
    */
    interface Contexts {
    }

}
declare namespace ScanModule {
    namespace Contexts {
        /**
        * @description Category containing objects.
        */
        let Objects: string
        
        /**
        * @description Category containing places.
        */
        let Places: string
        
        /**
        * @description Category containing cars.
        */
        let Cars: string
        
        /**
        * @description Category containing dogs.
        */
        let Dogs: string
        
    
    }

}

/**
* @description The base class for scenewide events.  SceneEvents can be created using [ScriptComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScriptComponent)'s [createEvent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#scriptcomponent--createEvent) method.
*/
interface SceneEvent extends IEventParameters {
    /**
    * @description Binds a callback function to this event.
    */
    bind(evCallback: (arg1: this) => void): void
    
    /**
    * @description Returns the typename of the SceneEvent.
    */
    getTypeName(): string
    
    /**
    * @description If true, the event is able to trigger. If false, the event will not trigger.
    */
    enabled: boolean
    
}

/**
* @description An object in the scene hierarchy, containing a [Transform](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Transform) and possibly [Components](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components).
* A script can access the SceneObject holding it through the method `script.getSceneObject()`.
*/
interface SceneObject extends SerializableWithUID {
    /**
    * @description Returns the Transform attached to the SceneObject.
    */
    getTransform(): Transform
    
    /**
    * @description Destroys the SceneObject.
    */
    destroy(): void
    
    /**
    * @description Returns the number of children the SceneObject has.
    */
    getChildrenCount(): number
    
    /**
    * @description Returns this SceneObject's child at index `index`.
    */
    getChild(index: number): SceneObject
    
    /**
    * @description Returns the first attached Component with type matching or deriving from `componentType`.
    */
    getComponent<K extends keyof ComponentNameMap>(componentType: K): ComponentNameMap[K]
    
    /**
    * @description Returns a list of attached components with types matching or deriving from `componentType`.
    */
    getComponents<K extends keyof ComponentNameMap>(componentType: K): ComponentNameMap[K][]
    
    /**
    * @description Copies `component` and adds it to the SceneObject, then returns it.
    */
    copyComponent<K extends Component>(component: K): K
    
    /**
    * @description Adds a new component of type `typeName` to the SceneObject.
    */
    createComponent<K extends keyof ComponentNameMap>(typeName: K): ComponentNameMap[K]
    
    /**
    * @description Sets the SceneObject's parent to `newParent` in the scene hierarchy.
    */
    setParent(newParent: SceneObject): void
    
    /**
    * @description Returns the SceneObject's parent in the hierarchy, or null if there isn't one.
    */
    getParent(): SceneObject
    
    /**
    * @description Returns whether the SceneObject has a parent in the scene hierarchy.
    */
    hasParent(): boolean
    
    /**
    * @description Unparents the SceneObject in the hierarchy, making it an orphan.
    */
    removeParent(): void
    
    /**
    * @description Changes the parent of the SceneObject without altering its world position, rotation, or scale.
    */
    setParentPreserveWorldTransform(newParent: SceneObject): void
    
    /**
    * @description Creates a deep copy of the passed in `sceneObject` (including its hierarchy), and parents it to this SceneObject.
    */
    copyWholeHierarchy(sceneObject: SceneObject): SceneObject
    
    /**
    * @description Creates a shallow copy of the passed in `sceneObject` (not including its hierarchy), and parents it to this SceneObject.
    */
    copySceneObject(sceneObject: SceneObject): SceneObject
    
    /**
    * @description Whether the SceneObject, including its components and children, is enabled or disabled.
    */
    enabled: boolean
    
    /**
    * @readonly
    
    * @description Returns true if this SceneObject and all of its parents are enabled.
    */
    isEnabledInHierarchy: boolean
    
    /**
    * @readonly
    
    * @description An event that will trigger when a SceneObject goes from disabled in the hiearchy to enabled in the hiearchy.
    */
    onEnabled: event0<void>
    
    /**
    * @readonly
    
    * @description An event that will trigger when a SceneObject goes from enabled in the hiearchy to disabled in the hiearchy.
    */
    onDisabled: event0<void>
    
    /**
    * @readonly
    
    * @description Get an array of the scene object's children.
    */
    children: SceneObject[]
    
    /**
    * @description The name of the SceneObject.
    */
    name: string
    
    /**
    * @description Gets or sets the LayerSet of layers this SceneObject belongs to.
    * This is used to determine which [Cameras](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Camera) will render the SceneObject.
    */
    layer: LayerSet
    
}

/**
* @description Base class for all object-based Event types in Lens Studio (ManipulateStartEvent, TapEvent, etc.).
*/
interface SceneObjectEvent extends SceneEvent {
    /**
    * @description Returns the SceneObject this Event is associated with.
    */
    getSceneObject(): SceneObject
    
}

/**
* @description Overrides the settings on a local [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) to fit a screen region on the device.
* See the [Screen Transform guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/scene-set-up/2d/screen-transform-overview) for more information.
*/
interface ScreenRegionComponent extends Component {
    /**
    * @description The region of the screen the local [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) will be fit to.
    */
    region: ScreenRegionType
    
}

/**
* @description Types of screen regions that can be used with [ScreenRegionComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenRegionComponent).
*/
declare enum ScreenRegionType {
    /**
    * @description The entire screen area of the device.
    */
    FullFrame,
    /**
    * @description The screen area shown to the user when recording. On some devices, this region is a subset of FullFrame region.
    */
    Capture,
    /**
    * @description The screen area shown to the user when previewing a Snap. On some devices, this region is a subset of Capture region.
    */
    Preview,
    /**
    * @description A screen area that will be visible on all devices and won't overlap Snapchat UI. Safe area to place any UI elements.
    */
    SafeRender,
    /**
    * @description The screen area where the round "Snap" button is drawn.
    */
    RoundButton
}

/**
* @description Texture providing the current Render Target being rendered.
*/
interface ScreenTextureProvider extends TextureProvider {
}

/**
* @description Used for positioning objects in 2d screen space. It overrides the regular [Transform](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Transform) component on the SceneObject it's attached to.

* See the [Screen Transform guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/scene-set-up/2d/screen-transform-overview) for more information.
*/
interface ScreenTransform extends Component {
    /**
    * @description Returns true if this ScreenTransform is in a valid screen hierarchy, which is required for anchoring to work.
    * To be in a valid screen hierarchy there must be a [Camera](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Camera) component upward in the parent hierarchy, and every
    * object between the Camera and this one must also have a ScreenTransform.
    */
    isInScreenHierarchy(): boolean
    
    /**
    * @description Converts from a screen position to a normalized (-1 to 1) position within this ScreenTransform's bounds.
    */
    screenPointToLocalPoint(screenPoint: vec2): vec2 | undefined
    
    /**
    * @description Converts from a screen position to a normalized (-1 to 1) position within the parent object's bounds.
    * This value is useful because it can be used directly for this ScreenTransform's anchor positioning.
    */
    screenPointToParentPoint(screenPoint: vec2): vec2 | undefined
    
    /**
    * @description Converts from a world position to a normalized (-1 to 1) position within this ScreenTransform's bounds.
    */
    worldPointToLocalPoint(worldPoint: vec3): vec2
    
    /**
    * @description Converts from a world position to a normalized (-1 to 1) position within the parent object's bounds.
    * This value is useful because it can be used directly for this ScreenTransform's anchor positioning.
    */
    worldPointToParentPoint(worldPoint: vec3): vec2
    
    /**
    * @description Converts from a normalized (-1 to 1) position within this ScreenTransform's bounds to a world position.
    */
    localPointToWorldPoint(relativeLocalPoint: vec2): vec3
    
    /**
    * @description Converts from a normalized (-1 to 1) position within this ScreenTransform's bounds to a screen position. This function will calculate the ScreenPoint by heuristically looking for a camera: first checking for a camera in it's parent's hierarchy, then looking for a camera with the same render layer, and finally just choosing the first camera in the scene.
    
    */
    localPointToScreenPoint(relativeLocalPoint: vec2): vec2
    
    /**
    * @description Returns true if the local point is within the boundaries of this ScreenTransform--that is: its position is within `-1` and `1` in both the x and y coordinates.
    
    */
    containsLocalPoint(localPoint: vec2): boolean
    
    /**
    * @description Returns true if the screen position is within the boundaries of this ScreenTransform.
    * Useful for checking if a touch event overlaps with this object. 
    * This function will calculate the ScreenPoint by heuristically looking for a camera: first checking for a camera in it's parent's hierarchy, then looking for a camera with the same render layer, and finally just choosing the first camera in the scene.
    
    */
    containsScreenPoint(screenPoint: vec2): boolean
    
    /**
    * @description Returns true if the world position is within the boundaries of this ScreenTransform.
    * The `z` value of the world position is ignored.
    */
    containsWorldPoint(worldPoint: vec3): boolean
    
    /**
    * @description The anchor rect positioning this ScreenTransform proportional to its parent's bounds.
    * For each field, a value of `0` equals the parent's center point, and value of `-1` or `1` (depending on the side) equals the parent's full boundary.
    
    * For example, a `top` value of `1.0` means this ScreenTransform's top edge will be exactly at the top edge of its parent.
    
    * A `bottom` value of `-0.5` means this ScreenTransform's bottom edge will be halfway between its parent's bottom edge and center.
    
    * A `right` value of `0` means this ScreenTransform's right edge will be exactly at its parent's center.
    
    * A `left` value of `-2` means this ScreenTransform's left edge will be twice as far from its parent's center as its parent's left edge is.
    */
    anchors: Rect
    
    /**
    * @description This rect is applied after `anchors` to determine the final boundaries of the ScreenTransform.
    * It adds an offset in world units (based on the parent [Camera's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Camera) `size` property) to each edge of the ScreenTransform's boundaries.
    
    * For example, a value of `0` for any side will have no effect on boundaries.
    
    *  A value of `1.0` for any side will offset that edge by `1.0` world unit.
    */
    offsets: Rect
    
    /**
    * @description Basic local scaling applied to the SceneObject.
    */
    scale: vec3
    
    /**
    * @description Basic local position in world units relative to the parent's center. Useful for animating screen elements with a simple offset.
    */
    position: vec3
    
    /**
    * @description Basic local rotation applied to the SceneObject.
    */
    rotation: quat
    
    /**
    * @description Normalized (x, y) position of the center point used in rotation. (-1, -1) being bottom left, (0, 0) being center, and (1, 1) being top right of the image.
    */
    pivot: vec2
    
}

/**
* @description Represents a JavaScript script which can be used to add logic in your Lens.
*/
interface ScriptAsset extends Asset {
}

/**
* @description Binds scripts to Events and executes them when triggered. Any script can access the ScriptComponent executing them through the variable `script`.
* See also: [Scripting Overview](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/scripting-overview#script-components), [Script Events Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/script-events#introduction).
*/
interface ScriptComponent extends Component {
    [index:string]: any
    
    /**
    * @description Adds a new SceneEvent, triggered by `eventType` events, to the ScriptComponent.
    */
    createEvent<K extends keyof EventNameMap>(eventType: K): EventNameMap[K]
    
    /**
    * @description Removes a previously added SceneEvent from the ScriptComponent.
    */
    removeEvent(event: SceneEvent): void
    
}

/**
* @description Base class for objects representing Script data.
*/
interface ScriptObject {
    /**
    * @description Returns the name of this object's type.
    */
    getTypeName(): string
    
    /**
    * @description Returns true if the object matches or derives from the passed in type.
    */
    isOfType(type: string): boolean
    
    /**
    * @description Returns true if this object is the same as `other`. Useful for checking if two references point to the same thing.
    */
    isSame(other: ScriptObject): boolean
    
}

/**
* @description Represents the Lens scene. Accessible through `global.scene`.
*/
interface ScriptScene extends ScriptObject {
    /**
    * @description Adds a new [SceneObject](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject) named `name` to the scene.
    */
    createSceneObject(name: string): SceneObject
    
    /**
    * @description Creates a new Render Target texture with a [RenderTargetProvider](https://lensstudio.snapchat.com/api/lens-studio/Classes/Providers#RenderTargetProvider) as its `control` property.
    */
    createRenderTargetTexture(): Texture
    
    /**
    * @description Create a texture containing the DepthStencilRenderTargetProvider.
    */
    createDepthStencilRenderTargetTexture(): Texture
    
    /**
    * @description Returns whether or not the scene is currently being recorded.
    */
    isRecording(): boolean
    
    /**
    * @description Returns a string describing the currently active device camera.
    
    * Returns "back" if the rear-facing (aka World) camera is active.
    
    * Returns "front" if the front-facing (aka Selfie) camera is active.
    
    * Otherwise, the camera is not initialized.
    */
    getCameraType(): string
    
    /**
    * @description Returns the number of [SceneObjects](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject) in the current scene.
    */
    getRootObjectsCount(): number
    
    /**
    * @description Returns the root [SceneObject](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject) at index `index` in the current scene.
    */
    getRootObject(index: number): SceneObject
    
    /**
    * @description The output Render Target of the actual recorded video.
    */
    captureTarget: Texture
    
    /**
    * @description The output Render Target that users will see in the live camera and during recording.
    */
    liveTarget: Texture
    
    /**
    * @description Similar to `liveTarget`, but this RenderTarget will not have predictive motion adjustments applied to it (only applicable on supported devices). [Learn more](https://docs.snap.com/spectacles/guides/lens-studio-integration/creating-lenses/#overlay-target)
    */
    liveOverlayTarget: Texture
    
    /**
    * @readonly
    
    * @description Returns true if the device supports Raytracing and Advanced Graphics Features is enabled in the project settings.
    */
    isRayTracingSupported: boolean
    
}

/**
* @description Segmentation model used for [SegmentationTextureProvider](https://lensstudio.snapchat.com/api/lens-studio/Classes/Providers#SegmentationTextureProvider).
*/
interface SegmentationModel extends BinAsset {
}

/**
* @description Controls a segmentation texture resource.
* Can be accessed through [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Texture#control-textureprovider) on a Segmentation texture.
* For more information, see the [Segmentation](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/segmentation/fullscreen-segmentation) guide.
*/
interface SegmentationTextureProvider extends TextureProvider {
    /**
    * @readonly
    
    * @description Triggered when the area segmented changes.
    */
    onMaskPercentageUpdated: event1<number, void>
    
}

/**
* @description Arguments used with the `InteractionComponent.onSelectEnd` event.
*/
interface SelectEndEventArgs extends ScriptObject {
}

/**
* @description Arguments used with the `InteractionComponent.onSelectStart` event.
*/
interface SelectStartEventArgs extends ScriptObject {
}

/**
* @description Low-level data class.
*/
interface SerializableWithUID extends ScriptObject {
    /**
    * @readonly
    */
    uniqueIdentifier: string
    
}

/**
* @description Base type for collision shapes.
*/
interface Shape extends ScriptObject {
}
declare namespace Shape {
    /**
    * @description Create a LevelsetShape with default settings.
    */
    export function createLevelsetShape(): LevelsetShape
    
    /**
    * @description Create a SphereShape with default settings.
    */
    export function createSphereShape(): SphereShape
    
    /**
    * @description Create a CapsuleShape with default settings.
    */
    export function createCapsuleShape(): CapsuleShape
    
    /**
    * @description Create a BoxShape with default settings.
    */
    export function createBoxShape(): BoxShape
    
    /**
    * @description Create a ConeShape with default settings.
    */
    export function createConeShape(): ConeShape
    
    /**
    * @description Create a CylinderShape with default settings.
    */
    export function createCylinderShape(): CylinderShape
    
    /**
    * @description Creates a new MeshShape.
    */
    export function createMeshShape(): MeshShape
    

}

interface SinglePlaneTrackingUpdatedEvent extends SceneEvent {
}

/**
* @description Represents skinning data for rigged meshes. See also: [MeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MeshVisual).
*/
interface Skin extends Component {
    /**
    * @description Get the bone on the skin.
    */
    getSkinBone(boneName: string): SceneObject
    
    setSkinBone(boneName: string, bone: SceneObject): void
    
    /**
    * @description Get all the bone names on the skin.
    */
    getSkinBoneNames(): string[]
    
    /**
    * @description Remove all bones on the skin.
    */
    clearBones(): void
    
}

/**
* @description Triggered when a smile ends on the tracked face.
*/
interface SmileFinishedEvent extends FaceTrackingEvent {
}

/**
* @description Triggered when a smile begins on the tracked face.
*/
interface SmileStartedEvent extends FaceTrackingEvent {
}

/**
* @description Details about friendship between the active user and another user.
*/
interface SnapchatFriendUserInfo {
    /**
    * @readonly
    
    * @description When this friendship started.
    */
    friendshipStart?: Date
    
    /**
    * @readonly
    
    * @description Last interaction between this user and the active one.
    */
    lastInteractionTime?: Date
    
}

/**
* @description Represents a Snapchat user in order to pass to other APIs or to retrieve certain details about the user like display name. 
*/
interface SnapchatUser extends ScriptObject {
    /**
    * @readonly
    
    * @description Name a Snapchat user has chosen to be shown as. Note that not all users set their display name, so you should treat this value as such.
    */
    displayName?: string
    
    /**
    * @readonly
    
    * @description Snapchat user's unique userName. Should not be relied on as a key because it can be changed.
    */
    userName: string
    
    /**
    * @readonly
    
    * @description Information about friendship between this friend and the active user like start date and streak.
    */
    friendInfo?: SnapchatFriendUserInfo
    
    /**
    * @readonly
    
    * @description Whether this user has a bitmoji that can be retireved via BitmojiModule.
    */
    hasBitmoji: boolean
    
    /**
    * @readonly
    
    * @description User's birth month and day. The month and day will start with the index of 1.
    */
    birthday?: SnapchatUserBirthday
    
    /**
    * @readonly
    
    * @description This user's zodiac.
    */
    zodiac?: Zodiac
    
}

/**
* @description Set the bone on the skin.
*/
interface SnapchatUserBirthday {
    /**
    * @readonly
    */
    day: number
    
    /**
    * @readonly
    */
    month: number
    
}

/**
* @description Provides a marker for tracking Snapcodes.
* For more information, see the [Marker Tracking](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/world/marker-tracking) guide.
*/
interface SnapcodeMarkerProvider extends MarkerProvider {
}

/**
* @description Called when the user taps on the capture button to record an image.
*/
interface SnapImageCaptureEvent extends SceneEvent {
}

/**
* @description Gets called when the user starts long pressing the capture button to record a Snap.
*/
interface SnapRecordStartEvent extends SceneEvent {
}

/**
* @description Gets called when the user stops long pressing the Snap button to end recording of a Snap.
*/
interface SnapRecordStopEvent extends SceneEvent {
}

/**
* @description Order that inTensor will be sorted when applying `TensorMath.argSortMasked()`.
*/
declare enum SortOrder {
    /**
    * @description Applies ascending sorting order when returning indices in TensorMath.argSortMasked().
    */
    Ascending,
    Descending
}

interface SpatialAudio extends ScriptObject {
    /**
    * @description Sets whether spatial audio is enabled or not.
    */
    enabled: boolean
    
    /**
    * @readonly
    
    * @description Distance effect settings.
    */
    distanceEffect: DistanceEffect
    
    /**
    * @readonly
    
    * @description Directivity effect settings.
    */
    directivityEffect: DirectivityEffect
    
    /**
    * @readonly
    
    * @description Position effect settings.
    */
    positionEffect: PositionEffect
    
}

interface SpatialMap {
}

/**
* @description Representation the signal strength over time at various frequencies present in a particular waveform. Created by applying Fast Fourier Transform (FFT) on the overlapping segments of the audio data.
*/
interface Spectrogram extends ScriptObject {
    /**
    * @description Processes audio data from the inTensor of inShape, writes result to the outTensor and returns the outTensor shape.
    */
    process(inTensor: Float32Array, inShape: vec3, outTensor: Float32Array): vec3
    
    /**
    * @readonly
    
    * @description Returns the maximum buffer length.
    */
    maxTensorSize: number
    
}

/**
* @description The builder class for Spectrogram.
*/
interface SpectrogramBuilder extends ScriptObject {
    /**
    * @description Creates a new Spectrogram.
    */
    build(): Spectrogram
    
    /**
    * @description Sets the frame size.
    */
    setFrameSize(frameSize: number): SpectrogramBuilder
    
    /**
    * @description Sets the number of samples between successive FFT segments.
    */
    setHopSize(hopSize: number): SpectrogramBuilder
    
    /**
    * @description Sets the length of the window.
    */
    setFFTSize(fftSize: number): SpectrogramBuilder
    
}

/**
* @description A sphere collision shape.
*/
interface SphereShape extends Shape {
    /**
    * @description Local radius of the sphere.
    */
    radius: number
    
}

interface SplineComponent extends Component {
}

/**
* @description Represents transform data for screen-aligned 2D sprites. Use on SceneObjects with a SpriteVisual Component.
* See also: [SpriteVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#deprecated-spritevisual).
*/
interface SpriteAligner extends Component {
    /**
    * @description The location of the point this sprite is bound to.
    */
    bindingPoint: vec2
    
    /**
    * @description The width and height of the SpriteVisual.
    */
    size: vec2
    
}

/**
* @description Stencil buffer clear option.
*/
declare enum StencilClearOption {
    /**
    * @description Stencil buffer clear operation will be skipped.
    */
    None,
    /**
    * @description Stencil buffer will be cleared by "clearStencilValue" property value.
    */
    CustomValue,
    /**
    * @description Stencil buffer will be cleared by texture from "maskTexture" property, if "maskTexture" is null then the stencil clear option will fallback to "CustomValue" mode.
    */
    CustomTexture
}

/**
* @description Specifies whether the front and/or back face stencil test will be applied. The initial value is "FrontAndBack"
*/
declare enum StencilFace {
    /**
    * @description The stencil test will be applied to the both front and back faces.
    */
    FrontAndBack,
    /**
    * @description The stencil test will be applied to the front face only.
    */
    Front,
    /**
    * @description The stencil test will be applied to the back face only.
    */
    Back
}

/**
* @description Specifies the stencil test function. The initial value is "Always".
*/
declare enum StencilFunction {
    /**
    * @description Always passes.
    */
    Always,
    /**
    * @description Always fails.
    */
    Never,
    /**
    * @description Passes if (referenceValue & readMask) < (stencil buffer value & readMask).
    */
    Less,
    /**
    * @description Passes if (referenceValue & readMask) <= (stencil buffer value & readMask).
    */
    LessEqual,
    /**
    * @description Passes if (referenceValue & readMask) > (stencil buffer value & readMask).
    */
    Greater,
    /**
    * @description Passes if (referenceValue & readMask) >= (stencil buffer value & readMask).
    */
    GreaterEqual,
    /**
    * @description Passes if (referenceValue & readMask) = (stencil buffer value & readMask).
    */
    Equal,
    /**
    * @description Passes if (referenceValue & readMask) != (stencil buffer value & readMask).
    */
    NotEqual
}

/**
* @description Options for specifying the action to take when stencil and depth tests resolve.
*/
declare enum StencilOperation {
    /**
    * @description Keep the current value.
    */
    Keep,
    /**
    * @description Set the stencil buffer value to 0.
    */
    Zero,
    /**
    * @description Sets the stencil buffer value to "referenceValue" property of StencilState.
    */
    Replace,
    /**
    * @description Increments the current stencil buffer value. Clamps to the maximum representable unsigned value.
    */
    IncrementClamp,
    /**
    * @description Increments the current stencil buffer value. Wraps stencil buffer value to zero when incrementing the maximum representable unsigned value.
    */
    IncrementWrap,
    /**
    * @description Decrements the current stencil buffer value. Clamps to 0.
    */
    DecrementClamp,
    /**
    * @description Decrements the current stencil buffer value. Wraps stencil buffer value to the maximum representable unsigned value when decrementing a stencil buffer value of zero.
    */
    DecrementWrap,
    /**
    * @description Bitwise inverts the current stencil buffer value.
    */
    Invert
}

/**
* @description The stencil test state for Pass.
*/
interface StencilState extends ScriptObject {
    /**
    * @description Enable or disable the stencil test for pass. The initial value is false.
    */
    enabled: boolean
    
    /**
    * @description Specifies whether the front and/or back face stencil test will be applied. The initial value is "FrontAndBack"
    */
    face: StencilFace
    
    /**
    * @description Specifies the action to take when the stencil test fails. The initial value is "Keep".
    */
    stencilFailureOperation: StencilOperation
    
    /**
    * @description Specifies the stencil action when the stencil test passes, but the depth test fails. The initial value is "Keep".
    */
    depthFailureOperation: StencilOperation
    
    /**
    * @description Specifies the stencil action when both the stencil test and the depth test pass, or when the stencil test passes and either there is no depth buffer or depth testing is not enabled. The initial value is "Keep".
    */
    depthStencilPassOperation: StencilOperation
    
    /**
    * @description Specifies the stencil test function. The initial value is "Always".
    */
    stencilCompareFunction: StencilFunction
    
    /**
    * @description Specifies the reference value for the stencil test. It is clamped to the range [0..pow(2, n)1], where n is the number of bitplanes in the stencil buffer. The initial value is 0.
    */
    referenceValue: number
    
    /**
    * @description Specifies a mask that is ANDed with both the reference value and the stored stencil value when the test is done. The initial value is 0xFF.
    */
    readMask: number
    
    /**
    * @description Specifies a bit mask to enable and disable writing of individual bits in the stencil planes. Initially, the mask is 0xFF.
    */
    writeMask: number
    
}

/**
* @description Settings for saving values in a MultiplayerSession.
*/
interface StorageOptions extends ScriptObject {
    /**
    * @description The Storage Scope to set, get, or delete values from.
    */
    scope: StorageScope
    
}
declare namespace StorageOptions {
    /**
    * @description Create a new StorageOptions object.
    */
    export function create(): StorageOptions
    

}

/**
* @description Specifies a scope for storing or retrieving values from a MultiplayerSession.
*/
declare enum StorageScope {
    /**
    * @description Stored variables are visible to the user only, and accessible until lens cache is cleared.
    */
    User,
    /**
    * @description Stored values are visible and editable by any user who is using the same session. Values are stored as long as the session that corresponds to chat exists.
    */
    Session
}

/**
* @description Options for stretching a mesh to fit a [ScreenTransform's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) bounding box.
* Used in [MeshVisual's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MeshVisual) `stretchMode` property, as long as the SceneObject has a [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) attached.
* Also used in [TextFill's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#TextFill) `textureStretch` property to control texture stretching when drawing text.

* See the [Image guide](https://lensstudio.snapchat.com/lens-studio/references/guides/adding-content/2d/image) for more information about stretch modes.
*/
declare enum StretchMode {
    /**
    * @description Scale uniformly so that both width and height fit within the bounds.
    */
    Fit,
    /**
    * @description Scale uniformly so that both width and height meet or exceed the bounds.
    */
    Fill,
    /**
    * @description Scale non-uniformly to match the exact width and height of the bounds.
    */
    Stretch,
    /**
    * @description Scale uniformly to match the same height as the bounds.
    */
    FitHeight,
    /**
    * @description Scale uniformly to match the same width as the bounds.
    */
    FitWidth,
    /**
    * @description Same as `Fill`, but when used with the [Image component](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Image) any area outside of the bounds is cropped out.
    */
    FillAndCut
}

interface Studio {
}
declare namespace Studio {
    /**
    * @description Prints out a message to the Logger window in Studio. When running on a device paired to Lens Studio, it will send the message directly to the Logger window in Studio to help with debugging.
    */
    export function log(message: unknown): void
    

}

/**
* @description Used with [DeviceTracking.surfaceOptions](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#DeviceTracking) to change settings for Surface tracking mode.
*/
interface SurfaceOptions extends ScriptObject {
    /**
    * @description If enabled, surface tracking will be improved using native AR tracking.
    */
    enhanceWithNativeAR: boolean
    
}

/**
* @description If a [DeviceTracking](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#DeviceTracking) component is present in the scene, this event is triggered when
* the tracking is restarted (typically when the Lens starts, or if the user taps the ground).
*/
interface SurfaceTrackingResetEvent extends SceneEvent {
}

declare enum TangentType {
    Const,
    Broken,
    Clamped,
    Free,
    Linear
}

/**
* @description This event is triggered when the user taps on the screen.
*/
interface TapEvent extends SceneObjectEvent {
}

/**
* @description Arguments used with the `InteractionComponent.onTap` event.
*/
interface TapEventArgs extends ScriptObject {
    /**
    * @readonly
    
    * @description The position of the touch on the screen. [0,0] being top left, and [1,1] being bottom right.
    */
    position: vec2
    
}

/**
* @description Namespace for mathematical operations on tensors. Useful with MLComponent.

**Tensor broadcasting rules**

* Tensor (channels, width, height) op Tensor (1, 1, 1)  =  the same as applying op with scalar

* Tensor (channels, width, height) op Tensor (channels, 1, 1)  =  the same as applying op per channel
*/
interface TensorMath {
}
declare namespace TensorMath {
    /**
    * @description Applies softMax function to `inTensor`, with the specified `inShape`. The result is put into `outTensor`.
    
    * If `inShape` = {width, height, channels}, then the shape of `outTensor` should be {1, 1, channels}.
    */
    export function softMax(inTensor: Float32Array, inShape: vec3, outTensor: Float32Array): void
    
    /**
    * @description Applies softArgMax function to the `inTensor`, with the specified `inShape`. The result is put into `outTensor`.
    
    * If `inShape` = {width, height, channels}, then the shape of `outTensor` should be {1, 2, channels}.
    */
    export function softArgMax(inTensor: Float32Array, inShape: vec3, outTensor: Float32Array, normalized: boolean): void
    
    /**
    * @description Returns the indices of the maximum values along an the channels of `inTensor`, with the specified `inShape`. The result is put into `outTensor`.
    
    * If `inShape` = {width, height, channels}, then the shape of `outTensor` should be {1, 2, channels}.
    */
    export function argMax(inTensor: Float32Array, inShape: vec3, outTensor: Uint32Array): void
    
    /**
    * @description Similar to `numpy.argsort()`, but in TensorMath, we have no kind and order parameters. Also, tensor is always 3D, and axis can be equal to 0(x), 1(y) or 2(z).
    */
    export function argSort(inTensor: Float32Array, shape: vec3, axis: number, outTensor: Uint32Array): void
    
    /**
    * @description Concatenates inTensorA and inTensorB along the specified axis and puts the result to the outTensor. Input arrays have to have same dimension along other 2 axes.
    */
    export function concat(inTensorA: Float32Array, inShapeA: vec3, inTensorB: Float32Array, inShapeB: vec3, axis: number, outTensor: Float32Array): void
    
    /**
    * @description Rearranges the inTensor of inShape according to the desired ordering and puts the result into outTensor.
    */
    export function permute(inTensor: Float32Array, inShape: vec3, permuteAxis: vec3, outTensor: Float32Array): void
    
    /**
    * @description Duplicates inTensor elements and store result in the outTensor.
    
    * Axis specifies the number of repeats along the axis e.g:
    
    * axis(1, 1, 1): outTensor will be the same as inTensor
    
    * axis(2, 1, 2): inTesnor = [1 2 3 | 4 5 6], inShape = [3 2 1] => 
    
    * outTensor = [[1 1 2 2 3 3 | 4 4 5 5 6 6] [1 1 2 2 3 3 | 4 4 5 5 6 6]] outShape = [6 2 2]
    */
    export function repeat(inTensor: Float32Array, inShape: vec3, axis: vec3, outTensor: Float32Array): void
    
    /**
    * @description Applies subpixelArgMax function to the `inTensor`, with the specified `inShape` and kernel size. The result is put into `outTensor`.
    
    * If `inShape` = {width, height, channels}, then the shape of `outTensor` should be {1, 2, channels}.
    */
    export function subpixelArgMax(inTensor: Float32Array, inShape: vec3, outTensor: Float32Array, kernelSize: number): void
    
    /**
    * @description Estimates the object pose given a set of object points (`inObjectPoints`), their corresponding image projections (`inImagePoints`), and the intrinsic camera matrix (`transform`).
    
    * `inObjectPoints` - Float32Array of size (point count * 3). 3D points of the object
    
    * `inImagePoints` - Float32Array of size (point count * 2). 2D points which are projections of the points in `inObjectPoints`, with some applied transformation we are attempting to find
    
    * `imagePointsShape` - Should be {2, point count, 1}
    
    * `transform` - Intrinsic camera matrix
    
    * `flags` - Currently unused, can be left as 0. Further functionality may be added in the future.
    
    * `outRotTrans` - Float32Array of size 6 where results are written. Describes object transformation:
    
    **vec3(outRotTrans[0], outRotTrans[1], outRotTrans[2])** - object rotation in Rodrigues format
    
    **vec3(outRotTrans[3], outRotTrans[4], outRotTrans[5])** - object position in 3D space
    */
    export function solvePnP(inObjectPoints: Float32Array, inImagePoints: Float32Array, imagePointsShape: vec3, transform: mat3, flags: number, outRotTrans: Float32Array): boolean
    
    /**
    * @description Estimates the object pose given a set of object points (inObjectPoints), their corresponding image projections (inImagePoints), and the intrinsic camera matrix (cameraIntrinsicsMatrix). Similar to solvePnP but has additional camera distortion coefficient settings.
    
    * `distortionCoeff` - Input vector of distortion coefficients. If the vector is null - the zero distortion coefficients are assumed. 
    
    * `distortionCoeffShape` - {numElements, 1, 1} where numElements can be 4, 5, 8 or 12.
    
    * `useExtrinsicGuess` - The function uses the provided outRotTrans values as initial approximations of the rotation and translation vectors and further optimizes them.
    
    */
    export function solvePnPExtended(inObjectPoints: Float32Array, inImagePoints: Float32Array, imagePointsShape: vec3, cameraIntrinsicsMatrix: mat3, distortionCoeff: Float32Array, distortionCoeffShape: vec3, useExtrinsicGuess: boolean, flags: number, outRotTrans: Float32Array): boolean
    
    /**
    * @description `solvePnPRansac` is very similar to `solvePnPExtended` except that it uses Random Sample Consensus ( RANSAC ) for robustly estimating the pose.
    
    * `iterationsCount` - The number of times the minimum number of points are picked and the parameters estimated
    
    * `reprojectionError` - As mentioned earlier in RANSAC the points for which the predictions are close enough are called inliers. This parameter value is the maximum allowed distance between the observed and computed point projections to consider it an inlier.
    
    * `confidence` -  Number of inliers. If the algorithm at some stage finds more inliers than confidence, it finishes.
    * `outInliers` - Output array that contains indices of inliers in objectPoints and imagePoints .
    * outRotTrans
    */
    export function solvePnPRansac(inObjectPoints: Float32Array, inImagePoints: Float32Array, imagePointsShape: vec3, cameraIntrinsicsMatrix: mat3, distortionCoeff: Float32Array, distortionCoeffShape: vec3, useExtrinsicGuess: boolean, iterationsCount: number, reprojectionError: number, confidence: number, flags: number, outInliers: Uint8Array, outRotTrans: Float32Array): boolean
    
    /**
    * @description Stabilizes image objects between two consecutive frames caused by camera or object movement.
    * Results are written to the `points` array.
    
    * `prevGrayscale` - Uint8Array of size (width * height). Grayscale data of previous frame
    
    * `grayscale` - Uint8Array of size (width * height). Grayscale data of current frame
    
    * `textureShape` - Should be {width, height, 1}
    
    * `prevPoints` - Float32Array of size (point count * 2). Positions of 2D points on the previous frame
    
    * `points` - Float32Array of size (point count * 2). Results are written here - positions of 2D points on the current frame
    
    * `pointsShape` - Should be {2, point count, 1}
    
    * `winSize` - Size of the search window for each pyramid level
    
    * `maxLevel` - Maximal pyramid level number, with zero based index
    
    * `maxCount` - Terminate if iteration count exceeds maxCount
    
    * `epsilon` - Terminate if window movement is less than epsilon
    */
    export function opticalFlow(prevGrayscale: Uint8Array, grayscale: Uint8Array, textureShape: vec3, prevPoints: Float32Array, points: Float32Array, pointsShape: vec3, winSize: vec2, maxLevel: number, maxCount: number, epsilon: number): void
    
    /**
    * @exposesUserData
    
    * @description Converts the texture to a set of 0-255 grayscale values, and outputs the result into `outTensor`.
    
    * `outTensor` should be a Uint8Array of shape {width, height, 1}.
    */
    export function textureToGrayscale(texture: Texture, grayscaleBuffer: Uint8Array, grayscaleBufferShape: vec3): void
    
    /**
    * @description Places the maximum values of `inTensor` into `outTensor`.
    
    * `outTensor` should have the shape {1, 1, channels}.
    */
    export function max(inTensor: Float32Array, inShape: vec3, outTensor: Float32Array): void
    
    /**
    * @description Places the minimum values of `inTensor` into `outTensor`.
    
    * `outTensor` should have the shape {1, 1, channels}.
    */
    export function min(inTensor: Float32Array, inShape: vec3, outTensor: Float32Array): void
    
    /**
    * @description Raises elements of `inTensor` to the power of `val`, and puts the results into `outTensor`.
    */
    export function power(inTensor: Float32Array, val: number, outTensor: Float32Array): void
    
    /**
    * @description Clamps all values of `inTensor` between `minVal` and `maxVal`, and puts the result into `outTensor`.
    */
    export function clamp(inTensor: Float32Array, minVal: number, maxVal: number, outTensor: Float32Array): void
    
    /**
    * @description Multiplies each element of `inTensor` by a scalar value and puts the result into `outTensor`.
    */
    export function mulScalar(inTensor: Float32Array, scalar: number, outTensor: Float32Array): void
    
    /**
    * @description Adds a scalar value to each element of `inTensor` and puts the result into `outTensor`.
    */
    export function addScalar(inTensor: Float32Array, scalar: number, outTensor: Float32Array): void
    
    /**
    * @description Calculate the sum of the inTensor elements and store result in the outTensor.
    
    * Axis specifies axis along which a sum is performed., e.g:
    
    * axis(0, 0, 0): the sum will be performed on the whole tensor
    
    * axis(0, 0, 1): the sum will be performed along the z axis. outTensor will store inShape.z values
    
    * axis(1, 1, 0): the sum will be performed along x and y axes. outTensor will store 
    
    * inShape.x * inShape.y values, where outTensor[y][x] is the sum of all inTensor[0..inShape.z-1][y][x] values
    */
    export function sum(inTensor: Float32Array, inShape: vec3, axis: vec3, outTensor: Float32Array): void
    
    /**
    * @description Converts inTensor from the power scale to the decibel scale.
    */
    export function powerToDb(inTensor: Float32Array, topDb: number, outTensor: Float32Array): void
    
    /**
    * @description Converts inTensor from the amplitude scale to the decibel scale.
    */
    export function amplitudeToDb(inTensor: Float32Array, outTensor: Float32Array): void
    
    /**
    * @description Adds `inTensorA` to `inTensorB` and puts the result into `outTensor`. See the broadcasting rules for elementwise operations.
    */
    export function addTensors(inTensorA: Float32Array, inShapeA: vec3, inTensorB: Float32Array, inShapeB: vec3, outTensor: Float32Array): void
    
    /**
    * @description Subtracts `inTensorB` from `inTensorA` and puts the result into `outTensor`. See the broadcasting rules for elementwise operations.
    */
    export function subTensors(inTensorA: Float32Array, inShapeA: vec3, inTensorB: Float32Array, inShapeB: vec3, outTensor: Float32Array): void
    
    /**
    * @description Multiplies `inTensorA` and `inTensorB` and puts the result into `outTensor`. See broadcasting rules for elementwise operations.
    */
    export function mulTensors(inTensorA: Float32Array, inShapeA: vec3, inTensorB: Float32Array, inShapeB: vec3, outTensor: Float32Array): void
    
    /**
    * @description Divides `inTensorA` by `inTensorB` and puts the result into `outTensor`. See broadcasting rules for elementwise operations.
    */
    export function divTensors(inTensorA: Float32Array, inShapeA: vec3, inTensorB: Float32Array, inShapeB: vec3, outTensor: Float32Array): void
    
    /**
    * @description Looks for contour points in a grayscale texture, or any one-channel tensor.
    
    * `grayscaledTexture` - Float32Array of size (width * height). Grayscale texture or one-channel tensor to evaluate
    
    * `textureShape` - Should be {width, height, 1}
    
    * `threshold` - Quality value threshold for found contour points
    
    * Each contour point found must satisfy these conditions:
    
    * 1. The point's quality value should be >= `threshold`
    
    * 2. The number of points which have values < `threshold` and lie in the rectangle with left corner vec2(x-`winSize`, y-`winSize`) and right corner vec2(x+`winSize`-1, y+`winSize`-1) should be <= `maxNearCount`
    
    * `outTensor` - Float32Array where results are written. Found contour points are written in the format: x0, y0, x1, y1, etc.
    * The number of points found will not exceed outTensor's size / 2.
    */
    export function getContour(grayscaledTexture: Float32Array, textureShape: vec3, threshold: number, winSize: number, maxNearCount: number, outTensor: Float32Array): number
    
    /**
    * @description Sorts 2D points by polar angle relative to the `center` point.
    
    * `inTensor` - Float32Array with size (point count * 2). 2D points to sort
    
    * `tensorShape` - Should be {2, point count, 1}
    
    * `center` - Center point to use for polar angle sorting
    */
    export function polarSort2d(inTensor: Float32Array, tensorShape: vec3, center: vec2): void
    
    /**
    * @description Smooths a polygon formed by input points. Works with points of any dimension count, for example `2` for 2D points or `3` for 3D points.
    * Results are written to `outTensor`.
    
    * `inTensor` - Float32Array of size (point dimension count * point count). Points of polygon to smooth out
    
    * `tensorShape` - Should be {point dimension count, point count, 1}
    
    * `step` - Smoothing value, higher value meaning higher smoothness
    
    * `outTensor` - Float32Array of size (point dimension count * point count). Smoothed points are written to this array
    */
    export function smoothPoints(inTensor: Float32Array, tensorShape: vec3, step: number, outTensor: Float32Array): void
    
    /**
    * @description Applies a rotation to each point in a set of 3D points, and places the results in `outPoints`.
    
    * `points` - Float32Array of size (point count * 3). Points to rotate
    
    * `pointsShape` - Should be {3, point count, 1}
    
    * `rotation` - Quaternion rotation to apply
    
    * `outPoints` - Float32Array of size (point count * 3). Resulting rotated points are placed here
    */
    export function rotatePoints3d(points: Float32Array, pointsShape: vec3, rotation: quat, outPoints: Float32Array): void
    
    /**
    * @description Checks for each 2D point whether it is inside of a rectangle.
    
    * `points` - Float32Array of size (point count * 2). 2D points to check
    
    * `pointsShape` - Should be {2, point count, 1}
    
    * `rect` - 2D rectangle points will be checked against
    
    * `output` - Uint8Array of size (point count). For each point, this will be filled with 1 if the point is inside the rectangle, or 0 otherwise.
    */
    export function isInRectangle(points: Float32Array, pointsShape: vec3, rect: Rect, output: Uint8Array): void
    
    /**
    * @description Finds minimum distances between each 2D point from one array, to 2D points in another array.
    
    * `from` - Float32Array of size ("from" point count * 2). 2D points from which min distances should be found
    
    * `fromShape` - Should be {2, "from" point count, 1}
    
    * `to` - Float32Array of size ("to" point count * 2). 2D points to which min distances should be found
    
    * `toShape` - Should be {2, "to" point count, 1}
    
    * `output` - Float32Array of size ("from" point count). For each point in the `from` array, the minimum distance to points from the `to` array will be written to this array.
    */
    export function findMinDistancesBetween(from: Float32Array, fromShape: vec3, to: Float32Array, toShape: vec3, output: Float32Array): void
    
    /**
    * @description If we consider the tensor as a 3D array, this function finds the indexes of the minimum element in each subarray of size `window`.
    
    * `tensor` - Float32Array of size (width * height * depth). 3D array of input data
    
    * `tensorShape` - Should be {width, height, depth}
    
    * `window` - Size of each subarray, in each of which will be found the index of the min element
    
    * `output` - Float32Array of size (width * height * depth). The index of the min value will be written into this array for each subarray.
    */
    export function minInSlideWindow(tensor: Float32Array, tensorShape: vec3, window: vec3, output: Uint32Array): void
    
    /**
    * @description If we consider the tensor as a 3D array, this function finds the indexes of the maximum element in each subarray of size `window`.
    
    * `tensor` - Float32Array of size (width * height * depth). 3D array of input data
    
    * `tensorShape` - Should be {width, height, depth}
    
    * `window` - Size of each subarray, in each of which will be found the index of the max element
    
    * `output` - Float32Array of size (width * height * depth). The index of the max value will be written into this array for each subarray.
    */
    export function maxInSlideWindow(tensor: Float32Array, tensorShape: vec3, window: vec3, output: Uint32Array): void
    
    /**
    * @description Calculates the length of all vectors in an array. Vectors can be of any dimension count.
    
    * `vectors` - Float32Array of size (vector dimension count * vector count). Vectors to measure the length of
    
    * `vectorsShape` - Should be {vector dimension count, vector count, 1}
    
    * `output` - Float32Array of size (vector count). For each vector in `vectors`, its length will be written to this array
    */
    export function getVectorsLength(vectors: Float32Array, vectorsShape: vec3, output: Float32Array): void
    
    /**
    * @description From the input points, finds the two points with the maximum distance between them and returns this distance. Works with points of any dimension count.
    
    * `points` - Float32Array of size (point dimension count * point count). Points to find the max distance between.
    
    * `pointsShape` - Should be {point dimension count, point count, 1}
    */
    export function maxDistanceBetweenPoints(points: Float32Array, pointsShape: vec3): number
    
    /**
    * @description Erodes an image by using a specific structuring element.
    */
    export function erode(inTensor: Float32Array, inShape: vec3, kernelTensor: Float32Array, kernelShape: vec3, anchor: vec2, iterations: number, borderType: TensorMath.BorderType, borderValue: vec4, outTensor: Float32Array): void
    
    /**
    * @description Dilates an image by using a specific structuring element.
    */
    export function dilate(inTensor: Float32Array, inShape: vec3, kernelTensor: Float32Array, kernelShape: vec3, anchor: vec2, iterations: number, borderType: TensorMath.BorderType, borderValue: vec4, outTensor: Float32Array): void
    
    /**
    * @description Returns contours. Note that: contours are sorted from the largest to the smallest; 2) We cannot return Array of TypedArray-s, so all contours data is stored in single outTensor and Array with sizes of each contour is returned.
    
    */
    export function findContours(inTensor: Uint8Array, inShape: vec3, mode: number, method: number, offset: vec2, outTensor: any): number[]
    
    /**
    * @description Return TensorMath's format of RotatedRect.
    
    */
    export function minAreaRect(inTensor: Float32Array, inShape: vec3): RotatedRect
    
    export function getRotatedRectPoints(rotatedRect: RotatedRect, outTensor: Float32Array): void
    
    /**
    * @description Fills a convex polygon.
    */
    export function fillConvexPoly(imgTensor: Float32Array, imgShape: vec3, pointsTensor: any, pointsShape: vec3, color: vec4, lineType: number, shift: number): void
    
    /**
    * @description Fills a polygon. Note that you should pass an Array of Int32Array-s (polygonsTensors).
    
    */
    export function fillPoly(imgTensor: Float32Array, imgShape: vec3, polygonsTensors: any[], color: vec4, lineType: number, shift: number, offset: vec2): void
    
    /**
    * @description Draws a line segment connecting two points.
    */
    export function drawLine(imgTensor: Float32Array, imgShape: vec3, point1: vec2, point2: vec2, color: vec4, thickness: number, lineType: number, shift: number): void
    
    /**
    * @description Multiply a matrix by a set of points through an efficient batch operation.
    */
    export function mulMatToPoints(pointsTensor: Float32Array, pointsShape: vec3, matrix: mat4, outTensor: Float32Array): void
    
    /**
    * @description Project 3D points into 2D space using an efficient batch operation.
    */
    export function projectPoints(pointsTensor: Float32Array, pointsShape: vec3, projectionMatrix: mat4, outTensor: Float32Array): void
    
    /**
    * @description Blurs an image tensor using the box filter.
    */
    export function applyBoxFilter(inTensor: Float32Array, inShape: vec3, kernelSize: vec2, anchor: vec2, normalize: boolean, borderType: TensorMath.BorderType, outTensor: Float32Array): void
    
    /**
    * @description Applies a fixed-level threshold to each array element.
    */
    export function applyThreshold(inTensor: Float32Array, threshold: number, maxValue: number, type: TensorMath.ThresholdMethod, outTensor: Float32Array): void
    
    /**
    * @description Approximates a polygonal curve with the specified precision.
    */
    export function approximatePolygonalCurve(inTensor: Float32Array, inShape: vec3, epsilon: number, closed: boolean, outTensor: Float32Array): number
    
    /**
    * @description Applies a list of indices of a tensor in sorted order of their corresponding values in the tensor to the given outTensor. Only indices whose corresponding values are not equal to 0 in the provided mask will be returned. Indices from the original tensor, before applying the mask, will be returned in the provided array. Returns the size of the list of indices applied to the outTensor (note: this will be equal to the number of non-zero values provided in the mask). eg:
    *  
    * ```js
    * var inTensor = [5.0, 2.0, 1.0, 3.0, 7.0, -1.0, -5.0]; 
    * var mask = [1.0, 1.0, 1.0, 1.0, 1.0, 0.0, 0.0]; 
    * var outTensor = new Uint32Array(7); 
    * var order = SortOrder.Ascending;
    * var size = TensorMath.argSortMasked(inTensor, mask, outTensor, order); 
    * print(outTensor.subarray(size)); // Expected: [2, 1, 3, 0, 4]
    * ```
    
    */
    export function argSortMasked(inTensor: Float32Array, mask: Float32Array, outTensor: Uint32Array, order: SortOrder): number
    
    export function applyNMS(inTensor: Float32Array, inShape: vec3, scores: Float32Array, scoreThreshold: number, iouThreshold: number, outTensor: Uint32Array): number
    

}

declare namespace TensorMath {
    enum BorderType {
        Constant,
        Replicate,
        Reflect,
        Reflect101
    }

}

declare namespace TensorMath {
    enum ThresholdMethod {
        Binary,
        BinaryInv,
        Trunc,
        ToZero,
        ToZeroInv
    }

}

/**
* @description Visual component that renders dynamic text.
* See the [Text guide](https://lensstudio.snapchat.com/lens-studio/references/guides/adding-content/text/2d-text) for more information.
*/
interface Text extends BaseMeshVisual {
    /**
    * @description Text string to be drawn.
    */
    text: string
    
    /**
    * @description Font asset used.
    */
    font: Font
    
    /**
    * @description Font size used.
    */
    size: number
    
    /**
    * @description Controls how text should be handled when it goes past the vertical boundaries defined by the world space rect or ScreenTransform.
    
    */
    verticalOverflow: VerticalOverflow
    
    /**
    * @description Controls how text should be handled when it goes past the horizontal boundaries defined by the world space rect or ScreenTransform.
    
    */
    horizontalOverflow: HorizontalOverflow
    
    /**
    * @description Settings for how the text is drawn, such as fill color or texture.
    */
    textFill: TextFill
    
    /**
    * @description Settings for how dropshadow is used in text drawing.
    */
    dropshadowSettings: DropshadowSettings
    
    /**
    * @description Settings for how text outline is used in text drawing.
    */
    outlineSettings: OutlineSettings
    
    /**
    * @description Settings for drawing a background behind the text.
    */
    backgroundSettings: BackgroundSettings
    
    /**
    * @description If enabled, the rendered text will always scale to fit the boundaries defined by the world space rect or ScreenTransform.
    
    */
    sizeToFit: boolean
    
    /**
    * @description Modifies the spacing between letters. Set to 0 by default, which uses the font's normal letter spacing.
    * Negative values will remove space between letters, and positive values will add more space between letters.
    */
    letterSpacing: number
    
    /**
    * @description Modifies the vertical spacing between lines, as a multiple of lines. 1 will be single spacing, 2 will be double spaced, and 0.5 would be half the normal line height.
    */
    lineSpacing: number
    
    /**
    * @description If enabled, the text material will use Depth Testing. Useful when Text exists in 3D space.
    */
    depthTest: boolean
    
    /**
    * @description Controls the masking of color channels with a vec4b representing each channel with a boolean.
    */
    colorMask: vec4b
    
    /**
    * @description Controls the boundaries the text is aligned and wraps within when not using ScreenTransform. Referred to as the Layout Rect in the Inspector panel.
    
    */
    worldSpaceRect: Rect
    
    capitilizationOverride: CapitilizationOverride
    
    /**
    * @description Makes the Text component editable. When this is enabled the Text can be clicked to open up the device keyboard and edit the contents.
    */
    editable: boolean
    
    /**
    * @description Use this property to control whether to show the input preview the keyboard. Note this preview also enables cursor movement.
    */
    showEditingPreview: boolean
    
    /**
    * @description Use this property to override the touch handling for when to open the device keyboard when the Text is editable.
    */
    touchHandler: InteractionComponent
    
    /**
    * @readonly
    
    * @description This event will notify you when the input keyboard opens.
    */
    onEditingStarted: event0<void>
    
    /**
    * @readonly
    
    * @description This event will notify you on every edit to the text while the user is typing.
    */
    onEditingUpdated: event1<string, void>
    
    /**
    * @readonly
    
    * @description This event will notify you when the user finishes editing the text.
    */
    onEditingFinished: event1<string, void>
    
}

/**
* @description Renders a given text with a 3D mesh.
*/
interface Text3D extends MaterialMeshVisual {
    /**
    * @description Splits the Text3D into individual RenderMeshVisuals. This destroys the Text3D component and you can no longer edit its properties like text.
    */
    split(): RenderMeshVisual[]
    
    /**
    * @description Text string to be drawn.
    */
    text: string
    
    /**
    * @description Font asset used.
    */
    font: Font
    
    /**
    * @description Font size used.
    */
    size: number
    
    /**
    * @description How deep the text meshes should be relative to the line height. 0-1 portion of the basic height which will be used for mesh depth. Values > 1 are valid.
    */
    extrusionDepth: number
    
    /**
    * @description Starting from the Text3D's local position control whether the meshes are extruded forwards, backwards, or both directions
    */
    extrudeDirection: number
    
    /**
    * @description Controls how text should be handled when it goes past the vertical boundaries defined by the world space rect or ScreenTransform.
    
    */
    verticalOverflow: VerticalOverflow
    
    /**
    * @description Controls how text should be handled when it goes past the horizontal boundaries defined by the world space rect or ScreenTransform.
    
    */
    horizontalOverflow: HorizontalOverflow
    
    /**
    * @description If enabled, the rendered text will always scale to fit the boundaries defined by the world space rect or ScreenTransform.
    
    */
    sizeToFit: boolean
    
    /**
    * @description Modifies the spacing between letters. Set to 0 by default, which uses the fonts normal letter spacing. Negative values will remove space between letters, and positive values will add more space between letters.
    */
    letterSpacing: number
    
    /**
    * @description Modifies the vertical spacing between lines, as a multiple of lines. 1 will be single spacing, 2 will be double spaced, and 0.5 would be half the normal line height.
    */
    lineSpacing: number
    
    /**
    * @description Controls the boundaries the text is aligned and wraps within when not using ScreenTransform. Referred to as the Layout Rect in the Inspector panel.
    
    */
    worldSpaceRect: Rect
    
    capitilizationOverride: CapitilizationOverride
    
    /**
    * @description Makes the Text component editable. When this is enabled the Text can be clicked to open up the device keyboard and edit the contents.
    */
    editable: boolean
    
    /**
    * @description Use this property to control whether to show the input preview the keyboard. Note this preview also enables cursor movement.
    */
    showEditingPreview: boolean
    
    /**
    * @description Use this property to override the touch handling for when to open the device keyboard when the Text is editable.
    */
    touchHandler: InteractionComponent
    
    /**
    * @readonly
    
    * @description This event will notify you when the user starts typing.
    */
    onEditingStarted: event0<void>
    
    /**
    * @readonly
    
    * @description This event will notify you on every edit to the text while the user is typing.
    */
    onEditingUpdated: event1<string, void>
    
    /**
    * @readonly
    
    * @description This event will notify you when the user finishes editing the text.
    */
    onEditingFinished: event1<string, void>
    
}

/**
* @description Fill settings used by several text related classes.
* Used in [Text's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) `textFill` property, [DropshadowSettings'](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#DropshadowSettings) `fill` property, and [OutlineSettings'](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#OutlineSettings) `fill` property.
*/
interface TextFill extends ScriptObject {
    /**
    * @description Controls which drawing method is used. Can switch between `Texture` mode (for drawing using a tiled texture) or `Solid` mode (for drawing a solid color).
    */
    mode: TextFillMode
    
    /**
    * @description If `mode` is set to `TextFillMode.Solid`, this will be used as the solid color used in drawing.
    */
    color: vec4
    
    /**
    * @description If `mode` is set to `TextFillMode.Texture`, this defines how many times the texture will tile across its drawing zone.
    */
    tileCount: number
    
    /**
    * @description If `mode` is set to `TextFillMode.Texture`, this defines what area should be used for tiling the texture.
    */
    tileZone: TileZone
    
    /**
    * @description If `mode` is set to `TextFillMode.Texture`, this defines what type of stretching is used when the
    * Texture's aspect ratio doesn't match the drawing area's aspect ratio.
    */
    textureStretch: StretchMode
    
    /**
    * @description If `mode` is set to `TextFillMode.Texture`, this will be used as the texture asset used in drawing.
    */
    texture: Texture
    
}

/**
* @description Used in [TextFill's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#TextFill) `mode` property. Controls which drawing method is used for the TextFill.
*/
declare enum TextFillMode {
    /**
    * @description Solid color will be used for drawing.
    */
    Solid,
    /**
    * @description Tiled texture will be used for drawing.
    */
    Texture
}

/**
* @description Provides access to the device's input system. Accessible through `global.textInputSystem`. 
*/
interface TextInputSystem extends ScriptObject {
    /**
    * @description Requests the client to pop up the keyboard with specific type and return key type, fill the inivisible text view with the initial text, and set the selected text range. When user starts editing the text, the new text string and the new selected range are returned to the lens by invoking the onTextChanged function.
    */
    requestKeyboard(options: TextInputSystem.KeyboardOptions): void
    
    /**
    * @description Requests the client to hide the keyboard and clear any existing text in the inivisible text view.
    */
    dismissKeyboard(): void
    
    /**
    * @description Requests the client to set the selected text range on the invisible text view.
    */
    setSelectedTextRange(range: vec2): void
    
    /**
    * @description Requests the client to set the editing ("cursor") position on the invisible text view.
    */
    setEditingPosition(position: number): void
    
}

declare namespace TextInputSystem {
    /**
    * @description The settings used for how the input keyboard should work.
    */
    class KeyboardOptions {
        /**
        * @description The type of keyboard to be used for input.
        */
        constructor()
        
        /**
        * @description Initial text to fill into the invisible text view.
        */
        initialText: string
        
        /**
        * @description Initial selected text range to set on the inivisible text view.
        */
        initialSelectedRange: vec2
        
        /**
        * @description Requested keyboard type.
        */
        keyboardType: TextInputSystem.KeyboardType
        
        /**
        * @description Requested return key type.
        */
        returnKeyType: TextInputSystem.ReturnKeyType
        
        /**
        * @description Sets whether a preview should be visible above the input keyboard.
        */
        enablePreview: boolean
        
        /**
        * @description Callback to be invoked every time the user presses a (non-return) key.
        */
        onTextChanged: (text: string, range: vec2) => void
        
        /**
        * @description Callback to be invoked when the user presses the return key.
        */
        onReturnKeyPressed: () => void
        
        /**
        * @description Callback to be invoked when keyboard is shown or dismissed.
        */
        onKeyboardStateChanged: (keyboardIsOpen: boolean) => void
        
        /**
        * @description Callback to be invoked when there is an error.
        */
        onError: (error: number, description: string) => void
        
    }

}

declare namespace TextInputSystem {
    /**
    * @description The different input style of keyboard input.
    */
    enum KeyboardType {
        /**
        * @description A keyboard type for entering normal text
        */
        Text,
        /**
        * @description A keyboard type for entering digits
        */
        Num,
        /**
        * @description A keyboard type for entering phone numbers
        */
        Phone,
        /**
        * @description A keyboard type for entering URLs
        */
        Url
    }

}

declare namespace TextInputSystem {
    /**
    * @description The return key style of keyboard input.
    */
    enum ReturnKeyType {
        /**
        * @description Return key label shows "Done"
        */
        Done,
        /**
        * @description Return key label shows "Go"
        */
        Go,
        /**
        * @description Return key label shows "Next"
        */
        Next,
        /**
        * @description Return key label shows "return"
        */
        Return,
        /**
        * @description Return key label shows "Search"
        */
        Search,
        /**
        * @description Return key label shows "Send"
        */
        Send
    }

}

/**
* @description Controls a text rendering texture. Can be accessed through the main rendering pass on a [Label](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#deprecated-label) component.
* The properties here mirror those on Label.
*/
interface TextProvider extends TextureProvider {
    /**
    * @description The text being rendered.
    */
    text: string
    
    /**
    * @description The font used for rendering text.
    */
    fontAsset: Font
    
    /**
    * @description The font size being used.
    */
    size: number
    
    /**
    * @description The color for rendering text.
    */
    textColor: vec4
    
    /**
    * @description If enabled, adds a dropshadow to the text.
    */
    useDropshadow: boolean
    
    /**
    * @description The horizontal and vertical offset used for dropshadow.
    */
    shadowOffset: vec2
    
    /**
    * @description The color used for dropshadow.
    */
    shadowColor: vec4
    
    /**
    * @description If enabled, renders an outline around the text.
    */
    useOutline: boolean
    
    /**
    * @description The strength of the outline effect.
    */
    outlineSize: number
    
    /**
    * @description The color used for the outline effect.
    */
    outlineColor: vec4
    
}

interface TextToSpeech {
}

declare namespace TextToSpeech {
    /**
    * @description Provides the configuration for the [Text to Speech Module](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#TextToSpeechModule).   It is used to control the language of the generated voice, the voice’s style and pace.
    */
    interface Options extends ScriptObject {
    }

}
declare namespace TextToSpeech {
    namespace Options {
        /**
        * @description Creates options for the [Text to Speech Module](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#TextToSpeechModule).
        */
        export function create(): TextToSpeech.Options
        
    
    }

}

declare namespace TextToSpeech {
    /**
    * @description Provides a map of what phoneme is said at what time in the synthesized speech.
    */
    interface PhonemeInfo extends ScriptObject {
        /**
        * @readonly
        
        * @description Phoneme start time in milliseconds.
        */
        startTime: number
        
        /**
        * @readonly
        
        * @description Phoneme end time in milliseconds.
        */
        endTime: number
        
        /**
        * @readonly
        
        * @description The phoneme found for the alloted time.
        */
        phoneme: string
        
        /**
        * @readonly
        
        * @description Is the phonemes part of an abusive word.
        */
        isAbusive: boolean
        
    }

}

declare namespace TextToSpeech {
    /**
    * @description The voice style on which the TextToSpeech will be synthesized, Varying from neutral style, to a more elaborated styles depending on the voice. Note that Sam’s voice still doesn’t have more than the neutral style, this will be added throughout the coming releases.
    */
    interface VoiceNames {
    }

}

declare namespace TextToSpeech {
    /**
    * @description Parameter returned in the `onTTSCompleteHandler` callback providing timing details for word pronunciation. 
    */
    interface WordInfo extends ScriptObject {
        /**
        * @readonly
        
        * @description The time in milliseconds when the word started in the audio. 
        */
        startTime: number
        
        /**
        * @readonly
        
        * @description The time in milliseconds when the word ended in the audio. 
        */
        endTime: number
        
        /**
        * @readonly
        
        * @description The words the synthesized audio was generated for (as text might be expanded during the synthesize process, there might be a slight variation between the input text and the words returned). 
        */
        word: string
        
    }

}

/**
* @description Allows generation of speech from a given text. You can use only one `TextToSpeechModule` in a Lens. However, its methods can be called multiple times in parallel if needed. 
*/
interface TextToSpeechModule extends Asset {
    /**
    * @description Generates speech ([AudioTrackAsset](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#AudioTrackAsset)) of a given text. `Input` should be the text to synthesize (Currently supports text in English only. Non English characters will be stripped). `Options` should be a [TextToSpeechOptions](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#TextToSpeechOptions). `onTTsComplete` should be a callback function which will be called once the audio generation is completed. The callback will receive two parameters: the generated audio file ([AudioTrackAsset](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#AudioTrackAsset)) and maps of word/timing ([WordInfos](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#TextToSpeechWordInfo)). `onTTSError` should be a callback function which will be called if there is an error. This callback will receive a message of the error code and its description. 
    */
    synthesize(input: string, options: TextToSpeech.Options, onTTSComplete: (audioTrackAsset: AudioTrackAsset, wordInfo: TextToSpeech.WordInfo[], phonemeInfo: TextToSpeech.PhonemeInfo[], voiceStyle: any) => void, onTTSError: (error: number, description: string) => void): void
    
}

/**
* @description Represents a texture file asset.
*/
interface Texture extends Asset {
    /**
    * @description Returns the width of the texture.
    */
    getWidth(): number
    
    /**
    * @description Returns the height of the texture.
    */
    getHeight(): number
    
    /**
    * @description Returns the Colorspace of the Texture.
    */
    getColorspace(): Colorspace
    
    /**
    * @description Returns a Texture that captures the current state of this Texture Asset.
    */
    copyFrame(): Texture
    
    /**
    * @description Creates a marker asset from the texture. 
    */
    createMarkerAsset(): MarkerAsset
    
    /**
    * @description The TextureProvider for the texture, which may control things like animation depending on the texture type.
    * See also: [AnimatedTextureFileProvider](https://lensstudio.snapchat.com/api/lens-studio/Classes/Providers#AnimatedTextureFileProvider).
    */
    control: TextureProvider
    
}

/**
* @description The base class for specialized Texture providers. Can be accessed through [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Texture#control-textureprovider).
*/
interface TextureProvider extends Provider {
    /**
    * @description Returns the texture's aspect ratio, which is calculated as width / height.
    */
    getAspect(): number
    
    /**
    * @description Returns the width of the texture in pixels.
    */
    getWidth(): number
    
    /**
    * @description Returns the height of the texture in pixels.
    */
    getHeight(): number
    
}

/**
* @description Defines the bounding area used for texture tiling with [TextFill's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#TextFill) `tileZone` property.
*/
declare enum TileZone {
    /**
    * @description The attached [ScreenTransform's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) bounding rectangle is used for texture tiling
    */
    Rect,
    /**
    * @description The Text component's drawn area (extents) is used for texture tiling
    */
    Extents,
    /**
    * @description Each character uses its own drawn area for texture tiling
    */
    Character,
    /**
    * @description The position of each character in screen space is used for tiling
    */
    Screen
}

/**
* @description This provider is returned by `global.touchSystem` and allows your lens to handle any touches on the screen, and optionally let certain touch types to pass through (let Snapchat handle the touch).

* Possible TouchType exception values:

* "TouchTypeNone"
* "TouchTypeTouch"
* "TouchTypeTap"
* "TouchTypeDoubleTap"
* "TouchTypeScale"
* "TouchTypePan"
* "TouchTypeSwipe"
*/
interface TouchDataProvider extends ScriptObject {
    /**
    * @description Returns a copy of `currentMask` with the `newException` flag set to true.
    */
    composeTouchBlockingExceptionMask(currentMask: number, newException: string): number
    
    /**
    * @description Allow or stop allowing a certain `TouchType` to pass through your lens. Useful for allowing Snapchat to handle certain `TouchType`, e.g. allowing `TouchTypeDoubleTap` to flip the camera.
    */
    enableTouchBlockingException(exception: string, enable: boolean): void
    
    /**
    * @description Set your lens to handle touches on the screen, preventing default Snapchat touch behavior from occuring. Useful for enabling full screen touches without any touch components. It is similar to creating a plane the size of the screen in front of the camera.
    */
    touchBlocking: boolean
    
    /**
    * @description The current touch mask.
    */
    touchBlockingExceptionMask: number
    
}

/**
* @description Triggered when a touch event ends.
*/
interface TouchEndEvent extends SceneObjectEvent {
    /**
    * @description Returns the normalized 2D screen position of the users touch.
    * The normalized coordinates range from ([0-1], [0-1]), (0,0) being top-left and (1,1) being bottom-right.
    */
    getTouchPosition(): vec2
    
    /**
    * @description Returns the ID of this specific touch. Useful for distinguishing between touches when multiple are occurring simultaneously.
    */
    getTouchId(): number
    
}

interface TouchEndEventArgs extends ScriptObject {
    /**
    * @readonly
    
    * @description Returns the ID of this specific touch. Useful for distinguishing between touches when multiple are occurring simultaneously.
    */
    touchId: number
    
    /**
    * @readonly
    
    * @description The position of the touch on the screen. [0,0] being top left, and [1,1] being bottom right.
    */
    position: vec2
    
}

/**
* @description Triggered when a touch position on the screen is moved.
*/
interface TouchMoveEvent extends SceneObjectEvent {
    /**
    * @description Returns the normalized 2D screen position of the users touch.
    * The normalized coordinates range from ([0-1], [0-1]), (0,0) being top-left and (1,1) being bottom-right.
    */
    getTouchPosition(): vec2
    
    /**
    * @description Returns the ID of this specific touch. Useful for distinguishing between touches when multiple are occurring simultaneously.
    */
    getTouchId(): number
    
}

/**
* @description Arguments used with the `InteractionComponent.onTouchMove` event.
*/
interface TouchMoveEventArgs extends ScriptObject {
    /**
    * @readonly
    
    * @description Returns the ID of this specific touch. Useful for distinguishing between touches when multiple are occurring simultaneously.
    */
    touchId: number
    
    /**
    * @readonly
    
    * @description The position of the touch on the screen. [0,0] being top left, and [1,1] being bottom right.
    */
    position: vec2
    
}

/**
* @description Triggered when a touch event starts.
*/
interface TouchStartEvent extends SceneObjectEvent {
    /**
    * @description Returns the normalized 2D screen position of the users touch.
    * The normalized coordinates range from ([0-1], [0-1]), (0,0) being top-left and (1,1) being bottom-right.
    */
    getTouchPosition(): vec2
    
    /**
    * @description Returns the ID of this specific touch. Useful for distinguishing between touches when multiple are occurring simultaneously.
    */
    getTouchId(): number
    
}

/**
* @description Arguments used with the `InteractionComponent.onTouchStart` event.
*/
interface TouchStartEventArgs extends ScriptObject {
    /**
    * @readonly
    
    * @description Returns the ID of this specific touch. Useful for distinguishing between touches when multiple are occurring simultaneously.
    */
    touchId: number
    
    /**
    * @readonly
    
    * @description The position of the touch on the screen. [0,0] being top left, and [1,1] being bottom right.
    */
    position: vec2
    
}

/**
* @description Represents a mesh generated by world tracking. Only available when world mesh tracking is supported and enabled.
*/
interface TrackedMesh extends ScriptObject {
    /**
    * @readonly
    
    * @description Returns the World Transformation matrix of the detected mesh.
    */
    transform: mat4
    
    /**
    * @readonly
    
    * @description Returns whether the tracked mesh is valid.
    */
    isValid: boolean
    
}

/**
* @description Classifications of mesh face. See [TrackedMeshHitTestResult](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#TrackedMeshHitTestResult).
*/
declare enum TrackedMeshFaceClassification {
    /**
    * @description None
    */
    None,
    /**
    * @description Wall
    */
    Wall,
    /**
    * @description Floor
    */
    Floor,
    /**
    * @description Ceiling
    */
    Ceiling,
    /**
    * @description Table
    */
    Table,
    /**
    * @description Seat
    */
    Seat,
    /**
    * @description Window
    */
    Window,
    /**
    * @description Door
    */
    Door
}

/**
* @description Provides histogram information about tracked world mesh faces in a given area. Returned by `DeviceTracking.calculateWorldMeshHistogram()`.
*/
interface TrackedMeshHistogramResult extends ScriptObject {
    /**
    * @readonly
    
    * @description Array of relative proportions for each classification, in the order described below. The values all add up to a total of 1.0.
    
    * The classification value order is:
    * 0: None
    * 1: Wall
    * 2: Floor
    * 3: Ceiling
    * 4: Table
    * 5: Seat
    * 6: Window
    * 7: Door
    */
    histogram: number[]
    
    /**
    * @readonly
    
    * @description Average normal direction, in world space, of the mesh faces.
    */
    avgNormal: vec3
    
}

/**
* @description Provides information about a TrackedMesh surface hit during a raycast. Is returned in an array when calling `DeviceTracking.hitTestWorldMesh()` or `DeviceTracking.raycastWorldMesh()`.
*/
interface TrackedMeshHitTestResult extends ScriptObject {
    /**
    * @readonly
    
    * @description Returns the TrackedMesh that was hit.
    */
    mesh: TrackedMesh
    
    /**
    * @readonly
    
    * @description Returns the world space position of the intersection point.
    */
    position: vec3
    
    /**
    * @readonly
    
    * @description Returns the world space normal direction of the intersection point.
    */
    normal: vec3
    
    /**
    * @readonly
    
    * @description Returns the classification of the mesh face at the intersection point. 
    */
    classification: TrackedMeshFaceClassification
    
}

/**
* @description A representation for plane detected by native tracking. Can be used with TrackedPoint.
*/
interface TrackedPlane extends ScriptObject {
    /**
    * @readonly
    
    * @description A coarse mesh describing the shape of the tracked plane. Can be displayed by RenderMeshVisual
    */
    mesh: RenderMesh
    
    /**
    * @readonly
    
    * @description The position, rotation, and scale of the detected plane
    */
    transform: mat4
    
    /**
    * @readonly
    
    * @description The center point of the detected plane
    */
    pivot: vec3
    
    /**
    * @readonly
    
    * @description The size of the detected plane, where it is described as width, 0, depth
    */
    size: vec3
    
    /**
    * @readonly
    
    * @description Whether the detected plane is still available to utilize
    */
    isValid: boolean
    
    /**
    * @readonly
    
    * @description Returns the orientation of the detected plane, either vertical or horizontal.
    
    */
    orientation: TrackedPlaneOrientation
    
}

declare enum TrackedPlaneOrientation {
    Horizontal,
    Vertical
}

/**
* @description A point on the real world not attached to any detected plane.
*/
interface TrackedPoint extends ScriptObject {
    /**
    * @readonly
    
    * @description Provides the world position of the tracked point.
    */
    position: vec3
    
    /**
    * @readonly
    
    * @description Provides the orientation of the tracked point.
    */
    orientation: quat
    
}

/**
* @description Allows you to bind the position and rotation of an object with this component to a [TrackedPoint](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#TrackedPoint).
*/
interface TrackedPointComponent extends Component {
    /**
    * @readonly
    
    * @description Returns whether the trackedPoint is valid.
    */
    isValid: boolean
    
    /**
    * @description The [TrackedPoint](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#TrackedPoint/) which this component will copy the position and rotation from.
    */
    trackedPoint: TrackedPoint
    
}

/**
* @description Controls the position, rotation, and scale of a [SceneObject](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject).  Every SceneObject automatically has a Transform attached.
*/
interface Transform extends ScriptObject {
    /**
    * @description Returns the Transform's local-to-world transformation matrix.
    */
    getWorldTransform(): mat4
    
    /**
    * @description Returns the Transform's world-to-local transformation matrix.
    */
    getInvertedWorldTransform(): mat4
    
    /**
    * @description Returns the Transform's position relative to its parent.
    */
    getLocalPosition(): vec3
    
    /**
    * @description Returns the Transform's position relative to the world.
    */
    getWorldPosition(): vec3
    
    /**
    * @description Returns the Transform's rotation relative to its parent.
    */
    getLocalRotation(): quat
    
    /**
    * @description Returns the Transform's rotation relative to the world.
    */
    getWorldRotation(): quat
    
    /**
    * @description Returns the Transform's scale relative to its parent.
    */
    getLocalScale(): vec3
    
    /**
    * @description Returns the Transform's scale relative to the world.
    */
    getWorldScale(): vec3
    
    /**
    * @description Sets the Transform's local transformation matrix.
    */
    setLocalTransform(transformMat: mat4): void
    
    /**
    * @description Sets the Transform's transformation matrix.
    */
    setWorldTransform(transformMat: mat4): void
    
    /**
    * @description Sets the Transform's position relative to its parent.
    */
    setLocalPosition(pos: vec3): void
    
    /**
    * @description Sets the Transform's position relative to the world.
    */
    setWorldPosition(pos: vec3): void
    
    /**
    * @description Sets the Transform's rotation relative to its parent.
    */
    setLocalRotation(rotation: quat): void
    
    /**
    * @description Sets the Transform's rotation relative to the world.
    */
    setWorldRotation(rotation: quat): void
    
    /**
    * @description Sets the Transform's scale relative to its parent.
    */
    setLocalScale(scale: vec3): void
    
    /**
    * @description Sets the Transform's scale relative to the world.
    * This may produce lossy results when parent objects are rotated, so use `setLocalScale()` instead if possible.
    */
    setWorldScale(scale: vec3): void
    
    /**
    * @description Returns the SceneObject the Transform is attached to.
    */
    getSceneObject(): SceneObject
    
    /**
    * @readonly
    
    * @description Returns the Transform's forward directional vector.
    */
    forward: vec3
    
    /**
    * @readonly
    
    * @description Returns the Transform's back directional vector.
    */
    back: vec3
    
    /**
    * @readonly
    
    * @description Returns the Transform's up directional vector.
    */
    up: vec3
    
    /**
    * @readonly
    
    * @description Returns the Transform's down directional vector.
    */
    down: vec3
    
    /**
    * @readonly
    
    * @description Returns the Transform's left directional vector.
    */
    left: vec3
    
    /**
    * @readonly
    
    * @description Returns the Transform's right directional vector.
    */
    right: vec3
    
    /**
    * @description When scaling a parent with segment scale enabled, instead of scaling child objects, it creates position offsets. This setting is used when exporting from certain 3d authoring tools.
    */
    segmentScaleCompensate: boolean
    
}

/**
* @description Applies additional transform processing on data for InputPlaceholders and OutputPlaceholders used with MLComponent.
* For more information, see the [MLComponent Overview](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/ml-component-overview).
*/
interface Transformer extends ScriptObject {
    /**
    * @readonly
    
    * @description Transformation matrix of this Transformer.
    */
    matrix: mat3
    
    /**
    * @readonly
    
    * @description Inverse transformation matrix of this Transformer.
    */
    inverseMatrix: mat3
    
}

/**
* @description Builds Transformer objects for MLComponent.
*/
interface TransformerBuilder extends ScriptObject {
    /**
    * @description Enables or disables horizontal flipping.
    */
    setFlipX(value: boolean): TransformerBuilder
    
    /**
    * @description Enables or disables vertical flipping.
    */
    setFlipY(value: boolean): TransformerBuilder
    
    /**
    * @description Sets the rotation type used.
    */
    setRotation(mode: TransformerRotation): TransformerBuilder
    
    /**
    * @description Sets the stretching type used.
    */
    setStretch(value: boolean): TransformerBuilder
    
    /**
    * @description Sets the vertical alignment type used.
    */
    setVerticalAlignment(mode: VerticalAlignment): TransformerBuilder
    
    /**
    * @description Sets the horizontal alignment type used.
    */
    setHorizontalAlignment(mode: HorizontalAlignment): TransformerBuilder
    
    /**
    * @description Sets the fill value used.
    */
    setFillColor(color: vec4): TransformerBuilder
    
    /**
    * @description Builds and returns a Transformer object based on the current settings.
    */
    build(): Transformer
    
}

/**
* @description Rotation types used by TransformBuilder.
*/
declare enum TransformerRotation {
    /**
    * @description No rotation
    */
    None,
    /**
    * @description Rotates by 90 degrees
    */
    Rotate90,
    /**
    * @description Rotates by 180 degrees
    */
    Rotate180,
    /**
    * @description Rotates by 270 degrees
    */
    Rotate270
}

/**
* @description Triangle hit information, available when a ray cast intersects a collision mesh.
*/
interface TriangleHit extends ScriptObject {
    /**
    * @readonly
    
    * @description Collision mesh containing the triangle.
    */
    mesh: CollisionMesh
    
    /**
    * @readonly
    
    * @description Index of the triangle in the mesh. Note, this is the index of the triangle in the collision mesh, which won't necessarily correspond to the same index on the render mesh (depending on type and bake settings).
    */
    index: number
    
    /**
    * @readonly
    
    * @description Vertex indices in the mesh. Note, these are the indices of the vertices in the collision mesh, which won't necessarily correspond to the same indices in the render mesh (depending on type and bake settings).
    */
    vertexIndices: number[]
    
    /**
    * @readonly
    
    * @description World-space vertex positions.
    */
    vertexPositions: vec3[]
    
    /**
    * @readonly
    
    * @description Barycentric coordinate of the hit on the triangle. This is used to interpolate values over the triangle surface. Essentially, these are weights for each of the 3 triangle vertices. For example, you can compute the hit position from vertex positions as: (V0\*b0 + V1\*b1 + V2\*b2). We already have the hit position in RayCastHit, but 'V' can be any interpolated value, such as color or texture coordinate.
    
    */
    barycentricCoordinate: vec3
    
}

/**
* @description Gets called when the user triggers the primary input on their device. For example touch on touch screens.
*/
interface TriggerPrimaryEvent extends SceneObjectEvent {
    /**
    * @readonly
    */
    position: vec2
    
}

/**
* @description Arguments used with the `InteractionComponent.onTriggerPrimary` event.
*/
interface TriggerPrimaryEventArgs extends ScriptObject {
}

/**
* @description Triggered when the lens turns off.
*/
interface TurnOffEvent extends SceneEvent {
}

/**
* @description Triggered every frame.
*/
interface UpdateEvent extends SceneEvent {
    /**
    * @description Returns the time elapsed (in seconds) between the current frame and previous frame.
    */
    getDeltaTime(): number
    
}

/**
* @description Provides information about the user such as display name, birthday, and current weather. Accessible through `global.userContextSystem`.

* All callbacks will execute as soon as the requested information is available. In some rare cases, the requested information may be completely unavailable, and the callback will never occur.

* Note that formatted or localized strings may appear differently to users depending on their region.
*/
interface UserContextSystem extends ScriptObject {
    /**
    * @description Provides the user's display name.
    */
    requestDisplayName(callback: (data: string) => void): void
    
    /**
    * @exposesUserData
    
    * @description Provides the user's birth date as a Date object.
    */
    requestBirthdate(callback: (data: Date) => void): void
    
    /**
    * @exposesUserData
    
    * @description Provides the user's birth date as localized string.
    */
    requestBirthdateFormatted(callback: (formattedData: string) => void): void
    
    /**
    * @description Provides the user's current altitude in meters.
    */
    requestAltitudeInMeters(callback: (data: number) => void): void
    
    /**
    * @description Provides the user's current altitude as a localized string.
    */
    requestAltitudeFormatted(callback: (formattedData: string) => void): void
    
    /**
    * @description Provides the user's current temperature in celsius.
    */
    requestTemperatureCelsius(callback: (data: number) => void): void
    
    /**
    * @description Provides the user's current temperature in fahrenheit.
    */
    requestTemperatureFahrenheit(callback: (data: number) => void): void
    
    /**
    * @description Provides the user's current temperature as a localized string.
    */
    requestTemperatureFormatted(callback: (formattedData: string) => void): void
    
    /**
    * @description Provides the user's current weather condition.
    */
    requestWeatherCondition(callback: (data: WeatherCondition) => void): void
    
    /**
    * @description Provides the user's current weather condition as a localized string.
    */
    requestWeatherLocalized(callback: (formattedData: string) => void): void
    
    /**
    * @exposesUserData
    
    * @description Provides the name of the city the user is currently located in.
    */
    requestCity(callback: (data: string) => void): void
    
    /**
    * @exposesUserData
    
    * @description Provides the user's username.
    */
    requestUsername(callback: (data: string) => void): void
    
    /**
    * @exposesUserData
    
    * @description Retrieve the Snapchatter's friends list in order to access details like display name, birthdate, or Bitmoji
    */
    getAllFriends(callback: (data: SnapchatUser[]) => void): void
    
    /**
    * @exposesUserData
    
    * @description Retrieve the Snapchatter's best friends in order to access details like display name, birthdate, or Bitmoji.
    */
    getBestFriends(callback: (data: SnapchatUser[]) => void): void
    
    /**
    * @exposesUserData
    
    * @description Gets the list of friends in the current context, such as 1:1 chats, 1:many chats, and group chats. 
    */
    getUsersInCurrentContext(callback: (data: SnapchatUser[]) => void): void
    
    /**
    * @exposesUserData
    
    * @description Retrieve a SnapchatUser representing the current user.
    */
    getCurrentUser(callback: (data: SnapchatUser) => void): void
    
    /**
    * @exposesUserData
    
    * @description Retrieve a SnapchatUser object for MyAI which you can use to access the MyAI Bitmoji or other details.
    */
    getMyAIUser(callback: (data: SnapchatUser) => void): void
    
}

/**
* @description A two dimensional vector.
* Vectors can only store finite numbers in the range Number.MIN_VALUE to Number.MAX_VALUE.

*/
declare class vec2 {
    /**
    * @description Creates a new instance of a vec2.
    */
    constructor(x: number, y: number)
    
    /**
    * @description Returns a string representation of the vector.
    */
    toString(): string
    
    /**
    * @description Returns the vector plus `vec`.
    */
    add(vec: vec2): vec2
    
    /**
    * @description Returns the vector minus `vec`.
    */
    sub(vec: vec2): vec2
    
    /**
    * @description Returns the component-wise multiplication product of the vector and `vec`.
    */
    mult(vec: vec2): vec2
    
    /**
    * @description Returns the component-wise multiplication product of the vector and `vec`.
    */
    scale(vec: vec2): vec2
    
    /**
    * @description Returns the division of the vector by the vector `vec`.
    */
    div(vec: vec2): vec2
    
    /**
    * @description Multiplies the components by the number `scale`.
    */
    uniformScale(scale: number): vec2
    
    /**
    * @description Returns whether this is equal to `vec`.
    */
    equal(vec: vec2): boolean
    
    /**
    * @description Returns a copy of the vector with its length clamped to `length`.
    */
    clampLength(length: number): vec2
    
    /**
    * @description Returns a copy of the vector with its length scaled to 1.
    */
    normalize(): vec2
    
    /**
    * @description Returns the angle between the vector and `vec`.
    */
    angleTo(vec: vec2): number
    
    /**
    * @description Returns the distance between the vector and the vector `vec`.
    */
    distance(vec: vec2): number
    
    /**
    * @description Like `distance()`, but returns the squared distance between vectors.
    */
    distanceSquared(vec: vec2): number
    
    /**
    * @description Returns the dot product of the vector and `vec`.
    */
    dot(vec: vec2): number
    
    /**
    * @description Returns a copy of the vector moved towards the point `point` by the amount `magnitude`.
    */
    moveTowards(point: vec2, magnitude: number): vec2
    
    /**
    * @description Returns a copy of the vector projected onto the vector `vec`.
    */
    project(vec: vec2): vec2
    
    /**
    * @description Projects the vector onto the plane represented by the normal `normal`.
    */
    projectOnPlane(normal: vec2): vec2
    
    /**
    * @description Returns a copy of the vector reflected across the plane defined by the normal `vec`.
    */
    reflect(vec: vec2): vec2
    
    /**
    * @description x component of the vec2.
    */
    x: number
    
    /**
    * @description y component of the vec2.
    */
    y: number
    
    /**
    * @description Alternate name for the x component.
    */
    r: number
    
    /**
    * @description Alternate name for the y component.
    */
    g: number
    
    /**
    * @description Returns the length of the vector.
    */
    length: number
    
    /**
    * @description Returns the squared length of the vector.
    */
    lengthSquared: number
    
}
declare namespace vec2 {
    /**
    * @description Returns a new vector containing the largest value of each component in the two vectors.
    */
    export function max(vecA: vec2, vecB: vec2): vec2
    
    /**
    * @description Returns a new vector containing the smallest value of each component in the two vectors.
    */
    export function min(vecA: vec2, vecB: vec2): vec2
    
    /**
    * @description Linearly interpolates between the two vectors `vecA` and `vecB` by the factor `t`.
    */
    export function lerp(vecA: vec2, vecB: vec2, t: number): vec2
    
    /**
    * @description Returns the vector (1, 1).
    */
    export function one(): vec2
    
    /**
    * @description Returns the vector (0, 0).
    */
    export function zero(): vec2
    
    /**
    * @description Returns the vector (0, 1).
    */
    export function up(): vec2
    
    /**
    * @description Returns the vector (0, -1).
    */
    export function down(): vec2
    
    /**
    * @description Returns the vector (-1, 0).
    */
    export function left(): vec2
    
    /**
    * @description Returns the vector (1, 0).
    */
    export function right(): vec2
    
    /**
    * @description Generate a random 2D direction vector. This is equivalent to a random point on a unit-radius circle.
    */
    export function randomDirection(): vec2
    

}

/**
* @description Represents an animation track using vec2 value keyframes.
*/
interface Vec2AnimationTrack extends AnimationTrack {
}

/**
* @description Represents an animation track using vec2 value keyframes.
*/
interface Vec2AnimationTrackKeyFramed extends Vec2AnimationTrack {
    /**
    * @description Removes all keyframes.
    */
    removeAllKeys(): void
    
    /**
    * @description Removes the keyframe at `index`.
    */
    removeKeyAt(index: number): void
    
    /**
    * @description Adds a keyframe value `value` at time `time`.
    */
    addKey(time: number, value: vec2): void
    
}

/**
* @description A three dimensional vector. 
* Vectors can only store finite numbers in the range Number.MIN_VALUE to Number.MAX_VALUE.

*/
declare class vec3 {
    /**
    * @description Creates a new instance of a vec3.
    */
    constructor(x: number, y: number, z: number)
    
    /**
    * @description Returns a string representation of the vector.
    */
    toString(): string
    
    /**
    * @description Returns the cross product of the vector and `vec`
    */
    cross(vec: vec3): vec3
    
    /**
    * @description Returns a copy of the vector rotated towards the `target` vector by `step` radians.
    
    * The vectors may be non-normalized. The function always returns a vector with the source vector's magnitude.
    * This prevents overshoot. If `step` exceeds the angle between vectors, it stops at the `target` direction.
    * If `step` is negative, this rotates the source vector away from `target`. It stops when the direction is precisely opposite to `target`.
    * If the vectors are in opposite directions, the result is rotated along an arbitrary (but consistent) axis.
    * If either vector is zero magnitude, it returns the source vector.
    
    */
    rotateTowards(target: vec3, step: number): vec3
    
    /**
    * @description Returns the vector plus `vec`.
    */
    add(vec: vec3): vec3
    
    /**
    * @description Returns the vector minus `vec`.
    */
    sub(vec: vec3): vec3
    
    /**
    * @description Returns the component-wise multiplication product of the vector and `vec`.
    */
    mult(vec: vec3): vec3
    
    /**
    * @description Returns the component-wise multiplication product of the vector and `vec`.
    */
    scale(vec: vec3): vec3
    
    /**
    * @description Returns the division of the vector by the vector `vec`.
    */
    div(vec: vec3): vec3
    
    /**
    * @description Multiplies the components by the number `scale`.
    */
    uniformScale(scale: number): vec3
    
    /**
    * @description Returns whether this is equal to `vec`.
    */
    equal(vec: vec3): boolean
    
    /**
    * @description Returns a copy of the vector with its length clamped to `length`.
    */
    clampLength(length: number): vec3
    
    /**
    * @description Returns a copy of the vector with its length scaled to 1.
    */
    normalize(): vec3
    
    /**
    * @description Returns the angle in radians between the vector and `vec`.
    */
    angleTo(vec: vec3): number
    
    /**
    * @description Returns the distance between the vector and the vector `vec`.
    */
    distance(vec: vec3): number
    
    /**
    * @description Like `distance()`, but returns the squared distance between vectors.
    */
    distanceSquared(vec: vec3): number
    
    /**
    * @description Returns the dot product of the vector and `vec`.
    */
    dot(vec: vec3): number
    
    /**
    * @description Returns a copy of the vector moved towards the point `point` by the amount `magnitude`.
    */
    moveTowards(point: vec3, magnitude: number): vec3
    
    /**
    * @description Returns a copy of the vector projected onto the vector `vec`.
    */
    project(vec: vec3): vec3
    
    /**
    * @description Projects the vector onto the plane represented by the normal `normal`.
    */
    projectOnPlane(normal: vec3): vec3
    
    /**
    * @description Returns a copy of the vector reflected across the plane defined by the normal `vec`.
    */
    reflect(vec: vec3): vec3
    
    /**
    * @description x component of the vec3.
    */
    x: number
    
    /**
    * @description y component of the vec3.
    */
    y: number
    
    /**
    * @description z component of the vec3.
    */
    z: number
    
    /**
    * @description Alternate name for the x component.
    */
    r: number
    
    /**
    * @description Alternate name for the y component.
    */
    g: number
    
    /**
    * @description Alternate name for the z component.
    */
    b: number
    
    /**
    * @description Returns the length of the vector.
    */
    length: number
    
    /**
    * @description Returns the squared length of the vector.
    */
    lengthSquared: number
    
}
declare namespace vec3 {
    /**
    * @description Makes the vectors `vecA` and `vecB` normalized and orthogonal to each other.
    */
    export function orthonormalize(vecA: vec3, vecB: vec3): void
    
    /**
    * @description Returns a new vector containing the largest value of each component in the two vectors.
    */
    export function max(vecA: vec3, vecB: vec3): vec3
    
    /**
    * @description Returns a new vector containing the smallest value of each component in the two vectors.
    */
    export function min(vecA: vec3, vecB: vec3): vec3
    
    /**
    * @description Linearly interpolates between the two vectors `vecA` and `vecB` by the factor `t`.
    */
    export function lerp(vecA: vec3, vecB: vec3, t: number): vec3
    
    /**
    * @description Spherically interpolates between the two vectors `vecA` and `vecB` by the factor `t`.
    */
    export function slerp(vecA: vec3, vecB: vec3, t: number): vec3
    
    /**
    * @description Returns the vector (1, 1, 1).
    */
    export function one(): vec3
    
    /**
    * @description Returns the vector (0, 0, 0).
    */
    export function zero(): vec3
    
    /**
    * @description Returns the vector (0, 1, 0).
    */
    export function up(): vec3
    
    /**
    * @description Returns the vector (0, -1, 0).
    */
    export function down(): vec3
    
    /**
    * @description Returns the vector (-1, 0, 0).
    */
    export function left(): vec3
    
    /**
    * @description Returns the vector (1, 0, 0).
    */
    export function right(): vec3
    
    /**
    * @description Returns the vector (0, 0, -1).
    */
    export function back(): vec3
    
    /**
    * @description Returns the vector (0, 0, 1).
    */
    export function forward(): vec3
    
    /**
    * @description Generate random 3D direction vector. This is equivalent to a random point on a unit-radius sphere.
    */
    export function randomDirection(): vec3
    

}

/**
* @description Represents an animation track using vec3 value keyframes.
*/
interface Vec3AnimationTrack extends AnimationTrack {
}

/**
* @description Represents an animation track using vec3 value keyframes.
*/
interface Vec3AnimationTrackKeyFramed extends Vec3AnimationTrack {
    /**
    * @description Removes all keyframes.
    */
    removeAllKeys(): void
    
    /**
    * @description Removes the keyframe at `index`.
    */
    removeKeyAt(index: number): void
    
    /**
    * @description Adds a keyframe value `value` at time `time`.
    */
    addKey(time: number, value: vec3): void
    
}

/**
* @description Represents an animation track using vec3 animation tracks.
*/
interface Vec3AnimationTrackXYZ extends Vec3AnimationTrack {
    /**
    * @description Returns the child track at index `index`
    */
    getChildTrackByIndex(index: number): AnimationTrack
    
    /**
    * @description Sets the child track at index `index` to `track`
    */
    setChildTrackByIndex(index: number, track: AnimationTrack): void
    
}

/**
* @description A four dimensional vector.
* Vectors can only store finite numbers in the range Number.MIN_VALUE to Number.MAX_VALUE.

*/
declare class vec4 {
    /**
    * @description Creates a new instance of a vec4.
    */
    constructor(x: number, y: number, z: number, w: number)
    
    /**
    * @description Returns a string representation of the vector.
    */
    toString(): string
    
    /**
    * @description Returns the vector plus `vec`.
    */
    add(vec: vec4): vec4
    
    /**
    * @description Returns the vector minus `vec`.
    */
    sub(vec: vec4): vec4
    
    /**
    * @description Returns the component-wise multiplication product of the vector and `vec`.
    */
    mult(vec: vec4): vec4
    
    /**
    * @description Returns the component-wise multiplication product of the vector and `vec`.
    */
    scale(vec: vec4): vec4
    
    /**
    * @description Returns the division of the vector by the vector `vec`.
    */
    div(vec: vec4): vec4
    
    /**
    * @description Multiplies the components by the number `scale`.
    */
    uniformScale(scale: number): vec4
    
    /**
    * @description Returns whether this is equal to `vec`.
    */
    equal(vec: vec4): boolean
    
    /**
    * @description Returns a copy of the vector with its length clamped to `length`.
    */
    clampLength(length: number): vec4
    
    /**
    * @description Returns a copy of the vector with its length scaled to 1.
    */
    normalize(): vec4
    
    /**
    * @description Returns the angle between the vector and `vec`.
    */
    angleTo(vec: vec4): number
    
    /**
    * @description Returns the distance between the vector and the vector `vec`.
    */
    distance(vec: vec4): number
    
    /**
    * @description Like `distance()`, but returns the squared distance between vectors.
    */
    distanceSquared(vec: vec4): number
    
    /**
    * @description Returns the dot product of the vector and `vec`.
    */
    dot(vec: vec4): number
    
    /**
    * @description Returns a copy of the vector moved towards the point `point` by the amount `magnitude`.
    */
    moveTowards(point: vec4, magnitude: number): vec4
    
    /**
    * @description Returns a copy of the vector projected onto the vector `vec`.
    */
    project(vec: vec4): vec4
    
    /**
    * @description Projects the vector onto the plane represented by the normal `normal`.
    */
    projectOnPlane(normal: vec4): vec4
    
    /**
    * @description Returns a copy of the vector reflected across the plane defined by the normal `vec`.
    */
    reflect(vec: vec4): vec4
    
    /**
    * @description x component of the vec4.
    */
    x: number
    
    /**
    * @description y component of the vec4.
    */
    y: number
    
    /**
    * @description z component of the vec4.
    */
    z: number
    
    /**
    * @description w component of the vec4.
    */
    w: number
    
    /**
    * @description Alternate name for the x component.
    */
    r: number
    
    /**
    * @description Alternate name for the y component.
    */
    g: number
    
    /**
    * @description Alternate name for the z component.
    */
    b: number
    
    /**
    * @description Alternate name for the w component.
    */
    a: number
    
    /**
    * @description Returns the length of the vector.
    */
    length: number
    
    /**
    * @description Returns the squared length of the vector.
    */
    lengthSquared: number
    
}
declare namespace vec4 {
    /**
    * @description Returns a new vector containing the largest value of each component in the two vectors.
    */
    export function max(vecA: vec4, vecB: vec4): vec4
    
    /**
    * @description Returns a new vector containing the smallest value of each component in the two vectors.
    */
    export function min(vecA: vec4, vecB: vec4): vec4
    
    /**
    * @description Linearly interpolates between the two vectors `vecA` and `vecB` by the factor `t`.
    */
    export function lerp(vecA: vec4, vecB: vec4, t: number): vec4
    
    /**
    * @description Returns the vector (1, 1, 1, 1).
    */
    export function one(): vec4
    
    /**
    * @description Returns the vector (0, 0, 0, 0).
    */
    export function zero(): vec4
    

}

/**
* @description Represents an animation track using vec4 value keyframes.
*/
interface Vec4AnimationTrack extends AnimationTrack {
}

/**
* @description Represents an animation track using vec4 value keyframes.
*/
interface Vec4AnimationTrackKeyFramed extends Vec4AnimationTrack {
    /**
    * @description Removes all keys.
    */
    removeAllKeys(): void
    
    /**
    * @description Removes key at index `index`.
    */
    removeKeyAt(index: number): void
    
    /**
    * @description Adds a key with value `value` at time `time`.
    */
    addKey(time: number, value: vec4): void
    
}

/**
* @description A vector containing 4 boolean values.
*/
declare class vec4b {
    /**
    * @description Creates a new instance of a vec4b.
    */
    constructor(x: boolean, y: boolean, z: boolean, w: boolean)
    
    /**
    * @description Returns a string representation of the vector.
    */
    toString(): string
    
    /**
    * @description x component of the vec4b.
    */
    x: boolean
    
    /**
    * @description y component of the vec4b.
    */
    y: boolean
    
    /**
    * @description z component of the vec4b.
    */
    z: boolean
    
    /**
    * @description w component of the vec4b.
    */
    w: boolean
    
    /**
    * @description Alternate name for the x component.
    */
    r: boolean
    
    /**
    * @description Alternate name for the y component.
    */
    g: boolean
    
    /**
    * @description Alternate name for the z component.
    */
    b: boolean
    
    /**
    * @description Alternate name for the w component.
    */
    a: boolean
    
}

/**
* @description Used to help control vertex animations on the SceneObject.
*/
interface VertexCache extends Component {
    /**
    * @description The current time of vertex animations on this SceneObject.
    */
    currentTime: number
    
    /**
    * @description The weight applied to vertex animations on this SceneObject.
    */
    weight: number
    
}

/**
* @description Provides settings for vertex physics in the ClothVisual component.
*/
interface VertexSimulationSettings extends ScriptObject {
    /**
    * @description Mass value.
    */
    mass: number
    
    /**
    * @description Mass value weight on Global Simulation Settings for this vertex.
    */
    massGlobalWeight: number
    
    /**
    * @description Stretch Stiffness value.
    */
    stretchStiffness: number
    
    /**
    * @description StretchStiffness value weight on Global Simulation Settings for this vertex.
    */
    stretchStiffnessGlobalWeight: number
    
    /**
    * @description Bend Stiffness value.
    */
    bendStiffness: number
    
    /**
    * @description BendStiffness value weight on Global Simulation Settings for this vertex.
    */
    bendStiffnessGlobalWeight: number
    
    /**
    * @description Friction value.
    */
    friction: number
    
    /**
    * @description Friction value weight on Global Simulation Settings for this vertex.
    */
    frictionGlobalWeight: number
    
}

/**
* @description Used by the `verticalAlignment` property in [MeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MeshVisual).
* When a [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) is attached to the same SceneObject, this determines how the mesh will be positioned vertically.
*/
declare enum VerticalAlignment {
    /**
    * @description The mesh will be aligned to the bottom side.
    */
    Bottom,
    /**
    * @description The mesh will be centered.
    */
    Center,
    /**
    * @description The mesh will be aligned to the top side.
    */
    Top
}

/**
* @description Options for handling vertical text overflow. Used by [Text component's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) `verticalOverflow` property.
*/
declare enum VerticalOverflow {
    /**
    * @description Text will continue to draw past the end of the vertical boundaries.
    */
    Overflow,
    /**
    * @description Text will be clipped at the end of the vertical boundaries.
    */
    Truncate,
    /**
    * @description Text will shrink to fit within the vertical boundaries.
    */
    Shrink
}

/**
* @description Defines a VFX to use with [VFX Component](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#VFXComponent). For more information, see the [VFX Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/graphics/particles/vfx-editor/introduction-and-concepts).
*/
interface VFXAsset extends Asset {
    /**
    * @description Clones the VFX asset and returns a copy. 
    */
    clone(): VFXAsset
    
    /**
    * @readonly
    
    * @description Controls properties for the VFXAsset. Any scriptable properties on a VFX Graph will automatically become properties of this Properties class. For example, if the VFX Graph defines a variable named `baseColor`, a script would be able to access that property as `vfxAsset.properties.baseColor`.
    */
    properties: Properties
    
    /**
    * @readonly
    
    * @description Returns a PassWrapper containing all simulation passes in a VFX asset
    */
    simulations: PassWrappers
    
    /**
    * @readonly
    
    * @description Returns a PassWrapper containing all feedback passes in a VFX asset
    */
    feedbacks: PassWrappers
    
    /**
    * @readonly
    
    * @description Returns a PassWrapper containing all output passes in a VFX asset
    */
    outputs: PassWrappers
    
    /**
    * @description When `Mesh` is selected as a Geometry Type in the VFX Output Container, the system will render particles using this mesh, otherwise particles will be rendered as quads. Refer to the Custom Mesh Emitter built-in asset as a starting point when working with custom meshes.
    */
    mesh: RenderMesh
    
}

/**
* @description A VFX visual used to show a [VFX Asset](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#VFXAsset).
*/
interface VFXComponent extends BaseMeshVisual {
    /**
    * @description Restart the VFX system.
    */
    restart(): void
    
    /**
    * @description Clear the VFX system.
    */
    clear(): void
    
    /**
    * @description The [VFX Asset](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#VFXAsset) that describes the VFX simulation.
    */
    asset: VFXAsset
    
}

/**
* @description Describes the current status of a [VideoTextureProvider](https://lensstudio.snapchat.com/api/lens-studio/Classes/Providers#VideoTextureProvider).
*/
declare enum VideoStatus {
    /**
    * @description The video playback has stopped
    */
    Stopped,
    /**
    * @description The video is being prepared
    */
    Preparing,
    /**
    * @description The video is playing
    */
    Playing,
    /**
    * @description The video playback is paused
    */
    Paused
}

/**
* @description Controls a video texture resource. Can be accessed through [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Texture#control-textureprovider).
*/
interface VideoTextureProvider extends TextureProvider {
    /**
    * @description Plays the video `playCount` times. If `playCount` is less than 0, it loops infinitely.
    */
    play(playCount: number): void
    
    /**
    * @description Pauses the video playback.
    */
    pause(): void
    
    /**
    * @description Resumes the video playback.
    */
    resume(): void
    
    /**
    * @description Stops the video playback.
    */
    stop(): void
    
    /**
    * @description Sets the current playback time to the specified time in seconds.
    */
    seek(value: number): boolean
    
    /**
    * @description The playback rate of the video. The rate is set when starting playback, meaning that updates to this property does not take effect until the video is stopped and started again. Defaults to 1.
    
    */
    playbackRate: number
    
    /**
    * @description The audio volume of the video resource, normalized from 0 to 1.
    */
    volume: number
    
    /**
    * @description Sets or returns playback start time in unit range.
    */
    relativeStartTime: number
    
    /**
    * @description Sets or returns playback end time in unit range.
    */
    relativeEndTime: number
    
    /**
    * @readonly
    
    * @description Returns the number of played cycles.
    */
    currentPlayCount: number
    
    /**
    * @readonly
    
    * @description Returns true if video file has been loaded and is ready for decoding and false otherwise.
    */
    isPlaybackReady: boolean
    
    /**
    * @readonly
    
    * @description Returns the duration of playback range in seconds, or zero if accessed while playback is in an unprepared state.
    */
    duration: number
    
    /**
    * @readonly
    
    * @description Returns the duration of loaded video file in seconds, or zero if accessed during playback being in unprepared state.
    */
    totalDuration: number
    
    /**
    * @readonly
    
    * @description Returns the current time in seconds, or zero if accessed during playback being in unprepared state.
    */
    currentTime: number
    
    /**
    * @readonly
    
    * @description Returns the time of the last acquired texture in seconds, or zero if accessed during playback being in unprepared state.
    */
    lastFrameTime: number
    
    /**
    * @readonly
    
    * @description A read-only property that returns the status of provider. Suggested as a substitution for the existing getStatus()
    */
    status: VideoStatus
    
    /**
    * @readonly
    
    * @description The event for being reported about playback start. When this event is triggered, lens developers can set video texture to material slots and see actual video frames.
    */
    onPlaybackReady: event0<void>
    
    /**
    * @readonly
    
    * @description The event for being reported about playback finished. When this event is triggered, lens developers can evict this texture from material slots to avoid disrupting user's experience.
    */
    onPlaybackDone: event0<void>
    
}

/**
* @description Base class for all visual Components (e.g. MeshVisual).
*/
interface Visual extends Component {
    /**
    * @description Returns the order of this Visual in the render queue.
    */
    getRenderOrder(): number
    
    /**
    * @description Sets the order of this Visual in the render queue.
    */
    setRenderOrder(value: number): void
    
}

interface VoiceML {
}

declare namespace VoiceML {
    /**
    * @description Additional parameters are used to provide additional data for NlpModels and NlpResponses. 
    */
    interface AdditionalParam extends ScriptObject {
        /**
        * @readonly
        
        * @description The key of the additional parameter.
        */
        key: string
        
        /**
        * @readonly
        
        * @description The value of the additional parameter.
        */
        value: string
        
    }

}

declare namespace VoiceML {
    /**
    * @description NlpModels are used to provide the VoiceML NLP engine information about how the transcript of the input audio should be processed. BaseNlpModels is the abstract base class all NlpModels ([NlpKeywordModel](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLNlpKeywordModel), [NlpIntentModel](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLNlpIntentModel)) inherit from.
    
    * You can specify multiple NlpModels to process the same audio, all of their results will be returned in [ListeningUpdateEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLListeningUpdateEventArgs)
    */
    interface BaseNlpModel extends ScriptObject {
        /**
        * @description Adds additional params to set in the ML engine.
        */
        addParam(key: string, value: string): void
        
    }

}

declare namespace VoiceML {
    /**
    * @description The abstract base class all NlpResponses inherit from. NlpResponses are used as the result from the VoiceML NLP engine with information after processing the transcript of the input audio.
    */
    interface BaseNlpResponse extends ScriptObject {
        /**
        * @readonly
        
        * @description Name of the model that sent the response.  In current studio release, you can expect the result to contain either  `VOICE_ENABLED_UI` like described in [NlpModel](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLNlpIntentModel).  If using `enableSystemCommands()`, `SYSTEM_COMMANDS` will be returned as well. 
        */
        modelName: string
        
        /**
        * @readonly
        
        * @description Additional params to send in the response.
        */
        additionalParams: VoiceML.AdditionalParam[]
        
        /**
        * @readonly
        
        * @description The status of the NLP response. in case of successful result, the value should be `VoiceMLModule.NlpResponsesStatusCodes.OK`, in case of failure, the value should be `VoiceMLModule.NlpResponsesStatusCodes.ERROR`. 
        */
        status: VoiceML.NlpResponseStatus
        
    }

}

declare namespace VoiceML {
    /**
    * @description An NLP model used to detect keywords in the transcript of the input audio.  For example, you can have keyword detection which will trigger every time the word "red" is said, and another trigger for the word "yellow". 
    */
    interface KeywordModelGroup extends ScriptObject {
        /**
        * @readonly
        
        * @description Name of the keyword group. 
        */
        name: string
        
        /**
        * @readonly
        
        * @description Array of keywords that are being sent to the ML model for detection with similar meaning. For example, putting the word "yellow" would trigger a response in the sentence "I saw a yellow bird". The keyword detection model supports slight variation in the word transcription. We recommend adding possible synonyms of the same keyword. So for example, should you want to detect "movie", you might want to consider adding "video" and possibly "feature" and "flick" as well to the list resulting in `["movie", "video", "feature","flick"]` as different people might express themselves differently.  
        */
        keywords: string[]
        
    }

}

declare namespace VoiceML {
    /**
    * @description ListeningErrorEventArgs object returns in onListeningError callback. It contains the error code and description of the error. 
    */
    interface ListeningErrorEventArgs extends ScriptObject {
        /**
        * @readonly
        
        * @description Returns the Error code number. 
        */
        error: number
        
        /**
        * @readonly
        
        * @description Returns the description of the error. 
        */
        description: string
        
    }

}

declare namespace VoiceML {
    /**
    * @description Provides the configuration for the audio input processing output. This can either include NLP processing using the ([nlpModels](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLBaseNlpModel)) or directly retrieving the transcription.  `speechContext` provides the ability to further improve the transcription accuracy given an assumed context. 
    */
    interface ListeningOptions extends ScriptObject {
        /**
        * @description In cases where specific words are expected from the users, the transcription accuracy of these words can be improved, by strengthening their likelihood in context. The strength is scaled `1-10` (10 being the strongest increase) the default value is `5`. 
        */
        addSpeechContext(phrases: string[], boost: number): void
        
        /**
        * @description Should complete transcription returned. Such transcriptions after the user stopped speaking. This transcription is marked with `isFinalTranscription=true` in the `OnListeningUpdate`. 
        */
        shouldReturnAsrTranscription: boolean
        
        /**
        * @description Should interim transcription returned. Such transcriptions are returned while the user still speaks, however they may be less accurate, and can be changed on following transcriptions. This interim results are marked with `isFinalTranscription=false` in the `OnListeningUpdate`. 
        */
        shouldReturnInterimAsrTranscription: boolean
        
        /**
        * @description Options for the ML model to be used. 
        */
        nlpModels: VoiceML.BaseNlpModel[]
        
        /**
        * @description An optional attribute to specify which speech recognizer ML model to use when transcribing.  When creating a new `ListeningOptions` the value of this attrbute is defaulted to `SPEECH_RECOGNIZER`.   The supported values are: `SPEECH_RECOGNIZER`. 
        */
        speechRecognizer: string
        
        /**
        * @description The language which VoiceML should listen to.
        */
        languageCode: string
        
        /**
        * @description Supports multiple speech contexts for increased transcription accuracy. 
        */
        speechContexts: VoiceML.SpeechContext[]
        
        /**
        * @description An array of VoiceML.QnaAction elements. It is used to pass the context in each QnaAction to the DialogML.
        */
        postProcessingActions: VoiceML.PostProcessingAction[]
        
    }

}
declare namespace VoiceML {
    namespace ListeningOptions {
        /**
        * @description Creates voice command options. 
        */
        export function create(): VoiceML.ListeningOptions
        
    
    }

}

declare namespace VoiceML {
    /**
    * @description The parameter when the callback registered on from `VoiceMLModule.onListeningUpdate` is called. This is to mark the input audio transcription (and possibly NlpModels as a result) was updated.
    */
    interface ListeningUpdateEventArgs extends ScriptObject {
        /**
        * @description In case a keyword detection model used, returns results from the keywords model.
        */
        getKeywordResponses(): VoiceML.NlpKeywordResponse[]
        
        /**
        * @description In case an NLP intent model used, returns intent response from the ML engine. 
        */
        getIntentResponses(): VoiceML.NlpIntentResponse[]
        
        /**
        * @description In case `enableSystemCommands()` function used, returns the command executed by the system. 
        */
        getCommandResponses(): VoiceML.NlpCommandResponse[]
        
        /**
        * @description Returns an array of QnaResponse elements, each representing an answer to the question in the eventArgs.transcript. It has properties answer(a string with the ML's response) and answerStatusCode, which can take values 
        */
        getQnaResponses(): VoiceML.QnaResponse[]
        
        /**
        * @readonly
        
        * @description Returns the transcription from the Automatic Speech Recognition. 
        */
        transcription: string
        
        /**
        * @readonly
        
        * @description Specifies whether the transcription returned is final, or partial (interim) which can be updated later as the sentence continues.
        */
        isFinalTranscription: boolean
        
    }

}

declare namespace VoiceML {
    /**
    * @description May be returned when [enableSystemCommands()](https://lensstudio.snapchat.com/api/classes/VoiceMLModule#enableSystemCommands) API is used.
    * The NlpCommandResponse contains the results of the servers' NLP command model classification on the last sentence and indicates that a predetermined system command was detected.
    * Current supported commands:
    
    * “Take a Snap”: takes a snapshot.
    * “Start Recording”: starts a video recording if not already started.
    * "Stop Recording”: stops an ongoing video recording, if applicable.
    
    */
    interface NlpCommandResponse extends VoiceML.BaseNlpResponse {
        /**
        * @readonly
        
        * @description The classification result of the NLP command model for the sentence. For example, for the sentence "Take a snap!" with the [enableSystemCommands()](https://lensstudio.snapchat.com/api/classes/VoiceMLModule#enableSystemCommands) API enabled, the result will be a [NlpCommandResponse](https://lensstudio.snapchat.com/api/classes/VoiceMLModule/NlpCommandResponse) holding a `take a snap` [Command](https://lensstudio.snapchat.com/api/classes/VoiceMLModule/NlpCommandRespons#command). In addition, the following values could be returned: 
        
        * `#SNAP_ERROR_INCONCLUSIVE`: two or more commands detected.
        * `#SNAP_ERROR_INDECISIVE`: no command detected.
        * `#SNAP_ERROR_NONVERBAL`: no transcribable human voice was detected.
        * `#SNAP_ERROR_SILENCE`: silence was detected. 
        
        * Anything starting with `#SNAP_ERROR_`: Errors that are not currently defined in this documentation and should be ignored.
        
        */
        command: string
        
    }

}

declare namespace VoiceML {
    /**
    * @description Specifies which NLP Intent model should run to classify the transcription of the input audio. NLP Intent classification meant to extract the meaning of a sentence rather than detecting certain keywords. Multiple intent models can be used on the same transcription, and will run only on complete sentences (`isFinalTranscription = True`). Supported intent models: `VOICE_ENABLED_UI`. 
    */
    interface NlpIntentModel extends VoiceML.BaseNlpModel {
        /**
        * @description A list of the intents the `Intent ML` engine should use when classifying the sentence, for example `["back", "next"]`, in case no list provided, all possible intents of the model are used.  
        
        * Supported intents for `VOICE_ENABLED_UI`: `next`, `back`, `left`, `right`, `up`, `down`, `first`, `second`, `third`, `fourth`, `fifth`, `sixth`, `seventh`, `eighth`, `ninth`, `tenth`. 
        */
        possibleIntents: string[]
        
    }

}

declare namespace VoiceML {
    /**
    * @description Returned when [NlpIntentModel](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLNlpIntentModel) was specificed in the ListeningOptions, it contains the results of the NLP Intent model classification on the last sentence. `NlpIntentResponse` will only run on complete sentences (`isFinalTranscription = true`).  
    
    */
    interface NlpIntentResponse extends VoiceML.BaseNlpResponse {
        /**
        * @readonly
        
        * @description The classification result of the NLP intent model for the sentence. For example for the sentence "show me the next item" with the model `VOICE_ENABLED_UI`, the result will be `next` for `VOICE_ENABLED_UI` the following intents are valid: `next`, `back`, `left`, `right`, `up`, `down`, `first`, `second`, `third`, `fourth`, `fifth`, `sixth`, `seventh`, `eighth`, `ninth`, `tenth`. In addition, the following values could be returned: 
        
        * `#SNAP_ERROR_INCONCLUSIVE`: two or more intents detected.
        * `#SNAP_ERROR_INDECISIVE`: no intent detected.
        * `#SNAP_ERROR_NONVERBAL`: no transcribable human voice was detected.
        * `#SNAP_ERROR_SILENCE`: silence was detected. 
        
        * Anything starting with `#SNAP_ERROR_`: Errors that are not currently defined in this documentation and should be ignored.
        */
        intent: string
        
    }

}

declare namespace VoiceML {
    /**
    * @description Contains helper functions for NlpIntentModel.
    */
    interface NlpIntentsModelOptions {
    }

}
declare namespace VoiceML {
    namespace NlpIntentsModelOptions {
        /**
        * @description Creates a new NlpIntentModel.
        */
        export function create(intentModelName: string): VoiceML.NlpIntentModel
        
    
    }

}

declare namespace VoiceML {
    /**
    * @description Used to detect usage of certain keywords from the input audio.
    
    * Keyword detection (whose results will be returned in [NlpKeywordResponse](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLNlpKeywordResponse) in the [ListeningUpdateEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLListeningUpdateEventArgs)) can happen in the mid input sentence (and in such case the the isFinalTranscription=false) or can happen at the end of the sentence (isFinalTranscription=true). Mid sentence detection have closer proximity to the time the word was spoken, but might be less accurate. 
    */
    interface NlpKeywordModel extends VoiceML.BaseNlpModel {
        /**
        * @description Adds a keyword group to the keywords model, the group name will be returned in NlpKeywordResponse to indicate one of the keywords in the group has been detected. Groups are used to include synonyms, or context similar words.  `addKeywordGroup("walk", ["walk", "jog", "strolls"])`. 
        */
        addKeywordGroup(name: string, keywords: string[]): void
        
        /**
        * @readonly
        
        * @description Holds group of keywords to be used in the ML model. 
        */
        keywordGroups: VoiceML.KeywordModelGroup[]
        
    }

}

declare namespace VoiceML {
    /**
    * @description Contains helper functions for NlpKeywordModel.
    */
    interface NlpKeywordModelOptions {
    }

}
declare namespace VoiceML {
    namespace NlpKeywordModelOptions {
        /**
        * @description Creates a new NlpKeywordModel.
        */
        export function create(): VoiceML.NlpKeywordModel
        
    
    }

}

declare namespace VoiceML {
    /**
    * @description NlpKeywordResponse will be returned if KeywordModel has been supplied as an input model in the ListeningOptions.  The keyword model allows detection of keywords (or short phrases) in an input audio. 
    */
    interface NlpKeywordResponse extends VoiceML.BaseNlpResponse {
        /**
        * @readonly
        
        * @description The keywords detected by the NLP keyword detection model.  In addition, the following values could be returned: 
        
        * `#SNAP_ERROR_INCONCLUSIVE`: two or more intents detected.
        * `#SNAP_ERROR_INDECISIVE`: no intent detected.
        * `#SNAP_ERROR_NONVERBAL`: no transcribable human voice was detected.
        * `#SNAP_ERROR_SILENCE`: silence was detected.
        
        * Anything starting with `#SNAP_ERROR_`: Errors that are not currently defined in this documentation and should be ignored
        */
        keywords: string[]
        
    }

}

declare namespace VoiceML {
    /**
    * @description The NLP Response Status indicates wether the NLP was successful in parsing the sentence. 
    */
    interface NlpResponseStatus extends ScriptObject {
        /**
        * @readonly
        
        * @description Nlp status code coming from the ML engine. 
        */
        code: number
        
        /**
        * @readonly
        
        * @description Nlp status description coming from the ML engine. 
        */
        description: string
        
    }

}

declare namespace VoiceML {
    /**
    * @description The `PostProcessingAction` is the base class for `QnaAction` and other post processing actions that need to processed after the transcription phase.
    */
    interface PostProcessingAction extends ScriptObject {
    }

}

declare namespace VoiceML {
    /**
    * @description The `PostProcessingActionResponse` is the base class for `QnaActionRespose` and other post processing actions responses. It holds the id and status properties for all deriving objects.
    */
    interface PostProcessingActionResponse extends ScriptObject {
        /**
        * @readonly
        
        * @description The id of the `PostProcessingAction` so returning results can be tied to post processing actions
        */
        id: number
        
        /**
        * @readonly
        
        * @description The status holds the description and code for the post processing actions. In case of success the description will be empty and the code will be 0
        */
        status: VoiceML.PostProcessingActionResponseStatus
        
    }

}

declare namespace VoiceML {
    interface PostProcessingActionResponseStatus extends ScriptObject {
        /**
        * @readonly
        */
        code: number
        
        /**
        * @readonly
        */
        description: string
        
    }

}

declare namespace VoiceML {
    interface QnaAction extends VoiceML.PostProcessingAction {
        /**
        * @description The context passed to the QnaAction. The QnaAction is passed to VoiceML.listeningOptions to use the DialogML within the VoiceML automatically
        */
        context: string
        
    }

}
declare namespace VoiceML {
    namespace QnaAction {
        /**
        * @description Parameter: string context. Creates a QnaAction object with a given context i.e. source text for the Dialog ML. 
        */
        export function create(context: string): VoiceML.QnaAction
        
    
    }

}

declare namespace VoiceML {
    interface QnaResponse extends VoiceML.PostProcessingActionResponse {
        /**
        * @readonly
        
        * @description The response string the module came up with in response to the eventArgs.transcript
        */
        answer: string
        
        /**
        * @readonly
        
        * @description The status of the QnaResponse
        */
        answerStatusCode: number
        
    }

}

declare namespace VoiceML {
    /**
    * @description Speech context is used in cases where specific words are expected from the users, the transcription accuracy of these words can be improved, by straightening their likelihood in context. The strength is scaled 1-10 (10 being the strongest increase) the default value is 5.
    */
    interface SpeechContext extends ScriptObject {
        /**
        * @readonly
        
        * @description Array of keyword to enhance by the ML engine. 
        */
        phrases: string[]
        
        /**
        * @readonly
        
        * @description The strength is scaled 1-10 (10 being the strongest increase) the default value is 5.
        */
        boost: number
        
    }

}

/**
* @description VoiceML Module allows voice input and commands. It enables transciption of the speech, detecting keywords within the transcription, intents as well as system commands (such as "Take a Snap"). You can use one VoiceML Module per Lens. 
*/
interface VoiceMLModule extends Asset {
    /**
    * @exposesUserData
    
    * @description Starts transcribing the user audio, the NLP model is connected and sends back results using an event, optionally could request for transcription and interim results.  Notice, you can only `startListening`, after microphone permissions have been granted. It is recommended to use `startListening` method only after `onListeningEnabled` callback was called.
    */
    startListening(options: VoiceML.ListeningOptions): void
    
    /**
    * @description Stops voice commands. 
    */
    stopListening(): void
    
    /**
    * @description Allows the user to provide voice commands for the VoiceML to execute on behalf of the users.  Current supported commands: "Take a Snap", "Start Recording", "Stop Recording". In case a command was detected, it will be automtically executed by the system and returned as part of the [NlpCommandResponse](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLNlpCommandResponse) in the `onListeningUpdate` callback. You can retrieve the command that was executed using the following snippet: 
    
    * ```js
    * var onUpdateListeningEventHandler = function(eventArgs) {
    *     var commandResponses = eventArgs.getCommandResponses();
    *     var nlpResponseText = "";
    
    *     for (var i = 0; i < commandResponses.length; i++) {
    *         var commandResponse = commandResponses[i];
    *         nlpResponseText += "Command Response: " + commandResponse.modelName + "\n command: " + commandResponse.command;     
    *     }
    * }
    * ``` 
    */
    enableSystemCommands(): void
    
    /**
    * @readonly
    
    * @description Registers a callback, which will be called with interim transcription or related NLP models. 
    */
    onListeningUpdate: event1<VoiceML.ListeningUpdateEventArgs, void>
    
    /**
    * @readonly
    
    * @description Registers a callback, which will be called in case the VoiceML module can't process the inputs. Most errors are due to network connectivity, or misconfigured NLP inputs. 
    */
    onListeningError: event1<VoiceML.ListeningErrorEventArgs, void>
    
    /**
    * @readonly
    
    * @description Registers a callback which will be called when microphone permissions are granted to the Lens, the microphone is initialized, and is actively listening. The expected design pattern is to start the listening session once those permissions have been granted: 
    
    * ```js
    * //@input Asset.VoiceMLModule vmlModule
    * var onListeningEnabledHandler = function(){
    *     script.vmlModule.startListening(options);  
    * }
    * script.vmlModule.onListeningEnabled.add(onListeningEnabledHandler); 
    * ```
    
    */
    onListeningEnabled: event0<void>
    
    /**
    * @readonly
    
    * @description Registers a callback which will be called when microphone permissions are taken from the lens. `stopListening()` is implicitly called in such case. 
    */
    onListeningDisabled: event0<void>
    
}

declare namespace VoiceMLModule {
    enum AnswerStatusCodes {
        /**
        * @description Should never show up in javascript frontend. Default value.
        */
        UNSET,
        /**
        * @description Answer found
        */
        STATUS_OK,
        /**
        * @description User transcript was not a question
        */
        NOT_A_QUESTION,
        /**
        * @description There was a question, but we don't know the answer
        */
        NO_ANSWER_FOUND
    }

}

declare namespace VoiceMLModule {
    /**
    * @description Status Codes for NLP Responses. 
    */
    enum NlpResponsesStatusCodes {
        /**
        * @description Ok Status Code for NLP Responses, this indicates a successful processing of the NLP model. 
        */
        OK,
        /**
        * @description Error Status Code for NLP Responses, this indicates an unsuccessful processing of the NLP model. 
        */
        ERROR
    }

}

declare namespace VoiceMLModule {
    interface SpeechRecognizer {
    }

}
declare namespace VoiceMLModule {
    namespace SpeechRecognizer {
        let Default: string
        
    
    }

}

/**
* @description Types of weather returned by [UserContextSystem's](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#UserContextSystem) `requestWeatherCondition()` callback.
*/
declare enum WeatherCondition {
    /**
    * @description Unknown or unsupported weather condition
    */
    Unknown,
    /**
    * @description Lightning
    */
    Lightning,
    /**
    * @description Low Visibility
    */
    LowVisibility,
    /**
    * @description Partly Cloudy
    */
    PartlyCloudy,
    /**
    * @description Clear Night
    */
    ClearNight,
    /**
    * @description Cloudy
    */
    Cloudy,
    /**
    * @description Rainy
    */
    Rainy,
    /**
    * @description Hail
    */
    Hail,
    /**
    * @description Snow
    */
    Snow,
    /**
    * @description Windy
    */
    Windy,
    /**
    * @description Sunny
    */
    Sunny
}

declare enum WeightedMode {
    Both,
    Left,
    None,
    Right
}

/**
* @description Groups physics objects in its subtree into an independent world simulation.
*/
interface WorldComponent extends Component {
    /**
    * @description Create an intersection probe for a specific world.
    */
    createProbe(): Probe
    
    /**
    * @description Update order of this world relative to other worlds (lower values are earlier). The implicit root world order is 0, so worlds with negative order will update before the root world, and worlds with non-negative order update after.
    */
    updateOrder: number
    
    /**
    * @description Reference to world settings. If not set, use the default world settings for the project.
    */
    worldSettings: Physics.WorldSettingsAsset
    
}

/**
* @description Holds settings for world mesh tracking in DeviceTracking component. Accessible through DeviceTracking.worldOptions.
*/
interface WorldOptions extends ScriptObject {
    pointCloudEnabled: boolean
    
    nativePlaneTrackingType: NativePlaneTrackingType
    
    /**
    * @description Enables or disables the generation of world meshes.
    */
    enableWorldMeshesTracking: boolean
    
    /**
    * @description Enables or disables world mesh classification gathering.
    */
    enableWorldMeshesClassificationTracking: boolean
    
}

/**
* @description Provider for RenderMesh data representing the estimated shape of real world objects generated from depth information. Only available when world mesh tracking is supported and enabled.
*/
interface WorldRenderObjectProvider extends RenderObjectProvider {
    /**
    * @description Mesh classification format being used.
    */
    meshClassificationFormat: MeshClassificationFormat
    
    /**
    * @description Enable or disable normal direction usage.
    */
    useNormals: boolean
    
    /**
    * @description Enable or disable world mesh tracking.
    */
    enableWorldMeshesTracking: boolean
    
    /**
    * @readonly
    
    * @description Returns the number of vertices of the mesh.
    */
    vertexCount: number
    
    /**
    * @readonly
    
    * @description Returns the number of faces of the mesh.
    */
    faceCount: number
    
}

/**
* @description Provides information about whether certain world tracking features are supported by the device.
*/
interface WorldTrackingCapabilities extends ScriptObject {
    /**
    * @readonly
    
    * @description Returns true if the device supports scene reconstruction.
    */
    sceneReconstructionSupported: boolean
    
    /**
    * @readonly
    */
    trackedWorldPointsSupported: boolean
    
    /**
    * @readonly
    */
    planesTrackingSupported: boolean
    
}

/**
* @description Triggered when new world tracking meshes are detected. Only available when a Device Tracking component is in the scene, and world mesh tracking is supported and enabled.
*/
interface WorldTrackingMeshesAddedEvent extends SceneEvent {
    /**
    * @description Returns an array of newly added Tracked Meshes.
    */
    getMeshes(): TrackedMesh[]
    
}

/**
* @description Triggered when some world tracking meshes are no longer detected. Only available when a Device Tracking component is in the scene, and world mesh tracking is supported and enabled.
*/
interface WorldTrackingMeshesRemovedEvent extends SceneEvent {
    /**
    * @description Returns an array of TrackedMeshes that are no longer detected.
    */
    getMeshes(): TrackedMesh[]
    
}

/**
* @description Triggered when world tracking meshes are updated. Only available when a Device Tracking component is in the scene, and world mesh tracking is supported and enabled.
*/
interface WorldTrackingMeshesUpdatedEvent extends SceneEvent {
    /**
    * @description Returns an array of TrackedMeshes that were updated.
    */
    getMeshes(): TrackedMesh[]
    
}

/**
* @description Triggered when plane(s) are newly detected. The `worldOptions.nativePlaneTrackingType` must be set to anything other than `NativePlaneTrackingType.None`. In addition, [Device Tracking](https://lensstudio.snapchat.com/api/classes/devicetracking) component must be set to `World` mode to orient the Camera relative to the planes correctly.

*/
interface WorldTrackingPlanesAddedEvent extends SceneEvent {
    /**
    * @description Returns an array of newly added [TrackedPlanes](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#trackedplane).
    */
    getPlanes(): TrackedPlane[]
    
}

/**
* @description Triggered when plane(s) are no longer detected.  This usually happens when two planes merge into one.  Planes persist when no longer seen by camera(s) and when previously detected objects move (e.g. a door is opened) to create a static scene. [Device Tracking](https://lensstudio.snapchat.com/api/classes/devicetracking) component must be set to `World` mode to orient the Camera relative to the planes correctly.

*/
interface WorldTrackingPlanesRemovedEvent extends SceneEvent {
    /**
    * @description Returns an array of [TrackedPlanes](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#trackedplane) that are no longer detected. 
    */
    getPlanes(): TrackedPlane[]
    
}

/**
* @description Triggered when currently detected plane(s) are updated.  This usually happens when a plane grows in size. The `worldOptions.nativePlaneTrackingType` must be set to anything other than `NativePlaneTrackingType.None`. In addition, [Device Tracking](https://lensstudio.snapchat.com/api/classes/devicetracking) component must be set to `World` mode to orient the Camera relative to the planes correctly.

*/
interface WorldTrackingPlanesUpdatedEvent extends SceneEvent {
    /**
    * @description Returns an array of [TrackedPlanes](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#trackedplane) that were updated.
    */
    getPlanes(): TrackedPlane[]
    
}

/**
* @description Describes how a texture should be sampled when using coordinates outside of the normal range.
*/
declare enum WrapMode {
    /**
    * @description Take the average of the border color and stretch it out.
    */
    ClampToEdge,
    /**
    * @description Repeats the current texture at the edge again.
    */
    Repeat,
    /**
    * @description Repeats the current texture at the edge again, but flipping the orientation.
    */
    MirroredRepeat,
    /**
    * @description Take the border color at the edge of a texture and stretch it out.
    */
    ClampToBorder
}

declare enum Zodiac {
    Aquarius,
    Aries,
    Cancer,
    Capricorn,
    Gemini,
    Leo,
    Libra,
    Pisces,
    Sagittarius,
    Scorpio,
    Taurus,
    Virgo
}



declare var script : ScriptComponent;

interface event0<R> {
    add(callback: () => R) : EventRegistration
    remove(eventRegistration: EventRegistration) : void
}
interface event1<T0,R> {
    add(callback: (arg0:T0) => R) : EventRegistration
    remove(eventRegistration: EventRegistration) : void
}
interface event2<T0,T1,R> {
    add(callback: (arg0:T0, arg1:T1) => R) : EventRegistration
    remove(eventRegistration: EventRegistration) : void
}
interface event3<T0,T1,T2,R> {
    add(callback: (arg0:T0, arg1:T1, arg2:T2) => R) : EventRegistration
    remove(eventRegistration: EventRegistration) : void
}
interface event4<T0,T1,T2,T3,R> {
    add(callback: (arg0:T0, arg1:T1, arg2:T2, arg3:T3) => R) : EventRegistration
    remove(eventRegistration: EventRegistration) : void
}
interface event5<T0,T1,T2,T3,T4,R> {
    add(callback: (arg0:T0, arg1:T1, arg2:T2, arg3:T3, arg4:T4) => R) : EventRegistration
    remove(eventRegistration: EventRegistration) : void
}

/**
* @deprecated

* @description Not for real use. Contains a fake instance of every class. Helpful for checking the API of a class.
*/
declare namespace _palette {
    /**
    * @description An object containing the position of an object relative to a LocationAsset. 
    */
    let Anchor: Anchor
    
    /**
    * @description Controls an animated texture resource. Can be accessed from [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#texture--control) on an animated texture.
    * See also: [2D Animation Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/adding-content/2d/2d-animation).
    */
    let AnimatedTextureFileProvider: AnimatedTextureFileProvider
    
    /**
    * @description Used by [AnimationMixer](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AnimationMixer) to animate a single object in the hierarchy.
    * These are automatically added to SceneObjects when importing animated FBX files.
    * See also: [Playing 3D Animation Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/additional-examples/playing-3d-animation), [AnimationMixer](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AnimationMixer), [AnimationLayer](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#AnimationLayer).
    */
    let Animation: Animation
    
    /**
    * @description Asset that contains multiple animation layers. Animation assets themselves do not handle playing or orchestrating animations. This is left to the animation player component to handle.
    */
    let AnimationAsset: AnimationAsset
    
    /**
    * @description Animation Clip is what an Animation Player uses to manage playback for a specific animation. It defines that animation by referencing an Animation Asset and providing start and end points, playback speed and direction, and blending information. 
    */
    let AnimationClip: AnimationClip
    
    /**
    * @description Used by [AnimationMixerLayer](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#AnimationMixerLayer) for setting animation looping behavior.
    */
    let AnimationClip_PostInfinity: AnimationClip.PostInfinity
    
    /**
    * @description Used by [AnimationMixerLayer](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#AnimationMixerLayer) for setting animation clip range type.
    */
    let AnimationClip_RangeType: AnimationClip.RangeType
    
    let AnimationCurve: AnimationCurve
    
    let AnimationCurveTrack: AnimationCurveTrack
    
    let AnimationKeyFrame: AnimationKeyFrame
    
    /**
    * @description Configures an animation layer for a single [SceneObject](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject).
    * Gives access to position, rotation, scale and blend shape animation tracks.
    * See also: [Playing 3D Animation Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/additional-examples/playing-3d-animation), [AnimationMixer](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AnimationMixer), [Animation](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Animation).
    */
    let AnimationLayer: AnimationLayer
    
    let AnimationLayerBlendMode: AnimationLayerBlendMode
    
    let AnimationLayerScaleMode: AnimationLayerScaleMode
    
    /**
    * @description Controls playback of animations on the attached [SceneObject](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject) and its child objects.
    * Please refer to the [Playing 3D Animation Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/additional-examples/playing-3d-animation) for setting up and playing animations.
    */
    let AnimationMixer: AnimationMixer
    
    /**
    * @description Controls animation playback for a single animation layer.
    * See also: [AnimationMixer](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#AnimationMixer).
    */
    let AnimationMixerLayer: AnimationMixerLayer
    
    /**
    * @description Component that handles playing animation clips as well as binding callbacks to user defined events.
    */
    let AnimationPlayer: AnimationPlayer
    
    let AnimationPropertyEventRegistration: AnimationPropertyEventRegistration
    
    /**
    * @description A layer containing different properties. Examples include position, rotation, scale or any other arbitrary properties a user would like to add and sample from.
    */
    let AnimationPropertyLayer: AnimationPropertyLayer
    
    /**
    * @description The base class for animation tracks.
    */
    let AnimationTrack: AnimationTrack
    
    /**
    * @description Base class for all assets used in the engine.
    */
    let Asset: Asset
    
    /**
    * @description Used by [Head.setAttachmentPointType()](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Head) to specify the type of attachment used with a Head binding.
    */
    let AttachmentPointType: AttachmentPointType
    
    /**
    * @description The curve that specifies how sound fades with the distance from Audio Component to the Audio Listener.****
    */
    let Audio_DistanceCurveType: Audio.DistanceCurveType
    
    /**
    * @description Used to play audio in a Lens.
    * You can assign an [AudioTrackAsset](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#AudioTrackAsset) to play through script or through the AudioComponent's inspector in Lens Studio.
    * See the [Playing Audio](https://lensstudio.snapchat.com/guides/audio) guide for more information.
    */
    let AudioComponent: AudioComponent
    
    /**
    * @description Configures an audio effect for [AudioEffectComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AudioEffectComponent).
    */
    let AudioEffectAsset: AudioEffectAsset
    
    /**
    * @description Used to add an audio effect to a Lens.
    * When present in the scene, it will automatically apply the selected audio effect to recordings made with the Lens.
    * See the [Audio Effect](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/audio/audio-effect) guide for more information.
    
    */
    let AudioEffectComponent: AudioEffectComponent
    
    /**
    * @description Provider for [AudioEffectAsset](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#AudioEffectAsset).
    */
    let AudioEffectProvider: AudioEffectProvider
    
    /**
    * @description A component that receives input from Audio Components that have Spatial Audio enabled. Calculates their positions relative to the scene object it is attached to,and properly mixes them.
    */
    let AudioListenerComponent: AudioListenerComponent
    
    /**
    * @description Provider of the Audio Output Audio Track asset. 
    */
    let AudioOutputProvider: AudioOutputProvider
    
    /**
    * @description Represents an audio file asset.
    * See also: [AudioComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AudioComponent).
    */
    let AudioTrackAsset: AudioTrackAsset
    
    /**
    * @description Base class for Audio Track providers.
    */
    let AudioTrackProvider: AudioTrackProvider
    
    /**
    * @description Cardinal axis enumeration.
    */
    let Axis: Axis
    
    /**
    * @description Settings for rendering the background on a [Text](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) component.
    * Accessible through the [Text](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) component's `backgroundSettings` property.
    */
    let BackgroundSettings: BackgroundSettings
    
    /**
    * @description The base class for all mesh rendering components.
    * Comparable to the former class "MeshVisual", which was split into the classes:
    * [BaseMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#BaseMeshVisual),
    * [MaterialMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MaterialMeshVisual),
    * and [RenderMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#RenderMeshVisual).
    */
    let BaseMeshVisual: BaseMeshVisual
    
    /**
    * @description Base class for MultiplayerSession options. This class is not used directly - use ConnectedLensSessionOptions instead.
    */
    let BaseMultiplayerSessionOptions: BaseMultiplayerSessionOptions
    
    /**
    * @description Base class for Input and Output Placeholders used by MLComponent.
    */
    let BasePlaceholder: BasePlaceholder
    
    /**
    * @description Provides basic information about a transformation.
    * See also: [DeviceTracking](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#DeviceTracking)
    */
    let BasicTransform: BasicTransform
    
    /**
    * @description File based asset.
    */
    let BinAsset: BinAsset
    
    let Bitmoji2DOptions: Bitmoji2DOptions
    
    let Bitmoji2DResource: Bitmoji2DResource
    
    let Bitmoji3DOptions: Bitmoji3DOptions
    
    /**
    * @description Provides information about the current user's 3D Bitmoji avatar to be downloaded via the RemoteMediaModule.
    */
    let Bitmoji3DResource: Bitmoji3DResource
    
    /**
    * @description Provides access to getting information about the current user's Bitmoji.
    */
    let BitmojiModule: BitmojiModule
    
    let BlendMode: BlendMode
    
    /**
    * @description Used to analyze the camera input and apply similar image artifacts to your AR objects in order to allow it to blend and match the real world better.
    */
    let BlurNoiseEstimation: BlurNoiseEstimation
    
    /**
    * @description Derived from ColliderComponent, attaching this to a SceneObject turns it into a dynamic rigid-body that is automatically moved by the physics simulation in response to gravity, collisions, and other forces.
    */
    let BodyComponent: BodyComponent
    
    /**
    * @description Provides depth values of the tracked body encoded as D24_UNORM_S8_UINT in screen resolution. Depth is normalized between 0-1, and the stencil component is ignored. The values may be remapped from normalized units (0 to 1) to negative centimeters from the camera (-near in cm to -far in cm) using the `Depth Map` mode of the `Texture 2D Sample` node in Material Editor.
    
    */
    let BodyDepthTextureProvider: BodyDepthTextureProvider
    
    /**
    * @description Provides surface normal values of the tracked body encoded as RGBA (x, y, z, confidence) in `R8G8B8A8_UNORM` format in screen resolution. The XYZ values may need to be remapped from [0 -> 1] to [-1 -> 1] using the "Normal Map" mode of the `Texture 2D Sample` node in Material Editor.
    */
    let BodyNormalsTextureProvider: BodyNormalsTextureProvider
    
    /**
    * @description Provider for full Body Mesh render object.
    */
    let BodyRenderObjectProvider: BodyRenderObjectProvider
    
    /**
    * @description Asset used to configure Body Tracking for the [ObjectTracking3D](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ObjectTracking3D) component.
    */
    let BodyTrackingAsset: BodyTrackingAsset
    
    /**
    * @description A box collision shape.
    */
    let BoxShape: BoxShape
    
    /**
    * @description Triggered when eyebrows are lowered on the tracked face.
    */
    let BrowsLoweredEvent: BrowsLoweredEvent
    
    /**
    * @description Triggered when eyebrows are raised on the tracked face.
    */
    let BrowsRaisedEvent: BrowsRaisedEvent
    
    /**
    * @description Triggered when eyebrows are returned to normal on the tracked face.
    */
    let BrowsReturnedToNormalEvent: BrowsReturnedToNormalEvent
    
    /**
    * @description Renders the scene to a Render Target texture.
    * A Camera will only render a SceneObject if the SceneObject's render layer is enabled on the Camera.
    * For more information, see the [Camera and Layers](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/scene-set-up/camera) guide.
    */
    let Camera: Camera
    
    /**
    * @description The base class from which ColorRenderTarget and DepthStencilRenderTarget are derived from
    */
    let Camera_BaseRenderTarget: Camera.BaseRenderTarget
    
    /**
    * @description Color based RenderTarget.
    */
    let Camera_ColorRenderTarget: Camera.ColorRenderTarget
    
    /**
    * @description Different faces of the cubemap that a camera can render into.
    */
    let Camera_CubemapFace: Camera.CubemapFace
    
    /**
    * @description Used in [Camera's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Camera) `depthBufferMode` property.
    * Each mode is suited for handling objects at a certain distance range.
    * For more information on depth modes, see the [Camera and Layers](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/scene-set-up/camera) guide.
    */
    let Camera_DepthBufferMode: Camera.DepthBufferMode
    
    /**
    * @description This class inherits from the BaseRenderTarget class. BaseRenderTarget class is not available for creation and is used like the base class for DepthStencilRenderTarget class to provide access to targetTexture, inputTexture and maskTexture properties.
    */
    let Camera_DepthStencilRenderTarget: Camera.DepthStencilRenderTarget
    
    /**
    * @description Used in [Camera's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Camera) `devicePropertyUsage` property.
    * Specifies which camera properties should be overridden by device properties.
    */
    let Camera_DeviceProperty: Camera.DeviceProperty
    
    /**
    * @description Returned from [Camera's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Camera) `type` property.
    */
    let Camera_Type: Camera.Type
    
    /**
    * @description Triggered when the device's back facing camera becomes active.
    */
    let CameraBackEvent: CameraBackEvent
    
    /**
    * @description Triggered when the device's front facing camera becomes active.
    */
    let CameraFrontEvent: CameraFrontEvent
    
    let CameraTextureProvider: CameraTextureProvider
    
    /**
    * @description A 2D canvas anchored in 3D space that acts as the root of the ScreenTransform hierarchy. ScreenTransform SceneObjects can be placed on the Canvas, and the Canvas can be sized and placed anywhere in 3D space. It is like a painter’s canvas for ScreenTransforms.
    */
    let Canvas: Canvas
    
    let Canvas_SortingType: Canvas.SortingType
    
    let Canvas_UnitType: Canvas.UnitType
    
    /**
    * @description Changes the capitalization of the text component. Useful when using dynamic texts.
    */
    let CapitilizationOverride: CapitilizationOverride
    
    /**
    * @description A capsule collision shape. Also known as a capped cylinder.
    */
    let CapsuleShape: CapsuleShape
    
    /**
    * @description Settings for how color will be cleared before rendering.
    */
    let ClearColorOption: ClearColorOption
    
    /**
    * @description Clears depth in the drawing order.
    */
    let ClearDepth: ClearDepth
    
    /**
    * @description Handles the mesh data of cloth and prepares it for cloth simulation. Also controls all the parameters of the cloth simulator and colliders.
    */
    let ClothVisual: ClothVisual
    
    /**
    * @description Cloth bend mode to use for cloth simulation.
    */
    let ClothVisual_BendMode: ClothVisual.BendMode
    
    /**
    * @description Options associated with the listValues method call.
    */
    let CloudStorageListOptions: CloudStorageListOptions
    
    /**
    * @description Provides access to Cloud Storage.
    */
    let CloudStorageModule: CloudStorageModule
    
    /**
    * @description Used to configure `Cloud Storage Module` with various options. Note: if `session` scoped storage is required, this option must be provided.
    
    */
    let CloudStorageOptions: CloudStorageOptions
    
    /**
    * @description Options associated with the getValue/deleteValue methods for Cloud Storage.
    */
    let CloudStorageReadOptions: CloudStorageReadOptions
    
    /**
    * @description Options associated with the setValue method for Cloud Storage.
    */
    let CloudStorageWriteOptions: CloudStorageWriteOptions
    
    /**
    * @description An instance of Cloud Storage that can store data in a multiplayer experience.
    */
    let CloudStore: CloudStore
    
    /**
    * @description Collider used by the [Hair Visual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#HairVisual) for its simulation.
    */
    let ColliderComponent: ColliderComponent
    
    /**
    * @description A state generated for ColliderComponent collision events.
    */
    let Collision: Collision
    
    /**
    * @description Args used for [ColliderComponent.onCollisionEnter](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), which is triggered when a collision begins.
    */
    let CollisionEnterEventArgs: CollisionEnterEventArgs
    
    /**
    * @description Args used for [ColliderComponent.onCollisionExit](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), which is triggered when a collision ends.
    */
    let CollisionExitEventArgs: CollisionExitEventArgs
    
    let CollisionMesh: CollisionMesh
    
    /**
    * @description Args used for [ColliderComponent.onCollisionStay](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), which is triggered every frame while a collision continues.
    */
    let CollisionStayEventArgs: CollisionStayEventArgs
    
    /**
    * @description Not usable from JS. Accessed via the Colocated Landmarks 2D Mesh resource. Contains the 2D keypoints when creating a Colocated map. Expanded by the material provided with template.
    */
    let ColocatedLandmarks2DRenderObjectProvider: ColocatedLandmarks2DRenderObjectProvider
    
    /**
    * @description Not usable from JS. Accessed via the Colocated Landmarks 3D Mesh resource. Contains the 3D landmarks when creating a Colocated map. Expanded by the material provided with template.
    */
    let ColocatedLandmarks3DRenderObjectProvider: ColocatedLandmarks3DRenderObjectProvider
    
    let ColocatedLandmarksRenderObjectProviderBase: ColocatedLandmarksRenderObjectProviderBase
    
    /**
    * @description Creates Colocated Connected Lenses experiences by enabling the creation and tracking of a shared space which can be used to place several users in the same coordinate frame. This shared space will be made available and can be tracked by any friend you invite to join your session via Snapcode. Users are expected to be located in the same room when using the colocated feature. This component needs to be attached to the camera. 
    */
    let ColocatedTrackingComponent: ColocatedTrackingComponent
    
    /**
    * @description Data type used for color values.
    */
    let Colorspace: Colorspace
    
    /**
    * @description The base class for all components.  Components are attached to [SceneObjects](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject).
    */
    let Component: Component
    
    /**
    * @description A cone collision shape.
    */
    let ConeShape: ConeShape
    
    /**
    * @description Event fired when the "Launch connected lens" button was pressed. Wait for this event to be triggered before creating a session, as having this event gaurantees the user has accepted the necessary disclosures to use a connected lens experience.
    */
    let ConnectedLensEnteredEvent: ConnectedLensEnteredEvent
    
    /**
    * @description Connected Lenses Module allows use of networked Lens communication capabilities (real-time communication, co-located session creation and joining, and shared persistent storage). It's recommended to only use one ConnectedLensModule per Lens.
    */
    let ConnectedLensModule: ConnectedLensModule
    
    /**
    * @description Information that is bootstrapped to the user who just connected to the session.
    */
    let ConnectedLensModule_ConnectionInfo: ConnectedLensModule.ConnectionInfo
    
    /**
    * @description Information about the host update.
    */
    let ConnectedLensModule_HostUpdateInfo: ConnectedLensModule.HostUpdateInfo
    
    /**
    * @description Provides extra context about a RealtimeStore's creation.
    */
    let ConnectedLensModule_RealtimeStoreCreationInfo: ConnectedLensModule.RealtimeStoreCreationInfo
    
    /**
    * @description Gives information about the Realtime Store delete operation. 
    */
    let ConnectedLensModule_RealtimeStoreDeleteInfo: ConnectedLensModule.RealtimeStoreDeleteInfo
    
    /**
    * @description Provides information about a key being removed from a RealtimeStore.
    */
    let ConnectedLensModule_RealtimeStoreKeyRemovalInfo: ConnectedLensModule.RealtimeStoreKeyRemovalInfo
    
    /**
    * @description Provides information about a RealtimeStore's ownership being updated.
    */
    let ConnectedLensModule_RealtimeStoreOwnershipUpdateInfo: ConnectedLensModule.RealtimeStoreOwnershipUpdateInfo
    
    /**
    * @description Gives information about the Realtime Store update operation.
    */
    let ConnectedLensModule_RealtimeStoreUpdateInfo: ConnectedLensModule.RealtimeStoreUpdateInfo
    
    /**
    * @description Used by ConnectedLensesModule to specify the session share type.
    */
    let ConnectedLensModule_SessionShareType: ConnectedLensModule.SessionShareType
    
    /**
    * @description Provides information about a user in a Connected Lens session.
    */
    let ConnectedLensModule_UserInfo: ConnectedLensModule.UserInfo
    
    /**
    * @description Settings for configuring a Connected Lens session.
    */
    let ConnectedLensSessionOptions: ConnectedLensSessionOptions
    
    /**
    * @description Type of the created Connected Lens session.
    */
    let ConnectedLensSessionOptions_SessionCreationType: ConnectedLensSessionOptions.SessionCreationType
    
    /**
    * @description Constraints body motion in configurable ways, for simulating physical objects such as joints and hinges.
    */
    let Constraint: Constraint
    
    /**
    * @description Used to apply a constraint to an object.
    */
    let ConstraintComponent: ConstraintComponent
    
    /**
    * @description Contact point between two colliding objects.  See also: [ColliderComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), [CollisionEnterEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#CollisionEnterEventArgs), [CollisionExitEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#CollisionExitEventArgs), [CollisionStayEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#CollisionStayEventArgs).
    */
    let Contact: Contact
    
    /**
    * @description Base class for Texture Providers that crop an input texture.
    */
    let CropTextureProvider: CropTextureProvider
    
    /**
    * @description Used with [Pass's](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Pass) `cullMode` property.
    * Determines which faces of a surface are culled (not rendered).
    */
    let CullMode: CullMode
    
    /**
    * @description A cylinder collision shape.
    */
    let CylinderShape: CylinderShape
    
    let DeformingCollisionMesh: DeformingCollisionMesh
    
    /**
    * @description Creates a buffer for the audio data.
    */
    let Delay: Delay
    
    /**
    * @description Builder class for the Delay.
    */
    let DelayBuilder: DelayBuilder
    
    /**
    * @description An event that gets triggered after a delay.
    */
    let DelayedCallbackEvent: DelayedCallbackEvent
    
    /**
    * @description Local estimate of the derivative of the input data along the selected axis. Outputs the derivative of the input features along the window.
    */
    let Delta: Delta
    
    /**
    * @description Builder class for Delta.
    */
    let DeltaBuilder: DeltaBuilder
    
    /**
    * @description Settings for the depth clear option modes on a DepthStencilRenderTargetProvider.
    */
    let DepthClearOption: DepthClearOption
    
    /**
    * @description Writes video feed depth information to the depth buffer, which automatically sets up depth occlusion for 3D visuals.
    * Only works in some cases, such as in Lenses for Spectacles 3.
    * See the [Lenses for Spectacles guide](https://lensstudio.snapchat.com/lens-studio/references/guides/distributing/spectacles/spectacles-3/lenses-for-spectacles-3) for more information.
    */
    let DepthSetter: DepthSetter
    
    /**
    * @description Access to a Depth Stencil Render Target that can output depth and stencil values from a Camera in Depth24/Stencil8 format.
    */
    let DepthStencilRenderTargetProvider: DepthStencilRenderTargetProvider
    
    /**
    * @description Provides depth information of the video feed that the Lens is being applied to when available, such as in Lenses for Spectacles 3.
    * Can be accessed from `mainPass.baseTex.control` of a Spectacles Depth material.
    
    * See the [Lenses for Spectacles guide](https://lensstudio.snapchat.com/lens-studio/references/guides/distributing/spectacles/spectacles-2021/lenses-for-spectacles) for more information.
    */
    let DepthTextureProvider: DepthTextureProvider
    
    /**
    * @description Provides information about the device's camera.
    */
    let DeviceCamera: DeviceCamera
    
    /**
    * @description Provides information about the device running the Lens. Accessible through `global.deviceInfoSystem`.
    */
    let DeviceInfoSystem: DeviceInfoSystem
    
    /**
    * @description Used to track a landmarker in the camera. Moves the SceneObject's transform to match the detected landmarker scene.
    * See the [Landmarker guide](https://lensstudio.snapchat.com/templates/landmarker) for more information.
    */
    let DeviceLocationTrackingComponent: DeviceLocationTrackingComponent
    
    /**
    * @description Moves or rotates the SceneObject to match device orientation.
    
    * If using "Surface" tracking mode, adding this to a SceneObject enables surface tracking for the scene, and moves the
    * object to a position and rotation that matches the physical camera's pose in the world. Surface tracking can also be enhanced
    * with native AR by enabling the "Use Native AR" option in the Inspector panel, or through script by setting the
    * component's [`surfaceOptions.enhanceWithNativeAR`](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SurfaceOptions) property.
    
    * If using "Rotation" tracking mode, adding this to a SceneObject will apply the device's real world rotation to the object.
    
    * If using "World" tracking mode, adding this to a SceneObject enables native AR tracking for the scene, and moves the
    * object to a position and rotation that matches the physical camera's pose in the world.
    
    * See the [Tracking Modes](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/world/tracking-modes) guide for more information.
    
    **Note:** This component was named "WorldTracking" in previous versions of Lens Studio.
    */
    let DeviceTracking: DeviceTracking
    
    /**
    * @description Tracking modes used by the [DeviceTracking](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#DeviceTracking) component to specify what type of tracking to use.
    */
    let DeviceTrackingMode: DeviceTrackingMode
    
    /**
    * @description Provides answer information in response to `DialogModule.askQuestions()`.
    */
    let Dialog_Answer: Dialog.Answer
    
    let DialogModule: DialogModule
    
    /**
    * @description An Audio Component effect that simulates sound attenuation based on the orientation of the transform relative to the [AudioListenerComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AudioListenerComponent). 
    */
    let DirectivityEffect: DirectivityEffect
    
    let DirectMultiplayerSessionOptions: DirectMultiplayerSessionOptions
    
    /**
    * @description An audio effect that simulates sound attenuation based on the distance between the Audio and the [AudioListenerComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#AudioListenerComponent). 
    */
    let DistanceEffect: DistanceEffect
    
    /**
    * @description Used in [Text's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) `dropShadowSettings` property.
    * Configures how dropshadow will appear on a Text component.
    */
    let DropshadowSettings: DropshadowSettings
    
    /**
    * @description A resource that is resolved at runtime.
    */
    let DynamicResource: DynamicResource
    
    let EventRegistration: EventRegistration
    
    /**
    * @description Expression names used with `FaceRenderObjectProvider.getExpressionWeightByName()`
    * and returned by `FaceRenderObjectProvider.getExpressionNames()`.
    */
    let Expressions: Expressions
    
    /**
    * @description Applies an eye color effect to a face.
    */
    let EyeColorVisual: EyeColorVisual
    
    /**
    * @description Texture Provider giving a cropped region of the input texture, calculated based on face position.
    * Can be accessed using `Texture.control` on a FaceCropTexture asset.
    * For more information, see the [Crop Textures](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/crop-textures#crop-textures) guide.
    */
    let FaceCropTextureProvider: FaceCropTextureProvider
    
    /**
    * @description Triggered when a new face is detected and starts being tracked.
    */
    let FaceFoundEvent: FaceFoundEvent
    
    /**
    * @description Controls the face image picker texture resource.
    * Can be accessed through [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#texture--control) on a Face Image Picker texture.
    * For more information, see the [Face Image Picker Texture guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/face/face-effects/face-image-picker-texture).
    */
    let FaceImagePickerTextureProvider: FaceImagePickerTextureProvider
    
    /**
    * @description Used with [FaceInsetVisual.faceRegion](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#FaceInsetVisual) for setting the face region to draw.
    */
    let FaceInsetRegion: FaceInsetRegion
    
    /**
    * @description Draws a section of a tracked face.
    */
    let FaceInsetVisual: FaceInsetVisual
    
    /**
    * @description Triggered when a face can no longer be tracked.  For example, if a face gets blocked from the camera's view, or gets too far away.
    */
    let FaceLostEvent: FaceLostEvent
    
    /**
    * @description Applies a face mask effect. See the [Face Mask Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/face/face-effects/face-mask) for more information.
    */
    let FaceMaskVisual: FaceMaskVisual
    
    /**
    * @description Mesh provider for a Face Mesh. Accessible through the `control` property on a Face Mesh `RenderMesh`.
    */
    let FaceRenderObjectProvider: FaceRenderObjectProvider
    
    /**
    * @description Applies a face stretch effect.
    * Face stretch features can be added to a FaceStretchVisual through the Inspector panel in Lens Studio.
    * See the [Face Stretch Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/face/face-effects/face-stretch) for more information.
    */
    let FaceStretchVisual: FaceStretchVisual
    
    /**
    * @description TextureProvider for face textures.
    * See the [Face Texture Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/face/face-effects/face-texture) for more information.
    * Can be accessed using [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Texture#control-textureprovider) on a face texture asset.
    */
    let FaceTextureProvider: FaceTextureProvider
    
    /**
    * @description This is the base class for all face tracking events. This event won't actually get triggered itself, so use one of the child classes instead.
    */
    let FaceTrackingEvent: FaceTrackingEvent
    
    /**
    * @description Provider for file based Audio Tracks.
    */
    let FileAudioTrackProvider: FileAudioTrackProvider
    
    /**
    * @description A file track provider of the Licensed Sounds from Asset LIbrary.
    */
    let FileLicensedSoundProvider: FileLicensedSoundProvider
    
    /**
    * @description A [TextureProvider](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#TextureProvider) for textures originating from files.
    */
    let FileTextureProvider: FileTextureProvider
    
    /**
    * @description Intersection filter settings. Unifies settings for world probes and collider overlap tests.   See also: [ColliderComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), [Physics](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#Physics).
    */
    let Filter: Filter
    
    let FilteringMode: FilteringMode
    
    let FixedCollisionMesh: FixedCollisionMesh
    
    /**
    * @description Fully constrain rotation and translation.  See also: [ConstraintComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ConstraintComponent).
    */
    let FixedConstraint: FixedConstraint
    
    /**
    * @description The base class for animation tracks using float values.
    */
    let FloatAnimationTrack: FloatAnimationTrack
    
    /**
    * @description Represents an animation track using float value keyframes.
    */
    let FloatAnimationTrackKeyFramed: FloatAnimationTrackKeyFramed
    
    /**
    * @description Represents an animation track using vec3 value keyframes for a bezier curve.
    */
    let FloatBezierAnimationTrackKeyFramed: FloatBezierAnimationTrackKeyFramed
    
    /**
    * @description Arguments used with the `InteractionComponent.onFocusEnd` event.
    */
    let FocusEndEventArgs: FocusEndEventArgs
    
    /**
    * @description Arguments used with the `InteractionComponent.onFocusStart` event.
    */
    let FocusStartEventArgs: FocusStartEventArgs
    
    /**
    * @description A font asset used for rendering text.
    * Used by [Text](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text).
    * For more information, see the [Text guide](https://lensstudio.snapchat.com/lens-studio/references/guides/adding-content/text/2d-text).
    */
    let Font: Font
    
    /**
    * @description Mode for setting frustum culling on Pass
    */
    let FrustumCullMode: FrustumCullMode
    
    let GaussianSplattingAsset: GaussianSplattingAsset
    
    let GaussianSplattingVisual: GaussianSplattingVisual
    
    /**
    * @description Class for storing and retrieving data based on keys.
    * Used by [PersistentStorageSystem](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#PersistentStorageSystem).
    * For more information, see the [Persistent Storage guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/persistent-storage).
    */
    let GeneralDataStore: GeneralDataStore
    
    /**
    * @description Namespace for location functionality.
    */
    let GeoLocation: GeoLocation
    
    /**
    * @description Enumeration of supported GPS location accuracy.
    */
    let GeoLocationAccuracy: GeoLocationAccuracy
    
    /**
    * @description The location of the device.
    */
    let GeoPosition: GeoPosition
    
    /**
    * @description Represents a GLTF 3D Model.
    */
    let GltfAsset: GltfAsset
    
    /**
    * @description Settings for importing a glTF Asset.
    * Use this with [GltfAsset](https://lensstudio.snapchat.com/api/classes/GltfAsset) component's `tryInstantiateWithSetting` method.
    */
    let GltfSettings: GltfSettings
    
    /**
    * @description Hair asset converted from an FBX containing splines to be used with [Hair Visual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#HairVisual).
    */
    let HairDataAsset: HairDataAsset
    
    /**
    * @description Component that renders hair simulation.
    */
    let HairVisual: HairVisual
    
    /**
    * @description Provides additional data for the tracked hand. You can figure out whether the tracked hand is the left hand by accessing the *isLeft* property [true/false], as well as the probability of this data through the  *isLeftProbability*  property [0-1].
    
    */
    let HandSpecificData: HandSpecificData
    
    /**
    * @description     
    */
    let HandTracking3DAsset: HandTracking3DAsset
    
    /**
    * @description Triggers haptic feedback on the device. (iOS Devices only)
    
    */
    let HapticFeedbackSystem: HapticFeedbackSystem
    
    /**
    * @description The method in which haptic feedback is provided. Use with the `HapticFeedbackSystem`.
    */
    let HapticFeedbackType: HapticFeedbackType
    
    /**
    * @description Binds the SceneObject to a tracked face.
    * See the [Head Attached 3D Objects Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/face/face-effects/head-attached-3d-objects) for more information.
    */
    let Head: Head
    
    /**
    * @description Constraints translation and a single axis of rotation.  See also: [ConstraintComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ConstraintComponent).
    */
    let HingeConstraint: HingeConstraint
    
    /**
    * @description Used to show and hide hints to the user.
    * For more information and useful helper scripts, see the [Scripting Hints Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/additional-examples/scripting-hints).
    
    * <table cellspacing=0 cellpadding=0><thead><tr><th>Hint ID</th><th>Hint Message</th></tr></thead><tbody><tr><td>&#8220;lens_hint_blow_a_kiss&#8221;</td><td>&#8220;Blow A Kiss&#8221;</td></tr><tr><td>&#8220;lens_hint_come_closer&#8221;</td><td>&#8220;Come Closer&#8221;</td></tr><tr><td>&#8220;lens_hint_do_not_smile&#8221;</td><td>&#8220;Do Not Smile&#8221;</td></tr><tr><td>&#8220;lens_hint_do_not_try_with_a_friend&#8221;</td><td>&#8220;Do Not Try With A Friend&#8221;</td></tr><tr><td>&#8220;lens_hint_find_face&#8221;</td><td>&#8220;Find Face&#8221;</td></tr><tr><td>&#8220;lens_hint_keep_raising_your_eyebrows&#8221;</td><td>&#8220;Keep Raising Your Eyebrows&#8221;</td></tr><tr><td>&#8220;lens_hint_kiss&#8221;</td><td>&#8220;Kiss&#8221;</td></tr><tr><td>&#8220;lens_hint_kiss_again&#8221;</td><td>&#8220;Kiss Again&#8221;</td></tr><tr><td>&#8220;lens_hint_look_around&#8221;</td><td>&#8220;Look Around&#8221;</td></tr><tr><td>&#8220;lens_hint_look_down&#8221;</td><td>&#8220;Look Down&#8221;</td></tr><tr><td>&#8220;lens_hint_look_left&#8221;</td><td>&#8220;Look Left&#8221;</td></tr><tr><td>&#8220;lens_hint_look_right&#8221;</td><td>&#8220;Look Right&#8221;</td></tr><tr><td>&#8220;lens_hint_look_up&#8221;</td><td>&#8220;Look Up&#8221;</td></tr><tr><td>&#8220;lens_hint_make_some_noise&#8221;</td><td>&#8220;Make Some Noise!&#8221;</td></tr><tr><td>&#8220;lens_hint_nod_your_head&#8221;</td><td>&#8220;Nod Your Head&#8221;</td></tr><tr><td>&#8220;lens_hint_now_kiss&#8221;</td><td>&#8220;Now Kiss&#8221;</td></tr><tr><td>&#8220;lens_hint_now_open_your_mouth&#8221;</td><td>&#8220;Now Open Your Mouth&#8221;</td></tr><tr><td>&#8220;lens_hint_now_raise_your_eyebrows&#8221;</td><td>&#8220;Now Raise Your Eyebrows&#8221;</td></tr><tr><td>&#8220;lens_hint_now_smile&#8221;</td><td>&#8220;Now Smile&#8221;</td></tr><tr><td>&#8220;lens_hint_open_your_mouth&#8221;</td><td>&#8220;Open Your Mouth&#8221;</td></tr><tr><td>&#8220;lens_hint_open_your_mouth_again&#8221;</td><td>&#8220;Open Your Mouth Again&#8221;</td></tr><tr><td>&#8220;lens_hint_raise_eyebrows_or_open_mouth&#8221;</td><td>&#8220;Raise Your Eyebrows / Or / Open Your Mouth&#8221;</td></tr><tr><td>&#8220;lens_hint_raise_your_eyebrows&#8221;</td><td>&#8220;Raise Your Eyebrows&#8221;</td></tr><tr><td>&#8220;lens_hint_raise_your_eyebrows_again&#8221;</td><td>&#8220;Raise Your Eyebrows Again&#8221;</td></tr><tr><td>&#8220;lens_hint_smile&#8221;</td><td>&#8220;Smile&#8221;</td></tr><tr><td>&#8220;lens_hint_smile_again&#8221;</td><td>&#8220;Smile Again&#8221;</td></tr><tr><td>&#8220;lens_hint_swap_camera&#8221;</td><td>&#8220;Swap Camera&#8221;</td></tr><tr><td>&#8220;lens_hint_tap&#8221;</td><td>&#8220;Tap!&#8221;</td></tr><tr><td>&#8220;lens_hint_tap_a_surface&#8221;</td><td>&#8220;Tap A Surface&#8221;</td></tr><tr><td>&#8220;lens_hint_tap_ground&#8221;</td><td>&#8220;Tap The Ground&#8221;</td></tr><tr><td>&#8220;lens_hint_tap_ground_to_place&#8221;</td><td>&#8220;Tap Ground To Place&#8221;</td></tr><tr><td>&#8220;lens_hint_tap_surface_to_place&#8221;</td><td>&#8220;Tap Surface To Place&#8221;</td></tr><tr><td>&#8220;lens_hint_try_friend&#8221;</td><td>&#8220;Try It With A Friend&#8221;</td></tr><tr><td>&#8220;lens_hint_try_rear_camera&#8221;</td><td>&#8220;Try It With Your Rear Camera&#8221;</td></tr><tr><td>&#8220;lens_hint_turn_around&#8221;</td><td>&#8220;Turn Around&#8221;</td></tr><tr><td>&#8220;lens_hint_walk_through_the_door&#8221;</td><td>&#8220;Walk Through The Door&#8221;</td></tr></tbody></table>
    */
    let HintsComponent: HintsComponent
    
    /**
    * @description Used by the `horizontalAlignment` property in [MeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MeshVisual).
    * When a [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) is attached to the same SceneObject, this determines how the mesh will be positioned horizontally.
    */
    let HorizontalAlignment: HorizontalAlignment
    
    /**
    * @description Options for wrapping text horizontally.
    * Used by [Text component's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) `horizontalOverflow` property.
    */
    let HorizontalOverflow: HorizontalOverflow
    
    /**
    * @description The base class for parameter objects passed into event callbacks.
    */
    let IEventParameters: IEventParameters
    
    /**
    * @description A 2D visual used for drawing texture assets.
    * Commonly used with [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) for drawing images on the screen.
    
    * See the [Image guide](https://lensstudio.snapchat.com/lens-studio/references/guides/adding-content/2d/image) for more information.
    */
    let Image: Image
    
    /**
    * @description Controls an image picker texture and UI.
    * Can be accessed through [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Texture#control-textureprovider) on an Image Picker texture.
    * For more information, see the [Image Picker Texture](https://lensstudio.snapchat.com/lens-studio/references/guides/adding-content/2d/media-picker-texture) guide.
    */
    let ImagePickerTextureProvider: ImagePickerTextureProvider
    
    /**
    * @description Builds InputPlaceHolders for MLComponent.
    */
    let InputBuilder: InputBuilder
    
    /**
    * @description Controls input data for a neural network used by an MLComponent.
    * For more information, see the [MLComponent Scripting](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/scripting-ml-component) guide.
    */
    let InputPlaceholder: InputPlaceholder
    
    /**
    * @description The base class for animation tracks using integer values.
    */
    let IntAnimationTrack: IntAnimationTrack
    
    /**
    * @description Allows the `MeshVisual` provided to this component to handle touches on the screen (blocking Snapchat from receiving the touches), and optionally let certain touch types to pass through (let Snapchat handle the touch).
    
    * Possible `TouchType` values:
    
    * "TouchTypeNone"
    * "TouchTypeTouch"
    * "TouchTypeTap"
    * "TouchTypeDoubleTap"
    * "TouchTypeScale"
    * "TouchTypePan"
    * "TouchTypeSwipe"
    */
    let InteractionComponent: InteractionComponent
    
    /**
    * @description Represents an animation track using stepped integer value keyframes.
    */
    let IntStepAnimationTrackKeyFramed: IntStepAnimationTrackKeyFramed
    
    /**
    * @description Represents an animation track using stepped integer value keyframes.
    */
    let IntStepNoLerpAnimationTrackKeyFramed: IntStepNoLerpAnimationTrackKeyFramed
    
    /**
    * @description Triggered when the tracked face ends a kiss.
    */
    let KissFinishedEvent: KissFinishedEvent
    
    /**
    * @description Triggered when the tracked face starts a kiss.
    */
    let KissStartedEvent: KissStartedEvent
    
    /**
    * @description This event is triggered at the end of every frame, after normal [UpdateEvents](https://lensstudio.snapchat.com/api/lens-studio/Classes/Events#UpdateEvent) trigger
    * but before rendering occurs.
    */
    let LateUpdateEvent: LateUpdateEvent
    
    /**
    * @description Used to describe a set of layers that an object belongs to or interacts with.
    
    * See [SceneObject's](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject) `layer` property, [Camera's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Camera) `renderLayer` property, and [LightSource's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#LightSource) `renderLayer` property.
    */
    let LayerSet: LayerSet
    
    /**
    * @description Collider asset generated from a mesh to be used with the [Hair Visual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#HairVisual) as part of the hair simulation.
    */
    let LevelsetColliderAsset: LevelsetColliderAsset
    
    /**
    * @description A levelset collision shape. A levelset is an asset that is generated to approximate a mesh.
    */
    let LevelsetShape: LevelsetShape
    
    /**
    * @description Acts as a source of light in the scene.
    * See the [Light and Shadows](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/graphics/light-and-shadow) guide for more information about lighting.
    */
    let LightSource: LightSource
    
    /**
    * @description Applies a liquify effect to anything rendered behind it.
    */
    let LiquifyVisual: LiquifyVisual
    
    let LoadStatus: LoadStatus
    
    /**
    * @description Asset used with the Localizations system to support custom localization strings.
    */
    let LocalizationsAsset: LocalizationsAsset
    
    /**
    * @description Helps convert data types to localized string representations. Accessible through `global.localizationSystem`.
    
    * Note that formatted or localized strings may appear differently to users depending on their region.
    * The example results given here are representative of a user in the United States, but may appear differently for users in other regions.
    */
    let LocalizationSystem: LocalizationSystem
    
    /**
    * @description A component which modifies the [Transform](https://lensstudio.snapchat.com/api/classes/Transform) of the object it is on to a position in the real world, based on a `LocationAsset` and a `position`.
    
    */
    let LocatedAtComponent: LocatedAtComponent
    
    /**
    * @description Provides a frame of reference in which to localize objects to the real world. Use with [LocatedAtComponent](https://lensstudio.snapchat.com/api/classes/LocatedAtComponent).
    */
    let LocationAsset: LocationAsset
    
    /**
    * @description Provides access to location cloud storage depending upon the LocationCloudStorageOptions.
    */
    let LocationCloudStorageModule: LocationCloudStorageModule
    
    /**
    * @description Used to configure LocationCloudStorage module with various options. 
    */
    let LocationCloudStorageOptions: LocationCloudStorageOptions
    
    /**
    * @description Instance of location cloud store which has a similar interface as cloud store.
    */
    let LocationCloudStore: LocationCloudStore
    
    /**
    * @description Used by [DeviceLocationTrackingComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#DeviceLocationTrackingComponent) to indicate the user's distance from the landmarker location.
    * See the [Landmarker guide](https://lensstudio.snapchat.com/templates/landmarker) for more information.
    */
    let LocationProximityStatus: LocationProximityStatus
    
    /**
    * @description Provides access to a location's Mesh--such as when working with City Scale AR. Usually used in conjunction with `LocationTextureProvider`.
    */
    let LocationRenderObjectProvider: LocationRenderObjectProvider
    
    /**
    * @description The LocationService allows the user to provide their location to lens applications if they so desire. For privacy reasons, the user is asked for permission to report location information.
    */
    let LocationService: LocationService
    
    /**
    * @description Provides access to a location's texture--such as when working with City Scale AR.
    */
    let LocationTextureProvider: LocationTextureProvider
    
    /**
    * @description Every frame, LookAtComponent rotates its SceneObject to face towards a target SceneObject.
    */
    let LookAtComponent: LookAtComponent
    
    /**
    * @description The "aim" and "up" vectors used with [LookAtComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#LookAtComponent) when determining rotation.
    * LookAtComponent will try to point the `Aim` axis of the SceneObject towards the target,
    * while keeping the `Up` axis of the SceneObject pointing towards `worldUpVector`.
    */
    let LookAtComponent_AimVectors: LookAtComponent.AimVectors
    
    /**
    * @description Modes used in `LookAtComponent.lookAtMode` to determine the rotation method being used.
    */
    let LookAtComponent_LookAtMode: LookAtComponent.LookAtMode
    
    /**
    * @description Used with [LookAtComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#LookAtComponent) to set the "up" vector when determining rotation.
    */
    let LookAtComponent_WorldUpVector: LookAtComponent.WorldUpVector
    
    /**
    * @description Namespace for Machine Learning related classes and methods.
    * For more information, see the [Machine Learning Overview](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-overview).
    */
    let MachineLearning: MachineLearning
    
    /**
    * @description Timing options for when MLComponent should start or stop running. Used with `MLComponent.runScheduled()`.
    * For more information, see the [MLComponent Scripting](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/scripting-ml-component) guide.
    */
    let MachineLearning_FrameTiming: MachineLearning.FrameTiming
    
    /**
    * @description Inference modes used by `MLComponent.inferenceMode`. Each mode describes how the neural network will be run.
    */
    let MachineLearning_InferenceMode: MachineLearning.InferenceMode
    
    /**
    * @description Describes the current state of the MLComponent model.
    * For more information, see the [MLComponent Scripting](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/scripting-ml-component) guide.
    */
    let MachineLearning_ModelState: MachineLearning.ModelState
    
    /**
    * @description Types of output used by OutputPlaceholder.
    */
    let MachineLearning_OutputMode: MachineLearning.OutputMode
    
    /**
    * @description Handles input information from user touch input via the [TouchComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#TouchComponent) to control Scale, Rotation, and Translation of objects.
    */
    let ManipulateComponent: ManipulateComponent
    
    /**
    * @description This event is triggered when manipulation on the object ends.
    */
    let ManipulateEndEvent: ManipulateEndEvent
    
    /**
    * @description Arguments used with the `ManipulateComponent.onManipulateEnd` event.
    */
    let ManipulateEndEventArgs: ManipulateEndEventArgs
    
    /**
    * @description Result object returned from [ManipulateComponent.intersectManipulateFrame()](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ManipulateComponent).
    */
    let ManipulateFrameIntersectResult: ManipulateFrameIntersectResult
    
    /**
    * @description This event is triggered when manipulation on the object begins.
    */
    let ManipulateStartEvent: ManipulateStartEvent
    
    /**
    * @description Arguments used with the `ManipulateComponent.onManipulateStart` event.
    */
    let ManipulateStartEventArgs: ManipulateStartEventArgs
    
    /**
    * @description Enum values specifying each type of manipulation. See [ManipulateComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ManipulateComponent).
    */
    let ManipulateType: ManipulateType
    
    /**
    * @description Module for providing Map utils. 
    */
    let MapModule: MapModule
    
    let MappingOptions: MappingOptions
    
    let MappingSession: MappingSession
    
    let MappingSession_MappingThrottling: MappingSession.MappingThrottling
    
    let MapTextureProvider: MapTextureProvider
    
    /**
    * @description Defines a marker to use for Marker Tracking with [MarkerTrackingComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MarkerTrackingComponent).
    * For more information, see the [Marker Tracking](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/world/marker-tracking) guide.
    */
    let MarkerAsset: MarkerAsset
    
    /**
    * @description Base class for marker providers.
    * For more information, see the [Marker Tracking](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/world/marker-tracking) guide.
    */
    let MarkerProvider: MarkerProvider
    
    /**
    * @description Used to track images in the camera. Moves the containing object's transform to match the detected image.
    * For more information, see the [Marker Tracking](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/world/marker-tracking) guide.
    */
    let MarkerTrackingComponent: MarkerTrackingComponent
    
    /**
    * @description Clips visuals and Interaction Component events within a tree hierarchy. Any Visual or Interaction Components will be clipped user defined 2D bounds. These 2D bounds are defined by a Screen Transform. Useful for clipping some screen transforms--for example a scroll view. 
    */
    let MaskingComponent: MaskingComponent
    
    /**
    * @description A 2x2 matrix.
    */
    let mat2: mat2
    
    /**
    * @description A 3x3 matrix.
    */
    let mat3: mat3
    
    /**
    * @description A 4x4 matrix.
    */
    let mat4: mat4
    
    /**
    * @description An asset that describes how visual objects should appear.
    * Each Material is a collection of [Passes](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Pass) which define the actual rendering passes.
    * Materials are used by [MeshVisuals](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MeshVisual) for drawing meshes in the scene.
    */
    let Material: Material
    
    /**
    * @description Base class for all MeshVisual components using [Materials](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#Material) to render.
    * Comparable to the former class "MeshVisual", which was split into the classes:
    * [BaseMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#BaseMeshVisual),
    * [MaterialMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#MaterialMeshVisual),
    * and [RenderMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#RenderMeshVisual).
    */
    let MaterialMeshVisual: MaterialMeshVisual
    
    /**
    * @description Provides useful math utility functions.
    */
    let MathUtils: MathUtils
    
    /**
    * @description Settings for the physical substance, such as friction and bounciness, of a collider. If unset, uses the default matter from the world settings.
    */
    let Matter: Matter
    
    /**
    * @description Base class for Texture Providers based on selectable media.
    */
    let MediaPickerTextureProvider: MediaPickerTextureProvider
    
    /**
    * @description Computes a mel scale spectrogram - a spectrogram where the frequencies are converted to the mel scale.
    */
    let MelSpectrogram: MelSpectrogram
    
    /**
    * @description A builder class for MelSpectrogram.
    */
    let MelSpectrogramBuilder: MelSpectrogramBuilder
    
    /**
    * @description A class for generating meshes at runtime.
    */
    let MeshBuilder: MeshBuilder
    
    /**
    * @description Formats of mesh classification used by WorldRenderObjectProvider.
    */
    let MeshClassificationFormat: MeshClassificationFormat
    
    /**
    * @description Possible index data types used by [MeshBuilder](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#MeshBuilder). `MeshIndexType.UInt16` is the value normally used.
    */
    let MeshIndexType: MeshIndexType
    
    let MeshRenderObjectProvider: MeshRenderObjectProvider
    
    let MeshShadowMode: MeshShadowMode
    
    /**
    * @description Allows meshes to be used as collision shapes, for ColliderComponent and BodyComponent.
    */
    let MeshShape: MeshShape
    
    /**
    * @description Mesh topology types used by [MeshBuilder](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#MeshBuilder).
    */
    let MeshTopology: MeshTopology
    
    /**
    * @description This class has been DEPRECATED starting in Lens Studio 2.3.
    * The `Component.MeshVisual` typename is now an alias for [BaseMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#BaseMeshVisual).
    * When upgrading a project to Lens Studio 2.3 or higher, any instances of the MeshVisual component will be upgraded to [RenderMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#RenderMeshVisual).
    
    * This class was split into the following three classes, to better distinguish the behaviors of child classes.
    
    * [BaseMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#BaseMeshVisual): Base class for all visual classes using meshes to render
    
    * [MaterialMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#MaterialMeshVisual): Child class of BaseMeshVisual, gives access to the [Materials](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#Material) used to render
    
    * [RenderMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#RenderMeshVisual): Child class of MaterialMeshVisual, gives access to the [RenderMesh](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#RenderMesh) used to render
    */
    let MeshVisual: MeshVisual
    
    /**
    * @description Mel-frequency cepstral coefficients.
    */
    let MFCC: MFCC
    
    /**
    * @description Builder class for the MFCC (Mel Frequency Cepstral Co-efficients).
    */
    let MFCCBuilder: MFCCBuilder
    
    /**
    * @description The Audio Track Provider of the audio from microphone.
    */
    let MicrophoneAudioProvider: MicrophoneAudioProvider
    
    /**
    * @description Binary ML model supplied by the user.
    */
    let MLAsset: MLAsset
    
    /**
    * @description Transforms inputs (Textures or Float32Array) into outputs (Textures or Float32Array) using a neural network.
    * The neural network is represented by an MLAsset, which is set as the `model` property.
    * For more information, see the [MLComponent Overview](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/ml-component-overview).
    */
    let MLComponent: MLComponent
    
    /**
    * @description Triggered when the tracked face's mouth closes.
    */
    let MouthClosedEvent: MouthClosedEvent
    
    /**
    * @description Triggered when the tracked face's mouth opens.
    */
    let MouthOpenedEvent: MouthOpenedEvent
    
    /**
    * @description An instance of a Connected Lens session among a group of participants who were successfully invited into the experience. 
    */
    let MultiplayerSession: MultiplayerSession
    
    /**
    * @description Tracking type used by the [Device Tracking](https://lensstudio.snapchat.com/api/classes/devicetracking) component to specify what type of plane to detect. 
    */
    let NativePlaneTrackingType: NativePlaneTrackingType
    
    let NoiseReduction: NoiseReduction
    
    let NoiseReductionBuilder: NoiseReductionBuilder
    
    /**
    * @description Base class for configuring object tracking in the [ObjectTracking3D](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ObjectTracking3D) component.
    */
    let Object3DAsset: Object3DAsset
    
    /**
    * @description A reusable object hierarchy stored as a resource.
    * Can be instantiated through script or brought into the scene through Lens Studio.
    * For more information, see the [Prefabs](https://lensstudio.snapchat.com/lens-studio/references/guides/general/prefabs) guide.
    */
    let ObjectPrefab: ObjectPrefab
    
    /**
    * @description Provides additional data for the tracked object. For example, with hand tracking, you can figure out whether the tracked hand is the left hand by accessing the `isLeft` property [true/false], as well as the probability of this data through the `isLeftProbability` property [0-1]. 
    */
    let ObjectSpecificData: ObjectSpecificData
    
    /**
    * @description Used to track objects in the camera. Moves the local [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) to match the detected image.
    
    * See the [Object Tracking guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/world/object-tracking) and the [Hand Gestures Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/body/hand-gestures) for more information.
    */
    let ObjectTracking: ObjectTracking
    
    /**
    * @description Component used for tracking objects in 3D space.
    */
    let ObjectTracking3D: ObjectTracking3D
    
    /**
    * @description Strategies for updating attached objects. Used by the [ObjectTracking3D](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ObjectTracking3D) component.
    */
    let ObjectTracking3D_TrackingMode: ObjectTracking3D.TrackingMode
    
    let ObjectTrackingMaskedTextureProvider: ObjectTrackingMaskedTextureProvider
    
    let ObjectTrackingNormalsTextureProvider: ObjectTrackingNormalsTextureProvider
    
    /**
    * @description Controls a segmentation texture and its placement using information provided by Object tracking.
    */
    let ObjectTrackingTextureProvider: ObjectTrackingTextureProvider
    
    /**
    * @description Triggered when the Lens starts, earlier than all OnStart events. Also fires immediately on a newly instantiated or copied object.
    */
    let OnAwakeEvent: OnAwakeEvent
    
    /**
    * @description Triggered when the associated ScriptComponent is destroyed while the lens is running.
    */
    let OnDestroyEvent: OnDestroyEvent
    
    /**
    * @description Triggered when the `ScriptComponent` this event is bound to is disabled.
    
    */
    let OnDisableEvent: OnDisableEvent
    
    /**
    * @description Triggered when the `ScriptComponent` this event is bound to is enabled.
    
    */
    let OnEnableEvent: OnEnableEvent
    
    /**
    * @description Triggered when the Lens starts, after all OnAwakeEvents have triggered. Also triggers later on newly instantiated or copied objects.
    */
    let OnStartEvent: OnStartEvent
    
    /**
    * @description Types of operating system that may be running on the device.
    */
    let OS: OS
    
    /**
    * @description Used in [Text's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) `outlineSettings` property.
    * Configures how text outlining will appear on a Text component.
    */
    let OutlineSettings: OutlineSettings
    
    /**
    * @description Builds OutputPlaceholders for MLComponent.
    */
    let OutputBuilder: OutputBuilder
    
    /**
    * @description Provides output data from the neural network used by an MLComponent.
    * For more information, see the [MLComponent Scripting](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/scripting-ml-component) guide.
    */
    let OutputPlaceholder: OutputPlaceholder
    
    /**
    * @description Exposes state generated for ColliderComponent overlap events.  See also: [ColliderComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), [OverlapEnterEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#OverlapEnterEventArgs), [OverlapExitEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#OverlapExitEventArgs), [OverlapStayEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#OverlapStayEventArgs).
    */
    let Overlap: Overlap
    
    /**
    * @description Args used for [ColliderComponent.onOverlapEnter](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), which is triggered when the collider begins overlapping another object. Typically used for volume triggers.
    */
    let OverlapEnterEventArgs: OverlapEnterEventArgs
    
    /**
    * @description Args used for [ColliderComponent.onOverlapEnd](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), which is triggered when the collider stops overlapping another object. Typically used for volume triggers.
    */
    let OverlapExitEventArgs: OverlapExitEventArgs
    
    /**
    * @description Args used for [ColliderComponent.onOverlapStay](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ColliderComponent), which is triggered every frame while the collider continues overlapping another object. Typically used for volume triggers.
    */
    let OverlapStayEventArgs: OverlapStayEventArgs
    
    /**
    * @description Controls how a mesh will get rendered. Each Pass acts as an interface for the shader it's associated with.
    * Any properties on a Pass's shader will automatically become properties on that Pass.
    * For example, if the shader defines a variable named `baseColor`, a script would be able to access that property as `material.mainPass.baseColor`.
    */
    let Pass: Pass
    
    /**
    * @description Similar to [Pass](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Pass), except used by [VFX Assets](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#VFXAsset).
    */
    let PassWrapper: PassWrapper
    
    /**
    * @description Allows for retrieval of a collection of Pass objects used by VFXAsset
    */
    let PassWrappers: PassWrappers
    
    /**
    * @description Allows data to be stored and retrieved between Lens sessions.
    * In other words, data can be saved on device and loaded back in the next time the Lens is opened.
    * Can be accessed with `global.persistentStorageSystem`.
    
    * See the [Persistent Storage guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/persistent-storage) for more information.
    */
    let PersistentStorageSystem: PersistentStorageSystem
    
    /**
    * @description Namespace containing physics classes and static physics methods.
    */
    let Physics: Physics
    
    /**
    * @description Namespace containing static helper methods for [Constraints](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Constraint).
    */
    let Physics_Constraint: Physics.Constraint
    
    /**
    * @description Constraint type used by a [Constraint](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Contraint).   See also: [Physics.Constraint.create()](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#PhysicsConstraint), [ConstraintComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ConstraintComponent).
    */
    let Physics_ConstraintType: Physics.ConstraintType
    
    /**
    * @description Script interface for applying collision filtering to colliders and ray/shape-casts.
    */
    let Physics_Filter: Physics.Filter
    
    /**
    * @description Type of force to use when applying force or torque to a [BodyComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#BodyComponent).
    */
    let Physics_ForceMode: Physics.ForceMode
    
    /**
    * @description Stores reusable settings uniform for a world (such as gravity magnitude and direction). See also: [WorldComponent.worldSettings](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#WorldComponent).
    */
    let Physics_WorldSettingsAsset: Physics.WorldSettingsAsset
    
    /**
    * @description Attaches the SceneObject to the mesh surface of a different SceneObject.
    * See the [Pin To Mesh](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/scene-set-up/3d/pin-to-mesh#adding-a-pin-to-mesh-component) guide for more information.
    */
    let PinToMeshComponent: PinToMeshComponent
    
    /**
    * @description Used with [PinToMeshComponent.orientation](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#PinToMeshComponent).
    */
    let PinToMeshComponent_Orientation: PinToMeshComponent.Orientation
    
    let PitchShifter: PitchShifter
    
    let PitchShifterBuilder: PitchShifterBuilder
    
    let PlaybackMode: PlaybackMode
    
    /**
    * @description Represents 3D locations of stationary features in the environment. The resulting cloud of points provides a sparse description of the 3D environment.
    
    */
    let PointCloud: PointCloud
    
    /**
    * @description A type of constraint that only allows rotation.  See also: [ConstraintComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ConstraintComponent).
    */
    let PointConstraint: PointConstraint
    
    /**
    * @description An Audio Component effect that allows the Lens to simulate sound based on the direction of the Audio Listener relative to the Audio Component.
    */
    let PositionEffect: PositionEffect
    
    /**
    * @description Uses an input color lookup table image to adjust the coloring of the Lens.
    * See the [Color Correction Post Effect guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/graphics/materials/post-effects#color-correction) for more information.
    */
    let PostEffectVisual: PostEffectVisual
    
    /**
    * @description Performs collision tests (such as ray casts) in one or more world. See [Physics.createGlobalProbe()](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#Physics), [Physics.createRootProbe()](https://lensstudio.snapchat.com/api/lens-studio/Classes/OtherClasses#Physics), [WorldComponent.createProbe()](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#WorldComponent).
    */
    let Probe: Probe
    
    /**
    * @description RenderObjectProvider for mesh objects generated procedurally.
    */
    let ProceduralMeshRenderObjectProvider: ProceduralMeshRenderObjectProvider
    
    /**
    * @description Provides a texture that can be written to or read from. Can be accessed using Texture.control on a Procedural Texture.
    */
    let ProceduralTextureProvider: ProceduralTextureProvider
    
    let Properties: Properties
    
    let PropertyOnEventArgs: PropertyOnEventArgs
    
    /**
    * @description Base class for all resource providers.
    */
    let Provider: Provider
    
    /**
    * @description A quaternion, used to represent rotation.
    */
    let quat: quat
    
    /**
    * @description The base class for animation tracks using quaternion values.
    */
    let QuaternionAnimationTrack: QuaternionAnimationTrack
    
    /**
    * @description Represents an animation track using quaternion value keyframes.
    */
    let QuaternionAnimationTrackKeyFramed: QuaternionAnimationTrackKeyFramed
    
    /**
    * @description Represents a rotation animation track using euler angles.
    */
    let QuaternionAnimationTrackXYZEuler: QuaternionAnimationTrackXYZEuler
    
    /**
    * @description Hit results of a ray-cast provided to script. See [Probe](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Probe).
    */
    let RayCastHit: RayCastHit
    
    /**
    * @description The options for the realtime store.
    */
    let RealtimeStoreCreateOptions: RealtimeStoreCreateOptions
    
    /**
    * @description The ownership model of a realtime store.
    */
    let RealtimeStoreCreateOptions_Ownership: RealtimeStoreCreateOptions.Ownership
    
    /**
    * @description The persistence model for a realtime store.
    */
    let RealtimeStoreCreateOptions_Persistence: RealtimeStoreCreateOptions.Persistence
    
    /**
    * @description An axis aligned rectangle.
    * Used by `anchors` and `offsets` in [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) to represent screen boundaries.
    * Rect can only store finite numbers in the range Number.MIN_VALUE to Number.MAX_VALUE.
    
    */
    let Rect: Rect
    
    /**
    * @description Applies ScreenTransform positioning to match the cropped region of a texture.
    * For more information, see the [Crop Textures](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/crop-textures) guide.
    */
    let RectangleSetter: RectangleSetter
    
    /**
    * @description Texture Provider providing a cropped region of the input texture. The region is specified by the cropRect in local space and rotation.
    * Can be accessed using Texture.control on a RectCropTexture asset, such as a Screen Crop Texture.
    * For more information, see the [Crop Textures](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/crop-textures) guide.
    */
    let RectCropTextureProvider: RectCropTextureProvider
    
    let RemoteApiRequest: RemoteApiRequest
    
    let RemoteApiResponse: RemoteApiResponse
    
    /**
    * @description Provides access to a remote media.
    */
    let RemoteMediaModule: RemoteMediaModule
    
    /**
    * @description Provides a reference to a remote asset (i.e. assets outside of the Lens size limit) that can be downloaded at runtime using script.
    */
    let RemoteReferenceAsset: RemoteReferenceAsset
    
    let RemoteServiceModule: RemoteServiceModule
    
    /**
    * @description Represents a mesh asset.
    * See also: [RenderMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#RenderMeshVisual).
    */
    let RenderMesh: RenderMesh
    
    /**
    * @description Renders a [RenderMesh](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#RenderMesh) asset in the scene.
    * Comparable to the former class "MeshVisual", which was split into the classes:
    * [BaseMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#BaseMeshVisual),
    * [MaterialMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#MaterialMeshVisual),
    * and [RenderMeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#RenderMeshVisual).
    */
    let RenderMeshVisual: RenderMeshVisual
    
    /**
    * @description Provider for RenderMesh data.
    */
    let RenderObjectProvider: RenderObjectProvider
    
    /**
    * @description Controls a camera texture resource.
    * Can be accessed through [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Texture#control-textureprovider) on a Camera texture.
    * For more information, see the [Camera and Layers](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/scene-set-up/camera) guide.
    */
    let RenderTargetProvider: RenderTargetProvider
    
    /**
    * @description Specifies what kind of render target it is. Some texture types, for example TextureCubemap, need additional properties set on the rendering camera to work correctly.
    */
    let RenderTargetProvider_TextureType: RenderTargetProvider.TextureType
    
    /**
    * @description Visual effect used to add subtle retouching effects to detected faces (soft skin, teeth whitening, etc.).
    * To learn more, visit the [Retouch Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/face/face-effects/face-retouch).
    */
    let RetouchVisual: RetouchVisual
    
    let RotatedRect: RotatedRect
    
    /**
    * @description Used with [DeviceTracking.rotationOptions](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#DeviceTracking) to change settings for Rotation tracking mode.
    */
    let RotationOptions: RotationOptions
    
    let Sampler: Sampler
    
    /**
    * @description Class for building Sampler.
    */
    let SamplerBuilder: SamplerBuilder
    
    /**
    * @description An accessor for Pass.samplers when using PassWrappers
    */
    let SamplerWrapper: SamplerWrapper
    
    let SamplerWrappers: SamplerWrappers
    
    /**
    * @exposes user data
    
    * @description Asset for detecting an object through the Scan system.
    */
    let ScanModule: ScanModule
    
    /**
    * @description Contexts used in `ScanModule.scan()`.
    */
    let ScanModule_Contexts: ScanModule.Contexts
    
    /**
    * @description The base class for scenewide events.  SceneEvents can be created using [ScriptComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScriptComponent)'s [createEvent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#scriptcomponent--createEvent) method.
    */
    let SceneEvent: SceneEvent
    
    /**
    * @description An object in the scene hierarchy, containing a [Transform](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Transform) and possibly [Components](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components).
    * A script can access the SceneObject holding it through the method `script.getSceneObject()`.
    */
    let SceneObject: SceneObject
    
    /**
    * @description Base class for all object-based Event types in Lens Studio (ManipulateStartEvent, TapEvent, etc.).
    */
    let SceneObjectEvent: SceneObjectEvent
    
    /**
    * @description Overrides the settings on a local [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) to fit a screen region on the device.
    * See the [Screen Transform guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/scene-set-up/2d/screen-transform-overview) for more information.
    */
    let ScreenRegionComponent: ScreenRegionComponent
    
    /**
    * @description Types of screen regions that can be used with [ScreenRegionComponent](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenRegionComponent).
    */
    let ScreenRegionType: ScreenRegionType
    
    /**
    * @description Texture providing the current Render Target being rendered.
    */
    let ScreenTextureProvider: ScreenTextureProvider
    
    /**
    * @description Used for positioning objects in 2d screen space. It overrides the regular [Transform](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#Transform) component on the SceneObject it's attached to.
    
    * See the [Screen Transform guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/scene-set-up/2d/screen-transform-overview) for more information.
    */
    let ScreenTransform: ScreenTransform
    
    /**
    * @description Represents a JavaScript script which can be used to add logic in your Lens.
    */
    let ScriptAsset: ScriptAsset
    
    /**
    * @description Binds scripts to Events and executes them when triggered. Any script can access the ScriptComponent executing them through the variable `script`.
    * See also: [Scripting Overview](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/scripting-overview#script-components), [Script Events Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/adding-interactivity/script-events#introduction).
    */
    let ScriptComponent: ScriptComponent
    
    /**
    * @description Base class for objects representing Script data.
    */
    let ScriptObject: ScriptObject
    
    /**
    * @description Represents the Lens scene. Accessible through `global.scene`.
    */
    let ScriptScene: ScriptScene
    
    /**
    * @description Segmentation model used for [SegmentationTextureProvider](https://lensstudio.snapchat.com/api/lens-studio/Classes/Providers#SegmentationTextureProvider).
    */
    let SegmentationModel: SegmentationModel
    
    /**
    * @description Controls a segmentation texture resource.
    * Can be accessed through [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Texture#control-textureprovider) on a Segmentation texture.
    * For more information, see the [Segmentation](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/segmentation/fullscreen-segmentation) guide.
    */
    let SegmentationTextureProvider: SegmentationTextureProvider
    
    /**
    * @description Arguments used with the `InteractionComponent.onSelectEnd` event.
    */
    let SelectEndEventArgs: SelectEndEventArgs
    
    /**
    * @description Arguments used with the `InteractionComponent.onSelectStart` event.
    */
    let SelectStartEventArgs: SelectStartEventArgs
    
    /**
    * @description Low-level data class.
    */
    let SerializableWithUID: SerializableWithUID
    
    /**
    * @description Base type for collision shapes.
    */
    let Shape: Shape
    
    let SinglePlaneTrackingUpdatedEvent: SinglePlaneTrackingUpdatedEvent
    
    /**
    * @description Represents skinning data for rigged meshes. See also: [MeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MeshVisual).
    */
    let Skin: Skin
    
    /**
    * @description Triggered when a smile ends on the tracked face.
    */
    let SmileFinishedEvent: SmileFinishedEvent
    
    /**
    * @description Triggered when a smile begins on the tracked face.
    */
    let SmileStartedEvent: SmileStartedEvent
    
    /**
    * @description Details about friendship between the active user and another user.
    */
    let SnapchatFriendUserInfo: SnapchatFriendUserInfo
    
    /**
    * @description Represents a Snapchat user in order to pass to other APIs or to retrieve certain details about the user like display name. 
    */
    let SnapchatUser: SnapchatUser
    
    /**
    * @description Set the bone on the skin.
    */
    let SnapchatUserBirthday: SnapchatUserBirthday
    
    /**
    * @description Provides a marker for tracking Snapcodes.
    * For more information, see the [Marker Tracking](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/tracking/world/marker-tracking) guide.
    */
    let SnapcodeMarkerProvider: SnapcodeMarkerProvider
    
    /**
    * @description Called when the user taps on the capture button to record an image.
    */
    let SnapImageCaptureEvent: SnapImageCaptureEvent
    
    /**
    * @description Gets called when the user starts long pressing the capture button to record a Snap.
    */
    let SnapRecordStartEvent: SnapRecordStartEvent
    
    /**
    * @description Gets called when the user stops long pressing the Snap button to end recording of a Snap.
    */
    let SnapRecordStopEvent: SnapRecordStopEvent
    
    /**
    * @description Order that inTensor will be sorted when applying `TensorMath.argSortMasked()`.
    */
    let SortOrder: SortOrder
    
    let SpatialAudio: SpatialAudio
    
    let SpatialMap: SpatialMap
    
    /**
    * @description Representation the signal strength over time at various frequencies present in a particular waveform. Created by applying Fast Fourier Transform (FFT) on the overlapping segments of the audio data.
    */
    let Spectrogram: Spectrogram
    
    /**
    * @description The builder class for Spectrogram.
    */
    let SpectrogramBuilder: SpectrogramBuilder
    
    /**
    * @description A sphere collision shape.
    */
    let SphereShape: SphereShape
    
    let SplineComponent: SplineComponent
    
    /**
    * @description Represents transform data for screen-aligned 2D sprites. Use on SceneObjects with a SpriteVisual Component.
    * See also: [SpriteVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#deprecated-spritevisual).
    */
    let SpriteAligner: SpriteAligner
    
    /**
    * @description Stencil buffer clear option.
    */
    let StencilClearOption: StencilClearOption
    
    /**
    * @description Specifies whether the front and/or back face stencil test will be applied. The initial value is "FrontAndBack"
    */
    let StencilFace: StencilFace
    
    /**
    * @description Specifies the stencil test function. The initial value is "Always".
    */
    let StencilFunction: StencilFunction
    
    /**
    * @description Options for specifying the action to take when stencil and depth tests resolve.
    */
    let StencilOperation: StencilOperation
    
    /**
    * @description The stencil test state for Pass.
    */
    let StencilState: StencilState
    
    /**
    * @description Settings for saving values in a MultiplayerSession.
    */
    let StorageOptions: StorageOptions
    
    /**
    * @description Specifies a scope for storing or retrieving values from a MultiplayerSession.
    */
    let StorageScope: StorageScope
    
    /**
    * @description Options for stretching a mesh to fit a [ScreenTransform's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) bounding box.
    * Used in [MeshVisual's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MeshVisual) `stretchMode` property, as long as the SceneObject has a [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) attached.
    * Also used in [TextFill's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#TextFill) `textureStretch` property to control texture stretching when drawing text.
    
    * See the [Image guide](https://lensstudio.snapchat.com/lens-studio/references/guides/adding-content/2d/image) for more information about stretch modes.
    */
    let StretchMode: StretchMode
    
    let Studio: Studio
    
    /**
    * @description Used with [DeviceTracking.surfaceOptions](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#DeviceTracking) to change settings for Surface tracking mode.
    */
    let SurfaceOptions: SurfaceOptions
    
    /**
    * @description If a [DeviceTracking](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#DeviceTracking) component is present in the scene, this event is triggered when
    * the tracking is restarted (typically when the Lens starts, or if the user taps the ground).
    */
    let SurfaceTrackingResetEvent: SurfaceTrackingResetEvent
    
    let TangentType: TangentType
    
    /**
    * @description This event is triggered when the user taps on the screen.
    */
    let TapEvent: TapEvent
    
    /**
    * @description Arguments used with the `InteractionComponent.onTap` event.
    */
    let TapEventArgs: TapEventArgs
    
    /**
    * @description Namespace for mathematical operations on tensors. Useful with MLComponent.
    
    **Tensor broadcasting rules**
    
    * Tensor (channels, width, height) op Tensor (1, 1, 1)  =  the same as applying op with scalar
    
    * Tensor (channels, width, height) op Tensor (channels, 1, 1)  =  the same as applying op per channel
    */
    let TensorMath: TensorMath
    
    let TensorMath_BorderType: TensorMath.BorderType
    
    let TensorMath_ThresholdMethod: TensorMath.ThresholdMethod
    
    /**
    * @description Visual component that renders dynamic text.
    * See the [Text guide](https://lensstudio.snapchat.com/lens-studio/references/guides/adding-content/text/2d-text) for more information.
    */
    let Text: Text
    
    /**
    * @description Renders a given text with a 3D mesh.
    */
    let Text3D: Text3D
    
    /**
    * @description Fill settings used by several text related classes.
    * Used in [Text's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) `textFill` property, [DropshadowSettings'](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#DropshadowSettings) `fill` property, and [OutlineSettings'](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#OutlineSettings) `fill` property.
    */
    let TextFill: TextFill
    
    /**
    * @description Used in [TextFill's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#TextFill) `mode` property. Controls which drawing method is used for the TextFill.
    */
    let TextFillMode: TextFillMode
    
    /**
    * @description Provides access to the device's input system. Accessible through `global.textInputSystem`. 
    */
    let TextInputSystem: TextInputSystem
    
    /**
    * @description The settings used for how the input keyboard should work.
    */
    let TextInputSystem_KeyboardOptions: TextInputSystem.KeyboardOptions
    
    /**
    * @description The different input style of keyboard input.
    */
    let TextInputSystem_KeyboardType: TextInputSystem.KeyboardType
    
    /**
    * @description The return key style of keyboard input.
    */
    let TextInputSystem_ReturnKeyType: TextInputSystem.ReturnKeyType
    
    /**
    * @description Controls a text rendering texture. Can be accessed through the main rendering pass on a [Label](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#deprecated-label) component.
    * The properties here mirror those on Label.
    */
    let TextProvider: TextProvider
    
    let TextToSpeech: TextToSpeech
    
    /**
    * @description Provides the configuration for the [Text to Speech Module](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#TextToSpeechModule).   It is used to control the language of the generated voice, the voice’s style and pace.
    */
    let TextToSpeech_Options: TextToSpeech.Options
    
    /**
    * @description Provides a map of what phoneme is said at what time in the synthesized speech.
    */
    let TextToSpeech_PhonemeInfo: TextToSpeech.PhonemeInfo
    
    /**
    * @description The voice style on which the TextToSpeech will be synthesized, Varying from neutral style, to a more elaborated styles depending on the voice. Note that Sam’s voice still doesn’t have more than the neutral style, this will be added throughout the coming releases.
    */
    let TextToSpeech_VoiceNames: TextToSpeech.VoiceNames
    
    /**
    * @description Parameter returned in the `onTTSCompleteHandler` callback providing timing details for word pronunciation. 
    */
    let TextToSpeech_WordInfo: TextToSpeech.WordInfo
    
    /**
    * @description Allows generation of speech from a given text. You can use only one `TextToSpeechModule` in a Lens. However, its methods can be called multiple times in parallel if needed. 
    */
    let TextToSpeechModule: TextToSpeechModule
    
    /**
    * @description Represents a texture file asset.
    */
    let Texture: Texture
    
    /**
    * @description The base class for specialized Texture providers. Can be accessed through [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Texture#control-textureprovider).
    */
    let TextureProvider: TextureProvider
    
    /**
    * @description Defines the bounding area used for texture tiling with [TextFill's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#TextFill) `tileZone` property.
    */
    let TileZone: TileZone
    
    /**
    * @description This provider is returned by `global.touchSystem` and allows your lens to handle any touches on the screen, and optionally let certain touch types to pass through (let Snapchat handle the touch).
    
    * Possible TouchType exception values:
    
    * "TouchTypeNone"
    * "TouchTypeTouch"
    * "TouchTypeTap"
    * "TouchTypeDoubleTap"
    * "TouchTypeScale"
    * "TouchTypePan"
    * "TouchTypeSwipe"
    */
    let TouchDataProvider: TouchDataProvider
    
    /**
    * @description Triggered when a touch event ends.
    */
    let TouchEndEvent: TouchEndEvent
    
    let TouchEndEventArgs: TouchEndEventArgs
    
    /**
    * @description Triggered when a touch position on the screen is moved.
    */
    let TouchMoveEvent: TouchMoveEvent
    
    /**
    * @description Arguments used with the `InteractionComponent.onTouchMove` event.
    */
    let TouchMoveEventArgs: TouchMoveEventArgs
    
    /**
    * @description Triggered when a touch event starts.
    */
    let TouchStartEvent: TouchStartEvent
    
    /**
    * @description Arguments used with the `InteractionComponent.onTouchStart` event.
    */
    let TouchStartEventArgs: TouchStartEventArgs
    
    /**
    * @description Represents a mesh generated by world tracking. Only available when world mesh tracking is supported and enabled.
    */
    let TrackedMesh: TrackedMesh
    
    /**
    * @description Classifications of mesh face. See [TrackedMeshHitTestResult](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#TrackedMeshHitTestResult).
    */
    let TrackedMeshFaceClassification: TrackedMeshFaceClassification
    
    /**
    * @description Provides histogram information about tracked world mesh faces in a given area. Returned by `DeviceTracking.calculateWorldMeshHistogram()`.
    */
    let TrackedMeshHistogramResult: TrackedMeshHistogramResult
    
    /**
    * @description Provides information about a TrackedMesh surface hit during a raycast. Is returned in an array when calling `DeviceTracking.hitTestWorldMesh()` or `DeviceTracking.raycastWorldMesh()`.
    */
    let TrackedMeshHitTestResult: TrackedMeshHitTestResult
    
    /**
    * @description A representation for plane detected by native tracking. Can be used with TrackedPoint.
    */
    let TrackedPlane: TrackedPlane
    
    let TrackedPlaneOrientation: TrackedPlaneOrientation
    
    /**
    * @description A point on the real world not attached to any detected plane.
    */
    let TrackedPoint: TrackedPoint
    
    /**
    * @description Allows you to bind the position and rotation of an object with this component to a [TrackedPoint](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#TrackedPoint).
    */
    let TrackedPointComponent: TrackedPointComponent
    
    /**
    * @description Controls the position, rotation, and scale of a [SceneObject](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#SceneObject).  Every SceneObject automatically has a Transform attached.
    */
    let Transform: Transform
    
    /**
    * @description Applies additional transform processing on data for InputPlaceholders and OutputPlaceholders used with MLComponent.
    * For more information, see the [MLComponent Overview](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/machine-learning/ml-component/ml-component-overview).
    */
    let Transformer: Transformer
    
    /**
    * @description Builds Transformer objects for MLComponent.
    */
    let TransformerBuilder: TransformerBuilder
    
    /**
    * @description Rotation types used by TransformBuilder.
    */
    let TransformerRotation: TransformerRotation
    
    /**
    * @description Triangle hit information, available when a ray cast intersects a collision mesh.
    */
    let TriangleHit: TriangleHit
    
    /**
    * @description Gets called when the user triggers the primary input on their device. For example touch on touch screens.
    */
    let TriggerPrimaryEvent: TriggerPrimaryEvent
    
    /**
    * @description Arguments used with the `InteractionComponent.onTriggerPrimary` event.
    */
    let TriggerPrimaryEventArgs: TriggerPrimaryEventArgs
    
    /**
    * @description Triggered when the lens turns off.
    */
    let TurnOffEvent: TurnOffEvent
    
    /**
    * @description Triggered every frame.
    */
    let UpdateEvent: UpdateEvent
    
    /**
    * @description Provides information about the user such as display name, birthday, and current weather. Accessible through `global.userContextSystem`.
    
    * All callbacks will execute as soon as the requested information is available. In some rare cases, the requested information may be completely unavailable, and the callback will never occur.
    
    * Note that formatted or localized strings may appear differently to users depending on their region.
    */
    let UserContextSystem: UserContextSystem
    
    /**
    * @description A two dimensional vector.
    * Vectors can only store finite numbers in the range Number.MIN_VALUE to Number.MAX_VALUE.
    
    */
    let vec2: vec2
    
    /**
    * @description Represents an animation track using vec2 value keyframes.
    */
    let Vec2AnimationTrack: Vec2AnimationTrack
    
    /**
    * @description Represents an animation track using vec2 value keyframes.
    */
    let Vec2AnimationTrackKeyFramed: Vec2AnimationTrackKeyFramed
    
    /**
    * @description A three dimensional vector. 
    * Vectors can only store finite numbers in the range Number.MIN_VALUE to Number.MAX_VALUE.
    
    */
    let vec3: vec3
    
    /**
    * @description Represents an animation track using vec3 value keyframes.
    */
    let Vec3AnimationTrack: Vec3AnimationTrack
    
    /**
    * @description Represents an animation track using vec3 value keyframes.
    */
    let Vec3AnimationTrackKeyFramed: Vec3AnimationTrackKeyFramed
    
    /**
    * @description Represents an animation track using vec3 animation tracks.
    */
    let Vec3AnimationTrackXYZ: Vec3AnimationTrackXYZ
    
    /**
    * @description A four dimensional vector.
    * Vectors can only store finite numbers in the range Number.MIN_VALUE to Number.MAX_VALUE.
    
    */
    let vec4: vec4
    
    /**
    * @description Represents an animation track using vec4 value keyframes.
    */
    let Vec4AnimationTrack: Vec4AnimationTrack
    
    /**
    * @description Represents an animation track using vec4 value keyframes.
    */
    let Vec4AnimationTrackKeyFramed: Vec4AnimationTrackKeyFramed
    
    /**
    * @description A vector containing 4 boolean values.
    */
    let vec4b: vec4b
    
    /**
    * @description Used to help control vertex animations on the SceneObject.
    */
    let VertexCache: VertexCache
    
    /**
    * @description Provides settings for vertex physics in the ClothVisual component.
    */
    let VertexSimulationSettings: VertexSimulationSettings
    
    /**
    * @description Used by the `verticalAlignment` property in [MeshVisual](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#MeshVisual).
    * When a [ScreenTransform](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#ScreenTransform) is attached to the same SceneObject, this determines how the mesh will be positioned vertically.
    */
    let VerticalAlignment: VerticalAlignment
    
    /**
    * @description Options for handling vertical text overflow. Used by [Text component's](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Text) `verticalOverflow` property.
    */
    let VerticalOverflow: VerticalOverflow
    
    /**
    * @description Defines a VFX to use with [VFX Component](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#VFXComponent). For more information, see the [VFX Guide](https://lensstudio.snapchat.com/lens-studio/references/guides/lens-features/graphics/particles/vfx-editor/introduction-and-concepts).
    */
    let VFXAsset: VFXAsset
    
    /**
    * @description A VFX visual used to show a [VFX Asset](https://lensstudio.snapchat.com/api/lens-studio/Classes/Assets#VFXAsset).
    */
    let VFXComponent: VFXComponent
    
    /**
    * @description Describes the current status of a [VideoTextureProvider](https://lensstudio.snapchat.com/api/lens-studio/Classes/Providers#VideoTextureProvider).
    */
    let VideoStatus: VideoStatus
    
    /**
    * @description Controls a video texture resource. Can be accessed through [Texture.control](https://lensstudio.snapchat.com/api/lens-studio/Classes/Components#Texture#control-textureprovider).
    */
    let VideoTextureProvider: VideoTextureProvider
    
    /**
    * @description Base class for all visual Components (e.g. MeshVisual).
    */
    let Visual: Visual
    
    let VoiceML: VoiceML
    
    /**
    * @description Additional parameters are used to provide additional data for NlpModels and NlpResponses. 
    */
    let VoiceML_AdditionalParam: VoiceML.AdditionalParam
    
    /**
    * @description NlpModels are used to provide the VoiceML NLP engine information about how the transcript of the input audio should be processed. BaseNlpModels is the abstract base class all NlpModels ([NlpKeywordModel](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLNlpKeywordModel), [NlpIntentModel](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLNlpIntentModel)) inherit from.
    
    * You can specify multiple NlpModels to process the same audio, all of their results will be returned in [ListeningUpdateEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLListeningUpdateEventArgs)
    */
    let VoiceML_BaseNlpModel: VoiceML.BaseNlpModel
    
    /**
    * @description The abstract base class all NlpResponses inherit from. NlpResponses are used as the result from the VoiceML NLP engine with information after processing the transcript of the input audio.
    */
    let VoiceML_BaseNlpResponse: VoiceML.BaseNlpResponse
    
    /**
    * @description An NLP model used to detect keywords in the transcript of the input audio.  For example, you can have keyword detection which will trigger every time the word "red" is said, and another trigger for the word "yellow". 
    */
    let VoiceML_KeywordModelGroup: VoiceML.KeywordModelGroup
    
    /**
    * @description ListeningErrorEventArgs object returns in onListeningError callback. It contains the error code and description of the error. 
    */
    let VoiceML_ListeningErrorEventArgs: VoiceML.ListeningErrorEventArgs
    
    /**
    * @description Provides the configuration for the audio input processing output. This can either include NLP processing using the ([nlpModels](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLBaseNlpModel)) or directly retrieving the transcription.  `speechContext` provides the ability to further improve the transcription accuracy given an assumed context. 
    */
    let VoiceML_ListeningOptions: VoiceML.ListeningOptions
    
    /**
    * @description The parameter when the callback registered on from `VoiceMLModule.onListeningUpdate` is called. This is to mark the input audio transcription (and possibly NlpModels as a result) was updated.
    */
    let VoiceML_ListeningUpdateEventArgs: VoiceML.ListeningUpdateEventArgs
    
    /**
    * @description May be returned when [enableSystemCommands()](https://lensstudio.snapchat.com/api/classes/VoiceMLModule#enableSystemCommands) API is used.
    * The NlpCommandResponse contains the results of the servers' NLP command model classification on the last sentence and indicates that a predetermined system command was detected.
    * Current supported commands:
    
    * “Take a Snap”: takes a snapshot.
    * “Start Recording”: starts a video recording if not already started.
    * "Stop Recording”: stops an ongoing video recording, if applicable.
    
    */
    let VoiceML_NlpCommandResponse: VoiceML.NlpCommandResponse
    
    /**
    * @description Specifies which NLP Intent model should run to classify the transcription of the input audio. NLP Intent classification meant to extract the meaning of a sentence rather than detecting certain keywords. Multiple intent models can be used on the same transcription, and will run only on complete sentences (`isFinalTranscription = True`). Supported intent models: `VOICE_ENABLED_UI`. 
    */
    let VoiceML_NlpIntentModel: VoiceML.NlpIntentModel
    
    /**
    * @description Returned when [NlpIntentModel](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLNlpIntentModel) was specificed in the ListeningOptions, it contains the results of the NLP Intent model classification on the last sentence. `NlpIntentResponse` will only run on complete sentences (`isFinalTranscription = true`).  
    
    */
    let VoiceML_NlpIntentResponse: VoiceML.NlpIntentResponse
    
    /**
    * @description Contains helper functions for NlpIntentModel.
    */
    let VoiceML_NlpIntentsModelOptions: VoiceML.NlpIntentsModelOptions
    
    /**
    * @description Used to detect usage of certain keywords from the input audio.
    
    * Keyword detection (whose results will be returned in [NlpKeywordResponse](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLNlpKeywordResponse) in the [ListeningUpdateEventArgs](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#VoiceMLListeningUpdateEventArgs)) can happen in the mid input sentence (and in such case the the isFinalTranscription=false) or can happen at the end of the sentence (isFinalTranscription=true). Mid sentence detection have closer proximity to the time the word was spoken, but might be less accurate. 
    */
    let VoiceML_NlpKeywordModel: VoiceML.NlpKeywordModel
    
    /**
    * @description Contains helper functions for NlpKeywordModel.
    */
    let VoiceML_NlpKeywordModelOptions: VoiceML.NlpKeywordModelOptions
    
    /**
    * @description NlpKeywordResponse will be returned if KeywordModel has been supplied as an input model in the ListeningOptions.  The keyword model allows detection of keywords (or short phrases) in an input audio. 
    */
    let VoiceML_NlpKeywordResponse: VoiceML.NlpKeywordResponse
    
    /**
    * @description The NLP Response Status indicates wether the NLP was successful in parsing the sentence. 
    */
    let VoiceML_NlpResponseStatus: VoiceML.NlpResponseStatus
    
    /**
    * @description The `PostProcessingAction` is the base class for `QnaAction` and other post processing actions that need to processed after the transcription phase.
    */
    let VoiceML_PostProcessingAction: VoiceML.PostProcessingAction
    
    /**
    * @description The `PostProcessingActionResponse` is the base class for `QnaActionRespose` and other post processing actions responses. It holds the id and status properties for all deriving objects.
    */
    let VoiceML_PostProcessingActionResponse: VoiceML.PostProcessingActionResponse
    
    let VoiceML_PostProcessingActionResponseStatus: VoiceML.PostProcessingActionResponseStatus
    
    let VoiceML_QnaAction: VoiceML.QnaAction
    
    let VoiceML_QnaResponse: VoiceML.QnaResponse
    
    /**
    * @description Speech context is used in cases where specific words are expected from the users, the transcription accuracy of these words can be improved, by straightening their likelihood in context. The strength is scaled 1-10 (10 being the strongest increase) the default value is 5.
    */
    let VoiceML_SpeechContext: VoiceML.SpeechContext
    
    /**
    * @description VoiceML Module allows voice input and commands. It enables transciption of the speech, detecting keywords within the transcription, intents as well as system commands (such as "Take a Snap"). You can use one VoiceML Module per Lens. 
    */
    let VoiceMLModule: VoiceMLModule
    
    let VoiceMLModule_AnswerStatusCodes: VoiceMLModule.AnswerStatusCodes
    
    /**
    * @description Status Codes for NLP Responses. 
    */
    let VoiceMLModule_NlpResponsesStatusCodes: VoiceMLModule.NlpResponsesStatusCodes
    
    let VoiceMLModule_SpeechRecognizer: VoiceMLModule.SpeechRecognizer
    
    /**
    * @description Types of weather returned by [UserContextSystem's](https://lensstudio.snapchat.com/api/lens-studio/Classes/ScriptObjects#UserContextSystem) `requestWeatherCondition()` callback.
    */
    let WeatherCondition: WeatherCondition
    
    let WeightedMode: WeightedMode
    
    /**
    * @description Groups physics objects in its subtree into an independent world simulation.
    */
    let WorldComponent: WorldComponent
    
    /**
    * @description Holds settings for world mesh tracking in DeviceTracking component. Accessible through DeviceTracking.worldOptions.
    */
    let WorldOptions: WorldOptions
    
    /**
    * @description Provider for RenderMesh data representing the estimated shape of real world objects generated from depth information. Only available when world mesh tracking is supported and enabled.
    */
    let WorldRenderObjectProvider: WorldRenderObjectProvider
    
    /**
    * @description Provides information about whether certain world tracking features are supported by the device.
    */
    let WorldTrackingCapabilities: WorldTrackingCapabilities
    
    /**
    * @description Triggered when new world tracking meshes are detected. Only available when a Device Tracking component is in the scene, and world mesh tracking is supported and enabled.
    */
    let WorldTrackingMeshesAddedEvent: WorldTrackingMeshesAddedEvent
    
    /**
    * @description Triggered when some world tracking meshes are no longer detected. Only available when a Device Tracking component is in the scene, and world mesh tracking is supported and enabled.
    */
    let WorldTrackingMeshesRemovedEvent: WorldTrackingMeshesRemovedEvent
    
    /**
    * @description Triggered when world tracking meshes are updated. Only available when a Device Tracking component is in the scene, and world mesh tracking is supported and enabled.
    */
    let WorldTrackingMeshesUpdatedEvent: WorldTrackingMeshesUpdatedEvent
    
    /**
    * @description Triggered when plane(s) are newly detected. The `worldOptions.nativePlaneTrackingType` must be set to anything other than `NativePlaneTrackingType.None`. In addition, [Device Tracking](https://lensstudio.snapchat.com/api/classes/devicetracking) component must be set to `World` mode to orient the Camera relative to the planes correctly.
    
    */
    let WorldTrackingPlanesAddedEvent: WorldTrackingPlanesAddedEvent
    
    /**
    * @description Triggered when plane(s) are no longer detected.  This usually happens when two planes merge into one.  Planes persist when no longer seen by camera(s) and when previously detected objects move (e.g. a door is opened) to create a static scene. [Device Tracking](https://lensstudio.snapchat.com/api/classes/devicetracking) component must be set to `World` mode to orient the Camera relative to the planes correctly.
    
    */
    let WorldTrackingPlanesRemovedEvent: WorldTrackingPlanesRemovedEvent
    
    /**
    * @description Triggered when currently detected plane(s) are updated.  This usually happens when a plane grows in size. The `worldOptions.nativePlaneTrackingType` must be set to anything other than `NativePlaneTrackingType.None`. In addition, [Device Tracking](https://lensstudio.snapchat.com/api/classes/devicetracking) component must be set to `World` mode to orient the Camera relative to the planes correctly.
    
    */
    let WorldTrackingPlanesUpdatedEvent: WorldTrackingPlanesUpdatedEvent
    
    /**
    * @description Describes how a texture should be sampled when using coordinates outside of the normal range.
    */
    let WrapMode: WrapMode
    
    let Zodiac: Zodiac
    
}

