import { createClient } from 'https://esm.sh/@supabase/supabase-js@2?target=es2020';
import { SUPABASE_URL, SUPABASE_ANON_KEY } from './config.js';

document.addEventListener('DOMContentLoaded', () => {
  const counterEl = document.getElementById('counter');
  const skyEl = document.getElementById('sky');
  const rtEl = document.getElementById('rtStatus');
  const todayStrEl = document.getElementById('todayStr');

  // Format today (UTC)
  const now = new Date();
  const yyyy = now.getUTCFullYear();
  const mm = String(now.getUTCMonth() + 1).padStart(2, '0');
  const dd = String(now.getUTCDate()).padStart(2, '0');
  const todayStr = `${yyyy}-${mm}-${dd}`;

  todayStrEl.textContent = `${todayStr} (UTC)`;

  let total = 0;
  let supabase = null;

  // Initialize Supabase if creds are present
  if (SUPABASE_URL && SUPABASE_ANON_KEY) {
    supabase = createClient(SUPABASE_URL, SUPABASE_ANON_KEY);
  }

  setRealtimeStatus(false);
  setTotal(0);

  function setRealtimeStatus(on) {
    rtEl.textContent = on ? 'connected' : 'disconnected';
    rtEl.style.color = on ? '#0a8f47' : '#5b728a';
  }
  function setTotal(n) {
    total = Math.max(0, Number(n || 0));
    counterEl.textContent = total.toLocaleString();
    adjustSkyDensity(total);
  }

  function adjustSkyDensity(n) {
    let target;
    // The density logic scales the number of balloons displayed relative
    // to the total count (n) to prevent excessive DOM elements at high totals.
    if (n <= 25) {
      // 1 balloon per +1 up to 25
      target = n;
    } else {
      // After 25, add 1 balloon for every +20 beyond 25 (slower scaling)
      target = 25 + Math.floor((n - 25) / 20);
    }

    target = Math.min(150, target);

    const current = skyEl.childElementCount;
    if (current < target) {
      for (let i = current; i < target; i++) {
        skyEl.appendChild(makeBalloon());
      }
    } else if (current > target) {
      for (let i = current; i > target; i--) {
        skyEl.removeChild(skyEl.lastElementChild);
      }
    }
  }

  function makeBalloon() {
    const b = document.createElement('div');
    b.className = 'balloon';

    b.style.left = 3 + Math.random() * 94 + '%';

    const dur = 7 + Math.random() * 6;
    b.style.setProperty('--dur', dur + 's');

    const endY = -(60 + Math.random() * 120);
    b.style.setProperty('--endY', endY + '%');

    const hue = 320 + Math.floor(Math.random() * 60);
    const fill = `hsl(${hue} 100% 72%)`;
    b.style.background = fill;
    b.style.color = fill;

    const s = document.createElement('div');
    s.className = 'string';
    b.appendChild(s);
    return b;
  }

  async function init() {
    setRealtimeStatus(false);

    if (supabase == null) {
      setTotal(0);
      return;
    }

    // Initial total via RPC get_kindness_total_all
    try {
      const { data, error } = await supabase.rpc('get_kindness_total_all');
      if (error) throw error;
      setTotal(Number(data || 0));
    } catch (e) {
      console.warn('Initial load failed:', e?.message || e);
      setTotal(0);
    }

    let refreshTimer;
    const refresh = () => {
      clearTimeout(refreshTimer);
      refreshTimer = setTimeout(async () => {
        try {
          const { data, error } = await supabase.rpc('get_kindness_total_all');
          if (error) throw error;
          setTotal(Number(data || 0));
        } catch (e) {
          console.warn('Refresh failed:', e?.message || e);
        }
      }, 100); // debounce 100ms
    };

    supabase
      .channel('kindness-realtime-all')
      .on(
        'postgres_changes',
        { event: '*', schema: 'public', table: 'kindness_totals' },
        refresh
      )
      .subscribe((status) => setRealtimeStatus(status === 'SUBSCRIBED'));
  }

  init();
});
