/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.spark.bigquery.write;

import com.google.cloud.bigquery.connector.common.BigQueryClient;
import com.google.cloud.bigquery.connector.common.BigQueryConnectorException;
import com.google.cloud.bigquery.connector.common.BigQueryUtil;
import com.google.cloud.spark.bigquery.PartitionOverwriteMode;
import com.google.cloud.spark.bigquery.SchemaConverters;
import com.google.cloud.spark.bigquery.SchemaConvertersConfiguration;
import com.google.cloud.spark.bigquery.SparkBigQueryConfig;
import com.google.cloud.spark.bigquery.SparkBigQueryUtil;
import com.google.cloud.spark.bigquery.SupportedCustomDataType;
import com.google.cloud.spark.bigquery.metrics.SparkBigQueryConnectorMetricsUtils;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.Field;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.FormatOptions;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.Job;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.JobInfo;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.JobStatistics;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.Schema;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.TableId;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.TableInfo;
import com.google.cloud.spark.bigquery.repackaged.com.google.common.collect.ImmutableList;
import com.google.cloud.spark.bigquery.repackaged.com.google.common.collect.ImmutableMap;
import com.google.cloud.spark.bigquery.repackaged.com.google.common.collect.Streams;
import com.google.cloud.spark.bigquery.util.HdfsUtils;
import com.google.cloud.spark.bigquery.write.IntermediateDataCleaner;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.spark.SparkContext;
import org.apache.spark.sql.Dataset;
import org.apache.spark.sql.Row;
import org.apache.spark.sql.SQLContext;
import org.apache.spark.sql.SaveMode;
import org.apache.spark.sql.types.StructField;
import org.apache.spark.sql.types.StructType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BigQueryWriteHelper {
    private static final Logger logger = LoggerFactory.getLogger(BigQueryWriteHelper.class);
    private final BigQueryClient bigQueryClient;
    private final SaveMode saveMode;
    private final SparkBigQueryConfig config;
    private final Dataset<Row> data;
    private final Optional<TableInfo> tableInfo;
    private final Configuration conf;
    private final Path gcsPath;
    private final Optional<IntermediateDataCleaner> createTemporaryPathDeleter;
    private final Schema tableSchema;
    private final JobInfo.WriteDisposition writeDisposition;
    private Optional<TableId> temporaryTableId = Optional.empty();
    private final SparkContext sparkContext;

    public BigQueryWriteHelper(BigQueryClient bigQueryClient, SQLContext sqlContext, SaveMode saveMode, SparkBigQueryConfig config, Dataset<Row> data, Optional<TableInfo> tableInfo) {
        this.bigQueryClient = bigQueryClient;
        BigQueryWriteHelper.verifySaveMode(saveMode);
        this.saveMode = saveMode;
        this.config = config;
        this.data = data;
        this.tableInfo = tableInfo;
        this.conf = sqlContext.sparkContext().hadoopConfiguration();
        this.gcsPath = SparkBigQueryUtil.createGcsPath(config, this.conf, sqlContext.sparkContext().applicationId());
        this.createTemporaryPathDeleter = config.getTemporaryGcsBucket().map(unused -> new IntermediateDataCleaner(this.gcsPath, this.conf));
        Schema schema = SchemaConverters.from(SchemaConvertersConfiguration.from(config)).toBigQuerySchema(data.schema());
        if (tableInfo.isPresent()) {
            schema = BigQueryUtil.adjustSchemaIfNeeded(schema, ((TableDefinition)tableInfo.get().getDefinition()).getSchema(), ((ImmutableList)config.getLoadSchemaUpdateOptions()).contains((Object)JobInfo.SchemaUpdateOption.ALLOW_FIELD_RELAXATION));
        }
        this.tableSchema = schema;
        this.writeDisposition = SparkBigQueryUtil.saveModeToWriteDisposition(saveMode);
        this.sparkContext = sqlContext.sparkContext();
    }

    public void writeDataFrameToBigQuery() {
        if (this.config.getCreateDisposition().map(cd2 -> !this.tableExists() && cd2 == JobInfo.CreateDisposition.CREATE_NEVER).orElse(false).booleanValue()) {
            throw new BigQueryConnectorException(String.format("For table %s Create Disposition is CREATE_NEVER and the table does not exists. Aborting the insert", this.friendlyTableName()));
        }
        try {
            this.createTemporaryPathDeleter.ifPresent(cleaner -> Runtime.getRuntime().addShutdownHook((Thread)cleaner));
            String format = this.config.getIntermediateFormat().getDataSource();
            this.data.write().format(format).save(this.gcsPath.toString());
            if (this.writeDisposition == JobInfo.WriteDisposition.WRITE_TRUNCATE && this.config.getPartitionOverwriteModeValue() == PartitionOverwriteMode.DYNAMIC && this.bigQueryClient.tableExists(this.config.getTableId()) && this.bigQueryClient.isTablePartitioned(this.config.getTableId())) {
                this.temporaryTableId = Optional.of(this.bigQueryClient.createTempTableAfterCheckingSchema(this.config.getTableId(), this.tableSchema, this.config.getEnableModeCheckForSchemaFields()).getTableId());
                this.loadDataToBigQuery();
                Job queryJob = this.bigQueryClient.overwriteDestinationWithTemporaryDynamicPartitons(this.temporaryTableId.get(), this.config.getTableId());
                this.bigQueryClient.waitForJob(queryJob);
            } else {
                this.loadDataToBigQuery();
            }
            this.updateMetadataIfNeeded();
        }
        catch (Exception e) {
            throw new BigQueryConnectorException("Failed to write to BigQuery", e);
        }
        finally {
            this.cleanTemporaryGcsPathIfNeeded();
        }
    }

    void loadDataToBigQuery() throws IOException {
        FileSystem fs = this.gcsPath.getFileSystem(this.conf);
        FormatOptions formatOptions = this.config.getIntermediateFormat().getFormatOptions();
        String suffix = "." + formatOptions.getType().toLowerCase();
        List<String> sourceUris = SparkBigQueryUtil.optimizeLoadUriListForSpark(Streams.stream(HdfsUtils.toJavaUtilIterator(fs.listFiles(this.gcsPath, false))).map(file -> file.getPath().toString()).filter(path -> path.toLowerCase().endsWith(suffix)).collect(Collectors.toList()));
        List<String> optimizedSourceUris = SparkBigQueryUtil.optimizeLoadUriListForSpark(sourceUris);
        JobInfo.WriteDisposition writeDisposition = SparkBigQueryUtil.saveModeToWriteDisposition(this.saveMode);
        TableId destinationTableId = this.temporaryTableId.orElse(this.config.getTableId());
        JobStatistics.LoadStatistics loadStatistics = this.bigQueryClient.loadDataIntoTable(this.config, optimizedSourceUris, formatOptions, writeDisposition, Optional.of(this.tableSchema), destinationTableId);
        long currentTimeMillis = System.currentTimeMillis();
        SparkBigQueryConnectorMetricsUtils.postWriteSessionMetrics(currentTimeMillis, SparkBigQueryConfig.WriteMethod.INDIRECT, loadStatistics.getOutputBytes(), Optional.of(this.config.getIntermediateFormat()), this.sparkContext);
    }

    String friendlyTableName() {
        return BigQueryUtil.friendlyTableName(this.config.getTableId());
    }

    void updateMetadataIfNeeded() {
        BigQueryWriteHelper.updateTableMetadataIfNeeded(this.data.schema(), this.config, this.bigQueryClient);
    }

    public static void updateTableMetadataIfNeeded(StructType sparkSchema, SparkBigQueryConfig config, BigQueryClient bigQueryClient) {
        Map fieldsToUpdate = Stream.of(sparkSchema.fields()).filter(field -> {
            Optional<SupportedCustomDataType> supportedCustomDataType = SupportedCustomDataType.of(field.dataType());
            return supportedCustomDataType.isPresent() || SchemaConverters.getDescriptionOrCommentOfField(field, supportedCustomDataType).isPresent();
        }).collect(Collectors.toMap(StructField::name, Function.identity()));
        if (!fieldsToUpdate.isEmpty() || !((ImmutableMap)config.getBigQueryTableLabels()).isEmpty()) {
            TableInfo originalTableInfo = bigQueryClient.getTable(config.getTableIdWithoutThePartition());
            TableInfo.Builder updatedTableInfo = originalTableInfo.toBuilder();
            if (!fieldsToUpdate.isEmpty()) {
                logger.debug("updating schema, found fields to update: {}", fieldsToUpdate.keySet());
                Object originalTableDefinition = originalTableInfo.getDefinition();
                Schema originalSchema = ((TableDefinition)originalTableDefinition).getSchema();
                Schema updatedSchema = Schema.of(originalSchema.getFields().stream().map(field -> Optional.ofNullable(fieldsToUpdate.get(field.getName())).map(sparkSchemaField -> BigQueryWriteHelper.updatedField(field, sparkSchemaField)).orElse((Field)field)).collect(Collectors.toList()));
                updatedTableInfo.setDefinition((TableDefinition)((TableDefinition.Builder)((TableDefinition)originalTableDefinition).toBuilder().setSchema(updatedSchema)).build());
            }
            if (!((ImmutableMap)config.getBigQueryTableLabels()).isEmpty()) {
                updatedTableInfo.setLabels(config.getBigQueryTableLabels()).build();
            }
            bigQueryClient.update(updatedTableInfo.build());
        }
    }

    static Field updatedField(Field field, StructField sparkSchemaField) {
        Field.Builder newField = field.toBuilder();
        Optional<String> bqDescription = SchemaConverters.getDescriptionOrCommentOfField(sparkSchemaField, SupportedCustomDataType.of(sparkSchemaField.dataType()));
        bqDescription.ifPresent(newField::setDescription);
        return newField.build();
    }

    void cleanTemporaryGcsPathIfNeeded() {
        logger.info("Cleaning temporary GCS path");
        this.createTemporaryPathDeleter.ifPresent(IntermediateDataCleaner::deletePath);
    }

    static void verifySaveMode(SaveMode saveMode) {
        if (saveMode == SaveMode.ErrorIfExists || saveMode == SaveMode.Ignore) {
            throw new UnsupportedOperationException("SaveMode " + saveMode + " is not supported");
        }
    }

    private boolean tableExists() {
        return this.tableInfo.isPresent();
    }
}

