/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.bigquery.connector.common;

import com.google.cloud.bigquery.connector.common.BigQueryConnectorException;
import com.google.cloud.bigquery.connector.common.BigQueryErrorCode;
import com.google.cloud.bigquery.connector.common.BigQueryJobCompletionListener;
import com.google.cloud.bigquery.connector.common.BigQueryUtil;
import com.google.cloud.bigquery.connector.common.ComparisonResult;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.RetryOption;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.BigQuery;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.BigQueryException;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.BigQueryOptions;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.Clustering;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.Dataset;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.DatasetId;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.EncryptionConfiguration;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.FormatOptions;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.Job;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.JobConfiguration;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.JobId;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.JobInfo;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.JobStatistics;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.LoadJobConfiguration;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.QueryJobConfiguration;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.RangePartitioning;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.Schema;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.StandardTableDefinition;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.Table;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.TableId;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.TableInfo;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.TableResult;
import com.google.cloud.spark.bigquery.repackaged.com.google.cloud.bigquery.TimePartitioning;
import com.google.cloud.spark.bigquery.repackaged.com.google.common.base.Preconditions;
import com.google.cloud.spark.bigquery.repackaged.com.google.common.cache.Cache;
import com.google.cloud.spark.bigquery.repackaged.com.google.common.collect.ImmutableList;
import com.google.cloud.spark.bigquery.repackaged.com.google.common.collect.ImmutableMap;
import com.google.cloud.spark.bigquery.repackaged.com.google.common.collect.ImmutableSet;
import com.google.cloud.spark.bigquery.repackaged.org.threeten.bp.Duration;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BigQueryClient {
    private static final Logger log = LoggerFactory.getLogger(BigQueryClient.class);
    private static final List<Runnable> CLEANUP_JOBS = new ArrayList<Runnable>();
    private final BigQuery bigQuery;
    private final Cache<String, TableInfo> destinationTableCache;
    private final Optional<String> materializationProject;
    private final Optional<String> materializationDataset;
    private final JobConfigurationFactory jobConfigurationFactory;
    private final Optional<BigQueryJobCompletionListener> jobCompletionListener;
    private final long bigQueryJobTimeoutInMinutes;

    public BigQueryClient(BigQuery bigQuery, Optional<String> materializationProject, Optional<String> materializationDataset, Cache<String, TableInfo> destinationTableCache, Map<String, String> labels, QueryJobConfiguration.Priority queryJobPriority, Optional<BigQueryJobCompletionListener> jobCompletionListener, long bigQueryJobTimeoutInMinutes) {
        this.bigQuery = bigQuery;
        this.materializationProject = materializationProject;
        this.materializationDataset = materializationDataset;
        this.destinationTableCache = destinationTableCache;
        this.jobConfigurationFactory = new JobConfigurationFactory(labels, queryJobPriority);
        this.jobCompletionListener = jobCompletionListener;
        this.bigQueryJobTimeoutInMinutes = bigQueryJobTimeoutInMinutes;
    }

    public static synchronized void runCleanupJobs() {
        log.info("Running cleanup jobs. Jobs count is " + CLEANUP_JOBS.size());
        for (Runnable job : CLEANUP_JOBS) {
            try {
                job.run();
            }
            catch (Exception e) {
                log.warn("Caught exception while running cleanup job. Continue to run the rest of the jobs", (Throwable)e);
            }
        }
        log.info("Clearing the cleanup jobs list");
        CLEANUP_JOBS.clear();
        log.info("Finished to run cleanup jobs.");
    }

    public JobInfo waitForJob(Job job) {
        try {
            Job completedJob = job.waitFor(RetryOption.initialRetryDelay(Duration.ofSeconds(1L)), RetryOption.totalTimeout(Duration.ofMinutes(this.bigQueryJobTimeoutInMinutes)));
            if (completedJob == null || completedJob.getStatus().getError() != null) {
                throw new UncheckedIOException(new IOException(completedJob != null ? completedJob.getStatus().getError().toString() : null));
            }
            if (!completedJob.isDone()) {
                completedJob.cancel();
                throw new IllegalStateException(String.format("Job aborted due to timeout  : %s minutes", this.bigQueryJobTimeoutInMinutes));
            }
            this.jobCompletionListener.ifPresent(jcl -> jcl.accept(completedJob));
            return completedJob;
        }
        catch (InterruptedException e) {
            throw new RuntimeException("Could not copy table from temporary sink to destination table.", e);
        }
    }

    private static Optional<String> createWhereClause(String[] filters) {
        if (filters.length == 0) {
            return Optional.empty();
        }
        return Optional.of(Stream.of(filters).collect(Collectors.joining(") AND (", "(", ")")));
    }

    public TableInfo getTable(TableId tableId) {
        return this.bigQuery.getTable(tableId, new BigQuery.TableOption[0]);
    }

    public boolean tableExists(TableId tableId) {
        return this.getTable(tableId) != null;
    }

    public TableInfo createTable(TableId tableId, Schema schema, CreateTableOptions options) {
        StandardTableDefinition.Builder tableDefinition = StandardTableDefinition.newBuilder().setSchema(schema);
        options.getClusteredFields().ifPresent(clusteredFields -> tableDefinition.setClustering(Clustering.newBuilder().setFields((List<String>)clusteredFields).build()));
        TableInfo.Builder tableInfo = TableInfo.newBuilder(tableId, tableDefinition.build());
        options.getKmsKeyName().ifPresent(keyName -> tableInfo.setEncryptionConfiguration(EncryptionConfiguration.newBuilder().setKmsKeyName((String)keyName).build()));
        if (!options.getBigQueryTableLabels().isEmpty()) {
            tableInfo.setLabels(options.getBigQueryTableLabels());
        }
        return this.bigQuery.create(tableInfo.build(), new BigQuery.TableOption[0]);
    }

    public TableInfo createTempTable(TableId destinationTableId, Schema schema) {
        TableId tempTableId = this.createTempTableId(destinationTableId);
        TableInfo tableInfo = TableInfo.newBuilder(tempTableId, StandardTableDefinition.of(schema)).build();
        Table tempTable = this.bigQuery.create(tableInfo, new BigQuery.TableOption[0]);
        CLEANUP_JOBS.add(() -> this.deleteTable(tempTable.getTableId()));
        return tempTable;
    }

    public TableInfo createTempTableAfterCheckingSchema(TableId destinationTableId, Schema schema, boolean enableModeCheckForSchemaFields) throws IllegalArgumentException {
        TableInfo destinationTable = this.getTable(destinationTableId);
        Schema tableSchema = ((TableDefinition)destinationTable.getDefinition()).getSchema();
        ComparisonResult schemaWritableResult = BigQueryUtil.schemaWritable(schema, tableSchema, false, enableModeCheckForSchemaFields);
        Preconditions.checkArgument(schemaWritableResult.valuesAreEqual(), new BigQueryConnectorException.InvalidSchemaException("Destination table's schema is not compatible with dataframe's schema. " + schemaWritableResult.makeMessage()));
        return this.createTempTable(destinationTableId, schema);
    }

    public TableId createTempTableId(TableId destinationTableId) {
        String tempProject = this.materializationProject.orElseGet(destinationTableId::getProject);
        String tempDataset = this.materializationDataset.orElseGet(destinationTableId::getDataset);
        String tableName = destinationTableId.getTable() + System.nanoTime();
        TableId tempTableId = tempProject == null ? TableId.of(tempDataset, tableName) : TableId.of(tempProject, tempDataset, tableName);
        return tempTableId;
    }

    public boolean deleteTable(TableId tableId) {
        log.info("Deleting table " + BigQueryClient.fullTableName(tableId));
        return this.bigQuery.delete(tableId);
    }

    private Job copyData(TableId sourceTableId, TableId destinationTableId, JobInfo.WriteDisposition writeDisposition) {
        String queryFormat = "SELECT * FROM `%s`";
        String temporaryTableName = BigQueryClient.fullTableName(sourceTableId);
        String sqlQuery = String.format(queryFormat, temporaryTableName);
        QueryJobConfiguration queryConfig = this.jobConfigurationFactory.createQueryJobConfigurationBuilder(sqlQuery, Collections.emptyMap()).setUseLegacySql(false).setDestinationTable(destinationTableId).setWriteDisposition(writeDisposition).build();
        return this.create(JobInfo.newBuilder(queryConfig).build());
    }

    public Job overwriteDestinationWithTemporaryDynamicPartitons(TableId temporaryTableId, TableId destinationTableId) {
        Object destinationDefinition = this.getTable(destinationTableId).getDefinition();
        String sqlQuery = null;
        if (destinationDefinition instanceof StandardTableDefinition) {
            String destinationTableName = BigQueryClient.fullTableName(destinationTableId);
            String temporaryTableName = BigQueryClient.fullTableName(temporaryTableId);
            StandardTableDefinition sdt = (StandardTableDefinition)destinationDefinition;
            TimePartitioning timePartitioning = sdt.getTimePartitioning();
            if (timePartitioning != null) {
                sqlQuery = BigQueryUtil.getQueryForTimePartitionedTable(destinationTableName, temporaryTableName, sdt, timePartitioning);
            } else {
                RangePartitioning rangePartitioning = sdt.getRangePartitioning();
                if (rangePartitioning != null) {
                    sqlQuery = BigQueryUtil.getQueryForRangePartitionedTable(destinationTableName, temporaryTableName, sdt, rangePartitioning);
                }
            }
            if (sqlQuery != null) {
                QueryJobConfiguration queryConfig = this.jobConfigurationFactory.createQueryJobConfigurationBuilder(sqlQuery, Collections.emptyMap()).setUseLegacySql(false).build();
                return this.create(JobInfo.newBuilder(queryConfig).build());
            }
        }
        return this.overwriteDestinationWithTemporary(temporaryTableId, destinationTableId);
    }

    public Job overwriteDestinationWithTemporary(TableId temporaryTableId, TableId destinationTableId) {
        String queryFormat = "MERGE `%s`\nUSING (SELECT * FROM `%s`)\nON FALSE\nWHEN NOT MATCHED THEN INSERT ROW\nWHEN NOT MATCHED BY SOURCE THEN DELETE";
        String destinationTableName = BigQueryClient.fullTableName(destinationTableId);
        String temporaryTableName = BigQueryClient.fullTableName(temporaryTableId);
        String sqlQuery = String.format(queryFormat, destinationTableName, temporaryTableName);
        QueryJobConfiguration queryConfig = this.jobConfigurationFactory.createQueryJobConfigurationBuilder(sqlQuery, Collections.emptyMap()).setUseLegacySql(false).build();
        return this.create(JobInfo.newBuilder(queryConfig).build());
    }

    public Job appendDestinationWithTemporary(TableId temporaryTableId, TableId destinationTableId) {
        return this.copyData(temporaryTableId, destinationTableId, JobInfo.WriteDisposition.WRITE_APPEND);
    }

    public String createTablePathForBigQueryStorage(TableId tableId) {
        Preconditions.checkNotNull(tableId, "tableId cannot be null");
        String project = tableId.getProject() != null ? tableId.getProject() : this.getProjectId();
        return String.format("projects/%s/datasets/%s/tables/%s", project, tableId.getDataset(), tableId.getTable());
    }

    public TableInfo getReadTable(ReadTableOptions options) {
        Optional<String> query = options.query();
        if (query.isPresent()) {
            this.validateViewsEnabled(options);
            String sql = query.get();
            return this.materializeQueryToTable(sql, options.expirationTimeInMinutes());
        }
        TableInfo table = this.getTable(options.tableId());
        if (table == null) {
            return null;
        }
        Object tableDefinition = table.getDefinition();
        TableDefinition.Type tableType = ((TableDefinition)tableDefinition).getType();
        if (TableDefinition.Type.TABLE == tableType || TableDefinition.Type.EXTERNAL == tableType || TableDefinition.Type.SNAPSHOT == tableType) {
            return table;
        }
        if (TableDefinition.Type.VIEW == tableType || TableDefinition.Type.MATERIALIZED_VIEW == tableType) {
            this.validateViewsEnabled(options);
            return table;
        }
        throw new BigQueryConnectorException(BigQueryErrorCode.UNSUPPORTED, String.format("Table type '%s' of table '%s.%s' is not supported", tableType, table.getTableId().getDataset(), table.getTableId().getTable()));
    }

    public Schema getReadTableSchema(ReadTableOptions options) {
        Optional<String> query = options.query();
        if (query.isPresent()) {
            this.validateViewsEnabled(options);
            String sql = query.get();
            return this.getQueryResultSchema(sql, Collections.emptyMap());
        }
        TableInfo table = this.getReadTable(options);
        return table != null ? ((TableDefinition)table.getDefinition()).getSchema() : null;
    }

    private void validateViewsEnabled(ReadTableOptions options) {
        if (!options.viewsEnabled()) {
            throw new BigQueryConnectorException(BigQueryErrorCode.UNSUPPORTED, String.format("Views are not enabled. You can enable views by setting '%s' to true. Notice additional cost may occur.", options.viewEnabledParamName()));
        }
    }

    DatasetId toDatasetId(TableId tableId) {
        return DatasetId.of(tableId.getProject(), tableId.getDataset());
    }

    public String getProjectId() {
        return ((BigQueryOptions)this.bigQuery.getOptions()).getProjectId();
    }

    Iterable<Dataset> listDatasets(String projectId) {
        return this.bigQuery.listDatasets(projectId, new BigQuery.DatasetListOption[0]).iterateAll();
    }

    public Iterable<Table> listTables(DatasetId datasetId, TableDefinition.Type ... types) {
        ImmutableSet<TableDefinition.Type> allowedTypes = ImmutableSet.copyOf(types);
        Iterable<Table> allTables = this.bigQuery.listTables(datasetId, new BigQuery.TableListOption[0]).iterateAll();
        return StreamSupport.stream(allTables.spliterator(), false).filter(table -> allowedTypes.contains(((TableDefinition)table.getDefinition()).getType())).collect(ImmutableList.toImmutableList());
    }

    TableId createDestinationTable(Optional<String> referenceProject, Optional<String> referenceDataset) {
        String project = this.materializationProject.orElse(referenceProject.orElse(null));
        String dataset = this.materializationDataset.orElse(referenceDataset.orElse(null));
        String name = String.format("_bqc_%s", UUID.randomUUID().toString().toLowerCase(Locale.ENGLISH).replace("-", ""));
        return project == null ? TableId.of(dataset, name) : TableId.of(project, dataset, name);
    }

    public Table update(TableInfo table) {
        return this.bigQuery.update(table, new BigQuery.TableOption[0]);
    }

    public Job createAndWaitFor(JobConfiguration.Builder jobConfiguration) {
        return this.createAndWaitFor((JobConfiguration)jobConfiguration.build());
    }

    public Job createAndWaitFor(JobConfiguration jobConfiguration) {
        JobInfo jobInfo = JobInfo.of(jobConfiguration);
        Job job = this.bigQuery.create(jobInfo, new BigQuery.JobOption[0]);
        Object returnedJob = null;
        log.info("Submitted job {}. jobId: {}", (Object)jobConfiguration, (Object)job.getJobId());
        try {
            Job completedJob = job.waitFor(new RetryOption[0]);
            if (completedJob == null) {
                throw new BigQueryException(0, String.format("Failed to run the job [%s], got null back", job));
            }
            if (completedJob.getStatus().getError() != null) {
                throw new BigQueryException(0, String.format("Failed to run the job [%s], due to '%s'", completedJob.getStatus().getError()));
            }
            return completedJob;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new BigQueryException(0, String.format("Failed to run the job [%s], task was interrupted", job), e);
        }
    }

    Job create(JobInfo jobInfo) {
        return this.bigQuery.create(jobInfo, new BigQuery.JobOption[0]);
    }

    public TableResult query(String sql) {
        try {
            return this.bigQuery.query(this.jobConfigurationFactory.createQueryJobConfigurationBuilder(sql, Collections.emptyMap()).build(), new BigQuery.JobOption[0]);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new BigQueryException(0, String.format("Failed to run the query [%s]", sql), e);
        }
    }

    String createSql(TableId table, ImmutableList<String> requiredColumns, String[] filters, OptionalLong snapshotTimeMillis) {
        String columns = requiredColumns.isEmpty() ? "*" : requiredColumns.stream().map(column -> String.format("`%s`", column)).collect(Collectors.joining(","));
        return this.createSql(table, columns, filters, snapshotTimeMillis);
    }

    String createSql(TableId table, String formattedQuery, String[] filters, OptionalLong snapshotTimeMillis) {
        String tableName = BigQueryClient.fullTableName(table);
        String whereClause = BigQueryClient.createWhereClause(filters).map(clause -> "WHERE " + clause).orElse("");
        String snapshotTimeClause = snapshotTimeMillis.isPresent() ? String.format("FOR SYSTEM_TIME AS OF TIMESTAMP_MILLIS(%d)", snapshotTimeMillis.getAsLong()) : "";
        return String.format("SELECT %s FROM `%s` %s %s", formattedQuery, tableName, whereClause, snapshotTimeClause);
    }

    public static String fullTableName(TableId tableId) {
        if (tableId.getProject() == null) {
            return String.format("%s.%s", tableId.getDataset(), tableId.getTable());
        }
        return String.format("%s.%s.%s", tableId.getProject(), tableId.getDataset(), tableId.getTable());
    }

    public long calculateTableSize(TableId tableId, Optional<String> filter) {
        return this.calculateTableSize(this.getTable(tableId), filter);
    }

    public long calculateTableSize(TableInfo tableInfo, Optional<String> filter) {
        TableDefinition.Type type = ((TableDefinition)tableInfo.getDefinition()).getType();
        if (!(type != TableDefinition.Type.EXTERNAL && type != TableDefinition.Type.TABLE || filter.isPresent())) {
            if (BigQueryUtil.isBigQueryNativeTable(tableInfo) && tableInfo.getRequirePartitionFilter() != null && tableInfo.getRequirePartitionFilter().booleanValue()) {
                ImmutableList<String> partitioningFields = BigQueryUtil.getPartitionFields(tableInfo);
                if (partitioningFields.isEmpty()) {
                    throw new IllegalStateException("Could not find partitioning columns for table requiring partition filter: " + tableInfo.getTableId());
                }
                String table = BigQueryClient.fullTableName(tableInfo.getTableId());
                return this.getNumberOfRows(String.format("SELECT COUNT(*) from `%s` WHERE %s IS NOT NULL", table, partitioningFields.get(0)));
            }
            String table = BigQueryClient.fullTableName(tableInfo.getTableId());
            return this.getNumberOfRows(String.format("SELECT COUNT(*) from `%s`", table));
        }
        if (type == TableDefinition.Type.VIEW || type == TableDefinition.Type.MATERIALIZED_VIEW || (type == TableDefinition.Type.TABLE || type == TableDefinition.Type.EXTERNAL) && filter.isPresent()) {
            String table = BigQueryClient.fullTableName(tableInfo.getTableId());
            String whereClause = filter.map(f -> "WHERE " + f).orElse("");
            return this.getNumberOfRows(String.format("SELECT COUNT(*) from `%s` %s", table, whereClause));
        }
        throw new IllegalArgumentException(String.format("Unsupported table type %s for table %s", type, BigQueryClient.fullTableName(tableInfo.getTableId())));
    }

    private long getNumberOfRows(String sql) {
        TableResult result = this.query(sql);
        long numberOfRows = result.iterateAll().iterator().next().get(0).getLongValue();
        return numberOfRows;
    }

    public TableInfo materializeQueryToTable(String querySql, int expirationTimeInMinutes) {
        TableId tableId = this.createDestinationTable(Optional.empty(), Optional.empty());
        return this.materializeTable(querySql, tableId, expirationTimeInMinutes);
    }

    public TableInfo materializeQueryToTable(String querySql, int expirationTimeInMinutes, Map<String, String> additionalQueryJobLabels) {
        TableId destinationTableId = this.createDestinationTable(Optional.empty(), Optional.empty());
        TempTableBuilder tableBuilder = new TempTableBuilder(this, querySql, destinationTableId, expirationTimeInMinutes, this.jobConfigurationFactory, additionalQueryJobLabels);
        return this.materializeTable(querySql, tableBuilder);
    }

    public TableInfo materializeViewToTable(String querySql, TableId viewId, int expirationTimeInMinutes) {
        TableId tableId = this.createDestinationTable(Optional.ofNullable(viewId.getProject()), Optional.ofNullable(viewId.getDataset()));
        return this.materializeTable(querySql, tableId, expirationTimeInMinutes);
    }

    public Schema getQueryResultSchema(String querySql, Map<String, String> additionalQueryJobLabels) {
        JobInfo jobInfo = JobInfo.of(this.jobConfigurationFactory.createQueryJobConfigurationBuilder(querySql, additionalQueryJobLabels).setDryRun(true).build());
        log.info("running query dryRun {}", (Object)querySql);
        Job completedJobInfo = this.create(jobInfo);
        if (completedJobInfo.getStatus().getError() != null) {
            throw BigQueryUtil.convertToBigQueryException(completedJobInfo.getStatus().getError());
        }
        JobStatistics.QueryStatistics queryStatistics = (JobStatistics.QueryStatistics)completedJobInfo.getStatistics();
        return queryStatistics.getSchema();
    }

    private TableInfo materializeTable(String querySql, TableId destinationTableId, int expirationTimeInMinutes) {
        try {
            return this.destinationTableCache.get(querySql, new TempTableBuilder(this, querySql, destinationTableId, expirationTimeInMinutes, this.jobConfigurationFactory, Collections.emptyMap()));
        }
        catch (Exception e) {
            throw new BigQueryConnectorException(BigQueryErrorCode.BIGQUERY_VIEW_DESTINATION_TABLE_CREATION_FAILED, String.format("Error creating destination table using the following query: [%s]", querySql), e);
        }
    }

    private TableInfo materializeTable(String querySql, TempTableBuilder tmpTableBuilder) {
        try {
            return this.destinationTableCache.get(querySql, tmpTableBuilder);
        }
        catch (Exception e) {
            throw new BigQueryConnectorException(BigQueryErrorCode.BIGQUERY_VIEW_DESTINATION_TABLE_CREATION_FAILED, String.format("Error creating destination table using the following query: [%s]", querySql), e);
        }
    }

    public JobStatistics.LoadStatistics loadDataIntoTable(LoadDataOptions options, List<String> sourceUris, FormatOptions formatOptions, JobInfo.WriteDisposition writeDisposition, Optional<Schema> schema, TableId destinationTable) {
        LoadJobConfiguration.Builder jobConfiguration = this.jobConfigurationFactory.createLoadJobConfigurationBuilder(destinationTable, sourceUris, formatOptions).setCreateDisposition(JobInfo.CreateDisposition.CREATE_IF_NEEDED).setWriteDisposition(writeDisposition);
        if (schema.isPresent()) {
            jobConfiguration.setSchema(schema.get());
        } else {
            jobConfiguration.setAutodetect(true);
        }
        options.getCreateDisposition().ifPresent(jobConfiguration::setCreateDisposition);
        if (options.getPartitionField().isPresent() || options.getPartitionType().isPresent()) {
            TimePartitioning.Builder timePartitionBuilder = TimePartitioning.newBuilder(options.getPartitionTypeOrDefault());
            options.getPartitionExpirationMs().ifPresent(timePartitionBuilder::setExpirationMs);
            options.getPartitionRequireFilter().ifPresent(timePartitionBuilder::setRequirePartitionFilter);
            options.getPartitionField().ifPresent(timePartitionBuilder::setField);
            jobConfiguration.setTimePartitioning(timePartitionBuilder.build());
        }
        if (options.getPartitionField().isPresent() && options.getPartitionRange().isPresent()) {
            RangePartitioning.Builder rangePartitionBuilder = RangePartitioning.newBuilder();
            options.getPartitionField().ifPresent(rangePartitionBuilder::setField);
            options.getPartitionRange().ifPresent(rangePartitionBuilder::setRange);
            jobConfiguration.setRangePartitioning(rangePartitionBuilder.build());
        }
        options.getClusteredFields().ifPresent(clusteredFields -> {
            Clustering clustering = Clustering.newBuilder().setFields((List<String>)clusteredFields).build();
            jobConfiguration.setClustering(clustering);
        });
        if (options.isUseAvroLogicalTypes()) {
            jobConfiguration.setUseAvroLogicalTypes(true);
        }
        if (!options.getDecimalTargetTypes().isEmpty()) {
            jobConfiguration.setDecimalTargetTypes((List)options.getDecimalTargetTypes());
        }
        if (!options.getLoadSchemaUpdateOptions().isEmpty()) {
            jobConfiguration.setSchemaUpdateOptions((List)options.getLoadSchemaUpdateOptions());
        }
        options.getKmsKeyName().ifPresent(destinationTableKmsKeyName -> jobConfiguration.setDestinationEncryptionConfiguration(EncryptionConfiguration.newBuilder().setKmsKeyName((String)destinationTableKmsKeyName).build()));
        Job finishedJob = null;
        try {
            finishedJob = this.createAndWaitFor(jobConfiguration);
            if (finishedJob.getStatus().getError() != null) {
                throw new BigQueryException(0, String.format("Failed to load to %s in job %s. BigQuery error was '%s'", BigQueryUtil.friendlyTableName(options.getTableId()), finishedJob.getJobId(), finishedJob.getStatus().getError().getMessage()), finishedJob.getStatus().getError());
            }
            log.info("Done loading to {}. jobId: {}", (Object)BigQueryUtil.friendlyTableName(options.getTableId()), (Object)finishedJob.getJobId());
            return (JobStatistics.LoadStatistics)finishedJob.getStatistics();
        }
        catch (Exception e) {
            if (finishedJob == null) {
                log.error("Unable to create the job to load to {}", (Object)BigQueryUtil.friendlyTableName(options.getTableId()));
                throw e;
            }
            TimePartitioning.Type partitionType = options.getPartitionTypeOrDefault();
            if (e.getMessage().equals(String.format("Cannot output %s partitioned data in LegacySQL", new Object[]{partitionType})) && formatOptions.equals(FormatOptions.parquet())) {
                throw new BigQueryException(0, String.format("%s time partitioning is not available for load jobs from PARQUET in this project yet. Please replace the intermediate format to AVRO or contact your account manager to enable this.", new Object[]{partitionType}), e);
            }
            JobId jobId = finishedJob.getJobId();
            log.warn(String.format("Failed to load the data into BigQuery, JobId for debug purposes is [%s:%s.%s]", jobId.getProject(), jobId.getLocation(), jobId.getJob()));
            throw new BigQueryException(0, "Problem loading data into BigQuery", e);
        }
    }

    public void createTableIfNeeded(TableId tableId, Schema bigQuerySchema, CreateTableOptions options) {
        if (!this.tableExists(tableId)) {
            this.createTable(tableId, bigQuerySchema, options);
        }
    }

    static class JobConfigurationFactory {
        private final ImmutableMap<String, String> labels;
        private final QueryJobConfiguration.Priority queryJobPriority;

        public JobConfigurationFactory(Map<String, String> labels, QueryJobConfiguration.Priority queryJobPriority) {
            this.labels = ImmutableMap.copyOf(labels);
            this.queryJobPriority = queryJobPriority;
        }

        QueryJobConfiguration.Builder createQueryJobConfigurationBuilder(String querySql, Map<String, String> additionalQueryJobLabels) {
            QueryJobConfiguration.Builder builder = QueryJobConfiguration.newBuilder(querySql).setPriority(this.queryJobPriority);
            HashMap<String, String> allLabels = new HashMap<String, String>(additionalQueryJobLabels);
            if (this.labels != null && !this.labels.isEmpty()) {
                allLabels.putAll(this.labels);
            }
            builder.setLabels(allLabels);
            return builder;
        }

        LoadJobConfiguration.Builder createLoadJobConfigurationBuilder(TableId tableId, List<String> sourceUris, FormatOptions formatOptions) {
            LoadJobConfiguration.Builder builder = LoadJobConfiguration.newBuilder(tableId, sourceUris, formatOptions);
            if (this.labels != null && !this.labels.isEmpty()) {
                builder.setLabels(this.labels);
            }
            return builder;
        }
    }

    static class TempTableBuilder
    implements Callable<TableInfo> {
        final BigQueryClient bigQueryClient;
        final String querySql;
        final TableId tempTable;
        final int expirationTimeInMinutes;
        final JobConfigurationFactory jobConfigurationFactory;
        final Map<String, String> additionalQueryJobLabels;

        TempTableBuilder(BigQueryClient bigQueryClient, String querySql, TableId tempTable, int expirationTimeInMinutes, JobConfigurationFactory jobConfigurationFactory, Map<String, String> additionalQueryJobLabels) {
            this.bigQueryClient = bigQueryClient;
            this.querySql = querySql;
            this.tempTable = tempTable;
            this.expirationTimeInMinutes = expirationTimeInMinutes;
            this.jobConfigurationFactory = jobConfigurationFactory;
            this.additionalQueryJobLabels = additionalQueryJobLabels;
        }

        @Override
        public TableInfo call() {
            return this.createTableFromQuery();
        }

        TableInfo createTableFromQuery() {
            log.info("DestinationTable is {}", (Object)this.tempTable);
            JobInfo jobInfo = JobInfo.of(this.jobConfigurationFactory.createQueryJobConfigurationBuilder(this.querySql, this.additionalQueryJobLabels).setDestinationTable(this.tempTable).build());
            log.info("running query {}", (Object)this.querySql);
            JobInfo completedJobInfo = this.bigQueryClient.waitForJob(this.bigQueryClient.create(jobInfo));
            if (completedJobInfo.getStatus().getError() != null) {
                throw BigQueryUtil.convertToBigQueryException(completedJobInfo.getStatus().getError());
            }
            CLEANUP_JOBS.add(() -> this.bigQueryClient.deleteTable(this.tempTable));
            TableInfo createdTable = this.bigQueryClient.getTable(this.tempTable);
            long expirationTime = createdTable.getCreationTime() + TimeUnit.MINUTES.toMillis(this.expirationTimeInMinutes);
            Table updatedTable = this.bigQueryClient.update(createdTable.toBuilder().setExpirationTime(expirationTime).build());
            return updatedTable;
        }

        Job waitForJob(Job job) {
            try {
                log.info("Job submitted : {}, {},  Job type : {}", new Object[]{job.getJobId(), job.getSelfLink(), ((JobConfiguration)job.getConfiguration()).getType()});
                Job completedJob = job.waitFor(new RetryOption[0]);
                log.info("Job has finished {} creationTime : {}, startTime : {}, endTime : {} ", new Object[]{completedJob.getJobId(), ((JobStatistics)completedJob.getStatistics()).getCreationTime(), ((JobStatistics)completedJob.getStatistics()).getStartTime(), ((JobStatistics)completedJob.getStatistics()).getEndTime()});
                log.debug("Job has finished. {}", (Object)completedJob);
                return completedJob;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new BigQueryException(0, String.format("Job %s has been interrupted", job.getJobId()), e);
            }
        }
    }

    public static interface CreateTableOptions {
        default public Optional<String> getKmsKeyName() {
            return Optional.empty();
        }

        default public Map<String, String> getBigQueryTableLabels() {
            return Collections.emptyMap();
        }

        default public Optional<ImmutableList<String>> getClusteredFields() {
            return Optional.empty();
        }

        public static CreateTableOptions of(final Optional<String> kmsKeyName, final Map<String, String> bigQueryTableLabels, final Optional<ImmutableList<String>> clusteredFields) {
            return new CreateTableOptions(){

                @Override
                public Optional<String> getKmsKeyName() {
                    return kmsKeyName;
                }

                @Override
                public Map<String, String> getBigQueryTableLabels() {
                    return bigQueryTableLabels;
                }

                @Override
                public Optional<ImmutableList<String>> getClusteredFields() {
                    return clusteredFields;
                }
            };
        }
    }

    public static interface LoadDataOptions {
        public TableId getTableId();

        public Optional<JobInfo.CreateDisposition> getCreateDisposition();

        public Optional<String> getPartitionField();

        public Optional<TimePartitioning.Type> getPartitionType();

        public Optional<RangePartitioning.Range> getPartitionRange();

        public TimePartitioning.Type getPartitionTypeOrDefault();

        public OptionalLong getPartitionExpirationMs();

        public Optional<Boolean> getPartitionRequireFilter();

        public Optional<ImmutableList<String>> getClusteredFields();

        public boolean isUseAvroLogicalTypes();

        public List<String> getDecimalTargetTypes();

        public List<JobInfo.SchemaUpdateOption> getLoadSchemaUpdateOptions();

        public boolean getEnableModeCheckForSchemaFields();

        public Optional<String> getKmsKeyName();
    }

    public static interface ReadTableOptions {
        public TableId tableId();

        public Optional<String> query();

        public boolean viewsEnabled();

        public String viewEnabledParamName();

        public int expirationTimeInMinutes();
    }
}

