(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     58385,       1919]*)
(*NotebookOutlinePosition[     74100,       2325]*)
(*  CellTagsIndexPosition[     71890,       2266]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["\<\
5.6.6 Two-Dimensional Boundary Layer Flows: Group \
Classification\
\>", "Subsection",
  CounterAssignments->{{"Title", 5}, {"SubTitle", 0}, {"SubSubTitle", 0}, {
      "Section", 0}, {"SubSection", 0}, {"SubSubSection", 0}, {
      "NumberedEquation", 60}, {"NumberedFigure", 0}, {"NumberedTable", 0}},
  CounterBoxOptions->{CounterFunction:>CapitalRomanNumeral},
  CellTags->"Two-Dimensional Boundary-Layer Flows"],

Cell[TextData[{
  "In this section, we examine the problem of an incompressible boundary \
layer flow over a flat plate. We will not solve the physical problem but \
discuss the symmetries of several related models. The models differ from each \
other in the behavior of the vertical velocity component above the boundary. \
We demonstrate that the models are the result of group classification of the \
general equation. We show that the symmetry of the different models is \
closely connected with the symmetry of the general model. In addition, we \
illustrate that ",
  StyleBox["MathLie ",
    FontSlant->"Italic"],
  "is capable of extracting these models from a symmetry calculation. The \
physical arrangement of the flow is given in the following figure. The flow \
above a plate is governed by the mainstream velocity ",
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  ". The coordinates are denoted by ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " and the velocity components in the two directions are ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`v\)]],
  ", respectively."
}], "Text",
  Evaluatable->False,
  CellTags->{
  "boundary-layer flow", "group classification", "symmetry calculation"}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .36923 
MathPictureStart
%% Graphics
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.24359 0.0732601 0.0307692 0.0732601 [
[ 0 0 0 0 ]
[ 1 .36923 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
[ ] 0 setdash
0 g
p
P
0 0 m
1 0 L
1 .36923 L
0 .36923 L
closepath
clip
newpath
p
p
0 0 1 r
.24359 .00879 m
.24359 .03077 L
.97619 .03077 L
.97619 .00879 L
F
P
p
.004 w
.24359 .0381 m
.46337 .0381 L
s
p
.46337 .0381 m
.41337 .0256 L
.41337 .0381 L
.41337 .0506 L
.46337 .0381 L
F
.46337 .0381 m
.41337 .0256 L
.41337 .0381 L
.41337 .0506 L
.46337 .0381 L
s
P
P
p
.004 w
.24359 .0381 m
.24359 .25055 L
s
p
.24359 .25055 m
.25609 .20055 L
.24359 .20055 L
.23109 .20055 L
.24359 .25055 L
F
.24359 .25055 m
.25609 .20055 L
.24359 .20055 L
.23109 .20055 L
.24359 .25055 L
s
P
P
[(x)] .5 .0381 0 0 Mshowa
[(y)] .24359 .28718 0 0 Mshowa
p
.004 w
.39011 .10403 m
.5 .10403 L
s
p
.5 .10403 m
.45 .09153 L
.45 .10403 L
.45 .11653 L
.5 .10403 L
F
.5 .10403 m
.45 .09153 L
.45 .10403 L
.45 .11653 L
.5 .10403 L
s
P
P
p
.004 w
.39011 .10403 m
.39011 .21392 L
s
p
.39011 .21392 m
.40261 .16392 L
.39011 .16392 L
.37761 .16392 L
.39011 .21392 L
F
.39011 .21392 m
.40261 .16392 L
.39011 .16392 L
.37761 .16392 L
.39011 .21392 L
s
P
P
[(u)] .57326 .10403 0 0 Mshowa
[(v)] .39011 .28718 0 0 Mshowa
p
1 0 0 r
p
.004 w
.02381 .03077 m
.20696 .03077 L
s
p
.20696 .03077 m
.15696 .01827 L
.15696 .03077 L
.15696 .04327 L
.20696 .03077 L
F
.20696 .03077 m
.15696 .01827 L
.15696 .03077 L
.15696 .04327 L
.20696 .03077 L
s
P
P
P
p
1 0 0 r
p
.004 w
.02381 .10403 m
.20696 .10403 L
s
p
.20696 .10403 m
.15696 .09153 L
.15696 .10403 L
.15696 .11653 L
.20696 .10403 L
F
.20696 .10403 m
.15696 .09153 L
.15696 .10403 L
.15696 .11653 L
.20696 .10403 L
s
P
P
P
p
1 0 0 r
p
.004 w
.02381 .17729 m
.20696 .17729 L
s
p
.20696 .17729 m
.15696 .16479 L
.15696 .17729 L
.15696 .18979 L
.20696 .17729 L
F
.20696 .17729 m
.15696 .16479 L
.15696 .17729 L
.15696 .18979 L
.20696 .17729 L
s
P
P
P
p
1 0 0 r
p
.004 w
.02381 .25055 m
.20696 .25055 L
s
p
.20696 .25055 m
.15696 .23805 L
.15696 .25055 L
.15696 .26305 L
.20696 .25055 L
F
.20696 .25055 m
.15696 .23805 L
.15696 .25055 L
.15696 .26305 L
.20696 .25055 L
s
P
P
P
p
1 0 0 r
p
.004 w
.02381 .32381 m
.20696 .32381 L
s
p
.20696 .32381 m
.15696 .31131 L
.15696 .32381 L
.15696 .33631 L
.20696 .32381 L
F
.20696 .32381 m
.15696 .31131 L
.15696 .32381 L
.15696 .33631 L
.20696 .32381 L
s
P
P
P
[(W)] .17033 .36044 0 0 Mshowa
P
% End of Graphics
MathPictureEnd\
\>"], "Graphics",
  PageWidth->Infinity,
  Evaluatable->False,
  ImageSize->{215.5, 83.5},
  ImageMargins->{{54.5, 0}, {0, 0}},
  ImageRegion->{{-0.015811, 1.01641}, {-0.34506, 1.30239}},
  FontFamily->"Courier New",
  FontSize->8,
  FontWeight->"Plain",
  FontSlant->"Plain",
  FontTracking->"Plain",
  FontColor->GrayLevel[0],
  Background->GrayLevel[1],
  FontVariations->{"Underline"->False,
  "Outline"->False,
  "Shadow"->False},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004=0000J1000`40O003h00Oogoo3Woo003oOol>
Ool003mool/07`=oo`009goo0W`05Woob`0O0goo000WOol6O00BOoo;01l3Ool002Moo`Ul00mool/0
7`=oo`009goo3G`02goo;`0O0`004P0O0P000P0O0P00P@0O0goo0002OoldO009Ool_01l8000>01l0
100001l07`00PP0O0goo000WOol<O00kOol<000;Ool20026Ool002Moo`Ml015ooch000Qoo`8008Io
o`009goo0g`05Goo00<007ooOol0;7oo30002Woo00@007ooOol008Eoo`00?goo00<007ooOol0;7oo
1`003Woo0P000Woo0P00Q7oo000oOol00`00Oomoo`0/Ool3002LOol003moo`03001oogoo0<]oo`00
?goo00<007ooOol0bgoo000oOol00`00Oomoo`3;Ool003moo`03001oogoo0<]oo`00?goo00<007oo
Ool0bgoo000oOol00`00Oomoo`3;Ool003moo`03001oogoo0<]oo`00?goo00<007ooOol0bgoo000o
Ool00`00Oomoo`3;Ool003moo`03001oogoo0<]oo`00?goo00<007ooOol0bgoo000WOol3O00EOol0
0`00Oomoo`0fOol3000NOol200000goo0000001_Ool002Moo`Ql011oo`03001oogoo03Ioo`P001Qo
o`03001oogoo0080071oo`009goo37`037oo00<007ooOol0=Woo300057oo00D007ooOomoo`00071o
o`000Woo=7`02Goo00<007ooOol09Goo80004Goo00D007ooOomoo`00071oo`009goo3G`02goo00<0
07ooOol09Goo00<007ooOol03Woo3@004goo00D007ooOomoo`00071oo`009goo2G`03goo00<007oo
Ool09Goo00<007ooOol03Woo2@005Woo0P000Woo0P00L7oo000WOol6O00BOol00`00Oomoo`0UOol0
0`00Oomoo`0>Ool6002?Ool002Moo`9l01Ioo`03001oogoo02Eoo`03001oogoo00ioo`8009=oo`00
?goo00<007ooOol09Goo00<007ooOol0Xgoo000oOol00`00Oomoo`0UOol00`00Oomoo`2SOol003mo
o`03001oogoo02Eoo`03001oogoo0:=oo`00?goo00<007ooOol09Goo00<007ooOol0Xgoo000oOol0
0`00Oomoo`0UOol00`00Oomoo`2SOol003moo`03001oogoo02Eoo`03001oogoo0:=oo`00?goo00<0
07ooOol09Goo00<007ooOol0Xgoo000oOol00`00Oomoo`0UOol00`00Oomoo`2SOol003moo`03001o
ogoo02Eoo`03001oogoo0:=oo`00?goo00<007ooOol09Goo00<007ooOol0Xgoo000oOol00`00Oomo
o`0UOol00`00Oomoo`2SOol003moo`03001oogoo02Eoo`03001oogoo0:=oo`009goo0W`05Woo00<0
07ooOol08Woo2000XGoo000WOol6O00BOol00`00Oomoo`0ROol8002QOol002Moo`Ul00moo`03001o
ogoo029oo`L00:9oo`009goo3G`02goo00<007ooOol08goo1P00XWoo0002OoldO009Ool00`00Oomo
o`0SOol6002ROol002Moo`al00aoo`03001oogoo02=oo`H00:9oo`009goo1g`04Goo00<007ooOol0
8goo1@00Xgoo000WOol3O00EOol00`00Oomoo`0TOol4002SOol003moo`03001oogoo02Aoo`@00:=o
o`00?goo00<007ooOol097oo0`00Y7oo000kOol8000SOol3002TOol003]oo`P002=oo`<00:Aoo`00
?7oo1`0097oo0P00Y7oo000lOol6000UOol00`00Oomoo`2SOol003aoo`H002Eoo`03001oogoo0:=o
o`00?7oo1P00bgoo000mOol5003;Ool003eoo`D00<]oo`00?Goo1000c7oo000nOol3003<Ool002Mo
o`9l01Eoo`<00<aoo`009goo1W`04Goo0`00c7oo000WOol9O00>Ool2003=Ool002Moo`el00]oo`03
001oogoo0<]oo`000Woo=7`02Goo00<007ooOol0bgoo000WOol<O03JOol002Moo`Ml0=moo`009goo
0g`0hgoo003oOol>Ool003eoo`<00<eoo`00?goo00<007ooOol0bgoo000oOol00`00Oomoo`0TOol2
002UOol003ioo`03001oo`0002Eoo`800:Eoo`00?Woo00<007oo000097oo00@007ooOol00:Aoo`00
?Goo00D007ooOomoo`0002=oo`04001oogoo002TOol003eoo`05001oogooOol0000ROol00`00Oomo
o`02Ool00`00Oomoo`2QOol003aoo`<00004Ool00000000POol30002Ool3002ROol00?moo`ioo`00
ogoo3Woo003oOol>Ool00?moo`ioo`009goo0g`0hgoo000WOol8O03NOol002Moo`al0=Yoo`000Woo
=7`0egoo000WOol=O03IOol002Moo`Ul0=eoo`009goo1W`0h7oo000WOol2O03TOol00?moo`ioo`00
ogoo3Woo000YOol00`00Ool0003QOol002Qoo`05001oo`00Ool0003POol002Qoo`05001oo`00Ool0
003POol002Qoo`05001oo`00Ool0003POol002Qoo`05001oo`00Ool0003POol002Qoo`05001oogoo
Ool0003POol002Qoo`05001oogooOol0003POol002Moo`<00004Ool00000003OOol00?moo`ioo`00
\
\>"],
  ImageRangeCache->{{{-4.25, 272.75}, {139.375, -31.5}} -> {-3.11564, \
-3.6625, 0.0615981, 0.0615981}}],

Cell[TextData[{
  "The general two-dimensional, steady, laminar incompressible boundary layer \
equations follow from the Navier-Stokes equations if we neglect buoyancy. The \
resulting equations are the continuity equation and the momentum equation for \
steady flow (",
  ButtonBox["Lamb [1945]):",
    ButtonData:>{"Literat.nb", "Lamb-1945"},
    ButtonStyle->"Hyperlink"]
}], "Text",
  Evaluatable->False,
  CellTags->{
  "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
    "continuity equation", "momentum equation", "steady flow"}],

Cell[BoxData[
    \(TraditionalForm\`\(u\_x + v\_y = 0, \)\)], "NumberedEquation"],

Cell[BoxData[
    FormBox[
      RowBox[{
        RowBox[{
          RowBox[{"u", " ", 
            FormBox[\(u\_x\),
              "TraditionalForm"]}], " ", "+", " ", \(v\ u\_y\), "-", " ", 
          \(\(W(x)\)\ W\_x\), " ", "-", " ", \(\[Eta]\ u\_\(x, x\)\)}], "=", 
        "0."}], TraditionalForm]], "NumberedEquation"],

Cell[TextData[{
  "Here, ",
  Cell[BoxData[
      \(TraditionalForm\`W = W(x)\)]],
  " is a known function describing the mainstream flow velocity in the ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " direction and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Eta]\)]],
  " is the kinematic viscosity. The boundary conditions to be satisfied are"
}], "Text",
  CellTags->"boundary condition"],

Cell[BoxData[
    \(TraditionalForm\`u(x, y = 0)\  = \ \(v(x, y = 0)\  = \ 0\)\)], 
  "NumberedEquation"],

Cell["and", "Text",
  Evaluatable->False],

Cell[BoxData[
    \(TraditionalForm\`u(x, \[Infinity])\  = \ \(W(x) . \)\)], 
  "NumberedEquation"],

Cell["If we denote by", "Text",
  Evaluatable->False],

Cell[BoxData[
    \(U = u[x, y]; V = v[x, y]; \)], "Input",
  PageWidth->Infinity],

Cell[TextData[{
  "the velocity fields in ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " directions, we can write the left-hand side of the two equations in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " by"
}], "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(bound = {\[PartialD]\_x U + \[PartialD]\_y V, 
        U\ \[PartialD]\_x U + V\ \[PartialD]\_y U - 
          W[x]\ \[PartialD]\_x W[x] - \[Eta]\ \[PartialD]\_{y, 2}U}; 
    bound // LTF\)], "Input",
  CellTags->"Prandtl's boundary layer equations"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(u\_x + v\_y == 0\)},
          {\(u\ u\_x + v\ u\_y - W\ W\_x - \[Eta]\ u\_\(y, y\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The equations ",
  StyleBox["bound",
    FontSlant->"Italic"],
  " constitute a system of equations in two dependent variables ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`v\)]],
  ". This is a system of partial differential equations of parabolic type. \
The equations are today known as Prandtel's boundary layer equations. If we \
now define a stream function by"
}], "Text",
  Evaluatable->False,
  CellTags->{
  "partial differential equations of parabolic type", 
    "Prandtel's boundary-layer equations", "stream function"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(stream = {
        u \[Rule] Function[{x, y}, \[PartialD]\_y \[CapitalPsi][x, y]], 
        v \[Rule] Function[{x, y}, \(-\[PartialD]\_x \[CapitalPsi][x, y]\)]}
        \)], "Input"],

Cell[BoxData[
    \({u \[Rule] Function[{x, y}, \[PartialD]\_y \[CapitalPsi][x, y]], 
      v \[Rule] Function[{x, y}, \(-\[PartialD]\_x \[CapitalPsi][x, y]\)]}
      \)], "Output"]
}, Open  ]],

Cell["\<\
we can reduce the two equations of motion to a single equation \
given by\
\>", "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(bound1 = bound /. stream; bound1 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {"True"},
          {
            \(\(-W\)\ W\_x + \[CapitalPsi]\_y\ \[CapitalPsi]\_\(x, y\) - 
                \[CapitalPsi]\_x\ \[CapitalPsi]\_\(y, y\) - 
                \[Eta]\ \[CapitalPsi]\_\(y, y, y\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "representing a non-linear partial differential equation for the stream \
function \[CapitalPsi] of third order. In the following, we will analyze this \
equation to derive explicit solutions for different mainstream velocities ",
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  ". Utilizing the functions of ",
  StyleBox["MathLie",
    FontSlant->"Italic"],
  ", we are able to determine the symmetries by"
}], "Text",
  Evaluatable->False,
  CellTags->"mainstream velocities"],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbound = 
      Infinitesimals[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]}, {
          \[CapitalPsi]}, {x, y}, {\[Eta]}]; symbound // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(\[Phi]\_1 == k1\)},
          {\(\[Xi]\_1 == 0\)},
          {\(\[Xi]\_2 == \[ScriptCapitalF]\_1\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], k1], 
        Equal[ 
          Subscript[ \[Xi], 1], 0], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Subscript[ \[ScriptCapitalF], 1]]}]]], "Output"]
}, Open  ]],

Cell["\<\
The result shows that the equation for the stream function with \
arbitrary mainstream velocity allows only a translation with respect to the \
stream function. In the following, we will examine the influence of the \
mainstream function on the symmetries.\
\>", "Text"],

Cell[CellGroupData[{

Cell["5.6.6.1 The Blasius Solution", "Subsubsection",
  CellTags->"The Blasius Solution"],

Cell[TextData[{
  "If we assume that the mainstream velocity is a constant ",
  Cell[BoxData[
      \(TraditionalForm\`W(x) = w\)]],
  " as in the problem discussed by ",
  ButtonBox["Blasius [1908],",
    ButtonData:>{"Literat.nb", "Blas-08"},
    ButtonStyle->"Hyperlink"],
  " the symmetry analysis results in the representation of infinitesimals:"
}], "Text",
  CellTags->"infinitesimals of Blasius' model"],

Cell[BoxData[
    \(\(mainstream = W \[Rule] Function[x, w]; \)\)], "Input",
  PageWidth->Infinity],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbound = 
      Infinitesimals[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
          mainstream, {\[CapitalPsi]}, {x, y}, {\[Eta], w}]; 
    symbound // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(\[Phi]\_1 == k1 + k2\ \[CapitalPsi]\)},
          {\(\[Xi]\_1 == k3 + k4\ x\)},
          {\(\[Xi]\_2 == \((\(-k2\) + k4)\)\ y + \[ScriptCapitalF]\_1\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], 
          Plus[ k1, 
            Times[ k2, \[CapitalPsi]]]], 
        Equal[ 
          Subscript[ \[Xi], 1], 
          Plus[ k3, 
            Times[ k4, x]]], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Plus[ 
            Times[ 
              Plus[ 
                Times[ -1, k2], k4], y], 
            Subscript[ \[ScriptCapitalF], 1]]]}]]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The similarity reduction for this model follows by using the function ",
  StyleBox["LieReduction[]", "MmaText"],
  ". Let us first examine the reduction of the four-dimensional group with \
respect to translations. This subgroup is selected by the group constants ",
  StyleBox["k3", "MmaText",
    FontSlant->"Italic"],
  ",",
  StyleBox[" ",
    FontSlant->"Italic"],
  StyleBox["k1", "MmaText",
    FontSlant->"Italic"],
  ", and the arbitrary function ",
  StyleBox["free[1]", "MmaText",
    FontSlant->"Italic"],
  " which we set equal to a constant. The related infinitesimals are thus \
given by"
}], "Text",
  Evaluatable->False,
  CellTags->"LieReduction[]"],

Cell[CellGroupData[{

Cell[BoxData[
    \(infi1 = 
      \({{\(xi[1]\)[x, y, \[CapitalPsi]], \(xi[2]\)[x, y, \[CapitalPsi]]}, {
              \(phi[1]\)[x, y, \[CapitalPsi]]}} /. symbound\) /. {
          k1 \[Rule] 1, k2 \[Rule] 0, k3 \[Rule] 1, k4 \[Rule] 0, 
          \(free[1]\)[x] \[Rule] c}\)], "Input"],

Cell[BoxData[
    \({{1, c}, {1}}\)], "Output"]
}, Open  ]],

Cell["The reduction of the equation follows for this subgroup to", "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(red1 = 
      LieReduction[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
          mainstream, {\[CapitalPsi]}, {x, y}, 
        infi1\[LeftDoubleBracket]1\[RightDoubleBracket], 
        infi1\[LeftDoubleBracket]2\[RightDoubleBracket]]; 
    LTF[Flatten[red1]] /. zeta1 \[Rule] \[Zeta]\_1\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(\(-c\)\ x + y - \[Zeta]\_1 == 0\)},
          {\(\(-x\) + \[CapitalPsi] - F\_1 == 0\)},
          {
            \(\(-F1\_\(\[Zeta]\_1, \[Zeta]\_1\)\) - 
                \[Eta]\ F1\_\(\[Zeta]\_1, \[Zeta]\_1, \[Zeta]\_1\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell["\<\
The related similarity solution is obtained if we integrate the \
third-order ordinary differential equation\
\>", "Text",
  Evaluatable->False,
  CellTags->"similarity solution"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sred1 = 
      DSolve[red1\[LeftDoubleBracket]3, 1\[RightDoubleBracket] == 0, F1, 
        zeta1]\)], "Input",
  PageWidth->Infinity],

Cell[BoxData[
    \({{F1 \[Rule] \((C[1] + E\^\(-\(#1\/\[Eta]\)\)\ C[3] + C[2]\ #1&)\)}}
      \)], "Output"]
}, Open  ]],

Cell["\<\
and insert the result into the representation of the solution\
\>", 
  "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(ssred1 = 
      Solve[Flatten[red1\[LeftDoubleBracket]2\[RightDoubleBracket] /. sred1], 
        \[CapitalPsi]]\)], "Input",
  PageWidth->Infinity],

Cell[BoxData[
    \({{\[CapitalPsi] \[Rule] 
          x + C[1] + \((\(-c\)\ x + y)\)\ C[2] + 
            E\^\(-\(\(\(-c\)\ x + y\)\/\[Eta]\)\)\ C[3]}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The stream function ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPsi]\)]],
  " is determined by three integration constants ",
  Cell[BoxData[
      \(TraditionalForm\`C[i], \ i = 1, 2, 3\)]],
  ". The solution was derived under the condition that the problem allows the \
invariance of translation in the independent and dependent variables. The \
corresponding components of the velocity fields follow by"
}], "Text",
  Evaluatable->False],

Cell[BoxData[
    \(\(rule\  = \ 
      \[CapitalPsi] \[Rule] Function[{x, y}, w]\  /. \ 
        \((ssred1\  /. \[CapitalPsi] \[Rule] w)\); \)\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(velocities\  = \ \({u[x, y], v[x, y]}\  /. \ stream\)\  /. \ rule\)], 
  "Input"],

Cell[BoxData[
    \({C[2] - \(E\^\(-\(\(\(-c\)\ x + y\)\/\[Eta]\)\)\ C[3]\)\/\[Eta], 
      \(-1\) + c\ C[2] - 
        \(c\ E\^\(-\(\(\(-c\)\ x + y\)\/\[Eta]\)\)\ C[3]\)\/\[Eta]}\)], 
  "Output"]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`c\)]],
  " is the group parameter and ",
  StyleBox["C",
    FontSlant->"Italic"],
  "[2] and ",
  StyleBox["C",
    FontSlant->"Italic"],
  "[3] are constants of integration."
}], "Text"],

Cell[TextData[{
  "Another subgroup of the Blasius model for the steady two-dimensional flow \
for a flat plate is the scaling invariance. This kind of invariance is \
selected from the total group if we set the group parameters ",
  StyleBox["k2",
    FontSlant->"Italic"],
  " and ",
  StyleBox["k4",
    FontSlant->"Italic"],
  " equal to a constant value. We get the infinitesimals from"
}], "Text",
  Evaluatable->False,
  CellTags->"Blasius model"],

Cell[CellGroupData[{

Cell[BoxData[
    \(infi1 = 
      \({{\(xi[1]\)[x, y, \[CapitalPsi]], \(xi[2]\)[x, y, \[CapitalPsi]]}, {
              \(phi[1]\)[x, y, \[CapitalPsi]]}} /. symbound\) /. {
          k1 \[Rule] 0, k2 \[Rule] c, k3 \[Rule] 0, k4 \[Rule] 1, 
          \(free[1]\)[x] \[Rule] 0}\)], "Input"],

Cell[BoxData[
    \({{x, \((1 - c)\)\ y}, {c\ \[CapitalPsi]}}\)], "Output"]
}, Open  ]],

Cell["The second reduction of the equation follows by ", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(red2 = 
      Simplify[PowerExpand[
          LieReduction[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
              mainstream, {\[CapitalPsi]}, {x, y}, 
            infi1\[LeftDoubleBracket]1\[RightDoubleBracket], 
            infi1\[LeftDoubleBracket]2\[RightDoubleBracket]]]]; 
    LTF[Flatten[red2]]\  /. \ zeta1 \[Rule] \[Zeta]\_1\)], "Input"],

Cell[BoxData[
    \(Solve::"tdep" \( : \ \) 
      "The equations appear to involve transcendental functions of the \
variables in an essentially non-algebraic way."\)], "Message"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(x\^\(\(-1\) + c\)\ y - \[Zeta]\_1 == 0\)},
          {\(x\^\(-c\)\ \[CapitalPsi] - F\_1 == 0\)},
          {
            \(y\^\(-\(\(4\ c\)\/\(\(-1\) + c\)\)\)\ 
                \[Zeta]\_1\%\(\(4\ c\)\/\(\(-1\) + c\)\)\ 
                \((\((\(-1\) + 2\ c)\)\ \((F\_1)\)\_\[Zeta]\_1\%2 - 
                    c\ F\_1\ \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1\) - 
                    \[Eta]\ \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1, 
                        \[Zeta]\_1\))\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "If we try to solve this type of equation by ",
  StyleBox["DSolve[]", "MmaText"],
  ", we get the result"
}], "Text",
  Evaluatable->False,
  CellTags->"DSolve[]"],

Cell[CellGroupData[{

Cell[BoxData[
    \(DSolve[red2\[LeftDoubleBracket]3\[RightDoubleBracket], F1, zeta1]\)], 
  "Input",
  PageWidth->Infinity],

Cell[BoxData[
    RowBox[{"DSolve", "[", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
            \(y\^\(-\(\(4\ c\)\/\(\(-1\) + c\)\)\)\), " ", 
              \(zeta1\^\(\(4\ c\)\/\(\(-1\) + c\)\)\), " ", 
              RowBox[{"(", 
                RowBox[{
                  RowBox[{\((\(-1\) + 2\ c)\), " ", 
                    SuperscriptBox[
                      RowBox[{
                        SuperscriptBox["F1", "\[Prime]",
                          MultilineFunction->None], "[", "zeta1", "]"}], 
                      "2"]}], "-", 
                  RowBox[{"c", " ", \(F1[zeta1]\), " ", 
                    RowBox[{
                      SuperscriptBox["F1", "\[DoublePrime]",
                        MultilineFunction->None], "[", "zeta1", "]"}]}], 
                  "-", 
                  RowBox[{"\[Eta]", " ", 
                    RowBox[{
                      SuperscriptBox["F1", 
                        TagBox[\((3)\),
                          Derivative],
                        MultilineFunction->None], "[", "zeta1", "]"}]}]}], 
                ")"}]}], "==", "0"}], "}"}], ",", "F1", ",", "zeta1"}], 
      "]"}]], "Output"]
}, Open  ]],

Cell[TextData[
"However, we can use Lie\[CloseCurlyQuote]s methods to examine the symmetries \
of this ordinary differential equation. The symmetries of this equation are \
given by"], "Text",
  Evaluatable->False,
  CellTags->"Lie's method"],

Cell[BoxData[
    \(\(red2eq\  = \ 
      Thread[red2\[LeftDoubleBracket]3, 1\[RightDoubleBracket]/
            y\^\(-\(\(4\ c\)\/\(\(-1\) + c\)\)\)\ 
          zeta1\^\(-\(\(4\ c\)\/\(\(-1\) + c\)\)\), Equal]; \)\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(iblasius = Infinitesimals[red2eq, F1, zeta1, {c, \[Eta]}]; 
    iblasius // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(\[Phi]\_1 == \(-F1\)\ k2\)},
          {\(\[Xi]\_1 == k1 + k2\ zeta1\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], 
          Times[ -1, F1, k2]], 
        Equal[ 
          Subscript[ \[Xi], 1], 
          Plus[ k1, 
            Times[ k2, zeta1]]]}]]], "Output"]
}, Open  ]],

Cell[TextData[{
  "demonstrating that the Blasius equation allows a two-dimensional symmetry \
group containing a translation with respect to the independent variable ",
  StyleBox["zeta1",
    FontSlant->"Italic"],
  " and a scaling in the independent and dependent variables. Since the \
Blasius equation is a third-order ODE but the symmetry group is of dimension \
two, we know from C",
  ButtonBox["hapter 4",
    ButtonData:>{"symmet1.nb", 
      "Symmetries of Ordinary Differential Equations"},
    ButtonStyle->"Hyperlink"],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 0]],
  "that at least a reduction of the order is possible. At this point, the \
solution procedure ends since the number of symmetries is smaller than the \
degree of the ODE."
}], "Text",
  Evaluatable->False,
  CellTags->{"Blasius equation", "reduction of the order"}]
}, Open  ]],

Cell[CellGroupData[{

Cell["5.6.6.2 Falkner-Skan Solution", "Subsubsection",
  CellTags->"Falkner Skan Solution"],

Cell[TextData[{
  "For the same geometrical situation, ",
  ButtonBox["Falkner and Skan [1931]",
    ButtonData:>{"Literat.nb", "Falk-31"},
    ButtonStyle->"Hyperlink"],
  " proposed that the mainstream velocity ",
  Cell[BoxData[
      \(TraditionalForm\`W(x)\)]],
  " is a power law function of the horizontal coordinate. We define this \
relation as"
}], "Text",
  Evaluatable->False,
  CellTags->{"power law", "Falkner", "Skan"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(mainstream = W \[Rule] Function[x, k\ x\^m]\)], "Input"],

Cell[BoxData[
    \(W \[Rule] Function[x, k\ x\^m]\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`k\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`m\)]],
  " are real constants. The symmetries of the stream function \[CapitalPsi] \
are determined for this case by"
}], "Text",
  Evaluatable->False,
  CellTags->"stream function"],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbound = 
      Infinitesimals[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
          mainstream, {\[CapitalPsi]}, {x, y}, {\[Eta], k, m}]; 
    symbound // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(\[Phi]\_1 == k1 + k2\ \[CapitalPsi]\)},
          {
            RowBox[{\(\[Xi]\_1\), "==", 
              StyleBox[\(\(2\ k2\ x\)\/\(1 + m\)\),
                ScriptLevel->0]}]},
          {
            RowBox[{\(\[Xi]\_2\), "==", 
              RowBox[{
                RowBox[{"-", 
                  StyleBox[\(\(k2\ \((\(-1\) + m)\)\ y\)\/\(1 + m\)\),
                    ScriptLevel->0]}], "+", \(\[ScriptCapitalF]\_1\)}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], 
          Plus[ k1, 
            Times[ k2, \[CapitalPsi]]]], 
        Equal[ 
          Subscript[ \[Xi], 1], 
          Times[ 2, k2, 
            Power[ 
              Plus[ 1, m], -1], x]], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Plus[ 
            Times[ -1, k2, 
              Plus[ -1, m], 
              Power[ 
                Plus[ 1, m], -1], y], 
            Subscript[ \[ScriptCapitalF], 1]]]}]]], "Output"]
}, Open  ]],

Cell[TextData[{
  "For the Falkner-Skan model, we find a two-dimensional discrete symmetry \
group and, in addition, an infinite dimensional group represented by ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          StyleBox[\(free[1]\),
            FontSlant->"Italic"], "=", \(\[ScriptCapitalF]\_1\)}], 
        TraditionalForm]]],
  ". Compared with the case when ",
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  " is arbitrary, the group is enlarged by an additional degree of freedom. \
With respect to the Blasius group, the dimension is reduced by two \
components. The main symmetries consist of a translation and a scaling. Let \
us first discuss the translation symmetry. The related infinitesimals follow \
by setting ",
  StyleBox["k1",
    FontSlant->"Italic"],
  " and ",
  StyleBox["free[1]",
    FontSlant->"Italic"],
  " to constants:"
}], "Text",
  Evaluatable->False,
  CellTags->"Falkner-Skan model"],

Cell[CellGroupData[{

Cell[BoxData[
    \(infi1 = 
      \({{\(xi[1]\)[x, y, \[CapitalPsi]], \(xi[2]\)[x, y, \[CapitalPsi]]}, {
              \(phi[1]\)[x, y, \[CapitalPsi]]}} /. symbound\) /. {
          k1 \[Rule] 1, k2 \[Rule] 0, \(free[1]\)[x] \[Rule] c}\)], "Input"],

Cell[BoxData[
    \({{0, c}, {1}}\)], "Output"]
}, Open  ]],

Cell["The reduction of the stream function equation follows by ", "Text",
  CellTags->"stream function equation"],

Cell[CellGroupData[{

Cell[BoxData[
    \(red1 = 
      Simplify[PowerExpand[
          LieReduction[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
              mainstream, {\[CapitalPsi]}, {x, y}, 
            infi1\[LeftDoubleBracket]1\[RightDoubleBracket], 
            infi1\[LeftDoubleBracket]2\[RightDoubleBracket]]]]; 
    LTF[Flatten[red1]] /. zeta1 \[Rule] \[Zeta]\_1\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(x - \[Zeta]\_1 == 0\)},
          {
            RowBox[{
              RowBox[{
                RowBox[{"-", 
                  StyleBox[\(y\/c\),
                    ScriptLevel->0]}], "+", "\[CapitalPsi]", "-", \(F\_1\)}], 
              "==", "0"}]},
          {\(\(-k\^2\)\ m\ \[Zeta]\_1\%\(2\ m\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The result is somehow surprising since it does not contain any reduction \
of the original equation. If we examine the equation for the stream function, \
we observe that all terms contain derivatives with respect to the coordinate \
",
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " which single out the similarity solution. Only the additive term \
containing the mainstream velocity ",
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  " remains in the reduction. The result shows us that under the symmetry of \
translations, only a solution depending linearly on the vertical coordinate ",
  
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " exists."
}], "Text",
  Evaluatable->False,
  CellTags->"similarity solution"],

Cell["\<\
The other type of symmetry contained in the Falkner-Skan case \
represents a scaling. The reduction for the scaling symmetry is given by\
\>",
   "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(infi1 = 
      \({{\(xi[1]\)[x, y, \[CapitalPsi]], \(xi[2]\)[x, y, \[CapitalPsi]]}, {
              \(phi[1]\)[x, y, \[CapitalPsi]]}} /. symbound\) /. {
          k1 \[Rule] 0, k2 \[Rule] c, \(free[1]\)[x] \[Rule] 0}\)], "Input"],

Cell[BoxData[
    \({{\(2\ c\ x\)\/\(1 + m\), 
        \(-\(\(c\ \((\(-1\) + m)\)\ y\)\/\(1 + m\)\)\)}, {c\ \[CapitalPsi]}}
      \)], "Output"]
}, Open  ]],

Cell[TextData[{
  "where we set the group parameters ",
  StyleBox["k2",
    FontSlant->"Italic"],
  " equal to a constant ",
  Cell[BoxData[
      \(TraditionalForm\`c\)]],
  ". The reduction of the equation for the stream function follows by "
}], "Text",
  CellTags->"reduction"],

Cell[CellGroupData[{

Cell[BoxData[
    \(red2 = 
      Simplify[PowerExpand[
          LieReduction[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
              mainstream, {\[CapitalPsi]}, {x, y}, 
            infi1\[LeftDoubleBracket]1\[RightDoubleBracket], 
            infi1\[LeftDoubleBracket]2\[RightDoubleBracket]]]]; 
    LTF[Flatten[red2]] /. zeta1 \[Rule] \[Zeta]\_1\)], "Input"],

Cell[BoxData[
    \(Solve::"tdep" \( : \ \) 
      "The equations appear to involve transcendental functions of the \
variables in an essentially non-algebraic way."\)], "Message"],

Cell[BoxData[
    TagBox[
      TagBox[GridBox[{
            {\(x\^\(1\/2\ \((\(-1\) + m)\)\)\ y - \[Zeta]\_1 == 0\)},
            {\(x\^\(1\/2\ \((\(-1\) - m)\)\)\ \[CapitalPsi] - F\_1 == 0\)},
            {
              \(\(-y\^\(-\(\(4\ m\)\/\(\(-1\) + m\)\)\)\)\ 
                  \[Zeta]\_1\%\(\(4\ m\)\/\(\(-1\) + m\)\)\ 
                  \((\(-2\)\ m\ \((F\_1)\)\_\[Zeta]\_1\%2 + 
                      \((1 + m)\)\ F\_1\ 
                        \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1\) + 
                      2\ \((k\^2\ m + 
                            \[Eta]\ \n\t
                              \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1, 
                                \[Zeta]\_1\))\))\) == 0\)}
            },
          RowSpacings->1,
          ColumnSpacings->3,
          RowAlignments->Baseline,
          ColumnAlignments->{Left}],
        (TableForm[ #]&)],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "Again, ",
  StyleBox["DSolve[]", "MmaText"],
  " is unable to find a solution. The type of the resulting equation is the \
same as the equation in the Blasius model. The same arguments apply here. The \
number of symmetries of the equation is not sufficient for an integration. \
This is one reason why the Falkner-Skan equation is an unsolved problem. We \
will not examine the numerical solution of this equation which actually was \
carried out by ",
  ButtonBox["Falkner and Skan [1931]",
    ButtonData:>{"Literat.nb", "Falk-31"},
    ButtonStyle->"Hyperlink"],
  " in their paper. However our interest is concerned with other \
possibilities to model the mainstream velocity. There is another case which \
enlarges the number of symmetries of the steady two-dimensional flow."
}], "Text",
  CellTags->{"DSolve[]", "Falkner-Skan equation", "mainstream velocity"}]
}, Open  ]],

Cell[CellGroupData[{

Cell["5.6.6.3 Exponential Mainstream Velocity", "Subsubsection",
  CellTags->{
  "Exponential mainstream velocity", "Exponential Mainstream Velocity"}],

Cell["\<\
Another way to choose the mainstream velocity is to assume an \
exponential increase in the horizontal direction. The function of this type \
is given by\
\>", "Text",
  Evaluatable->False,
  CellTags->"exponential mainstream velocity"],

Cell[CellGroupData[{

Cell[BoxData[
    \(mainstream = W \[Rule] Function[x, k\ Exp[a\ x]]\)], "Input"],

Cell[BoxData[
    \(W \[Rule] Function[x, k\ Exp[a\ x]]\)], "Output"]
}, Open  ]],

Cell["\<\
The infinitesimals for this sort of the mainstream velocity \
are\
\>", "Text",
  Evaluatable->False,
  CellTags->"infinitesimals"],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbound = 
      Infinitesimals[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
          mainstream, {\[CapitalPsi]}, {x, y}, {\[Nu], k, a}]; 
    symbound // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(\[Phi]\_1 == k1 - k2\ \[CapitalPsi]\)},
          {
            RowBox[{\(\[Xi]\_1\), "==", 
              RowBox[{"-", 
                StyleBox[\(\(2\ k2\)\/a\),
                  ScriptLevel->0]}]}]},
          {\(\[Xi]\_2 == k2\ y + \[ScriptCapitalF]\_1\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], 
          Plus[ k1, 
            Times[ -1, k2, \[CapitalPsi]]]], 
        Equal[ 
          Subscript[ \[Xi], 1], 
          Times[ -2, 
            Power[ a, -1], k2]], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Plus[ 
            Times[ k2, y], 
            Subscript[ \[ScriptCapitalF], 1]]]}]]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The result contains a two-dimensional finite symmetry group representing a \
translation and some sort of scaling. The undetermined function ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          RowBox[{
            RowBox[{
              StyleBox["free",
                FontSlant->"Italic"], "[", "1", "]"}], "[", "x", "]"}], "=", 
          \(\[ScriptCapitalF]\_1\)}], TraditionalForm]]],
  " extends the finite group to an infinite one. Again, the symmetry group is \
extended if we compare it with the general case in which the mainstream \
velocity is an arbitrary function. The reductions for the scaling symmetry \
follow by"
}], "Text",
  Evaluatable->False,
  CellTags->{"symmetry group", "finite group"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(infi1 = 
      \({{\(xi[1]\)[x, y, \[CapitalPsi]], \(xi[2]\)[x, y, \[CapitalPsi]]}, {
              \(phi[1]\)[x, y, \[CapitalPsi]]}} /. symbound\) /. {
          k1 \[Rule] 0, k2 \[Rule] c, \(free[1]\)[x] \[Rule] 0}\)], "Input"],

Cell[BoxData[
    \({{\(-\(\(2\ c\)\/a\)\), c\ y}, {\(-c\)\ \[CapitalPsi]}}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(red2 = 
      Simplify[PowerExpand[
          LieReduction[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
              mainstream, {\[CapitalPsi]}, {x, y}, 
            infi1\[LeftDoubleBracket]1\[RightDoubleBracket], 
            infi1\[LeftDoubleBracket]2\[RightDoubleBracket]]]]; 
    LTF[Flatten[red2]] /. zeta1 \[Rule] \[Zeta]\_1\)], "Input"],

Cell[BoxData[
    \(Solve::"tdep" \( : \ \) 
      "The equations appear to involve transcendental functions of the \
variables in an essentially non-algebraic way."\)], "Message"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(E\^\(\(a\ x\)\/2\)\ y - \[Zeta]\_1 == 0\)},
          {\(E\^\(-\(\(a\ x\)\/2\)\)\ \[CapitalPsi] - F\_1 == 0\)},
          {
            \(\(-\[Zeta]\_1\%4\)\ 
                \((2\ a\ k\^2 - 2\ a\ \((F\_1)\)\_\[Zeta]\_1\%2 + 
                    a\ F\_1\ \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1\) + 
                    2\ \[Eta]\ 
                      \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1, \[Zeta]\_1\))
                  \) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell["\<\
The reduced equation is again of the Falkner-Skan type. Thus, both \
types of the mainstream velocity result into the same type of  equation.\
\>",
   "Text",
  Evaluatable->False,
  CellTags->{"reduced equation", "Faklner-Skan equation"}]
}, Open  ]],

Cell[CellGroupData[{

Cell["5.6.6.4 Group Classification", "Subsubsection",
  CellTags->{"Group classification", "Group Classification"}],

Cell[TextData[{
  "The general topic behind the calculations carried out above is the problem \
of group classification of a partial differential equation. The question is \
formulated as follows. Assume we have a system of equations containing a \
certain arbitrariness, expressed in the dependence of the equations on \
certain parameters or functions. These equations admit a certain group \
\[GothicCapitalG]. If we now change the arbitrariness to a specific form, we \
may observe that the group \[GothicCapitalG] is enlarged. This behavior of \
enlargement of a group was the result of our previous discussion. The \
question now is: Can we find the specific forms for the mainstream velocity ",
  
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  " discussed in the previous sections by using the functions of ",
  StyleBox["MathLie",
    FontSlant->"Italic"],
  "? The problem of group classification is closely connected with the common \
factors occurring in the determining equations. These common factors are \
eliminated by the functions of ",
  StyleBox["MathLie",
    FontSlant->"Italic"],
  ". The information removed from the determining equations is not lost but \
collected in a global variable called ",
  StyleBox["EliminatedFactors", "MmaText",
    FontSlant->"Italic"],
  ". This list collects all factors removed by the functions ",
  StyleBox["Lie[]", "MmaText"],
  ", ",
  StyleBox["LieSolve[]", "MmaText"],
  ", ",
  StyleBox["Infinitesimals[]", "MmaText"],
  ", ",
  StyleBox["DeterminingEquations[],", "MmaText"],
  " and ",
  StyleBox["PDESolve[]", "MmaText"],
  ". If we need to solve the classification problem, we have to examine the \
list ",
  StyleBox["EliminatedFactors", "MmaText",
    FontSlant->"Italic"],
  ".  "
}], "Text",
  Evaluatable->False,
  CellTags->{
  "group classification", "enlargement of a group", "EliminatedFactors", 
    "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
    "PDESolve[]", "common factors"}],

Cell[TextData[{
  "The following considerations will illustrate the special cases for the \
mainstream velocity discussed above. All models discussed so far follow from \
a group classification and can be calculated from the eliminated common \
factors. We start the determination of the general classification problem by \
calling the function ",
  StyleBox["Infinitesimals[]:", "MmaText"]
}], "Text",
  Evaluatable->False,
  CellTags->{"mainstream velocity", "Infinitesimals[]"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbound = 
      Infinitesimals[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]}, {
          \[CapitalPsi]}, {x, y}, {\[Nu]}]; symbound // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(\[Phi]\_1 == k1\)},
          {\(\[Xi]\_1 == 0\)},
          {\(\[Xi]\_2 == \[ScriptCapitalF]\_1\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], k1], 
        Equal[ 
          Subscript[ \[Xi], 1], 0], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Subscript[ \[ScriptCapitalF], 1]]}]]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The factors which were cancelled in the derivation of the determining \
equations can be inspected just by reading the variable ",
  StyleBox["EliminatedFactors:", "MmaText",
    FontSlant->"Italic"]
}], "Text",
  Evaluatable->False,
  CellTags->"determining equations"],

Cell[CellGroupData[{

Cell[BoxData[
    \(elFactor = EliminatedFactors; elFactor // \ LieTraditionalForm\)], 
  "Input"],

Cell[BoxData[
    \({\(-1\), 1, 4\/3, \(-3\)\ W\ W\_x, 
      W\_x\/\(3\ W\) + W\_\(x, x\)\/\(3\ W\_x\), 
      W\_x\%3\/\(W\ \((W\_x\%2 + W\ W\_\(x, x\))\)\) - 
        \(W\_x\ W\_\(x, x\)\)\/\(W\_x\%2 + W\ W\_\(x, x\)\) + 
        \(W\ W\_\(x, x\)\%2\)\/\(W\_x\ \((W\_x\%2 + W\ W\_\(x, x\))\)\) - 
        \(W\ W\_\(x, x, x\)\)\/\(W\_x\%2 + W\ W\_\(x, x\)\), 
      \(-\(\(3\ W\_x\ W\_\(x, x\)\)\/\(4\ 
                \((W\_x\%2 + W\ W\_\(x, x\))\)\)\)\) - 
        \(W\ W\_\(x, x, x\)\)\/\(4\ \((W\_x\%2 + W\ W\_\(x, x\))\)\), 
      \(-\(W\_x\%3\/\(W\ \((W\_x\%2 + W\ W\_\(x, x\))\)\)\)\) + 
        \(W\_x\ W\_\(x, x\)\)\/\(4\ \((W\_x\%2 + W\ W\_\(x, x\))\)\) - 
        \(W\ W\_\(x, x\)\%2\)\/\(W\_x\ \((W\_x\%2 + W\ W\_\(x, x\))\)\) + 
        \(3\ W\ W\_\(x, x, x\)\)\/\(4\ \((W\_x\%2 + W\ W\_\(x, x\))\)\), 
      \(9\ W\_x\%5\ W\_\(x, x\)\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) - \(27\ W\ W\_x\%3\ 
            W\_\(x, x\)\%2\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) + \(18\ W\^2\ W\_x\ 
            W\_\(x, x\)\%3\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) + \(15\ W\ W\_x\%4\ 
            W\_\(x, x, x\)\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) - \(18\ W\^2\ W\_x\%2\ W\_\(x, x\)\ 
            W\_\(x, x, x\)\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) + \(3\ W\^3\ W\_\(x, x\)\%2\ 
            W\_\(x, x, x\)\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) - \(6\ W\^3\ W\_x\ 
            W\_\(x, x, x\)\%2\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) + \(3\ W\^2\ W\_x\%3\ 
            W\_\(x, x, x, x\)\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) + \(3\ W\^3\ W\_x\ W\_\(x, x\)\ 
            W\_\(x, x, x, x\)\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))\)\)}
      \)], "Output"]
}, Open  ]],

Cell[TextData[{
  "In addition to three numerical factors, the list contains six relations \
which determine the mainstream velocity ",
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  " by a differential equation. In the following calculations, we will show \
that all models for ",
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  " discussed so far are contained in these equations. Let us start with the \
first equation which is extracted from position four of the list ",
  StyleBox["elFactor", "MmaText",
    FontSlant->"Italic"],
  ". Applying ",
  StyleBox["DSolve[]", "MmaText"],
  " to this equation, we get"
}], "Text",
  Evaluatable->False,
  CellTags->"DSolve[]"],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq1 = elFactor\[LeftDoubleBracket]4\[RightDoubleBracket] == 0; 
    eq1 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(\(-3\)\ W\ W\_x == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(DSolve[eq1, W, x]\)], "Input"],

Cell[BoxData[
    \({{W \[Rule] \((C[1]&)\)}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The result shows that any constant is sufficient to satisfy this equation. \
The constant case for the mainstream velocity is the model discussed by ",
  ButtonBox["Blasius [1908]",
    ButtonData:>{"Literat.nb", "Blas-08"},
    ButtonStyle->"Hyperlink"],
  ". The fifth equation from the list of common factors gives us"
}], "Text",
  Evaluatable->False,
  CellTags->"Blasius model"],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq2 = elFactor\[LeftDoubleBracket]5\[RightDoubleBracket] == 0; 
    eq2 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            RowBox[{
              RowBox[{
                StyleBox[\(W\_x\/\(3\ W\)\),
                  ScriptLevel->0], "+", 
                StyleBox[\(W\_\(x, x\)\/\(3\ W\_x\)\),
                  ScriptLevel->0]}], "==", "0"}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(DSolve[eq2, W, x]\)], "Input"],

Cell[BoxData[
    \({{W \[Rule] \((\(-\@\(E\^\(3\ C[2]\)\ \((2\ #1 - 2\ C[1])\)\)\)&)\)}, {
        W \[Rule] \((\@\(E\^\(3\ C[2]\)\ \((2\ #1 - 2\ C[1])\)\)&)\)}}\)], 
  "Output"]
}, Open  ]],

Cell[TextData[{
  "This solution is a special case of the ",
  ButtonBox["Falkner-Skan",
    ButtonData:>{"Literat.nb", "Falk-31"},
    ButtonStyle->"Hyperlink"],
  " model with ",
  Cell[BoxData[
      \(TraditionalForm\`m = 1/2\)]],
  ". Thus, the eliminated prefactors contain at least the special case of ",
  Cell[BoxData[
      FormBox[
        RowBox[{\(W(x)\), "=", 
          RowBox[{"k", " ", 
            FormBox[\(x\^\(1/2\)\),
              "TraditionalForm"]}]}], TraditionalForm]]],
  ". The sixth equation of our list contains a very complicated non-linear \
third-order ordinary differential equation which we use in the form"
}], "Text",
  Evaluatable->False,
  CellTags->"Falkner-Skan model"],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq3 = 
      Numerator[
        Together[elFactor\[LeftDoubleBracket]6\[RightDoubleBracket]]]; 
    eq3 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            \(W\_x\%4 - W\ W\_x\%2\ W\_\(x, x\) + W\^2\ W\_\(x, x\)\%2 - 
                W\^2\ W\_x\ W\_\(x, x, x\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "If we try to solve this equation by using ",
  StyleBox["DSolve[],", "MmaText"],
  " we end up with"
}], "Text",
  Evaluatable->False,
  CellTags->"DSolve[]"],

Cell[CellGroupData[{

Cell[BoxData[
    \(DSolve[eq3 == 0, W, x]\)], "Input"],

Cell[BoxData[
    RowBox[{"DSolve", "[", 
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox[
              RowBox[{
                SuperscriptBox["W", "\[Prime]",
                  MultilineFunction->None], "[", "x", "]"}], "4"], "-", 
            RowBox[{\(W[x]\), " ", 
              SuperscriptBox[
                RowBox[{
                  SuperscriptBox["W", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], " ", 
              RowBox[{
                SuperscriptBox["W", "\[DoublePrime]",
                  MultilineFunction->None], "[", "x", "]"}]}], "+", 
            RowBox[{\(W[x]\^2\), " ", 
              SuperscriptBox[
                RowBox[{
                  SuperscriptBox["W", "\[DoublePrime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"]}], "-", 
            RowBox[{\(W[x]\^2\), " ", 
              RowBox[{
                SuperscriptBox["W", "\[Prime]",
                  MultilineFunction->None], "[", "x", "]"}], " ", 
              RowBox[{
                SuperscriptBox["W", 
                  TagBox[\((3)\),
                    Derivative],
                  MultilineFunction->None], "[", "x", "]"}]}]}], "==", "0"}], 
        ",", "W", ",", "x"}], "]"}]], "Output"]
}, Open  ]],

Cell["However, the equation is solved by an exponential function", "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq3 /. W \[Rule] Function[x, k\ Exp[a\ x]]\)], "Input"],

Cell[BoxData[
    \(0\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "This result shows that the exponential model discussed earlier is also \
consistent with the determining equations for the mainstream velocity. The \
seventh equation of our list ",
  StyleBox["elFactor", "MmaText",
    FontSlant->"Italic"],
  " is connected with the equation ",
  StyleBox["eq2", "MmaText",
    FontSlant->"Italic"],
  ". We can show this by just integrating the equation with respect to ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  ":"
}], "Text",
  Evaluatable->False,
  CellTags->"determining equations"],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq4 = 
      Numerator[
        Together[elFactor\[LeftDoubleBracket]7\[RightDoubleBracket]]]; 
    eq4 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(\(-3\)\ W\_x\ W\_\(x, x\) - W\ W\_\(x, x, x\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[Integral]eq4 \[DifferentialD]x\)], "Input"],

Cell[BoxData[
    RowBox[{
      RowBox[{"-", 
        SuperscriptBox[
          RowBox[{
            SuperscriptBox["W", "\[Prime]",
              MultilineFunction->None], "[", "x", "]"}], "2"]}], "-", 
      RowBox[{\(W[x]\), " ", 
        RowBox[{
          SuperscriptBox["W", "\[DoublePrime]",
            MultilineFunction->None], "[", "x", "]"}]}]}]], "Output"]
}, Open  ]],

Cell[TextData[{
  "Thus, no more information is gained by considering this equation. The \
eighth equation is a third-order ODE which cannot be treated by ",
  StyleBox["DSolve[]", "MmaText"],
  ". The solution of this third-order ODE is again a special case of the \
Falkner-Skan type with ",
  Cell[BoxData[
      \(TraditionalForm\`m = \(-1\)/2\)]],
  ":"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq5 = 
      Numerator[
        Together[elFactor\[LeftDoubleBracket]8\[RightDoubleBracket]]]; 
    eq5 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            \(\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\) == 0
              \)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Simplify[eq5 /. W \[Rule] Function[x, k\ x\^m]]\)], "Input"],

Cell[BoxData[
    \(\(-2\)\ k\^4\ m\^2\ \((\(-1\) + m + 2\ m\^2)\)\ 
      x\^\(4\ \((\(-1\) + m)\)\)\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The last relation is a fourth-order ODE not solvable by ",
  StyleBox["DSolve[]", "MmaText"],
  ". A particular solution of this equation is, however, given by the \
Falkner-Skan relation for the mainstream velocity. We can check this by"
}], "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq6 = 
      Numerator[
        Together[elFactor\[LeftDoubleBracket]9\[RightDoubleBracket]]]; 
    eq6 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            \(\(-3\)\ 
                \((3\ W\_x\%5\ W\_\(x, x\) - 9\ W\ W\_x\%3\ W\_\(x, x\)\%2 + 
                    6\ W\^2\ W\_x\ W\_\(x, x\)\%3 + 
                    5\ W\ W\_x\%4\ W\_\(x, x, x\) - 
                    6\ W\^2\ W\_x\%2\ W\_\(x, x\)\ W\_\(x, x, x\) + 
                    W\^3\ W\_\(x, x\)\%2\ W\_\(x, x, x\) - 
                    2\ W\^3\ W\_x\ W\_\(x, x, x\)\%2 + 
                    W\^2\ W\_x\%3\ W\_\(x, x, x, x\) + 
                    W\^3\ W\_x\ W\_\(x, x\)\ W\_\(x, x, x, x\))\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        ColumnWidths->0.95,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Simplify[eq6 /. W \[Rule] Function[x, k\ x\^m]]\)], "Input"],

Cell[BoxData[
    \(0\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Thus, we demonstrated that the group classification problem for the steady \
two-dimensional flow is solved by three types of mainstream velocities: (i) ",
  
  Cell[BoxData[
      \(TraditionalForm\`W(x)\  = \(const . \)\)]],
  ", (ii)",
  Cell[BoxData[
      FormBox[
        RowBox[{" ", 
          RowBox[{\(W(x)\), " ", "=", " ", 
            RowBox[{"k", " ", 
              FormBox[\(x\^m\),
                "TraditionalForm"]}]}]}], TraditionalForm]]],
  ", and (iii) ",
  Cell[BoxData[
      FormBox[
        RowBox[{\(W(x)\), " ", "=", " ", 
          RowBox[{"k", " ", 
            FormBox[
              SuperscriptBox["e", 
                StyleBox["ax",
                  FontSlant->"Italic"]],
              "TraditionalForm"]}]}], TraditionalForm]]],
  ". Special cases also contained in the classification of type (ii) are the \
cases with ",
  Cell[BoxData[
      \(TraditionalForm\`m = 1/2\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`m = \(-1\)/2\)]],
  ". The maximal group order of four occurs for the type (i), all other types \
possess a lower group order."
}], "Text",
  Evaluatable->False,
  CellTags->"group classification problem"],

Cell[TextData[ButtonBox["Next notebook",
  ButtonData:>{"Jet.nb", None},
  ButtonStyle->"Hyperlink"]], "Text",
  CellFrame->{{0, 0}, {0, 0.5}}]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 1024}, {0, 695}},
ScreenStyleEnvironment->"Working",
WindowToolbars->"EditBar",
Evaluator->"Local",
WindowSize->{685, 668},
WindowMargins->{{20, Automatic}, {Automatic, 0}},
PrintingCopies->1,
PrintingStartingPageNumber->319,
PrintingPageRange->{1, Automatic},
PageHeaders->{{Cell[ 
        TextData[ {
          StyleBox[ 
            CounterBox[ "Page"], "PageNumber"], "     ", 
          StyleBox[ 
          "Point Symmetries of Partial Differential Equations", FontSlant -> 
            "Italic"]}], "Header"], Inherited, None}, {
    None, Inherited, Cell[ 
        TextData[ {
          StyleBox[ 
            CounterBox[ 
            "Section", CounterFunction :> (Part[ {"Working Examples"}, #]&)], 
            FontSlant -> "Italic"], "     ", 
          StyleBox[ 
            CounterBox[ "Page"], "PageNumber"]}], "Header"]}},
PageHeaderLines->{False, False},
PrintingOptions->{"FirstPageHeader"->True,
"FirstPageFooter"->False,
"FacingPages"->True},
PrivateNotebookOptions->{"ColorPalette"->{RGBColor, -1}},
ShowCellLabel->True,
RenderingOptions->{"ObjectDithering"->True,
"RasterDithering"->False},
Magnification->1.25,
StyleDefinitions -> "BookStyles.nb"
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "Two-Dimensional Boundary-Layer Flows"->{
    Cell[1739, 51, 426, 8, 49, "Subsection",
      CounterAssignments->{{"Title", 5}, {"SubTitle", 0}, {
          "SubSubTitle", 0}, {"Section", 0}, {"SubSection", 0}, {
          "SubSubSection", 0}, {"NumberedEquation", 60}, {
          "NumberedFigure", 0}, {"NumberedTable", 0}},
      CellTags->"Two-Dimensional Boundary-Layer Flows"]},
  "boundary-layer flow"->{
    Cell[2168, 61, 1334, 32, 222, "Text",
      Evaluatable->False,
      CellTags->{
      "boundary-layer flow", "group classification", 
        "symmetry calculation"}]},
  "group classification"->{
    Cell[2168, 61, 1334, 32, 222, "Text",
      Evaluatable->False,
      CellTags->{
      "boundary-layer flow", "group classification", "symmetry calculation"}],
    
    Cell[40800, 1388, 1987, 45, 306, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "symmetry calculation"->{
    Cell[2168, 61, 1334, 32, 222, "Text",
      Evaluatable->False,
      CellTags->{
      "boundary-layer flow", "group classification", 
        "symmetry calculation"}]},
  "Navier-Stokes equations"->{
    Cell[9843, 395, 556, 12, 75, "Text",
      Evaluatable->False,
      CellTags->{
      "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
        "continuity equation", "momentum equation", "steady flow"}]},
  "boundary-layer equation"->{
    Cell[9843, 395, 556, 12, 75, "Text",
      Evaluatable->False,
      CellTags->{
      "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
        "continuity equation", "momentum equation", "steady flow"}]},
  "buoyancy"->{
    Cell[9843, 395, 556, 12, 75, "Text",
      Evaluatable->False,
      CellTags->{
      "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
        "continuity equation", "momentum equation", "steady flow"}]},
  "continuity equation"->{
    Cell[9843, 395, 556, 12, 75, "Text",
      Evaluatable->False,
      CellTags->{
      "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
        "continuity equation", "momentum equation", "steady flow"}]},
  "momentum equation"->{
    Cell[9843, 395, 556, 12, 75, "Text",
      Evaluatable->False,
      CellTags->{
      "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
        "continuity equation", "momentum equation", "steady flow"}]},
  "steady flow"->{
    Cell[9843, 395, 556, 12, 75, "Text",
      Evaluatable->False,
      CellTags->{
      "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
        "continuity equation", "momentum equation", "steady flow"}]},
  "boundary condition"->{
    Cell[10815, 422, 401, 12, 54, "Text",
      CellTags->"boundary condition"]},
  "Prandtl's boundary layer equations"->{
    Cell[11959, 470, 269, 5, 28, "Input",
      CellTags->"Prandtl's boundary layer equations"]},
  "partial differential equations of parabolic type"->{
    Cell[12553, 489, 609, 17, 75, "Text",
      Evaluatable->False,
      CellTags->{
      "partial differential equations of parabolic type", 
        "Prandtel's boundary-layer equations", "stream function"}]},
  "Prandtel's boundary-layer equations"->{
    Cell[12553, 489, 609, 17, 75, "Text",
      Evaluatable->False,
      CellTags->{
      "partial differential equations of parabolic type", 
        "Prandtel's boundary-layer equations", "stream function"}]},
  "stream function"->{
    Cell[12553, 489, 609, 17, 75, "Text",
      Evaluatable->False,
      CellTags->{
      "partial differential equations of parabolic type", 
        "Prandtel's boundary-layer equations", "stream function"}],
    Cell[28088, 997, 310, 11, 54, "Text",
      Evaluatable->False,
      CellTags->"stream function"]},
  "mainstream velocities"->{
    Cell[14250, 548, 499, 12, 75, "Text",
      Evaluatable->False,
      CellTags->"mainstream velocities"]},
  "The Blasius Solution"->{
    Cell[15802, 598, 89, 1, 37, "Subsubsection",
      CellTags->"The Blasius Solution"]},
  "infinitesimals of Blasius' model"->{
    Cell[15894, 601, 411, 10, 75, "Text",
      CellTags->"infinitesimals of Blasius' model"]},
  "LieReduction[]"->{
    Cell[17438, 653, 687, 19, 96, "Text",
      Evaluatable->False,
      CellTags->"LieReduction[]"]},
  "similarity solution"->{
    Cell[19415, 715, 189, 5, 54, "Text",
      Evaluatable->False,
      CellTags->"similarity solution"],
    Cell[32105, 1128, 751, 19, 138, "Text",
      Evaluatable->False,
      CellTags->"similarity solution"]},
  "Blasius model"->{
    Cell[21622, 799, 454, 12, 75, "Text",
      Evaluatable->False,
      CellTags->"Blasius model"],
    Cell[48666, 1586, 403, 9, 75, "Text",
      Evaluatable->False,
      CellTags->"Blasius model"]},
  "DSolve[]"->{
    Cell[23843, 863, 182, 6, 33, "Text",
      Evaluatable->False,
      CellTags->"DSolve[]"],
    Cell[35275, 1221, 887, 16, 180, "Text",
      CellTags->{"DSolve[]", "Falkner-Skan equation", "mainstream velocity"}],
    
    Cell[47445, 1540, 679, 18, 117, "Text",
      Evaluatable->False,
      CellTags->"DSolve[]"],
    Cell[51177, 1673, 177, 6, 33, "Text",
      Evaluatable->False,
      CellTags->"DSolve[]"]},
  "Lie's method"->{
    Cell[25383, 909, 242, 5, 54, "Text",
      Evaluatable->False,
      CellTags->"Lie's method"]},
  "Blasius equation"->{
    Cell[26489, 947, 851, 19, 138, "Text",
      Evaluatable->False,
      CellTags->{"Blasius equation", "reduction of the order"}]},
  "reduction of the order"->{
    Cell[26489, 947, 851, 19, 138, "Text",
      Evaluatable->False,
      CellTags->{"Blasius equation", "reduction of the order"}]},
  "Falkner Skan Solution"->{
    Cell[27377, 971, 91, 1, 37, "Subsubsection",
      CellTags->"Falkner Skan Solution"]},
  "power law"->{
    Cell[27471, 974, 434, 12, 54, "Text",
      Evaluatable->False,
      CellTags->{"power law", "Falkner", "Skan"}]},
  "Falkner"->{
    Cell[27471, 974, 434, 12, 54, "Text",
      Evaluatable->False,
      CellTags->{"power law", "Falkner", "Skan"}]},
  "Skan"->{
    Cell[27471, 974, 434, 12, 54, "Text",
      Evaluatable->False,
      CellTags->{"power law", "Falkner", "Skan"}]},
  "Falkner-Skan model"->{
    Cell[29778, 1056, 931, 25, 138, "Text",
      Evaluatable->False,
      CellTags->"Falkner-Skan model"],
    Cell[49941, 1631, 711, 19, 96, "Text",
      Evaluatable->False,
      CellTags->"Falkner-Skan model"]},
  "stream function equation"->{
    Cell[31048, 1095, 113, 1, 33, "Text",
      CellTags->"stream function equation"]},
  "reduction"->{
    Cell[33480, 1170, 282, 9, 54, "Text",
      CellTags->"reduction"]},
  "Falkner-Skan equation"->{
    Cell[35275, 1221, 887, 16, 180, "Text",
      CellTags->{
      "DSolve[]", "Falkner-Skan equation", "mainstream velocity"}]},
  "mainstream velocity"->{
    Cell[35275, 1221, 887, 16, 180, "Text",
      CellTags->{"DSolve[]", "Falkner-Skan equation", "mainstream velocity"}],
    
    Cell[42790, 1435, 481, 9, 96, "Text",
      Evaluatable->False,
      CellTags->{"mainstream velocity", "Infinitesimals[]"}]},
  "Exponential mainstream velocity"->{
    Cell[36199, 1242, 151, 2, 37, "Subsubsection",
      CellTags->{
      "Exponential mainstream velocity", 
        "Exponential Mainstream Velocity"}]},
  "Exponential Mainstream Velocity"->{
    Cell[36199, 1242, 151, 2, 37, "Subsubsection",
      CellTags->{
      "Exponential mainstream velocity", 
        "Exponential Mainstream Velocity"}]},
  "exponential mainstream velocity"->{
    Cell[36353, 1246, 246, 6, 54, "Text",
      Evaluatable->False,
      CellTags->"exponential mainstream velocity"]},
  "infinitesimals"->{
    Cell[36792, 1263, 141, 5, 33, "Text",
      Evaluatable->False,
      CellTags->"infinitesimals"]},
  "symmetry group"->{
    Cell[38028, 1308, 741, 17, 117, "Text",
      Evaluatable->False,
      CellTags->{"symmetry group", "finite group"}]},
  "finite group"->{
    Cell[38028, 1308, 741, 17, 117, "Text",
      Evaluatable->False,
      CellTags->{"symmetry group", "finite group"}]},
  "reduced equation"->{
    Cell[40396, 1374, 249, 6, 54, "Text",
      Evaluatable->False,
      CellTags->{"reduced equation", "Faklner-Skan equation"}]},
  "Faklner-Skan equation"->{
    Cell[40396, 1374, 249, 6, 54, "Text",
      Evaluatable->False,
      CellTags->{"reduced equation", "Faklner-Skan equation"}]},
  "Group classification"->{
    Cell[40682, 1385, 115, 1, 37, "Subsubsection",
      CellTags->{"Group classification", "Group Classification"}]},
  "Group Classification"->{
    Cell[40682, 1385, 115, 1, 37, "Subsubsection",
      CellTags->{"Group classification", "Group Classification"}]},
  "enlargement of a group"->{
    Cell[40800, 1388, 1987, 45, 306, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "EliminatedFactors"->{
    Cell[40800, 1388, 1987, 45, 306, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "Lie[]"->{
    Cell[40800, 1388, 1987, 45, 306, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "LieSolve[]"->{
    Cell[40800, 1388, 1987, 45, 306, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "Infinitesimals[]"->{
    Cell[40800, 1388, 1987, 45, 306, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}],
    Cell[42790, 1435, 481, 9, 96, "Text",
      Evaluatable->False,
      CellTags->{"mainstream velocity", "Infinitesimals[]"}]},
  "DeterminingEquations[]"->{
    Cell[40800, 1388, 1987, 45, 306, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "PDESolve[]"->{
    Cell[40800, 1388, 1987, 45, 306, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "common factors"->{
    Cell[40800, 1388, 1987, 45, 306, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "determining equations"->{
    Cell[44018, 1473, 289, 7, 54, "Text",
      Evaluatable->False,
      CellTags->"determining equations"],
    Cell[52985, 1732, 552, 15, 96, "Text",
      Evaluatable->False,
      CellTags->"determining equations"]},
  "group classification problem"->{
    Cell[57021, 1875, 1190, 35, 117, "Text",
      Evaluatable->False,
      CellTags->"group classification problem"]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"Two-Dimensional Boundary-Layer Flows", 60096, 1966},
  {"boundary-layer flow", 60466, 1973},
  {"group classification", 60669, 1979},
  {"symmetry calculation", 61155, 1991},
  {"Navier-Stokes equations", 61361, 1997},
  {"boundary-layer equation", 61620, 2003},
  {"buoyancy", 61864, 2009},
  {"continuity equation", 62119, 2015},
  {"momentum equation", 62372, 2021},
  {"steady flow", 62619, 2027},
  {"boundary condition", 62873, 2033},
  {"Prandtl's boundary layer equations", 62997, 2036},
  {"partial differential equations of parabolic type", 63151, 2039},
  {"Prandtel's boundary-layer equations", 63408, 2045},
  {"stream function", 63645, 2051},
  {"mainstream velocities", 63992, 2060},
  {"The Blasius Solution", 64131, 2064},
  {"infinitesimals of Blasius' model", 64262, 2067},
  {"LieReduction[]", 64380, 2070},
  {"similarity solution", 64511, 2074},
  {"Blasius model", 64750, 2081},
  {"DSolve[]", 64971, 2088},
  {"Lie's method", 65412, 2101},
  {"Blasius equation", 65537, 2105},
  {"reduction of the order", 65702, 2109},
  {"Falkner Skan Solution", 65866, 2113},
  {"power law", 65975, 2116},
  {"Falkner", 66110, 2120},
  {"Skan", 66242, 2124},
  {"Falkner-Skan model", 66388, 2128},
  {"stream function equation", 66638, 2135},
  {"reduction", 66743, 2138},
  {"Falkner-Skan equation", 66845, 2141},
  {"mainstream velocity", 67003, 2145},
  {"Exponential mainstream velocity", 67301, 2152},
  {"Exponential Mainstream Velocity", 67497, 2157},
  {"exponential mainstream velocity", 67693, 2162},
  {"infinitesimals", 67836, 2166},
  {"symmetry group", 67962, 2170},
  {"finite group", 68106, 2174},
  {"reduced equation", 68254, 2178},
  {"Faklner-Skan equation", 68416, 2182},
  {"Group classification", 68577, 2186},
  {"Group Classification", 68724, 2189},
  {"enlargement of a group", 68873, 2192},
  {"EliminatedFactors", 69186, 2199},
  {"Lie[]", 69487, 2206},
  {"LieSolve[]", 69793, 2213},
  {"Infinitesimals[]", 70105, 2220},
  {"DeterminingEquations[]", 70553, 2230},
  {"PDESolve[]", 70859, 2237},
  {"common factors", 71169, 2244},
  {"determining equations", 71486, 2251},
  {"group classification problem", 71744, 2258}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 426, 8, 49, "Subsection",
  CounterAssignments->{{"Title", 5}, {"SubTitle", 0}, {"SubSubTitle", 0}, {
      "Section", 0}, {"SubSection", 0}, {"SubSubSection", 0}, {
      "NumberedEquation", 60}, {"NumberedFigure", 0}, {"NumberedTable", 0}},
  CellTags->"Two-Dimensional Boundary-Layer Flows"],
Cell[2168, 61, 1334, 32, 222, "Text",
  Evaluatable->False,
  CellTags->{
  "boundary-layer flow", "group classification", "symmetry calculation"}],
Cell[3505, 95, 6335, 298, 135, 2603, 239, "GraphicsData", "PostScript", \
"Graphics",
  Evaluatable->False],
Cell[9843, 395, 556, 12, 75, "Text",
  Evaluatable->False,
  CellTags->{
  "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
    "continuity equation", "momentum equation", "steady flow"}],
Cell[10402, 409, 82, 1, 30, "NumberedEquation"],
Cell[10487, 412, 325, 8, 30, "NumberedEquation"],
Cell[10815, 422, 401, 12, 54, "Text",
  CellTags->"boundary condition"],
Cell[11219, 436, 105, 2, 28, "NumberedEquation"],
Cell[11327, 440, 41, 1, 33, "Text",
  Evaluatable->False],
Cell[11371, 443, 99, 2, 28, "NumberedEquation"],
Cell[11473, 447, 53, 1, 33, "Text",
  Evaluatable->False],
Cell[11529, 450, 82, 2, 27, "Input"],
Cell[11614, 454, 320, 12, 54, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[11959, 470, 269, 5, 28, "Input",
  CellTags->"Prandtl's boundary layer equations"],
Cell[12231, 477, 307, 9, 45, "Output"]
}, Open  ]],
Cell[12553, 489, 609, 17, 75, "Text",
  Evaluatable->False,
  CellTags->{
  "partial differential equations of parabolic type", 
    "Prandtel's boundary-layer equations", "stream function"}],

Cell[CellGroupData[{
Cell[13187, 510, 202, 4, 28, "Input"],
Cell[13392, 516, 181, 3, 30, "Output"]
}, Open  ]],
Cell[13588, 522, 119, 4, 33, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[13732, 530, 72, 1, 27, "Input"],
Cell[13807, 533, 428, 12, 43, "Output"]
}, Open  ]],
Cell[14250, 548, 499, 12, 75, "Text",
  Evaluatable->False,
  CellTags->"mainstream velocities"],

Cell[CellGroupData[{
Cell[14774, 564, 180, 3, 27, "Input"],
Cell[14957, 569, 525, 17, 57, "Output"]
}, Open  ]],
Cell[15497, 589, 280, 5, 75, "Text"],

Cell[CellGroupData[{
Cell[15802, 598, 89, 1, 37, "Subsubsection",
  CellTags->"The Blasius Solution"],
Cell[15894, 601, 411, 10, 75, "Text",
  CellTags->"infinitesimals of Blasius' model"],
Cell[16308, 613, 99, 2, 27, "Input"],

Cell[CellGroupData[{
Cell[16432, 619, 202, 4, 45, "Input"],
Cell[16637, 625, 786, 25, 57, "Output"]
}, Open  ]],
Cell[17438, 653, 687, 19, 96, "Text",
  Evaluatable->False,
  CellTags->"LieReduction[]"],

Cell[CellGroupData[{
Cell[18150, 676, 288, 5, 45, "Input"],
Cell[18441, 683, 47, 1, 30, "Output"]
}, Open  ]],
Cell[18503, 687, 96, 1, 33, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[18624, 692, 330, 6, 63, "Input"],
Cell[18957, 700, 443, 12, 59, "Output"]
}, Open  ]],
Cell[19415, 715, 189, 5, 54, "Text",
  Evaluatable->False,
  CellTags->"similarity solution"],

Cell[CellGroupData[{
Cell[19629, 724, 153, 4, 27, "Input"],
Cell[19785, 730, 109, 2, 34, "Output"]
}, Open  ]],
Cell[19909, 735, 110, 4, 33, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[20044, 743, 167, 4, 27, "Input"],
Cell[20214, 749, 166, 3, 33, "Output"]
}, Open  ]],
Cell[20395, 755, 468, 11, 96, "Text",
  Evaluatable->False],
Cell[20866, 768, 156, 3, 27, "Input"],

Cell[CellGroupData[{
Cell[21047, 775, 101, 2, 27, "Input"],
Cell[21151, 779, 196, 4, 51, "Output"]
}, Open  ]],
Cell[21362, 786, 257, 11, 33, "Text"],
Cell[21622, 799, 454, 12, 75, "Text",
  Evaluatable->False,
  CellTags->"Blasius model"],

Cell[CellGroupData[{
Cell[22101, 815, 288, 5, 45, "Input"],
Cell[22392, 822, 75, 1, 30, "Output"]
}, Open  ]],
Cell[22482, 826, 64, 0, 33, "Text"],

Cell[CellGroupData[{
Cell[22571, 830, 380, 7, 63, "Input"],
Cell[22954, 839, 180, 3, 69, "Message"],
Cell[23137, 844, 691, 16, 73, "Output"]
}, Open  ]],
Cell[23843, 863, 182, 6, 33, "Text",
  Evaluatable->False,
  CellTags->"DSolve[]"],

Cell[CellGroupData[{
Cell[24050, 873, 124, 3, 27, "Input"],
Cell[24177, 878, 1191, 28, 56, "Output"]
}, Open  ]],
Cell[25383, 909, 242, 5, 54, "Text",
  Evaluatable->False,
  CellTags->"Lie's method"],
Cell[25628, 916, 226, 4, 34, "Input"],

Cell[CellGroupData[{
Cell[25879, 924, 112, 2, 27, "Input"],
Cell[25994, 928, 480, 16, 41, "Output"]
}, Open  ]],
Cell[26489, 947, 851, 19, 138, "Text",
  Evaluatable->False,
  CellTags->{"Blasius equation", "reduction of the order"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[27377, 971, 91, 1, 37, "Subsubsection",
  CellTags->"Falkner Skan Solution"],
Cell[27471, 974, 434, 12, 54, "Text",
  Evaluatable->False,
  CellTags->{"power law", "Falkner", "Skan"}],

Cell[CellGroupData[{
Cell[27930, 990, 76, 1, 27, "Input"],
Cell[28009, 993, 64, 1, 30, "Output"]
}, Open  ]],
Cell[28088, 997, 310, 11, 54, "Text",
  Evaluatable->False,
  CellTags->"stream function"],

Cell[CellGroupData[{
Cell[28423, 1012, 205, 4, 45, "Input"],
Cell[28631, 1018, 1132, 35, 93, "Output"]
}, Open  ]],
Cell[29778, 1056, 931, 25, 138, "Text",
  Evaluatable->False,
  CellTags->"Falkner-Skan model"],

Cell[CellGroupData[{
Cell[30734, 1085, 249, 4, 45, "Input"],
Cell[30986, 1091, 47, 1, 30, "Output"]
}, Open  ]],
Cell[31048, 1095, 113, 1, 33, "Text",
  CellTags->"stream function equation"],

Cell[CellGroupData[{
Cell[31186, 1100, 376, 7, 63, "Input"],
Cell[31565, 1109, 525, 16, 77, "Output"]
}, Open  ]],
Cell[32105, 1128, 751, 19, 138, "Text",
  Evaluatable->False,
  CellTags->"similarity solution"],
Cell[32859, 1149, 185, 5, 54, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[33069, 1158, 249, 4, 45, "Input"],
Cell[33321, 1164, 144, 3, 42, "Output"]
}, Open  ]],
Cell[33480, 1170, 282, 9, 54, "Text",
  CellTags->"reduction"],

Cell[CellGroupData[{
Cell[33787, 1183, 376, 7, 63, "Input"],
Cell[34166, 1192, 180, 3, 69, "Message"],
Cell[34349, 1197, 911, 21, 103, "Output"]
}, Open  ]],
Cell[35275, 1221, 887, 16, 180, "Text",
  CellTags->{"DSolve[]", "Falkner-Skan equation", "mainstream velocity"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[36199, 1242, 151, 2, 37, "Subsubsection",
  CellTags->{
  "Exponential mainstream velocity", "Exponential Mainstream Velocity"}],
Cell[36353, 1246, 246, 6, 54, "Text",
  Evaluatable->False,
  CellTags->"exponential mainstream velocity"],

Cell[CellGroupData[{
Cell[36624, 1256, 81, 1, 27, "Input"],
Cell[36708, 1259, 69, 1, 30, "Output"]
}, Open  ]],
Cell[36792, 1263, 141, 5, 33, "Text",
  Evaluatable->False,
  CellTags->"infinitesimals"],

Cell[CellGroupData[{
Cell[36958, 1272, 204, 4, 45, "Input"],
Cell[37165, 1278, 848, 27, 75, "Output"]
}, Open  ]],
Cell[38028, 1308, 741, 17, 117, "Text",
  Evaluatable->False,
  CellTags->{"symmetry group", "finite group"}],

Cell[CellGroupData[{
Cell[38794, 1329, 249, 4, 45, "Input"],
Cell[39046, 1335, 89, 1, 42, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[39172, 1341, 376, 7, 63, "Input"],
Cell[39551, 1350, 180, 3, 69, "Message"],
Cell[39734, 1355, 647, 16, 73, "Output"]
}, Open  ]],
Cell[40396, 1374, 249, 6, 54, "Text",
  Evaluatable->False,
  CellTags->{"reduced equation", "Faklner-Skan equation"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[40682, 1385, 115, 1, 37, "Subsubsection",
  CellTags->{"Group classification", "Group Classification"}],
Cell[40800, 1388, 1987, 45, 306, "Text",
  Evaluatable->False,
  CellTags->{
  "group classification", "enlargement of a group", "EliminatedFactors", 
    "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
    "PDESolve[]", "common factors"}],
Cell[42790, 1435, 481, 9, 96, "Text",
  Evaluatable->False,
  CellTags->{"mainstream velocity", "Infinitesimals[]"}],

Cell[CellGroupData[{
Cell[43296, 1448, 179, 3, 27, "Input"],
Cell[43478, 1453, 525, 17, 57, "Output"]
}, Open  ]],
Cell[44018, 1473, 289, 7, 54, "Text",
  Evaluatable->False,
  CellTags->"determining equations"],

Cell[CellGroupData[{
Cell[44332, 1484, 98, 2, 27, "Input"],
Cell[44433, 1488, 2997, 49, 558, "Output"]
}, Open  ]],
Cell[47445, 1540, 679, 18, 117, "Text",
  Evaluatable->False,
  CellTags->"DSolve[]"],

Cell[CellGroupData[{
Cell[48149, 1562, 111, 2, 27, "Input"],
Cell[48263, 1566, 239, 8, 30, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[48539, 1579, 50, 1, 27, "Input"],
Cell[48592, 1582, 59, 1, 30, "Output"]
}, Open  ]],
Cell[48666, 1586, 403, 9, 75, "Text",
  Evaluatable->False,
  CellTags->"Blasius model"],

Cell[CellGroupData[{
Cell[49094, 1599, 111, 2, 27, "Input"],
Cell[49208, 1603, 449, 14, 45, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[49694, 1622, 50, 1, 27, "Input"],
Cell[49747, 1625, 179, 3, 40, "Output"]
}, Open  ]],
Cell[49941, 1631, 711, 19, 96, "Text",
  Evaluatable->False,
  CellTags->"Falkner-Skan model"],

Cell[CellGroupData[{
Cell[50677, 1654, 143, 4, 27, "Input"],
Cell[50823, 1660, 339, 10, 31, "Output"]
}, Open  ]],
Cell[51177, 1673, 177, 6, 33, "Text",
  Evaluatable->False,
  CellTags->"DSolve[]"],

Cell[CellGroupData[{
Cell[51379, 1683, 55, 1, 27, "Input"],
Cell[51437, 1686, 1284, 31, 50, "Output"]
}, Open  ]],
Cell[52736, 1720, 96, 1, 33, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[52857, 1725, 75, 1, 27, "Input"],
Cell[52935, 1728, 35, 1, 30, "Output"]
}, Open  ]],
Cell[52985, 1732, 552, 15, 96, "Text",
  Evaluatable->False,
  CellTags->"determining equations"],

Cell[CellGroupData[{
Cell[53562, 1751, 143, 4, 27, "Input"],
Cell[53708, 1757, 269, 8, 30, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[54014, 1770, 65, 1, 38, "Input"],
Cell[54082, 1773, 369, 10, 30, "Output"]
}, Open  ]],
Cell[54466, 1786, 370, 9, 75, "Text"],

Cell[CellGroupData[{
Cell[54861, 1799, 143, 4, 27, "Input"],
Cell[55007, 1805, 368, 11, 31, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[55412, 1821, 80, 1, 27, "Input"],
Cell[55495, 1824, 115, 2, 30, "Output"]
}, Open  ]],
Cell[55625, 1829, 291, 6, 75, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[55941, 1839, 143, 4, 27, "Input"],
Cell[56087, 1845, 764, 18, 51, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[56888, 1868, 80, 1, 27, "Input"],
Cell[56971, 1871, 35, 1, 30, "Output"]
}, Open  ]],
Cell[57021, 1875, 1190, 35, 117, "Text",
  Evaluatable->False,
  CellTags->"group classification problem"],
Cell[58214, 1912, 143, 3, 44, "Text"]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

