(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     57690,       1910]*)
(*NotebookOutlinePosition[     73394,       2316]*)
(*  CellTagsIndexPosition[     71184,       2257]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["\<\
5.6.6 Two-Dimensional Boundary Layer Flows: Group \
Classification\
\>", "Subsection",
  CounterAssignments->{{"Title", 5}, {"SubTitle", 0}, {"SubSubTitle", 0}, {
      "Section", 0}, {"SubSection", 0}, {"SubSubSection", 0}, {
      "NumberedEquation", 60}, {"NumberedFigure", 0}, {"NumberedTable", 0}},
  CounterBoxOptions->{CounterFunction:>CapitalRomanNumeral},
  CellTags->"Two-Dimensional Boundary-Layer Flows"],

Cell[TextData[{
  "In this section, we examine the problem of an incompressible boundary \
layer flow over a flat plate. We will not solve the physical problem but \
discuss the symmetries of several related models. The models differ from each \
other in the behavior of the vertical velocity component above the boundary. \
We demonstrate that the models are the result of group classification of the \
general equation. We show that the symmetry of the different models is \
closely connected with the symmetry of the general model. In addition, we \
illustrate that ",
  StyleBox["MathLie ",
    FontSlant->"Italic"],
  "is capable of extracting these models from a symmetry calculation. The \
physical arrangement of the flow is given in the following figure. The flow \
above a plate is governed by the mainstream velocity ",
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  ". The coordinates are denoted by ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " and the velocity components in the two directions are ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`v\)]],
  ", respectively."
}], "Text",
  Evaluatable->False,
  CellTags->{
  "boundary-layer flow", "group classification", "symmetry calculation"}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .36923 
MathPictureStart
%% Graphics
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.24359 0.0732601 0.0307692 0.0732601 [
[ 0 0 0 0 ]
[ 1 .36923 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
[ ] 0 setdash
0 g
p
P
0 0 m
1 0 L
1 .36923 L
0 .36923 L
closepath
clip
newpath
p
p
0 0 1 r
.24359 .00879 m
.24359 .03077 L
.97619 .03077 L
.97619 .00879 L
F
P
p
.004 w
.24359 .0381 m
.46337 .0381 L
s
p
.46337 .0381 m
.41337 .0256 L
.41337 .0381 L
.41337 .0506 L
.46337 .0381 L
F
.46337 .0381 m
.41337 .0256 L
.41337 .0381 L
.41337 .0506 L
.46337 .0381 L
s
P
P
p
.004 w
.24359 .0381 m
.24359 .25055 L
s
p
.24359 .25055 m
.25609 .20055 L
.24359 .20055 L
.23109 .20055 L
.24359 .25055 L
F
.24359 .25055 m
.25609 .20055 L
.24359 .20055 L
.23109 .20055 L
.24359 .25055 L
s
P
P
[(x)] .5 .0381 0 0 Mshowa
[(y)] .24359 .28718 0 0 Mshowa
p
.004 w
.39011 .10403 m
.5 .10403 L
s
p
.5 .10403 m
.45 .09153 L
.45 .10403 L
.45 .11653 L
.5 .10403 L
F
.5 .10403 m
.45 .09153 L
.45 .10403 L
.45 .11653 L
.5 .10403 L
s
P
P
p
.004 w
.39011 .10403 m
.39011 .21392 L
s
p
.39011 .21392 m
.40261 .16392 L
.39011 .16392 L
.37761 .16392 L
.39011 .21392 L
F
.39011 .21392 m
.40261 .16392 L
.39011 .16392 L
.37761 .16392 L
.39011 .21392 L
s
P
P
[(u)] .57326 .10403 0 0 Mshowa
[(v)] .39011 .28718 0 0 Mshowa
p
1 0 0 r
p
.004 w
.02381 .03077 m
.20696 .03077 L
s
p
.20696 .03077 m
.15696 .01827 L
.15696 .03077 L
.15696 .04327 L
.20696 .03077 L
F
.20696 .03077 m
.15696 .01827 L
.15696 .03077 L
.15696 .04327 L
.20696 .03077 L
s
P
P
P
p
1 0 0 r
p
.004 w
.02381 .10403 m
.20696 .10403 L
s
p
.20696 .10403 m
.15696 .09153 L
.15696 .10403 L
.15696 .11653 L
.20696 .10403 L
F
.20696 .10403 m
.15696 .09153 L
.15696 .10403 L
.15696 .11653 L
.20696 .10403 L
s
P
P
P
p
1 0 0 r
p
.004 w
.02381 .17729 m
.20696 .17729 L
s
p
.20696 .17729 m
.15696 .16479 L
.15696 .17729 L
.15696 .18979 L
.20696 .17729 L
F
.20696 .17729 m
.15696 .16479 L
.15696 .17729 L
.15696 .18979 L
.20696 .17729 L
s
P
P
P
p
1 0 0 r
p
.004 w
.02381 .25055 m
.20696 .25055 L
s
p
.20696 .25055 m
.15696 .23805 L
.15696 .25055 L
.15696 .26305 L
.20696 .25055 L
F
.20696 .25055 m
.15696 .23805 L
.15696 .25055 L
.15696 .26305 L
.20696 .25055 L
s
P
P
P
p
1 0 0 r
p
.004 w
.02381 .32381 m
.20696 .32381 L
s
p
.20696 .32381 m
.15696 .31131 L
.15696 .32381 L
.15696 .33631 L
.20696 .32381 L
F
.20696 .32381 m
.15696 .31131 L
.15696 .32381 L
.15696 .33631 L
.20696 .32381 L
s
P
P
P
[(W)] .17033 .36044 0 0 Mshowa
P
% End of Graphics
MathPictureEnd\
\>"], "Graphics",
  PageWidth->Infinity,
  Evaluatable->False,
  ImageSize->{215.5, 83.5},
  ImageMargins->{{54.5, 0}, {0, 0}},
  ImageRegion->{{-0.015811, 1.01641}, {-0.34506, 1.30239}},
  FontFamily->"Courier New",
  FontSize->8,
  FontWeight->"Plain",
  FontSlant->"Plain",
  FontTracking->"Plain",
  FontColor->GrayLevel[0],
  Background->GrayLevel[1],
  FontVariations->{"Underline"->False,
  "Outline"->False,
  "Shadow"->False},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40003G0000Da000`40O003h00Oegoo003GOol0039o
oj807`=oo`007goo0g`047ooXP0O0goo000OOol9O00:OolV01l2001j01l3Ool00003Ooml07`002Ql
00MoobH07`H000X07`80000301l0000006L07`=oo`007goo2W`0;goo2P001goo00<007oo0000Jgoo
000OOol6O00=Oolb0006Ool00`00Oomoo`1ZOol001moo`9l015oo`03001oogoo02=oo`X000Ioo`80
0003Ool0000006Yoo`00<Woo00<007ooOol08goo1P00NGoo000bOol00`00Oomoo`0SOol2001mOol0
039oo`03001oogoo0:9oo`00<Woo00<007ooOol0XWoo000bOol00`00Oomoo`2ROol0039oo`03001o
ogoo0:9oo`00<Woo00<007ooOol0XWoo000bOol00`00Oomoo`2ROol0039oo`03001oogoo0:9oo`00
<Woo00<007ooOol0XWoo000OOol2O00AOol00`00Oomoo`0[Ool2001eOol001moo`Il00eoo`03001o
ogoo02]oo`H001Aoo`@005Uoo`007goo2W`02Goo00<007ooOol0:goo2P003goo00@007ooOol005Yo
o`0000=oog`0O000:7`01goo00<007ooOol07Goo6P003Goo00@007ooOol005Yoo`007goo2G`02Woo
00<007ooOol07Goo00<007ooOol02goo2@003goo0P0000=oo`000000FWoo000OOol3O00@Ool00`00
Oomoo`0MOol00`00Oomoo`0;Ool3001dOol0039oo`03001oogoo01eoo`03001oogoo089oo`00<Woo
00<007ooOol07Goo00<007ooOol0PWoo000bOol00`00Oomoo`0MOol00`00Oomoo`22Ool0039oo`03
001oogoo01eoo`03001oogoo089oo`00<Woo00<007ooOol07Goo00<007ooOol0PWoo000bOol00`00
Oomoo`0MOol00`00Oomoo`22Ool0039oo`03001oogoo01eoo`03001oogoo089oo`00<Woo00<007oo
Ool07Goo00<007ooOol0PWoo000bOol00`00Oomoo`0MOol00`00Oomoo`22Ool0039oo`03001oogoo
01eoo`03001oogoo089oo`007goo0W`04Goo00<007ooOol06goo1P00PGoo000OOol6O00=Ool00`00
Oomoo`0KOol60021Ool001moo`Yl00Uoo`03001oogoo01]oo`D0089oo`0000=oog`0O000:7`01goo
00<007ooOol077oo1000PWoo000OOol:O009Ool00`00Oomoo`0LOol40022Ool001moo`Il00eoo`03
001oogoo01aoo`@0089oo`007goo0W`04Goo00<007ooOol077oo0`00Pgoo000bOol00`00Oomoo`0L
Ool30023Ool002moo`L001]oo`<008=oo`00;goo1`0077oo0P00Pgoo000`Ool5000MOol00`00Oomo
o`22Ool0031oo`D001eoo`03001oogoo089oo`00<7oo1@00XWoo000`Ool5002ROol0035oo`<00:=o
o`00<Goo0`00Xgoo000OOol2O00@Ool3002SOol001moo`Il00aoo`<00:=oo`007goo2W`02Goo00<0
07ooOol0XWoo00000gooO01l000XO007Ool00`00Oomoo`2ROol001moo`Yl0:ioo`007goo1W`0/Woo
000OOol2O02fOol0031oo`800:Eoo`00<Goo00<007ooOol0Xgoo000bOol00`00Oomoo`0LOol20024
Ool0035oo`03001oo`0001eoo`8008Aoo`00<7oo00@007ooOol001aoo`04001oogoo0023Ool002mo
o`<00003Ool0000001Yoo`80009oo`80089oo`00egoo003GOol00=Moo`007goo0W`0]Woo000OOol6
O02bOol001moo`Yl0:ioo`0000=oog`0O000:7`0[7oo000OOol:O02^Ool001moo`Il0;9oo`007goo
0W`0]Woo003GOol00=Moo`008Goo00<007oo0000/goo000POol01@00Ool007oo0000/Woo000POol0
1@00Ool007oo0000/Woo000POol01@00Ool007oo0000/Woo000POol01@00Oomoogoo0000/Woo000O
Ool300000goo0000002bOol00=Moo`00\
\>"],
  ImageRangeCache->{{{-3.375, 218}, {111.313, -25.1875}} -> {-3.11697, \
-3.66142, 0.0616607, 0.0616607}}],

Cell[TextData[{
  "The general two-dimensional, steady, laminar incompressible boundary layer \
equations follow from the Navier-Stokes equations if we neglect buoyancy. The \
resulting equations are the continuity equation and the momentum equation for \
steady flow (",
  ButtonBox["Lamb [1945]):",
    ButtonData:>{"Literat.nb", "Lamb-1945"},
    ButtonStyle->"Hyperlink"]
}], "Text",
  Evaluatable->False,
  CellTags->{
  "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
    "continuity equation", "momentum equation", "steady flow"}],

Cell[BoxData[
    \(TraditionalForm\`\(u\_x + v\_y = 0, \)\)], "NumberedEquation"],

Cell[BoxData[
    FormBox[
      RowBox[{
        RowBox[{
          RowBox[{"u", " ", 
            FormBox[\(u\_x\),
              "TraditionalForm"]}], " ", "+", " ", \(v\ u\_y\), "-", " ", 
          \(\(W(x)\)\ W\_x\), " ", "-", " ", \(\[Eta]\ u\_\(x, x\)\)}], "=", 
        "0."}], TraditionalForm]], "NumberedEquation"],

Cell[TextData[{
  "Here, ",
  Cell[BoxData[
      \(TraditionalForm\`W = W(x)\)]],
  " is a known function describing the mainstream flow velocity in the ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " direction and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Eta]\)]],
  " is the kinematic viscosity. The boundary conditions to be satisfied are"
}], "Text",
  CellTags->"boundary condition"],

Cell[BoxData[
    \(TraditionalForm\`u(x, y = 0)\  = \ \(v(x, y = 0)\  = \ 0\)\)], 
  "NumberedEquation"],

Cell["and", "Text",
  Evaluatable->False],

Cell[BoxData[
    \(TraditionalForm\`u(x, \[Infinity])\  = \ \(W(x) . \)\)], 
  "NumberedEquation"],

Cell["If we denote by", "Text",
  Evaluatable->False],

Cell[BoxData[
    \(U = u[x, y]; V = v[x, y]; \)], "Input",
  PageWidth->Infinity],

Cell[TextData[{
  "the velocity fields in ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " directions, we can write the left-hand side of the two equations in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " by"
}], "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(bound = {\[PartialD]\_x U + \[PartialD]\_y V, 
        U\ \[PartialD]\_x U + V\ \[PartialD]\_y U - 
          W[x]\ \[PartialD]\_x W[x] - \[Eta]\ \[PartialD]\_{y, 2}U}; 
    bound // LTF\)], "Input",
  CellTags->"Prandtl's boundary layer equations"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(u\_x + v\_y == 0\)},
          {\(u\ u\_x + v\ u\_y - W\ W\_x - \[Eta]\ u\_\(y, y\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The equations ",
  StyleBox["bound",
    FontSlant->"Italic"],
  " constitute a system of equations in two dependent variables ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`v\)]],
  ". This is a system of partial differential equations of parabolic type. \
The equations are today known as Prandtel's boundary layer equations. If we \
now define a stream function by"
}], "Text",
  Evaluatable->False,
  CellTags->{
  "partial differential equations of parabolic type", 
    "Prandtel's boundary-layer equations", "stream function"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(stream = {
        u \[Rule] Function[{x, y}, \[PartialD]\_y \[CapitalPsi][x, y]], 
        v \[Rule] Function[{x, y}, \(-\[PartialD]\_x \[CapitalPsi][x, y]\)]}
        \)], "Input"],

Cell[BoxData[
    \({u \[Rule] Function[{x, y}, \[PartialD]\_y \[CapitalPsi][x, y]], 
      v \[Rule] Function[{x, y}, \(-\[PartialD]\_x \[CapitalPsi][x, y]\)]}
      \)], "Output"]
}, Open  ]],

Cell["\<\
we can reduce the two equations of motion to a single equation \
given by\
\>", "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(bound1 = bound /. stream; bound1 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {"True"},
          {
            \(\(-W\)\ W\_x + \[CapitalPsi]\_y\ \[CapitalPsi]\_\(x, y\) - 
                \[CapitalPsi]\_x\ \[CapitalPsi]\_\(y, y\) - 
                \[Eta]\ \[CapitalPsi]\_\(y, y, y\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "representing a non-linear partial differential equation for the stream \
function \[CapitalPsi] of third order. In the following, we will analyze this \
equation to derive explicit solutions for different mainstream velocities ",
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  ". Utilizing the functions of ",
  StyleBox["MathLie",
    FontSlant->"Italic"],
  ", we are able to determine the symmetries by"
}], "Text",
  Evaluatable->False,
  CellTags->"mainstream velocities"],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbound = 
      Infinitesimals[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]}, {
          \[CapitalPsi]}, {x, y}, {\[Eta]}]; symbound // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(\[Phi]\_1 == k1\)},
          {\(\[Xi]\_1 == 0\)},
          {\(\[Xi]\_2 == \[ScriptCapitalF]\_1\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], k1], 
        Equal[ 
          Subscript[ \[Xi], 1], 0], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Subscript[ \[ScriptCapitalF], 1]]}]]], "Output"]
}, Open  ]],

Cell["\<\
The result shows that the equation for the stream function with \
arbitrary mainstream velocity allows only a translation with respect to the \
stream function. In the following, we will examine the influence of the \
mainstream function on the symmetries.\
\>", "Text"],

Cell[CellGroupData[{

Cell["5.6.6.1 The Blasius Solution", "Subsubsection",
  CellTags->"The Blasius Solution"],

Cell[TextData[{
  "If we assume that the mainstream velocity is a constant ",
  Cell[BoxData[
      \(TraditionalForm\`W(x) = w\)]],
  " as in the problem discussed by ",
  ButtonBox["Blasius [1908],",
    ButtonData:>{"Literat.nb", "Blas-08"},
    ButtonStyle->"Hyperlink"],
  " the symmetry analysis results in the representation of infinitesimals:"
}], "Text",
  CellTags->"infinitesimals of Blasius' model"],

Cell[BoxData[
    \(\(mainstream = W \[Rule] Function[x, w]; \)\)], "Input",
  PageWidth->Infinity],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbound = 
      Infinitesimals[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
          mainstream, {\[CapitalPsi]}, {x, y}, {\[Eta], w}]; 
    symbound // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(\[Phi]\_1 == k1 + k2\ \[CapitalPsi]\)},
          {\(\[Xi]\_1 == k3 + k4\ x\)},
          {\(\[Xi]\_2 == \((\(-k2\) + k4)\)\ y + \[ScriptCapitalF]\_1\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], 
          Plus[ k1, 
            Times[ k2, \[CapitalPsi]]]], 
        Equal[ 
          Subscript[ \[Xi], 1], 
          Plus[ k3, 
            Times[ k4, x]]], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Plus[ 
            Times[ 
              Plus[ 
                Times[ -1, k2], k4], y], 
            Subscript[ \[ScriptCapitalF], 1]]]}]]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The similarity reduction for this model follows by using the function ",
  StyleBox["LieReduction[]", "MmaText"],
  ". Let us first examine the reduction of the four-dimensional group with \
respect to translations. This subgroup is selected by the group constants ",
  StyleBox["k3", "MmaText",
    FontSlant->"Italic"],
  ",",
  StyleBox[" ",
    FontSlant->"Italic"],
  StyleBox["k1", "MmaText",
    FontSlant->"Italic"],
  ", and the arbitrary function ",
  StyleBox["free[1]", "MmaText",
    FontSlant->"Italic"],
  " which we set equal to a constant. The related infinitesimals are thus \
given by"
}], "Text",
  Evaluatable->False,
  CellTags->"LieReduction[]"],

Cell[CellGroupData[{

Cell[BoxData[
    \(infi1 = 
      \({{\(xi[1]\)[x, y, \[CapitalPsi]], \(xi[2]\)[x, y, \[CapitalPsi]]}, {
              \(phi[1]\)[x, y, \[CapitalPsi]]}} /. symbound\) /. {
          k1 \[Rule] 1, k2 \[Rule] 0, k3 \[Rule] 1, k4 \[Rule] 0, 
          \(free[1]\)[x] \[Rule] c}\)], "Input"],

Cell[BoxData[
    \({{1, c}, {1}}\)], "Output"]
}, Open  ]],

Cell["The reduction of the equation follows for this subgroup to", "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(red1 = 
      LieReduction[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
          mainstream, {\[CapitalPsi]}, {x, y}, 
        infi1\[LeftDoubleBracket]1\[RightDoubleBracket], 
        infi1\[LeftDoubleBracket]2\[RightDoubleBracket]]; 
    LTF[Flatten[red1]] /. zeta1 \[Rule] \[Zeta]\_1\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(\(-c\)\ x + y - \[Zeta]\_1 == 0\)},
          {\(\(-x\) + \[CapitalPsi] - F\_1 == 0\)},
          {
            \(\(-F1\_\(\[Zeta]\_1, \[Zeta]\_1\)\) - 
                \[Eta]\ F1\_\(\[Zeta]\_1, \[Zeta]\_1, \[Zeta]\_1\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell["\<\
The related similarity solution is obtained if we integrate the \
third-order ordinary differential equation\
\>", "Text",
  Evaluatable->False,
  CellTags->"similarity solution"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sred1 = 
      DSolve[red1\[LeftDoubleBracket]3, 1\[RightDoubleBracket] == 0, F1, 
        zeta1]\)], "Input",
  PageWidth->Infinity],

Cell[BoxData[
    \({{F1 \[Rule] \((C[1] + E\^\(-\(#1\/\[Eta]\)\)\ C[3] + C[2]\ #1&)\)}}
      \)], "Output"]
}, Open  ]],

Cell["\<\
and insert the result into the representation of the solution\
\>", 
  "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(ssred1 = 
      Solve[Flatten[red1\[LeftDoubleBracket]2\[RightDoubleBracket] /. sred1], 
        \[CapitalPsi]]\)], "Input",
  PageWidth->Infinity],

Cell[BoxData[
    \({{\[CapitalPsi] \[Rule] 
          x + C[1] + \((\(-c\)\ x + y)\)\ C[2] + 
            E\^\(-\(\(\(-c\)\ x + y\)\/\[Eta]\)\)\ C[3]}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The stream function ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPsi]\)]],
  " is determined by three integration constants ",
  Cell[BoxData[
      \(TraditionalForm\`C[i], \ i = 1, 2, 3\)]],
  ". The solution was derived under the condition that the problem allows the \
invariance of translation in the independent and dependent variables. The \
corresponding components of the velocity fields follow by"
}], "Text",
  Evaluatable->False],

Cell[BoxData[
    \(\(rule\  = \ 
      \[CapitalPsi] \[Rule] Function[{x, y}, w]\  /. \ 
        \((ssred1\  /. \[CapitalPsi] \[Rule] w)\); \)\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(velocities\  = \ \({u[x, y], v[x, y]}\  /. \ stream\)\  /. \ rule\)], 
  "Input"],

Cell[BoxData[
    \({C[2] - \(E\^\(-\(\(\(-c\)\ x + y\)\/\[Eta]\)\)\ C[3]\)\/\[Eta], 
      \(-1\) + c\ C[2] - 
        \(c\ E\^\(-\(\(\(-c\)\ x + y\)\/\[Eta]\)\)\ C[3]\)\/\[Eta]}\)], 
  "Output"]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`c\)]],
  " is the group parameter and ",
  StyleBox["C",
    FontSlant->"Italic"],
  "[2] and ",
  StyleBox["C",
    FontSlant->"Italic"],
  "[3] are constants of integration."
}], "Text"],

Cell[TextData[{
  "Another subgroup of the Blasius model for the steady two-dimensional flow \
for a flat plate is the scaling invariance. This kind of invariance is \
selected from the total group if we set the group parameters ",
  StyleBox["k2",
    FontSlant->"Italic"],
  " and ",
  StyleBox["k4",
    FontSlant->"Italic"],
  " equal to a constant value. We get the infinitesimals from"
}], "Text",
  Evaluatable->False,
  CellTags->"Blasius model"],

Cell[CellGroupData[{

Cell[BoxData[
    \(infi1 = 
      \({{\(xi[1]\)[x, y, \[CapitalPsi]], \(xi[2]\)[x, y, \[CapitalPsi]]}, {
              \(phi[1]\)[x, y, \[CapitalPsi]]}} /. symbound\) /. {
          k1 \[Rule] 0, k2 \[Rule] c, k3 \[Rule] 0, k4 \[Rule] 1, 
          \(free[1]\)[x] \[Rule] 0}\)], "Input"],

Cell[BoxData[
    \({{x, \((1 - c)\)\ y}, {c\ \[CapitalPsi]}}\)], "Output"]
}, Open  ]],

Cell["The second reduction of the equation follows by ", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(red2 = 
      Simplify[PowerExpand[
          LieReduction[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
              mainstream, {\[CapitalPsi]}, {x, y}, 
            infi1\[LeftDoubleBracket]1\[RightDoubleBracket], 
            infi1\[LeftDoubleBracket]2\[RightDoubleBracket]]]]; 
    LTF[Flatten[red2]]\  /. \ zeta1 \[Rule] \[Zeta]\_1\)], "Input"],

Cell[BoxData[
    \(Solve::"tdep" \( : \ \) 
      "The equations appear to involve transcendental functions of the \
variables in an essentially non-algebraic way."\)], "Message"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(x\^\(\(-1\) + c\)\ y - \[Zeta]\_1 == 0\)},
          {\(x\^\(-c\)\ \[CapitalPsi] - F\_1 == 0\)},
          {
            \(y\^\(-\(\(4\ c\)\/\(\(-1\) + c\)\)\)\ 
                \[Zeta]\_1\%\(\(4\ c\)\/\(\(-1\) + c\)\)\ 
                \((\((\(-1\) + 2\ c)\)\ \((F\_1)\)\_\[Zeta]\_1\%2 - 
                    c\ F\_1\ \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1\) - 
                    \[Eta]\ \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1, 
                        \[Zeta]\_1\))\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "If we try to solve this type of equation by ",
  StyleBox["DSolve[]", "MmaText"],
  ", we get the result"
}], "Text",
  Evaluatable->False,
  CellTags->"DSolve[]"],

Cell[CellGroupData[{

Cell[BoxData[
    \(DSolve[red2\[LeftDoubleBracket]3\[RightDoubleBracket], F1, zeta1]\)], 
  "Input",
  PageWidth->Infinity],

Cell[BoxData[
    RowBox[{"DSolve", "[", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
            \(y\^\(-\(\(4\ c\)\/\(\(-1\) + c\)\)\)\), " ", 
              \(zeta1\^\(\(4\ c\)\/\(\(-1\) + c\)\)\), " ", 
              RowBox[{"(", 
                RowBox[{
                  RowBox[{\((\(-1\) + 2\ c)\), " ", 
                    SuperscriptBox[
                      RowBox[{
                        SuperscriptBox["F1", "\[Prime]",
                          MultilineFunction->None], "[", "zeta1", "]"}], 
                      "2"]}], "-", 
                  RowBox[{"c", " ", \(F1[zeta1]\), " ", 
                    RowBox[{
                      SuperscriptBox["F1", "\[DoublePrime]",
                        MultilineFunction->None], "[", "zeta1", "]"}]}], 
                  "-", 
                  RowBox[{"\[Eta]", " ", 
                    RowBox[{
                      SuperscriptBox["F1", 
                        TagBox[\((3)\),
                          Derivative],
                        MultilineFunction->None], "[", "zeta1", "]"}]}]}], 
                ")"}]}], "==", "0"}], "}"}], ",", "F1", ",", "zeta1"}], 
      "]"}]], "Output"]
}, Open  ]],

Cell[TextData[
"However, we can use Lie\[CloseCurlyQuote]s methods to examine the symmetries \
of this ordinary differential equation. The symmetries of this equation are \
given by"], "Text",
  Evaluatable->False,
  CellTags->"Lie's method"],

Cell[BoxData[
    \(\(red2eq\  = \ 
      Thread[red2\[LeftDoubleBracket]3, 1\[RightDoubleBracket]/
            y\^\(-\(\(4\ c\)\/\(\(-1\) + c\)\)\)\ 
          zeta1\^\(-\(\(4\ c\)\/\(\(-1\) + c\)\)\), Equal]; \)\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(iblasius = Infinitesimals[red2eq, F1, zeta1, {c, \[Eta]}]; 
    iblasius // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(\[Phi]\_1 == \(-F1\)\ k2\)},
          {\(\[Xi]\_1 == k1 + k2\ zeta1\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], 
          Times[ -1, F1, k2]], 
        Equal[ 
          Subscript[ \[Xi], 1], 
          Plus[ k1, 
            Times[ k2, zeta1]]]}]]], "Output"]
}, Open  ]],

Cell[TextData[{
  "demonstrating that the Blasius equation allows a two-dimensional symmetry \
group containing a translation with respect to the independent variable ",
  StyleBox["zeta1",
    FontSlant->"Italic"],
  " and a scaling in the independent and dependent variables. Since the \
Blasius equation is a third-order ODE but the symmetry group is of dimension \
two, we know from C",
  ButtonBox["hapter 4",
    ButtonData:>{"symmet1.nb", 
      "Symmetries of Ordinary Differential Equations"},
    ButtonStyle->"Hyperlink"],
  StyleBox[" ",
    FontColor->RGBColor[1, 0, 0]],
  "that at least a reduction of the order is possible. At this point, the \
solution procedure ends since the number of symmetries is smaller than the \
degree of the ODE."
}], "Text",
  Evaluatable->False,
  CellTags->{"Blasius equation", "reduction of the order"}]
}, Open  ]],

Cell[CellGroupData[{

Cell["5.6.6.2 Falkner-Skan Solution", "Subsubsection",
  CellTags->"Falkner Skan Solution"],

Cell[TextData[{
  "For the same geometrical situation, ",
  ButtonBox["Falkner and Skan [1931]",
    ButtonData:>{"Literat.nb", "Falk-31"},
    ButtonStyle->"Hyperlink"],
  " proposed that the mainstream velocity ",
  Cell[BoxData[
      \(TraditionalForm\`W(x)\)]],
  " is a power law function of the horizontal coordinate. We define this \
relation as"
}], "Text",
  Evaluatable->False,
  CellTags->{"power law", "Falkner", "Skan"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(mainstream = W \[Rule] Function[x, k\ x\^m]\)], "Input"],

Cell[BoxData[
    \(W \[Rule] Function[x, k\ x\^m]\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`k\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`m\)]],
  " are real constants. The symmetries of the stream function \[CapitalPsi] \
are determined for this case by"
}], "Text",
  Evaluatable->False,
  CellTags->"stream function"],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbound = 
      Infinitesimals[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
          mainstream, {\[CapitalPsi]}, {x, y}, {\[Eta], k, m}]; 
    symbound // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(\[Phi]\_1 == k1 + k2\ \[CapitalPsi]\)},
          {
            RowBox[{\(\[Xi]\_1\), "==", 
              StyleBox[\(\(2\ k2\ x\)\/\(1 + m\)\),
                ScriptLevel->0]}]},
          {
            RowBox[{\(\[Xi]\_2\), "==", 
              RowBox[{
                RowBox[{"-", 
                  StyleBox[\(\(k2\ \((\(-1\) + m)\)\ y\)\/\(1 + m\)\),
                    ScriptLevel->0]}], "+", \(\[ScriptCapitalF]\_1\)}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], 
          Plus[ k1, 
            Times[ k2, \[CapitalPsi]]]], 
        Equal[ 
          Subscript[ \[Xi], 1], 
          Times[ 2, k2, 
            Power[ 
              Plus[ 1, m], -1], x]], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Plus[ 
            Times[ -1, k2, 
              Plus[ -1, m], 
              Power[ 
                Plus[ 1, m], -1], y], 
            Subscript[ \[ScriptCapitalF], 1]]]}]]], "Output"]
}, Open  ]],

Cell[TextData[{
  "For the Falkner-Skan model, we find a two-dimensional discrete symmetry \
group and, in addition, an infinite dimensional group represented by ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          StyleBox[\(free[1]\),
            FontSlant->"Italic"], "=", \(\[ScriptCapitalF]\_1\)}], 
        TraditionalForm]]],
  ". Compared with the case when ",
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  " is arbitrary, the group is enlarged by an additional degree of freedom. \
With respect to the Blasius group, the dimension is reduced by two \
components. The main symmetries consist of a translation and a scaling. Let \
us first discuss the translation symmetry. The related infinitesimals follow \
by setting ",
  StyleBox["k1",
    FontSlant->"Italic"],
  " and ",
  StyleBox["free[1]",
    FontSlant->"Italic"],
  " to constants:"
}], "Text",
  Evaluatable->False,
  CellTags->"Falkner-Skan model"],

Cell[CellGroupData[{

Cell[BoxData[
    \(infi1 = 
      \({{\(xi[1]\)[x, y, \[CapitalPsi]], \(xi[2]\)[x, y, \[CapitalPsi]]}, {
              \(phi[1]\)[x, y, \[CapitalPsi]]}} /. symbound\) /. {
          k1 \[Rule] 1, k2 \[Rule] 0, \(free[1]\)[x] \[Rule] c}\)], "Input"],

Cell[BoxData[
    \({{0, c}, {1}}\)], "Output"]
}, Open  ]],

Cell["The reduction of the stream function equation follows by ", "Text",
  CellTags->"stream function equation"],

Cell[CellGroupData[{

Cell[BoxData[
    \(red1 = 
      Simplify[PowerExpand[
          LieReduction[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
              mainstream, {\[CapitalPsi]}, {x, y}, 
            infi1\[LeftDoubleBracket]1\[RightDoubleBracket], 
            infi1\[LeftDoubleBracket]2\[RightDoubleBracket]]]]; 
    LTF[Flatten[red1]] /. zeta1 \[Rule] \[Zeta]\_1\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(x - \[Zeta]\_1 == 0\)},
          {
            RowBox[{
              RowBox[{
                RowBox[{"-", 
                  StyleBox[\(y\/c\),
                    ScriptLevel->0]}], "+", "\[CapitalPsi]", "-", \(F\_1\)}], 
              "==", "0"}]},
          {\(\(-k\^2\)\ m\ \[Zeta]\_1\%\(2\ m\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The result is somehow surprising since it does not contain any reduction \
of the original equation. If we examine the equation for the stream function, \
we observe that all terms contain derivatives with respect to the coordinate \
",
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " which single out the similarity solution. Only the additive term \
containing the mainstream velocity ",
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  " remains in the reduction. The result shows us that under the symmetry of \
translations, only a solution depending linearly on the vertical coordinate ",
  
  Cell[BoxData[
      \(TraditionalForm\`y\)]],
  " exists."
}], "Text",
  Evaluatable->False,
  CellTags->"similarity solution"],

Cell["\<\
The other type of symmetry contained in the Falkner-Skan case \
represents a scaling. The reduction for the scaling symmetry is given by\
\>",
   "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(infi1 = 
      \({{\(xi[1]\)[x, y, \[CapitalPsi]], \(xi[2]\)[x, y, \[CapitalPsi]]}, {
              \(phi[1]\)[x, y, \[CapitalPsi]]}} /. symbound\) /. {
          k1 \[Rule] 0, k2 \[Rule] c, \(free[1]\)[x] \[Rule] 0}\)], "Input"],

Cell[BoxData[
    \({{\(2\ c\ x\)\/\(1 + m\), 
        \(-\(\(c\ \((\(-1\) + m)\)\ y\)\/\(1 + m\)\)\)}, {c\ \[CapitalPsi]}}
      \)], "Output"]
}, Open  ]],

Cell[TextData[{
  "where we set the group parameters ",
  StyleBox["k2",
    FontSlant->"Italic"],
  " equal to a constant ",
  Cell[BoxData[
      \(TraditionalForm\`c\)]],
  ". The reduction of the equation for the stream function follows by "
}], "Text",
  CellTags->"reduction"],

Cell[CellGroupData[{

Cell[BoxData[
    \(red2 = 
      Simplify[PowerExpand[
          LieReduction[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
              mainstream, {\[CapitalPsi]}, {x, y}, 
            infi1\[LeftDoubleBracket]1\[RightDoubleBracket], 
            infi1\[LeftDoubleBracket]2\[RightDoubleBracket]]]]; 
    LTF[Flatten[red2]] /. zeta1 \[Rule] \[Zeta]\_1\)], "Input"],

Cell[BoxData[
    \(Solve::"tdep" \( : \ \) 
      "The equations appear to involve transcendental functions of the \
variables in an essentially non-algebraic way."\)], "Message"],

Cell[BoxData[
    TagBox[
      TagBox[GridBox[{
            {\(x\^\(1\/2\ \((\(-1\) + m)\)\)\ y - \[Zeta]\_1 == 0\)},
            {\(x\^\(1\/2\ \((\(-1\) - m)\)\)\ \[CapitalPsi] - F\_1 == 0\)},
            {
              \(\(-y\^\(-\(\(4\ m\)\/\(\(-1\) + m\)\)\)\)\ 
                  \[Zeta]\_1\%\(\(4\ m\)\/\(\(-1\) + m\)\)\ 
                  \((\(-2\)\ m\ \((F\_1)\)\_\[Zeta]\_1\%2 + 
                      \((1 + m)\)\ F\_1\ 
                        \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1\) + 
                      2\ \((k\^2\ m + 
                            \[Eta]\ \n\t
                              \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1, 
                                \[Zeta]\_1\))\))\) == 0\)}
            },
          RowSpacings->1,
          ColumnSpacings->3,
          RowAlignments->Baseline,
          ColumnAlignments->{Left}],
        (TableForm[ #]&)],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "Again, ",
  StyleBox["DSolve[]", "MmaText"],
  " is unable to find a solution. The type of the resulting equation is the \
same as the equation in the Blasius model. The same arguments apply here. The \
number of symmetries of the equation is not sufficient for an integration. \
This is one reason why the Falkner-Skan equation is an unsolved problem. We \
will not examine the numerical solution of this equation which actually was \
carried out by ",
  ButtonBox["Falkner and Skan [1931]",
    ButtonData:>{"Literat.nb", "Falk-31"},
    ButtonStyle->"Hyperlink"],
  " in their paper. However our interest is concerned with other \
possibilities to model the mainstream velocity. There is another case which \
enlarges the number of symmetries of the steady two-dimensional flow."
}], "Text",
  CellTags->{"DSolve[]", "Falkner-Skan equation", "mainstream velocity"}]
}, Open  ]],

Cell[CellGroupData[{

Cell["5.6.6.3 Exponential Mainstream Velocity", "Subsubsection",
  CellTags->{
  "Exponential mainstream velocity", "Exponential Mainstream Velocity"}],

Cell["\<\
Another way to choose the mainstream velocity is to assume an \
exponential increase in the horizontal direction. The function of this type \
is given by\
\>", "Text",
  Evaluatable->False,
  CellTags->"exponential mainstream velocity"],

Cell[CellGroupData[{

Cell[BoxData[
    \(mainstream = W \[Rule] Function[x, k\ Exp[a\ x]]\)], "Input"],

Cell[BoxData[
    \(W \[Rule] Function[x, k\ Exp[a\ x]]\)], "Output"]
}, Open  ]],

Cell["\<\
The infinitesimals for this sort of the mainstream velocity \
are\
\>", "Text",
  Evaluatable->False,
  CellTags->"infinitesimals"],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbound = 
      Infinitesimals[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
          mainstream, {\[CapitalPsi]}, {x, y}, {\[Nu], k, a}]; 
    symbound // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(\[Phi]\_1 == k1 - k2\ \[CapitalPsi]\)},
          {
            RowBox[{\(\[Xi]\_1\), "==", 
              RowBox[{"-", 
                StyleBox[\(\(2\ k2\)\/a\),
                  ScriptLevel->0]}]}]},
          {\(\[Xi]\_2 == k2\ y + \[ScriptCapitalF]\_1\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], 
          Plus[ k1, 
            Times[ -1, k2, \[CapitalPsi]]]], 
        Equal[ 
          Subscript[ \[Xi], 1], 
          Times[ -2, 
            Power[ a, -1], k2]], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Plus[ 
            Times[ k2, y], 
            Subscript[ \[ScriptCapitalF], 1]]]}]]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The result contains a two-dimensional finite symmetry group representing a \
translation and some sort of scaling. The undetermined function ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          RowBox[{
            RowBox[{
              StyleBox["free",
                FontSlant->"Italic"], "[", "1", "]"}], "[", "x", "]"}], "=", 
          \(\[ScriptCapitalF]\_1\)}], TraditionalForm]]],
  " extends the finite group to an infinite one. Again, the symmetry group is \
extended if we compare it with the general case in which the mainstream \
velocity is an arbitrary function. The reductions for the scaling symmetry \
follow by"
}], "Text",
  Evaluatable->False,
  CellTags->{"symmetry group", "finite group"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(infi1 = 
      \({{\(xi[1]\)[x, y, \[CapitalPsi]], \(xi[2]\)[x, y, \[CapitalPsi]]}, {
              \(phi[1]\)[x, y, \[CapitalPsi]]}} /. symbound\) /. {
          k1 \[Rule] 0, k2 \[Rule] c, \(free[1]\)[x] \[Rule] 0}\)], "Input"],

Cell[BoxData[
    \({{\(-\(\(2\ c\)\/a\)\), c\ y}, {\(-c\)\ \[CapitalPsi]}}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(red2 = 
      Simplify[PowerExpand[
          LieReduction[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]} /. 
              mainstream, {\[CapitalPsi]}, {x, y}, 
            infi1\[LeftDoubleBracket]1\[RightDoubleBracket], 
            infi1\[LeftDoubleBracket]2\[RightDoubleBracket]]]]; 
    LTF[Flatten[red2]] /. zeta1 \[Rule] \[Zeta]\_1\)], "Input"],

Cell[BoxData[
    \(Solve::"tdep" \( : \ \) 
      "The equations appear to involve transcendental functions of the \
variables in an essentially non-algebraic way."\)], "Message"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(E\^\(\(a\ x\)\/2\)\ y - \[Zeta]\_1 == 0\)},
          {\(E\^\(-\(\(a\ x\)\/2\)\)\ \[CapitalPsi] - F\_1 == 0\)},
          {
            \(\(-\[Zeta]\_1\%4\)\ 
                \((2\ a\ k\^2 - 2\ a\ \((F\_1)\)\_\[Zeta]\_1\%2 + 
                    a\ F\_1\ \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1\) + 
                    2\ \[Eta]\ 
                      \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1, \[Zeta]\_1\))
                  \) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell["\<\
The reduced equation is again of the Falkner-Skan type. Thus, both \
types of the mainstream velocity result into the same type of  equation.\
\>",
   "Text",
  Evaluatable->False,
  CellTags->{"reduced equation", "Faklner-Skan equation"}]
}, Open  ]],

Cell[CellGroupData[{

Cell["5.6.6.4 Group Classification", "Subsubsection",
  CellTags->{"Group classification", "Group Classification"}],

Cell[TextData[{
  "The general topic behind the calculations carried out above is the problem \
of group classification of a partial differential equation. The question is \
formulated as follows. Assume we have a system of equations containing a \
certain arbitrariness, expressed in the dependence of the equations on \
certain parameters or functions. These equations admit a certain group \
\[GothicCapitalG]. If we now change the arbitrariness to a specific form, we \
may observe that the group \[GothicCapitalG] is enlarged. This behavior of \
enlargement of a group was the result of our previous discussion. The \
question now is: Can we find the specific forms for the mainstream velocity ",
  
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  " discussed in the previous sections by using the functions of ",
  StyleBox["MathLie",
    FontSlant->"Italic"],
  "? The problem of group classification is closely connected with the common \
factors occurring in the determining equations. These common factors are \
eliminated by the functions of ",
  StyleBox["MathLie",
    FontSlant->"Italic"],
  ". The information removed from the determining equations is not lost but \
collected in a global variable called ",
  StyleBox["EliminatedFactors", "MmaText",
    FontSlant->"Italic"],
  ". This list collects all factors removed by the functions ",
  StyleBox["Lie[]", "MmaText"],
  ", ",
  StyleBox["LieSolve[]", "MmaText"],
  ", ",
  StyleBox["Infinitesimals[]", "MmaText"],
  ", ",
  StyleBox["DeterminingEquations[],", "MmaText"],
  " and ",
  StyleBox["PDESolve[]", "MmaText"],
  ". If we need to solve the classification problem, we have to examine the \
list ",
  StyleBox["EliminatedFactors", "MmaText",
    FontSlant->"Italic"],
  ".  "
}], "Text",
  Evaluatable->False,
  CellTags->{
  "group classification", "enlargement of a group", "EliminatedFactors", 
    "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
    "PDESolve[]", "common factors"}],

Cell[TextData[{
  "The following considerations will illustrate the special cases for the \
mainstream velocity discussed above. All models discussed so far follow from \
a group classification and can be calculated from the eliminated common \
factors. We start the determination of the general classification problem by \
calling the function ",
  StyleBox["Infinitesimals[]:", "MmaText"]
}], "Text",
  Evaluatable->False,
  CellTags->{"mainstream velocity", "Infinitesimals[]"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(symbound = 
      Infinitesimals[{bound1\[LeftDoubleBracket]2\[RightDoubleBracket]}, {
          \[CapitalPsi]}, {x, y}, {\[Nu]}]; symbound // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(\[Phi]\_1 == k1\)},
          {\(\[Xi]\_1 == 0\)},
          {\(\[Xi]\_2 == \[ScriptCapitalF]\_1\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], k1], 
        Equal[ 
          Subscript[ \[Xi], 1], 0], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Subscript[ \[ScriptCapitalF], 1]]}]]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The factors which were cancelled in the derivation of the determining \
equations can be inspected just by reading the variable ",
  StyleBox["EliminatedFactors:", "MmaText",
    FontSlant->"Italic"]
}], "Text",
  Evaluatable->False,
  CellTags->"determining equations"],

Cell[CellGroupData[{

Cell[BoxData[
    \(elFactor = EliminatedFactors; elFactor // \ LieTraditionalForm\)], 
  "Input"],

Cell[BoxData[
    \({\(-1\), 1, 4\/3, \(-3\)\ W\ W\_x, 
      W\_x\/\(3\ W\) + W\_\(x, x\)\/\(3\ W\_x\), 
      W\_x\%3\/\(W\ \((W\_x\%2 + W\ W\_\(x, x\))\)\) - 
        \(W\_x\ W\_\(x, x\)\)\/\(W\_x\%2 + W\ W\_\(x, x\)\) + 
        \(W\ W\_\(x, x\)\%2\)\/\(W\_x\ \((W\_x\%2 + W\ W\_\(x, x\))\)\) - 
        \(W\ W\_\(x, x, x\)\)\/\(W\_x\%2 + W\ W\_\(x, x\)\), 
      \(-\(\(3\ W\_x\ W\_\(x, x\)\)\/\(4\ 
                \((W\_x\%2 + W\ W\_\(x, x\))\)\)\)\) - 
        \(W\ W\_\(x, x, x\)\)\/\(4\ \((W\_x\%2 + W\ W\_\(x, x\))\)\), 
      \(-\(W\_x\%3\/\(W\ \((W\_x\%2 + W\ W\_\(x, x\))\)\)\)\) + 
        \(W\_x\ W\_\(x, x\)\)\/\(4\ \((W\_x\%2 + W\ W\_\(x, x\))\)\) - 
        \(W\ W\_\(x, x\)\%2\)\/\(W\_x\ \((W\_x\%2 + W\ W\_\(x, x\))\)\) + 
        \(3\ W\ W\_\(x, x, x\)\)\/\(4\ \((W\_x\%2 + W\ W\_\(x, x\))\)\), 
      \(9\ W\_x\%5\ W\_\(x, x\)\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) - \(27\ W\ W\_x\%3\ 
            W\_\(x, x\)\%2\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) + \(18\ W\^2\ W\_x\ 
            W\_\(x, x\)\%3\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) + \(15\ W\ W\_x\%4\ 
            W\_\(x, x, x\)\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) - \(18\ W\^2\ W\_x\%2\ W\_\(x, x\)\ 
            W\_\(x, x, x\)\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) + \(3\ W\^3\ W\_\(x, x\)\%2\ 
            W\_\(x, x, x\)\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) - \(6\ W\^3\ W\_x\ 
            W\_\(x, x, x\)\%2\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) + \(3\ W\^2\ W\_x\%3\ 
            W\_\(x, x, x, x\)\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))
              \)\) + \(3\ W\^3\ W\_x\ W\_\(x, x\)\ 
            W\_\(x, x, x, x\)\)\/\(\((W\_x\%2 + W\ W\_\(x, x\))\)\ 
            \((\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\))\)\)}
      \)], "Output"]
}, Open  ]],

Cell[TextData[{
  "In addition to three numerical factors, the list contains six relations \
which determine the mainstream velocity ",
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  " by a differential equation. In the following calculations, we will show \
that all models for ",
  Cell[BoxData[
      \(TraditionalForm\`W\)]],
  " discussed so far are contained in these equations. Let us start with the \
first equation which is extracted from position four of the list ",
  StyleBox["elFactor", "MmaText",
    FontSlant->"Italic"],
  ". Applying ",
  StyleBox["DSolve[]", "MmaText"],
  " to this equation, we get"
}], "Text",
  Evaluatable->False,
  CellTags->"DSolve[]"],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq1 = elFactor\[LeftDoubleBracket]4\[RightDoubleBracket] == 0; 
    eq1 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(\(-3\)\ W\ W\_x == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(DSolve[eq1, W, x]\)], "Input"],

Cell[BoxData[
    \({{W \[Rule] \((C[1]&)\)}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The result shows that any constant is sufficient to satisfy this equation. \
The constant case for the mainstream velocity is the model discussed by ",
  ButtonBox["Blasius [1908]",
    ButtonData:>{"Literat.nb", "Blas-08"},
    ButtonStyle->"Hyperlink"],
  ". The fifth equation from the list of common factors gives us"
}], "Text",
  Evaluatable->False,
  CellTags->"Blasius model"],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq2 = elFactor\[LeftDoubleBracket]5\[RightDoubleBracket] == 0; 
    eq2 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            RowBox[{
              RowBox[{
                StyleBox[\(W\_x\/\(3\ W\)\),
                  ScriptLevel->0], "+", 
                StyleBox[\(W\_\(x, x\)\/\(3\ W\_x\)\),
                  ScriptLevel->0]}], "==", "0"}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(DSolve[eq2, W, x]\)], "Input"],

Cell[BoxData[
    \({{W \[Rule] \((\(-\@\(E\^\(3\ C[2]\)\ \((2\ #1 - 2\ C[1])\)\)\)&)\)}, {
        W \[Rule] \((\@\(E\^\(3\ C[2]\)\ \((2\ #1 - 2\ C[1])\)\)&)\)}}\)], 
  "Output"]
}, Open  ]],

Cell[TextData[{
  "This solution is a special case of the ",
  ButtonBox["Falkner-Skan",
    ButtonData:>{"Literat.nb", "Falk-31"},
    ButtonStyle->"Hyperlink"],
  " model with ",
  Cell[BoxData[
      \(TraditionalForm\`m = 1/2\)]],
  ". Thus, the eliminated prefactors contain at least the special case of ",
  Cell[BoxData[
      FormBox[
        RowBox[{\(W(x)\), "=", 
          RowBox[{"k", " ", 
            FormBox[\(x\^\(1/2\)\),
              "TraditionalForm"]}]}], TraditionalForm]]],
  ". The sixth equation of our list contains a very complicated non-linear \
third-order ordinary differential equation which we use in the form"
}], "Text",
  Evaluatable->False,
  CellTags->"Falkner-Skan model"],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq3 = 
      Numerator[
        Together[elFactor\[LeftDoubleBracket]6\[RightDoubleBracket]]]; 
    eq3 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            \(W\_x\%4 - W\ W\_x\%2\ W\_\(x, x\) + W\^2\ W\_\(x, x\)\%2 - 
                W\^2\ W\_x\ W\_\(x, x, x\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "If we try to solve this equation by using ",
  StyleBox["DSolve[],", "MmaText"],
  " we end up with"
}], "Text",
  Evaluatable->False,
  CellTags->"DSolve[]"],

Cell[CellGroupData[{

Cell[BoxData[
    \(DSolve[eq3 == 0, W, x]\)], "Input"],

Cell[BoxData[
    RowBox[{"DSolve", "[", 
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox[
              RowBox[{
                SuperscriptBox["W", "\[Prime]",
                  MultilineFunction->None], "[", "x", "]"}], "4"], "-", 
            RowBox[{\(W[x]\), " ", 
              SuperscriptBox[
                RowBox[{
                  SuperscriptBox["W", "\[Prime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"], " ", 
              RowBox[{
                SuperscriptBox["W", "\[DoublePrime]",
                  MultilineFunction->None], "[", "x", "]"}]}], "+", 
            RowBox[{\(W[x]\^2\), " ", 
              SuperscriptBox[
                RowBox[{
                  SuperscriptBox["W", "\[DoublePrime]",
                    MultilineFunction->None], "[", "x", "]"}], "2"]}], "-", 
            RowBox[{\(W[x]\^2\), " ", 
              RowBox[{
                SuperscriptBox["W", "\[Prime]",
                  MultilineFunction->None], "[", "x", "]"}], " ", 
              RowBox[{
                SuperscriptBox["W", 
                  TagBox[\((3)\),
                    Derivative],
                  MultilineFunction->None], "[", "x", "]"}]}]}], "==", "0"}], 
        ",", "W", ",", "x"}], "]"}]], "Output"]
}, Open  ]],

Cell["However, the equation is solved by an exponential function", "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq3 /. W \[Rule] Function[x, k\ Exp[a\ x]]\)], "Input"],

Cell[BoxData[
    \(0\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "This result shows that the exponential model discussed earlier is also \
consistent with the determining equations for the mainstream velocity. The \
seventh equation of our list ",
  StyleBox["elFactor", "MmaText",
    FontSlant->"Italic"],
  " is connected with the equation ",
  StyleBox["eq2", "MmaText",
    FontSlant->"Italic"],
  ". We can show this by just integrating the equation with respect to ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  ":"
}], "Text",
  Evaluatable->False,
  CellTags->"determining equations"],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq4 = 
      Numerator[
        Together[elFactor\[LeftDoubleBracket]7\[RightDoubleBracket]]]; 
    eq4 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(\(-3\)\ W\_x\ W\_\(x, x\) - W\ W\_\(x, x, x\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\[Integral]eq4 \[DifferentialD]x\)], "Input"],

Cell[BoxData[
    RowBox[{
      RowBox[{"-", 
        SuperscriptBox[
          RowBox[{
            SuperscriptBox["W", "\[Prime]",
              MultilineFunction->None], "[", "x", "]"}], "2"]}], "-", 
      RowBox[{\(W[x]\), " ", 
        RowBox[{
          SuperscriptBox["W", "\[DoublePrime]",
            MultilineFunction->None], "[", "x", "]"}]}]}]], "Output"]
}, Open  ]],

Cell[TextData[{
  "Thus, no more information is gained by considering this equation. The \
eighth equation is a third-order ODE which cannot be treated by ",
  StyleBox["DSolve[]", "MmaText"],
  ". The solution of this third-order ODE is again a special case of the \
Falkner-Skan type with ",
  Cell[BoxData[
      \(TraditionalForm\`m = \(-1\)/2\)]],
  ":"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq5 = 
      Numerator[
        Together[elFactor\[LeftDoubleBracket]8\[RightDoubleBracket]]]; 
    eq5 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            \(\(-4\)\ W\_x\%4 + W\ W\_x\%2\ W\_\(x, x\) - 
                4\ W\^2\ W\_\(x, x\)\%2 + 3\ W\^2\ W\_x\ W\_\(x, x, x\) == 0
              \)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Simplify[eq5 /. W \[Rule] Function[x, k\ x\^m]]\)], "Input"],

Cell[BoxData[
    \(\(-2\)\ k\^4\ m\^2\ \((\(-1\) + m + 2\ m\^2)\)\ 
      x\^\(4\ \((\(-1\) + m)\)\)\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The last relation is a fourth-order ODE not solvable by ",
  StyleBox["DSolve[]", "MmaText"],
  ". A particular solution of this equation is, however, given by the \
Falkner-Skan relation for the mainstream velocity. We can check this by"
}], "Text",
  Evaluatable->False],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq6 = 
      Numerator[
        Together[elFactor\[LeftDoubleBracket]9\[RightDoubleBracket]]]; 
    eq6 // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            \(\(-3\)\ 
                \((3\ W\_x\%5\ W\_\(x, x\) - 9\ W\ W\_x\%3\ W\_\(x, x\)\%2 + 
                    6\ W\^2\ W\_x\ W\_\(x, x\)\%3 + 
                    5\ W\ W\_x\%4\ W\_\(x, x, x\) - 
                    6\ W\^2\ W\_x\%2\ W\_\(x, x\)\ W\_\(x, x, x\) + 
                    W\^3\ W\_\(x, x\)\%2\ W\_\(x, x, x\) - 
                    2\ W\^3\ W\_x\ W\_\(x, x, x\)\%2 + 
                    W\^2\ W\_x\%3\ W\_\(x, x, x, x\) + 
                    W\^3\ W\_x\ W\_\(x, x\)\ W\_\(x, x, x, x\))\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        ColumnWidths->0.95,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Simplify[eq6 /. W \[Rule] Function[x, k\ x\^m]]\)], "Input"],

Cell[BoxData[
    \(0\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Thus, we demonstrated that the group classification problem for the steady \
two-dimensional flow is solved by three types of mainstream velocities: (i) ",
  
  Cell[BoxData[
      \(TraditionalForm\`W(x)\  = \(const . \)\)]],
  ", (ii)",
  Cell[BoxData[
      FormBox[
        RowBox[{" ", 
          RowBox[{\(W(x)\), " ", "=", " ", 
            RowBox[{"k", " ", 
              FormBox[\(x\^m\),
                "TraditionalForm"]}]}]}], TraditionalForm]]],
  ", and (iii) ",
  Cell[BoxData[
      FormBox[
        RowBox[{\(W(x)\), " ", "=", " ", 
          RowBox[{"k", " ", 
            FormBox[
              SuperscriptBox["e", 
                StyleBox["ax",
                  FontSlant->"Italic"]],
              "TraditionalForm"]}]}], TraditionalForm]]],
  ". Special cases also contained in the classification of type (ii) are the \
cases with ",
  Cell[BoxData[
      \(TraditionalForm\`m = 1/2\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`m = \(-1\)/2\)]],
  ". The maximal group order of four occurs for the type (i), all other types \
possess a lower group order."
}], "Text",
  Evaluatable->False,
  CellTags->"group classification problem"],

Cell[TextData[ButtonBox["Next notebook",
  ButtonData:>{"Jet.nb", None},
  ButtonStyle->"Hyperlink"]], "Text",
  CellFrame->{{0, 0}, {0, 0.5}}]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 1024}, {0, 695}},
ScreenStyleEnvironment->"Working",
WindowToolbars->"EditBar",
Evaluator->"Local",
WindowSize->{685, 668},
WindowMargins->{{20, Automatic}, {Automatic, 0}},
PrintingCopies->1,
PrintingStartingPageNumber->319,
PrintingPageRange->{1, Automatic},
PageHeaders->{{Cell[ 
        TextData[ {
          StyleBox[ 
            CounterBox[ "Page"], "PageNumber"], "     ", 
          StyleBox[ 
          "Point Symmetries of Partial Differential Equations", FontSlant -> 
            "Italic"]}], "Header"], Inherited, None}, {
    None, Inherited, Cell[ 
        TextData[ {
          StyleBox[ 
            CounterBox[ 
            "Section", CounterFunction :> (Part[ {"Working Examples"}, #]&)], 
            FontSlant -> "Italic"], "     ", 
          StyleBox[ 
            CounterBox[ "Page"], "PageNumber"]}], "Header"]}},
PageHeaderLines->{False, False},
PrintingOptions->{"FirstPageHeader"->True,
"FirstPageFooter"->False,
"FacingPages"->True},
PrivateNotebookOptions->{"ColorPalette"->{RGBColor, -1}},
ShowCellLabel->True,
RenderingOptions->{"ObjectDithering"->True,
"RasterDithering"->False},
Magnification->1,
StyleDefinitions -> "BookStyles.nb"
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "Two-Dimensional Boundary-Layer Flows"->{
    Cell[1739, 51, 426, 8, 41, "Subsection",
      CounterAssignments->{{"Title", 5}, {"SubTitle", 0}, {
          "SubSubTitle", 0}, {"Section", 0}, {"SubSection", 0}, {
          "SubSubSection", 0}, {"NumberedEquation", 60}, {
          "NumberedFigure", 0}, {"NumberedTable", 0}},
      CellTags->"Two-Dimensional Boundary-Layer Flows"]},
  "boundary-layer flow"->{
    Cell[2168, 61, 1334, 32, 146, "Text",
      Evaluatable->False,
      CellTags->{
      "boundary-layer flow", "group classification", 
        "symmetry calculation"}]},
  "group classification"->{
    Cell[2168, 61, 1334, 32, 146, "Text",
      Evaluatable->False,
      CellTags->{
      "boundary-layer flow", "group classification", "symmetry calculation"}],
    
    Cell[40105, 1379, 1987, 45, 203, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "symmetry calculation"->{
    Cell[2168, 61, 1334, 32, 146, "Text",
      Evaluatable->False,
      CellTags->{
      "boundary-layer flow", "group classification", 
        "symmetry calculation"}]},
  "Navier-Stokes equations"->{
    Cell[9148, 386, 556, 12, 61, "Text",
      Evaluatable->False,
      CellTags->{
      "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
        "continuity equation", "momentum equation", "steady flow"}]},
  "boundary-layer equation"->{
    Cell[9148, 386, 556, 12, 61, "Text",
      Evaluatable->False,
      CellTags->{
      "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
        "continuity equation", "momentum equation", "steady flow"}]},
  "buoyancy"->{
    Cell[9148, 386, 556, 12, 61, "Text",
      Evaluatable->False,
      CellTags->{
      "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
        "continuity equation", "momentum equation", "steady flow"}]},
  "continuity equation"->{
    Cell[9148, 386, 556, 12, 61, "Text",
      Evaluatable->False,
      CellTags->{
      "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
        "continuity equation", "momentum equation", "steady flow"}]},
  "momentum equation"->{
    Cell[9148, 386, 556, 12, 61, "Text",
      Evaluatable->False,
      CellTags->{
      "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
        "continuity equation", "momentum equation", "steady flow"}]},
  "steady flow"->{
    Cell[9148, 386, 556, 12, 61, "Text",
      Evaluatable->False,
      CellTags->{
      "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
        "continuity equation", "momentum equation", "steady flow"}]},
  "boundary condition"->{
    Cell[10120, 413, 401, 12, 44, "Text",
      CellTags->"boundary condition"]},
  "Prandtl's boundary layer equations"->{
    Cell[11264, 461, 269, 5, 23, "Input",
      CellTags->"Prandtl's boundary layer equations"]},
  "partial differential equations of parabolic type"->{
    Cell[11858, 480, 609, 17, 61, "Text",
      Evaluatable->False,
      CellTags->{
      "partial differential equations of parabolic type", 
        "Prandtel's boundary-layer equations", "stream function"}]},
  "Prandtel's boundary-layer equations"->{
    Cell[11858, 480, 609, 17, 61, "Text",
      Evaluatable->False,
      CellTags->{
      "partial differential equations of parabolic type", 
        "Prandtel's boundary-layer equations", "stream function"}]},
  "stream function"->{
    Cell[11858, 480, 609, 17, 61, "Text",
      Evaluatable->False,
      CellTags->{
      "partial differential equations of parabolic type", 
        "Prandtel's boundary-layer equations", "stream function"}],
    Cell[27393, 988, 310, 11, 27, "Text",
      Evaluatable->False,
      CellTags->"stream function"]},
  "mainstream velocities"->{
    Cell[13555, 539, 499, 12, 61, "Text",
      Evaluatable->False,
      CellTags->"mainstream velocities"]},
  "The Blasius Solution"->{
    Cell[15107, 589, 89, 1, 31, "Subsubsection",
      CellTags->"The Blasius Solution"]},
  "infinitesimals of Blasius' model"->{
    Cell[15199, 592, 411, 10, 44, "Text",
      CellTags->"infinitesimals of Blasius' model"]},
  "LieReduction[]"->{
    Cell[16743, 644, 687, 19, 67, "Text",
      Evaluatable->False,
      CellTags->"LieReduction[]"]},
  "similarity solution"->{
    Cell[18720, 706, 189, 5, 27, "Text",
      Evaluatable->False,
      CellTags->"similarity solution"],
    Cell[31410, 1119, 751, 19, 95, "Text",
      Evaluatable->False,
      CellTags->"similarity solution"]},
  "Blasius model"->{
    Cell[20927, 790, 454, 12, 61, "Text",
      Evaluatable->False,
      CellTags->"Blasius model"],
    Cell[47971, 1577, 403, 9, 44, "Text",
      Evaluatable->False,
      CellTags->"Blasius model"]},
  "DSolve[]"->{
    Cell[23148, 854, 182, 6, 29, "Text",
      Evaluatable->False,
      CellTags->"DSolve[]"],
    Cell[34580, 1212, 887, 16, 114, "Text",
      CellTags->{"DSolve[]", "Falkner-Skan equation", "mainstream velocity"}],
    
    Cell[46750, 1531, 679, 18, 82, "Text",
      Evaluatable->False,
      CellTags->"DSolve[]"],
    Cell[50482, 1664, 177, 6, 29, "Text",
      Evaluatable->False,
      CellTags->"DSolve[]"]},
  "Lie's method"->{
    Cell[24688, 900, 242, 5, 44, "Text",
      Evaluatable->False,
      CellTags->"Lie's method"]},
  "Blasius equation"->{
    Cell[25794, 938, 851, 19, 95, "Text",
      Evaluatable->False,
      CellTags->{"Blasius equation", "reduction of the order"}]},
  "reduction of the order"->{
    Cell[25794, 938, 851, 19, 95, "Text",
      Evaluatable->False,
      CellTags->{"Blasius equation", "reduction of the order"}]},
  "Falkner Skan Solution"->{
    Cell[26682, 962, 91, 1, 31, "Subsubsection",
      CellTags->"Falkner Skan Solution"]},
  "power law"->{
    Cell[26776, 965, 434, 12, 44, "Text",
      Evaluatable->False,
      CellTags->{"power law", "Falkner", "Skan"}]},
  "Falkner"->{
    Cell[26776, 965, 434, 12, 44, "Text",
      Evaluatable->False,
      CellTags->{"power law", "Falkner", "Skan"}]},
  "Skan"->{
    Cell[26776, 965, 434, 12, 44, "Text",
      Evaluatable->False,
      CellTags->{"power law", "Falkner", "Skan"}]},
  "Falkner-Skan model"->{
    Cell[29083, 1047, 931, 25, 95, "Text",
      Evaluatable->False,
      CellTags->"Falkner-Skan model"],
    Cell[49246, 1622, 711, 19, 61, "Text",
      Evaluatable->False,
      CellTags->"Falkner-Skan model"]},
  "stream function equation"->{
    Cell[30353, 1086, 113, 1, 27, "Text",
      CellTags->"stream function equation"]},
  "reduction"->{
    Cell[32785, 1161, 282, 9, 44, "Text",
      CellTags->"reduction"]},
  "Falkner-Skan equation"->{
    Cell[34580, 1212, 887, 16, 114, "Text",
      CellTags->{
      "DSolve[]", "Falkner-Skan equation", "mainstream velocity"}]},
  "mainstream velocity"->{
    Cell[34580, 1212, 887, 16, 114, "Text",
      CellTags->{"DSolve[]", "Falkner-Skan equation", "mainstream velocity"}],
    
    Cell[42095, 1426, 481, 9, 63, "Text",
      Evaluatable->False,
      CellTags->{"mainstream velocity", "Infinitesimals[]"}]},
  "Exponential mainstream velocity"->{
    Cell[35504, 1233, 151, 2, 31, "Subsubsection",
      CellTags->{
      "Exponential mainstream velocity", 
        "Exponential Mainstream Velocity"}]},
  "Exponential Mainstream Velocity"->{
    Cell[35504, 1233, 151, 2, 31, "Subsubsection",
      CellTags->{
      "Exponential mainstream velocity", 
        "Exponential Mainstream Velocity"}]},
  "exponential mainstream velocity"->{
    Cell[35658, 1237, 246, 6, 44, "Text",
      Evaluatable->False,
      CellTags->"exponential mainstream velocity"]},
  "infinitesimals"->{
    Cell[36097, 1254, 141, 5, 27, "Text",
      Evaluatable->False,
      CellTags->"infinitesimals"]},
  "symmetry group"->{
    Cell[37333, 1299, 741, 17, 78, "Text",
      Evaluatable->False,
      CellTags->{"symmetry group", "finite group"}]},
  "finite group"->{
    Cell[37333, 1299, 741, 17, 78, "Text",
      Evaluatable->False,
      CellTags->{"symmetry group", "finite group"}]},
  "reduced equation"->{
    Cell[39701, 1365, 249, 6, 44, "Text",
      Evaluatable->False,
      CellTags->{"reduced equation", "Faklner-Skan equation"}]},
  "Faklner-Skan equation"->{
    Cell[39701, 1365, 249, 6, 44, "Text",
      Evaluatable->False,
      CellTags->{"reduced equation", "Faklner-Skan equation"}]},
  "Group classification"->{
    Cell[39987, 1376, 115, 1, 31, "Subsubsection",
      CellTags->{"Group classification", "Group Classification"}]},
  "Group Classification"->{
    Cell[39987, 1376, 115, 1, 31, "Subsubsection",
      CellTags->{"Group classification", "Group Classification"}]},
  "enlargement of a group"->{
    Cell[40105, 1379, 1987, 45, 203, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "EliminatedFactors"->{
    Cell[40105, 1379, 1987, 45, 203, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "Lie[]"->{
    Cell[40105, 1379, 1987, 45, 203, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "LieSolve[]"->{
    Cell[40105, 1379, 1987, 45, 203, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "Infinitesimals[]"->{
    Cell[40105, 1379, 1987, 45, 203, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}],
    Cell[42095, 1426, 481, 9, 63, "Text",
      Evaluatable->False,
      CellTags->{"mainstream velocity", "Infinitesimals[]"}]},
  "DeterminingEquations[]"->{
    Cell[40105, 1379, 1987, 45, 203, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "PDESolve[]"->{
    Cell[40105, 1379, 1987, 45, 203, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "common factors"->{
    Cell[40105, 1379, 1987, 45, 203, "Text",
      Evaluatable->False,
      CellTags->{
      "group classification", "enlargement of a group", "EliminatedFactors", 
        "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
        "PDESolve[]", "common factors"}]},
  "determining equations"->{
    Cell[43323, 1464, 289, 7, 46, "Text",
      Evaluatable->False,
      CellTags->"determining equations"],
    Cell[52290, 1723, 552, 15, 63, "Text",
      Evaluatable->False,
      CellTags->"determining equations"]},
  "group classification problem"->{
    Cell[56326, 1866, 1190, 35, 78, "Text",
      Evaluatable->False,
      CellTags->"group classification problem"]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"Two-Dimensional Boundary-Layer Flows", 59398, 1957},
  {"boundary-layer flow", 59768, 1964},
  {"group classification", 59971, 1970},
  {"symmetry calculation", 60457, 1982},
  {"Navier-Stokes equations", 60663, 1988},
  {"boundary-layer equation", 60922, 1994},
  {"buoyancy", 61166, 2000},
  {"continuity equation", 61421, 2006},
  {"momentum equation", 61674, 2012},
  {"steady flow", 61921, 2018},
  {"boundary condition", 62175, 2024},
  {"Prandtl's boundary layer equations", 62299, 2027},
  {"partial differential equations of parabolic type", 62453, 2030},
  {"Prandtel's boundary-layer equations", 62710, 2036},
  {"stream function", 62947, 2042},
  {"mainstream velocities", 63294, 2051},
  {"The Blasius Solution", 63433, 2055},
  {"infinitesimals of Blasius' model", 63564, 2058},
  {"LieReduction[]", 63682, 2061},
  {"similarity solution", 63813, 2065},
  {"Blasius model", 64051, 2072},
  {"DSolve[]", 64272, 2079},
  {"Lie's method", 64712, 2092},
  {"Blasius equation", 64837, 2096},
  {"reduction of the order", 65001, 2100},
  {"Falkner Skan Solution", 65164, 2104},
  {"power law", 65273, 2107},
  {"Falkner", 65408, 2111},
  {"Skan", 65540, 2115},
  {"Falkner-Skan model", 65686, 2119},
  {"stream function equation", 65935, 2126},
  {"reduction", 66040, 2129},
  {"Falkner-Skan equation", 66142, 2132},
  {"mainstream velocity", 66300, 2136},
  {"Exponential mainstream velocity", 66598, 2143},
  {"Exponential Mainstream Velocity", 66794, 2148},
  {"exponential mainstream velocity", 66990, 2153},
  {"infinitesimals", 67133, 2157},
  {"symmetry group", 67259, 2161},
  {"finite group", 67402, 2165},
  {"reduced equation", 67549, 2169},
  {"Faklner-Skan equation", 67711, 2173},
  {"Group classification", 67872, 2177},
  {"Group Classification", 68019, 2180},
  {"enlargement of a group", 68168, 2183},
  {"EliminatedFactors", 68481, 2190},
  {"Lie[]", 68782, 2197},
  {"LieSolve[]", 69088, 2204},
  {"Infinitesimals[]", 69400, 2211},
  {"DeterminingEquations[]", 69848, 2221},
  {"PDESolve[]", 70154, 2228},
  {"common factors", 70464, 2235},
  {"determining equations", 70781, 2242},
  {"group classification problem", 71039, 2249}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 426, 8, 41, "Subsection",
  CounterAssignments->{{"Title", 5}, {"SubTitle", 0}, {"SubSubTitle", 0}, {
      "Section", 0}, {"SubSection", 0}, {"SubSubSection", 0}, {
      "NumberedEquation", 60}, {"NumberedFigure", 0}, {"NumberedTable", 0}},
  CellTags->"Two-Dimensional Boundary-Layer Flows"],
Cell[2168, 61, 1334, 32, 146, "Text",
  Evaluatable->False,
  CellTags->{
  "boundary-layer flow", "group classification", "symmetry calculation"}],
Cell[3505, 95, 5640, 289, 108, 2603, 239, "GraphicsData", "PostScript", \
"Graphics",
  Evaluatable->False],
Cell[9148, 386, 556, 12, 61, "Text",
  Evaluatable->False,
  CellTags->{
  "Navier-Stokes equations", "boundary-layer equation", "buoyancy", 
    "continuity equation", "momentum equation", "steady flow"}],
Cell[9707, 400, 82, 1, 24, "NumberedEquation"],
Cell[9792, 403, 325, 8, 24, "NumberedEquation"],
Cell[10120, 413, 401, 12, 44, "Text",
  CellTags->"boundary condition"],
Cell[10524, 427, 105, 2, 22, "NumberedEquation"],
Cell[10632, 431, 41, 1, 27, "Text",
  Evaluatable->False],
Cell[10676, 434, 99, 2, 22, "NumberedEquation"],
Cell[10778, 438, 53, 1, 27, "Text",
  Evaluatable->False],
Cell[10834, 441, 82, 2, 23, "Input"],
Cell[10919, 445, 320, 12, 27, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[11264, 461, 269, 5, 23, "Input",
  CellTags->"Prandtl's boundary layer equations"],
Cell[11536, 468, 307, 9, 34, "Output"]
}, Open  ]],
Cell[11858, 480, 609, 17, 61, "Text",
  Evaluatable->False,
  CellTags->{
  "partial differential equations of parabolic type", 
    "Prandtel's boundary-layer equations", "stream function"}],

Cell[CellGroupData[{
Cell[12492, 501, 202, 4, 23, "Input"],
Cell[12697, 507, 181, 3, 23, "Output"]
}, Open  ]],
Cell[12893, 513, 119, 4, 27, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[13037, 521, 72, 1, 23, "Input"],
Cell[13112, 524, 428, 12, 32, "Output"]
}, Open  ]],
Cell[13555, 539, 499, 12, 61, "Text",
  Evaluatable->False,
  CellTags->"mainstream velocities"],

Cell[CellGroupData[{
Cell[14079, 555, 180, 3, 23, "Input"],
Cell[14262, 560, 525, 17, 44, "Output"]
}, Open  ]],
Cell[14802, 580, 280, 5, 61, "Text"],

Cell[CellGroupData[{
Cell[15107, 589, 89, 1, 31, "Subsubsection",
  CellTags->"The Blasius Solution"],
Cell[15199, 592, 411, 10, 44, "Text",
  CellTags->"infinitesimals of Blasius' model"],
Cell[15613, 604, 99, 2, 23, "Input"],

Cell[CellGroupData[{
Cell[15737, 610, 202, 4, 23, "Input"],
Cell[15942, 616, 786, 25, 44, "Output"]
}, Open  ]],
Cell[16743, 644, 687, 19, 67, "Text",
  Evaluatable->False,
  CellTags->"LieReduction[]"],

Cell[CellGroupData[{
Cell[17455, 667, 288, 5, 39, "Input"],
Cell[17746, 674, 47, 1, 23, "Output"]
}, Open  ]],
Cell[17808, 678, 96, 1, 27, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[17929, 683, 330, 6, 39, "Input"],
Cell[18262, 691, 443, 12, 44, "Output"]
}, Open  ]],
Cell[18720, 706, 189, 5, 27, "Text",
  Evaluatable->False,
  CellTags->"similarity solution"],

Cell[CellGroupData[{
Cell[18934, 715, 153, 4, 23, "Input"],
Cell[19090, 721, 109, 2, 29, "Output"]
}, Open  ]],
Cell[19214, 726, 110, 4, 27, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[19349, 734, 167, 4, 23, "Input"],
Cell[19519, 740, 166, 3, 26, "Output"]
}, Open  ]],
Cell[19700, 746, 468, 11, 61, "Text",
  Evaluatable->False],
Cell[20171, 759, 156, 3, 23, "Input"],

Cell[CellGroupData[{
Cell[20352, 766, 101, 2, 23, "Input"],
Cell[20456, 770, 196, 4, 41, "Output"]
}, Open  ]],
Cell[20667, 777, 257, 11, 27, "Text"],
Cell[20927, 790, 454, 12, 61, "Text",
  Evaluatable->False,
  CellTags->"Blasius model"],

Cell[CellGroupData[{
Cell[21406, 806, 288, 5, 39, "Input"],
Cell[21697, 813, 75, 1, 23, "Output"]
}, Open  ]],
Cell[21787, 817, 64, 0, 27, "Text"],

Cell[CellGroupData[{
Cell[21876, 821, 380, 7, 55, "Input"],
Cell[22259, 830, 180, 3, 37, "Message"],
Cell[22442, 835, 691, 16, 56, "Output"]
}, Open  ]],
Cell[23148, 854, 182, 6, 29, "Text",
  Evaluatable->False,
  CellTags->"DSolve[]"],

Cell[CellGroupData[{
Cell[23355, 864, 124, 3, 23, "Input"],
Cell[23482, 869, 1191, 28, 44, "Output"]
}, Open  ]],
Cell[24688, 900, 242, 5, 44, "Text",
  Evaluatable->False,
  CellTags->"Lie's method"],
Cell[24933, 907, 226, 4, 28, "Input"],

Cell[CellGroupData[{
Cell[25184, 915, 112, 2, 23, "Input"],
Cell[25299, 919, 480, 16, 32, "Output"]
}, Open  ]],
Cell[25794, 938, 851, 19, 95, "Text",
  Evaluatable->False,
  CellTags->{"Blasius equation", "reduction of the order"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[26682, 962, 91, 1, 31, "Subsubsection",
  CellTags->"Falkner Skan Solution"],
Cell[26776, 965, 434, 12, 44, "Text",
  Evaluatable->False,
  CellTags->{"power law", "Falkner", "Skan"}],

Cell[CellGroupData[{
Cell[27235, 981, 76, 1, 23, "Input"],
Cell[27314, 984, 64, 1, 23, "Output"]
}, Open  ]],
Cell[27393, 988, 310, 11, 27, "Text",
  Evaluatable->False,
  CellTags->"stream function"],

Cell[CellGroupData[{
Cell[27728, 1003, 205, 4, 23, "Input"],
Cell[27936, 1009, 1132, 35, 74, "Output"]
}, Open  ]],
Cell[29083, 1047, 931, 25, 95, "Text",
  Evaluatable->False,
  CellTags->"Falkner-Skan model"],

Cell[CellGroupData[{
Cell[30039, 1076, 249, 4, 39, "Input"],
Cell[30291, 1082, 47, 1, 23, "Output"]
}, Open  ]],
Cell[30353, 1086, 113, 1, 27, "Text",
  CellTags->"stream function equation"],

Cell[CellGroupData[{
Cell[30491, 1091, 376, 7, 55, "Input"],
Cell[30870, 1100, 525, 16, 58, "Output"]
}, Open  ]],
Cell[31410, 1119, 751, 19, 95, "Text",
  Evaluatable->False,
  CellTags->"similarity solution"],
Cell[32164, 1140, 185, 5, 44, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[32374, 1149, 249, 4, 39, "Input"],
Cell[32626, 1155, 144, 3, 35, "Output"]
}, Open  ]],
Cell[32785, 1161, 282, 9, 44, "Text",
  CellTags->"reduction"],

Cell[CellGroupData[{
Cell[33092, 1174, 376, 7, 55, "Input"],
Cell[33471, 1183, 180, 3, 37, "Message"],
Cell[33654, 1188, 911, 21, 80, "Output"]
}, Open  ]],
Cell[34580, 1212, 887, 16, 114, "Text",
  CellTags->{"DSolve[]", "Falkner-Skan equation", "mainstream velocity"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[35504, 1233, 151, 2, 31, "Subsubsection",
  CellTags->{
  "Exponential mainstream velocity", "Exponential Mainstream Velocity"}],
Cell[35658, 1237, 246, 6, 44, "Text",
  Evaluatable->False,
  CellTags->"exponential mainstream velocity"],

Cell[CellGroupData[{
Cell[35929, 1247, 81, 1, 23, "Input"],
Cell[36013, 1250, 69, 1, 23, "Output"]
}, Open  ]],
Cell[36097, 1254, 141, 5, 27, "Text",
  Evaluatable->False,
  CellTags->"infinitesimals"],

Cell[CellGroupData[{
Cell[36263, 1263, 204, 4, 23, "Input"],
Cell[36470, 1269, 848, 27, 58, "Output"]
}, Open  ]],
Cell[37333, 1299, 741, 17, 78, "Text",
  Evaluatable->False,
  CellTags->{"symmetry group", "finite group"}],

Cell[CellGroupData[{
Cell[38099, 1320, 249, 4, 39, "Input"],
Cell[38351, 1326, 89, 1, 34, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[38477, 1332, 376, 7, 55, "Input"],
Cell[38856, 1341, 180, 3, 37, "Message"],
Cell[39039, 1346, 647, 16, 56, "Output"]
}, Open  ]],
Cell[39701, 1365, 249, 6, 44, "Text",
  Evaluatable->False,
  CellTags->{"reduced equation", "Faklner-Skan equation"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[39987, 1376, 115, 1, 31, "Subsubsection",
  CellTags->{"Group classification", "Group Classification"}],
Cell[40105, 1379, 1987, 45, 203, "Text",
  Evaluatable->False,
  CellTags->{
  "group classification", "enlargement of a group", "EliminatedFactors", 
    "Lie[]", "LieSolve[]", "Infinitesimals[]", "DeterminingEquations[]", 
    "PDESolve[]", "common factors"}],
Cell[42095, 1426, 481, 9, 63, "Text",
  Evaluatable->False,
  CellTags->{"mainstream velocity", "Infinitesimals[]"}],

Cell[CellGroupData[{
Cell[42601, 1439, 179, 3, 23, "Input"],
Cell[42783, 1444, 525, 17, 44, "Output"]
}, Open  ]],
Cell[43323, 1464, 289, 7, 46, "Text",
  Evaluatable->False,
  CellTags->"determining equations"],

Cell[CellGroupData[{
Cell[43637, 1475, 98, 2, 23, "Input"],
Cell[43738, 1479, 2997, 49, 372, "Output"]
}, Open  ]],
Cell[46750, 1531, 679, 18, 82, "Text",
  Evaluatable->False,
  CellTags->"DSolve[]"],

Cell[CellGroupData[{
Cell[47454, 1553, 111, 2, 23, "Input"],
Cell[47568, 1557, 239, 8, 23, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[47844, 1570, 50, 1, 23, "Input"],
Cell[47897, 1573, 59, 1, 23, "Output"]
}, Open  ]],
Cell[47971, 1577, 403, 9, 44, "Text",
  Evaluatable->False,
  CellTags->"Blasius model"],

Cell[CellGroupData[{
Cell[48399, 1590, 111, 2, 23, "Input"],
Cell[48513, 1594, 449, 14, 36, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[48999, 1613, 50, 1, 23, "Input"],
Cell[49052, 1616, 179, 3, 28, "Output"]
}, Open  ]],
Cell[49246, 1622, 711, 19, 61, "Text",
  Evaluatable->False,
  CellTags->"Falkner-Skan model"],

Cell[CellGroupData[{
Cell[49982, 1645, 143, 4, 23, "Input"],
Cell[50128, 1651, 339, 10, 24, "Output"]
}, Open  ]],
Cell[50482, 1664, 177, 6, 29, "Text",
  Evaluatable->False,
  CellTags->"DSolve[]"],

Cell[CellGroupData[{
Cell[50684, 1674, 55, 1, 23, "Input"],
Cell[50742, 1677, 1284, 31, 23, "Output"]
}, Open  ]],
Cell[52041, 1711, 96, 1, 27, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[52162, 1716, 75, 1, 23, "Input"],
Cell[52240, 1719, 35, 1, 23, "Output"]
}, Open  ]],
Cell[52290, 1723, 552, 15, 63, "Text",
  Evaluatable->False,
  CellTags->"determining equations"],

Cell[CellGroupData[{
Cell[52867, 1742, 143, 4, 23, "Input"],
Cell[53013, 1748, 269, 8, 23, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[53319, 1761, 65, 1, 33, "Input"],
Cell[53387, 1764, 369, 10, 23, "Output"]
}, Open  ]],
Cell[53771, 1777, 370, 9, 63, "Text"],

Cell[CellGroupData[{
Cell[54166, 1790, 143, 4, 23, "Input"],
Cell[54312, 1796, 368, 11, 24, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[54717, 1812, 80, 1, 23, "Input"],
Cell[54800, 1815, 115, 2, 23, "Output"]
}, Open  ]],
Cell[54930, 1820, 291, 6, 46, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[55246, 1830, 143, 4, 23, "Input"],
Cell[55392, 1836, 764, 18, 40, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[56193, 1859, 80, 1, 23, "Input"],
Cell[56276, 1862, 35, 1, 23, "Output"]
}, Open  ]],
Cell[56326, 1866, 1190, 35, 78, "Text",
  Evaluatable->False,
  CellTags->"group classification problem"],
Cell[57519, 1903, 143, 3, 35, "Text"]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

