(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    284649,       7350]*)
(*NotebookOutlinePosition[    308067,       7927]*)
(*  CellTagsIndexPosition[    304146,       7832]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["8", "ChapterLine",
  Evaluatable->False,
  CounterAssignments->{{"Title", 7}, {"SubTitle", 0}, {"SubSubTitle", 0}, {
      "Section", 0}, {"SubSection", 0}, {"SubSubSection", 0}, {
      "NumberedEquation", 0}, {"NumberedFigure", 0}, {"NumberedTable", 0}},
  CounterBoxOptions->{CounterFunction:>CapitalRomanNumeral},
  AspectRatioFixed->True,
  CellTags->"Approximate Symmetries of Partial Differential Equations"],

Cell["\<\
Approximate Symmetries of Partial
Differential Equations\
\>", "Title",
  Evaluatable->False,
  CounterAssignments->{{"Title", 7}, {"SubTitle", 0}, {"SubSubTitle", 0}, {
      "Section", 0}, {"SubSection", 0}, {"SubSubSection", 0}, {
      "NumberedEquation", 0}, {"NumberedFigure", 0}, {"NumberedTable", 0}},
  CounterBoxOptions->{CounterFunction:>CapitalRomanNumeral},
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell[TextData[{
  "8.",
  CounterBox["Section"],
  ".",
  " Introduction"
}], "Section",
  CounterAssignments->{{"Title", 8}, {"SubTitle", 0}, {"SubSubTitle", 0}, {
      "Section", 0}, {"SubSection", 0}, {"SubSubSection", 0}, {
      "NumberedEquation", 0}, {"NumberedFigure", 0}, {"NumberedTable", 0}},
  CellTags->"Introduction"],

Cell[TextData[{
  "The theory of approximate symmetries was developed by Baikov, Gazizov, and \
",
  ButtonBox["Ibragimov [1989]",
    ButtonData:>{"Literat.nb", "Baik-89"},
    ButtonStyle->"Hyperlink"],
  " in the 1980s. The idea behind this development was the extension of Lie\
\[CloseCurlyQuote]s theory to situations in which a small perturbation of the \
original equation is encountered. For such cases, the question arises of how \
the point symmetries or the group of the equations are altered if a small \
perturbation is added to the original equation. This question initiated the \
development of a group analysis method that is stable under small \
perturbations of the differential equation. The present chapter discusses the \
method of approximate symmetries. The method is based on the concept of an \
approximate group of transformations. Approximate symmetries are useful for \
partial differential equations depending on a small parameter \[Epsilon]. \
This parameter is usually used in the standard theories to examine the \
differential equation in some limit. On the other hand, this parameter is \
also useful in the examination of Lie point symmetries."
}], "Text",
  Evaluatable->False,
  TextJustification->1,
  AspectRatioFixed->True,
  CellTags->{
  "approximate symmetries", "extension of Lie's theory", 
    "small perturbation", "approximate group", "Lie point symmetries"}],

Cell[TextData[
"The basics of the theory were recently developed by Baikov et al. [1991]. \
These authors showed that the main part of Lie\[CloseCurlyQuote]s theory can \
be used in an approximate calculus taking into account the smallness of the \
critical parameter in the theory. The new theory maintains the essential \
features of the standard Lie theory. This chapter provides a concise \
introduction to the theory of approximate transformation groups and regular \
approximate symmetries of differential equations with a small parameter."], 
  "Text",
  Evaluatable->False,
  TextJustification->1,
  AspectRatioFixed->True,
  CellTags->{
  "Lie's theory", "approximate calculus", "critical parameter", 
    "approximate transformation groups", "approximate symmetries"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "8.",
  CounterBox["Section"],
  ".",
  " Approximations"
}], "Section"],

Cell[TextData[{
  "Discussing approximate symmetries, we first have to define the term \
approximation. The following terms are used to fix the notation. We assume \
that ",
  StyleBox["x",
    FontSlant->"Italic"],
  " = ",
  Cell[BoxData[
      \(TraditionalForm\`\((x\_1, \ x\_2, \ \[Ellipsis], \ x\_N)\)\)]],
  " are the independent coordinates of functions which are analytic in their \
arguments. Let us also assume that \[Epsilon] is a small parameter on which \
our functions additionally depend. We will denote the involved infinitesimal \
small functions of order ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^\(p + 1\)\)]],
  " by ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Theta]\_p\)(x, \ \[Epsilon])\)]],
  ", where ",
  Cell[BoxData[
      \(TraditionalForm\`p \[LessEqual] 0\)]],
  " is a positive constant. This condition is expressed by ",
  Cell[BoxData[
      \(TraditionalForm
      \`\[Theta]\_p\ \((x, \ \[Epsilon])\)\  = \ o(\[Epsilon]\^p)\)]],
  ". An alternate representation of this condition is"
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True,
  CellTags->{"approximation", "infinitesimal small functions"}],

Cell[BoxData[
    \(TraditionalForm
    \`lim\+\(\[Epsilon] \[Rule] 0\)\(\(\[Theta]\_p\)(x, \[Epsilon])
            \)\/\[Epsilon]\^p = \ 0. \)], "NumberedEquation"],

Cell["\<\
Using this notation, we can state what we mean by an approximation.\
\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["Definition: Approximation", "Definition",
  CellTags->"Definition: Approximation"],

Cell[TextData[{
  "Let ",
  Cell[BoxData[
      \(TraditionalForm\`f\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`g\)]],
  " be analytic functions in ",
  StyleBox["x.",
    FontSlant->"Italic"],
  " We define an approximation of order ",
  Cell[BoxData[
      \(TraditionalForm\`p\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`f\  \[Tilde] \ g\)]],
  ", by the relation"
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[BoxData[
    \(TraditionalForm
    \`f(z, \[Epsilon])\  = \ g(z, \[Epsilon])\  + \ o(\[Epsilon]\^p)\)], 
  "NumberedEquation"],

Cell[TextData[{
  "for some fixed value of ",
  Cell[BoxData[
      \(TraditionalForm\`p \[LessEqual] 0\)]],
  ". \[EmptyCircle]"
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
This definition is the basis of all the calculations we will carry \
out in the following sections.\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "8.",
  CounterBox["Section"],
  ".",
  " One-Parameter Approximation Group"
}], "Section",
  Evaluatable->False,
  AspectRatioFixed->True,
  CellTags->"One-Parameter Approximation Group"],

Cell[TextData[{
  "Following the discussion of ",
  ButtonBox["Baikov et al. [1991]",
    ButtonData:>{"Literat.nb", "Baik-91"},
    ButtonStyle->"Hyperlink"],
  ", we define a one-parameter approximation group for a set of vector \
functions ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          FormBox[\(f\_i\),
            "TraditionalForm"], "(", \(x, \[Epsilon]\), ")"}], 
        TraditionalForm]]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`i = 0, \[Ellipsis], p\)]],
  ", with coordinates ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          FormBox[\(f\_i\%j\),
            "TraditionalForm"], "(", \(x, \[Epsilon]\), ")"}], 
        TraditionalForm]]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`j = 1, \[Ellipsis], N\)]],
  ". The one-parameter family ",
  Cell[BoxData[
      \(TraditionalForm\`\[GothicCapitalG]\)]],
  " of approximate transformations is thus"
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True,
  CellTags->{
  "one-parameter approximation group", "approximate transformation"}],

Cell[BoxData[
    FormBox[
      RowBox[{
      \(x\^\(*j\)\  \[Tilde] \ 
          \[Sum]\+\(i = 0\)\%p\( \[Delta]\^i\) 
              \(\(f\_i\%j\)(x, \[Epsilon])\)\), ",", "     ", 
        FormBox[\(j = 1, \[Ellipsis], N\),
          "TraditionalForm"], ","}], TraditionalForm]], "NumberedEquation",
  CellTags->"eq-3"],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          RowBox[{
            RowBox[{
              FormBox[
                RowBox[{
                  FormBox[
                    RowBox[{
                      FormBox[\(x = \((x\_1\)\),
                        "TraditionalForm"], ",", \(x\_2\)}],
                    "TraditionalForm"], ",", "\[Ellipsis]", ",", \(x\_N\)}],
                "TraditionalForm"], ")"}], " ", "\[Element]", 
            \(\[DoubleStruckCapitalR]\^N\)}], " "}], TraditionalForm]]],
  " are the old coordinates and  ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          RowBox[{
            FormBox[
              RowBox[{
                FormBox[
                  RowBox[{
                    FormBox[\(x\^*\),
                      "TraditionalForm"], "=", \((\(x\_1\%*\)\)}],
                  "TraditionalForm"], ",", \(x\_2\%*\)}],
              "TraditionalForm"], 
            FormBox[\(, \[Ellipsis]\[ThinSpace], \(x\_N\%*\)\),
              "TraditionalForm"]}], ")"}], TraditionalForm]]],
  " are the new coordinates, and \[Epsilon] and \[Delta] are the group \
parameter and the perturbation parameter, respectively. This transformation \
satisfies the following conditions:"
}], "Text",
  ShowCellBracket->True,
  Evaluatable->False,
  AspectRatioFixed->True,
  CellTags->{"group parameter", "perturbation parameter"}],

Cell[BoxData[
    \(TraditionalForm\`\(f(x, \[Epsilon]\  = \ 0, \[Delta])\  \[Tilde] \ x, 
    \)\)], "NumberedEquation"],

Cell["\<\
the approximate identity element. Furthermore, it is assumed that \
the transformation\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True,
  CellTags->"approximate identity element"],

Cell[TextData[Cell[BoxData[
    FormBox[
      RowBox[{
        FormBox[\(x\^*\),
          "TraditionalForm"], " ", "=", " ", 
        \(f(x, \[Epsilon], \[Delta])\)}], TraditionalForm]]]], 
  "NumberedEquation",
  Evaluatable->False,
  TextAlignment->Center,
  AspectRatioFixed->True],

Cell[TextData[{
  "is defined for any value of \[Epsilon] of a small neighborhood of ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon] = 0\)]],
  " and that this neighborhood allows the relation ",
  Cell[BoxData[
      \(TraditionalForm\`f(x, \[Epsilon], \[Delta])\  \[Tilde] \ x\)]],
  "  at ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\  = 0\)]],
  "."
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[TextData[
"The set \[GothicCapitalG] of transformation is called a local one-parameter \
approximate transformation group if"], "Text",
  Evaluatable->False,
  AspectRatioFixed->True,
  CellTags->"local one-parameter approximate transformation group"],

Cell[TextData[Cell[BoxData[
    \(TraditionalForm
    \`f(f(x, \[Epsilon], \[Delta]), \[Gamma], \[Delta])\  \[Tilde] \ 
      F(x, \[Epsilon] + \[Gamma], \[Delta])\)]]], "NumberedEquation"],

Cell[TextData[{
  "for all transformations ",
  Cell[BoxData[
      \(TraditionalForm\`x = f\)]],
  "."
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["Example 1", "Example"],

Cell[TextData[{
  "Let us consider an example with ",
  Cell[BoxData[
      \(TraditionalForm\`N = 1\)]],
  ". The following two functions are equal in a first-order approximation. \
The functions ",
  Cell[BoxData[
      \(TraditionalForm\`f\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`g\)]],
  " depend on the independent variable ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " and on the two parameters ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Delta]\)]],
  ". ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\)]],
  " denotes the group parameter and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Delta]\)]],
  " the small perturbation parameter."
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True,
  CellTags->"first-order approximation"],

Cell[BoxData[
    \(f[x_, \[Epsilon]_, \[Delta]_] := 
      x + \[Epsilon]\ \((1 + \[Delta]\ x + \(\[Delta]\ \[Epsilon]\)\/2)\)\)], 
  "Input"],

Cell[BoxData[
    \(g[x_, \[Epsilon]_, \[Delta]_] := 
      x + \[Epsilon]\ \((1 + \[Delta]\ x)\)\ 
          \((1 + \(\[Epsilon]\ \[Delta]\)\/2)\)\)], "Input"],

Cell["The difference between the two functions is given by", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell[TextData[
"f[x,\[Epsilon],\[Delta]]-g[x,\[Epsilon],\[Delta]]//Expand"], "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    \(\(-\(1\/2\)\)\ x\ \[Delta]\^2\ \[Epsilon]\^2\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "which is a function proportional to the square of the small approximation \
parameter ",
  Cell[BoxData[
      \(TraditionalForm\`\[Delta]\)]],
  ". Thus, in first-order approximation, the two functions are equal. The two \
functions ",
  Cell[BoxData[
      \(TraditionalForm\`f\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`g\)]],
  " satisfy, in addition, the relation"
}], "Text",
  Evaluatable->False,
  TextJustification->1,
  AspectRatioFixed->True],

Cell[CellGroupData[{

Cell[TextData[
"f[g[x,\[Epsilon],\[Delta]],\[Phi],\[Delta]] - \
f[x,\[Epsilon]+\[Phi],\[Delta]] //Expand"], "Input",
  AspectRatioFixed->True],

Cell[BoxData[
    \(1\/2\ x\ \[Delta]\^2\ \[Epsilon]\^2 + 
      x\ \[Delta]\^2\ \[Epsilon]\ \[Phi] + 
      1\/2\ \[Delta]\^2\ \[Epsilon]\^2\ \[Phi] + 
      1\/2\ x\ \[Delta]\^3\ \[Epsilon]\^2\ \[Phi]\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "which is the approximate association relation of the related group. The \
difference is a quadratic function of the group parameters ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\)]],
  " and at least of order two in the small perturbation ",
  Cell[BoxData[
      \(TraditionalForm\`\[Delta]\)]],
  ". This example demonstrates the general behavior of an approximate group \
in first-order approximation. The essential point is that the above relations \
are satisfied up to the order of approximation; i.e., the approximation order \
(here, first-order) does not occur in the relations. Note that only higher \
orders in the perturbation parameter are present."
}], "Text",
  Evaluatable->False,
  TextJustification->1,
  AspectRatioFixed->True,
  CellTags->"approximate association relation"],

Cell[TextData[
"To determine the group properties of an approximate group transformation, we \
use similar tools as in the case of Lie point groups. The applied \
mathematical objects in Lie\[CloseCurlyQuote]s theory are the group \
generator, Lie\[CloseCurlyQuote]s equation, infinitesimal transformations, \
etc. In the following, we will discuss these objects for approximate \
groups."], "Text",
  Evaluatable->False,
  TextJustification->1,
  AspectRatioFixed->True,
  CellTags->{
  "group generator", "Lie's equation", "infinitesimal transformations"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "8.",
  CounterBox["Section"],
  ".",
  " Approximate Group Generator"
}], "Section",
  Evaluatable->False,
  AspectRatioFixed->True,
  CellTags->"Approximate group generator"],

Cell["\<\
In close analogy to Lie point groups, we introduce the main tool of \
symmetry analysis at this point. The group generator or vector field of an \
approximate group is given by a first-order differential operator of the form\
\
\>", "Text",
  Evaluatable->False,
  AspectRatioFixed->True,
  CellTags->{
  "Lie point group", "group generator", "vector field", 
    "vector field of an approximate group"}],

Cell[BoxData[
    \(TraditionalForm
    \`v\&\[RightVector]\  = 
      \[Sum]\+\(i = 1\)\%p\ 
        \(\(\[Xi]\_i\)(x, \[Delta])\)\ \[PartialD]\/\[PartialD]x\_i\)], 
  "NumberedEquation",
  CellTags->"eq-7"],

Cell["such that", "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[BoxData[
    \(TraditionalForm
    \`\(\(\[Xi]\_i\)(x, \[Delta])\  \[Tilde] \ 
      \(\[Xi]\_i\%0\)(x)\  + \ \[Delta]\ \(\(\[Xi]\_i\%1\)(x)\)\  + \ 
        \[CenterEllipsis] + \ \[Delta]\^p\ \(\(\[Xi]\_i\%p\)(x)\), \)\)], 
  "NumberedEquation",
  CellTags->"eq-8"],

Cell[TextData[{
  "where the components of the vector field ",
  Cell[BoxData[
      \(TraditionalForm\`\((\[Xi]\^0, \[Xi]\^1, \[Ellipsis], \[Xi]\^p)\)\)]],
  " are given by the expansion coefficients of the transformation"
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell[BoxData[
    FormBox[
      RowBox[{\(\[Xi]\_i\%\[Nu]\), " ", "=", " ", 
        RowBox[{
        \(\[PartialD]\(\(f\_i\%\[Nu]\)(x, \[Epsilon], \[Delta])
                  \)\/\[PartialD]\[Epsilon]\), \( | \_\(\[Epsilon] = 0\)\), 
          "     ", 
          
          FormBox[\(\[Nu]\  = \ 0, 1, \[Ellipsis], p; \ \ \ \ i = 1, 2, 
            \[Ellipsis], \(N . \)\),
            "TraditionalForm"]}]}], TraditionalForm]], "NumberedEquation",
  CellTags->"eq-9"],

Cell["Thus, an approximate vector field is given by", "Text",
  Evaluatable->False,
  AspectRatioFixed->True,
  CellTags->"approximate vector field"],

Cell[BoxData[
    \(TraditionalForm
    \`\(v\& \[RightVector] \)\  \[Tilde] 
      \((\ \(\[Xi]\_i\%0\)(x)\  + \ \[Delta]\ \(\(\[Xi]\_i\%1\)(x)\)\  + \ 
            \[CenterEllipsis] + \(\[Delta]\^p\) \(\(\[Xi]\_i\%p\)(x)\))\) 
        \[PartialD]\/\[PartialD]x\_i . \)], "NumberedEquation"],

Cell[TextData[{
  "The main difference between an ordinary vector field ",
  Cell[BoxData[
      \(TraditionalForm\`v\&\[RightVector]\)]],
  " of Lie point symmetries and an approximate vector field is an expansion \
of the coefficients of the vector field with respect to the perturbation \
parameter ",
  Cell[BoxData[
      \(TraditionalForm\`\[Delta]\)]],
  ". These coefficients follow from a Taylor expansion of the transformation \
",
  Cell[BoxData[
      \(TraditionalForm\`f\)]],
  " with respect to the group parameter \[Epsilon] and with respect to the \
perturbation parameter ",
  Cell[BoxData[
      \(TraditionalForm\`\[Delta]\)]],
  "."
}], "Text",
  Evaluatable->False,
  AspectRatioFixed->True,
  CellTags->{"Lie point symmetries", "Taylor expansion"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "8.",
  CounterBox["Section"],
  ".",
  " The Determining Equations and an Algorithm of Calculation"
}], "Section",
  Evaluatable->False,
  AspectRatioFixed->True,
  CellTags->"The Determining Equations and an Algorithm of Calculation"],

Cell[TextData[{
  "The purpose of this section is to discuss the connection between the \
classical Lie theory and the theory of approximate group analysis. We sketch \
the main steps of the algorithm for calculating approximate symmetries. A \
detailed presentation and proofs of the statements are contained in the work \
of ",
  ButtonBox["Baikov et al. [1989]",
    ButtonData:>{"Literat.nb", "Baik-89"},
    ButtonStyle->"Hyperlink"],
  "."
}], "Text",
  CellTags->{
  "classical Lie theory", "approximate group analysis", 
    "algorithm of approximate group analysis", "approximate symmetries"}],

Cell[TextData[{
  "Let \[GothicCapitalG] be an approximate group of transformations given by \
equation (8.",
  ButtonBox["3",
    ButtonData:>"eq-3",
    ButtonStyle->"Hyperlink"],
  "). Let us further assume that the order of approximation is ",
  Cell[BoxData[
      \(TraditionalForm\`p \[GreaterEqual] q\)]],
  ". The approximate equation may be given by"
}], "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{\(\[CapitalDelta](x, \[Epsilon])\), " ", "=", " ", 
        FormBox[
          \(\(\[CapitalDelta]\_0\)(x)\  + \ 
              \[Epsilon]\ \(\(\[CapitalDelta]\_1\)(x)\)\  + \ 
              \[CenterEllipsis] + \ 
              \(\[Epsilon]\^q\) \(\(\[CapitalDelta]\_q\)(x)\)\  = \ 
            \(o(\[Epsilon]\^q) . \)\),
          "TraditionalForm"]}], TraditionalForm]], "NumberedEquation",
  CellTags->"eq-11"],

Cell[TextData[
"Relation (8.11) is said to be an approximate invariant with respect to \
\[GothicCapitalG] if"], "Text",
  CellTags->"approximate invariant"],

Cell[BoxData[
    \(TraditionalForm
    \`\[CapitalDelta](f(x, \[Delta], \[Epsilon]), \[Epsilon])\  = \ 
      o(\[Epsilon]\^q)\)], "NumberedEquation"],

Cell[TextData[{
  "whenever ",
  Cell[BoxData[
      \(TraditionalForm\`x = \((x\_1, \[Ellipsis], x\_N)\)\)]],
  " satisfies equation (8.11)."
}], "Text"],

Cell[TextData[{
  "Assume the approximate vector field ",
  Cell[BoxData[
      \(TraditionalForm\`v\&\[RightVector]\)]],
  " is given by equation (8.",
  ButtonBox["7",
    ButtonData:>"eq-7",
    ButtonStyle->"Hyperlink"],
  "). Then, equation (8.11) is approximately invariant under the approximate \
group \[GothicCapitalG] if and only if"
}], "Text",
  CellTags->{"approximate vector field", "approximate group"}],

Cell[BoxData[
    \(TraditionalForm
    \`\(pr\^\((k)\)\) v\&\[RightVector]\ 
      \(\[CapitalDelta](x, \[Epsilon])
          \( | \_\(\[CapitalDelta](x, \[Epsilon]) = o(\[Epsilon]\^q)\)\)\( = 
          \ o(\[Epsilon]\^p) . \)\)\)], "NumberedEquation"],

Cell[TextData[{
  "This relation is called the determining equation for approximate \
symmetries. Comparing this expression with Lie\[CloseCurlyQuote]s theory, we \
realize that the original condition of invariance is altered in such a way \
that the exact vanishing is dropped. In relation (8.13), only an approximate \
vanishing is needed to derive the determining equations for the \
infinitesimals. If the determining equation (8.13) is satisfied, we say that \
equation (8.12) admits the approximate operator ",
  Cell[BoxData[
      \(TraditionalForm\`v\&\[RightVector]\)]],
  "."
}], "Text",
  CellTags->"determining equation for approximate symmetries"],

Cell[TextData[{
  "To demonstrate the relations discussed so far, let us consider the simple \
case with ",
  Cell[BoxData[
      \(TraditionalForm\`q = \(p = 1\)\)]],
  ". Relations (8.11) and (8.7) simplify to"
}], "Text"],

Cell[BoxData[
    \(TraditionalForm
    \`\(\[CapitalDelta]\_0\)(x)\  + \ 
        \[Epsilon]\ \(\(\[CapitalDelta]\_1\)(x)\)\  \[TildeTilde] \ 0\)], 
  "NumberedEquation"],

Cell["and", "Text"],

Cell[BoxData[
    \(TraditionalForm
    \`\(v\&\[RightVector]\  = \ 
      v\&\[RightVector]\^0\  + \ \[Epsilon]\ v\&\[RightVector]\^1\  
        \[Congruent] \ 
        \(\(\[Xi]\_i\%0\)(x)\)\ \[PartialD]\/\[PartialD]x\_i\  + \ 
          \(\(\[Epsilon]\[Xi]\_i\%1\)(x)\)\ \[PartialD]\/\[PartialD]x\_i, 
    \)\)], "NumberedEquation"],

Cell["\<\
respectively. The determining equation (8.13) then reduces to the \
relation\
\>", "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{
        RowBox[{
          RowBox[{\(pr\^\((k)\)\), "(", 
            
            FormBox[\(\ 
              v\&\[RightVector]\^0\  + \ \[Epsilon]\ v\&\[RightVector]\^1\),
              "TraditionalForm"], ")"}], 
          RowBox[{"(", 
            
            FormBox[\(\(\[CapitalDelta]\_0\)(x)\  + \ 
                \[Epsilon]\ \(\(\[CapitalDelta]\_1\)(x)\)\),
              "TraditionalForm"], ")"}]}], 
        \( | \_\(\(\[CapitalDelta]\_0\)(x)\  + \ 
                \[Epsilon]\ \(\(\[CapitalDelta]\_1\)(x)\) = 0\)\), 
        \( = \(o(\[Epsilon]) . \)\)}], TraditionalForm]], "NumberedEquation"],

Cell[TextData[{
  "This relation contains an algorithm for the calculation of first-order \
approximate symmetries. The algorithm is based on a theorem stated in ",
  ButtonBox["Baikov et al. [1989]",
    ButtonData:>{"Literat.nb", "Baik-89"},
    ButtonStyle->"Hyperlink"],
  ". "
}], "Text",
  CellTags->"algorithm to calculate first-order approximate symmetries"],

Cell["Theorem: First-order approximations", "Theorem",
  CellTags->"Theorem: First-order approximations"],

Cell["\<\
In the first-order approximation, the determining equations for \
approximate symmetries follow from the system of relations\
\>", "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{
        RowBox[{\(pr\^\((k)\)\), 
          
          FormBox[\(\(v\&\[RightVector]\^0\) 
                \(\(\[CapitalDelta]\_0\)(x)\)\  = \ 
              \(\[Lambda](x)\)\ \(\(\[CapitalDelta]\_0\)(x)\)\),
            "TraditionalForm"]}], ","}], TraditionalForm]], "NumberedEquation"],

Cell["and ", "Text"],

Cell[BoxData[
    \(TraditionalForm
    \`\(pr\^\((k)\)\) \(v\&\[RightVector]\^1\) 
          \(\(\[CapitalDelta]\_0\)(x)\)\  + \ 
        \(pr\^\((k)\)\) \(v\&\[RightVector]\^0\) 
          \(\(\[CapitalDelta]\_1\)(x)\)\  = \ 
      \(\[Lambda](x)\)\ \(\(\[CapitalDelta]\_1\)(x) . \)\)], 
  "NumberedEquation"],

Cell[TextData[{
  "The auxiliary factor ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda](x)\)]],
  " is determined from (8.17) and afterward substituted into (8.18), where \
(8.18) itself must hold for the solution ",
  Cell[BoxData[
      \(TraditionalForm\`x\)]],
  " of the unperturbed equation ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[CapitalDelta]\_0\)(x) = 0\)]],
  ". \[EmptyCircle]"
}], "Text",
  CellTags->"auxiliary factor"],

Cell["\<\
The above theorem provides an algorithm for the calculation of \
first-order approximate symmetries. The algorithm consists of the following \
four steps:\
\>", "Text",
  CellTags->"algorithm of first-order approximation"],

Cell[TextData[{
  "Find the exact symmetries generated by ",
  Cell[BoxData[
      \(TraditionalForm\`\(v\& \[RightVector] \)\^0\)]],
  " of the unperturbed equation. This step is equivalent to the classical \
theory of Lie. In equations, we have"
}], "ListItem",
  CellDingbat->"1."],

Cell[BoxData[
    FormBox[
      RowBox[{
        FormBox[\(pr\^\((k)\)\),
          "TraditionalForm"], 
        FormBox[
          \(\(v\&\[RightVector]\^0\) \(\(\[CapitalDelta]\_0\)(x)\) 
            \(|\_\(\[CapitalDelta]\_0 = 0\)\( = 0\) . \)\),
          "TraditionalForm"]}], TraditionalForm]], "NumberedEquation"],

Cell[TextData[{
  "If we know the symmetries in the zero approximation ",
  Cell[BoxData[
      \(TraditionalForm\`\(v\& \[RightVector] \)\^0\)]],
  ", we can use them to calculate the auxiliary function ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda](x)\)]],
  " if the perturbation ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Epsilon]\[CapitalDelta]\_1\)(x)\)]],
  " is given. The deficiency ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda] = H\)]],
  " follows by"
}], "ListItem",
  CellDingbat->"2.",
  CellTags->"deficiency"],

Cell[BoxData[
    FormBox[
      RowBox[{"H", " ", "\[TildeTilde]", " ", 
        RowBox[{\(1\/\[Epsilon]\), 
          FormBox[\(pr\^\((k)\)\),
            "TraditionalForm"], 
          \(\(v\&\[RightVector]\^0\)(
            \(\[CapitalDelta]\_0\)(x)\  + \ 
              \[Epsilon]\ \(\(\[CapitalDelta]\_1\)(x)\))\), " ", 
          \(|\_\(\[CapitalDelta]\_0 + \[Epsilon]\[CapitalDelta]\_1 = 0\) . 
            \)}]}], TraditionalForm]], "NumberedEquation"],

Cell["\<\
The symmetries of the first-order approximation then follow from \
the relation\
\>", "ListItem",
  CellDingbat->"3."],

Cell[BoxData[
    FormBox[
      RowBox[{
        FormBox[
          FormBox[\(pr\^\((k)\)\),
            "TraditionalForm"],
          "TraditionalForm"], 
        FormBox[
          \(\(v\&\[RightVector]\^1\)(\(\[CapitalDelta]\_0\)(x))
                \( | \_\(\[CapitalDelta]\_0 = 0\)\)\ \(+\ H\)\  = \ 
            \(0 . \)\),
          "TraditionalForm"]}], TraditionalForm]], "NumberedEquation"],

Cell["Check the consistency of the approximation at the end.", "ListItem",
  CellDingbat->"4."],

Cell[TextData[{
  "We remark that in the approximate group analysis of differential \
equations, the prolongation formulas are the same as in the classical Lie \
algorithm. Thus, it is straightforward to use the functions of ",
  StyleBox["MathLie",
    FontSlant->"Italic"],
  " to derive approximate symmetries. The package ",
  StyleBox["MathLie",
    FontSlant->"Italic"],
  " offers the function ",
  StyleBox["ApproximateSymmetries[]", "MmaText"],
  " to carry out the calculations in a single step. The following examples \
will demonstrate the application of the function."
}], "Text",
  CellTags->{
  "prolongation formula", "classical Lie algorithm", 
    "ApproximateSymmetries[]"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "8.",
  CounterBox["Section"],
  ".",
  " Examples"
}], "Section",
  CellTags->"Examples"],

Cell[TextData[{
  "The following two examples are taken from ",
  ButtonBox["Baikov et al. [1989]",
    ButtonData:>{"Literat.nb", "Baik-89"},
    ButtonStyle->"Hyperlink"],
  ". They serve to illustrate the calculation of approximate symmetries in \
connection with computer algebra. Our results are identical with those \
published by Baikov and co-workers."
}], "Text",
  CellTags->"computer algebra"],

Cell[CellGroupData[{

Cell["8.6.1 Isentropic Liquid", "Subsection",
  CellTags->"Isentropic Liquid"],

Cell[TextData[{
  "Let us consider the problem of a liquid in a pipe (",
  ButtonBox["Baikov et al. [1989]",
    ButtonData:>{"Literat.nb", "Baik-89"},
    ButtonStyle->"Hyperlink"],
  "). The system of equations of motion for the fluid density \[Rho] and the \
velocity field ",
  Cell[BoxData[
      \(TraditionalForm\`v\)]],
  " in a one-dimensional space is given by "
}], "Text",
  CellTags->"liquid in a pipe"],

Cell[BoxData[
    \(TraditionalForm\`\[Rho]\_t\  + \ \((\[Rho]\ v)\)\_x\  = \ 0\)], 
  "NumberedEquation"],

Cell["and ", "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{
        RowBox[{
          RowBox[{
            RowBox[{"\[Rho]", " ", 
              FormBox[\(v\_t\),
                "TraditionalForm"]}], " ", "+", " ", \(\[Rho]\ v\ v\_x\), 
            " ", "+", " ", \(p\_x\), " ", "-", " ", 
            \(\[Epsilon]\ \[Rho]\ v\)}], " ", "=", "0"}], ","}], 
      TraditionalForm]], "NumberedEquation"],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`p\)]],
  " is the pressure of the liquid and \[Epsilon] is the hydraulic-friction \
coefficient. In Lagrange coordinates ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`q = \[Integral]\[Rho] \[DifferentialD]x\)]],
  ", the system becomes"
}], "Text",
  CellTags->"hydraulic-friction coefficient"],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqOfMotion
      \  = {\[PartialD]\_t\((1\/\[Rho][q, t])\)\  - \[PartialD]\_q v[q, t] == 
          0, \[PartialD]\_t v[q, t] + \[PartialD]\_q p[q, t] - 
            \[Epsilon]\ v[q, t] == 0}; eqOfMotion // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            RowBox[{
              RowBox[{\(-v\_q\), "-", 
                StyleBox[\(\[Rho]\_t\/\[Rho]\^2\),
                  ScriptLevel->0]}], "==", "0"}]},
          {\(\(-v\)\ \[Epsilon] + p\_q + v\_t == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell["hence, for", "Text"],

Cell[BoxData[
    \(\(rule1\  = \ \[Rho] \[Rule] Function[{q, t}, 1/u[q, t]]; \)\)], "Input"],

Cell["we obtain the equations", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqation\  = \ eqOfMotion\  /. \ rule1; eqation // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(u\_t - v\_q == 0\)},
          {\(\(-v\)\ \[Epsilon] + p\_q + v\_t == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "Differentiating the first equation with respect to ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  " and the second with respect to ",
  Cell[BoxData[
      \(TraditionalForm\`q\)]],
  ", and replacing the spatial derivative of ",
  Cell[BoxData[
      \(TraditionalForm\`v\)]],
  " by the temporal derivative of ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  ", we end up with the relation"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{
      RowBox[{"equation", " ", "=", " ", 
        RowBox[{
          RowBox[{
            RowBox[{"(", 
              RowBox[{
              \((\[PartialD]\_t eqation\[LeftDoubleBracket]1, 1
                        \[RightDoubleBracket] + 
                    \[PartialD]\_q eqation\[LeftDoubleBracket]2, 1
                        \[RightDoubleBracket])\), " ", "/.", " ", 
                RowBox[{"Solve", "[", 
                  RowBox[{
                  \(eqation\[LeftDoubleBracket]1\[RightDoubleBracket]\), 
                    ",", 
                    RowBox[{
                      SuperscriptBox["v", 
                        TagBox[\((1, 0)\),
                          Derivative],
                        MultilineFunction->None], "[", \(q, t\), "]"}]}], 
                  "]"}]}], " ", ")"}], "\[LeftDoubleBracket]", "1", 
            "\[RightDoubleBracket]"}], "==", "0"}]}], ";", 
      \(equation // LTF\)}]], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(\(-\[Epsilon]\)\ u\_t + p\_\(q, q\) + u\_\(t, t\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell["If we now represent the pressure by the expression", "Text",
  CellTags->"pressure"],

Cell[BoxData[
    \(\(rule2\  = \ 
      p \[Rule] 
        Function[{q, t}, u[q, t]\^\(\[Sigma] + 1\)\/\(\[Sigma] + 1\)]; \)\)], 
  "Input"],

Cell[TextData[
"with \[Sigma] a constant, we can represent the equation as"], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(equation\  = equation\  /. \ rule2 // Simplify; equation // LTF\)], 
  "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            \(u\^\(\(-1\) + \[Sigma]\)\ \[Sigma]\ u\_q\%2 - 
                \[Epsilon]\ u\_t + u\^\[Sigma]\ u\_\(q, q\) + u\_\(t, t\) == 0
              \)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "This equation depends on two parameters \[Sigma] and \[Epsilon] denoting \
the isentropic exponent and the perturbation parameter, respectively. In the \
following, we consider the hydraulic-friction coefficient \[Epsilon] as a \
small quantity. Thus, we can discuss the symmetries of the equation under the \
condition that \[Epsilon] creates a disturbance of the original fluid \
equations. The approximate symmetries of the equation follow by applying the \
function ",
  StyleBox["ApproximateSymmetries[]", "MmaText"],
  " to the derived equation:"
}], "Text",
  CellTags->{
  "isentropic exponent", "hydraulic-friction coefficient", 
    "ApproximateSymmetries[]"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(infinitesimals\  = \ 
      ApproximateSymmetries[equation, {u}, {q, t}, {\[Sigma], \[Epsilon]}, 
        \[Epsilon]]; infinitesimals\  // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{\(\[Phi]\_1\), "==", 
              RowBox[{
                StyleBox[\(\(4\ k4\ u\)\/\(4 + \[Sigma]\)\),
                  ScriptLevel->0], "+", 
                StyleBox[\(\(k4\ u\ \[Sigma]\)\/\(4 + \[Sigma]\)\),
                  ScriptLevel->0], "+", 
                RowBox[{"\[Epsilon]", " ", 
                  RowBox[{"(", 
                    RowBox[{
                      RowBox[{"-", 
                        
                        StyleBox[
                          \(\(k4\ t\ u\ \[Sigma]\)\/\(4 + \[Sigma]\)\),
                          ScriptLevel->0]}], 
                      StyleBox["+",
                        ScriptLevel->0], 
                      
                      StyleBox[
                        \(\(u\ \((2\ k3\ t + k8\ \((4 + \[Sigma])\))
                                \)\)\/\(4 + \[Sigma]\)\),
                        ScriptLevel->0]}], ")"}]}]}]}]},
          {
            RowBox[{\(\[Xi]\_1\), "==", 
              RowBox[{"k2", "+", \(k3\ q\), "+", 
                RowBox[{"\[Epsilon]", " ", 
                  RowBox[{"(", 
                    RowBox[{"k5", "+", \(k7\ q\), "+", 
                      StyleBox[\(\(k8\ q\ \[Sigma]\)\/2\),
                        ScriptLevel->0]}], ")"}]}]}]}]},
          {
            RowBox[{\(\[Xi]\_2\), "==", 
              RowBox[{"k1", "+", 
                RowBox[{"t", " ", 
                  RowBox[{"(", 
                    RowBox[{"k3", "-", 
                      StyleBox[\(\(k4\ \[Sigma]\)\/2\),
                        ScriptLevel->0]}], ")"}]}], "+", 
                RowBox[{"\[Epsilon]", " ", 
                  RowBox[{"(", 
                    RowBox[{"k6", "+", \(k7\ t\), "+", 
                      
                      StyleBox[
                        \(\(t\^2\ \[Sigma]\ 
                              \((\(-2\)\ k3 + k4\ \[Sigma])\)\)\/\(4\ 
                              \((4 + \[Sigma])\)\)\),
                        ScriptLevel->0]}], ")"}]}]}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], 
          Plus[ 
            Times[ 4, k4, u, 
              Power[ 
                Plus[ 4, \[Sigma]], -1]], 
            Times[ k4, u, \[Sigma], 
              Power[ 
                Plus[ 4, \[Sigma]], -1]], 
            Times[ \[Epsilon], 
              Plus[ 
                Times[ -1, k4, t, u, \[Sigma], 
                  Power[ 
                    Plus[ 4, \[Sigma]], -1]], 
                Times[ u, 
                  Power[ 
                    Plus[ 4, \[Sigma]], -1], 
                  Plus[ 
                    Times[ 2, k3, t], 
                    Times[ k8, 
                      Plus[ 4, \[Sigma]]]]]]]]], 
        Equal[ 
          Subscript[ \[Xi], 1], 
          Plus[ k2, 
            Times[ k3, q], 
            Times[ \[Epsilon], 
              Plus[ k5, 
                Times[ k7, q], 
                Times[ 
                  Rational[ 1, 2], k8, q, \[Sigma]]]]]], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Plus[ k1, 
            Times[ t, 
              Plus[ k3, 
                Times[ 
                  Rational[ -1, 2], k4, \[Sigma]]]], 
            Times[ \[Epsilon], 
              Plus[ k6, 
                Times[ k7, t], 
                Times[ 
                  Rational[ 1, 4], 
                  Power[ t, 2], \[Sigma], 
                  Power[ 
                    Plus[ 4, \[Sigma]], -1], 
                  Plus[ 
                    Times[ -2, k3], 
                    Times[ k4, \[Sigma]]]]]]]]}]]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The result is a finite eight-dimensional symmetry group depending on the \
small parameter \[Epsilon] and the parameter \[Sigma]. It was essential in \
the above calculation that both parameters \[Sigma] and \[Epsilon] are given \
in the parameters list. The last argument of the function ",
  StyleBox["ApproximateSymmetries[]", "MmaText"],
  " contains only the name of the perturbation parameter \[Epsilon]. In this \
way, we are able to select one of the parameters as a small quantity. The \
coefficients of the eight different vector fields are accessible by"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(cases\  = 
      \({{\(xi[1]\)[q, t, u], \(xi[2]\)[q, t, u]}, {\(phi[1]\)[q, t, u]}} /. 
            infinitesimals\) /. 
          \((Map[\((Thread[{k1, k2, k3, k4, k5, k6, k7, k8} \[Rule] #])\)&, 
              Permutations[{1, 0, 0, 0, 0, 0, 0, 0}]])\) // Simplify\)], 
  "Input"],

Cell[BoxData[
    \({{{0, 1}, {0}}, {{1, 0}, {0}}, {{q, 
          t - \(t\^2\ \[Epsilon]\ \[Sigma]\)\/\(2\ \((4 + \[Sigma])\)\)}, {
          \(2\ t\ u\ \[Epsilon]\)\/\(4 + \[Sigma]\)}}, {{0, 
          \(t\ \[Sigma]\ 
              \((\(-8\) + \((\(-2\) + t\ \[Epsilon])\)\ \[Sigma])\)\)\/\(4\ 
              \((4 + \[Sigma])\)\)}, {
          \(u\ \((4 + \[Sigma] - t\ \[Epsilon]\ \[Sigma])\)\)\/\(4 + 
              \[Sigma]\)}}, {{\[Epsilon], 0}, {0}}, {{0, \[Epsilon]}, {0}}, {{
          q\ \[Epsilon], t\ \[Epsilon]}, {0}}, {{
          \(q\ \[Epsilon]\ \[Sigma]\)\/2, 0}, {u\ \[Epsilon]}}}\)], "Output"]
}, Open  ]],

Cell[TextData[
"We select one sublist from the above list to reduce the original equation \
and find a similarity representation. Let us examine, for example, the fourth \
subgroup which contains the approximation parameter \[Epsilon]. The reduction \
follows by"], "Text",
  CellTags->"similarity representation"],

Cell[CellGroupData[{

Cell[BoxData[
    \(red1\  = \ 
      LieReduction[equation, {u}, {q, t}, 
          cases\[LeftDoubleBracket]4, 1\[RightDoubleBracket], 
          cases\[LeftDoubleBracket]4, 2\[RightDoubleBracket]] // Simplify; 
    \(red1 /. zeta1 \[Rule] \[Zeta]\_1\  // Flatten\) // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(q - \[Zeta]\_1 == 0\)},
          {
            \(t\^\(2/\[Sigma]\)\ u\ 
                  \((\(-8\) - 2\ \[Sigma] + t\ \[Epsilon]\ \[Sigma])\)\^\(2/
                      \[Sigma]\) - F\_1 == 0\)},
          {
            \(\((\(-8\) - 2\ \[Sigma] + t\ \[Epsilon]\ \[Sigma])\)\^\(2/
                    \[Sigma]\)\ 
                \((4\ \((
                        64 + 64\ \[Sigma] - 
                          4\ \((\(-5\) + 2\ t\^2\ \[Epsilon]\^2)\)\ 
                            \[Sigma]\^2 + 
                          \((2 - 2\ t\^2\ \[Epsilon]\^2 + t\^3\ \[Epsilon]\^3)
                              \)\ \[Sigma]\^3)\)\ F\_1\%2 + 
                    \[Sigma]\^3\ F\_1\%\[Sigma]\ \((F\_1)\)\_\[Zeta]\_1\%2 + 
                    \[Sigma]\^2\ F\_1\%\(1 + \[Sigma]\)\ 
                      \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1\))\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        ColumnWidths->0.95,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The result shows that the similarity variable ",
  Cell[BoxData[
      \(TraditionalForm\`\[Zeta]\_1 = q\)]],
  " and the similarity representation is given by ",
  Cell[BoxData[
      \(u = 
        t\^\(\(-2\)/\[Sigma]\)\ 
          \((\(-8\) - 2\ \[Sigma] + t\ \[Epsilon]\ \[Sigma])\)\^\(\(-2\)/
              \[Sigma]\)\ \(F\_1\) \((q)\)\)], "InlineFormula",
    FontFamily->"Times"],
  ". This expression depends on \[Epsilon] in a certain power of \[Epsilon]. \
Since our approximation order in \[Epsilon] is one, we have to expand the \
similarity solution in \[Epsilon] to first order. The expansion is carried \
out by"
}], "Text",
  CellTags->"approximation order"],

Cell[CellGroupData[{

Cell[BoxData[
    \(v1\  = \ 
      \(Series[u /. 
              Solve[red1\[LeftDoubleBracket]2\[RightDoubleBracket], u], {
              \[Epsilon], 0, 1}] // Normal\) // Simplify\)], "Input"],

Cell[BoxData[
    \({\(t\^\(\(-2\)/\[Sigma]\)\ 
          \((\(-8\) - 2\ \[Sigma])\)\^\(\(-2\)/\[Sigma]\)\ 
          \((4 + t\ \[Epsilon] + \[Sigma])\)\ F1[q]\)\/\(4 + \[Sigma]\)}\)], 
  "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(rule3\  = \ 
      u \[Rule] 
        Function@@{{q, t}, v1\[LeftDoubleBracket]1\[RightDoubleBracket]}\)], 
  "Input"],

Cell[BoxData[
    \(u \[Rule] 
      Function[{q, t}, 
        \(t\^\(\(-2\)/\[Sigma]\)\ 
            \((\(-8\) - 2\ \[Sigma])\)\^\(\(-2\)/\[Sigma]\)\ 
            \((4 + t\ \[Epsilon] + \[Sigma])\)\ F1[q]\)\/\(4 + \[Sigma]\)]
        \)], "Output"]
}, Open  ]],

Cell[TextData[{
  "On the other hand, the arbitrary function ",
  StyleBox["F",
    FontSlant->"Italic"],
  "1 has to satisfy a second-order ODE. Examining this equation, we realize \
that the second-order ODE contains coefficients depending on ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  ". However, this dependence will directly lead to inconsistent results. \
Since we are looking for solutions linear in \[Epsilon], we have to eliminate \
the time-dependent terms. The coefficients containing ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  " contribute terms in \[Epsilon] of the order two or higher. A consistent \
similarity reduction is only possible if we eliminate those terms by choosing \
",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon] = 0\)]],
  ". The second-order ODE reduces thus to"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(redh\  = \ 
      red1\[LeftDoubleBracket]3\[RightDoubleBracket] /. \[Epsilon] \[Rule] 0; 
    redh /. zeta1 \[Rule] \[Zeta]\_1\  // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            \(\((\(-8\) - 2\ \[Sigma])\)\^\(2/\[Sigma]\)\ 
                \((4\ \((64 + 64\ \[Sigma] + 20\ \[Sigma]\^2 + 2\ \[Sigma]\^3)
                        \)\ F\_1\%2 + 
                    \[Sigma]\^3\ F\_1\%\[Sigma]\ \((F\_1)\)\_\[Zeta]\_1\%2\n
                    \t + \[Sigma]\^2\ F\_1\%\(1 + \[Sigma]\)\ 
                      \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1\))\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "free of \[Epsilon] and ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  ". The solution of this approximation equation follows by"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol1\  = \ DSolve[redh, F1, zeta1]\)], "Input"],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{"F1", "\[Rule]", 
            RowBox[{"(", 
              RowBox[{
                RowBox[{
                  RowBox[{"InverseFunction", "[", 
                    RowBox[{
                      RowBox[{\(C[2]\), "-", 
                        RowBox[{
                        \((#1\^\(\(-1\) - \[Sigma]\)\ 
                              \((C[1] - 256\ #1\^\(2 + \[Sigma]\) - 
                                  128\ \[Sigma]\ #1\^\(2 + \[Sigma]\) - 
                                  16\ \[Sigma]\^2\ #1\^\(2 + \[Sigma]\))\))
                            \), "/", 
                          RowBox[{"(", 
                            RowBox[{
                            "8", " ", "\[Sigma]", " ", 
                              \(\((4 + \[Sigma])\)\^2\), " ", 
                              RowBox[{"\[Sqrt]", 
                                RowBox[{"(", 
                                  
                                  StyleBox[
                                    \(1\/\[Sigma]\^2
                                        \((#1\^\(\(-2\)\ \[Sigma]\)\ 
                                        \((C[1] - 256\ #1\^\(2 + \[Sigma]\) - 
                                        128\ \[Sigma]\ #1\^\(2 + \[Sigma]\) - 
                                        16\ \[Sigma]\^2\ #1\^\(2 + \[Sigma]\))
                                        \))\)\),
                                    ScriptLevel->0], ")"}]}]}], ")"}]}], "-", 
                        \(\((\((\(-2\) - \[Sigma])\)\ C[1]\ 
                              Hypergeometric2F1[1\/\(\(-2\) - \[Sigma]\), 
                                1\/2, 1 + 1\/\(\(-2\) - \[Sigma]\), 
                                \(-\(\(\((\(-256\) - 128\ \[Sigma] - 
                                        16\ \[Sigma]\^2)\)\ #1
                                        \^\(2 + \[Sigma]\)\)\/C[1]\)\)]\ #1
                                \^\(\(-1\) + 1\/4\ \((\(-2\) - \[Sigma])\) - 
                                  \[Sigma] + \(2 + \[Sigma]\)\/4\)\ 
                              \@\(C[1] - 256\ #1\^\(2 + \[Sigma]\) - 
                                  128\ \[Sigma]\ #1\^\(2 + \[Sigma]\) - 
                                  16\ \[Sigma]\^2\ #1\^\(2 + \[Sigma]\)\)\ 
                              \@\(1 + 
                                  \(\((\(-256\) - 128\ \[Sigma] - 
                                        16\ \[Sigma]\^2)\)\ #1
                                        \^\(2 + \[Sigma]\)\)\/C[1]\))\)/
                          \((8\ \[Sigma]\ \((2 + \[Sigma])\)\ 
                              \((4 + \[Sigma])\)\^2\ 
                              \[Sqrt]\((
                                  \(1\/\[Sigma]\^2\) 
                                    \((#1\^\(\(-2\)\ \[Sigma]\)\ 
                                        \((C[1] - 256\ #1\^\(2 + \[Sigma]\) - 
                                        128\ \[Sigma]\ #1\^\(2 + \[Sigma]\) - 
                                        16\ \[Sigma]\^2\ #1\^\(2 + \[Sigma]\))
                                        \))\))\)\ 
                              \@\(C[1] + 
                                  \((\(-256\) - 128\ \[Sigma] - 
                                        16\ \[Sigma]\^2)\)\ #1
                                      \^\(2 + \[Sigma]\)\))\)\)}], "&"}], 
                    "]"}], "[", "#1", "]"}], "&"}], ")"}]}], "}"}], ",", 
        RowBox[{"{", 
          RowBox[{"F1", "\[Rule]", 
            RowBox[{"(", 
              RowBox[{
                RowBox[{
                  RowBox[{"InverseFunction", "[", 
                    RowBox[{
                      RowBox[{\(C[2]\), "+", 
                        RowBox[{
                        \((#1\^\(\(-1\) - \[Sigma]\)\ 
                              \((C[1] - 256\ #1\^\(2 + \[Sigma]\) - 
                                  128\ \[Sigma]\ #1\^\(2 + \[Sigma]\) - 
                                  16\ \[Sigma]\^2\ #1\^\(2 + \[Sigma]\))\))
                            \), "/", 
                          RowBox[{"(", 
                            RowBox[{
                            "8", " ", "\[Sigma]", " ", 
                              \(\((4 + \[Sigma])\)\^2\), " ", 
                              RowBox[{"\[Sqrt]", 
                                RowBox[{"(", 
                                  
                                  StyleBox[
                                    \(1\/\[Sigma]\^2
                                        \((#1\^\(\(-2\)\ \[Sigma]\)\ 
                                        \((C[1] - 256\ #1\^\(2 + \[Sigma]\) - 
                                        128\ \[Sigma]\ #1\^\(2 + \[Sigma]\) - 
                                        16\ \[Sigma]\^2\ #1\^\(2 + \[Sigma]\))
                                        \))\)\),
                                    ScriptLevel->0], ")"}]}]}], ")"}]}], "+", 
                        \(\((\((\(-2\) - \[Sigma])\)\ C[1]\ 
                              Hypergeometric2F1[1\/\(\(-2\) - \[Sigma]\), 
                                1\/2, 1 + 1\/\(\(-2\) - \[Sigma]\), 
                                \(-\(\(\((\(-256\) - 128\ \[Sigma] - 
                                        16\ \[Sigma]\^2)\)\ #1
                                        \^\(2 + \[Sigma]\)\)\/C[1]\)\)]\ #1
                                \^\(\(-1\) + 1\/4\ \((\(-2\) - \[Sigma])\) - 
                                  \[Sigma] + \(2 + \[Sigma]\)\/4\)\ 
                              \@\(C[1] - 256\ #1\^\(2 + \[Sigma]\) - 
                                  128\ \[Sigma]\ #1\^\(2 + \[Sigma]\) - 
                                  16\ \[Sigma]\^2\ #1\^\(2 + \[Sigma]\)\)\ 
                              \@\(1 + 
                                  \(\((\(-256\) - 128\ \[Sigma] - 
                                        16\ \[Sigma]\^2)\)\ #1
                                        \^\(2 + \[Sigma]\)\)\/C[1]\))\)/
                          \((8\ \[Sigma]\ \((2 + \[Sigma])\)\ 
                              \((4 + \[Sigma])\)\^2\ 
                              \[Sqrt]\((
                                  \(1\/\[Sigma]\^2\) 
                                    \((#1\^\(\(-2\)\ \[Sigma]\)\ 
                                        \((C[1] - 256\ #1\^\(2 + \[Sigma]\) - 
                                        128\ \[Sigma]\ #1\^\(2 + \[Sigma]\) - 
                                        16\ \[Sigma]\^2\ #1\^\(2 + \[Sigma]\))
                                        \))\))\)\ 
                              \@\(C[1] + 
                                  \((\(-256\) - 128\ \[Sigma] - 
                                        16\ \[Sigma]\^2)\)\ #1
                                      \^\(2 + \[Sigma]\)\))\)\)}], "&"}], 
                    "]"}], "[", "#1", "]"}], "&"}], ")"}]}], "}"}]}], "}"}]], 
  "Output"]
}, Open  ]],

Cell[TextData[{
  "The results are two solutions for function ",
  Cell[BoxData[
      \(TraditionalForm\`F\_1\)]],
  ". Both representations contain special functions of the hypergeometric \
type ",
  Cell[BoxData[
      \(TraditionalForm\`\_2\[NegativeVeryThinSpace]F\_1\)]],
  " depending on the isentropic exponent \[Sigma]. The representation of the \
solution in the original variables ",
  Cell[BoxData[
      \(TraditionalForm\`q\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  " follows by substituting the results into the approximated similarity \
solution. We choose here the first solution for the representation:"
}], "Text",
  CellTags->"isentropic exponent"],

Cell[CellGroupData[{

Cell[BoxData[
    \(solution\  = \ 
      \(u[q, t]\  /. \ rule3\)\  /. \ 
        sol1\[LeftDoubleBracket]1\[RightDoubleBracket]\)], "Input"],

Cell[BoxData[
    RowBox[{\(1\/\(4 + \[Sigma]\)\), 
      RowBox[{"(", 
        RowBox[{
        \(t\^\(\(-2\)/\[Sigma]\)\), " ", 
          \(\((\(-8\) - 2\ \[Sigma])\)\^\(\(-2\)/\[Sigma]\)\), " ", 
          \((4 + t\ \[Epsilon] + \[Sigma])\), " ", 
          RowBox[{
            RowBox[{"InverseFunction", "[", 
              RowBox[{
                RowBox[{\(C[2]\), "-", 
                  RowBox[{
                  \((#1\^\(\(-1\) - \[Sigma]\)\ 
                        \((C[1] - 256\ #1\^\(2 + \[Sigma]\) - 
                            128\ \[Sigma]\ #1\^\(2 + \[Sigma]\) - 
                            16\ \[Sigma]\^2\ #1\^\(2 + \[Sigma]\))\))\), 
                    "/", 
                    RowBox[{"(", 
                      RowBox[{
                      "8", " ", "\[Sigma]", " ", \(\((4 + \[Sigma])\)\^2\), 
                        " ", 
                        RowBox[{"\[Sqrt]", 
                          RowBox[{"(", 
                            
                            StyleBox[
                              \(1\/\[Sigma]\^2\((#1\^\(\(-2\)\ \[Sigma]\)\ 
                                  \((C[1] - 256\ #1\^\(2 + \[Sigma]\) - 
                                      128\ \[Sigma]\ #1\^\(2 + \[Sigma]\) - 
                                      16\ \[Sigma]\^2\ #1\^\(2 + \[Sigma]\))
                                    \))\)\),
                              ScriptLevel->0], ")"}]}]}], ")"}]}], "-", 
                  \(\((\((\(-2\) - \[Sigma])\)\ C[1]\ 
                        Hypergeometric2F1[1\/\(\(-2\) - \[Sigma]\), 1\/2, 
                          1 + 1\/\(\(-2\) - \[Sigma]\), 
                          \(-\(\(\((\(-256\) - 128\ \[Sigma] - 
                                      16\ \[Sigma]\^2)\)\ #1
                                    \^\(2 + \[Sigma]\)\)\/C[1]\)\)]\ #1
                          \^\(\(-1\) + 1\/4\ \((\(-2\) - \[Sigma])\) - 
                            \[Sigma] + \(2 + \[Sigma]\)\/4\)\ 
                        \@\(C[1] - 256\ #1\^\(2 + \[Sigma]\) - 
                            128\ \[Sigma]\ #1\^\(2 + \[Sigma]\) - 
                            16\ \[Sigma]\^2\ #1\^\(2 + \[Sigma]\)\)\ 
                        \@\(1 + 
                            \(\((\(-256\) - 128\ \[Sigma] - 16\ \[Sigma]\^2)
                                  \)\ #1\^\(2 + \[Sigma]\)\)\/C[1]\))\)/
                    \((8\ \[Sigma]\ \((2 + \[Sigma])\)\ 
                        \((4 + \[Sigma])\)\^2\ 
                        \[Sqrt]\((
                            \(1\/\[Sigma]\^2\) 
                              \((#1\^\(\(-2\)\ \[Sigma]\)\ 
                                  \((C[1] - 256\ #1\^\(2 + \[Sigma]\) - 
                                      128\ \[Sigma]\ #1\^\(2 + \[Sigma]\) - 
                                      16\ \[Sigma]\^2\ #1\^\(2 + \[Sigma]\))
                                    \))\))\)\ 
                        \@\(C[1] + 
                            \((\(-256\) - 128\ \[Sigma] - 16\ \[Sigma]\^2)
                                \)\ #1\^\(2 + \[Sigma]\)\))\)\)}], "&"}], 
              "]"}], "[", "q", "]"}]}], ")"}]}]], "Output",
  PageBreakBelow->True]
}, Open  ]],

Cell[TextData[{
  StyleBox["C",
    FontSlant->"Italic"],
  "[1] and ",
  StyleBox["C",
    FontSlant->"Italic"],
  "[2] are constants of integration and ",
  StyleBox["InverseFunction[]", "MmaText"],
  " represents the inverse of the function ",
  StyleBox["f", "TI",
    FontSlant->"Italic"],
  "."
}], "Text",
  CellTags->"InverseFunction[]"],

Cell[TextData[{
  "The original equation of the isentropic fluid in Lagrange coordinates can \
be examined either for specific values of the isentropic parameter \[Sigma] \
or for different models for the pressure. Let us discuss two cases for a \
specific value of \[Sigma]. The change of pressure in the model is left to \
the reader as an exercise. The following calculation of the approximate \
symmetries assume that ",
  Cell[BoxData[
      \(TraditionalForm\`\[Sigma] = \(-4\)/3\)]],
  ":"
}], "Text",
  CellTags->"isentropic fluid"],

Cell[CellGroupData[{

Cell[BoxData[
    \(infinitesimals\  = \ 
      ApproximateSymmetries[
        equation /. \[Sigma] \[Rule] \(-4\)/3, {u}, {q, t}, {\[Epsilon]}, 
        \[Epsilon], 
        SubstitutionRules \[Rule] {\[PartialD]\_\(t, t\)u[q, t]}]; 
    infinitesimals // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{\(\[Phi]\_1\), "==", 
              RowBox[{
                StyleBox[\(\(3\ k4\ u\)\/2\),
                  ScriptLevel->0], "-", 
                StyleBox[\(\(3\ k6\ u\)\/2\),
                  ScriptLevel->0], "-", \(3\ k7\ q\ u\)}]}]},
          {
            RowBox[{\(\ \ \ \ \ \ \ \ \ \ +\), 
              RowBox[{
                RowBox[{"(", 
                  RowBox[{
                    RowBox[{"-", 
                      StyleBox[\(\(3\ k2\ u\)\/2\),
                        ScriptLevel->0]}], "+", 
                    StyleBox[\(\(3\ k9\ u\)\/2\),
                      ScriptLevel->0], "-", \(3\ k3\ q\ u\), "+", 
                    StyleBox[\(\(3\ k4\ t\ u\)\/4\),
                      ScriptLevel->0]}], ")"}], " ", "\[Epsilon]"}]}]},
          {
            \(\[Xi]\_1 == 
              k5 + q\ \((k6 + k7\ q)\) + 
                \((k10 + q\ \((k2 + k3\ q)\))\)\ \[Epsilon]\)},
          {
            RowBox[{\(\[Xi]\_2\), "==", 
              RowBox[{"k1", "+", \(k4\ t\), "+", 
                RowBox[{
                  RowBox[{"(", 
                    RowBox[{"k8", "+", \(k9\ t\), "+", 
                      StyleBox[\(\(k4\ t\^2\)\/4\),
                        ScriptLevel->0]}], ")"}], " ", "\[Epsilon]"}]}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], 
          Plus[ 
            Times[ 
              Rational[ 3, 2], k4, u], 
            Times[ 
              Rational[ -3, 2], k6, u], 
            Times[ -3, k7, q, u], 
            Times[ 
              Plus[ 
                Times[ 
                  Rational[ -3, 2], k2, u], 
                Times[ 
                  Rational[ 3, 2], k9, u], 
                Times[ -3, k3, q, u], 
                Times[ 
                  Rational[ 3, 4], k4, t, u]], \[Epsilon]]]], 
        Equal[ 
          Subscript[ \[Xi], 1], 
          Plus[ k5, 
            Times[ q, 
              Plus[ k6, 
                Times[ k7, q]]], 
            Times[ 
              Plus[ k10, 
                Times[ q, 
                  Plus[ k2, 
                    Times[ k3, q]]]], \[Epsilon]]]], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Plus[ k1, 
            Times[ k4, t], 
            Times[ 
              Plus[ k8, 
                Times[ k9, t], 
                Times[ 
                  Rational[ 1, 4], k4, 
                  Power[ t, 2]]], \[Epsilon]]]]}]]], "Output"]
}, Open  ]],

Cell["\<\
The result of the calculation is a 10-dimensional finite symmetry \
group in the first-order approximation. The coefficients of the generating \
vector fields are\
\>", "Text",
  CellTags->{"finite symmetry group", "vector field"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(cases2\  = \ 
      \({{\(xi[1]\)[q, t, u], \(xi[2]\)[q, t, u]}, {\(phi[1]\)[q, t, u]}} /. 
          infinitesimals\) /. 
        \((Map[\((Thread[{k1, k2, k3, k4, k5, k6, k7, k8, k9, k10} \[Rule] #])
                \)&, Permutations[{1, 0, 0, 0, 0, 0, 0, 0, 0, 0}]])\)\)], 
  "Input"],

Cell[BoxData[
    \({{{0, 1}, {0}}, {{q\ \[Epsilon], 0}, {
          \(-\(\(3\ u\ \[Epsilon]\)\/2\)\)}}, {{q\^2\ \[Epsilon], 0}, {
          \(-3\)\ q\ u\ \[Epsilon]}}, {{0, t + \(t\^2\ \[Epsilon]\)\/4}, {
          \(3\ u\)\/2 + \(3\ t\ u\ \[Epsilon]\)\/4}}, {{1, 0}, {0}}, {{q, 
          0}, {\(-\(\(3\ u\)\/2\)\)}}, {{q\^2, 0}, {\(-3\)\ q\ u}}, {{0, 
          \[Epsilon]}, {0}}, {{0, t\ \[Epsilon]}, {
          \(3\ u\ \[Epsilon]\)\/2}}, {{\[Epsilon], 0}, {0}}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "A specific reduction for the third vector field with ",
  Cell[BoxData[
      FormBox[
        RowBox[{\(v\&\[RightVector]\_3\), " ", "=", " ", 
          RowBox[{"\[Epsilon]", " ", 
            StyleBox["q",
              FontSlant->"Italic"], " ", \(\[PartialD]\_q\ \(-3\)\), 
            "\[Epsilon]", " ", "q", " ", "u", " ", \(\[PartialD]\_u\)}]}], 
        TraditionalForm]]],
  " follows by"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(red2\  = \ 
      LieReduction[equation\  /. \ \[Sigma] \[Rule] \(-4\)/3, {u}, {q, t}, 
          cases2\[LeftDoubleBracket]3, 1\[RightDoubleBracket], 
          cases2\[LeftDoubleBracket]3, 2\[RightDoubleBracket]] // Simplify; 
    \(red2 /. zeta1 \[Rule] \[Zeta]\_1 // Flatten\) // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {\(t - \[Zeta]\_1 == 0\)},
          {\(q\^3\ u - F\_1 == 0\)},
          {
            \(\(-\[Epsilon]\)\ F1\_\(\[Zeta]\_1\) + 
                F1\_\(\[Zeta]\_1, \[Zeta]\_1\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The similarity representation is given by the similarity solution ",
  Cell[BoxData[
      \(TraditionalForm\`u = \(\(F\_1\)(t)\)/q\^3\)]],
  " with similarity variable ",
  Cell[BoxData[
      \(TraditionalForm\`\[Zeta]\_1 = t\)]],
  ". The auxiliary function ",
  Cell[BoxData[
      \(TraditionalForm\`F\_1\)]],
  " has to satisfy a second-order ODE. We realize that the similarity \
solution does not depend on \[Epsilon]. However, \[Epsilon] occurs in the \
determining equation of ",
  Cell[BoxData[
      \(TraditionalForm\`F\_1\)]],
  ". The solution of the equation for ",
  Cell[BoxData[
      \(TraditionalForm\`F\_1\)]],
  " is given by"
}], "Text",
  CellTags->{
  "similarity variable", "similarity solution", "determining equation"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(solh\  = \ 
      DSolve[red2\[LeftDoubleBracket]3\[RightDoubleBracket]\ , F1, zeta1]\)], 
  "Input"],

Cell[BoxData[
    \({{F1 \[Rule] \((\(E\^\(\[Epsilon]\ #1\)\ C[1]\)\/\[Epsilon] + C[2]&)\)}}
      \)], "Output"]
}, Open  ]],

Cell[TextData[
"This function actually does not depend linearly on \[Epsilon] and, thus, it \
does not fit into the approximation scheme of a first-order approximation. \
However, it is a solution of the reduced equation. The representation of this \
expression in Lagrange coordinates reads"], "Text",
  CellTags->"Lagrange coordinates"],

Cell[CellGroupData[{

Cell[BoxData[
    \(solution2\  = \ 
      Flatten[Solve[
          red2\[LeftDoubleBracket]2\[RightDoubleBracket]\  /. \ solh, u]]\)], 
  "Input"],

Cell[BoxData[
    \({u \[Rule] 
        \(E\^\(t\ \[Epsilon]\)\ C[1] + \[Epsilon]\ C[2]\)\/\(q\^3\ 
            \[Epsilon]\)}\)], "Output"]
}, Open  ]],

Cell["\<\
A graphical representation of the solution for specific values of \
the parameters is given as follows:\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Plot3D[
      Evaluate[\(u\  /. solution2\)\  /. {C[1] \[Rule] 1, C[2] \[Rule] 2, 
            \[Epsilon] \[Rule]  .1}], {q, 1, 2}, {t, 0, 20}, 
      AxesLabel \[Rule] {"\<q\>", "\<t\>", "\<u\>"}, Mesh \[Rule] False, 
      PlotPoints \[Rule] 35]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .81114 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% SurfaceGraphics
/Courier findfont 10  scalefont  setfont
% Scaling calculations
1.11022e-16 1.04977 -0.0679587 1.04977 [
[.02757 .24618 -6 -8.70276 ]
[.02757 .24618 0 .29724 ]
[.14598 .19958 -17.886 -9 ]
[.14598 .19958 .11404 0 ]
[.27013 .15073 -17.1354 -9 ]
[.27013 .15073 .86461 0 ]
[.40046 .09948 -16.3848 -9 ]
[.40046 .09948 1.61518 0 ]
[.53743 .04564 -15.6342 -9 ]
[.53743 .04564 2.36575 0 ]
[.68156 -0.01098 -4.96123 -9 ]
[.68156 -0.01098 1.03877 0 ]
[.29165 .07573 -9.31117 -15 ]
[.29165 .07573 .68883 0 ]
[.70096 -0.00478 0 -6.26206 ]
[.70096 -0.00478 6 2.73794 ]
[.78313 .12104 0 -6.13858 ]
[.78313 .12104 6 2.86142 ]
[.8565 .23339 0 -6.03127 ]
[.8565 .23339 12 2.96873 ]
[.9224 .33431 0 -5.93715 ]
[.9224 .33431 12 3.06285 ]
[.98191 .42546 0 -5.85393 ]
[.98191 .42546 12 3.14607 ]
[.91861 .21225 0 -10.0521 ]
[.91861 .21225 10 4.94788 ]
[.02411 .26511 -6 -2.74232 ]
[.02411 .26511 0 6.25768 ]
[.01687 .31052 -12 -2.78071 ]
[.01687 .31052 0 6.21929 ]
[.0094 .3573 -12 -2.82036 ]
[.0094 .3573 0 6.17964 ]
[.00171 .4055 -12 -2.86135 ]
[.00171 .4055 0 6.13865 ]
[-0.00621 .45518 -12 -2.90373 ]
[-0.00621 .45518 0 6.09627 ]
[-0.05457 .39542 -10 -4.72232 ]
[-0.05457 .39542 0 10.2777 ]
[ 0 0 0 0 ]
[ 1 .81114 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.03716 .25514 m
.68874 0 L
s
.03716 .25514 m
.04196 .25962 L
s
[(1)] .02757 .24618 1 .93395 Mshowa
.1552 .20892 m
.15981 .21359 L
s
[(1.2)] .14598 .19958 .98733 1 Mshowa
.27893 .16047 m
.28333 .16533 L
s
[(1.4)] .27013 .15073 .90393 1 Mshowa
.40878 .10962 m
.41294 .11469 L
s
[(1.6)] .40046 .09948 .82054 1 Mshowa
.54521 .0562 m
.54911 .06148 L
s
[(1.8)] .53743 .04564 .73714 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(2)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.06616 .24378 m
.06901 .2465 L
s
.09549 .2323 m
.09832 .23504 L
s
.12517 .22067 m
.12797 .22345 L
s
.18558 .19702 m
.18832 .19985 L
s
.21633 .18498 m
.21903 .18784 L
s
.24744 .1728 m
.25012 .17569 L
s
.3108 .14799 m
.31341 .15094 L
s
.34306 .13535 m
.34564 .13834 L
s
.37572 .12257 m
.37826 .12558 L
s
.44225 .09652 m
.44471 .09959 L
s
.47614 .08324 m
.47856 .08635 L
s
.51046 .06981 m
.51284 .07294 L
s
.58041 .04242 m
.5827 .04562 L
s
.61605 .02846 m
.6183 .03169 L
s
.65216 .01432 m
.65436 .01759 L
s
gsave
.29165 .07573 -70.3112 -19 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 23 translate 1 -1 scale
63.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(q) show
69.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
%%DocumentNeededResources: font Courier
%%DocumentNeededFonts: Courier
%%DocumentFonts: font Courier
grestore
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.68874 0 m
.68263 .00239 L
s
[(0)] .70096 -0.00478 -1 .39157 Mshowa
.7708 .12553 m
.76464 .12778 L
s
[(5)] .78313 .12104 -1 .36413 Mshowa
.84407 .23761 m
.83786 .23973 L
s
[(10)] .8565 .23339 -1 .34028 Mshowa
.9099 .3383 m
.90365 .34029 L
s
[(15)] .9224 .33431 -1 .31937 Mshowa
.96935 .42924 m
.96306 .43113 L
s
[(20)] .98191 .42546 -1 .30087 Mshowa
.125 Mabswid
.70593 .0263 m
.70226 .02771 L
s
.72272 .05198 m
.71904 .05338 L
s
.73912 .07706 m
.73543 .07845 L
s
.75514 .10158 m
.75145 .10294 L
s
.78611 .14895 m
.78241 .15028 L
s
.80108 .17185 m
.79737 .17317 L
s
.81573 .19425 m
.81201 .19555 L
s
.83006 .21617 m
.82633 .21745 L
s
.8578 .2586 m
.85407 .25986 L
s
.87123 .27915 m
.8675 .28039 L
s
.88439 .29928 m
.88065 .3005 L
s
.89727 .31899 m
.89353 .3202 L
s
.92227 .35722 m
.91851 .3584 L
s
.93439 .37576 m
.93063 .37693 L
s
.94627 .39394 m
.94251 .3951 L
s
.95792 .41176 m
.95416 .41291 L
s
gsave
.91861 .21225 -61 -14.0521 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 23 translate 1 -1 scale
63.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(t) show
69.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
%%DocumentNeededResources: font Courier
%%DocumentNeededFonts: Courier
%%DocumentFonts: font Courier
grestore
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
.03634 .26033 m
.04245 .25795 L
s
[(0)] .02411 .26511 1 -0.3906 Mshowa
.02913 .30584 m
.03525 .3035 L
s
[(10)] .01687 .31052 1 -0.38207 Mshowa
.0217 .35271 m
.02784 .35042 L
s
[(20)] .0094 .3573 1 -0.37325 Mshowa
.01404 .40101 m
.02021 .39876 L
s
[(30)] .00171 .4055 1 -0.36414 Mshowa
.00615 .4508 m
.01234 .4486 L
s
[(40)] -0.00621 .45518 1 -0.35473 Mshowa
.125 Mabswid
.03491 .26933 m
.03858 .2679 L
s
.03348 .27838 m
.03715 .27696 L
s
.03204 .28748 m
.03571 .28606 L
s
.03058 .29663 m
.03426 .29522 L
s
.02766 .3151 m
.03134 .3137 L
s
.02618 .32442 m
.02986 .32303 L
s
.0247 .3338 m
.02838 .33241 L
s
.0232 .34322 m
.02689 .34184 L
s
.02019 .36225 m
.02388 .36088 L
s
.01866 .37185 m
.02236 .37049 L
s
.01713 .38151 m
.02083 .38015 L
s
.01559 .39123 m
.01929 .38988 L
s
.01249 .41084 m
.01619 .4095 L
s
.01092 .42074 m
.01462 .4194 L
s
.00934 .4307 m
.01304 .42937 L
s
.00775 .44071 m
.01146 .43939 L
s
.00455 .46094 m
.00826 .45963 L
s
.00293 .47114 m
.00664 .46984 L
s
.0013 .48141 m
.00502 .48012 L
s
gsave
-0.05457 .39542 -71 -8.72232 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 23 translate 1 -1 scale
63.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(u) show
69.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
%%DocumentNeededResources: font Courier
%%DocumentNeededFonts: Courier
%%DocumentFonts: font Courier
grestore
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
.40529 .59895 m
.03716 .25514 L
s
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.40529 .59895 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
0 0 m
1 0 L
1 .81114 L
0 .81114 L
closepath
clip
newpath
.645 .763 .934 r
.38829 .8034 .39787 .81114 .41384 .80695 .40433 .79917 Mtetra
.40433 .79917 .41384 .80695 .42991 .80275 .42045 .79492 Mtetra
.42045 .79492 .42991 .80275 .44608 .79852 .43668 .79064 Mtetra
.43668 .79064 .44608 .79852 .46234 .79426 .453 .78633 Mtetra
.658 .766 .929 r
.453 .78633 .46234 .79426 .47869 .78998 .46943 .78152 Mtetra
.844 .772 .782 r
.46943 .78152 .47869 .78998 .49519 .78107 .48613 .76095 Mtetra
.761 .641 .729 r
.48613 .76095 .49519 .78107 .51178 .76126 .50276 .74197 Mtetra
.757 .645 .737 r
.50276 .74197 .51178 .76126 .52833 .74292 .51936 .72437 Mtetra
.754 .649 .745 r
.51936 .72437 .52833 .74292 .54486 .72586 .53594 .70798 Mtetra
.75 .653 .753 r
.53594 .70798 .54486 .72586 .56138 .70994 .55252 .69265 Mtetra
.747 .656 .76 r
.55252 .69265 .56138 .70994 .57791 .69501 .5691 .67824 Mtetra
.743 .66 .768 r
.5691 .67824 .57791 .69501 .59446 .68095 .58571 .66466 Mtetra
.74 .663 .775 r
.58571 .66466 .59446 .68095 .61104 .66767 .60235 .65181 Mtetra
.737 .667 .782 r
.60235 .65181 .61104 .66767 .62766 .65507 .61903 .6396 Mtetra
.734 .67 .788 r
.61903 .6396 .62766 .65507 .64432 .64309 .63576 .62796 Mtetra
.73 .673 .794 r
.63576 .62796 .64432 .64309 .66104 .63165 .65255 .61683 Mtetra
.727 .676 .8 r
.65255 .61683 .66104 .63165 .67782 .6207 .66939 .60616 Mtetra
.724 .679 .806 r
.66939 .60616 .67782 .6207 .69466 .61019 .6863 .5959 Mtetra
.722 .682 .811 r
.6863 .5959 .69466 .61019 .71157 .60006 .70329 .586 Mtetra
.719 .685 .816 r
.70329 .586 .71157 .60006 .72855 .59029 .72034 .57644 Mtetra
.716 .687 .821 r
.72034 .57644 .72855 .59029 .74561 .58083 .73748 .56716 Mtetra
.714 .69 .825 r
.73748 .56716 .74561 .58083 .76275 .57165 .75469 .55815 Mtetra
.711 .692 .83 r
.75469 .55815 .76275 .57165 .77998 .56273 .77199 .54938 Mtetra
.709 .694 .834 r
.77199 .54938 .77998 .56273 .79729 .55404 .78938 .54083 Mtetra
.707 .696 .838 r
.78938 .54083 .79729 .55404 .81469 .54555 .80686 .53246 Mtetra
.705 .698 .841 r
.80686 .53246 .81469 .54555 .83218 .53725 .82444 .52427 Mtetra
.702 .7 .845 r
.82444 .52427 .83218 .53725 .84977 .52912 .84211 .51624 Mtetra
.701 .702 .848 r
.84211 .51624 .84977 .52912 .86746 .52114 .85987 .50835 Mtetra
.699 .703 .851 r
.85987 .50835 .86746 .52114 .88524 .5133 .87774 .50058 Mtetra
.697 .705 .854 r
.87774 .50058 .88524 .5133 .90312 .50558 .89571 .49293 Mtetra
.695 .706 .856 r
.89571 .49293 .90312 .50558 .92111 .49798 .91378 .48539 Mtetra
.694 .708 .859 r
.91378 .48539 .92111 .49798 .9392 .49047 .93195 .47793 Mtetra
.692 .709 .861 r
.93195 .47793 .9392 .49047 .9574 .48305 .95024 .47055 Mtetra
.691 .71 .863 r
.95024 .47055 .9574 .48305 .9757 .47571 .96863 .46325 Mtetra
.645 .763 .934 r
.37861 .79557 .38829 .8034 .40433 .79917 .3947 .7913 Mtetra
.3947 .7913 .40433 .79917 .42045 .79492 .41089 .787 Mtetra
.41089 .787 .42045 .79492 .43668 .79064 .42718 .78267 Mtetra
.42718 .78267 .43668 .79064 .453 .78633 .44356 .77832 Mtetra
.827 .782 .809 r
.44356 .77832 .453 .78633 .46943 .78152 .46028 .76111 Mtetra
.763 .641 .726 r
.46028 .76111 .46943 .78152 .48613 .76095 .47701 .74143 Mtetra
.76 .645 .734 r
.47701 .74143 .48613 .76095 .50276 .74197 .49369 .72323 Mtetra
.756 .649 .743 r
.49369 .72323 .50276 .74197 .51936 .72437 .51034 .70632 Mtetra
.752 .653 .751 r
.51034 .70632 .51936 .72437 .53594 .70798 .52697 .69054 Mtetra
.749 .656 .758 r
.52697 .69054 .53594 .70798 .55252 .69265 .5436 .67576 Mtetra
.745 .66 .766 r
.5436 .67576 .55252 .69265 .5691 .67824 .56024 .66185 Mtetra
.742 .664 .773 r
.56024 .66185 .5691 .67824 .58571 .66466 .57691 .6487 Mtetra
.738 .667 .78 r
.57691 .6487 .58571 .66466 .60235 .65181 .59361 .63624 Mtetra
.735 .67 .787 r
.59361 .63624 .60235 .65181 .61903 .6396 .61035 .62439 Mtetra
.732 .674 .793 r
.61035 .62439 .61903 .6396 .63576 .62796 .62715 .61307 Mtetra
.729 .677 .799 r
.62715 .61307 .63576 .62796 .65255 .61683 .644 .60223 Mtetra
.726 .68 .805 r
.644 .60223 .65255 .61683 .66939 .60616 .66091 .59182 Mtetra
.723 .682 .811 r
.66091 .59182 .66939 .60616 .6863 .5959 .6779 .5818 Mtetra
.72 .685 .816 r
.6779 .5818 .6863 .5959 .70329 .586 .69495 .57211 Mtetra
.717 .688 .821 r
.69495 .57211 .70329 .586 .72034 .57644 .71208 .56273 Mtetra
.715 .69 .825 r
.71208 .56273 .72034 .57644 .73748 .56716 .72928 .55363 Mtetra
.712 .692 .83 r
.72928 .55363 .73748 .56716 .75469 .55815 .74658 .54477 Mtetra
.71 .695 .834 r
.74658 .54477 .75469 .55815 .77199 .54938 .76395 .53614 Mtetra
.707 .697 .838 r
.76395 .53614 .77199 .54938 .78938 .54083 .78142 .52771 Mtetra
.705 .699 .841 r
.78142 .52771 .78938 .54083 .80686 .53246 .79898 .51946 Mtetra
.703 .7 .845 r
.79898 .51946 .80686 .53246 .82444 .52427 .81663 .51137 Mtetra
.701 .702 .848 r
.81663 .51137 .82444 .52427 .84211 .51624 .83438 .50342 Mtetra
.699 .704 .851 r
.83438 .50342 .84211 .51624 .85987 .50835 .85223 .49561 Mtetra
.697 .706 .854 r
.85223 .49561 .85987 .50835 .87774 .50058 .87018 .48791 Mtetra
.696 .707 .857 r
.87018 .48791 .87774 .50058 .89571 .49293 .88823 .48032 Mtetra
.694 .708 .859 r
.88823 .48032 .89571 .49293 .91378 .48539 .90639 .47282 Mtetra
.692 .71 .861 r
.90639 .47282 .91378 .48539 .93195 .47793 .92465 .46541 Mtetra
.691 .711 .864 r
.92465 .46541 .93195 .47793 .95024 .47055 .94303 .45807 Mtetra
.69 .712 .866 r
.94303 .45807 .95024 .47055 .96863 .46325 .96151 .4508 Mtetra
.645 .763 .934 r
.36882 .78766 .37861 .79557 .3947 .7913 .38497 .78334 Mtetra
.38497 .78334 .3947 .7913 .41089 .787 .40122 .77899 Mtetra
.40122 .77899 .41089 .787 .42718 .78267 .41757 .77461 Mtetra
.789 .783 .847 r
.41757 .77461 .42718 .78267 .44356 .77832 .43423 .76179 Mtetra
.788 .676 .741 r
.43423 .76179 .44356 .77832 .46028 .76111 .45107 .74132 Mtetra
.762 .645 .732 r
.45107 .74132 .46028 .76111 .47701 .74143 .46784 .72245 Mtetra
.758 .649 .74 r
.46784 .72245 .47701 .74143 .49369 .72323 .48456 .70498 Mtetra
.755 .652 .748 r
.48456 .70498 .49369 .72323 .51034 .70632 .50126 .68871 Mtetra
.751 .656 .756 r
.50126 .68871 .51034 .70632 .52697 .69054 .51794 .67351 Mtetra
.747 .66 .764 r
.51794 .67351 .52697 .69054 .5436 .67576 .53462 .65923 Mtetra
.744 .664 .772 r
.53462 .65923 .5436 .67576 .56024 .66185 .55132 .64578 Mtetra
.74 .667 .779 r
.55132 .64578 .56024 .66185 .57691 .6487 .56805 .63305 Mtetra
.737 .671 .786 r
.56805 .63305 .57691 .6487 .59361 .63624 .58481 .62095 Mtetra
.733 .674 .792 r
.58481 .62095 .59361 .63624 .61035 .62439 .60162 .60943 Mtetra
.73 .677 .798 r
.60162 .60943 .61035 .62439 .62715 .61307 .61848 .59841 Mtetra
.727 .68 .804 r
.61848 .59841 .62715 .61307 .644 .60223 .6354 .58784 Mtetra
.724 .683 .81 r
.6354 .58784 .644 .60223 .66091 .59182 .65238 .57767 Mtetra
.721 .686 .815 r
.65238 .57767 .66091 .59182 .6779 .5818 .66943 .56786 Mtetra
.718 .688 .82 r
.66943 .56786 .6779 .5818 .69495 .57211 .68655 .55836 Mtetra
.715 .691 .825 r
.68655 .55836 .69495 .57211 .71208 .56273 .70375 .54916 Mtetra
.713 .693 .829 r
.70375 .54916 .71208 .56273 .72928 .55363 .72103 .54022 Mtetra
.71 .695 .834 r
.72103 .54022 .72928 .55363 .74658 .54477 .7384 .5315 Mtetra
.708 .697 .838 r
.7384 .5315 .74658 .54477 .76395 .53614 .75586 .52299 Mtetra
.706 .699 .841 r
.75586 .52299 .76395 .53614 .78142 .52771 .7734 .51467 Mtetra
.704 .701 .845 r
.7734 .51467 .78142 .52771 .79898 .51946 .79104 .50652 Mtetra
.701 .703 .848 r
.79104 .50652 .79898 .51946 .81663 .51137 .80877 .49852 Mtetra
.7 .705 .851 r
.80877 .49852 .81663 .51137 .83438 .50342 .8266 .49065 Mtetra
.698 .706 .854 r
.8266 .49065 .83438 .50342 .85223 .49561 .84453 .48291 Mtetra
.696 .708 .857 r
.84453 .48291 .85223 .49561 .87018 .48791 .86257 .47527 Mtetra
.694 .709 .859 r
.86257 .47527 .87018 .48791 .88823 .48032 .8807 .46773 Mtetra
.693 .71 .862 r
.8807 .46773 .88823 .48032 .90639 .47282 .89895 .46028 Mtetra
.691 .712 .864 r
.89895 .46028 .90639 .47282 .92465 .46541 .9173 .4529 Mtetra
.69 .713 .866 r
.9173 .4529 .92465 .46541 .94303 .45807 .93575 .44559 Mtetra
.688 .714 .868 r
.93575 .44559 .94303 .45807 .96151 .4508 .95432 .43835 Mtetra
.645 .763 .934 r
.35892 .77966 .36882 .78766 .38497 .78334 .37513 .77529 Mtetra
.37513 .77529 .38497 .78334 .40122 .77899 .39144 .77089 Mtetra
.721 .778 .898 r
.39144 .77089 .40122 .77899 .41757 .77461 .40797 .76305 Mtetra
.824 .73 .762 r
.40797 .76305 .41757 .77461 .43423 .76179 .42494 .7417 Mtetra
.765 .644 .728 r
.42494 .7417 .43423 .76179 .45107 .74132 .44181 .72209 Mtetra
.761 .648 .737 r
.44181 .72209 .45107 .74132 .46784 .72245 .45862 .70399 Mtetra
.757 .652 .746 r
.45862 .70399 .46784 .72245 .48456 .70498 .47538 .68718 Mtetra
.753 .656 .754 r
.47538 .68718 .48456 .70498 .50126 .68871 .49212 .67152 Mtetra
.75 .66 .762 r
.49212 .67152 .50126 .68871 .51794 .67351 .50885 .65685 Mtetra
.746 .664 .77 r
.50885 .65685 .51794 .67351 .53462 .65923 .52558 .64305 Mtetra
.742 .667 .777 r
.52558 .64305 .53462 .65923 .55132 .64578 .54234 .63002 Mtetra
.739 .671 .784 r
.54234 .63002 .55132 .64578 .56805 .63305 .55913 .61767 Mtetra
.735 .674 .791 r
.55913 .61767 .56805 .63305 .58481 .62095 .57595 .60591 Mtetra
.732 .677 .797 r
.57595 .60591 .58481 .62095 .60162 .60943 .59282 .59469 Mtetra
.728 .68 .803 r
.59282 .59469 .60162 .60943 .61848 .59841 .60974 .58395 Mtetra
.725 .683 .809 r
.60974 .58395 .61848 .59841 .6354 .58784 .62673 .57362 Mtetra
.722 .686 .815 r
.62673 .57362 .6354 .58784 .65238 .57767 .64378 .56368 Mtetra
.719 .689 .82 r
.64378 .56368 .65238 .57767 .66943 .56786 .6609 .55406 Mtetra
.716 .691 .825 r
.6609 .55406 .66943 .56786 .68655 .55836 .67809 .54475 Mtetra
.714 .693 .829 r
.67809 .54475 .68655 .55836 .70375 .54916 .69537 .53571 Mtetra
.711 .696 .834 r
.69537 .53571 .70375 .54916 .72103 .54022 .71272 .52691 Mtetra
.709 .698 .838 r
.71272 .52691 .72103 .54022 .7384 .5315 .73016 .51832 Mtetra
.706 .7 .841 r
.73016 .51832 .7384 .5315 .75586 .52299 .7477 .50993 Mtetra
.704 .702 .845 r
.7477 .50993 .75586 .52299 .7734 .51467 .76532 .50171 Mtetra
.702 .703 .848 r
.76532 .50171 .7734 .51467 .79104 .50652 .78303 .49365 Mtetra
.7 .705 .851 r
.78303 .49365 .79104 .50652 .80877 .49852 .80085 .48572 Mtetra
.698 .707 .854 r
.80085 .48572 .80877 .49852 .8266 .49065 .81876 .47792 Mtetra
.696 .708 .857 r
.81876 .47792 .8266 .49065 .84453 .48291 .83677 .47024 Mtetra
.695 .71 .86 r
.83677 .47024 .84453 .48291 .86257 .47527 .85489 .46265 Mtetra
.693 .711 .862 r
.85489 .46265 .86257 .47527 .8807 .46773 .87311 .45516 Mtetra
.691 .712 .865 r
.87311 .45516 .8807 .46773 .89895 .46028 .89144 .44774 Mtetra
.69 .714 .867 r
.89144 .44774 .89895 .46028 .9173 .4529 .90988 .4404 Mtetra
.689 .715 .869 r
.90988 .4404 .9173 .4529 .93575 .44559 .92842 .43311 Mtetra
.687 .716 .871 r
.92842 .43311 .93575 .44559 .95432 .43835 .94708 .42588 Mtetra
.645 .763 .934 r
.34891 .77157 .35892 .77966 .37513 .77529 .36518 .76715 Mtetra
.36518 .76715 .37513 .77529 .39144 .77089 .38155 .7627 Mtetra
.842 .776 .789 r
.38155 .7627 .39144 .77089 .40797 .76305 .39858 .74263 Mtetra
.767 .644 .725 r
.39858 .74263 .40797 .76305 .42494 .7417 .41557 .72219 Mtetra
.764 .648 .734 r
.41557 .72219 .42494 .7417 .44181 .72209 .43248 .70339 Mtetra
.76 .652 .743 r
.43248 .70339 .44181 .72209 .45862 .70399 .44933 .68599 Mtetra
.756 .656 .751 r
.44933 .68599 .45862 .70399 .47538 .68718 .46613 .66981 Mtetra
.752 .66 .76 r
.46613 .66981 .47538 .68718 .49212 .67152 .48291 .6547 Mtetra
.748 .664 .768 r
.48291 .6547 .49212 .67152 .50885 .65685 .49969 .64053 Mtetra
.744 .667 .775 r
.49969 .64053 .50885 .65685 .52558 .64305 .51648 .62717 Mtetra
.741 .671 .782 r
.51648 .62717 .52558 .64305 .54234 .63002 .5333 .61453 Mtetra
.737 .674 .789 r
.5333 .61453 .54234 .63002 .55913 .61767 .55014 .60254 Mtetra
.733 .678 .796 r
.55014 .60254 .55913 .61767 .57595 .60591 .56702 .5911 Mtetra
.73 .681 .802 r
.56702 .5911 .57595 .60591 .59282 .59469 .58396 .58016 Mtetra
.727 .684 .808 r
.58396 .58016 .59282 .59469 .60974 .58395 .60095 .56967 Mtetra
.724 .686 .814 r
.60095 .56967 .60974 .58395 .62673 .57362 .618 .55958 Mtetra
.72 .689 .819 r
.618 .55958 .62673 .57362 .64378 .56368 .63511 .54983 Mtetra
.718 .692 .824 r
.63511 .54983 .64378 .56368 .6609 .55406 .6523 .5404 Mtetra
.715 .694 .829 r
.6523 .5404 .6609 .55406 .67809 .54475 .66957 .53125 Mtetra
.712 .696 .833 r
.66957 .53125 .67809 .54475 .69537 .53571 .68692 .52236 Mtetra
.71 .698 .837 r
.68692 .52236 .69537 .53571 .71272 .52691 .70435 .51368 Mtetra
.707 .7 .841 r
.70435 .51368 .71272 .52691 .73016 .51832 .72186 .50521 Mtetra
.705 .702 .845 r
.72186 .50521 .73016 .51832 .7477 .50993 .73947 .49692 Mtetra
.703 .704 .848 r
.73947 .49692 .7477 .50993 .76532 .50171 .75717 .4888 Mtetra
.7 .706 .852 r
.75717 .4888 .76532 .50171 .78303 .49365 .77496 .48081 Mtetra
.699 .707 .855 r
.77496 .48081 .78303 .49365 .80085 .48572 .79286 .47296 Mtetra
.697 .709 .857 r
.79286 .47296 .80085 .48572 .81876 .47792 .81085 .46522 Mtetra
.695 .71 .86 r
.81085 .46522 .81876 .47792 .83677 .47024 .82895 .45759 Mtetra
.693 .712 .863 r
.82895 .45759 .83677 .47024 .85489 .46265 .84715 .45005 Mtetra
.692 .713 .865 r
.84715 .45005 .85489 .46265 .87311 .45516 .86545 .44259 Mtetra
.69 .714 .867 r
.86545 .44259 .87311 .45516 .89144 .44774 .88387 .4352 Mtetra
.689 .715 .869 r
.88387 .4352 .89144 .44774 .90988 .4404 .90239 .42788 Mtetra
.687 .716 .871 r
.90239 .42788 .90988 .4404 .92842 .43311 .92103 .42061 Mtetra
.686 .717 .873 r
.92103 .42061 .92842 .43311 .94708 .42588 .93978 .4134 Mtetra
.645 .763 .934 r
.33878 .76339 .34891 .77157 .36518 .76715 .35512 .75892 Mtetra
.805 .78 .83 r
.35512 .75892 .36518 .76715 .38155 .7627 .37199 .74417 Mtetra
.782 .661 .73 r
.37199 .74417 .38155 .7627 .39858 .74263 .38913 .72281 Mtetra
.766 .648 .731 r
.38913 .72281 .39858 .74263 .41557 .72219 .40615 .70322 Mtetra
.762 .652 .74 r
.40615 .70322 .41557 .72219 .43248 .70339 .42308 .68516 Mtetra
.758 .656 .748 r
.42308 .68516 .43248 .70339 .44933 .68599 .43997 .66842 Mtetra
.754 .66 .757 r
.43997 .66842 .44933 .68599 .46613 .66981 .45681 .65282 Mtetra
.75 .664 .765 r
.45681 .65282 .46613 .66981 .48291 .6547 .47364 .63823 Mtetra
.746 .667 .773 r
.47364 .63823 .48291 .6547 .49969 .64053 .49047 .62452 Mtetra
.743 .671 .781 r
.49047 .62452 .49969 .64053 .51648 .62717 .50732 .61157 Mtetra
.739 .674 .788 r
.50732 .61157 .51648 .62717 .5333 .61453 .52418 .5993 Mtetra
.735 .678 .795 r
.52418 .5993 .5333 .61453 .55014 .60254 .54109 .58763 Mtetra
.732 .681 .801 r
.54109 .58763 .55014 .60254 .56702 .5911 .55803 .57649 Mtetra
.728 .684 .807 r
.55803 .57649 .56702 .5911 .58396 .58016 .57503 .56582 Mtetra
.725 .687 .813 r
.57503 .56582 .58396 .58016 .60095 .56967 .59208 .55556 Mtetra
.722 .689 .818 r
.59208 .55556 .60095 .56967 .618 .55958 .6092 .54567 Mtetra
.719 .692 .824 r
.6092 .54567 .618 .55958 .63511 .54983 .62638 .53612 Mtetra
.716 .694 .828 r
.62638 .53612 .63511 .54983 .6523 .5404 .64364 .52685 Mtetra
.713 .697 .833 r
.64364 .52685 .6523 .5404 .66957 .53125 .66098 .51786 Mtetra
.71 .699 .837 r
.66098 .51786 .66957 .53125 .68692 .52236 .6784 .50909 Mtetra
.708 .701 .841 r
.6784 .50909 .68692 .52236 .70435 .51368 .6959 .50054 Mtetra
.705 .703 .845 r
.6959 .50054 .70435 .51368 .72186 .50521 .71349 .49217 Mtetra
.703 .705 .848 r
.71349 .49217 .72186 .50521 .73947 .49692 .73117 .48398 Mtetra
.701 .706 .852 r
.73117 .48398 .73947 .49692 .75717 .4888 .74895 .47593 Mtetra
.699 .708 .855 r
.74895 .47593 .75717 .4888 .77496 .48081 .76683 .46802 Mtetra
.697 .709 .858 r
.76683 .46802 .77496 .48081 .79286 .47296 .7848 .46023 Mtetra
.695 .711 .86 r
.7848 .46023 .79286 .47296 .81085 .46522 .80288 .45254 Mtetra
.694 .712 .863 r
.80288 .45254 .81085 .46522 .82895 .45759 .82106 .44495 Mtetra
.692 .713 .865 r
.82106 .44495 .82895 .45759 .84715 .45005 .83934 .43745 Mtetra
.69 .715 .867 r
.83934 .43745 .84715 .45005 .86545 .44259 .85773 .43002 Mtetra
.689 .716 .869 r
.85773 .43002 .86545 .44259 .88387 .4352 .87623 .42266 Mtetra
.688 .717 .871 r
.87623 .42266 .88387 .4352 .90239 .42788 .89485 .41535 Mtetra
.686 .718 .873 r
.89485 .41535 .90239 .42788 .92103 .42061 .91357 .4081 Mtetra
.685 .719 .875 r
.91357 .4081 .92103 .42061 .93978 .4134 .93241 .40088 Mtetra
.734 .778 .889 r
.32854 .75511 .33878 .76339 .35512 .75892 .34515 .74641 Mtetra
.822 .721 .753 r
.34515 .74641 .35512 .75892 .37199 .74417 .36244 .724 Mtetra
.769 .647 .727 r
.36244 .724 .37199 .74417 .38913 .72281 .3796 .70354 Mtetra
.765 .651 .736 r
.3796 .70354 .38913 .72281 .40615 .70322 .39665 .68474 Mtetra
.761 .655 .745 r
.39665 .68474 .40615 .70322 .42308 .68516 .41362 .66737 Mtetra
.757 .659 .754 r
.41362 .66737 .42308 .68516 .43997 .66842 .43054 .65124 Mtetra
.753 .663 .763 r
.43054 .65124 .43997 .66842 .45681 .65282 .44742 .63619 Mtetra
.749 .667 .771 r
.44742 .63619 .45681 .65282 .47364 .63823 .4643 .62209 Mtetra
.745 .671 .779 r
.4643 .62209 .47364 .63823 .49047 .62452 .48118 .6088 Mtetra
.741 .674 .786 r
.48118 .6088 .49047 .62452 .50732 .61157 .49808 .59623 Mtetra
.737 .678 .793 r
.49808 .59623 .50732 .61157 .52418 .5993 .515 .58431 Mtetra
.733 .681 .8 r
.515 .58431 .52418 .5993 .54109 .58763 .53196 .57294 Mtetra
.73 .684 .806 r
.53196 .57294 .54109 .58763 .55803 .57649 .54896 .56207 Mtetra
.726 .687 .812 r
.54896 .56207 .55803 .57649 .57503 .56582 .56602 .55163 Mtetra
.723 .69 .818 r
.56602 .55163 .57503 .56582 .59208 .55556 .58314 .54159 Mtetra
.72 .692 .823 r
.58314 .54159 .59208 .55556 .6092 .54567 .60032 .5319 Mtetra
.717 .695 .828 r
.60032 .5319 .6092 .54567 .62638 .53612 .61758 .52252 Mtetra
.714 .697 .833 r
.61758 .52252 .62638 .53612 .64364 .52685 .6349 .51341 Mtetra
.711 .699 .837 r
.6349 .51341 .64364 .52685 .66098 .51786 .65231 .50455 Mtetra
.709 .701 .841 r
.65231 .50455 .66098 .51786 .6784 .50909 .6698 .4959 Mtetra
.706 .703 .845 r
.6698 .4959 .6784 .50909 .6959 .50054 .68738 .48746 Mtetra
.704 .705 .848 r
.68738 .48746 .6959 .50054 .71349 .49217 .70505 .47919 Mtetra
.702 .707 .852 r
.70505 .47919 .71349 .49217 .73117 .48398 .72281 .47107 Mtetra
.7 .708 .855 r
.72281 .47107 .73117 .48398 .74895 .47593 .74067 .4631 Mtetra
.698 .71 .858 r
.74067 .4631 .74895 .47593 .76683 .46802 .75862 .45525 Mtetra
.696 .711 .86 r
.75862 .45525 .76683 .46802 .7848 .46023 .77667 .44751 Mtetra
.694 .713 .863 r
.77667 .44751 .7848 .46023 .80288 .45254 .79483 .43987 Mtetra
.692 .714 .865 r
.79483 .43987 .80288 .45254 .82106 .44495 .81309 .43232 Mtetra
.691 .715 .868 r
.81309 .43232 .82106 .44495 .83934 .43745 .83146 .42484 Mtetra
.689 .716 .87 r
.83146 .42484 .83934 .43745 .85773 .43002 .84994 .41744 Mtetra
.688 .717 .872 r
.84994 .41744 .85773 .43002 .87623 .42266 .86853 .41009 Mtetra
.686 .718 .873 r
.86853 .41009 .87623 .42266 .89485 .41535 .88723 .40279 Mtetra
.685 .719 .875 r
.88723 .40279 .89485 .41535 .91357 .4081 .90604 .39555 Mtetra
.684 .72 .877 r
.90604 .39555 .91357 .4081 .93241 .40088 .92498 .38833 Mtetra
.843 .773 .784 r
.31818 .74674 .32854 .75511 .34515 .74641 .33551 .72585 Mtetra
.772 .647 .723 r
.33551 .72585 .34515 .74641 .36244 .724 .35282 .7044 Mtetra
.768 .651 .733 r
.35282 .7044 .36244 .724 .3796 .70354 .36999 .68478 Mtetra
.764 .655 .742 r
.36999 .68478 .3796 .70354 .39665 .68474 .38707 .66671 Mtetra
.759 .659 .751 r
.38707 .66671 .39665 .68474 .41362 .66737 .40407 .64999 Mtetra
.755 .663 .76 r
.40407 .64999 .41362 .66737 .43054 .65124 .42103 .63443 Mtetra
.751 .667 .768 r
.42103 .63443 .43054 .65124 .44742 .63619 .43796 .61989 Mtetra
.747 .671 .776 r
.43796 .61989 .44742 .63619 .4643 .62209 .45488 .60623 Mtetra
.743 .674 .784 r
.45488 .60623 .4643 .62209 .48118 .6088 .47181 .59334 Mtetra
.739 .678 .791 r
.47181 .59334 .48118 .6088 .49808 .59623 .48876 .58113 Mtetra
.735 .681 .798 r
.48876 .58113 .49808 .59623 .515 .58431 .50574 .56951 Mtetra
.732 .684 .805 r
.50574 .56951 .515 .58431 .53196 .57294 .52276 .55843 Mtetra
.728 .687 .811 r
.52276 .55843 .53196 .57294 .54896 .56207 .53982 .54781 Mtetra
.725 .69 .817 r
.53982 .54781 .54896 .56207 .56602 .55163 .55694 .5376 Mtetra
.721 .693 .822 r
.55694 .5376 .56602 .55163 .58314 .54159 .57412 .52776 Mtetra
.718 .695 .827 r
.57412 .52776 .58314 .54159 .60032 .5319 .59137 .51824 Mtetra
.715 .697 .832 r
.59137 .51824 .60032 .5319 .61758 .52252 .6087 .50902 Mtetra
.712 .7 .836 r
.6087 .50902 .61758 .52252 .6349 .51341 .62609 .50005 Mtetra
.71 .702 .841 r
.62609 .50005 .6349 .51341 .65231 .50455 .64357 .49131 Mtetra
.707 .704 .845 r
.64357 .49131 .65231 .50455 .6698 .4959 .66114 .48278 Mtetra
.705 .705 .848 r
.66114 .48278 .6698 .4959 .68738 .48746 .67879 .47443 Mtetra
.702 .707 .852 r
.67879 .47443 .68738 .48746 .70505 .47919 .69653 .46624 Mtetra
.7 .709 .855 r
.69653 .46624 .70505 .47919 .72281 .47107 .71437 .4582 Mtetra
.698 .71 .858 r
.71437 .4582 .72281 .47107 .74067 .4631 .7323 .45029 Mtetra
.696 .712 .861 r
.7323 .45029 .74067 .4631 .75862 .45525 .75034 .44249 Mtetra
.694 .713 .863 r
.75034 .44249 .75862 .45525 .77667 .44751 .76847 .4348 Mtetra
.693 .714 .866 r
.76847 .4348 .77667 .44751 .79483 .43987 .78671 .4272 Mtetra
.691 .716 .868 r
.78671 .4272 .79483 .43987 .81309 .43232 .80506 .41967 Mtetra
.689 .717 .87 r
.80506 .41967 .81309 .43232 .83146 .42484 .82351 .41222 Mtetra
.688 .718 .872 r
.82351 .41222 .83146 .42484 .84994 .41744 .84208 .40483 Mtetra
.687 .719 .874 r
.84208 .40483 .84994 .41744 .86853 .41009 .86075 .3975 Mtetra
.685 .72 .876 r
.86075 .3975 .86853 .41009 .88723 .40279 .87954 .39021 Mtetra
.684 .721 .877 r
.87954 .39021 .88723 .40279 .90604 .39555 .89845 .38296 Mtetra
.683 .722 .879 r
.89845 .38296 .90604 .39555 .92498 .38833 .91747 .37574 Mtetra
.789 .667 .729 r
.30831 .72843 .31818 .74674 .33551 .72585 .32579 .70588 Mtetra
.771 .65 .729 r
.32579 .70588 .33551 .72585 .35282 .7044 .34311 .68533 Mtetra
.767 .654 .738 r
.34311 .68533 .35282 .7044 .36999 .68478 .36031 .66648 Mtetra
.762 .659 .748 r
.36031 .66648 .36999 .68478 .38707 .66671 .37741 .6491 Mtetra
.758 .663 .757 r
.37741 .6491 .38707 .66671 .40407 .64999 .39444 .63298 Mtetra
.754 .667 .765 r
.39444 .63298 .40407 .64999 .42103 .63443 .41144 .61796 Mtetra
.749 .67 .774 r
.41144 .61796 .42103 .63443 .43796 .61989 .42841 .60388 Mtetra
.745 .674 .782 r
.42841 .60388 .43796 .61989 .45488 .60623 .44538 .59064 Mtetra
.741 .678 .789 r
.44538 .59064 .45488 .60623 .47181 .59334 .46236 .57812 Mtetra
.737 .681 .796 r
.46236 .57812 .47181 .59334 .48876 .58113 .47936 .56623 Mtetra
.733 .684 .803 r
.47936 .56623 .48876 .58113 .50574 .56951 .4964 .55491 Mtetra
.73 .687 .809 r
.4964 .55491 .50574 .56951 .52276 .55843 .51348 .54409 Mtetra
.726 .69 .815 r
.51348 .54409 .52276 .55843 .53982 .54781 .5306 .5337 Mtetra
.723 .693 .821 r
.5306 .5337 .53982 .54781 .55694 .5376 .54778 .52369 Mtetra
.719 .695 .826 r
.54778 .52369 .55694 .5376 .57412 .52776 .56503 .51404 Mtetra
.716 .698 .831 r
.56503 .51404 .57412 .52776 .59137 .51824 .58235 .50468 Mtetra
.713 .7 .836 r
.58235 .50468 .59137 .51824 .6087 .50902 .59974 .4956 Mtetra
.711 .702 .84 r
.59974 .4956 .6087 .50902 .62609 .50005 .6172 .48676 Mtetra
.708 .704 .844 r
.6172 .48676 .62609 .50005 .64357 .49131 .63476 .47813 Mtetra
.705 .706 .848 r
.63476 .47813 .64357 .49131 .66114 .48278 .65239 .4697 Mtetra
.703 .708 .851 r
.65239 .4697 .66114 .48278 .67879 .47443 .67012 .46144 Mtetra
.701 .709 .855 r
.67012 .46144 .67879 .47443 .69653 .46624 .68794 .45332 Mtetra
.699 .711 .858 r
.68794 .45332 .69653 .46624 .71437 .4582 .70585 .44535 Mtetra
.697 .712 .861 r
.70585 .44535 .71437 .4582 .7323 .45029 .72387 .43749 Mtetra
.695 .714 .863 r
.72387 .43749 .7323 .45029 .75034 .44249 .74198 .42974 Mtetra
.693 .715 .866 r
.74198 .42974 .75034 .44249 .76847 .4348 .7602 .42208 Mtetra
.691 .716 .868 r
.7602 .42208 .76847 .4348 .78671 .4272 .77852 .41451 Mtetra
.69 .717 .87 r
.77852 .41451 .78671 .4272 .80506 .41967 .79695 .40701 Mtetra
.688 .718 .872 r
.79695 .40701 .80506 .41967 .82351 .41222 .81549 .39958 Mtetra
.687 .719 .874 r
.81549 .39958 .82351 .41222 .84208 .40483 .83414 .3922 Mtetra
.686 .72 .876 r
.83414 .3922 .84208 .40483 .86075 .3975 .8529 .38487 Mtetra
.684 .721 .877 r
.8529 .38487 .86075 .3975 .87954 .39021 .87178 .37758 Mtetra
.683 .722 .879 r
.87178 .37758 .87954 .39021 .89845 .38296 .89077 .37032 Mtetra
.682 .723 .88 r
.89077 .37032 .89845 .38296 .91747 .37574 .90989 .3631 Mtetra
.774 .649 .725 r
.2985 .70805 .30831 .72843 .32579 .70588 .31599 .68646 Mtetra
.77 .654 .734 r
.31599 .68646 .32579 .70588 .34311 .68533 .33332 .66674 Mtetra
.765 .658 .744 r
.33332 .66674 .34311 .68533 .36031 .66648 .35054 .64862 Mtetra
.761 .662 .753 r
.35054 .64862 .36031 .66648 .37741 .6491 .36766 .63187 Mtetra
.756 .666 .762 r
.36766 .63187 .37741 .6491 .39444 .63298 .38473 .61632 Mtetra
.752 .67 .771 r
.38473 .61632 .39444 .63298 .41144 .61796 .40177 .60178 Mtetra
.748 .674 .779 r
.40177 .60178 .41144 .61796 .42841 .60388 .41879 .58815 Mtetra
.743 .678 .787 r
.41879 .58815 .42841 .60388 .44538 .59064 .4358 .57529 Mtetra
.739 .681 .794 r
.4358 .57529 .44538 .59064 .46236 .57812 .45283 .56311 Mtetra
.735 .684 .801 r
.45283 .56311 .46236 .57812 .47936 .56623 .46989 .55153 Mtetra
.731 .687 .808 r
.46989 .55153 .47936 .56623 .4964 .55491 .48698 .54048 Mtetra
.728 .69 .814 r
.48698 .54048 .4964 .55491 .51348 .54409 .50411 .52989 Mtetra
.724 .693 .82 r
.50411 .52989 .51348 .54409 .5306 .5337 .5213 .51972 Mtetra
.721 .696 .825 r
.5213 .51972 .5306 .5337 .54778 .52369 .53855 .5099 Mtetra
.718 .698 .831 r
.53855 .5099 .54778 .52369 .56503 .51404 .55586 .50041 Mtetra
.715 .7 .835 r
.55586 .50041 .56503 .51404 .58235 .50468 .57324 .49121 Mtetra
.712 .702 .84 r
.57324 .49121 .58235 .50468 .59974 .4956 .5907 .48226 Mtetra
.709 .704 .844 r
.5907 .48226 .59974 .4956 .6172 .48676 .60823 .47353 Mtetra
.706 .706 .848 r
.60823 .47353 .6172 .48676 .63476 .47813 .62586 .46501 Mtetra
.704 .708 .851 r
.62586 .46501 .63476 .47813 .65239 .4697 .64357 .45666 Mtetra
.701 .71 .855 r
.64357 .45666 .65239 .4697 .67012 .46144 .66137 .44847 Mtetra
.699 .711 .858 r
.66137 .44847 .67012 .46144 .68794 .45332 .67926 .44043 Mtetra
.697 .713 .861 r
.67926 .44043 .68794 .45332 .70585 .44535 .69726 .43251 Mtetra
.695 .714 .863 r
.69726 .43251 .70585 .44535 .72387 .43749 .71535 .42469 Mtetra
.693 .715 .866 r
.71535 .42469 .72387 .43749 .74198 .42974 .73354 .41698 Mtetra
.692 .717 .868 r
.73354 .41698 .74198 .42974 .7602 .42208 .75184 .40936 Mtetra
.69 .718 .87 r
.75184 .40936 .7602 .42208 .77852 .41451 .77024 .40181 Mtetra
.689 .719 .872 r
.77024 .40181 .77852 .41451 .79695 .40701 .78876 .39433 Mtetra
.687 .72 .874 r
.78876 .39433 .79695 .40701 .81549 .39958 .80738 .3869 Mtetra
.686 .721 .876 r
.80738 .3869 .81549 .39958 .83414 .3922 .82612 .37953 Mtetra
.685 .722 .878 r
.82612 .37953 .83414 .3922 .8529 .38487 .84497 .3722 Mtetra
.683 .722 .879 r
.84497 .3722 .8529 .38487 .87178 .37758 .86394 .3649 Mtetra
.682 .723 .881 r
.86394 .3649 .87178 .37758 .89077 .37032 .88303 .35764 Mtetra
.681 .724 .882 r
.88303 .35764 .89077 .37032 .90989 .3631 .90223 .3504 Mtetra
.773 .653 .73 r
.28859 .68824 .2985 .70805 .31599 .68646 .30609 .66754 Mtetra
.768 .657 .74 r
.30609 .66754 .31599 .68646 .33332 .66674 .32344 .64859 Mtetra
.764 .662 .75 r
.32344 .64859 .33332 .66674 .35054 .64862 .34067 .63115 Mtetra
.759 .666 .759 r
.34067 .63115 .35054 .64862 .36766 .63187 .35783 .615 Mtetra
.755 .67 .768 r
.35783 .615 .36766 .63187 .38473 .61632 .37494 .59996 Mtetra
.75 .674 .776 r
.37494 .59996 .38473 .61632 .40177 .60178 .39201 .58589 Mtetra
.746 .677 .784 r
.39201 .58589 .40177 .60178 .41879 .58815 .40907 .57266 Mtetra
.742 .681 .792 r
.40907 .57266 .41879 .58815 .4358 .57529 .42613 .56016 Mtetra
.737 .684 .799 r
.42613 .56016 .4358 .57529 .45283 .56311 .44322 .5483 Mtetra
.733 .687 .806 r
.44322 .5483 .45283 .56311 .46989 .55153 .46032 .537 Mtetra
.73 .69 .813 r
.46032 .537 .46989 .55153 .48698 .54048 .47747 .5262 Mtetra
.726 .693 .819 r
.47747 .5262 .48698 .54048 .50411 .52989 .49466 .51583 Mtetra
.722 .696 .824 r
.49466 .51583 .50411 .52989 .5213 .51972 .51191 .50585 Mtetra
.719 .698 .83 r
.51191 .50585 .5213 .51972 .53855 .5099 .52922 .49621 Mtetra
.716 .701 .834 r
.52922 .49621 .53855 .5099 .55586 .50041 .5466 .48687 Mtetra
.713 .703 .839 r
.5466 .48687 .55586 .50041 .57324 .49121 .56405 .4778 Mtetra
.71 .705 .843 r
.56405 .4778 .57324 .49121 .5907 .48226 .58157 .46897 Mtetra
.707 .707 .847 r
.58157 .46897 .5907 .48226 .60823 .47353 .59918 .46035 Mtetra
.705 .708 .851 r
.59918 .46035 .60823 .47353 .62586 .46501 .61687 .45192 Mtetra
.702 .71 .854 r
.61687 .45192 .62586 .46501 .64357 .45666 .63466 .44365 Mtetra
.7 .712 .858 r
.63466 .44365 .64357 .45666 .66137 .44847 .65253 .43553 Mtetra
.698 .713 .86 r
.65253 .43553 .66137 .44847 .67926 .44043 .67051 .42754 Mtetra
.696 .715 .863 r
.67051 .42754 .67926 .44043 .69726 .43251 .68858 .41966 Mtetra
.694 .716 .866 r
.68858 .41966 .69726 .43251 .71535 .42469 .70675 .41189 Mtetra
.692 .717 .868 r
.70675 .41189 .71535 .42469 .73354 .41698 .72502 .40421 Mtetra
.69 .718 .87 r
.72502 .40421 .73354 .41698 .75184 .40936 .7434 .39661 Mtetra
.689 .719 .873 r
.7434 .39661 .75184 .40936 .77024 .40181 .76189 .38908 Mtetra
.687 .72 .874 r
.76189 .38908 .77024 .40181 .78876 .39433 .78049 .38161 Mtetra
.686 .721 .876 r
.78049 .38161 .78876 .39433 .80738 .3869 .7992 .37419 Mtetra
.685 .722 .878 r
.7992 .37419 .80738 .3869 .82612 .37953 .81802 .36681 Mtetra
.684 .723 .88 r
.81802 .36681 .82612 .37953 .84497 .3722 .83696 .35948 Mtetra
.682 .724 .881 r
.83696 .35948 .84497 .3722 .86394 .3649 .85602 .35217 Mtetra
.681 .724 .882 r
.85602 .35217 .86394 .3649 .88303 .35764 .8752 .34489 Mtetra
.68 .725 .884 r
.8752 .34489 .88303 .35764 .90223 .3504 .8945 .33763 Mtetra
.772 .657 .736 r
.27859 .66896 .28859 .68824 .30609 .66754 .2961 .64909 Mtetra
.767 .661 .746 r
.2961 .64909 .30609 .66754 .32344 .64859 .31346 .63086 Mtetra
.762 .665 .755 r
.31346 .63086 .32344 .64859 .34067 .63115 .33072 .61404 Mtetra
.757 .669 .764 r
.33072 .61404 .34067 .63115 .35783 .615 .34791 .59844 Mtetra
.753 .673 .773 r
.34791 .59844 .35783 .615 .37494 .59996 .36505 .58389 Mtetra
.748 .677 .782 r
.36505 .58389 .37494 .59996 .39201 .58589 .38216 .57025 Mtetra
.744 .681 .79 r
.38216 .57025 .39201 .58589 .40907 .57266 .39927 .55739 Mtetra
.74 .684 .797 r
.39927 .55739 .40907 .57266 .42613 .56016 .41638 .54522 Mtetra
.735 .687 .804 r
.41638 .54522 .42613 .56016 .44322 .5483 .43351 .53366 Mtetra
.731 .69 .811 r
.43351 .53366 .44322 .5483 .46032 .537 .45067 .52262 Mtetra
.728 .693 .817 r
.45067 .52262 .46032 .537 .47747 .5262 .46787 .51205 Mtetra
.724 .696 .823 r
.46787 .51205 .47747 .5262 .49466 .51583 .48513 .50189 Mtetra
.72 .698 .829 r
.48513 .50189 .49466 .51583 .51191 .50585 .50243 .49209 Mtetra
.717 .701 .834 r
.50243 .49209 .51191 .50585 .52922 .49621 .51981 .4826 Mtetra
.714 .703 .838 r
.51981 .4826 .52922 .49621 .5466 .48687 .53725 .47341 Mtetra
.711 .705 .843 r
.53725 .47341 .5466 .48687 .56405 .4778 .55477 .46446 Mtetra
.708 .707 .847 r
.55477 .46446 .56405 .4778 .58157 .46897 .57236 .45573 Mtetra
.705 .709 .851 r
.57236 .45573 .58157 .46897 .59918 .46035 .59004 .4472 Mtetra
.703 .71 .854 r
.59004 .4472 .59918 .46035 .61687 .45192 .60781 .43885 Mtetra
.701 .712 .857 r
.60781 .43885 .61687 .45192 .63466 .44365 .62566 .43065 Mtetra
.698 .714 .86 r
.62566 .43065 .63466 .44365 .65253 .43553 .64361 .42259 Mtetra
.696 .715 .863 r
.64361 .42259 .65253 .43553 .67051 .42754 .66166 .41465 Mtetra
.694 .716 .866 r
.66166 .41465 .67051 .42754 .68858 .41966 .67981 .40682 Mtetra
.693 .717 .868 r
.67981 .40682 .68858 .41966 .70675 .41189 .69806 .39908 Mtetra
.691 .719 .871 r
.69806 .39908 .70675 .41189 .72502 .40421 .71642 .39142 Mtetra
.689 .72 .873 r
.71642 .39142 .72502 .40421 .7434 .39661 .73488 .38384 Mtetra
.688 .721 .875 r
.73488 .38384 .7434 .39661 .76189 .38908 .75345 .37631 Mtetra
.686 .722 .876 r
.75345 .37631 .76189 .38908 .78049 .38161 .77213 .36885 Mtetra
.685 .722 .878 r
.77213 .36885 .78049 .38161 .7992 .37419 .79093 .36143 Mtetra
.684 .723 .88 r
.79093 .36143 .7992 .37419 .81802 .36681 .80985 .35405 Mtetra
.683 .724 .881 r
.80985 .35405 .81802 .36681 .83696 .35948 .82888 .3467 Mtetra
.681 .725 .883 r
.82888 .3467 .83696 .35948 .85602 .35217 .84803 .33938 Mtetra
.68 .726 .884 r
.84803 .33938 .85602 .35217 .8752 .34489 .8673 .33208 Mtetra
.68 .726 .885 r
.8673 .33208 .8752 .34489 .8945 .33763 .88669 .3248 Mtetra
.77 .66 .741 r
.2685 .65016 .27859 .66896 .2961 .64909 .28601 .63106 Mtetra
.765 .664 .751 r
.28601 .63106 .2961 .64909 .31346 .63086 .30339 .6135 Mtetra
.76 .669 .761 r
.30339 .6135 .31346 .63086 .33072 .61404 .32067 .59727 Mtetra
.756 .673 .77 r
.32067 .59727 .33072 .61404 .34791 .59844 .33789 .58218 Mtetra
.751 .677 .779 r
.33789 .58218 .34791 .59844 .36505 .58389 .35506 .56808 Mtetra
.746 .68 .787 r
.35506 .56808 .36505 .58389 .38216 .57025 .37222 .55483 Mtetra
.742 .684 .795 r
.37222 .55483 .38216 .57025 .39927 .55739 .38937 .54232 Mtetra
.738 .687 .802 r
.38937 .54232 .39927 .55739 .41638 .54522 .40653 .53046 Mtetra
.734 .69 .809 r
.40653 .53046 .41638 .54522 .43351 .53366 .42371 .51917 Mtetra
.73 .693 .816 r
.42371 .51917 .43351 .53366 .45067 .52262 .44092 .50838 Mtetra
.726 .696 .822 r
.44092 .50838 .45067 .52262 .46787 .51205 .45818 .49802 Mtetra
.722 .699 .827 r
.45818 .49802 .46787 .51205 .48513 .50189 .4755 .48804 Mtetra
.719 .701 .833 r
.4755 .48804 .48513 .50189 .50243 .49209 .49287 .4784 Mtetra
.715 .703 .837 r
.49287 .4784 .50243 .49209 .51981 .4826 .5103 .46907 Mtetra
.712 .705 .842 r
.5103 .46907 .51981 .4826 .53725 .47341 .52781 .46 Mtetra
.709 .707 .846 r
.52781 .46 .53725 .47341 .55477 .46446 .54539 .45116 Mtetra
.706 .709 .85 r
.54539 .45116 .55477 .46446 .57236 .45573 .56306 .44253 Mtetra
.704 .711 .854 r
.56306 .44253 .57236 .45573 .59004 .4472 .58081 .43409 Mtetra
.701 .712 .857 r
.58081 .43409 .59004 .4472 .60781 .43885 .59865 .4258 Mtetra
.699 .714 .86 r
.59865 .4258 .60781 .43885 .62566 .43065 .61658 .41766 Mtetra
.697 .715 .863 r
.61658 .41766 .62566 .43065 .64361 .42259 .6346 .40965 Mtetra
.695 .717 .866 r
.6346 .40965 .64361 .42259 .66166 .41465 .65273 .40175 Mtetra
.693 .718 .868 r
.65273 .40175 .66166 .41465 .67981 .40682 .67096 .39395 Mtetra
.691 .719 .871 r
.67096 .39395 .67981 .40682 .69806 .39908 .68929 .38624 Mtetra
.69 .72 .873 r
.68929 .38624 .69806 .39908 .71642 .39142 .70772 .3786 Mtetra
.688 .721 .875 r
.70772 .3786 .71642 .39142 .73488 .38384 .72627 .37102 Mtetra
.687 .722 .877 r
.72627 .37102 .73488 .38384 .75345 .37631 .74493 .36351 Mtetra
.685 .723 .878 r
.74493 .36351 .75345 .37631 .77213 .36885 .7637 .35604 Mtetra
.684 .724 .88 r
.7637 .35604 .77213 .36885 .79093 .36143 .78258 .34861 Mtetra
.683 .724 .881 r
.78258 .34861 .79093 .36143 .80985 .35405 .80158 .34122 Mtetra
.682 .725 .883 r
.80158 .34122 .80985 .35405 .82888 .3467 .8207 .33386 Mtetra
.681 .726 .884 r
.8207 .33386 .82888 .3467 .84803 .33938 .83994 .32652 Mtetra
.68 .727 .885 r
.83994 .32652 .84803 .33938 .8673 .33208 .85931 .31919 Mtetra
.679 .727 .886 r
.85931 .31919 .8673 .33208 .88669 .3248 .87879 .31188 Mtetra
.769 .664 .747 r
.25829 .63181 .2685 .65016 .28601 .63106 .27581 .61342 Mtetra
.764 .668 .757 r
.27581 .61342 .28601 .63106 .30339 .6135 .29321 .59649 Mtetra
.759 .672 .766 r
.29321 .59649 .30339 .6135 .32067 .59727 .31052 .5808 Mtetra
.754 .676 .775 r
.31052 .5808 .32067 .59727 .33789 .58218 .32777 .56619 Mtetra
.749 .68 .784 r
.32777 .56619 .33789 .58218 .35506 .56808 .34498 .55251 Mtetra
.745 .684 .792 r
.34498 .55251 .35506 .56808 .37222 .55483 .36218 .53962 Mtetra
.74 .687 .8 r
.36218 .53962 .37222 .55483 .38937 .54232 .37937 .52744 Mtetra
.736 .69 .807 r
.37937 .52744 .38937 .54232 .40653 .53046 .39658 .51587 Mtetra
.732 .693 .814 r
.39658 .51587 .40653 .53046 .42371 .51917 .41381 .50482 Mtetra
.728 .696 .82 r
.41381 .50482 .42371 .51917 .44092 .50838 .43108 .49425 Mtetra
.724 .699 .826 r
.43108 .49425 .44092 .50838 .45818 .49802 .4484 .48408 Mtetra
.72 .701 .831 r
.4484 .48408 .45818 .49802 .4755 .48804 .46577 .47428 Mtetra
.717 .703 .836 r
.46577 .47428 .4755 .48804 .49287 .4784 .4832 .46479 Mtetra
.713 .706 .841 r
.4832 .46479 .49287 .4784 .5103 .46907 .5007 .45559 Mtetra
.71 .708 .845 r
.5007 .45559 .5103 .46907 .52781 .46 .51828 .44663 Mtetra
.707 .709 .85 r
.51828 .44663 .52781 .46 .54539 .45116 .53593 .4379 Mtetra
.705 .711 .853 r
.53593 .4379 .54539 .45116 .56306 .44253 .55366 .42935 Mtetra
.702 .713 .857 r
.55366 .42935 .56306 .44253 .58081 .43409 .57148 .42098 Mtetra
.7 .714 .86 r
.57148 .42098 .58081 .43409 .59865 .4258 .58939 .41276 Mtetra
.698 .716 .863 r
.58939 .41276 .59865 .4258 .61658 .41766 .6074 .40467 Mtetra
.695 .717 .866 r
.6074 .40467 .61658 .41766 .6346 .40965 .6255 .3967 Mtetra
.694 .718 .868 r
.6255 .3967 .6346 .40965 .65273 .40175 .64371 .38883 Mtetra
.692 .719 .871 r
.64371 .38883 .65273 .40175 .67096 .39395 .66201 .38106 Mtetra
.69 .72 .873 r
.66201 .38106 .67096 .39395 .68929 .38624 .68042 .37336 Mtetra
.688 .721 .875 r
.68042 .37336 .68929 .38624 .70772 .3786 .69894 .36573 Mtetra
.687 .722 .877 r
.69894 .36573 .70772 .3786 .72627 .37102 .71757 .35817 Mtetra
.685 .723 .878 r
.71757 .35817 .72627 .37102 .74493 .36351 .73631 .35065 Mtetra
.684 .724 .88 r
.73631 .35065 .74493 .36351 .7637 .35604 .75517 .34317 Mtetra
.683 .725 .882 r
.75517 .34317 .7637 .35604 .78258 .34861 .77414 .33574 Mtetra
.682 .726 .883 r
.77414 .33574 .78258 .34861 .80158 .34122 .79323 .32833 Mtetra
.681 .726 .884 r
.79323 .32833 .80158 .34122 .8207 .33386 .81244 .32094 Mtetra
.68 .727 .886 r
.81244 .32094 .8207 .33386 .83994 .32652 .83177 .31358 Mtetra
.679 .728 .887 r
.83177 .31358 .83994 .32652 .85931 .31919 .85123 .30623 Mtetra
.678 .728 .888 r
.85123 .30623 .85931 .31919 .87879 .31188 .87081 .29889 Mtetra
.767 .667 .752 r
.24798 .61386 .25829 .63181 .27581 .61342 .26552 .59614 Mtetra
.762 .671 .762 r
.26552 .59614 .27581 .61342 .29321 .59649 .28293 .57978 Mtetra
.757 .676 .772 r
.28293 .57978 .29321 .59649 .31052 .5808 .30027 .56461 Mtetra
.752 .679 .781 r
.30027 .56461 .31052 .5808 .32777 .56619 .31755 .55044 Mtetra
.747 .683 .789 r
.31755 .55044 .32777 .56619 .34498 .55251 .3348 .53714 Mtetra
.743 .687 .797 r
.3348 .53714 .34498 .55251 .36218 .53962 .35203 .5246 Mtetra
.738 .69 .805 r
.35203 .5246 .36218 .53962 .37937 .52744 .36927 .51272 Mtetra
.734 .693 .812 r
.36927 .51272 .37937 .52744 .39658 .51587 .38653 .50141 Mtetra
.73 .696 .818 r
.38653 .50141 .39658 .51587 .41381 .50482 .40382 .4906 Mtetra
.726 .699 .824 r
.40382 .4906 .41381 .50482 .43108 .49425 .42114 .48023 Mtetra
.722 .701 .83 r
.42114 .48023 .43108 .49425 .4484 .48408 .43852 .47024 Mtetra
.718 .704 .835 r
.43852 .47024 .4484 .48408 .46577 .47428 .45595 .46059 Mtetra
.715 .706 .84 r
.45595 .46059 .46577 .47428 .4832 .46479 .47344 .45124 Mtetra
.712 .708 .845 r
.47344 .45124 .4832 .46479 .5007 .45559 .49101 .44216 Mtetra
.709 .71 .849 r
.49101 .44216 .5007 .45559 .51828 .44663 .50865 .4333 Mtetra
.706 .711 .853 r
.50865 .4333 .51828 .44663 .53593 .4379 .52637 .42466 Mtetra
.703 .713 .856 r
.52637 .42466 .53593 .4379 .55366 .42935 .54417 .41619 Mtetra
.701 .715 .86 r
.54417 .41619 .55366 .42935 .57148 .42098 .56206 .40788 Mtetra
.698 .716 .863 r
.56206 .40788 .57148 .42098 .58939 .41276 .58005 .39971 Mtetra
.696 .717 .865 r
.58005 .39971 .58939 .41276 .6074 .40467 .59813 .39167 Mtetra
.694 .719 .868 r
.59813 .39167 .6074 .40467 .6255 .3967 .61631 .38373 Mtetra
.692 .72 .87 r
.61631 .38373 .6255 .3967 .64371 .38883 .63459 .37589 Mtetra
.69 .721 .873 r
.63459 .37589 .64371 .38883 .66201 .38106 .65297 .36813 Mtetra
.689 .722 .875 r
.65297 .36813 .66201 .38106 .68042 .37336 .67146 .36045 Mtetra
.687 .723 .877 r
.67146 .36045 .68042 .37336 .69894 .36573 .69007 .35282 Mtetra
.686 .724 .878 r
.69007 .35282 .69894 .36573 .71757 .35817 .70878 .34526 Mtetra
.684 .724 .88 r
.70878 .34526 .71757 .35817 .73631 .35065 .72761 .33773 Mtetra
.683 .725 .882 r
.72761 .33773 .73631 .35065 .75517 .34317 .74655 .33025 Mtetra
.682 .726 .883 r
.74655 .33025 .75517 .34317 .77414 .33574 .76561 .32279 Mtetra
.681 .727 .884 r
.76561 .32279 .77414 .33574 .79323 .32833 .78479 .31536 Mtetra
.68 .727 .886 r
.78479 .31536 .79323 .32833 .81244 .32094 .80409 .30795 Mtetra
.679 .728 .887 r
.80409 .30795 .81244 .32094 .83177 .31358 .82352 .30056 Mtetra
.678 .728 .888 r
.82352 .30056 .83177 .31358 .85123 .30623 .84307 .29318 Mtetra
.677 .729 .889 r
.84307 .29318 .85123 .30623 .87081 .29889 .86275 .2858 Mtetra
.765 .671 .758 r
.23756 .59628 .24798 .61386 .26552 .59614 .25511 .57918 Mtetra
.76 .675 .768 r
.25511 .57918 .26552 .59614 .28293 .57978 .27255 .56337 Mtetra
.755 .679 .777 r
.27255 .56337 .28293 .57978 .30027 .56461 .28991 .54866 Mtetra
.75 .683 .786 r
.28991 .54866 .30027 .56461 .31755 .55044 .30722 .53491 Mtetra
.745 .686 .794 r
.30722 .53491 .31755 .55044 .3348 .53714 .32451 .52197 Mtetra
.741 .69 .802 r
.32451 .52197 .3348 .53714 .35203 .5246 .34179 .50975 Mtetra
.736 .693 .81 r
.34179 .50975 .35203 .5246 .36927 .51272 .35907 .49814 Mtetra
.732 .696 .816 r
.35907 .49814 .36927 .51272 .38653 .50141 .37638 .48707 Mtetra
.728 .699 .823 r
.37638 .48707 .38653 .50141 .40382 .4906 .39372 .47648 Mtetra
.724 .701 .829 r
.39372 .47648 .40382 .4906 .42114 .48023 .4111 .46629 Mtetra
.72 .704 .834 r
.4111 .46629 .42114 .48023 .43852 .47024 .42853 .45647 Mtetra
.716 .706 .839 r
.42853 .45647 .43852 .47024 .45595 .46059 .44603 .44696 Mtetra
.713 .708 .844 r
.44603 .44696 .45595 .46059 .47344 .45124 .46358 .43774 Mtetra
.71 .71 .848 r
.46358 .43774 .47344 .45124 .49101 .44216 .48121 .42876 Mtetra
.707 .712 .852 r
.48121 .42876 .49101 .44216 .50865 .4333 .49892 .42 Mtetra
.704 .713 .856 r
.49892 .42 .50865 .4333 .52637 .42466 .51671 .41143 Mtetra
.702 .715 .859 r
.51671 .41143 .52637 .42466 .54417 .41619 .53458 .40303 Mtetra
.699 .716 .862 r
.53458 .40303 .54417 .41619 .56206 .40788 .55254 .39477 Mtetra
.697 .718 .865 r
.55254 .39477 .56206 .40788 .58005 .39971 .5706 .38665 Mtetra
.695 .719 .868 r
.5706 .38665 .58005 .39971 .59813 .39167 .58876 .37864 Mtetra
.693 .72 .87 r
.58876 .37864 .59813 .39167 .61631 .38373 .60701 .37073 Mtetra
.691 .721 .873 r
.60701 .37073 .61631 .38373 .63459 .37589 .62537 .36291 Mtetra
.689 .722 .875 r
.62537 .36291 .63459 .37589 .65297 .36813 .64384 .35516 Mtetra
.688 .723 .877 r
.64384 .35516 .65297 .36813 .67146 .36045 .66241 .34749 Mtetra
.686 .724 .879 r
.66241 .34749 .67146 .36045 .69007 .35282 .6811 .33986 Mtetra
.685 .725 .88 r
.6811 .33986 .69007 .35282 .70878 .34526 .69989 .33229 Mtetra
.683 .726 .882 r
.69989 .33229 .70878 .34526 .72761 .33773 .71881 .32475 Mtetra
.682 .726 .883 r
.71881 .32475 .72761 .33773 .74655 .33025 .73783 .31725 Mtetra
.681 .727 .885 r
.73783 .31725 .74655 .33025 .76561 .32279 .75698 .30977 Mtetra
.68 .728 .886 r
.75698 .30977 .76561 .32279 .78479 .31536 .77626 .30232 Mtetra
.679 .728 .887 r
.77626 .30232 .78479 .31536 .80409 .30795 .79565 .29488 Mtetra
.678 .729 .888 r
.79565 .29488 .80409 .30795 .82352 .30056 .81517 .28745 Mtetra
.677 .729 .889 r
.81517 .28745 .82352 .30056 .84307 .29318 .83481 .28003 Mtetra
.676 .73 .89 r
.83481 .28003 .84307 .29318 .86275 .2858 .85459 .27261 Mtetra
.764 .674 .764 r
.22703 .57905 .23756 .59628 .25511 .57918 .24459 .56252 Mtetra
.758 .678 .773 r
.24459 .56252 .25511 .57918 .27255 .56337 .26205 .54722 Mtetra
.753 .682 .783 r
.26205 .54722 .27255 .56337 .28991 .54866 .27944 .53295 Mtetra
.748 .686 .791 r
.27944 .53295 .28991 .54866 .30722 .53491 .29679 .51957 Mtetra
.743 .689 .799 r
.29679 .51957 .30722 .53491 .32451 .52197 .31411 .50697 Mtetra
.739 .693 .807 r
.31411 .50697 .32451 .52197 .34179 .50975 .33144 .49504 Mtetra
.734 .696 .814 r
.33144 .49504 .34179 .50975 .35907 .49814 .34877 .48369 Mtetra
.73 .699 .821 r
.34877 .48369 .35907 .49814 .37638 .48707 .36613 .47284 Mtetra
.726 .701 .827 r
.36613 .47284 .37638 .48707 .39372 .47648 .38352 .46244 Mtetra
.722 .704 .833 r
.38352 .46244 .39372 .47648 .4111 .46629 .40095 .45243 Mtetra
.718 .706 .838 r
.40095 .45243 .4111 .46629 .42853 .45647 .41845 .44276 Mtetra
.714 .708 .843 r
.41845 .44276 .42853 .45647 .44603 .44696 .436 .43338 Mtetra
.711 .71 .847 r
.436 .43338 .44603 .44696 .46358 .43774 .45362 .42427 Mtetra
.708 .712 .851 r
.45362 .42427 .46358 .43774 .48121 .42876 .47131 .41538 Mtetra
.705 .714 .855 r
.47131 .41538 .48121 .42876 .49892 .42 .48909 .4067 Mtetra
.702 .715 .859 r
.48909 .4067 .49892 .42 .51671 .41143 .50694 .3982 Mtetra
.7 .717 .862 r
.50694 .3982 .51671 .41143 .53458 .40303 .52489 .38986 Mtetra
.698 .718 .865 r
.52489 .38986 .53458 .40303 .55254 .39477 .54292 .38165 Mtetra
.695 .719 .868 r
.54292 .38165 .55254 .39477 .5706 .38665 .56106 .37356 Mtetra
.693 .72 .87 r
.56106 .37356 .5706 .38665 .58876 .37864 .57929 .36558 Mtetra
.691 .721 .873 r
.57929 .36558 .58876 .37864 .60701 .37073 .59762 .35769 Mtetra
.69 .722 .875 r
.59762 .35769 .60701 .37073 .62537 .36291 .61606 .34989 Mtetra
.688 .723 .877 r
.61606 .34989 .62537 .36291 .64384 .35516 .6346 .34215 Mtetra
.686 .724 .879 r
.6346 .34215 .64384 .35516 .66241 .34749 .65326 .33447 Mtetra
.685 .725 .88 r
.65326 .33447 .66241 .34749 .6811 .33986 .67203 .32684 Mtetra
.684 .726 .882 r
.67203 .32684 .6811 .33986 .69989 .33229 .69091 .31925 Mtetra
.682 .727 .883 r
.69091 .31925 .69989 .33229 .71881 .32475 .70991 .3117 Mtetra
.681 .727 .885 r
.70991 .3117 .71881 .32475 .73783 .31725 .72903 .30417 Mtetra
.68 .728 .886 r
.72903 .30417 .73783 .31725 .75698 .30977 .74826 .29667 Mtetra
.679 .728 .887 r
.74826 .29667 .75698 .30977 .77626 .30232 .76763 .28919 Mtetra
.678 .729 .888 r
.76763 .28919 .77626 .30232 .79565 .29488 .78711 .28172 Mtetra
.677 .73 .889 r
.78711 .28172 .79565 .29488 .81517 .28745 .80673 .27425 Mtetra
.677 .73 .89 r
.80673 .27425 .81517 .28745 .83481 .28003 .82647 .26679 Mtetra
.676 .731 .891 r
.82647 .26679 .83481 .28003 .85459 .27261 .84634 .25933 Mtetra
.762 .677 .769 r
.21638 .56212 .22703 .57905 .24459 .56252 .23396 .54614 Mtetra
.756 .681 .779 r
.23396 .54614 .24459 .56252 .26205 .54722 .25144 .5313 Mtetra
.751 .685 .788 r
.25144 .5313 .26205 .54722 .27944 .53295 .26886 .51744 Mtetra
.746 .689 .796 r
.26886 .51744 .27944 .53295 .29679 .51957 .28624 .50442 Mtetra
.741 .692 .804 r
.28624 .50442 .29679 .51957 .31411 .50697 .30361 .49212 Mtetra
.737 .695 .812 r
.30361 .49212 .31411 .50697 .33144 .49504 .32097 .48046 Mtetra
.732 .698 .819 r
.32097 .48046 .33144 .49504 .34877 .48369 .33835 .46935 Mtetra
.728 .701 .825 r
.33835 .46935 .34877 .48369 .36613 .47284 .35576 .45871 Mtetra
.724 .704 .831 r
.35576 .45871 .36613 .47284 .38352 .46244 .37321 .44849 Mtetra
.72 .706 .836 r
.37321 .44849 .38352 .46244 .40095 .45243 .3907 .43863 Mtetra
.716 .708 .842 r
.3907 .43863 .40095 .45243 .41845 .44276 .40825 .42909 Mtetra
.713 .71 .846 r
.40825 .42909 .41845 .44276 .436 .43338 .42586 .41983 Mtetra
.709 .712 .85 r
.42586 .41983 .436 .43338 .45362 .42427 .44355 .41082 Mtetra
.706 .714 .854 r
.44355 .41082 .45362 .42427 .47131 .41538 .46131 .40202 Mtetra
.704 .715 .858 r
.46131 .40202 .47131 .41538 .48909 .4067 .47915 .39341 Mtetra
.701 .717 .861 r
.47915 .39341 .48909 .4067 .50694 .3982 .49707 .38497 Mtetra
.698 .718 .864 r
.49707 .38497 .50694 .3982 .52489 .38986 .51509 .37667 Mtetra
.696 .719 .867 r
.51509 .37667 .52489 .38986 .54292 .38165 .5332 .3685 Mtetra
.694 .721 .87 r
.5332 .3685 .54292 .38165 .56106 .37356 .55141 .36045 Mtetra
.692 .722 .872 r
.55141 .36045 .56106 .37356 .57929 .36558 .56971 .35249 Mtetra
.69 .723 .875 r
.56971 .35249 .57929 .36558 .59762 .35769 .58812 .34461 Mtetra
.688 .724 .877 r
.58812 .34461 .59762 .35769 .61606 .34989 .60664 .33681 Mtetra
.687 .725 .879 r
.60664 .33681 .61606 .34989 .6346 .34215 .62527 .32907 Mtetra
.685 .725 .88 r
.62527 .32907 .6346 .34215 .65326 .33447 .64401 .32139 Mtetra
.684 .726 .882 r
.64401 .32139 .65326 .33447 .67203 .32684 .66286 .31374 Mtetra
.683 .727 .883 r
.66286 .31374 .67203 .32684 .69091 .31925 .68182 .30614 Mtetra
.682 .728 .885 r
.68182 .30614 .69091 .31925 .70991 .3117 .70091 .29857 Mtetra
.68 .728 .886 r
.70091 .29857 .70991 .3117 .72903 .30417 .72012 .29102 Mtetra
.679 .729 .887 r
.72012 .29102 .72903 .30417 .74826 .29667 .73945 .28349 Mtetra
.678 .729 .888 r
.73945 .28349 .74826 .29667 .76763 .28919 .7589 .27597 Mtetra
.678 .73 .89 r
.7589 .27597 .76763 .28919 .78711 .28172 .77848 .26846 Mtetra
.677 .73 .891 r
.77848 .26846 .78711 .28172 .80673 .27425 .79819 .26095 Mtetra
.676 .731 .891 r
.79819 .26095 .80673 .27425 .82647 .26679 .81802 .25344 Mtetra
.675 .731 .892 r
.81802 .25344 .82647 .26679 .84634 .25933 .83799 .24593 Mtetra
.76 .681 .774 r
.20561 .54548 .21638 .56212 .23396 .54614 .2232 .53 Mtetra
.755 .685 .784 r
.2232 .53 .23396 .54614 .25144 .5313 .24071 .51559 Mtetra
.749 .688 .793 r
.24071 .51559 .25144 .5313 .26886 .51744 .25816 .50211 Mtetra
.744 .692 .801 r
.25816 .50211 .26886 .51744 .28624 .50442 .27558 .48942 Mtetra
.739 .695 .809 r
.27558 .48942 .28624 .50442 .30361 .49212 .29299 .47741 Mtetra
.734 .698 .816 r
.29299 .47741 .30361 .49212 .32097 .48046 .3104 .466 Mtetra
.73 .701 .823 r
.3104 .466 .32097 .48046 .33835 .46935 .32783 .4551 Mtetra
.726 .704 .829 r
.32783 .4551 .33835 .46935 .35576 .45871 .34528 .44465 Mtetra
.722 .706 .835 r
.34528 .44465 .35576 .45871 .37321 .44849 .36278 .43459 Mtetra
.718 .708 .84 r
.36278 .43459 .37321 .44849 .3907 .43863 .38033 .42487 Mtetra
.714 .71 .845 r
.38033 .42487 .3907 .43863 .40825 .42909 .39794 .41546 Mtetra
.711 .712 .849 r
.39794 .41546 .40825 .42909 .42586 .41983 .41562 .4063 Mtetra
.708 .714 .854 r
.41562 .4063 .42586 .41983 .44355 .41082 .43337 .39738 Mtetra
.705 .716 .857 r
.43337 .39738 .44355 .41082 .46131 .40202 .45119 .38866 Mtetra
.702 .717 .861 r
.45119 .38866 .46131 .40202 .47915 .39341 .4691 .38011 Mtetra
.699 .718 .864 r
.4691 .38011 .47915 .39341 .49707 .38497 .4871 .37172 Mtetra
.697 .72 .867 r
.4871 .37172 .49707 .38497 .51509 .37667 .50518 .36346 Mtetra
.695 .721 .87 r
.50518 .36346 .51509 .37667 .5332 .3685 .52337 .35533 Mtetra
.693 .722 .872 r
.52337 .35533 .5332 .3685 .55141 .36045 .54165 .34729 Mtetra
.691 .723 .874 r
.54165 .34729 .55141 .36045 .56971 .35249 .56003 .33935 Mtetra
.689 .724 .877 r
.56003 .33935 .56971 .35249 .58812 .34461 .57852 .33148 Mtetra
.687 .725 .878 r
.57852 .33148 .58812 .34461 .60664 .33681 .59712 .32368 Mtetra
.686 .726 .88 r
.59712 .32368 .60664 .33681 .62527 .32907 .61583 .31593 Mtetra
.684 .726 .882 r
.61583 .31593 .62527 .32907 .64401 .32139 .63465 .30823 Mtetra
.683 .727 .883 r
.63465 .30823 .64401 .32139 .66286 .31374 .65358 .30058 Mtetra
.682 .728 .885 r
.65358 .30058 .66286 .31374 .68182 .30614 .67264 .29295 Mtetra
.681 .728 .886 r
.67264 .29295 .68182 .30614 .70091 .29857 .69181 .28535 Mtetra
.68 .729 .887 r
.69181 .28535 .70091 .29857 .72012 .29102 .71111 .27777 Mtetra
.679 .73 .889 r
.71111 .27777 .72012 .29102 .73945 .28349 .73053 .27021 Mtetra
.678 .73 .89 r
.73053 .27021 .73945 .28349 .7589 .27597 .75007 .26265 Mtetra
.677 .731 .891 r
.75007 .26265 .7589 .27597 .77848 .26846 .76975 .2551 Mtetra
.676 .731 .892 r
.76975 .2551 .77848 .26846 .79819 .26095 .78955 .24754 Mtetra
.675 .732 .892 r
.78955 .24754 .79819 .26095 .81802 .25344 .80948 .23999 Mtetra
.675 .732 .893 r
.80948 .23999 .81802 .25344 .83799 .24593 .82955 .23242 Mtetra
.758 .684 .78 r
.19472 .5291 .20561 .54548 .2232 .53 .21233 .51408 Mtetra
.753 .688 .789 r
.21233 .51408 .2232 .53 .24071 .51559 .22986 .50008 Mtetra
.747 .691 .798 r
.22986 .50008 .24071 .51559 .25816 .50211 .24735 .48694 Mtetra
.742 .695 .806 r
.24735 .48694 .25816 .50211 .27558 .48942 .2648 .47455 Mtetra
.737 .698 .814 r
.2648 .47455 .27558 .48942 .29299 .47741 .28225 .46281 Mtetra
.732 .701 .821 r
.28225 .46281 .29299 .47741 .3104 .466 .29971 .45163 Mtetra
.728 .703 .827 r
.29971 .45163 .3104 .466 .32783 .4551 .31718 .44093 Mtetra
.724 .706 .833 r
.31718 .44093 .32783 .4551 .34528 .44465 .33469 .43065 Mtetra
.72 .708 .839 r
.33469 .43065 .34528 .44465 .36278 .43459 .35225 .42074 Mtetra
.716 .71 .844 r
.35225 .42074 .36278 .43459 .38033 .42487 .36986 .41116 Mtetra
.712 .712 .848 r
.36986 .41116 .38033 .42487 .39794 .41546 .38752 .40185 Mtetra
.709 .714 .853 r
.38752 .40185 .39794 .41546 .41562 .4063 .40526 .39279 Mtetra
.706 .716 .857 r
.40526 .39279 .41562 .4063 .43337 .39738 .42307 .38394 Mtetra
.703 .717 .86 r
.42307 .38394 .43337 .39738 .45119 .38866 .44097 .37528 Mtetra
.7 .719 .863 r
.44097 .37528 .45119 .38866 .4691 .38011 .45894 .36679 Mtetra
.698 .72 .866 r
.45894 .36679 .4691 .38011 .4871 .37172 .47701 .35844 Mtetra
.696 .721 .869 r
.47701 .35844 .4871 .37172 .50518 .36346 .49517 .35022 Mtetra
.693 .722 .872 r
.49517 .35022 .50518 .36346 .52337 .35533 .51343 .34211 Mtetra
.691 .723 .874 r
.51343 .34211 .52337 .35533 .54165 .34729 .53178 .33409 Mtetra
.69 .724 .876 r
.53178 .33409 .54165 .34729 .56003 .33935 .55024 .32615 Mtetra
.688 .725 .878 r
.55024 .32615 .56003 .33935 .57852 .33148 .56881 .31828 Mtetra
.686 .726 .88 r
.56881 .31828 .57852 .33148 .59712 .32368 .58749 .31048 Mtetra
.685 .727 .882 r
.58749 .31048 .59712 .32368 .61583 .31593 .60628 .30272 Mtetra
.683 .727 .883 r
.60628 .30272 .61583 .31593 .63465 .30823 .62518 .29501 Mtetra
.682 .728 .885 r
.62518 .29501 .63465 .30823 .65358 .30058 .6442 .28733 Mtetra
.681 .729 .886 r
.6442 .28733 .65358 .30058 .67264 .29295 .66334 .27968 Mtetra
.68 .729 .888 r
.66334 .27968 .67264 .29295 .69181 .28535 .68261 .27204 Mtetra
.679 .73 .889 r
.68261 .27204 .69181 .28535 .71111 .27777 .70199 .26443 Mtetra
.678 .73 .89 r
.70199 .26443 .71111 .27777 .73053 .27021 .7215 .25682 Mtetra
.677 .731 .891 r
.7215 .25682 .73053 .27021 .75007 .26265 .74114 .24922 Mtetra
.676 .731 .892 r
.74114 .24922 .75007 .26265 .76975 .2551 .76091 .24163 Mtetra
.675 .732 .893 r
.76091 .24163 .76975 .2551 .78955 .24754 .78081 .23402 Mtetra
.78081 .23402 .78955 .24754 .80948 .23999 .80084 .22642 Mtetra
.674 .733 .894 r
.80084 .22642 .80948 .23999 .82955 .23242 .82101 .2188 Mtetra
.756 .687 .785 r
.1837 .51295 .19472 .5291 .21233 .51408 .20133 .49836 Mtetra
.75 .691 .794 r
.20133 .49836 .21233 .51408 .22986 .50008 .21889 .48473 Mtetra
.745 .694 .803 r
.21889 .48473 .22986 .50008 .24735 .48694 .23641 .47192 Mtetra
.74 .697 .811 r
.23641 .47192 .24735 .48694 .2648 .47455 .2539 .45981 Mtetra
.735 .7 .818 r
.2539 .45981 .2648 .47455 .28225 .46281 .27139 .44831 Mtetra
.73 .703 .825 r
.27139 .44831 .28225 .46281 .29971 .45163 .28889 .43734 Mtetra
.726 .706 .831 r
.28889 .43734 .29971 .45163 .31718 .44093 .30642 .42682 Mtetra
.722 .708 .837 r
.30642 .42682 .31718 .44093 .33469 .43065 .32398 .41671 Mtetra
.718 .71 .842 r
.32398 .41671 .33469 .43065 .35225 .42074 .34159 .40693 Mtetra
.714 .712 .847 r
.34159 .40693 .35225 .42074 .36986 .41116 .35926 .39746 Mtetra
.711 .714 .852 r
.35926 .39746 .36986 .41116 .38752 .40185 .37699 .38825 Mtetra
.707 .716 .856 r
.37699 .38825 .38752 .40185 .40526 .39279 .39479 .37927 Mtetra
.704 .717 .859 r
.39479 .37927 .40526 .39279 .42307 .38394 .41267 .3705 Mtetra
.701 .719 .863 r
.41267 .3705 .42307 .38394 .44097 .37528 .43063 .3619 Mtetra
.699 .72 .866 r
.43063 .3619 .44097 .37528 .45894 .36679 .44867 .35345 Mtetra
.696 .721 .869 r
.44867 .35345 .45894 .36679 .47701 .35844 .46681 .34513 Mtetra
.694 .722 .871 r
.46681 .34513 .47701 .35844 .49517 .35022 .48504 .33694 Mtetra
.692 .724 .874 r
.48504 .33694 .49517 .35022 .51343 .34211 .50337 .32884 Mtetra
.69 .724 .876 r
.50337 .32884 .51343 .34211 .53178 .33409 .5218 .32083 Mtetra
.688 .725 .878 r
.5218 .32083 .53178 .33409 .55024 .32615 .54034 .31289 Mtetra
.687 .726 .88 r
.54034 .31289 .55024 .32615 .56881 .31828 .55899 .30502 Mtetra
.685 .727 .882 r
.55899 .30502 .56881 .31828 .58749 .31048 .57775 .2972 Mtetra
.684 .728 .883 r
.57775 .2972 .58749 .31048 .60628 .30272 .59662 .28943 Mtetra
.682 .728 .885 r
.59662 .28943 .60628 .30272 .62518 .29501 .61561 .2817 Mtetra
.681 .729 .886 r
.61561 .2817 .62518 .29501 .6442 .28733 .63472 .27399 Mtetra
.68 .73 .888 r
.63472 .27399 .6442 .28733 .66334 .27968 .65394 .26631 Mtetra
.679 .73 .889 r
.65394 .26631 .66334 .27968 .68261 .27204 .6733 .25864 Mtetra
.678 .731 .89 r
.6733 .25864 .68261 .27204 .70199 .26443 .69277 .25098 Mtetra
.677 .731 .891 r
.69277 .25098 .70199 .26443 .7215 .25682 .71237 .24334 Mtetra
.676 .732 .892 r
.71237 .24334 .7215 .25682 .74114 .24922 .73211 .23569 Mtetra
.676 .732 .893 r
.73211 .23569 .74114 .24922 .76091 .24163 .75197 .22804 Mtetra
.675 .732 .894 r
.75197 .22804 .76091 .24163 .78081 .23402 .77197 .22039 Mtetra
.674 .733 .894 r
.77197 .22039 .78081 .23402 .80084 .22642 .7921 .21272 Mtetra
.673 .733 .895 r
.7921 .21272 .80084 .22642 .82101 .2188 .81237 .20504 Mtetra
.754 .69 .79 r
.17255 .49701 .1837 .51295 .20133 .49836 .19021 .48282 Mtetra
.748 .694 .799 r
.19021 .48282 .20133 .49836 .21889 .48473 .2078 .46954 Mtetra
.743 .697 .808 r
.2078 .46954 .21889 .48473 .23641 .47192 .22535 .45702 Mtetra
.738 .7 .815 r
.22535 .45702 .23641 .47192 .2539 .45981 .24288 .44518 Mtetra
.733 .703 .822 r
.24288 .44518 .2539 .45981 .27139 .44831 .26041 .4339 Mtetra
.728 .706 .829 r
.26041 .4339 .27139 .44831 .28889 .43734 .27796 .42312 Mtetra
.724 .708 .835 r
.27796 .42312 .28889 .43734 .30642 .42682 .29554 .41277 Mtetra
.72 .71 .841 r
.29554 .41277 .30642 .42682 .32398 .41671 .31315 .4028 Mtetra
.716 .712 .846 r
.31315 .4028 .32398 .41671 .34159 .40693 .33082 .39314 Mtetra
.712 .714 .85 r
.33082 .39314 .34159 .40693 .35926 .39746 .34854 .38378 Mtetra
.709 .716 .855 r
.34854 .38378 .35926 .39746 .37699 .38825 .36633 .37466 Mtetra
.706 .718 .859 r
.36633 .37466 .37699 .38825 .39479 .37927 .3842 .36575 Mtetra
.703 .719 .862 r
.3842 .36575 .39479 .37927 .41267 .3705 .40214 .35703 Mtetra
.7 .72 .865 r
.40214 .35703 .41267 .3705 .43063 .3619 .42017 .34848 Mtetra
.697 .722 .868 r
.42017 .34848 .43063 .3619 .44867 .35345 .43828 .34007 Mtetra
.695 .723 .871 r
.43828 .34007 .44867 .35345 .46681 .34513 .45649 .33178 Mtetra
.693 .724 .874 r
.45649 .33178 .46681 .34513 .48504 .33694 .4748 .3236 Mtetra
.691 .725 .876 r
.4748 .3236 .48504 .33694 .50337 .32884 .4932 .31551 Mtetra
.689 .726 .878 r
.4932 .31551 .50337 .32884 .5218 .32083 .51171 .30751 Mtetra
.687 .726 .88 r
.51171 .30751 .5218 .32083 .54034 .31289 .53033 .29957 Mtetra
.686 .727 .882 r
.53033 .29957 .54034 .31289 .55899 .30502 .54906 .29168 Mtetra
.684 .728 .883 r
.54906 .29168 .55899 .30502 .57775 .2972 .5679 .28385 Mtetra
.683 .729 .885 r
.5679 .28385 .57775 .2972 .59662 .28943 .58685 .27606 Mtetra
.682 .729 .886 r
.58685 .27606 .59662 .28943 .61561 .2817 .60592 .26829 Mtetra
.68 .73 .888 r
.60592 .26829 .61561 .2817 .63472 .27399 .62512 .26056 Mtetra
.679 .73 .889 r
.62512 .26056 .63472 .27399 .65394 .26631 .64443 .25284 Mtetra
.678 .731 .89 r
.64443 .25284 .65394 .26631 .6733 .25864 .66387 .24513 Mtetra
.677 .731 .891 r
.66387 .24513 .6733 .25864 .69277 .25098 .68344 .23743 Mtetra
.676 .732 .892 r
.68344 .23743 .69277 .25098 .71237 .24334 .70314 .22974 Mtetra
.676 .732 .893 r
.70314 .22974 .71237 .24334 .73211 .23569 .72296 .22204 Mtetra
.675 .733 .894 r
.72296 .22204 .73211 .23569 .75197 .22804 .74292 .21434 Mtetra
.674 .733 .894 r
.74292 .21434 .75197 .22804 .77197 .22039 .76302 .20663 Mtetra
.674 .733 .895 r
.76302 .20663 .77197 .22039 .7921 .21272 .78325 .1989 Mtetra
.673 .734 .896 r
.78325 .1989 .7921 .21272 .81237 .20504 .80362 .19116 Mtetra
.752 .693 .795 r
.16127 .48125 .17255 .49701 .19021 .48282 .17895 .46744 Mtetra
.746 .696 .804 r
.17895 .46744 .19021 .48282 .2078 .46954 .19657 .45447 Mtetra
.741 .7 .812 r
.19657 .45447 .2078 .46954 .22535 .45702 .21416 .44224 Mtetra
.736 .703 .82 r
.21416 .44224 .22535 .45702 .24288 .44518 .23173 .43063 Mtetra
.731 .705 .827 r
.23173 .43063 .24288 .44518 .26041 .4339 .24931 .41956 Mtetra
.726 .708 .833 r
.24931 .41956 .26041 .4339 .27796 .42312 .2669 .40896 Mtetra
.722 .71 .839 r
.2669 .40896 .27796 .42312 .29554 .41277 .28453 .39876 Mtetra
.718 .712 .844 r
.28453 .39876 .29554 .41277 .31315 .4028 .3022 .38891 Mtetra
.714 .714 .849 r
.3022 .38891 .31315 .4028 .33082 .39314 .31992 .37937 Mtetra
.71 .716 .853 r
.31992 .37937 .33082 .39314 .34854 .38378 .33771 .3701 Mtetra
.707 .718 .858 r
.33771 .3701 .34854 .38378 .36633 .37466 .35556 .36105 Mtetra
.704 .719 .861 r
.35556 .36105 .36633 .37466 .3842 .36575 .37349 .35221 Mtetra
.701 .721 .865 r
.37349 .35221 .3842 .36575 .40214 .35703 .39149 .34354 Mtetra
.698 .722 .868 r
.39149 .34354 .40214 .35703 .42017 .34848 .40959 .33503 Mtetra
.696 .723 .871 r
.40959 .33503 .42017 .34848 .43828 .34007 .42777 .32664 Mtetra
.694 .724 .873 r
.42777 .32664 .43828 .34007 .45649 .33178 .44605 .31838 Mtetra
.692 .725 .876 r
.44605 .31838 .45649 .33178 .4748 .3236 .46443 .31021 Mtetra
.69 .726 .878 r
.46443 .31021 .4748 .3236 .4932 .31551 .48291 .30212 Mtetra
.688 .727 .88 r
.48291 .30212 .4932 .31551 .51171 .30751 .5015 .29411 Mtetra
.686 .727 .882 r
.5015 .29411 .51171 .30751 .53033 .29957 .5202 .28616 Mtetra
.685 .728 .883 r
.5202 .28616 .53033 .29957 .54906 .29168 .539 .27827 Mtetra
.683 .729 .885 r
.539 .27827 .54906 .29168 .5679 .28385 .55793 .27041 Mtetra
.682 .73 .886 r
.55793 .27041 .5679 .28385 .58685 .27606 .57696 .26259 Mtetra
.681 .73 .888 r
.57696 .26259 .58685 .27606 .60592 .26829 .59612 .25479 Mtetra
.68 .731 .889 r
.59612 .25479 .60592 .26829 .62512 .26056 .6154 .24702 Mtetra
.678 .731 .89 r
.6154 .24702 .62512 .26056 .64443 .25284 .63481 .23926 Mtetra
.678 .732 .891 r
.63481 .23926 .64443 .25284 .66387 .24513 .65434 .23151 Mtetra
.677 .732 .892 r
.65434 .23151 .66387 .24513 .68344 .23743 .674 .22377 Mtetra
.676 .732 .893 r
.674 .22377 .68344 .23743 .70314 .22974 .69379 .21602 Mtetra
.675 .733 .894 r
.69379 .21602 .70314 .22974 .72296 .22204 .71371 .20827 Mtetra
.674 .733 .895 r
.71371 .20827 .72296 .22204 .74292 .21434 .73377 .20051 Mtetra
.674 .734 .895 r
.73377 .20051 .74292 .21434 .76302 .20663 .75396 .19274 Mtetra
.673 .734 .896 r
.75396 .19274 .76302 .20663 .78325 .1989 .77429 .18495 Mtetra
.672 .734 .897 r
.77429 .18495 .78325 .1989 .80362 .19116 .79476 .17714 Mtetra
.75 .696 .8 r
.14986 .46567 .16127 .48125 .17895 .46744 .16756 .4522 Mtetra
.744 .699 .809 r
.16756 .4522 .17895 .46744 .19657 .45447 .18521 .43953 Mtetra
.739 .702 .817 r
.18521 .43953 .19657 .45447 .21416 .44224 .20284 .42755 Mtetra
.734 .705 .824 r
.20284 .42755 .21416 .44224 .23173 .43063 .22045 .41616 Mtetra
.729 .708 .831 r
.22045 .41616 .23173 .43063 .24931 .41956 .23807 .40528 Mtetra
.724 .71 .837 r
.23807 .40528 .24931 .41956 .2669 .40896 .25571 .39483 Mtetra
.72 .712 .842 r
.25571 .39483 .2669 .40896 .28453 .39876 .27339 .38477 Mtetra
.716 .714 .847 r
.27339 .38477 .28453 .39876 .3022 .38891 .29112 .37504 Mtetra
.712 .716 .852 r
.29112 .37504 .3022 .38891 .31992 .37937 .3089 .3656 Mtetra
.709 .718 .856 r
.3089 .3656 .31992 .37937 .33771 .3701 .32674 .3564 Mtetra
.705 .719 .86 r
.32674 .3564 .33771 .3701 .35556 .36105 .34466 .34743 Mtetra
.702 .721 .864 r
.34466 .34743 .35556 .36105 .37349 .35221 .36265 .33864 Mtetra
.699 .722 .867 r
.36265 .33864 .37349 .35221 .39149 .34354 .38072 .33001 Mtetra
.697 .723 .87 r
.38072 .33001 .39149 .34354 .40959 .33503 .39889 .32153 Mtetra
.695 .724 .873 r
.39889 .32153 .40959 .33503 .42777 .32664 .41714 .31317 Mtetra
.692 .725 .875 r
.41714 .31317 .42777 .32664 .44605 .31838 .43549 .30491 Mtetra
.69 .726 .877 r
.43549 .30491 .44605 .31838 .46443 .31021 .45395 .29675 Mtetra
.688 .727 .879 r
.45395 .29675 .46443 .31021 .48291 .30212 .4725 .28866 Mtetra
.687 .728 .881 r
.4725 .28866 .48291 .30212 .5015 .29411 .49117 .28064 Mtetra
.685 .728 .883 r
.49117 .28064 .5015 .29411 .5202 .28616 .50994 .27268 Mtetra
.684 .729 .885 r
.50994 .27268 .5202 .28616 .539 .27827 .52883 .26476 Mtetra
.682 .73 .886 r
.52883 .26476 .539 .27827 .55793 .27041 .54784 .25688 Mtetra
.681 .73 .888 r
.54784 .25688 .55793 .27041 .57696 .26259 .56696 .24902 Mtetra
.68 .731 .889 r
.56696 .24902 .57696 .26259 .59612 .25479 .5862 .24119 Mtetra
.679 .731 .89 r
.5862 .24119 .59612 .25479 .6154 .24702 .60557 .23338 Mtetra
.678 .732 .891 r
.60557 .23338 .6154 .24702 .63481 .23926 .62507 .22558 Mtetra
.677 .732 .892 r
.62507 .22558 .63481 .23926 .65434 .23151 .64469 .21778 Mtetra
.676 .733 .893 r
.64469 .21778 .65434 .23151 .674 .22377 .66444 .20998 Mtetra
.675 .733 .894 r
.66444 .20998 .674 .22377 .69379 .21602 .68432 .20218 Mtetra
.674 .733 .895 r
.68432 .20218 .69379 .21602 .71371 .20827 .70434 .19437 Mtetra
.674 .734 .895 r
.70434 .19437 .71371 .20827 .73377 .20051 .72449 .18655 Mtetra
.673 .734 .896 r
.72449 .18655 .73377 .20051 .75396 .19274 .74479 .17871 Mtetra
.673 .734 .897 r
.74479 .17871 .75396 .19274 .77429 .18495 .76522 .17086 Mtetra
.672 .735 .897 r
.76522 .17086 .77429 .18495 .79476 .17714 .78579 .16298 Mtetra
.748 .699 .805 r
.13831 .45023 .14986 .46567 .16756 .4522 .15604 .43707 Mtetra
.742 .702 .813 r
.15604 .43707 .16756 .4522 .18521 .43953 .17372 .42468 Mtetra
.737 .705 .821 r
.17372 .42468 .18521 .43953 .20284 .42755 .19138 .41293 Mtetra
.732 .707 .828 r
.19138 .41293 .20284 .42755 .22045 .41616 .20903 .40174 Mtetra
.727 .71 .834 r
.20903 .40174 .22045 .41616 .23807 .40528 .2267 .39103 Mtetra
.722 .712 .84 r
.2267 .39103 .23807 .40528 .25571 .39483 .24439 .38074 Mtetra
.718 .714 .846 r
.24439 .38074 .25571 .39483 .27339 .38477 .26213 .3708 Mtetra
.714 .716 .851 r
.26213 .3708 .27339 .38477 .29112 .37504 .27991 .36118 Mtetra
.71 .718 .855 r
.27991 .36118 .29112 .37504 .3089 .3656 .29774 .35182 Mtetra
.707 .719 .859 r
.29774 .35182 .3089 .3656 .32674 .3564 .31565 .3427 Mtetra
.704 .721 .863 r
.31565 .3427 .32674 .3564 .34466 .34743 .33363 .33378 Mtetra
.701 .722 .866 r
.33363 .33378 .34466 .34743 .36265 .33864 .35168 .32503 Mtetra
.698 .723 .869 r
.35168 .32503 .36265 .33864 .38072 .33001 .36982 .31644 Mtetra
.696 .724 .872 r
.36982 .31644 .38072 .33001 .39889 .32153 .38806 .30798 Mtetra
.693 .725 .875 r
.38806 .30798 .39889 .32153 .41714 .31317 .40638 .29963 Mtetra
.691 .726 .877 r
.40638 .29963 .41714 .31317 .43549 .30491 .42481 .29138 Mtetra
.689 .727 .879 r
.42481 .29138 .43549 .30491 .45395 .29675 .44334 .28322 Mtetra
.687 .728 .881 r
.44334 .28322 .45395 .29675 .4725 .28866 .46197 .27512 Mtetra
.686 .729 .883 r
.46197 .27512 .4725 .28866 .49117 .28064 .48071 .26709 Mtetra
.684 .729 .885 r
.48071 .26709 .49117 .28064 .50994 .27268 .49957 .2591 Mtetra
.683 .73 .886 r
.49957 .2591 .50994 .27268 .52883 .26476 .51854 .25116 Mtetra
.681 .731 .887 r
.51854 .25116 .52883 .26476 .54784 .25688 .53763 .24324 Mtetra
.68 .731 .889 r
.53763 .24324 .54784 .25688 .56696 .24902 .55683 .23535 Mtetra
.679 .732 .89 r
.55683 .23535 .56696 .24902 .5862 .24119 .57616 .22748 Mtetra
.678 .732 .891 r
.57616 .22748 .5862 .24119 .60557 .23338 .59562 .21962 Mtetra
.677 .733 .892 r
.59562 .21962 .60557 .23338 .62507 .22558 .6152 .21177 Mtetra
.676 .733 .893 r
.6152 .21177 .62507 .22558 .64469 .21778 .63492 .20392 Mtetra
.675 .733 .894 r
.63492 .20392 .64469 .21778 .66444 .20998 .65476 .19607 Mtetra
.675 .734 .895 r
.65476 .19607 .66444 .20998 .68432 .20218 .67474 .18821 Mtetra
.674 .734 .895 r
.67474 .18821 .68432 .20218 .70434 .19437 .69486 .18034 Mtetra
.673 .734 .896 r
.69486 .18034 .70434 .19437 .72449 .18655 .71511 .17245 Mtetra
.673 .735 .897 r
.71511 .17245 .72449 .18655 .74479 .17871 .7355 .16455 Mtetra
.672 .735 .897 r
.7355 .16455 .74479 .17871 .76522 .17086 .75604 .15662 Mtetra
.672 .735 .898 r
.75604 .15662 .76522 .17086 .78579 .16298 .77671 .14867 Mtetra
.746 .701 .809 r
.12662 .43491 .13831 .45023 .15604 .43707 .14438 .42206 Mtetra
.74 .704 .818 r
.14438 .42206 .15604 .43707 .17372 .42468 .16209 .40991 Mtetra
.735 .707 .825 r
.16209 .40991 .17372 .42468 .19138 .41293 .17979 .39838 Mtetra
.729 .71 .832 r
.17979 .39838 .19138 .41293 .20903 .40174 .19749 .38738 Mtetra
.725 .712 .838 r
.19749 .38738 .20903 .40174 .2267 .39103 .2152 .37682 Mtetra
.72 .714 .844 r
.2152 .37682 .2267 .39103 .24439 .38074 .23294 .36666 Mtetra
.716 .716 .849 r
.23294 .36666 .24439 .38074 .26213 .3708 .25073 .35683 Mtetra
.712 .718 .854 r
.25073 .35683 .26213 .3708 .27991 .36118 .26856 .3473 Mtetra
.708 .719 .858 r
.26856 .3473 .27991 .36118 .29774 .35182 .28646 .33802 Mtetra
.705 .721 .862 r
.28646 .33802 .29774 .35182 .31565 .3427 .30442 .32896 Mtetra
.702 .722 .865 r
.30442 .32896 .31565 .3427 .33363 .33378 .32246 .32009 Mtetra
.699 .723 .869 r
.32246 .32009 .33363 .33378 .35168 .32503 .34059 .31138 Mtetra
.697 .725 .872 r
.34059 .31138 .35168 .32503 .36982 .31644 .3588 .30281 Mtetra
.694 .726 .874 r
.3588 .30281 .36982 .31644 .38806 .30798 .3771 .29437 Mtetra
.692 .726 .877 r
.3771 .29437 .38806 .30798 .40638 .29963 .3955 .28603 Mtetra
.69 .727 .879 r
.3955 .28603 .40638 .29963 .42481 .29138 .41399 .27778 Mtetra
.688 .728 .881 r
.41399 .27778 .42481 .29138 .44334 .28322 .4326 .26961 Mtetra
.686 .729 .883 r
.4326 .26961 .44334 .28322 .46197 .27512 .45131 .2615 Mtetra
.685 .73 .884 r
.45131 .2615 .46197 .27512 .48071 .26709 .47013 .25344 Mtetra
.683 .73 .886 r
.47013 .25344 .48071 .26709 .49957 .2591 .48907 .24543 Mtetra
.682 .731 .887 r
.48907 .24543 .49957 .2591 .51854 .25116 .50812 .23745 Mtetra
.681 .731 .889 r
.50812 .23745 .51854 .25116 .53763 .24324 .52729 .2295 Mtetra
.679 .732 .89 r
.52729 .2295 .53763 .24324 .55683 .23535 .54658 .22157 Mtetra
.678 .732 .891 r
.54658 .22157 .55683 .23535 .57616 .22748 .566 .21366 Mtetra
.677 .733 .892 r
.566 .21366 .57616 .22748 .59562 .21962 .58555 .20575 Mtetra
.676 .733 .893 r
.58555 .20575 .59562 .21962 .6152 .21177 .60522 .19784 Mtetra
.676 .734 .894 r
.60522 .19784 .6152 .21177 .63492 .20392 .62503 .18994 Mtetra
.675 .734 .895 r
.62503 .18994 .63492 .20392 .65476 .19607 .64497 .18203 Mtetra
.674 .734 .895 r
.64497 .18203 .65476 .19607 .67474 .18821 .66504 .1741 Mtetra
.673 .735 .896 r
.66504 .1741 .67474 .18821 .69486 .18034 .68525 .16617 Mtetra
.673 .735 .897 r
.68525 .16617 .69486 .18034 .71511 .17245 .70561 .15821 Mtetra
.672 .735 .897 r
.70561 .15821 .71511 .17245 .7355 .16455 .7261 .15024 Mtetra
.672 .735 .898 r
.7261 .15024 .7355 .16455 .75604 .15662 .74674 .14224 Mtetra
.671 .736 .899 r
.74674 .14224 .75604 .15662 .77671 .14867 .76752 .13421 Mtetra
.743 .704 .814 r
.11478 .41971 .12662 .43491 .14438 .42206 .13257 .40713 Mtetra
.738 .707 .822 r
.13257 .40713 .14438 .42206 .16209 .40991 .15032 .39521 Mtetra
.732 .709 .829 r
.15032 .39521 .16209 .40991 .17979 .39838 .16806 .38388 Mtetra
.727 .712 .836 r
.16806 .38388 .17979 .39838 .19749 .38738 .1858 .37304 Mtetra
.723 .714 .842 r
.1858 .37304 .19749 .38738 .2152 .37682 .20356 .36263 Mtetra
.718 .716 .847 r
.20356 .36263 .2152 .37682 .23294 .36666 .22135 .35259 Mtetra
.714 .718 .852 r
.22135 .35259 .23294 .36666 .25073 .35683 .23919 .34286 Mtetra
.71 .719 .857 r
.23919 .34286 .25073 .35683 .26856 .3473 .25708 .33341 Mtetra
.707 .721 .861 r
.25708 .33341 .26856 .3473 .28646 .33802 .27504 .32419 Mtetra
.704 .722 .864 r
.27504 .32419 .28646 .33802 .30442 .32896 .29306 .31518 Mtetra
.7 .724 .868 r
.29306 .31518 .30442 .32896 .32246 .32009 .31117 .30635 Mtetra
.698 .725 .871 r
.31117 .30635 .32246 .32009 .34059 .31138 .32936 .29767 Mtetra
.695 .726 .874 r
.32936 .29767 .34059 .31138 .3588 .30281 .34764 .28912 Mtetra
.693 .727 .876 r
.34764 .28912 .3588 .30281 .3771 .29437 .36601 .28069 Mtetra
.691 .728 .878 r
.36601 .28069 .3771 .29437 .3955 .28603 .38448 .27235 Mtetra
.689 .728 .881 r
.38448 .27235 .3955 .28603 .41399 .27778 .40305 .26409 Mtetra
.687 .729 .882 r
.40305 .26409 .41399 .27778 .4326 .26961 .42173 .25591 Mtetra
.685 .73 .884 r
.42173 .25591 .4326 .26961 .45131 .2615 .44052 .24778 Mtetra
.684 .73 .886 r
.44052 .24778 .45131 .2615 .47013 .25344 .45942 .2397 Mtetra
.682 .731 .887 r
.45942 .2397 .47013 .25344 .48907 .24543 .47844 .23165 Mtetra
.681 .732 .889 r
.47844 .23165 .48907 .24543 .50812 .23745 .49757 .22364 Mtetra
.68 .732 .89 r
.49757 .22364 .50812 .23745 .52729 .2295 .51683 .21565 Mtetra
.679 .732 .891 r
.51683 .21565 .52729 .2295 .54658 .22157 .53621 .20767 Mtetra
.678 .733 .892 r
.53621 .20767 .54658 .22157 .566 .21366 .55571 .19971 Mtetra
.677 .733 .893 r
.55571 .19971 .566 .21366 .58555 .20575 .57535 .19175 Mtetra
.676 .734 .894 r
.57535 .19175 .58555 .20575 .60522 .19784 .59511 .18379 Mtetra
.675 .734 .895 r
.59511 .18379 .60522 .19784 .62503 .18994 .61501 .17582 Mtetra
.674 .734 .896 r
.61501 .17582 .62503 .18994 .64497 .18203 .63505 .16785 Mtetra
.674 .735 .896 r
.63505 .16785 .64497 .18203 .66504 .1741 .65522 .15986 Mtetra
.673 .735 .897 r
.65522 .15986 .66504 .1741 .68525 .16617 .67553 .15185 Mtetra
.672 .735 .898 r
.67553 .15185 .68525 .16617 .70561 .15821 .69598 .14382 Mtetra
.672 .736 .898 r
.69598 .14382 .70561 .15821 .7261 .15024 .71658 .13577 Mtetra
.671 .736 .899 r
.71658 .13577 .7261 .15024 .74674 .14224 .73732 .1277 Mtetra
.73732 .1277 .74674 .14224 .76752 .13421 .7582 .11959 Mtetra
.741 .706 .818 r
.1028 .40461 .11478 .41971 .13257 .40713 .12062 .39227 Mtetra
.736 .709 .826 r
.12062 .39227 .13257 .40713 .15032 .39521 .1384 .38057 Mtetra
.73 .711 .833 r
.1384 .38057 .15032 .39521 .16806 .38388 .15618 .36942 Mtetra
.725 .714 .839 r
.15618 .36942 .16806 .38388 .1858 .37304 .17397 .35873 Mtetra
.721 .716 .845 r
.17397 .35873 .1858 .37304 .20356 .36263 .19177 .34845 Mtetra
.716 .718 .85 r
.19177 .34845 .20356 .36263 .22135 .35259 .20962 .33851 Mtetra
.712 .719 .855 r
.20962 .33851 .22135 .35259 .23919 .34286 .22751 .32887 Mtetra
.709 .721 .859 r
.22751 .32887 .23919 .34286 .25708 .33341 .24546 .31949 Mtetra
.705 .722 .863 r
.24546 .31949 .25708 .33341 .27504 .32419 .26348 .31033 Mtetra
.702 .724 .867 r
.26348 .31033 .27504 .32419 .29306 .31518 .28157 .30136 Mtetra
.699 .725 .87 r
.28157 .30136 .29306 .31518 .31117 .30635 .29974 .29256 Mtetra
.696 .726 .873 r
.29974 .29256 .31117 .30635 .32936 .29767 .31799 .2839 Mtetra
.694 .727 .876 r
.31799 .2839 .32936 .29767 .34764 .28912 .33634 .27537 Mtetra
.692 .728 .878 r
.33634 .27537 .34764 .28912 .36601 .28069 .35478 .26693 Mtetra
.689 .729 .88 r
.35478 .26693 .36601 .28069 .38448 .27235 .37332 .25859 Mtetra
.688 .729 .882 r
.37332 .25859 .38448 .27235 .40305 .26409 .39197 .25032 Mtetra
.686 .73 .884 r
.39197 .25032 .40305 .26409 .42173 .25591 .41073 .24211 Mtetra
.684 .731 .886 r
.41073 .24211 .42173 .25591 .44052 .24778 .42959 .23396 Mtetra
.683 .731 .887 r
.42959 .23396 .44052 .24778 .45942 .2397 .44857 .22585 Mtetra
.681 .732 .888 r
.44857 .22585 .45942 .2397 .47844 .23165 .46767 .21777 Mtetra
.68 .732 .89 r
.46767 .21777 .47844 .23165 .49757 .22364 .48689 .20971 Mtetra
.679 .733 .891 r
.48689 .20971 .49757 .22364 .51683 .21565 .50623 .20168 Mtetra
.678 .733 .892 r
.50623 .20168 .51683 .21565 .53621 .20767 .5257 .19365 Mtetra
.677 .734 .893 r
.5257 .19365 .53621 .20767 .55571 .19971 .5453 .18563 Mtetra
.676 .734 .894 r
.5453 .18563 .55571 .19971 .57535 .19175 .56502 .17762 Mtetra
.675 .734 .895 r
.56502 .17762 .57535 .19175 .59511 .18379 .58488 .1696 Mtetra
.674 .735 .896 r
.58488 .1696 .59511 .18379 .61501 .17582 .60487 .16157 Mtetra
.60487 .16157 .61501 .17582 .63505 .16785 .625 .15352 Mtetra
.673 .735 .897 r
.625 .15352 .63505 .16785 .65522 .15986 .64527 .14546 Mtetra
.672 .735 .898 r
.64527 .14546 .65522 .15986 .67553 .15185 .66568 .13739 Mtetra
.672 .736 .898 r
.66568 .13739 .67553 .15185 .69598 .14382 .68623 .12928 Mtetra
.671 .736 .899 r
.68623 .12928 .69598 .14382 .71658 .13577 .70693 .12116 Mtetra
.70693 .12116 .71658 .13577 .73732 .1277 .72778 .113 Mtetra
.67 .736 .9 r
.72778 .113 .73732 .1277 .7582 .11959 .74877 .10481 Mtetra
.739 .708 .822 r
.09067 .38958 .1028 .40461 .12062 .39227 .10852 .37747 Mtetra
.733 .711 .83 r
.10852 .37747 .12062 .39227 .1384 .38057 .12634 .36596 Mtetra
.728 .714 .837 r
.12634 .36596 .1384 .38057 .15618 .36942 .14416 .35497 Mtetra
.723 .716 .843 r
.14416 .35497 .15618 .36942 .17397 .35873 .16199 .34443 Mtetra
.719 .718 .848 r
.16199 .34443 .17397 .35873 .19177 .34845 .17985 .33426 Mtetra
.714 .719 .853 r
.17985 .33426 .19177 .34845 .20962 .33851 .19774 .32441 Mtetra
.71 .721 .858 r
.19774 .32441 .20962 .33851 .22751 .32887 .21569 .31485 Mtetra
.707 .722 .862 r
.21569 .31485 .22751 .32887 .24546 .31949 .2337 .30553 Mtetra
.703 .724 .866 r
.2337 .30553 .24546 .31949 .26348 .31033 .25178 .29642 Mtetra
.7 .725 .869 r
.25178 .29642 .26348 .31033 .28157 .30136 .26993 .28749 Mtetra
.698 .726 .872 r
.26993 .28749 .28157 .30136 .29974 .29256 .28816 .27871 Mtetra
.695 .727 .875 r
.28816 .27871 .29974 .29256 .31799 .2839 .30649 .27006 Mtetra
.693 .728 .877 r
.30649 .27006 .31799 .2839 .33634 .27537 .3249 .26153 Mtetra
.69 .729 .88 r
.3249 .26153 .33634 .27537 .35478 .26693 .34342 .25309 Mtetra
.688 .729 .882 r
.34342 .25309 .35478 .26693 .37332 .25859 .36203 .24474 Mtetra
.686 .73 .884 r
.36203 .24474 .37332 .25859 .39197 .25032 .38076 .23645 Mtetra
.685 .731 .885 r
.38076 .23645 .39197 .25032 .41073 .24211 .39959 .22822 Mtetra
.683 .731 .887 r
.39959 .22822 .41073 .24211 .42959 .23396 .41853 .22003 Mtetra
.682 .732 .888 r
.41853 .22003 .42959 .23396 .44857 .22585 .43759 .21189 Mtetra
.681 .732 .89 r
.43759 .21189 .44857 .22585 .46767 .21777 .45678 .20377 Mtetra
.679 .733 .891 r
.45678 .20377 .46767 .21777 .48689 .20971 .47608 .19567 Mtetra
.678 .733 .892 r
.47608 .19567 .48689 .20971 .50623 .20168 .49551 .18758 Mtetra
.677 .734 .893 r
.49551 .18758 .50623 .20168 .5257 .19365 .51506 .1795 Mtetra
.676 .734 .894 r
.51506 .1795 .5257 .19365 .5453 .18563 .53475 .17143 Mtetra
.675 .734 .895 r
.53475 .17143 .5453 .18563 .56502 .17762 .55456 .16335 Mtetra
.675 .735 .896 r
.55456 .16335 .56502 .17762 .58488 .1696 .57451 .15526 Mtetra
.674 .735 .896 r
.57451 .15526 .58488 .1696 .60487 .16157 .5946 .14717 Mtetra
.673 .735 .897 r
.5946 .14717 .60487 .16157 .625 .15352 .61483 .13905 Mtetra
.673 .736 .898 r
.61483 .13905 .625 .15352 .64527 .14546 .63519 .13092 Mtetra
.672 .736 .898 r
.63519 .13092 .64527 .14546 .66568 .13739 .6557 .12276 Mtetra
.671 .736 .899 r
.6557 .12276 .66568 .13739 .68623 .12928 .67636 .11458 Mtetra
.67636 .11458 .68623 .12928 .70693 .12116 .69716 .10637 Mtetra
.67 .737 .9 r
.69716 .10637 .70693 .12116 .72778 .113 .71811 .09813 Mtetra
.71811 .09813 .72778 .113 .74877 .10481 .73921 .08986 Mtetra
.737 .711 .826 r
.07838 .37461 .09067 .38958 .10852 .37747 .09627 .36271 Mtetra
.731 .713 .834 r
.09627 .36271 .10852 .37747 .12634 .36596 .11413 .35139 Mtetra
.726 .715 .84 r
.11413 .35139 .12634 .36596 .14416 .35497 .13199 .34054 Mtetra
.721 .718 .846 r
.13199 .34054 .14416 .35497 .16199 .34443 .14987 .33012 Mtetra
.717 .719 .851 r
.14987 .33012 .16199 .34443 .17985 .33426 .16777 .32005 Mtetra
.713 .721 .856 r
.16777 .32005 .17985 .33426 .19774 .32441 .18572 .31029 Mtetra
.709 .722 .861 r
.18572 .31029 .19774 .32441 .21569 .31485 .20373 .3008 Mtetra
.705 .724 .864 r
.20373 .3008 .21569 .31485 .2337 .30553 .22179 .29153 Mtetra
.702 .725 .868 r
.22179 .29153 .2337 .30553 .25178 .29642 .23993 .28245 Mtetra
.699 .726 .871 r
.23993 .28245 .25178 .29642 .26993 .28749 .25815 .27355 Mtetra
.696 .727 .874 r
.25815 .27355 .26993 .28749 .28816 .27871 .27645 .26479 Mtetra
.694 .728 .877 r
.27645 .26479 .28816 .27871 .30649 .27006 .29484 .25615 Mtetra
.691 .729 .879 r
.29484 .25615 .30649 .27006 .3249 .26153 .31333 .24761 Mtetra
.689 .73 .881 r
.31333 .24761 .3249 .26153 .34342 .25309 .33191 .23916 Mtetra
.687 .73 .883 r
.33191 .23916 .34342 .25309 .36203 .24474 .3506 .23079 Mtetra
.685 .731 .885 r
.3506 .23079 .36203 .24474 .38076 .23645 .3694 .22248 Mtetra
.684 .732 .887 r
.3694 .22248 .38076 .23645 .39959 .22822 .38831 .21422 Mtetra
.682 .732 .888 r
.38831 .21422 .39959 .22822 .41853 .22003 .40733 .206 Mtetra
.681 .733 .889 r
.40733 .206 .41853 .22003 .43759 .21189 .42648 .19781 Mtetra
.68 .733 .891 r
.42648 .19781 .43759 .21189 .45678 .20377 .44574 .18964 Mtetra
.679 .734 .892 r
.44574 .18964 .45678 .20377 .47608 .19567 .46513 .18149 Mtetra
.677 .734 .893 r
.46513 .18149 .47608 .19567 .49551 .18758 .48464 .17335 Mtetra
.677 .734 .894 r
.48464 .17335 .49551 .18758 .51506 .1795 .50429 .16522 Mtetra
.676 .735 .895 r
.50429 .16522 .51506 .1795 .53475 .17143 .52406 .15708 Mtetra
.675 .735 .896 r
.52406 .15708 .53475 .17143 .55456 .16335 .54397 .14894 Mtetra
.674 .735 .896 r
.54397 .14894 .55456 .16335 .57451 .15526 .56401 .14078 Mtetra
.673 .736 .897 r
.56401 .14078 .57451 .15526 .5946 .14717 .5842 .13261 Mtetra
.673 .736 .898 r
.5842 .13261 .5946 .14717 .61483 .13905 .60452 .12443 Mtetra
.672 .736 .898 r
.60452 .12443 .61483 .13905 .63519 .13092 .62498 .11622 Mtetra
.672 .736 .899 r
.62498 .11622 .63519 .13092 .6557 .12276 .64559 .10798 Mtetra
.671 .737 .899 r
.64559 .10798 .6557 .12276 .67636 .11458 .66635 .09972 Mtetra
.671 .737 .9 r
.66635 .09972 .67636 .11458 .69716 .10637 .68726 .09143 Mtetra
.67 .737 .9 r
.68726 .09143 .69716 .10637 .71811 .09813 .70832 .0831 Mtetra
.67 .737 .901 r
.70832 .0831 .71811 .09813 .73921 .08986 .72953 .07473 Mtetra
.735 .713 .83 r
.06594 .35969 .07838 .37461 .09627 .36271 .08386 .34799 Mtetra
.729 .715 .837 r
.08386 .34799 .09627 .36271 .11413 .35139 .10176 .33682 Mtetra
.724 .717 .844 r
.10176 .33682 .11413 .35139 .13199 .34054 .11967 .32612 Mtetra
.719 .719 .849 r
.11967 .32612 .13199 .34054 .14987 .33012 .13759 .3158 Mtetra
.715 .721 .854 r
.13759 .3158 .14987 .33012 .16777 .32005 .15555 .30582 Mtetra
.711 .722 .859 r
.15555 .30582 .16777 .32005 .18572 .31029 .17355 .29614 Mtetra
.707 .724 .863 r
.17355 .29614 .18572 .31029 .20373 .3008 .19161 .2867 Mtetra
.704 .725 .867 r
.19161 .2867 .20373 .3008 .22179 .29153 .20974 .27747 Mtetra
.7 .726 .87 r
.20974 .27747 .22179 .29153 .23993 .28245 .22794 .26843 Mtetra
.697 .727 .873 r
.22794 .26843 .23993 .28245 .25815 .27355 .24622 .25954 Mtetra
.695 .728 .876 r
.24622 .25954 .25815 .27355 .27645 .26479 .26459 .25078 Mtetra
.692 .729 .878 r
.26459 .25078 .27645 .26479 .29484 .25615 .28305 .24214 Mtetra
.69 .73 .881 r
.28305 .24214 .29484 .25615 .31333 .24761 .3016 .2336 Mtetra
.688 .731 .883 r
.3016 .2336 .31333 .24761 .33191 .23916 .32026 .22514 Mtetra
.686 .731 .885 r
.32026 .22514 .33191 .23916 .3506 .23079 .33903 .21674 Mtetra
.684 .732 .886 r
.33903 .21674 .3506 .23079 .3694 .22248 .3579 .2084 Mtetra
.683 .732 .888 r
.3579 .2084 .3694 .22248 .38831 .21422 .37689 .2001 Mtetra
.681 .733 .889 r
.37689 .2001 .38831 .21422 .40733 .206 .39599 .19184 Mtetra
.68 .733 .891 r
.39599 .19184 .40733 .206 .42648 .19781 .41522 .18361 Mtetra
.679 .734 .892 r
.41522 .18361 .42648 .19781 .44574 .18964 .43456 .17539 Mtetra
.678 .734 .893 r
.43456 .17539 .44574 .18964 .46513 .18149 .45404 .16719 Mtetra
.677 .734 .894 r
.45404 .16719 .46513 .18149 .48464 .17335 .47364 .15899 Mtetra
.676 .735 .895 r
.47364 .15899 .48464 .17335 .50429 .16522 .49337 .15079 Mtetra
.675 .735 .896 r
.49337 .15079 .50429 .16522 .52406 .15708 .51324 .14259 Mtetra
.674 .735 .896 r
.51324 .14259 .52406 .15708 .54397 .14894 .53324 .13438 Mtetra
.674 .736 .897 r
.53324 .13438 .54397 .14894 .56401 .14078 .55338 .12615 Mtetra
.673 .736 .898 r
.55338 .12615 .56401 .14078 .5842 .13261 .57365 .11791 Mtetra
.672 .736 .898 r
.57365 .11791 .5842 .13261 .60452 .12443 .59408 .10964 Mtetra
.672 .736 .899 r
.59408 .10964 .60452 .12443 .62498 .11622 .61464 .10135 Mtetra
.671 .737 .899 r
.61464 .10135 .62498 .11622 .64559 .10798 .63535 .09303 Mtetra
.671 .737 .9 r
.63535 .09303 .64559 .10798 .66635 .09972 .65621 .08469 Mtetra
.67 .737 .9 r
.65621 .08469 .66635 .09972 .68726 .09143 .67723 .0763 Mtetra
.67 .737 .901 r
.67723 .0763 .68726 .09143 .70832 .0831 .69839 .06789 Mtetra
.669 .737 .901 r
.69839 .06789 .70832 .0831 .72953 .07473 .71971 .05943 Mtetra
.733 .715 .834 r
.05334 .34481 .06594 .35969 .08386 .34799 .0713 .33328 Mtetra
.727 .717 .841 r
.0713 .33328 .08386 .34799 .10176 .33682 .08924 .32226 Mtetra
.722 .719 .847 r
.08924 .32226 .10176 .33682 .11967 .32612 .10719 .31167 Mtetra
.717 .721 .852 r
.10719 .31167 .11967 .32612 .13759 .3158 .12516 .30146 Mtetra
.713 .722 .857 r
.12516 .30146 .13759 .3158 .15555 .30582 .14317 .29156 Mtetra
.709 .724 .862 r
.14317 .29156 .15555 .30582 .17355 .29614 .16123 .28194 Mtetra
.705 .725 .865 r
.16123 .28194 .17355 .29614 .19161 .2867 .17935 .27254 Mtetra
.702 .726 .869 r
.17935 .27254 .19161 .2867 .20974 .27747 .19753 .26335 Mtetra
.699 .727 .872 r
.19753 .26335 .20974 .27747 .22794 .26843 .2158 .25433 Mtetra
.696 .728 .875 r
.2158 .25433 .22794 .26843 .24622 .25954 .23414 .24545 Mtetra
.693 .729 .878 r
.23414 .24545 .24622 .25954 .26459 .25078 .25257 .2367 Mtetra
.691 .73 .88 r
.25257 .2367 .26459 .25078 .28305 .24214 .2711 .22805 Mtetra
.689 .731 .882 r
.2711 .22805 .28305 .24214 .3016 .2336 .28973 .21949 Mtetra
.687 .731 .884 r
.28973 .21949 .3016 .2336 .32026 .22514 .30846 .211 Mtetra
.685 .732 .886 r
.30846 .211 .32026 .22514 .33903 .21674 .3273 .20258 Mtetra
.684 .732 .888 r
.3273 .20258 .33903 .21674 .3579 .2084 .34625 .1942 Mtetra
.682 .733 .889 r
.34625 .1942 .3579 .2084 .37689 .2001 .36532 .18587 Mtetra
.681 .733 .89 r
.36532 .18587 .37689 .2001 .39599 .19184 .38451 .17756 Mtetra
.679 .734 .892 r
.38451 .17756 .39599 .19184 .41522 .18361 .40381 .16927 Mtetra
.678 .734 .893 r
.40381 .16927 .41522 .18361 .43456 .17539 .42324 .161 Mtetra
.677 .735 .894 r
.42324 .161 .43456 .17539 .45404 .16719 .4428 .15274 Mtetra
.676 .735 .895 r
.4428 .15274 .45404 .16719 .47364 .15899 .46249 .14448 Mtetra
.675 .735 .895 r
.46249 .14448 .47364 .15899 .49337 .15079 .48231 .13622 Mtetra
.674 .736 .896 r
.48231 .13622 .49337 .15079 .51324 .14259 .50227 .12795 Mtetra
.674 .736 .897 r
.50227 .12795 .51324 .14259 .53324 .13438 .52237 .11966 Mtetra
.673 .736 .898 r
.52237 .11966 .53324 .13438 .55338 .12615 .5426 .11136 Mtetra
.672 .736 .898 r
.5426 .11136 .55338 .12615 .57365 .11791 .56297 .10304 Mtetra
.672 .737 .899 r
.56297 .10304 .57365 .11791 .59408 .10964 .58349 .09469 Mtetra
.671 .737 .899 r
.58349 .09469 .59408 .10964 .61464 .10135 .60416 .08632 Mtetra
.671 .737 .9 r
.60416 .08632 .61464 .10135 .63535 .09303 .62498 .07791 Mtetra
.67 .737 .9 r
.62498 .07791 .63535 .09303 .65621 .08469 .64594 .06948 Mtetra
.67 .737 .901 r
.64594 .06948 .65621 .08469 .67723 .0763 .66706 .061 Mtetra
.669 .738 .901 r
.66706 .061 .67723 .0763 .69839 .06789 .68833 .05249 Mtetra
.669 .738 .902 r
.68833 .05249 .69839 .06789 .71971 .05943 .70976 .04394 Mtetra
.731 .717 .838 r
.04058 .32995 .05334 .34481 .0713 .33328 .05857 .31858 Mtetra
.725 .719 .844 r
.05857 .31858 .0713 .33328 .08924 .32226 .07656 .30769 Mtetra
.72 .721 .85 r
.07656 .30769 .08924 .32226 .10719 .31167 .09455 .29721 Mtetra
.716 .722 .855 r
.09455 .29721 .10719 .31167 .12516 .30146 .11257 .28708 Mtetra
.711 .724 .86 r
.11257 .28708 .12516 .30146 .14317 .29156 .13063 .27725 Mtetra
.707 .725 .864 r
.13063 .27725 .14317 .29156 .16123 .28194 .14875 .26768 Mtetra
.704 .726 .868 r
.14875 .26768 .16123 .28194 .17935 .27254 .16692 .25832 Mtetra
.7 .727 .871 r
.16692 .25832 .17935 .27254 .19753 .26335 .18517 .24916 Mtetra
.697 .728 .874 r
.18517 .24916 .19753 .26335 .2158 .25433 .2035 .24015 Mtetra
.695 .729 .877 r
.2035 .24015 .2158 .25433 .23414 .24545 .22191 .23127 Mtetra
.692 .73 .879 r
.22191 .23127 .23414 .24545 .25257 .2367 .24041 .22251 Mtetra
.69 .731 .882 r
.24041 .22251 .25257 .2367 .2711 .22805 .25901 .21385 Mtetra
.688 .732 .884 r
.25901 .21385 .2711 .22805 .28973 .21949 .27771 .20527 Mtetra
.686 .732 .886 r
.27771 .20527 .28973 .21949 .30846 .211 .29651 .19676 Mtetra
.684 .733 .887 r
.29651 .19676 .30846 .211 .3273 .20258 .31543 .1883 Mtetra
.683 .733 .889 r
.31543 .1883 .3273 .20258 .34625 .1942 .33446 .17988 Mtetra
.681 .734 .89 r
.33446 .17988 .34625 .1942 .36532 .18587 .3536 .1715 Mtetra
.68 .734 .891 r
.3536 .1715 .36532 .18587 .38451 .17756 .37287 .16315 Mtetra
.679 .734 .893 r
.37287 .16315 .38451 .17756 .40381 .16927 .39226 .15481 Mtetra
.678 .735 .894 r
.39226 .15481 .40381 .16927 .42324 .161 .41178 .14648 Mtetra
.677 .735 .895 r
.41178 .14648 .42324 .161 .4428 .15274 .43142 .13815 Mtetra
.676 .735 .895 r
.43142 .13815 .4428 .15274 .46249 .14448 .4512 .12983 Mtetra
.675 .736 .896 r
.4512 .12983 .46249 .14448 .48231 .13622 .47111 .12149 Mtetra
.674 .736 .897 r
.47111 .12149 .48231 .13622 .50227 .12795 .49116 .11315 Mtetra
.673 .736 .898 r
.49116 .11315 .50227 .12795 .52237 .11966 .51135 .10479 Mtetra
.673 .737 .898 r
.51135 .10479 .52237 .11966 .5426 .11136 .53168 .09641 Mtetra
.672 .737 .899 r
.53168 .09641 .5426 .11136 .56297 .10304 .55215 .088 Mtetra
.671 .737 .899 r
.55215 .088 .56297 .10304 .58349 .09469 .57277 .07957 Mtetra
.671 .737 .9 r
.57277 .07957 .58349 .09469 .60416 .08632 .59354 .07111 Mtetra
.67 .737 .9 r
.59354 .07111 .60416 .08632 .62498 .07791 .61446 .06262 Mtetra
.67 .738 .901 r
.61446 .06262 .62498 .07791 .64594 .06948 .63553 .05409 Mtetra
.63553 .05409 .64594 .06948 .66706 .061 .65675 .04552 Mtetra
.669 .738 .902 r
.65675 .04552 .66706 .061 .68833 .05249 .67814 .03691 Mtetra
.67814 .03691 .68833 .05249 .70976 .04394 .69968 .02826 Mtetra
.729 .719 .841 r
.02766 .3151 .04058 .32995 .05857 .31858 .04569 .30388 Mtetra
.723 .721 .847 r
.04569 .30388 .05857 .31858 .07656 .30769 .06371 .2931 Mtetra
.718 .722 .853 r
.06371 .2931 .07656 .30769 .09455 .29721 .08175 .28272 Mtetra
.714 .724 .858 r
.08175 .28272 .09455 .29721 .11257 .28708 .09982 .27266 Mtetra
.71 .725 .862 r
.09982 .27266 .11257 .28708 .13063 .27725 .11794 .26289 Mtetra
.706 .726 .866 r
.11794 .26289 .13063 .27725 .14875 .26768 .13611 .25336 Mtetra
.702 .728 .87 r
.13611 .25336 .14875 .26768 .16692 .25832 .15434 .24403 Mtetra
.699 .729 .873 r
.15434 .24403 .16692 .25832 .18517 .24916 .17265 .23488 Mtetra
.696 .729 .876 r
.17265 .23488 .18517 .24916 .2035 .24015 .19104 .22588 Mtetra
.693 .73 .879 r
.19104 .22588 .2035 .24015 .22191 .23127 .20952 .217 Mtetra
.691 .731 .881 r
.20952 .217 .22191 .23127 .24041 .22251 .22809 .20823 Mtetra
.689 .732 .883 r
.22809 .20823 .24041 .22251 .25901 .21385 .24676 .19955 Mtetra
.687 .732 .885 r
.24676 .19955 .25901 .21385 .27771 .20527 .26553 .19094 Mtetra
.685 .733 .887 r
.26553 .19094 .27771 .20527 .29651 .19676 .28441 .1824 Mtetra
.683 .733 .888 r
.28441 .1824 .29651 .19676 .31543 .1883 .3034 .1739 Mtetra
.682 .734 .89 r
.3034 .1739 .31543 .1883 .33446 .17988 .32251 .16544 Mtetra
.68 .734 .891 r
.32251 .16544 .33446 .17988 .3536 .1715 .34173 .15701 Mtetra
.679 .735 .892 r
.34173 .15701 .3536 .1715 .37287 .16315 .36108 .1486 Mtetra
.678 .735 .893 r
.36108 .1486 .37287 .16315 .39226 .15481 .38056 .1402 Mtetra
.677 .735 .894 r
.38056 .1402 .39226 .15481 .41178 .14648 .40016 .13181 Mtetra
.676 .736 .895 r
.40016 .13181 .41178 .14648 .43142 .13815 .41989 .12342 Mtetra
.675 .736 .896 r
.41989 .12342 .43142 .13815 .4512 .12983 .43976 .11502 Mtetra
.674 .736 .897 r
.43976 .11502 .4512 .12983 .47111 .12149 .45976 .10661 Mtetra
.673 .736 .898 r
.45976 .10661 .47111 .12149 .49116 .11315 .4799 .09819 Mtetra
.673 .737 .898 r
.4799 .09819 .49116 .11315 .51135 .10479 .50019 .08975 Mtetra
.672 .737 .899 r
.50019 .08975 .51135 .10479 .53168 .09641 .52061 .08129 Mtetra
.52061 .08129 .53168 .09641 .55215 .088 .54119 .0728 Mtetra
.671 .737 .9 r
.54119 .0728 .55215 .088 .57277 .07957 .56191 .06428 Mtetra
.56191 .06428 .57277 .07957 .59354 .07111 .58278 .05572 Mtetra
.67 .738 .901 r
.58278 .05572 .59354 .07111 .61446 .06262 .6038 .04714 Mtetra
.6038 .04714 .61446 .06262 .63553 .05409 .62497 .03851 Mtetra
.669 .738 .902 r
.62497 .03851 .63553 .05409 .65675 .04552 .64631 .02984 Mtetra
.64631 .02984 .65675 .04552 .67814 .03691 .6678 .02113 Mtetra
.6678 .02113 .67814 .03691 .69968 .02826 .68946 .01238 Mtetra
0 g
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.03716 .25514 m
.68874 0 L
s
.03716 .25514 m
.04196 .25962 L
s
[(1)] .02757 .24618 1 .93395 Mshowa
.1552 .20892 m
.15981 .21359 L
s
[(1.2)] .14598 .19958 .98733 1 Mshowa
.27893 .16047 m
.28333 .16533 L
s
[(1.4)] .27013 .15073 .90393 1 Mshowa
.40878 .10962 m
.41294 .11469 L
s
[(1.6)] .40046 .09948 .82054 1 Mshowa
.54521 .0562 m
.54911 .06148 L
s
[(1.8)] .53743 .04564 .73714 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(2)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.06616 .24378 m
.06901 .2465 L
s
.09549 .2323 m
.09832 .23504 L
s
.12517 .22067 m
.12797 .22345 L
s
.18558 .19702 m
.18832 .19985 L
s
.21633 .18498 m
.21903 .18784 L
s
.24744 .1728 m
.25012 .17569 L
s
.3108 .14799 m
.31341 .15094 L
s
.34306 .13535 m
.34564 .13834 L
s
.37572 .12257 m
.37826 .12558 L
s
.44225 .09652 m
.44471 .09959 L
s
.47614 .08324 m
.47856 .08635 L
s
.51046 .06981 m
.51284 .07294 L
s
.58041 .04242 m
.5827 .04562 L
s
.61605 .02846 m
.6183 .03169 L
s
.65216 .01432 m
.65436 .01759 L
s
gsave
.29165 .07573 -70.3112 -19 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 23 translate 1 -1 scale
63.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(q) show
69.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
%%DocumentNeededResources: font Courier
%%DocumentNeededFonts: Courier
%%DocumentFonts: font Courier
grestore
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellMargins->{{Inherited, Inherited}, {0, 0}},
  ImageSize->{180.5, 165.5},
  ImageMargins->{{60, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40002d0000YA000`40O003h00O]7oo002dOol00;Ao
o`00]7oo002dOol00;Aoo`00]7oo002dOol00;Aoo`00]7oo002dOol00;Aoo`00]7oo002dOol00;Ao
o`00]7oo001eOol4000kOol007Eoo`03001oogoo03aoo`00MWoo00<007ooOol0>goo001gOol00`00
Oomoo`0jOol007Eoo`04001oogoo000kOol007Ioo`8000Ioo`8003Aoo`00OGoo00@007ooOol003=o
o`00OGoo00@007ooOol003=oo`00OGoo00@007ooOol003=oo`00EGoo1@000Woo0P0017oo0P006Goo
00@007ooOol003=oo`00>Woo0`006Woo00<007ooOol01goo00@007ooOol001Aoo`8000=oo`8003Ao
o`00>goo00<007ooOol06Goo00<007ooOol01goo00@007ooOol0019oo`800003Ool0000003Qoo`00
>Goo0`006goo00<007ooOol027oo0P0047oo0`000goo0P00>7oo000hOol01000Oomoo`006goo00<0
07ooOol01goo00@007ooOol000eoo`800009Ool007ooE_aBg5;l001Fo00003Moo`00>7oo00@007oo
Ool001Yoo`8000Yoo`8000]oo`<0009oo`0:E_aBfeKlD]]Fo5;K001Bfgoo000fOol003Uoo`@0031o
o`80000?Ool007ooOomBo5;LE_aBg5;lD]aFo000D_aBg00003Ioo`00JWoo0`0017oo00iBfeKlD]]F
o5;KE_aBfeKlD]/005;KE_aoo`00=Goo000oOol50002Ool20003Ool3000IOol40003Ool04U;LE_aB
g5;lD]aFo5;LD_aBg5KlD]aBo5;L001Bg5;lOol003Aoo`00@Goo00<007ooOol01goo00@007ooOol0
01Ioo`8000=oo`0EE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo000E_aBfeKl008003=o
o`00@Goo00<007ooOol01goo00@007ooOol001=oo`@000=oo`0HE_aBg5;lD]aFo5;LD_aBg5KlD]aB
o5;LE_aBg5;lD]aFo5;L001Bg5KlD]aoo`00<goo0011Ool00`00Oomoo`07Ool3000BOol20005Ool0
6e;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_`005KlD]]Fo5;KOol0000bOol0
045oo`03001oogoo00Qoo`03001oogoo00ioo`@000=oo`0ND]aBo5;LE_aBg5;lD]aFo5;LD_aBg5Kl
D]aBo5;LE_aBg5;lD]aFo5;LD_aBg000D]aBo5;LE_aBg000<Woo0010Ool2000:Ool3000<Ool20004
Ool08U;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_`005Kl
D]]Fo5;KE_aoo`00<Goo001HOol40004Ool095;LE_aBg5KlD]aFo5;LD_aBg5KlD]aBo5;LE_aBg5;l
D]aFo5;LD_aBg5KlD]aBo5;LE_aBg000D]aFo5;LD_aBg5KlOol0031oo`00EWoo0P001Woo02IFo5;K
E_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKl001Fo5;K
E_aBfeKlD]]oo`001goo0P009goo000[Ool50002Ool20005Ool3000GOol30005Ool0:EKlD]]Fo5;L
E_aBg5KlD]aFo5;LE_aBg5KlD]aFo5;LE_aBg5;lD]aFo5;LD_aBg5KlD]aBo5;LE_aBg5;lD]`005;L
D_aBg5KlD]aBo5;LOol00P0017oo00@007ooOol002Ioo`00;Goo00<007ooOol02Woo00<007ooOol0
57oo0P001Goo02aFo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]F
o5;KE_aBfeKlD]]Fo5;KE_aBfeKl001Fo5;KE_aBfeKlD]]Fo5;K0P001goo00<007ooOol097oo000]
Ool00`00Oomoo`07Ool5000BOol40003Ool0<EKlD]]Fo5;LE_aBfeKlD]aFo5;KE_aBg5KlD]]Fo5;L
E_aBfeKlD]aFo5;LE_aBg5KlD]aFo5;LE_aBg5KlD]aFo5;LD_aBg5KlD]`005;LE_aBg5;lD]aFo5;L
D_aBg00000Eoo`8002Moo`00;Goo00<007ooOol01goo00@007ooOol0015oo`8000Ioo`0cE_aBfeKl
D]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKl
D]]Fo5;KE_aBfeKl001Fo5;KE_aBfeKlD]]Fo5;KE_aoo`0000Aoo`03001oogoo02Ioo`00;Goo00<0
07ooOol027oo00<007oo00003Woo0`001Woo03ABg5KlD]]Fo5;LE_aBfeKlD]aFo5;KE_aBg5KlD]]F
o5;LE_aBfeKlD]aFo5;KE_aBg5KlD]]Fo5;LE_aBfeKlD]aFo5;LD_aBg5KlD]aBo5;LE_`005;lD]aF
o5;LD_aBg5KlD]aBo08000=oo`<002Ioo`00;7oo0P002goo0P002goo0`001Woo03UBfeKlD]]Fo5;K
E_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;K
E_aBfeKlD]]Fo5;KE_aBfeKlD]/005;KE_aBfeKlD]]Fo5;KE_aBfgoo0000;7oo0014Ool30005Ool0
?EKKE_aBfeKlE]aFo5;KE_aFg5KlD]]Fo5;LE_aBfeKlD]aFo5;KE_aBg5KlD]]Fo5;LE_aBfeKlD]aF
o5;KE_aBg5KlD]]Fo5;LE_aBfeKlD]aFo5;KE_aBg5KlD]]Bo000E_aBg5;lD]aFo5;LD_aBg5KlD]ao
o`0002]oo`005goo1@000Woo0P000goo10006Woo0`001goo03aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;K
E_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;K
E_aBfeKlD]]Fo5;KE_aBf`00D]]Fo5;KE_aBfeKlD]]Fo5;KE_`2Ool00`00Oomoo`0XOol001Uoo`03
001oogoo00Moo`03001oogoo01Uoo`<000Moo`11E_aFfeKlD]]Fo5KKE_aBfeKlE]aFo5;KE_aFg5Kl
D]]Fo5;LE_aBfeKlD]aFo5;KE_aBg5KlD]]Fo5;LE_aBfeKlD]aFo5;KE_aBg5KlD]]Fo5;LE_aBfeKl
D]aFo5;KE_aBg5Kl001Fo5;LE_aBfeKlD]aFo5;LD_aBg5KlOol0000ZOol001Uoo`03001oogoo00Qo
o`03001oogoo01Eoo`<000Moo`15E_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aB
feKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aB
feKlD]]Fo5;K001BfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aoo`0002Uoo`006Goo00<007ooOol02Goo
00<007ooOol04Woo0P0000=oo`00Ool017oo04IFg5KKE_aFfeKlE]]Fo5;KE]aFfeKlD]]Fo5KKE_aB
feKlE]]Fo5;KE]aFfeKlD]]Fo5KLE_aBfeKlE]aFo5;KE_aBg5KlD]]Fo5;LE_aBfeKlD]aFo5;KE_aB
g5KlD]]Fo5;LE_aBfeKlD]aFo5;KE_`005KlD]]Fo5;LE_aBfeKlD]aFo5;KE_aBg09oo`03001oogoo
02Ioo`006Goo00<007ooOol01goo00@007ooOol0011oo`<000Ioo`1;E_aBfeKlD]]Fo5;KE_aBfeKl
D]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKl
D]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;K001BfeKlD]]Fo5;KE_aBfeKl
D]]Fo5;KE_aBfgoo008002Moo`0067oo0P002Woo0P003goo0P0000=oo`00Ool01Goo00EFo5KKE]]F
feKl00=Ff`16E_aFfeKLE]]Fo5;KE]aFfeKlD]]Fg5KKE_aBfeKLE]]Fo5;KE]aFfeKlD]]Fg5KKE_aB
feKLE]]Fo5;KE]aBg5KlD]]Fg5;LE_aBfeKlD]aFo5;KE]aBg5KlD]]Fo5;LE_aBfeKlD]aFo000E]aB
g5KlD]]Fo5;LE_aBfeKLD]aFo5;KE_aBg7oo000WOol0039oo`<000Qoo`1AD]]Fo5;KE_aBfeKlD]]F
o5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]F
o5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;K001BfeKlD]]F
o5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]oo`0001=oo`80015oo`00<7oo0P0027oo0eKK00EFo5KKE]]F
feKl00=Ff`05E_aFfeKKE]]Fo003E]/0@5KlE]]Fg5KKE_aBfeKLE]]Fo5;KE]aFfeKlD]]Fg5KKE_aB
feKLE]]Fo5;KE]aFfeKlD]]Fg5KKE_aBfeKLE]]Fo5;KE]aBg5KlD]]Fg5;LE_aBfeKLD]aFo5;KE]`0
05KlD]]Fg5;LE_aBfeKLD]aFo5;KE]aBg5KlD]]Fg7oo0P004Goo00@007ooOol0011oo`004Goo1@00
5goo0`001goo05IBfeKlE]]Fo5;KE_aFfeKlD]]Fo5KKE_aBfeKlE]]Fo5;KE_aBfeKlD]]Fo5;KE_aB
feKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aB
feKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]/005;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aB
feKl0P004Goo00<007ooOol04Goo000COol00`00Oomoo`0EOol20009Ool00eKKE_]Ff`02E]/01EKk
E]]FfeKKE_/00eKK00EFo5KKE]]FfeKl00=Ff`05E_aFfeKKE]]Fo003E]/0>UKlE]]Fg5KKE_aBfeKL
E]]Fo5;KE]aFfeKlD]]Fg5KKE_aBfeKLE]]Fo5;KE]aFfeKlD]]Fg5KKE_aBfeKLD]]Fo5;KE]aBg5Kl
D]]Fg5;LE_aBf`00D]aFo5;KE]aBg5KlD]]Fg5;LE_aBfeKLD]aFo5;K0Woo00<007ooOol00goo1@00
0Woo0P0000=oo`0000000`0047oo000COol00`00Oomoo`0BOol30009Ool0G5KlE]]Fo5KKE_aFfeKl
E]]Fo5;KE_aFfeKlD]]Fo5KKE_aBfeKlE]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKl
D]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKl
D]]Fo5;KE_aBfeKl001Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfgoo0007Ool01@00Oomo
ogoo00000Woo00<007oo00004goo000COol00`00Oomoo`0?Ool3000:Ool3E]/01EKkE]]FfeKKE_/0
0eKK00EFneKKE]]FfeKk00=Ff`05E_]FfeKKE]]Fo003E]/01EKlE]]FfeKKE_`00eKK00EFo5KKE]]F
feKl00=Ff`0bE_aBfeKLE]]Fo5;KE]aFfeKlD]]Fg5KKE_aBfeKLE]]Fo5;KE]aFfeKlD]]Fg5KKE_aB
feKLD]]Fo5;KE]aBf`00D]]Fg5;LE_aBfeKLD]]Fo5;KE]aBfeKlD]]Fg5;LE_aBf`8000Ioo`05001o
ogooOol00002Ool00`00Oomoo`0COol001=oo`03001oogoo00eoo`<000Uoo`1RE]]Jo5KKE_]FfeKl
E]]FneKKE_aFfeKlE]]Fo5KKE_]BfeKlE]]Fo5;KE_aFfeKlD]]Fo5KKE_aBfeKlE]]Fo5;KE_aBfeKl
D]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKl
D]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKl001Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKl
D]/000Ioo`05001oogooOol00002Ool00`00Oomoo`0COol0019oo`8000aoo`<000]oo`UFf`05E_]F
feKKE]]Fn`03E]/01EKkE]]FfeKKE_/00eKK00EFneKKE]]FfeKl00=Ff`05E_aFfeKKE]]Fo003E]/0
1EKlE]]FfeKKE_`00eKK00IFo5KKE]]FfeKlD]/2E]/0:eKlD]]Fg5KKE_aBfeKLE]]Fo5;KE]aFfeKl
D]]Fg5KKE_aBfeKLE]]Fo5;K001FfeKlD]]Fg5;KE_aBfeKLD]]Fo5;KE]aBfeKlD]]Fg5;KE_aoo`00
00Eoo`05001oogooOol00002Ool00`00Oomoo`0COol001ioo`800003Ool007oo00Uoo`1UF_]Ffe[k
E]]JneKKF_]FfeKkE]]Jo5KKE_]FfeKlE]]FneKKE_aFfeKlE]]Fo5KKE_aBfeKlE]]Fo5;KE_aFfeKl
D]]Fo5KKE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKl
D]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKl001Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKl
D]]Fo5;KE_aBfeKlD]]Fo0020003Ool20004Ool2000FOol001]oo`<000]ooaQFf`03E_]FfeKK00UF
f`05E_]FfeKKE]]Fn`03E]/01EKlE]]FfeKKE_`00eKK00EFo5KKE]]FfeKl00=Ff`06E_aFfeKKE]]F
o5;K0UKK00mFo5;KE]]FfeKlD]]Fg5KKE_aBfeKLE]]Fo5;KE]`00P0001EBfeKLE]]Fo5;KE]aFfeKl
D]]Fg5;KE_aBfeKLD]]Fo5;KE]aBfgoo00008Goo000IOol2000;Ool0De[kE]YJneKJF_]FfU[kE]]J
neKJF_]FfeKkE]]JneKKE_]Ffe[kE]]FneKKF_]FfeKkE]]Fo5KKE_]FfeKlE]]FneKKE_aFfeKlD]]F
o5KKE_aBfeKlE]]Fo5;KE_aFfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]F
o5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_`00`0001QFo5;K001BfeKlD]]Fo5;KE_aBfeKlD]]F
o5;KE_aBfeKlD]]Fo5;KE_aBfgoo000POol001Moo`8000aoo`03F]]Ffe[K00=Ff`03F]]FfeKK03EF
f`05E_aFfeKKE]]Fg003E]/01UKlE]]FfeKKE_aBf`9Ff`03E]aBfeKK00<0009Ff`0GE_aBfeKL001F
o5;KE]aFfeKlD]]Fg5KKE_aBfeKLE]]Fo5;KE]aFfeKlD]]Fg002Ool00`00Oomoo`0MOol001Moo`80
00]oo`1?F]]FfU[kE]YJneKJF_]FfU[kE]YJneKJF_]FfU[kE]]JneKJF_]FfeKkE]]JneKKE_]Ffe[k
E]]FneKKE_aFfeKkE]]Fo5KKE_]FfeKlE]]FneKKE_aFfeKkD]]Fo5KKE_aBfeKlE]]Fo5;KE_aBfeKl
D]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_`00`0001mFo5;KE_aBfeKl
D]]Fo5;KE_`005KlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlOol0000OOol0015o
o`8000Aoo`<000Moo`0=F]]FfU[KF]YJfeKJF]]Ffe[KE]]JfeKKF]/0?eKK00=Fg5KKE]/00`000UKK
00IFg5;KE]]FfeKLD]/2E]/01UKLD]]Ff`00E]aBf`9Ff`0:E_aBfeKLE]]Fg5;KE]aFfeKlD]/2E]/0
1EKlD]]Fg7oo00007Woo000@Ool01000Oomoo`000goo00@007ooOol000Eoo`1>F_]FfU[KE]YJneKJ
F]]FfU[kE]YJfeKJF_]FfU[KE]YJneKJF]]FfU[kE]]JneKJF_]Ffe[kE]YJneKKE_]FfU[kE]]FneKK
F_]FfeKkE]]JneKKE_]FfeKkE]]FneKKE_aFfeKkE]]Fo5KKE_aBfeKlE]]Fo5;KE_aFfeKlD]]Fo5KK
E_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;K0P0002MFo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aB
feKl001Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo7oo00007Goo000@Ool01000
Oomoo`000Woo0P000goo00D007ooOomJfU[K00=JfP0AF]]FfU[KF]YJfeKJF]]Ffe[KE]YJfeKKF]]F
fe[KE]]Jf`0gE]/30005E]/00eKLE]]Ff`05E]/02UKLE]]FfeKKE]aBf`00E]]Fg5;K0UKK00IFg5;K
E]]FfeKLD]/2E]/025KLD]]FfeKKE]aBfgoo000MOol0011oo`04001oogoo0002Ool20003Ool0CE[J
F]]FfU[KE]YJfeKJF_]FfU[KE]YJneKJF]]FfU[kE]YJfeKJF_]FfU[KE]YJneKJF]]FfU[kE]YJfeKJ
F_]Ffe[KE]YJneKKF]]FfU[kE]]FneKKF_]FfeKkE]]FneKKE_]Ffe[kE]]FneKKE_aFfeKkE]]Fo5KK
E_]BfeKlE]]Fne;KE_aFfeKkD]]Fo5KKE_aBfeKlE]/00`0002eBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;K
E_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;K001BfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;K
E_aBfgoo000077oo000@Ool01000Oomoo`000Woo0P000Woo2e[J015Jfe[JF]YJfU[KE]YJfeKJF]]F
fU[KE]]JfeKJF]]Ffe[K00=Ff`03F]]FfeKK02UFf`<001UFf`06E]aBf`00E]]Fg5;K0UKK00IFg5;K
E]]FfeKLD]/2E]/01EKLD]]FfeKKE]`00Woo00<007ooOol06Goo000AOol20003Ool20000;GooF]YJ
fU[JF]]JfU[KF]YJfe[JF]]FfU[KE]YJfeKJF]]FfU[KE]YJfeKJF]]FfU[kE]YJfeKJF_]FfU[KE]YJ
neKJF]]FfU[kE]YJfeKJF_]Ffe[KE]YJn`02E]/01UKJF_]FfeKKE]YJn`9Ff`03E]YFneKK009Ff`0>
E_]FfeKKE]]Fo5KKE_]FfeKlE]]Fne;KE_aFf`<0000dE]]Fne;KE_aFfeKkD]]Fo5KKE_aBfeKlD]]F
o5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]/005;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]F
o5;KE_aBfeKlD]]Fo7oo000KOol000Uoo`D0009oo`8000Aoo`03001ooe[J00iJfP05F]]JfU[JF]YJ
f`03F]X02U[KE]YJfeKKF]]FfU[KE]]JfeKJ0UKK00=JfeKJE]/08EKK0`008UKK00<005KKE]/015KK
00IFg5;KE]]FfeKLD]/2E]/015KLD]]FfeKK0Woo00<007ooOol00goo1@000Woo0P0037oo000;Ool0
1@00Oomoogoo00000Woo00T007ooOomoo`00F]YNfU[IG]X00e[J00=Jfe[JF]X00e[J02]JfeKJF]]J
fU[KE]YJfeKJF]]FfU[KE]YJfeKJF]]FfU[KE]YJneKJF]]FfU[KE]YJfeKJF_]FfU[KE]YJneKJF]]F
fU[kE]]JfeKJF_]Ffe[KE]YJn`02E]/01UKJF_]FfeKKE]YJn`=Ff`03E_]Ff`000080009Ff`06D]]F
o5KKE]]BfeKl0UKK00IBfeKlE]]Fne;KE_`2E]/09e;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aB
feKlD]]Fo5;KE_`005KlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBfeKlD]]Fo5;KE_aBf`02Ool00`00Oomo
o`04Ool01@00Oomoogoo00000Woo00<007ooOol02Goo000;Ool01@00Oomoogoo00000Woo00D007oo
Oomoo`0001IJfP0BF]]JfU[JF]YJfeKJF]YFfU[KE]YJfeKJF]]FfU[KE]]JfeKJ0UKK00=JfeKKE]/0
5EKK0`00:UKK00<005KKE]/025KK00=Fg5;KE]/00UKK00EBfeKKE]]oo`0000Ioo`03001oogoo00Ao
o`03001oogoo00Uoo`002goo00D007ooOomoo`00009oo`03001oogoo00800009OomNfU[IG]YJ^EkJ
F]YNfUZi00AJfP05F]]JfU[JF]YJf`02F]X09UKJF]]FfU[KE]YJfeKJF]]FfU[KE]YJfeKJF]]FfU[K
E]YJfeKJF]]FfU[KE]YJfeKJF_]FfU[KE]YJneKJF]]FfU[kE]YJfeKJF_/2E]/00eKJF_]Ff`030003
E]/01EKkE]]FfeKKE_/00eKK00EFneKKE]]FfeKk009Ff`06D]]FneKKE]]BfeKl0UKK00IBfeKlE]]F
fe;KE_`2E]/08E;KE_aBfeKKD]]Fo5;KE]aBfeKlD]/005;KE_aBfeKLD]]Fo5;KE_aBfeKlD]]Fo5;K
E_aBfeKLD]]Fo5;KOol00005Ool00`00Oomoo`02Ool2000<Ool000]oo`05001oogooOol00002Ool0
0`00Oomoo`020002Ool2F]X00ekJF]YJfP0IF]X02U[KE]YJfeKJF]]FfU[KE]YJfeKJ0UKK00IJfeKJ
E]]Ffe[KE]X3E]/00eKJE]]Ff`05E]/2000aE]/00`00E]]Ff`0@E]/2Ool00`00Oomoo`02Ool00`00
Oomoo`02Ool00`00Oomoo`0;Ool000Yoo`8000Aoo`8000=oo`8000=oo`09G]YJfEkJF[UNfU[JG]YJ
^EkJ009JfP06F[YJfU[JF]YJ^U[K0U[J02AJ^U[KE]YJfeJjF]]FfU[KE]YJfeKJF]]FfU[KE]YJfeKJ
F]]FfU[KE]YJfeKJF]]FfU[KE]YJfeKJF]]FfU[KE]YJfeKJF]/300002eKKE]YJneKJE]]FfU[KE]YF
feKJF_/00UKK00=FfUKkE]/00UKK00EFneKKE]]FfeKk00=Ff`05E_]FfeKKD]]Fo002E]/01U;KE_aF
feKKD]]Fo09Ff`0ND]]Fo5KKE]]BfeKlD]]Ffe;KE_aBf`00D]]Fo5;KE]aBfeKlD]]Fg5;KE_aBfeKL
D]]Fo5;KE]aBfeKl0P000goo0P0017oo0`002goo000EOol20004Ool2F]X00ekJF]UJfP0IF]X03U[K
E]YJfU[JF]]FfU[JE]YJfeKJF]]FfU[KE]X2E]/01E[KE]YFfeKKF]/00`00>5KK00<005KKE]/03eKK
0Woo00<007ooOol057oo0008Ool40003Ool20004Ool00`00Oomoo`03Ool02UZiG]YJfEkJF[UNfU[I
G]YJ^EkJ0U[J00EJ^EkJF]YJfUZi00=JfP0LF[YJfe[JF]YF^U[KE]YJfeJjF]]FfU[KE[YJfeKJF]]F
^U[KE]YJfeKJF]]FfU[KE]YJfeKJF]/300004e[KE]YJfeKJF]]FfU[KE]YJfeKJF]]FfU[KE]YJfeKJ
E]]FfU[K009Ff`06E]YJfeKKE]]FfUKk0UKK00IFfUKkE]]FfeKJE_/3E]/01EKkE]]Ffe;KE_`00UKK
00IBfeKkE]]Ffe;KE_/2E]/06U;KE_aFfeKKD]]Fo000E]]BfeKlD]]Ffe;KE_aBfeKLD]]Fo5;KE]]B
feKlD]]Ffe;KOol2000EOol000Qoo`03001oogoo00=oo`04001oogoo0003Ool00`00Oomoo`04Ool0
1E[IG]YJfEkJF]T00U[J00=NfU[JF]X05U[J00EFfU[JF]YJfUKJ009JfP05F]]FfU[KE]YJf`030000
1U[KE]YFfeKJF]]FfP=Ff`03E]YFfeKK03=Ff`03001FfeKK00mFf`9oo`03001oogoo01=oo`002Goo
00<007ooOol00Woo00@007ooOol000=oo`03001oogoo00Eoo`0AF[UNfUZiG]YJ^EkJF[UNfUZiG]YJ
^EkJF[UNfUZiF]YJ^P03F]X01UZiF]YJfU[JE[YJf`9JfP0<E[YJfeKJF]YF^U[KE]YJfeJjF]]FfU[K
0`0001mJfeKJF]]FfU[KE]YJfeKJF]]FfU[KE]YJfeKJF]]FfU[KE]YJfeKJF]]FfU[KE]YFfeKJF]]F
fUKKE]YJf`02E]/01EKJE]]FfeKKE]X00eKK00EFfUKKE]]FfeKJ00=Ff`05D]]FneKKE]]Bf`03E]/0
1U;KE]]FfeKKD]]Fn`9Ff`06D]]Ff`00E]]BfeKl0UKK00aBfeKlD]]Ffe;KE_aBfeKKD]]Fo5;KE]/2
Ool00`00Oomoo`0BOol000Yoo`05001oogooOol00002Ool01000Oomoo`002Goo00QJfEkIF]UNfU[I
F]YJfEkJ15[J00=JfE[JF]X05E[J0`0001IJfUKJF]YFfU[KE]YJfUKJF]]FfUKKE]YJfeKJE]]FfU[K
E]YFfeKJE]]FfP=Ff`05E]YFfeKKE]]FfP0YE]/00`00E]]Ff`0>E]/2Ool2000COol000Qoo`04001o
ogoo0002Ool01000Oomoo`000Woo0P002Goo021J^EkJF[UNfUZiG]YJ^EkJF[UNfUZiG]YJ^EkJF[UN
fUZiG]YJ^E[JF[YJfUZjF]YF^U[JF[YJfUJjF]YJ^U[J0`00039JfeJjF]]FfU[KE[YJfeKJF]]F^U[K
E]YJfeJjF]]FfU[KE[YJfeKJF]]FfU[KE]YJfeJjF]]FfU[KE[YJfeKJF]]FfU[KE]YFfeKJF]]FfUKK
E]YJfeKJE]]FfU[KE]YFfeKJ0eKK00EFfUKKE]]FfeKJ00=Ff`05E]YFfeKKE]]FfP03E]/01U;KE]]F
feKKD]/0009Ff`05D]]FfeKKE]]Bf`03E]/01e;KE]aBfeKKD]]Fg5;K009oo`03001oogoo015oo`00
2Goo0P0017oo0P000goo0P002Woo00aJ^EkIG]UNfU[IF]YJfEkJF]UJfU[IG]X@F]X30005F]X01EKJ
F]YJfU[JE]X00U[J00iJfeKJF]YFfU[KE]YJfUKJF]]FfUKKE]YJfeKJ0eKK00EFfUKKE]]FfeKJ00=F
f`03E]YFfeKK02IFf`03001FfeKK00iFf`=oo`03001oogoo011oo`0057oo00<007ooOol02Woo01UJ
^EkJF[UNfUZiG]YJ^EkJF[UNfUZiG]YJ^EkJF[UNfUZiG]YJ^EkJF[UJfUZjF]YJ^@020000AU[JE[YJ
fUZjF]YF^U[KE[YJfUJjF]]F^U[KE[YJfeJjF]]F^U[KE]YJfeJjF]]FfU[KE[YJfeKJF]]F^U[KE]YJ
feJjF]]FfU[KE[YJfeKJF]]F^U[KE]YJfeJjF]]FfU[KE[YJfeKJE]]F^U[KE]YFfeKJF]]FfUKKE]YF
feKJE]]FfUKKE]YFfeKJ0eKK00EFfUKKE]]Ffe;J00=Ff`05D]/005KKE]]Bf`03E]/01E;KE]]FfeKK
D]/00eKK00ABfeKKOomoo`80015oo`0017oo100037oo0P002goo019NfEZiG]UN^EkJF[UNfEZiG]UJ
^E[JF[UNfUZiF]YJfE[JF[T4F]X3000DF]X01EKJF]YFfU[KE]X00U[J00UJfeKJF]YFfU[KE]YJfUKJ
F]/00eKJ00IJfeKJE]]FfUKKE]X3E]/01EKJE]]FfeKKE]X07eKK00<005KKE]/03UKK0goo00<007oo
Ool00goo10000goo0P000goo0003Ool01@00Oomoo`00Ool00P0017oo0P0017oo00<007ooOol02goo
019NfEZiG]YJ^EkJF[UNfUZiG]YJ^EkJF[UNfUZiG]YJ^EkJF[T30000DEZiG]YJ^E[JF[UJfUZiF]YF
^U[JF[YJfUJjF]YJ^U[JE[YJfUJjF]YF^U[KE[YJfeJjF]]F^U[KE[YJfeJjF]]F^U[KE]YJfeJjF]]F
fU[KE[YJfeKJF]]F^U[KE]YJfeJjF]]FfU[KE[YJfeKJF]]F^U[KE]YFfeJjF]]FfUKKE[YJfeKJE]]F
^U[KE]YFfeJjF]]FfUKKE[YFfeKJE]]F^P03E]/01EJjE]]FfeKK00000eKK00EBfUKKE]]Ffe:j00=F
f`05D]YFfeKKE]]BfP03Ool00`00Oomoo`02Ool00`00Oomoo`03Ool01000Oomoo`000Woo0003Ool0
0`00Oomoo`020002Ool01000Oomoo`000Woo00D007ooOomoo`0000eoo`0?G[UNfEZiG]UN^EkIF[UN
fEjiG]YJ^EkIF[UNfUZi00<00005F[UJfU[IF]YJ^P03F]X01EZjF]YJfU[JF[X00e[J00=J^U[JF]X0
3E[J00QFfU[JE]YJfUKJF]YFfU[J0eKJ01IJfeKJE]YFfU[KE]YFfeKJF]]FfUKKE]YFfeKJE]]FfUKK
E]YFfeKJE]]FfP=Ff`03E]YFfeKK01EFf`03001FfeKK00iFf`9oo`8000Aoo`03001oogoo009oo`04
001oogoo0002Ool0009oo`800003Ool00000009oo`05001oogooOol00002Ool00`00Oomoo`02000>
Ool02eZiG]UJ^EkIF[UNfEZiG]YJ^EkJF[T00`0006UJ^EkJF[UNfUZiG]YJ^E[JF[UNfUZiF]YJ^EkJ
F[UJfUZiF]YJ^E[JE[UJfUZjF]YF^U[JE[YJfUJjF]YF^U[JE[YJfeJjF]YF^U[KE[YJfUJjF]]F^U[K
E[YJfeJjF]]F^U[KE[YJfeJjF]]FfU[KE[YJfeJjF]]F^U[KE[YJfeJjF]]FfU[KE[YJfeKJE]]F^U[K
E]YFfeJjF]]FfUKKE[YJfeKJE]]F^UKKE]YFfeJjE]]FfUKKE[YFfeKJE]/005KKE]YFfeJjE]]FfUKK
E[X00eKK00EB^UKKE]]Ffe:j009oo`03001oogoo00=oo`05001oogooOol00002Ool10001Ool1Ool0
00Yoo`04001oogoo0002Ool00`00Oomoo`02000?Ool00ejiG]UN^@02G[T00ekIF[T0000200004EZi
G]UJ^EkIF[UJfEZiG]YJ^E[IF[UJfUZiF]YJ^E[JF[T00e[J00EJ^U[JF]YJfUZj00=JfP05F[YJfU[J
F]YJ^P03F]X01EZjF]YJfU[JF[X00e[J00QFfU[JE]YJfUKJF]YFfU[J0eKJ01IJfeKJE]YFfU[KE]YF
feKJE]]FfUKKE]YFfeKJE]]FfUKKE]YFfeKJE]]FfP=Ff`05E]YFfeKKE]]FfP0>E]/00`00E]]Ff`0>
E]/2Ool20002Ool01000Oomoo`000Woo00@007ooOol0009oo`001goo00@007ooOol0009oo`04001o
ogoo0002Ool00`00Oomoo`0?Ool00eZiG]UJ^0030000NeZiG]UJ^EkJF[UNfUZiG]YJ^EkJF[UNfUZi
G]YJ^EkJF[UNfUZiF]YJ^EkJF[UJfUZiG]YJ^E[JF[UJfUZjF]YF^E[JE[YJfUJiF]YF^U[JE[YJfUJj
F]YF^U[KE[YJfUJjF]]F^U[KE[YJfeJjF]]F^U[KE[YJfeJjF]]F^U[KE[YJfeJjF]]F^U[KE[YJfeJj
F]]F^U[KE[YJfeKJF]]F^U[KE[YFfeJjF]]F^UKKE[YJfeKJE]]F^U[KE[YFfeJjE]]FfUKKE[YFfeKJ
001F^UKKE]YFfeJjE]]FfUKKE[YFfeKJE]]B^UKKE]YFf`00Ool00003Ool20004Ool20003Ool000Qo
o`8000Aoo`8000=oo`80011oo`<000=N^@0=G]UJ^EjiG[UNfEZiG[UJ^EkIF[UN^EZiG]X00eZi00YJ
fUZiF[UJ^E[JF[UJfUZjF]YJ^@=JfP05F[UJfU[JF]YJ^P03F]X01EZjF]YJfU[JF[X00e[J00EJ^U[J
F]YJfUJj00=JfP0<E]YJfUKJF]YFfU[JE]YJfUKJF]YFfU[K0eKJ00EJfeKJE]YFfU[K00=FfP0BE]]F
fUKKE]YFfeKJE]]FfUKKE]YFfeKJE]]FfUKKE]YFfeKJ0eKK00EFfUKKE]]FfeKJ009Ff`03001FfUKK
009Ff`03E]YFfeKK00QFf`9oo`03001oogoo00aoo`004goo0P003Goo0`000Woo07mN^5kIF[QNfEZi
G]UJ^5kIF[UNfEZhG]UJ^EkIF[UNfUZiG]YJ^EkJF[UNfUZiG]YJ^E[JF[UNfUZiF]YJ^E[JF[UJfUZi
F]YJ^E[JF[UJfUZjF]YF^E[JE[YJfUJiF]YF^U[JE[YJfeJjF]YF^U[KE[YJfUJjF]]F^U[KE[YJfeJj
F]]F^U[KE[YJfeJjF]]F^U[KE[YJfeJjF]]F^U[KE[YJfeJjF]]F^U[KE[YJfeJjF]]F^U[KE[YJfeJj
F]]F^UKKE[YJfeJjE]]F^U[KE[YFfeJjF]/005KKE[YFfeJjE]]F^UKKE[YFfeJjE]]FfUKKE[YFfeJj
Ool0000>Ool000Qoo`<0009oo`8000Aoo`03001oogoo00Yoo`8000Ioo`UN^@0=G]UJ^EjiF[UNfEZi
G[UJ^EkIF[UN^EZiF]X00eZi01IJfUZiF[UJ^E[JF[UJfUZjF]YJ^E[JF[YJfUZjF]YJ^U[JF[YJfUZj
F]YJ^P=JfP05F[YJfU[JF]YF^P03F]X03EJjF]YFfU[JE[YJfUKJF]YF^U[JE]YJfUJj009FfP06F]YF
^UKJE]YJfUJj0UKJ01iFfeKJE]YFfUKKE]YFfeKJE]]FfUKKE]YFfeKJE]]FfUKKE]YFfeKJE]]FfUKK
E]YFfeKJE]/005KKE]X3E]/01EKJE]]FfeKKE]X01UKK00=oo`00Ool037oo0009Ool01000Oomoo`00
0Woo00<007ooOol00P002Goo0`0027oo07mNfEjhG]UJ^5kIF[UNfEZhG]UJ^EkIF[QNfEZiG]UJ^EkJ
F[UNfEZiG]YJ^EkJF[UNfUZiG]YJ^EkJF[UJfUZiG]YJ^E[JF[UJfUZiF]YJ^E[JF[UJfUZiF]YJ^E[J
E[UJfUZjF]YF^U[JE[YJfUJjF]YF^U[JE[YJfUJjF]YF^U[JE[YJfUJjF]]F^U[JE[YJfeJjF]]F^U[K
E[YJfeJjF]]F^U[KE[YJfeJjF]]F^U[KE[YJfeJjF]]F^U[KE[YJfeJjE]]F^U[KE[YFfeJjF]]F^UKK
E[YJf`00E]]F^UKKE[YFfeJjE]]F^UKKE[YFfeJjE]]F^UKKOol0000=Ool000Ioo`D00005Ool007oo
Ool00002Ool20006Ool3000<Ool015jiG[QN^Ejh0eji00=N^5jiG[T00eji00YJ^EjiF[UN^EZiG]UJ
^EjiF[UNf@=J^@0bF]UJ^EZiF[UJfUZiF]YJ^E[JF[UJfUZjF]YJ^E[JF[YJfUZiF]YJ^U[JF[YJfUZj
F]YJ^U[JF[YJfUZjF]YJ^U[JE[YJfUZjF]YF^U[JE[YJfUJjF]YFfU[JE[YJfUKJF]YF^P9FfP06F]YF
^UKJE]YJfUJj0UKJ00IFfeJjE]YFfUKKE[X2E]X05UKKE[YFfUKJE]]FfUKKE]YFfeJjE]]FfUKKE[YF
f`00E]]F^UKKE]YFfeKJ0eKK00MFfUKKE]YFfeJjE]/0000=Ool000Ioo`04001oogoo0002Ool01000
Oomoo`000Woo00<007ooOol00Woo0`0047oo07eNfEjhG[UN^5kIF[QN^EZhG]UJ^5kIF[QNfEZiG]UJ
^5kIF[UNfEZiG]UJ^EkIF[UNfUZiG]UJ^EkJF[UNfUZiG]YJ^E[JF[UNfUZiF]YJ^E[JF[UJfUJiF]YJ
^E[JE[UJfUZiF]YF^E[JF[UJfUJiF]YF^U[JE[UJfUJjF]YF^E[JE[YJfUJjF]]F^U[JE[YJfeJjF]YF
^U[KE[YJfUJjF]]F^U[KE[YJfeJjF]]F^U[KE[YJfeJjF]]F^U[KE[YJfeJjF]]F^U[KE[YJfeJjF]]F
^UKKE[X005JjE]]F^U[KE[YFfeJjF]]F^UKKE[YJfeJjE]]F^P0000eoo`001goo00<007oo00000Woo
00@007ooOol0009oo`D001=oo`=N^004G[UN^5jiG[P5G[T025jhG[UN^EjiF[UN^EZiG[T3F[T01Eji
F[UJ^EZiG[T02EZi00]J^UZiF]YJ^EZjF[UJfUZiF]YJ^E[J00=J^P05F]YJ^UZjF[YJfP03F[X09U[J
F[YJfUZjF]YF^U[JF[YJfUJjF]YF^U[JE[YJfUJjF]YF^U[JE[YJfUJjE]YF^U[JE[YFfUJjF]YF^UKJ
E[YFfUJjE]YF^UKJE[X2E]X01UKKE[YFfUKJE]]F^P9FfP06E]]F^P00E]YFfeJj0UKJ00IFfeJjE]]F
fUKKE[X3Ool00`00Oomoo`0;Ool000Qoo`8000=oo`8000=oo`8001Moo`1fG[QNfEjhG[UJ^5kIG[QN
^EZhG]UJ^5jiF[QNfEZhG[UJ^5kIF[UN^EZiG]UJ^EjiF[QNfUZiG]UJ^EkJF[UNfUZiG]YJ^EkJF[UN
fUZiF]YJ^E[JF[UJfUZiF]YJ^E[JF[UJfUZiF]YF^E[JF[UJfUJiF]YF^E[JE[UJfUJjF]YF^E[JE[YJ
fUJjF]YF^U[JE[YJfeJjF]YF^U[KE[YJfUJjF]]F^U[KE[YJfeJjF]]F^U[KE[YJfeJjF]]F^U[KE[YJ
feJjF]]F^U[KE[YJfeJjF]]F^U[KE[X005JjE]]F^U[KE[YFfeJjF]/5Ool00`00Oomoo`0;Ool001=o
o`03001oogoo01Ioo`EN^008G[UN^5jiG[QN^EjhG[UJ^@=N^@08F[UN^EjiG[UJ^EjiF[UN^@=J^@05
G[UJ^EZiF[UN^@07F[T01EZjF[UJ^EZiF]X00eZi00IJfUZiF[YJ^U[JF[T2F[X01E[JF[UJ^UZjF]X0
0eZj039JfUZjF[YJ^U[JF[YJfUZjF]YF^U[JE[YJfUJjF[YF^U[JE[YJfUJjF]YF^UKJE[YJfUJjE]YF
^U[JE[YFfUJjF]YF^UKJE[YFfUJjE]YF^UKJE[YFfUJjE]]F^P00E[YFfeJj0UKJ00=FfgooOol017oo
00<007ooOol02goo000DOol2000GOol0LEjhH[UN^5jiF[QN^EjhG[UJ^5jiG[QN^EZhG]UJ^5jiF[QN
fEZhG[UJ^5kIF[QN^EZiG]UJ^EkIF[UNfEZiG]UJ^EkJF[UJfUZiG]YJ^E[JF[UNfUZiF]YJ^E[JF[UJ
fUZiF]YJ^E[JF[UJfUZiF]YF^E[JF[UJfUJiF]YF^U[JE[UJfUJjF]YF^E[JE[YJfUJjF]YF^U[JE[YJ
fUJjF]YF^U[JE[YJfUJjF]YF^U[JE[YJfeJjF]YF^U[KE[YJfUJjF]]F^U[JE[YJfeJjF]]F^U[KE[YJ
f`00F]]F^UKKE[X02Goo00<007ooOol02Woo000FOol00`00Oomoo`0DOol4G[P00ejiG[QN^003G[P0
15jiG[QN^Ejh0eji00EJ^5jiG[UN^EZi00=N^@04F[UN^EZiG[T3F[T00ejiF[UJ^@09F[T03UZjF[UJ
^EZiF[YJ^EZjF[UJ^UZiF[YJ^EZjF[T2F[X01E[JF[YJ^UZjF]X00eZj00IJfUJjF[YJ^U[JE[X2F[X0
3E[JE[YJ^UZjF]YF^UZjE[YJfUJjF[YF^U[J00=F^P0=F]YF^UJjE[YJfUJjE]YF^U[JE[YFfUJjE]X0
0eJj00=FfP00Ool02goo00<007ooOol02Woo000GOol00`00Oomoo`0DOol00ejhGYQR^003G[P0I5ji
G[QN^EZhG[UN^5jiF[QN^EZhG[UJ^5kIF[QN^EZhG]UJ^5jiF[QNfEZiG[UJ^5kIF[UN^EZiG]UJ^EkI
F[UNfEZiF]YJ^EkJF[UJfUZiG]YJ^E[JF[UJfUZiF]YJ^E[JF[UJfUJiF]YJ^E[JE[UJfUZiF]YF^E[J
E[UJfUJiF]YF^E[JE[UJfUJiF]YF^E[JE[YJfUJiF]YF^U[JE[UJfUJjF]YF^U[JE[YJfUJjF]]F^U[J
E[YJfeJjF]YF^U[KE[YJfUJj0goo00<007ooOol02Goo00<007ooOol02Woo000HOol00`00Oomoo`0D
Ool8G[P01EjiG[QN^5jhG[T00ejh00IN^EjhG[UN^5jiF[P3G[T025ZiG[UJ^EjiF[UN^EZiG[TCF[T0
3UZjF[UJ^UZiF[YJ^EZjF[UJ^UZiF[YJ^EZjF[T7F[X055JjF[YJ^UZjE[YJ^UJjF]YF^UZjE[YJfUJj
F[YF^UZjE[YJ^UJjF]X3E[X01E[JE[YF^UJjF]X00UJj1Goo00<007ooOol02Goo00<007ooOol02Woo
000IOol00`00Oomoo`0COol015jhH[QNUf:h0Ujh02INV6:hG[QN^EZhG[UN^5jiF[QN^EjhG[UJ^5ji
F[QN^EZhG[UJ^5jiF[QN^EZhG[UJ^5jiF[UN^EZhG[UJ^EjiF[QN^EZiG[UJ^EkJ0eZi03ENfUZiF[UJ
^E[JF[UJfUZiF]YJ^EZjE[UJfUZiF[YF^E[JF[UJ^UJiF]YF^E[JE[UJfUJiF]YF^E[JE[UJfUJiF]YF
^E[JE[UJfUJjF]YF^E[JE[YJfUJiF]YF^U[JE[YJfUJjF]YF^U[J00Uoo`03001oogoo00Qoo`03001o
ogoo00Yoo`006Woo0P0057oo4Ejh00=N^EjhG[P00Uji00aN^5jiF[UN^EZhG[UJ^EjiF[UN^EZiG[T3
F[T00ejiF[UJ^@0EF[T01EZjF[UJ^EZiF[X00eZi00=J^UZiF[X00UZj00EJ^EZjF[YJ^UZi00=J^P0@
E[YJ^UZjF[YF^UZjE[YJ^UJjF[YF^UZjE[YJ^UJjF]X;Ool00`00Oomoo`07Ool00`00Oomoo`0:Ool0
01aoo`03001oogoo015oo`05H[QN^6:hGYMR^002G[P01EjGH[QN^5jhGYP00Ujh02=N^EZHG[UJ^5ji
FYQN^EZhG[UJV5jiF[QN^EZHG[UJ^5jiFYQN^EZiG[UJV5jiF[UN^EZhG[UJ^EjiF[UN^EZiG[UJ^EkJ
00=J^@0/G]YJ^EZjF[UJfUZiF[YJ^E[JF[UJ^UJiF]YJ^EZjE[UJfUZiF[YF^E[JE[UJ^UJiF]YF^E[J
E[UJfUJiF[YF^E[JE[UJ^UJiF]YF^E[JE[UJfUJjF]YF^@ioo`03001oogoo00Moo`03001oogoo00Uo
o`007Goo00<007ooOol04Goo4Ujh00EN^EjhG[QN^5ji00=N^00:G[UJ^5jiF[UN^EZhG[UJ^EjiF[P2
F[T01EjiF[UJ^EZiG[T05eZi00iJ^UZiF[UJ^EZjF[UJ^UZiF[YJ^EZjF[UJ^UJi0eZj00EF^EZjF[YJ
^UJi00=J^P03E[YJ^Woo00moo`03001oogoo00Moo`03001oogoo00Uoo`007Woo00<007ooOol04Goo
03ER^5jGH[QNUf:hGYQN^5jGG[QNV5jhGYMN^EjHG[QJV5jiGYQN^5ZHG[UJ^5jiFYQN^EZhG[UJV5ji
F[QN^EZHG[UJ^5jiFYQN^EZhG[UJV5jiF[QN^EZHG[UJ^EjiF[UN^EZiG[UJ^Ejj00=J^@0QG[YJ^EZi
F[UJ^UZiF[YJ^E[JF[UJ^UJiF[YJ^EZjE[UJfUJiF[YF^E[JE[UJ^UJiF]YF^EZjE[UJfUJiF[YF^E[J
01Aoo`03001oogoo00Ioo`03001oogoo00Uoo`007goo0P004Woo00AN^5jgH[QNU`=N^005GYMN^5jh
G[QNV003G[P00ejHG[QN^008G[P02EjiF[QN^EjhG[UJ^5jiF[QN^@03F[T01EjiF[UJ^EZiG[T06eZi
00EJ^UZiF[UJ^EZj00=J^@07F[YJ^EZjF[UJ^UZiF[X067oo00<007ooOol01Goo00<007ooOol02Goo
000QOol00`00Oomoo`0?Ool0=UjGH[QNUf:hGYMR^5jGH[QNUf:hGYMN^5jGG[QNV5jhGYMN^5jHG[QJ
V5jiFYQN^5ZHG[UJV5jiFYQN^EZhG[UJV5jiF[QN^EZHG[UJ^5jiFYQN^EZhG[UJV5jiF[QN^EZHG[UJ
^5ZiFYQN^@9J^@06FYUN^EZiF[UJVEji0eZi00EJ^UZiF[UJ^EZj009J^@0<E[UJ^UZiF[YF^EZjF[UJ
^UJiF[YF^EZj6Woo00<007ooOol017oo00<007ooOol02Goo000ROol00`00Oomoo`0?Ool02EjGG[MN
Uf:hGYMN^5jGG[QNV003G[P01EjHG[QN^5jhGYP02ejh00EJ^5jhG[QN^5Zh009N^009G[UJ^5jiF[QN
^EZhF[UJ^5ji01mJ^@05F[YJ^EZiF[UJ^P02F[TKOol00`00Oomoo`04Ool00`00Oomoo`09Ool002=o
o`03001oogoo00ioo`0kH[QNUf:hGYMR^5jGH[QNUf:hGYMR^5jGG[QNUf:hGYMN^5jGG[QNV5jhFYQN
^EjHG[QJV5jiFYQN^5ZHG[UJV5jiFYQN^EZHG[UJV5jiFYQN^EZHG[UJV5jiFYQN^EZHG[UJV5jiF[QN
^EZHG[UJ^5jiFYQN^@02F[T01EZHF[UJ^EZiFYT00eZi00YJVEZiF[UJ^UJiF[YJ^EZjEYUJ^Qmoo`03
001oogoo00=oo`03001oogoo00Uoo`0097oo00<007ooOol03Woo019R^5jGG[MNUejhGYMN^5jGG[QN
UejhGYMN^5jHG[QNV5jhGYP3G[P01EjHG[QN^5jhGYP00ejh00ENV5jhG[QN^EZh009N^00:G[UJ^5jh
F[QN^EZhG[UJ^5jiF[P2F[T00ejiF[QJ^@02F[T00eZhF[UJ^@0EF[TQOol00`00Oomoo`03Ool00`00
Oomoo`08Ool002Eoo`8000ioo`1<GYMR^5jGH[QNUf:hGYMR^5jGG[QNUf:hGYMN^5jGH[QNUejhGYMN
^5jGG[QJV5jhGYMN^5ZHG[QNV5jhFYQN^EZHG[QJV5jiFYQN^EZHG[UJV5jiFYQN^EZHG[UJV5jiFYQN
^EZHG[UJV5jiFYQN^EZHG[UJV5jiFYQN^EZHG[UJVEZiFYQN^EZIF[UJVEZiFYUJ^EZIF[TROol00`00
Oomoo`03Ool00`00Oomoo`08Ool002Moo`03001oogoo00aoo`03GYMR]ejG009NU`0FH[QNUejGGYMN
^5jGG[QNUejhGYMN^5jHG[QNUejhGYQN^5jHG[QNV5jhGYP3G[P01EjHG[QN^5jhGYP00ejh00UJV5jh
F[QN^EZHG[QJ^5jiFYP00UZi00IN^EZHF[UJ^EjiF[P3F[T00eZhF[UJ^@0;F[TVOol00`00Oomoo`02
Ool00`00Oomoo`08Ool002Qoo`03001oogoo00aoo`16GYMR^5jGH[MNUf:hGYMR^5jGH[QNUf:hGYMN
^5jGH[QNUejhGYMR^5jGG[QJUejhGYQN^5ZGG[QJV5jhFYQN^5ZHG[QJV5jhFYQN^5ZHG[UJV5jhFYQN
^EZHG[UJV5jiFYQN^EZHG[UJV5jiFYQN^EZHG[UJV5jiFYQJ^EZHG[UJV5ZiFYQN^EZIF[TXOol01@00
Oomoogoo00002Woo000YOol00`00Oomoo`0;Ool01f:GGYMRUejGHYMNUf:g00=NU`05G[MNUejGGYMN
^003GYL01UjhGYMNV5jHG[QNU`9NV005G[QNV5jHGYQN^003GYP075jhGYQN^5jHG[QJV5jhGYQN^5ZH
G[QJV5jhFYQJ^5ZHG[QJV5ZiF[QJ^EZHF[UJ^5ZiFYQJ^EZh0eZi00=J^5ZiFYT0:goo00@007ooOol0
00Yoo`00:Woo0P0037oo045R]ejGH[MNUV:hGYMR]ejGH[QNUf:hGYMR^5jGG[QNUf:hGYMN^5jGG[QN
UejhGYMN^5jGG[QJUejhGYMN^5ZGG[QJV5jhFYMN^EZHG[QJUejiFYQN^5ZHG[UJV5jhFYQN^EZHG[UJ
V5jiFYQN^EZHG[UJV5jiFYQN^EZHG[UJV5ji02eoo`04001oogoo000:Ool002aoo`03001oogoo00Uo
o`04GYMRUejGHYL3GYL01F:GGYMNUejGHYL01EjG00ANV5jGGYQNU`=NV005GYMNV5jHGYQNU`03GYP0
15jGGYQNV5jh0ejH01IN^5ZHGYQNV5jhFYQN^5jHG[QJV5jhFYQN^5ZHF[QJV5jiFYQJ^5ZHF[UJV09J
^003F[Uoogoo02eoo`03001oo`0000Yoo`00;Goo00<007ooOol02Goo03eNUV:gGYMRUejGH[MNUf:G
GYMR^5jGH[MNUf:hGYMR^5jGH[QNUejhGYMR^5jGG[QNUejhGYMN^5ZGG[QNUejhFYMN^5ZGG[QJUejh
FYMN^5ZHG[QJV5jhFYQN^5ZHG[QJV5jiFYQN^EZHG[UJV5jiFYQN^EZHG[UJV00bOol00`00Ool00009
Ool002ioo`03001oogoo00Uoo`04GYMRUf:GHYL3GYL01F:GGYMNUejGHYL03EjG00ANV5jGGYQNU`=N
V003GYMNV5jH00ANV00DG[QJV5jHGYQN^5ZHGYQJV5jhFYQNV5ZHG[QJV5jHFYQN^5ZHF[QJV3Aoo`80
00Uoo`00;goo00<007ooOol027oo03YRUejFHYMNUV:GGYMR]ejFH[MNUf:gGYMRUejGH[MNUf:GGYMR
^5jGHYMNUf:hGYMNV5jGG[QNUejHFYMN^5jGGYQJUejhGYMN^5ZGG[QJUejhFYMN^5ZGG[QJUejhFYQN
^5ZGG[QJV5jhFYQN^5ZHG[QJV3Eoo`8000Uoo`00<7oo0P002Goo00=RUejFHYL00V:G00ANUf:GGYMR
U`mNU`05GYQNUejGGYMNV003GYL015jHGYMNV5jG1EjH00eJV5jHGYQNV5ZHGYQJV5jHFYQNV5ZHGYQJ
V00hOol20009Ool0039oo`03001oogoo00Ioo`0eGYIRUejFHYMNUV:GGYIRUejFHYMNUV:GGYIRUejG
HYMNUf:GGYMRUejGGYQNUf:HGYMNV5jGHYQNUejHFYMNV5jGGYQJUejhFYMNV5ZGG[QJUejHFYMN^5ZH
GYQJUejhFYQNV5ZGG[QJV00eOol4000;Ool003=oo`03001oogoo00Ioo`9RU`08GYIRUejGHYMNUV:G
GYMRU`=NU`03HYMNUejG00UNU`05GYQNUejGGYMNV003GYL00ejHGYMNU`02GYP01EjGGYQNV5jHGYL0
0ejH00EJUejHGYQNV5ZH039oo`@000moo`00=7oo00<007ooOol01Goo01ERUejFHYMNUV:GGYIRUeif
HYMNUV:GGYIRUejGHYMNUV:GGYMRUejGHYL00ejG01YRV5jGGYMNUf:HGYMNV5jGGYQNUejHFYMNV5jG
GYQJUejhFYMNV5ZGG[QJUejHFYMN^5ZG;goo10004goo000eOol20006Ool026:GHYIRUUjFHYMNUf:G
GYH2GYL01V:GGYIRUejGHYMNUQINU`09GYQNUejHGYMNV5jGGYQNUejH02eoo`@001Moo`00=goo00<0
07ooOol00goo01YNUV:GGYIRUeifHYMNUV:GGWIRUejFHYMNMV:GGYIRUeifHYMNUV:GGWIRUejGHYMN
MV:G0UjG00INMejHGYMNUeigGYP2GYL02UZGGYQNUejHFYMNV5jGGYQJUejH:Woo10006goo000hOol0
0`00Oomoo`03Ool01EjFHYIRUV:FGYH00f:F00ENUV:GGYIRUejF009NU`06HYMNUUjGGYMRUejF5UjG
00=NV5jGOol09Woo0`007goo000iOol00`00Oomoo`02Ool0;6:GGWIRUeifHYINMV:GGWIRUeifHYMN
MV:GGWIRUeifHYMNMV:GGWIRUeifHYMNMV:GGWMRUeifGYMNMejHGWMNUeigGYQNUejHFWMNV5jGGYQJ
MejHGYLTOol4000ROol003Yoo`8000=oo`ERUP0=GYIRUV:FHYMNUV:FGYIRUejFGYMNUV:GGYH00UjG
00=RUejGGYL04ejG8Goo10009Woo000lOol09P00OomooeifHYINMV:GHYIRUeifHYMNMV:GGWIRUeif
HYMNMV:GGWIRUeifHYMNMV:GGWIRUeifGYMNMV:GGWINUeifHYMNMUjGGWL3GYL00eYgGYMoo`0NOol4
000ZOol003eoo`03001oogoo00IRUP0=GWIRUUjFHYMNMV:FGYIRUeifHYINUV:GGWH00UjF00=NUeif
GYL00UjG00=NUUjGGYL02EjG77oo1000;Woo000nOol09`00OomRUeifH[MRUVJgGWIRUUifHYMNMV:G
GWIRUeifHYMNMV:GGWIRUeifHYMNMV:GGWIRUeifHYMNMV:GGWINUeifHYMNMUjGGWIRU`0IOol4000b
Ool003moo`03001oof:F009RUP07J_QRUV:FHYIVef:FGYH00f:F00=NUV:GGWH00UjF00IRUeifGYMN
UUjGGWH3GYL01EifGYMNUejGGWH00ejG5goo0`00=Woo0010Ool200008FJgJ_Q[66KhHYMVef[GI]MR
Uf:FHYMRUV:GGWIRUUifHYMNMV:GGWIRUeifHYMNMV:GGWIRUeifHYMNMUjGGWIRU`0DOol4000iOol0
049oo`04J_Q[66/HJaP4I]L4HYH02EifHYINMV:FGWINUUifHYINMP02GYH02F:GGWINUUjFGYMNMUjG
GYINU`0AOol4000mOol004=oo`0NI_Q[65/KGaaVef[GHaUW6V:FH[MZn6/HGWIRUfKGJ_QNMV:GHYIR
]eifHYMNMV:FGWIRUeifHYMNMV:G3Woo1000@Goo0014Ool3Fa/015lLIaUW6VLI0V:F0V/H00IRUUif
I]MVef:FGWH2H[L02V:GGWINUUifGYMNMUjFGWINUeif2Woo1000AGoo0015Ool02TklDae>oE<MH_UW
6V<IJaQVn6/I0V[h00IVef[hI]MR]f:FI[L2HYH01eifHYMNMV:GGWIRUeif00Moo`@004Uoo`00Agoo
0U;m00IC7DkmD_eBoE/LFa/2Fa`03VKHI]MS6V<JH[MRUVLIGWIRUUifJaQNMUjFGWH5Ool3001=Ool0
04Qoo`09Dae>oE<MC_aC7DkmDaeJne/L00=S6P0:IaYW6F/IIaQ[6EifJaQZn6/HGWH2Ool4001@Ool0
04Uoo`9Bo@06Dae>oE;mD_eC7Dkm0U;m00IC7DkmD_eBoF/IIaT3JaP016[h00000000E7oo001:Ool0
4U<MC_eC7DklDae>oE<MC_aC7DkmDae>o5<MC_eC7E;lDad005Qoo`00Bgoo0U;m00IC7DkmD_eBoE<M
C_d2D_d00e<MC_eBo@1LOol004eoo`07C_eC7DklDae>oE<M0000H7oo001>Ool00e;mDae>o@1SOol0
0;Aoo`00]7oo002dOol00;Aoo`00]7oo002dOol00;Aoo`00]7oo002dOol00;Aoo`00]7oo002dOol0
0;Aoo`00]7oo002dOol00001\
\>"],
  ImageRangeCache->{{{0, 179.5}, {164.5, 0}} -> {-0.114677, -0.0979773, \
0.00626889, 0.00626889}}]
}, Open  ]],

Cell[TextData[{
  "Another case discussed by ",
  ButtonBox["Baikov et al. [1989]",
    ButtonData:>{"Literat.nb", "Baik-89"},
    ButtonStyle->"Hyperlink"],
  " is the isentropic motion with ",
  Cell[BoxData[
      \(TraditionalForm\`\[Sigma] = \(-4\)\)]],
  ". The equation of motion for this case reads"
}], "Text",
  CellTags->"isentropic motion"],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq\  = \ equation /. \[Sigma] \[Rule] \(-4\); eq // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            RowBox[{
              RowBox[{
                RowBox[{"-", 
                  StyleBox[\(\(4\ u\_q\%2\)\/u\^5\),
                    ScriptLevel->0]}], "-", \(\[Epsilon]\ u\_t\), "+", 
                StyleBox[\(u\_\(q, q\)\/u\^4\),
                  ScriptLevel->0], "+", \(u\_\(t, t\)\)}], "==", "0"}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell["\<\
The first-order approximate symmetries of this equation follows \
by\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(infinitesimals\  = \ 
      ApproximateSymmetries[eq, {u}, {q, t}, {\[Epsilon]}, \[Epsilon], 
        SubstitutionRules \[Rule] {\[PartialD]\_\(x, x\)u[q, t]}]; 
    infinitesimals // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{\(\[Phi]\_1\), "==", 
              RowBox[{
                RowBox[{"-", 
                  StyleBox[\(\(k4\ u\)\/2\),
                    ScriptLevel->0]}], "+", 
                RowBox[{
                  StyleBox[\(1\/2\),
                    ScriptLevel->0], " ", \((\(-k2\) + k6 + 2\ k7\ t)\), " ", 
                  "u", " ", "\[Epsilon]"}]}]}]},
          {\(\[Xi]\_1 == k3 + k4\ q + \((k8 + k2\ q)\)\ \[Epsilon]\)},
          {\(\[Xi]\_2 == k1 + \((k5 + t\ \((k6 + k7\ t)\))\)\ \[Epsilon]\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], 
          Plus[ 
            Times[ 
              Rational[ -1, 2], k4, u], 
            Times[ 
              Rational[ 1, 2], 
              Plus[ 
                Times[ -1, k2], k6, 
                Times[ 2, k7, t]], u, \[Epsilon]]]], 
        Equal[ 
          Subscript[ \[Xi], 1], 
          Plus[ k3, 
            Times[ k4, q], 
            Times[ 
              Plus[ k8, 
                Times[ k2, q]], \[Epsilon]]]], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Plus[ k1, 
            Times[ 
              Plus[ k5, 
                Times[ t, 
                  Plus[ k6, 
                    Times[ k7, t]]]], \[Epsilon]]]]}]]], "Output"]
}, Open  ]],

Cell["\<\
representing an eight-dimensional finite symmetry group. Again, we \
can use the infinitesimals to derive analytic solutions for the isentropic \
model. The generating vector fields of the subgroups read\
\>", "Text",
  CellTags->{"infinitesimals", "isentropic model", "generating vector field"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(Map[
        \((Fold[Plus, 0, 
                Map[Fold[NonCommutativeMultiply, 1, #]&, 
                    Transpose[{
                        Flatten[#], {\*"\"\<\!\(\[PartialD]\_q\)\>\"", 
                          \*"\"\<\!\(\[PartialD]\_t\)\>\"", 
                          \*"\"\<\!\(\[PartialD]\_u\)\>\""}}]] /. 
                  \ _**0**_ \[Rule] 0]\  /. \ 1**a_**b_ \[Rule] a**b)\)&, 
        \({{\(xi[1]\)[q, t, u], \(xi[2]\)[q, t, u]}, {
                \(phi[1]\)[q, t, u]}} /. infinitesimals\) /. 
          \((Map[\((Thread[{k1, k2, k3, k4, k5, k6, k7, k8} \[Rule] #])\)&, 
              Permutations[{1, 0, 0, 0, 0, 0, 0, 0}]])\)] // TableForm\)], 
  "Input"],

Cell[BoxData[GridBox[{
        {\(1**"\!\(\[PartialD]\_t\)"\)},
        {\(\((q\ \[Epsilon])\)**"\!\(\[PartialD]\_q\)" + 
            \((\(-\(\(u\ \[Epsilon]\)\/2\)\))\)**"\!\(\[PartialD]\_u\)"\)},
        {\(1**"\!\(\[PartialD]\_q\)"\)},
        {\(q**"\!\(\[PartialD]\_q\)" + 
            \((\(-\(u\/2\)\))\)**"\!\(\[PartialD]\_u\)"\)},
        {\(\[Epsilon]**"\!\(\[PartialD]\_t\)"\)},
        {\(\((t\ \[Epsilon])\)**"\!\(\[PartialD]\_t\)" + 
            \(\(u\ \[Epsilon]\)\/2\)**"\!\(\[PartialD]\_u\)"\)},
        {\(\((t\^2\ \[Epsilon])\)**"\!\(\[PartialD]\_t\)" + 
            \((t\ u\ \[Epsilon])\)**"\!\(\[PartialD]\_u\)"\)},
        {\(\[Epsilon]**"\!\(\[PartialD]\_q\)"\)}
        },
      RowSpacings->1,
      ColumnSpacings->3,
      RowAlignments->Baseline,
      ColumnAlignments->{Left}]], "Output"]
}, Open  ]],

Cell[TextData[{
  "Here, we used the function ",
  StyleBox["NonCommutativeMultiply[]", "MmaText"],
  " (",
  StyleBox["**", "MmaText"],
  ") to keep the ordering of the operators in the representation of the \
vector fields. The coefficients of these differential equations are"
}], "Text",
  CellTags->"NonCommutativeMultiply[]"],

Cell[CellGroupData[{

Cell[BoxData[
    \(cases3\  = \ 
      \({{\(xi[1]\)[q, t, u], \(xi[2]\)[q, t, u]}, {\(phi[1]\)[q, t, u]}} /. 
          infinitesimals\) /. 
        \((Map[\((Thread[{k1, k2, k3, k4, k5, k6, k7, k8} \[Rule] #])\)&, 
            Permutations[{1, 0, 0, 0, 0, 0, 0, 0}]])\)\)], "Input"],

Cell[BoxData[
    \({{{0, 1}, {0}}, {{q\ \[Epsilon], 0}, {
          \(-\(\(u\ \[Epsilon]\)\/2\)\)}}, {{1, 0}, {0}}, {{q, 0}, {
          \(-\(u\/2\)\)}}, {{0, \[Epsilon]}, {0}}, {{0, t\ \[Epsilon]}, {
          \(u\ \[Epsilon]\)\/2}}, {{0, t\^2\ \[Epsilon]}, {
          t\ u\ \[Epsilon]}}, {{\[Epsilon], 0}, {0}}}\)], "Output"]
}, Open  ]],

Cell["\<\
One of the possible reductions can be calculated by combining two \
sub-groups. For the present calculation, we combine the third and seventh \
sub-group:\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(red3\  = \ 
      LieReduction[eq, {u}, {q, t}, 
          cases3\[LeftDoubleBracket]7, 1\[RightDoubleBracket] + 
            cases3\[LeftDoubleBracket]3, 1\[RightDoubleBracket], 
          cases3\[LeftDoubleBracket]7, 2\[RightDoubleBracket] + 
            cases3\[LeftDoubleBracket]3, 2\[RightDoubleBracket]] // Simplify; 
    \(red3 /. zeta1 \[Rule] \[Zeta]\_1\  // Flatten\) // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            RowBox[{
              RowBox[{
                StyleBox[\(1\/t\),
                  ScriptLevel->0], "+", \(q\ \[Epsilon]\), "-", 
                \(\[Zeta]\_1\)}], "==", "0"}]},
          {
            RowBox[{
              RowBox[{
                RowBox[{"-", 
                  StyleBox[\(u\/t\),
                    ScriptLevel->0]}], "-", \(F\_1\)}], "==", "0"}]},
          {
            \(t\^3\ \[Epsilon]\ F\_1\%6 + 
                4\ \[Epsilon]\^2\ \((F\_1)\)\_\[Zeta]\_1\%2 - 
                \[Epsilon]\^2\ F\_1\ \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1\) - 
                F\_1\%5\ 
                  \((t\^2\ \[Epsilon]\ \((F\_1)\)\_\(\[Zeta]\_1\) + 
                      \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1\))\)\n\t == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "Solving for the unknown field ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  ", we find an explicit similarity representation of the solution:"
}], "Text",
  CellTags->"similarity representation"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Solve[\(-\(u\/t\)\) == F1[1\/t + q\ \[Epsilon]], u]\)], "Input"],

Cell[BoxData[
    \({{u \[Rule] \(-t\)\ F1[1\/t + q\ \[Epsilon]]}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The resulting similarity representation is given by ",
  Cell[BoxData[
      \(TraditionalForm\`u = \(-t\)\ \(\(F\_1\)(1/t + q\ \[Epsilon])\)\)]],
  ", where ",
  Cell[BoxData[
      \(TraditionalForm\`F\_1\)]],
  " has to satisfy a second-order ODE depending on ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  " and \[Epsilon]. We eliminate this dependence by choosing ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon] = 0\)]],
  " in the determining equation of ",
  Cell[BoxData[
      \(TraditionalForm\`F\_1\)]],
  ". The solution of the resulting equation follows with"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(solh\  = 
      DSolve[red2\[LeftDoubleBracket]3\[RightDoubleBracket] /. 
          \[Epsilon] \[Rule] 0\ , F1, zeta1]\)], "Input"],

Cell[BoxData[
    \({{F1 \[Rule] \((C[1] + C[2]\ #1&)\)}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "representing a linear function in ",
  StyleBox["zeta1",
    FontSlant->"Italic"],
  ". The solution in Lagrange coordinates ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`q\)]],
  " reads"
}], "Text",
  CellTags->"Lagrange coordinates"],

Cell[CellGroupData[{

Cell[BoxData[
    \(solution3\  = \ 
      Solve[Flatten[
            red3\[LeftDoubleBracket]2\[RightDoubleBracket]\  /. \ solh], 
          u] // Flatten\)], "Input"],

Cell[BoxData[
    \({u \[Rule] \(-t\)\ C[1] - C[2] - q\ t\ \[Epsilon]\ C[2]}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "This solution depends linearly on \[Epsilon] and thus is consistent with \
the approximation order of the procedure. The choice of numerical values for \
the integration constants ",
  StyleBox["C",
    FontSlant->"Italic"],
  "[1], ",
  StyleBox["C",
    FontSlant->"Italic"],
  "[2] and the approximation parameter \[Epsilon] allows us to outline this \
solution in a contour plot."
}], "Text",
  CellTags->"contour plot"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ContourPlot[
      Evaluate[\(u /. solution3\) /. {C[1] \[Rule] 1, C[2] \[Rule] \(-1\), 
            \[Epsilon] \[Rule]  .9}], {q, \(-3\), 3}, {t, 0, 20}, 
      AxesLabel \[Rule] {"\<q\>", "\<t\>"}, ColorFunction \[Rule] Hue, 
      Axes \[Rule] True]\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% ContourGraphics
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.160256 0.0192308 0.0480769 [
[.01923 -0.0125 -6 -9 ]
[.01923 -0.0125 6 0 ]
[.17949 -0.0125 -6 -9 ]
[.17949 -0.0125 6 0 ]
[.33974 -0.0125 -6 -9 ]
[.33974 -0.0125 6 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.66026 -0.0125 -3 -9 ]
[.66026 -0.0125 3 0 ]
[.82051 -0.0125 -3 -9 ]
[.82051 -0.0125 3 0 ]
[.98077 -0.0125 -3 -9 ]
[.98077 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01923 -6 -4.5 ]
[-0.0125 .01923 0 4.5 ]
[-0.0125 .25962 -6 -4.5 ]
[-0.0125 .25962 0 4.5 ]
[-0.0125 .5 -12 -4.5 ]
[-0.0125 .5 0 4.5 ]
[-0.0125 .74038 -12 -4.5 ]
[-0.0125 .74038 0 4.5 ]
[-0.0125 .98077 -12 -4.5 ]
[-0.0125 .98077 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[1.025 0 0 -7.5 ]
[1.025 0 10 7.5 ]
[0 1.025 -5 0 ]
[0 1.025 5 15 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.01923 0 m
.01923 .00625 L
s
[(-3)] .01923 -0.0125 0 1 Mshowa
.17949 0 m
.17949 .00625 L
s
[(-2)] .17949 -0.0125 0 1 Mshowa
.33974 0 m
.33974 .00625 L
s
[(-1)] .33974 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(0)] .5 -0.0125 0 1 Mshowa
.66026 0 m
.66026 .00625 L
s
[(1)] .66026 -0.0125 0 1 Mshowa
.82051 0 m
.82051 .00625 L
s
[(2)] .82051 -0.0125 0 1 Mshowa
.98077 0 m
.98077 .00625 L
s
[(3)] .98077 -0.0125 0 1 Mshowa
.125 Mabswid
.05128 0 m
.05128 .00375 L
s
.08333 0 m
.08333 .00375 L
s
.11538 0 m
.11538 .00375 L
s
.14744 0 m
.14744 .00375 L
s
.21154 0 m
.21154 .00375 L
s
.24359 0 m
.24359 .00375 L
s
.27564 0 m
.27564 .00375 L
s
.30769 0 m
.30769 .00375 L
s
.37179 0 m
.37179 .00375 L
s
.40385 0 m
.40385 .00375 L
s
.4359 0 m
.4359 .00375 L
s
.46795 0 m
.46795 .00375 L
s
.53205 0 m
.53205 .00375 L
s
.5641 0 m
.5641 .00375 L
s
.59615 0 m
.59615 .00375 L
s
.62821 0 m
.62821 .00375 L
s
.69231 0 m
.69231 .00375 L
s
.72436 0 m
.72436 .00375 L
s
.75641 0 m
.75641 .00375 L
s
.78846 0 m
.78846 .00375 L
s
.85256 0 m
.85256 .00375 L
s
.88462 0 m
.88462 .00375 L
s
.91667 0 m
.91667 .00375 L
s
.94872 0 m
.94872 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01923 m
.00625 .01923 L
s
[(0)] -0.0125 .01923 1 0 Mshowa
0 .25962 m
.00625 .25962 L
s
[(5)] -0.0125 .25962 1 0 Mshowa
0 .5 m
.00625 .5 L
s
[(10)] -0.0125 .5 1 0 Mshowa
0 .74038 m
.00625 .74038 L
s
[(15)] -0.0125 .74038 1 0 Mshowa
0 .98077 m
.00625 .98077 L
s
[(20)] -0.0125 .98077 1 0 Mshowa
.125 Mabswid
0 .06731 m
.00375 .06731 L
s
0 .11538 m
.00375 .11538 L
s
0 .16346 m
.00375 .16346 L
s
0 .21154 m
.00375 .21154 L
s
0 .30769 m
.00375 .30769 L
s
0 .35577 m
.00375 .35577 L
s
0 .40385 m
.00375 .40385 L
s
0 .45192 m
.00375 .45192 L
s
0 .54808 m
.00375 .54808 L
s
0 .59615 m
.00375 .59615 L
s
0 .64423 m
.00375 .64423 L
s
0 .69231 m
.00375 .69231 L
s
0 .78846 m
.00375 .78846 L
s
0 .83654 m
.00375 .83654 L
s
0 .88462 m
.00375 .88462 L
s
0 .93269 m
.00375 .93269 L
s
.25 Mabswid
0 0 m
0 1 L
s
.01923 .99375 m
.01923 1 L
s
.17949 .99375 m
.17949 1 L
s
.33974 .99375 m
.33974 1 L
s
.5 .99375 m
.5 1 L
s
.66026 .99375 m
.66026 1 L
s
.82051 .99375 m
.82051 1 L
s
.98077 .99375 m
.98077 1 L
s
.125 Mabswid
.05128 .99625 m
.05128 1 L
s
.08333 .99625 m
.08333 1 L
s
.11538 .99625 m
.11538 1 L
s
.14744 .99625 m
.14744 1 L
s
.21154 .99625 m
.21154 1 L
s
.24359 .99625 m
.24359 1 L
s
.27564 .99625 m
.27564 1 L
s
.30769 .99625 m
.30769 1 L
s
.37179 .99625 m
.37179 1 L
s
.40385 .99625 m
.40385 1 L
s
.4359 .99625 m
.4359 1 L
s
.46795 .99625 m
.46795 1 L
s
.53205 .99625 m
.53205 1 L
s
.5641 .99625 m
.5641 1 L
s
.59615 .99625 m
.59615 1 L
s
.62821 .99625 m
.62821 1 L
s
.69231 .99625 m
.69231 1 L
s
.72436 .99625 m
.72436 1 L
s
.75641 .99625 m
.75641 1 L
s
.78846 .99625 m
.78846 1 L
s
.85256 .99625 m
.85256 1 L
s
.88462 .99625 m
.88462 1 L
s
.91667 .99625 m
.91667 1 L
s
.94872 .99625 m
.94872 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .01923 m
1 .01923 L
s
.99375 .25962 m
1 .25962 L
s
.99375 .5 m
1 .5 L
s
.99375 .74038 m
1 .74038 L
s
.99375 .98077 m
1 .98077 L
s
.125 Mabswid
.99625 .06731 m
1 .06731 L
s
.99625 .11538 m
1 .11538 L
s
.99625 .16346 m
1 .16346 L
s
.99625 .21154 m
1 .21154 L
s
.99625 .30769 m
1 .30769 L
s
.99625 .35577 m
1 .35577 L
s
.99625 .40385 m
1 .40385 L
s
.99625 .45192 m
1 .45192 L
s
.99625 .54808 m
1 .54808 L
s
.99625 .59615 m
1 .59615 L
s
.99625 .64423 m
1 .64423 L
s
.99625 .69231 m
1 .69231 L
s
.99625 .78846 m
1 .78846 L
s
.99625 .83654 m
1 .83654 L
s
.99625 .88462 m
1 .88462 L
s
.99625 .93269 m
1 .93269 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 0 m
1 0 L
s
gsave
1.025 0 -61 -11.5 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 23 translate 1 -1 scale
63.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(q) show
69.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
%%DocumentNeededResources: font Courier
%%DocumentNeededFonts: Courier
%%DocumentFonts: font Courier
grestore
0 0 m
0 1 L
s
gsave
0 1.025 -66 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 23 translate 1 -1 scale
63.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(t) show
69.000000 14.437500 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
%%DocumentNeededResources: font Courier
%%DocumentNeededFonts: Courier
%%DocumentFonts: font Courier
grestore
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
.2 0 1 r
.01923 .98077 m
.98077 .98077 L
.98077 .01923 L
.01923 .01923 L
F
0 g
.5 Mabswid
0 .4 1 r
.01923 .07818 m
.08791 .08504 L
.11263 .08791 L
.15659 .0937 L
.22527 .105 L
.29396 .12033 L
.36264 .14235 L
.39535 .15659 L
.43132 .17662 L
.48959 .22527 L
.5 .23733 L
.53671 .29396 L
.56498 .36264 L
.56868 .37427 L
.58382 .43132 L
.59729 .5 L
.60738 .56868 L
.61524 .63736 L
.62152 .70604 L
.62666 .77473 L
.63094 .84341 L
.63457 .91209 L
.63736 .97339 L
.63767 .98077 L
.01923 .98077 L
F
0 g
.01923 .07818 m
.08791 .08504 L
.11263 .08791 L
.15659 .0937 L
.22527 .105 L
.29396 .12033 L
.36264 .14235 L
.39535 .15659 L
.43132 .17662 L
.48959 .22527 L
.5 .23733 L
.53671 .29396 L
.56498 .36264 L
.56868 .37427 L
.58382 .43132 L
.59729 .5 L
.60738 .56868 L
.61524 .63736 L
.62152 .70604 L
.62666 .77473 L
.63094 .84341 L
.63457 .91209 L
.63736 .97339 L
.63767 .98077 L
s
0 1 1 r
.01923 .21213 m
.06126 .22527 L
.08791 .23458 L
.15659 .26294 L
.21546 .29396 L
.22527 .29991 L
.29396 .3501 L
.30798 .36264 L
.36264 .42214 L
.36966 .43132 L
.41372 .5 L
.43132 .53429 L
.44676 .56868 L
.47246 .63736 L
.49302 .70604 L
.5 .73295 L
.50984 .77473 L
.52386 .84341 L
.53572 .91209 L
.54589 .98077 L
.01923 .98077 L
F
0 g
.01923 .21213 m
.06126 .22527 L
.08791 .23458 L
.15659 .26294 L
.21546 .29396 L
.22527 .29991 L
.29396 .3501 L
.30798 .36264 L
.36264 .42214 L
.36966 .43132 L
.41372 .5 L
.43132 .53429 L
.44676 .56868 L
.47246 .63736 L
.49302 .70604 L
.5 .73295 L
.50984 .77473 L
.52386 .84341 L
.53572 .91209 L
.54589 .98077 L
s
0 1 .4 r
.01923 .34608 m
.05099 .36264 L
.08791 .38412 L
.1555 .43132 L
.15659 .43218 L
.22527 .49482 L
.23015 .5 L
.28614 .56868 L
.29396 .57986 L
.32969 .63736 L
.36264 .70193 L
.36453 .70604 L
.39303 .77473 L
.41678 .84341 L
.43132 .89196 L
.43688 .91209 L
.45411 .98077 L
.01923 .98077 L
F
0 g
.01923 .34608 m
.05099 .36264 L
.08791 .38412 L
.1555 .43132 L
.15659 .43218 L
.22527 .49482 L
.23015 .5 L
.28614 .56868 L
.29396 .57986 L
.32969 .63736 L
.36264 .70193 L
.36453 .70604 L
.39303 .77473 L
.41678 .84341 L
.43132 .89196 L
.43688 .91209 L
.45411 .98077 L
s
.2 1 0 r
.01923 .48004 m
.04659 .5 L
.08791 .53367 L
.12552 .56868 L
.15659 .60142 L
.18692 .63736 L
.22527 .68973 L
.23603 .70604 L
.27621 .77473 L
.29396 .80962 L
.3097 .84341 L
.33804 .91209 L
.36233 .98077 L
.01923 .98077 L
F
0 g
.01923 .48004 m
.04659 .5 L
.08791 .53367 L
.12552 .56868 L
.15659 .60142 L
.18692 .63736 L
.22527 .68973 L
.23603 .70604 L
.27621 .77473 L
.29396 .80962 L
.3097 .84341 L
.33804 .91209 L
.36233 .98077 L
s
.8 1 0 r
.01923 .61399 m
.04414 .63736 L
.08791 .68321 L
.10753 .70604 L
.15659 .77066 L
.1594 .77473 L
.20262 .84341 L
.22527 .88464 L
.23919 .91209 L
.27054 .98077 L
.01923 .98077 L
F
0 g
.01923 .61399 m
.04414 .63736 L
.08791 .68321 L
.10753 .70604 L
.15659 .77066 L
.1594 .77473 L
.20262 .84341 L
.22527 .88464 L
.23919 .91209 L
.27054 .98077 L
s
1 .6 0 r
.01923 .74794 m
.04258 .77473 L
.08791 .83275 L
.09554 .84341 L
.14035 .91209 L
.15659 .9399 L
.17876 .98077 L
.01923 .98077 L
F
0 g
.01923 .74794 m
.04258 .77473 L
.08791 .83275 L
.09554 .84341 L
.14035 .91209 L
.15659 .9399 L
.17876 .98077 L
s
1 0 0 r
.01923 .8819 m
.04151 .91209 L
.08698 .98077 L
.01923 .98077 L
F
0 g
.01923 .8819 m
.04151 .91209 L
.08698 .98077 L
s
.8 0 1 r
.98077 .18249 m
.91791 .22527 L
.91209 .2304 L
.85795 .29396 L
.84341 .31811 L
.82197 .36264 L
.79799 .43132 L
.78085 .5 L
.77473 .53048 L
.768 .56868 L
.75801 .63736 L
.75002 .70604 L
.74347 .77473 L
.73802 .84341 L
.73341 .91209 L
.72946 .98077 L
.98077 .98077 L
F
0 g
.98077 .18249 m
.91791 .22527 L
.91209 .2304 L
.85795 .29396 L
.84341 .31811 L
.82197 .36264 L
.79799 .43132 L
.78085 .5 L
.77473 .53048 L
.768 .56868 L
.75801 .63736 L
.75002 .70604 L
.74347 .77473 L
.73802 .84341 L
.73341 .91209 L
.72946 .98077 L
s
1 0 .6 r
.98077 .47403 m
.96442 .5 L
.92863 .56868 L
.91209 .60751 L
.90078 .63736 L
.87851 .70604 L
.86029 .77473 L
.8451 .84341 L
.84341 .85187 L
.83225 .91209 L
.82124 .98077 L
.98077 .98077 L
F
0 g
.98077 .47403 m
.96442 .5 L
.92863 .56868 L
.91209 .60751 L
.90078 .63736 L
.87851 .70604 L
.86029 .77473 L
.8451 .84341 L
.84341 .85187 L
.83225 .91209 L
.82124 .98077 L
s
1 0 0 r
.98077 .76558 m
.9771 .77473 L
.95218 .84341 L
.9311 .91209 L
.91302 .98077 L
.98077 .98077 L
F
0 g
.98077 .76558 m
.9771 .77473 L
.95218 .84341 L
.9311 .91209 L
.91302 .98077 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{179.625, 183.625},
  ImageMargins->{{80, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40002c0000]a000`40O003h00O/goo002cOol0019o
o`8001Eoo`@001Eoo`D0015oo`8001Eoo`D001=oo`@001Eoo`80015oo`004Goo00@007ooOol001Ao
o`03001oogoo01Qoo`03001oogoo011oo`04001oogoo000FOol00`00Oomoo`0COol00`00Oomoo`0E
Ool01000Oomoo`0047oo000COol00`00Oomoo`0DOol00`00Oomoo`0GOol00`00Oomoo`0@Ool01000
Oomoo`005Woo00<007ooOol057oo00<007ooOol05Woo00<007ooOol03goo000;Ool40005Ool00`00
Oomoo`0<Ool40004Ool00`00Oomoo`0>Ool40004Ool00`00Oomoo`0@Ool01000Oomoo`005Woo00<0
07ooOol05Goo00<007ooOol05Woo00<007ooOol03Woo000AOol01000Oomoo`0057oo00@007ooOol0
01Moo`03001oogoo011oo`04001oogoo000FOol00`00Oomoo`0COol01000Oomoo`0057oo00@007oo
Ool0011oo`004Woo0P005Woo0P005goo0P004goo0P005Woo0P005Woo0P005Woo0P004Goo002]Ool3
0003Ool00:ioo`03001oogoo009oo`00[7oo0`0017oo002[Ool01000Oomoo`0017oo000=OonH0006
Ool01000Oomoo`0017oo0008Ool20003Ool01000Oomoo`005goo00<007ooOol05Woo00<007ooOol0
5Goo00<007ooOol05Goo00<007ooOol05Goo00<007ooOol05Goo00@007ooOol000Moo`@000=oo`00
1goo00@007ooOol0009oo`03001oogoo09Aoo`03001oogoo00aoo`001goo00@007ooOol0009oo`80
09Aoo`8000ioo`001goo00@007ooOol0009oo`2D001oogoo61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61l3Ool00`00Oomoo`0<Ool000Moo`04001oogoo0002Ool0
1@00Oomooa@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l0
0a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD
7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD
7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l0
1APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l011PO51lD7a@O0goo
00<007ooOol037oo0008Ool20003Ool0U000OomooaPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO0goo00<007ooOol037oo000=Ool00`00Oomoo`0351l01APO51lD
7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH
7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D
7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH
7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD
7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0661lD7a@O51lH7a@O0goo00<007ooOol037oo000=Ool0
U000OomooaPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO0goo
00<007ooOol037oo000=Ool01@00Oomooa@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH
7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD
7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH
7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D
7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH
7`0351l011PO51lD7a@O0goo00<007ooOol037oo000=Ool20000TWoo61lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61l3Ool00`00Oomoo`0<Ool000eoo`03001oogoo
00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O
51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO
51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O
00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO
00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00HH7a@O51lD7aPO51l3Ool00`00Oomo
o`0<Ool000eoo`03001oogoo00H0002;61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61l00goo00<007ooOol037oo000=Ool00`00Oomoo`060Il<000351l01APO51lD7a@O61l00a@O00DH
7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D
7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH
7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD
7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l011PO51lD7a@O0goo00<007oo
Ool037oo000=Ool05@00Oomoo`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`060000
NAPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO00=oo`03001oogoo00aoo`003Goo00<007ooOol0606O1@000Q@O00DH7a@O51lD7aPO00<D7`05
61lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`03
51l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O
61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O
51lD7aPO00<D7`0561lD7a@O51lH7`0351l00aPO51moo`02Ool00`00Oomoo`0<Ool000eoo`0P001o
ogoo0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`D0001_61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO00=oo`03001oogoo
00aoo`003Goo0P0000=oo`6O0Il0806O1P0000<D7aPO51l00Q@O00DH7a@O51lD7aPO00<D7`0561lD
7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l0
1APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l0
0a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00@H7a@O51lD
7`=oo`03001oogoo00aoo`003Goo02/007ooOol1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`040000IAPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61l00goo00<007ooOol037oo000=Ool00`00Oomoo`0/0Il300001APO51lD
7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH
7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D
7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH
7`0351l00aPO51moo`02Ool00`00Oomoo`0<Ool000eoo`0b001oogoo0Il1O`6O0Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`6O0Gl1W`@0001N51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61l3Ool00`00Oomoo`0<Ool000eoo`03001oogoo03<1
W`<000<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD
7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l0
1APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l0
0a@O0goo00<007ooOol037oo000=Ool0>@00Oomoo`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl1W`5o0Il1O`020000FAPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61l00goo00<007ooOol037oo000=Ool00`00Oomoo`0h0Il20000
0a@O61lD7`0251l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l0
1APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l0
0a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00<H7a@OOol0
0Woo00<007ooOol037oo000=Ool20000>goo0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl00P0005DH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7`03Ool00`00Oomoo`0<Ool000eoo`03001oogoo03`1W`8000<D7`05
61lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`03
51l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O
61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l3Ool00`00Oomoo`0<Ool000eoo`11
001oogoo0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`020000DAPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7`03
Ool00`00Oomoo`0<Ool000eoo`03001oogoo0401W`04000D7a@O61l351l01APO51lD7a@O61l00a@O
00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO
00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O
51lH7`0351l00aPO51l00002Ool00`00Oomoo`0<Ool000eoo`2D001oogoo0Il1O`6O0Gl1W`5o0Il1
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`0061lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O0003Ool00`00Oomoo`0<Ool000eoo`03001o
ogoo0481W`04000D7a@O61l351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O
51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO
51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0200001F0OOomoogoo00003Woo
000=Ool0U000Oomoo`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O000H7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7`00I1mP
7f@O0goo00<007ooOol037oo000=Ool00`00Oomoo`140Il0100051lD7aPO0a@O00DH7a@O51lD7aPO
00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O
51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01QPO
001T7f0OI1mP7`=oo`03001oogoo00aoo`003Goo0P0008]oo`000001W`5o0Il1O`6O0Gl1W`5o0Il1
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`0061lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7`0200001F@OH1mT7f0OI1l00goo00<007ooOol037oo000=Ool00`00Oomoo`02
0ol400100Il0100051lD7aPO0a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD
7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH
7a@O51lD7aPO00<D7`0<61lD7a@O51l0060OI1mP7f@OH1mT7f0O0goo00<007ooOol037oo000=Ool0
0`00Oomoo`060ol30000R05o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`6O0Gl1W`5o0Il001PO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O001P7f@OH1mT7f0OI1mP7f@O0goo00<007ooOol0
37oo000=Ool00`00Oomoo`090ol4000k0Il0100051lD7aPO0a@O00DH7a@O51lD7aPO00<D7`0561lD
7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l0
1APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0:001P7f@OH1mT7f0OI1mP7f@OH1l3Ool00`00
Oomoo`0<Ool000eoo`03001oogoo00d3o`8000220Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o
0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o
0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il001PO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7`00H1mT7f0OI1mP7f@OH1mT7f0OI1l3Ool00`00Oomo
o`0<Ool000Qoo`8000=oo`03001oogoo00l3o`<003P1W`04000D7a@O61l351l01APO51lD7a@O61l0
0a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD
7aPO00<D7`0561lD7a@O51lH7`0351l041PO51lD7a@O001P7f@OH1mT7f0OI1mP7f@OH1mT7f0O0goo
00<007ooOol037oo0007Ool01000Oomoo`000Woo00<007ooOol04P?o0P0007d1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl001@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7`00H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O00=o
o`03001oogoo00aoo`002Woo00<007ooOol00P0000=oo`?o0ol04P?o0P00=@6O00D001@O51lD7aPO
00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O
51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0>001P7f@OH1mT7f0OI1mP7f@O
H1mT7f0OI1mP7`9oo`8000ioo`0027oo0P000goo00<007ooOol05P?o0P0007T1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`0051lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO001T7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7`03Ool00`00Oomo
o`0<Ool000Qoo`03001oogoo009oo`03001oogoo01P3o`800381W`03000H7a@O008D7`0561lD7a@O
51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO
51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0@61l006@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O
H1l3Ool00`00Oomoo`0<Ool000Qoo`<0009oo`03001oogoo01X3o`80001e0Il1O`6O0Gl1W`5o0Il1
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il001PO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7`00I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O00=oo`03001oogoo00aoo`003Goo
00<007ooOol070?o0P00;`6O00<001@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`03
51l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l05QPO51lD7a@O
61l006@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O0goo00<007ooOol037oo000=Ool00`00
Oomoo`0N0ol0L`000Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`0051lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO001T7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O
00=oo`03001oogoo00aoo`003Goo00<007ooOol07`?o00<0006O0Il0;06O00@001@O51lH7`<D7`05
61lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`03
51l01APO51lD7a@O61l00a@O01@H7`00I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O
0goo00<007ooOol037oo000=Ool200000goo0ol3o`0N0ol20000K`6O0Gl1W`5o0Il1O`6O0Gl1W`5o
0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o
0Il1O`6O0Gl1W`0061lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51l0060OI1mP7f@OH1mT7f0O
I1mP7f@OH1mT7f0OI1mP7f@OH1mT7`03Ool00`00Oomoo`0<Ool000eoo`03001oogoo0283o`030001
W`6O02X1W`05000D7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD
7a@O51lH7`0351l01APO51lD7a@O61l00a@O01XH7a@O51lD7aPO51l0060OI1mP7f@OH1mT7f0OI1mP
7f@OH1mT7f0OI1mP7f@OH1mT7f0O0goo00<007ooOol037oo000=Ool00`00Oomoo`0S0ol0KP000Il1
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`0051lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7`00I1mP
7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1l3Ool00`00Oomoo`0<Ool000eoo`03001o
ogoo02@3o`8002X1W`03000H7a@O008D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH
7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00Q@O01D006@OH1mT7f0OI1mP
7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l00goo00<007ooOol037oo000=Ool00`00Oomoo`0V
0ol0J`000Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o
0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O000H7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51l0060O
I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1l00goo00<007ooOol037oo000=Ool0
0`00Oomoo`0W0ol00`000Il1W`0W0Il00`0051lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD
7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O01XH7a@O51lD7`00H1mT7f0OI1mP
7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O0goo00<007ooOol037oo000=Ool01000Oomo
o`009`?o0P0006L1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`0051lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61l006@OH1mT
7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7`03Ool00`00Oomoo`0<Ool000eoo`80
0004Ool3k000000W0ol00`000Il1W`0U0Il0100051lD7aPO0a@O00DH7a@O51lD7aPO00<D7`0561lD
7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0H61l006@OH1mT7f0OI1mP
7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O0goo00<007ooOol037oo000=Ool01P00Oomo
o`?/0n/3k08002H3o`1V0001W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il001PO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51l0060OI1mP
7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O0goo00<007ooOol037oo000=Ool0
0`00Oomoo`050n`00`000ol3o`0T0ol00`000Il1W`0T0Il01@0051lD7a@O61l00a@O00DH7a@O51lD
7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O01hH7a@O51lD7aPO51l0060OI1mP
7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l3Ool00`00Oomoo`0<Ool000eo
o`09001oogoo0n`3j`?/0n/3k0?[008002D3o`1T0001W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o
0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`0061lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61l006@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1l3Ool00`00Oomo
o`0<Ool000eoo`03001oogoo00P3k08002@3o`030001W`6O0281W`03000D7aPO00<D7`0561lD7a@O
51lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0251l06@00
I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O00=oo`03001oogoo
00aoo`003Goo00d007ooOol3k0?[0n`3j`?/0n/3k0?[0n`3j`02000S0ol0HP000Il1O`6O0Gl1W`5o
0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o
000D7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O001P7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O
I1l3Ool00`00Oomoo`0<Ool000eoo`03001oogoo00`3k0800283o`030001W`6O0241W`04000D7a@O
61l351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l07QPO51lD
7a@O61l006@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7`=oo`03
001oogoo00aoo`003Goo018007ooOol3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[000R0ol0H000
0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o
0Il1O`6O0Gl001@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO001T7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O
I1mP7f@OH1mT7`=oo`03001oogoo00aoo`003Goo0P0000=oo`?/0n`03@?/00<000?o0ol080?o00<0
006O0Il0806O00D001@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O00DH
7a@O51lD7aPO00<D7`0L61lD7`00H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT
7f0OI1mP7f@OH1l3Ool00`00Oomoo`0<Ool000eoo`0C001oogoo0n`3j`?/0n/3k0?[0n`3j`?/0n/3
k0?[0n`3j`?/0n/00P008@?o05h0006O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`0061lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO001T7f0OI1mP7f@OH1mT7f0OI1mP7f@O
H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7`=oo`03001oogoo00aoo`003Goo00<007ooOol04P?/
00<000?o0ol07`?o00<0006O0Il07P6O00<001@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O
51lH7`0351l01APO51lD7a@O61l00a@O01hH7a@O51lD7`00H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O
H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l3Ool00`00Oomoo`0<Ool000eoo`0F001oogoo0n`3j`?/
0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`2000P0ol0G0000Il1O`6O0Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`0051lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51l0060OI1mP7f@O
H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1l3Ool00`00Oomoo`0<Ool0
00eoo`03001oogoo01D3k0030003o`?o01d3o`030001W`6O01h1W`04000D7a@O61l351l01APO51lD
7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l071PO001T7f0OI1mP7f@OH1mT
7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O0goo00<007ooOol037oo000=Ool0
6P00Oomoo`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[000O0ol0F`00
0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O
0Gl001@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO
51lH7a@O001P7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O
00=oo`03001oogoo00aoo`003Goo00<007ooOol05`?/00<000?o0ol070?o00<0006O0Il07@6O00<0
01PO51l00Q@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O01hH
7a@O001P7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l3
Ool00`00Oomoo`0<Ool000eoo`80000JOol3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/
0n/3k0?[0n`3j`?/0n/001h3o`1J0001W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`0061lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7`00I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT
7f0OI1mP7f@OH1mT7f0OI1mP7f@O0goo00<007ooOol037oo000=Ool00`00Oomoo`0I0n`00`000ol3
o`0L0ol00`000Il1W`0L0Il00`0051lH7`0351l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D
7`0561lD7a@O51lH7`0351l07000H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT
7f0OI1mP7f@OH1mT7f0O0goo00<007ooOol037oo000=Ool07@00Oomoo`?/0n/3k0?[0n`3j`?/0n/3
k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/00P007@?o05P0006O0Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il001PO51lH7a@O61lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O001P7f@OH1mT7f0OI1mP7f@O
H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7`=oo`03001oogoo00aoo`003Goo
00<007ooOol070?/00<000?o0ol06P?o00<0006O0Il06`6O00D001@O51lD7aPO00<D7`0561lD7a@O
51lH7`0351l01APO51lD7a@O61l00a@O02<H7a@O51lD7aPO001T7f0OI1mP7f@OH1mT7f0OI1mP7f@O
H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7`00009oo`03001oogoo00aoo`003Goo0240
07ooOol000?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3
j`?/000070?o05L0005o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O000H7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7`00H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT
7f0O00000goo00<007ooOol037oo00000goo000000030002Ool20003Ool01@00OomooaOP000070?/
00<000?o0ol06P?o00<0006O0Il06P6O00<001PO51l00Q@O00DH7a@O51lD7aPO00<D7`0561lD7a@O
51lH7`0351l01APO51lD7a@O61l00Q@O01d006@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O
I1mP7f@OH1mT7f0OI1mP7f@OH1mT7`0000=oo`03001oogoo00aoo`000goo00D007ooOomoo`00009o
o`04001oogoo0002Ool081_P5n0000000n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[
0n`3j`?/0n/3k0?[0n`3j`?/000K0ol0EP000Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl001@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7a@O61l006@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT
7f0OI1mP7f@OH1mT7`00O1<3Ool00`00Oomoo`0<Ool000=oo`05001oogooOol00002Ool00`00Oomo
o`0200001Woo5n0Gh1OP6n0001/3k0030003o`?o01T3o`030001W`6O01T1W`04000D7a@O61l351l0
1APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0R61lD7a@O51l0060OI1mP7f@OH1mT7f0OI1mP
7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7`00O182Ool2000>Ool000=oo`05001o
ogooOol00002Ool01000Oomoo`000Woo00DKh1OP6n0Gh1_P0080000K0n/3k0?[0n`3j`?/0n/3k0?[
0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k00001/3o`1D0001W`5o0Il1O`6O0Gl1W`5o
0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o000D7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO001T7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O
I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1l007`B0goo00<007ooOol037oo0003Ool01@00Oomoogoo
00000Woo00@007ooOol0009oo`035n0Kh1OP008Gh0036n0Gh00001X3k0030003o`?o01P3o`030001
W`6O01P1W`03000H7a@O008D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00a@O028H7a@O51lD
7aPO51l0060OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1l0
009l4P=oo`03001oogoo00aoo`000Woo0P0017oo0P000goo02H007ooOolKh1OP6n0Gh1_P5n0Kh1OP
0003j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`006`?o05<0
005o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o000D7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7`00H1mT7f0OI1mP7f@OH1mT
7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O001l4W`C00=oo`03001oogoo00ao
o`003Goo00<007ooOol00aOP00HKh1OP5n0Gh1_P5n02000H0n`00`000ol3o`0H0ol00`000Il1W`0G
0Il0100051lD7aPO0a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l07aPO51lD7`00I1mP
7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7`0000=l4P=oo`03
001oogoo00aoo`003Goo02P007ooOolKh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P0003k0?[0n`3j`?/0n/3
k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/001X3o`1@0001W`5o0Il1O`6O0Gl1W`5o
0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl001@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO51lH7a@O001P7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O
I1mP7f@OH1mT7f0OI1mP7f@O001l4`9l4P=oo`03001oogoo00aoo`003Goo00D007ooOolGh1_P00<G
h0056n0Gh1OP5n0Kh0025n000`000n`3k00F0n`00`000ol3o`0G0ol00`000Il1W`0G0Il01@0051lD
7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l071PO001T7f0OI1mP7f@OH1mT
7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7`0017`B0goo00<007ooOol037oo000=
Ool0:P00Oomooa_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P0003k0?[0n`3j`?/0n/3k0?[0n`3j`?/
0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/001T3o`1=0001O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il001PO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7`00H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP
7f@OH1l00003O1801G`COomoogoo00003Woo000=Ool200001Goo5n0Gh1OP6n000aOP00DKh1OP5n0G
h1_P008Gh0030003k0?/01H3k0030003o`?o01H3o`030001W`6O01H1W`03000D7aPO00<D7`0561lD
7a@O51lH7`0351l01APO51lD7a@O61l00a@O01dH7a@O001P7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP
7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7`0000El4P=oo`03001oogoo00aoo`003Goo02/007ooOolK
h1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP0003j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3
k0?[0n`3j`?/0n/3k0?[0n`0000H0ol0C0000Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1
O`6O0Gl1W`5o0Il1O`6O000H7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD
7aPO001T7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7`000W`B
00=l4g`BO1800goo00<007ooOol037oo000=Ool01@00OomooaOP6n000aOP00DKh1OP5n0Gh1_P00<G
h0036n0Gh00001P3k0030003o`?o01H3o`030001W`6O01D1W`05000D7a@O51lH7`0351l01APO51lD
7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0J001P7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT
7f0OI1mP7f@OH1mT7f0OI1l000Il4P=oo`03001oogoo00aoo`003Goo02d007ooOolKh1OP6n0Gh1_P
5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh0000n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/
0n/3k0?[0n`3j`?/00005`?o04`0006O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O
0Gl1W`5o0Il001PO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61l006@O
H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7`00O19l4`=l4P05O1=o
ogooOol0000>Ool000eoo`03001oogoo00<Gh0056n0Gh1OP5n0Kh0035n001a_P5n0Gh1OP6n0Gh000
01P3k0030003o`?o01@3o`030001W`6O01D1W`03000D7aPO00<D7`0561lD7a@O51lH7`0351l01APO
51lD7a@O61l00a@O01/H7a@O001P7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O
H1mT7f0O00001g`B0goo00<007ooOol037oo000=Ool0;`00Oomooa_P5n0Kh1OP6n0Gh1_P5n0Kh1OP
6n0Gh1_P5n0Kh1OP6n0Gh0000n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[
0n`3j`?/00005P?o04T0005o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O
000H7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO001T7f0OI1mP7f@O
H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O001l4`03O1800g`CO19l4P03Ool00`00
Oomoo`0<Ool000eoo`05001oogoo5n0Kh0035n001A_P5n0Gh1OP6n000aOP00LKh1OP5n0Gh1_P5n00
000G0n`00`000ol3o`0D0ol00`000Il1W`0D0Il01@0051lD7a@O61l00a@O00DH7a@O51lD7aPO00<D
7`0561lD7a@O51lH7`0351l06000H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT
7f0OI1l000Ql4P=oo`03001oogoo00aoo`003Goo0P0002iooa_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0G
h1_P5n0Kh1OP6n0Gh1_P5n0000?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3
j`?/0n/001H3o`160001W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il001PO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61l006@OH1mT7f0OI1mP7f@O
H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O0003O1801G`CO19l4W`BO1<00goo00<007ooOol0
37oo000=Ool00`00Oomoo`035n001A_P5n0Gh1OP6n000aOP00DKh1OP5n0Gh1_P00<Gh0036n0Gh000
01H3k0030003o`?o01@3o`030001W`6O01@1W`03000H7a@O008D7`0561lD7a@O51lH7`0351l01APO
51lD7a@O61l00a@O01TH7`00I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O
H1l00009O183Ool00`00Oomoo`0<Ool000eoo`0a001oogoo000Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP
6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0000?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[
0n`3j`?/00005@?o04D0006O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o
000D7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51l0060OI1mP7f@OH1mT7f0O
I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O00000W`B00El4g`BO19l4W`C009l4P=oo`03001o
ogoo00aoo`003Goo00D007ooOomSh00000<Gh0056n0Gh1OP5n0Kh0035n001A_P5n0Gh1OP6n000aOP
00<Kh1OP00005@?/00<000?o0ol050?o00<0006O0Il04`6O00@001@O51lH7`<D7`0561lD7a@O51lH
7`0351l01APO51lD7a@O61l00Q@O01L006@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP
7f@OH1mT7`0000Yl4P=oo`03001oogoo00aoo`003Goo038007ooOomWh6?P000Gh1_P5n0Kh1OP6n0G
h1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0000?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3
j`?/0n/3k0?[0n`3j`005@?o04D0005o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o
0Il1O`0051lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O001P7f@OH1mT7f0O
I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l007`BO1<00g`B00El4g`BO19l4W`C00=oo`03
001oogoo00aoo`003Goo00L007ooOomSh6OPHn0000035n001A_P5n0Gh1OP6n000aOP00DKh1OP5n0G
h1_P00<Gh0036n0Gh00001@3k0030003o`?o01@3o`030001W`6O0181W`03000H7a@O008D7`0561lD
7a@O51lH7`0351l01APO51lD7a@O61l00a@O01LH7`00I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP
7f@OH1mT7f0OI1mP7`0000]l4P=oo`03001oogoo00aoo`003Goo03<007ooOomWh6?PIn1Sh0005n0K
h1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP0003j`?/0n/3k0?[0n`3j`?/0n/3
k0?[0n`3j`?/0n/3k0?[0n`3j`?/00005@?o0480006O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl001@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7`00H1mT
7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7`00O1<3O1801G`CO19l4W`BO1<00W`B
0goo00<007ooOol037oo000=Ool200001gooHn1Wh6?PIn1Sh00000<Gh0056n0Gh1OP5n0Kh0035n00
1A_P5n0Gh1OP6n000aOP00<Kh1OP000050?/00<000?o0ol04P?o00<0006O0Il04P6O00@001@O51lH
7`<D7`0561lD7a@O51lH7`0351l01APO51lD7a@O61l00Q@O01H006@OH1mT7f0OI1mP7f@OH1mT7f0O
I1mP7f@OH1mT7f0OI1mP7f@OH1l000]l4P=oo`03001oogoo00aoo`003Goo03@007ooOomWh6?PIn1S
h6OPHn0001OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0000?/0n/3k0?[0n`3
j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/001D3o`0o0001O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o
0Il1O`6O0Gl1W`5o0Il1O`0051lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
001P7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7`0000=l4P05O1=l4W`BO19l4`03
O1801G`COomoogoo00003Woo000=Ool02`00Oomoof?PIn1Sh6OPHn1Wh6?P00000aOP00DKh1OP5n0G
h1_P00<Gh0056n0Gh1OP5n0Kh0035n000a_P0003k00C0n`00`000ol3o`0B0ol00`000Il1W`0A0Il0
0`0061lD7`0251l01APO51lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0F61l006@OH1mT7f0OI1mP
7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O000<O183Ool00`00Oomoo`0<Ool000eoo`0e001oogoo
In1Sh6OPHn1Wh6?PIn1Sh0005n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh000
0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`0000D0ol0@@000Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl001@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7`00H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l007`BO1<00g`B
00El4g`BO19l4W`C009l4P=oo`03001oogoo00aoo`003Goo00d007ooOomSh6OPHn1Wh6?PIn1Sh6OP
Hn0000035n001A_P5n0Gh1OP6n000aOP00DKh1OP5n0Gh1_P00<Gh0030003k0?/0183k0030003o`?o
0183o`030001W`6O0141W`03000D7aPO00<D7`0561lD7a@O51lH7`0351l06aPO51lD7a@O61lD7`00
H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l0000=O183Ool00`00Oomoo`0<Ool0
00eoo`0f001oogooIn1Sh6OPHn1Wh6?PIn1Sh6OPHn0001OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP
6n0Gh1_P5n0Kh1OP0003j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`0050?o03h0
006O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O000H7a@O61lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7aPO51lH7`00I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1l0
07`C0g`B00El4g`BO19l4W`C00=l4P05O1=oogooOol0000>Ool000eoo`0?001oogooHn1Wh6?PIn1S
h6OPHn1Wh6?PIn1Sh00000<Gh0056n0Gh1OP5n0Kh0035n001A_P5n0Gh1OP6n000QOP00<000?/0n`0
4P?/00<000?o0ol04@?o00<0006O0Il04@6O00D001@O51lD7aPO00<D7`0561lD7a@O51lH7`0351l0
6APO51lD7a@O001P7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7`0000el4P=oo`03
001oogoo00aoo`003Goo0P0003EoofOPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh0005n0Kh1OP6n0Gh1_P
5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P0003k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/
0n/3k00001<3o`0m0001W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`0061lD7aPO
51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61l006@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O
H1mT7f0OI1mP7f@O00000W`B00El4g`BO19l4W`C00=l4P03O1=l4W`B00=oo`03001oogoo00aoo`00
3Goo014007ooOomSh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh00000<Gh0056n0Gh1OP5n0Kh0035n00
1A_P5n0Gh1OP6n000QOP00<000?/0n`04@?/00<000?o0ol04@?o00<0006O0Il0406O00<001@O61l0
0a@O00DH7a@O51lD7aPO00<D7`0J61lD7a@O51lH7a@O001P7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP
7f@OH1mT7f0OI1l000il4P=oo`03001oogoo00aoo`003Goo03P007ooOomWh6?PIn1Sh6OPHn1Wh6?P
In1Sh6OPHn1Wh6?P000Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0000?/0n/3k0?[
0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`004`?o03d0005o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl1W`0061lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61l006@OH1mT
7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7`00O19l4`03O1801G`CO19l4W`BO1<00g`B00El
4gooOomoo`0000ioo`003Goo01<007ooOomSh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn000003
5n001A_P5n0Gh1OP6n000aOP00DKh1OP5n0Gh1_P008Gh0030003k0?/00l3k0030003o`?o0143o`03
0001W`6O0101W`05000D7a@O51lH7`0351l01APO51lD7a@O61l00a@O01PH7a@O51lD7`00H1mT7f0O
I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O000>O183Ool00`00Oomoo`0<Ool000eoo`0i001o
ogooIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OP000Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0K
h1OP6n0Gh1_P5n0000?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k00001<3o`0j0001
W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O000H7a@O61lD7aPO51lH7a@O61lD7aPO51lH
7a@O61lD7aPO51lH7`00I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7`00O1<3O1801G`C
O19l4W`BO1<00g`B00=l4g`BO1800goo00<007ooOol037oo000=Ool05@00Oomoof?PIn1Sh6OPHn1W
h6?PIn1Sh6OPHn1Wh6?PIn1Sh6OP000Kh0035n001A_P5n0Gh1OP6n000aOP00LKh1OP5n0Gh1_P5n00
000A0n`00`000ol3o`0A0ol00`000Il1W`0?0Il00`0051lH7`0351l01APO51lD7a@O61l00a@O01TH
7a@O51lD7aPO51l0060OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l0000?O183Ool00`00
Oomoo`0<Ool00003Ool0000000<0009oo`8000=oo`0j001oogooIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1W
h6?PIn1Sh6OPHn1Wh0006n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh0000n`3j`?/0n/3
k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[000B0ol0>@000Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o
0Il1O`6O0Gl1W`0061lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61l006@OH1mT7f0O
I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1l00002O1801G`CO19l4W`BO1<00g`B00El4g`BO19l4W`C
00=oo`03001oogoo00aoo`000goo00D007ooOomoo`00009oo`04001oogoo0002Ool056?PIn1Sh6OP
Hn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh0006n035n001A_P5n0Gh1OP6n000aOP00HKh1OP5n0G
h1_P000A0n`00`000ol3o`0@0ol00`000Il1W`0?0Il01@0051lD7a@O61l00a@O00DH7a@O51lD7aPO
00<D7`0G61lD7a@O51l0060OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l0000?O183Ool0
0`00Oomoo`0<Ool000=oo`03001oogoo00Aoo`03001oogoo0080000iOomWh6?PIn1Sh6OPHn1Wh6?P
In1Sh6OPHn1Wh6?PIn1Sh6OPHn0001OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0000?[
0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0000143o`0j0001W`5o0Il1O`6O0Gl1W`5o
0Il1O`6O0Gl1W`5o0Il1O`6O0Gl001@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51l0060O
I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l007`BO1<3O1801G`CO19l4W`BO1<00g`B00=l
4g`BO1800Woo0P003Woo0003Ool00`00Oomoo`02Ool20003Ool06@00Oomoo`00In1Sh6OPHn1Wh6?P
In1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?P000Gh1_P00<Gh0056n0Gh1OP5n0Kh0035n001A_P5n0Gh1OP
00004@?/00<000?o0ol03`?o00<0006O0Il0406O00<001PO51l00Q@O00DH7a@O51lD7aPO00<D7`0H
61lD7a@O51lH7`00I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7`0047`B0goo00<007oo
Ool037oo0003Ool00`00Oomoo`02Ool00`00Oomoo`02Ool0?000OomoogiP001Wh6?PIn1Sh6OPHn1W
h6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn0001OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P0003
k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/00143o`0g0001O`6O0Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`6O0Gl1W`5o000D7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O001P7f@O
H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O00000g`B00El4g`BO19l4W`C00=l4P05O1=l4W`B
O19l4`03Ool00`00Oomoo`0<Ool0009oo`8000Aoo`<0009oo`03001oogoo009n@00C001Wh6?PIn1S
h6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OP00000QOP00DKh1OP5n0Gh1_P00<Gh0056n0Gh1OP5n0K
h0025n000`000n`3k00?0n`00`000ol3o`0?0ol00`000Il1W`0?0Il0100051lD7aPO0a@O00DH7a@O
51lD7aPO00<D7`0F61lD7a@O001T7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O000@O180
1000Oomoo`003Woo000=Ool00`00Oomoo`02OT00>7iP001Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1W
h6?PIn1Sh6OP000Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh0000n/3k0?[0n`3j`?/0n/3
k0?[0n`3j`?/0n/3k0?[0n`3j`?/000@0ol0=P000Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o
0Il1O`0051lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7`00H1mT7f0OI1mP7f@OH1mT7f0O
I1mP7f@OH1mT7f0OI1l0009l4P05O1=l4W`BO19l4`03O1801G`CO19l4W`BO1<00W`B00@007ooOol0
00ioo`003Goo00<007ooOol017i0018006OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?P0003
5n001A_P5n0Gh1OP6n000aOP00LKh1OP5n0Gh1_P5n00000A0n`00`000ol3o`0>0ol00`000Il1W`0?
0Il00`0061lD7`0251l01APO51lD7a@O61l00a@O01LH7a@O51lD7aPO001T7f0OI1mP7f@OH1mT7f0O
I1mP7f@OH1mT7f0OI1mP7`00015l4P04001oogoo000>Ool000eoo`04001oogooOV03OT00=WiP001W
h6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh0005n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0K
h1OP6n0Gh0000n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`00103o`0g0001O`6O0Gl1W`5o
0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o000D7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O
001P7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7`00O1<00g`B00El4g`BO19l4W`C00=l4P06
O1=l4W`BO19l4`000Woo00<007ooOol037oo000=Ool200000gooOT1n@004OT004P00In1Sh6OPHn1W
h6?PIn1Sh6OPHn1Wh6?PIn1Sh6OP000Kh0<Gh0056n0Gh1OP5n0Kh0035n001Q_P5n0Gh1OP6n000143
k0030003o`?o00d3o`030001W`6O00l1W`04000D7a@O61l351l01APO51lD7a@O61l00a@O01DH7a@O
51l006@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O000047`B00D007ooOomoo`0000ioo`00
3Goo00<007ooOol00Wi003UnH7i0OT1n@7iP001Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn00
01_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0000?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3
j`?/0n/3k0?[000040?o03@0006O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o000D7aPO51lH
7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O001P7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l0
00=l4P05O1=l4W`BO19l4`03O1801g`CO19l4W`BO1=l4P0000=oo`03001oogoo00aoo`003Goo00<0
07ooOol027i0018006OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh0006n035n001A_P5n0Gh1OP
6n000aOP00DKh1OP5n0Gh0000103k0030003o`?o00h3o`030001W`6O00h1W`03000H7a@O008D7`05
61lD7a@O51lH7`0351l05QPO51lD7a@O61l006@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7`00
47`B00<007`0Ool00Woo00<007ooOol037oo000=Ool01000OomoogiP0gi003QnH7i0OT1n@000Hn1W
h6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn0001_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0K
h1OP0003j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0003`?o03H0006O0Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`6O0Gl1W`5o0Il001PO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51lH7`00I1mP7f@O
H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l007`BO1<3O1801G`CO19l4W`BO1<00g`B00El4g`BO180
07`000=oo`03001oogoo00aoo`003Goo00<007ooOol02Gi001<006?PIn1Sh6OPHn1Wh6?PIn1Sh6OP
Hn1Wh6?PIn1Sh0005n0Kh0035n001A_P5n0Gh1OP6n000aOP00@Kh1OP5n000103k0030003o`?o00d3
o`030001W`6O00l1W`03000D7aPO00<D7`0561lD7a@O51lH7`0351l051PO51l0060OI1mP7f@OH1mT
7f0OI1mP7f@OH1mT7f0OI1mP7f@O000@O1800`00O01oo`02Ool00`00Oomoo`0<Ool000eoo`03001o
ogoo009n@005OV1n@7i0OT1nH003OT00<P00Hn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?P000G
h1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh0000n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3
j`?/000?0ol0=0000Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`0061lD7aPO51lH7a@O
61lD7aPO51lH7a@O61lD7aPO001T7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7`000g`B00El
4g`BO19l4W`C00=l4P05O1=l4W`BO1800002O003Ool00`00Oomoo`0<Ool000eoo`03001oogoo00]n
@00C001Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn0001OP6n000aOP00DKh1OP5n0Gh1_P00<G
h0036n0Gh00000l3k0030003o`?o00h3o`030001W`6O00h1W`05000D7a@O51lH7`0351l01APO51lD
7a@O61l00a@O0140060OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7`00011l4P03001l07`000=o
o`03001oogoo00aoo`003Goo0P0000=oogiPOT000Wi000EnH7i0OT1n@7iP00=n@00a001Sh6OPHn1W
h6?PIn1Sh6OPHn1Wh6?PIn1Sh6OP000Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh0000n/3
k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[00003`?o0380005o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O000H7a@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O61l006@OH1mT7f0OI1mP7f@OH1mT
7f0OI1mP7f@OH1mT7`000W`B00El4g`BO19l4W`C00=l4P06O1=l4W`BO19l4`000g`00goo00<007oo
Ool037oo000=Ool00`00Oomoo`0<OT004@00In1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OP0000
0QOP00DKh1OP5n0Gh1_P00<Gh0066n0Gh1OP5n0Kh0003`?/00<000?o0ol03@?o00<0006O0Il03P6O
00<001@O61l00a@O00DH7a@O51lD7aPO00<D7`0C61lD7`00H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O
H1mT7f0O00003g`B00@007`0O01l00=oo`03001oogoo00aoo`003Goo00<007ooOol00Wi000EnH7i0
OT1n@7iP00=n@00cOV1n@7i0001Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn0001_P5n0Kh1OP
6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0000?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`0000l3o`0c
0001O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`0061lD7aPO51lH7a@O61lD7aPO51lH7a@O
61lD7aPO001T7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1l007`C00=l4P05O1=l4W`BO19l4`03
O18017`CO19l4P000g`00goo00<007ooOol037oo000=Ool00`00Oomoo`0=OT004@00Hn1Wh6?PIn1S
h6OPHn1Wh6?PIn1Sh6OPHn1Wh6?P00000aOP00DKh1OP5n0Gh1_P00<Gh0056n0Gh1OP5n00000?0n`0
0`000ol3o`0<0ol00`000Il1W`0>0Il01@0051lD7a@O61l00a@O00DH7a@O51lD7aPO00<D7`0A001P
7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l0000>O1800`00O01l0002O003Ool00`00Oomoo`0<
Ool000eoo`04001oogooOV03OT001GiPOT1n@7i0OV000gi0039nH7i0001Sh6OPHn1Wh6?PIn1Sh6OP
Hn1Wh6?PIn1Sh6OPHn0001OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0000?/0n/3k0?[0n`3j`?/
0n/3k0?[0n`3j`?/0n/3k0003`?o0340006O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`0061lD
7aPO51lH7a@O61lD7aPO51lH7a@O61lD7aPO001T7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1l0
0002O1801G`CO19l4W`BO1<00g`B00El4g`BO19l4P0000Al00=oo`03001oogoo00aoo`003Goo00<0
07ooOol03gi0014006?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn0001_P00<Gh0056n0Gh1OP5n0K
h0035n0011_P5n0Gh0003`?/00<000?o0ol030?o00<0006O0Il03@6O00<001@O61l00a@O00DH7a@O
51lD7aPO00<D7`0B61lD7`00H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7`003g`B00<007`0O000
0W`00goo00<007ooOol037oo000=Ool00`00Oomoo`02OT001GiPOT1n@7i0OV000gi003EnH7i0OT1n
@7iPOT0006?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn0001_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0G
h1_P5n0000?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`0000h3o`0b0001W`5o0Il1O`6O0Gl1
W`5o0Il1O`6O0Gl1W`5o0Il001PO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51l0060OI1mP7f@OH1mT
7f0OI1mP7f@OH1mT7f0OI1mP7`00O19l4`=l4P05O1=l4W`BO19l4`03O1800g`C001l0004O003Ool0
0`00Oomoo`0<Ool000eoo`800003Ool007i000in@00B001Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1W
h6?P000Gh1_P0aOP00DKh1OP5n0Gh1_P00<Gh0036n0000?/00l3k0030003o`?o00/3o`030001W`6O
00d1W`05000D7a@O51lH7`0351l01APO51lD7a@O61l00Q@O014006@OH1mT7f0OI1mP7f@OH1mT7f0O
I1mP7f@OH1mT7`0000il4P03001l07`000=l00=oo`03001oogoo00aoo`003Goo00D007ooOoml0000
009n@005OV1n@7i0OT1nH003OT00=7iPOT1n@7i0OV0006OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OP
Hn0001OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0000?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/
0n/3k0003@?o0300006O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`0061lD7aPO51lH7a@O61lD
7aPO51lH7a@O61lD7`00H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0O0003O1801G`CO19l4W`B
O1<00g`B00Al4g`BO18000El00=oo`03001oogoo00aoo`003Goo00<007ooOol00W`000<007i0OT00
3Gi000l006OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OP00000QOP00DKh1OP5n0Gh1_P00<Gh0056n0G
h1OP5n00000?0n`00`000ol3o`0<0ol00`000Il1W`0<0Il00`0051lH7`0351l01APO51lD7a@O61l0
0a@O014H7`00I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7`0000ml4P03001l07`000=l00=oo`03
001oogoo00aoo`003Goo00<007ooOol00g`000@007i0OT1nH0=n@005OV1n@7i0OT1nH003OT00;P00
Hn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh0006n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh000
0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0003P?o02h0005o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il001PO51lH7a@O61lD7aPO51lH7a@O61lD7aPO51l0060OI1mP7f@OH1mT7f0OI1mP7f@OH1mT
7f0OI1l0009l4P05O1=l4W`BO19l4`03O1801G`CO19l4W`B00001W`00goo00<007ooOol037oo000=
Ool00`00Oomoo`04O0000`00OT1n@00<OT003`00Hn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn000003
5n001A_P5n0Gh1OP6n000aOP00@Kh1OP5n0000h3k0030003o`?o00`3o`030001W`6O00`1W`05000D
7a@O51lH7`0351l01APO51lD7a@O61l00Q@O010006@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l0
00il4P03001l07`000Al00=oo`03001oogoo00aoo`003Goo00<007ooOol01G`000@007i0OT1nH0=n
@005OV1n@7i0OT1nH002OT00;@00In1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn0001_P5n0Kh1OP6n0G
h1_P5n0Kh1OP6n0Gh1_P5n0000?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[0n`0000>0ol0;`000Gl1
W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`0061lD7aPO51lH7a@O61lD7aPO51lH7a@O61lD7`00H1mT
7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1mT7`00O1<00g`B00El4g`BO19l4W`C00=l4P03O1=l4P0000Il
00=oo`03001oogoo00aoo`003Goo00<007ooOol01G`000<007i0OT0037i0010006OPHn1Wh6?PIn1S
h6OPHn1Wh6?PIn1Sh6OP000Kh0<Gh0056n0Gh1OP5n0Kh0035n000a_P5n00000>0n`00`000ol3o`0;
0ol00`000Il1W`0<0Il00`0051lH7`0351l01APO51lD7a@O61l00a@O014H7`00I1mP7f@OH1mT7f0O
I1mP7f@OH1mT7f0OI1mP7`0000el4P03001l07`000El00=oo`03001oogoo00aoo`003Goo00<007oo
Ool01W`000D007i0OT1n@7iP00=n@00cOV1n@7i0OT1nH7i0001Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?P
In1Sh0005n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh0000n`3j`?/0n/3k0?[0n`3j`?/0n/3k0?[
0n`3j`0000h3o`0/0001W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl1W`0061lD7aPO51lH7a@O61lD7aPO
51lH7a@O61lD7`00H1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l000=l4P05O1=l4W`BO19l4`03O180
17`CO19l4P001g`00goo00<007ooOol037oo000=Ool200000gooO01l0005O0000`00OT1n@00;OT00
4@00Hn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn0001OP6n000aOP00DKh1OP5n0Gh1_P00<Gh0030003
k0?/00`3k0030003o`?o00`3o`030001W`6O00/1W`05000D7a@O51lH7`0351l01APO51lD7a@O61l0
0Q@O00l006@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O00003W`B00<007`0O0001G`00goo00<007oo
Ool037oo000=Ool00`00Oomoo`07O0000`00OV1n@002OT001GiPOT1n@7i0OV000gi002enH000In1S
h6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn0001_P5n0Kh1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0000?[0n`3
j`?/0n/3k0?[0n`3j`?/0n/3k0?[00003P?o02h0006O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`6O0Gl0
01@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O001P7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7`00O19l
4`=l4P05O1=l4W`BO19l4`03O1800g`C001l0006O003Ool00`00Oomoo`0<Ool000eoo`03001oogoo
00Ql0003001n@7i000]n@00?001Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh000008Gh0056n0Gh1OP
5n0Kh0035n0011_P5n0Gh0003P?/00<000?o0ol02`?o00<0006O0Il0306O00<001PO51l00Q@O00DH
7a@O51lD7aPO00<D7`0@61l006@OH1mT7f0OI1mP7f@OH1mT7f0OI1mP7f@O000=O1800`00O01l0006
O003Ool00`00Oomoo`0<Ool000eoo`03001oogoo00Ql0003001n@7iP00=n@005OV1n@7i0OT1nH003
OT00;000Hn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn0001OP6n0Gh1_P5n0Kh1OP6n0Gh1_P5n0Kh1OP
6n0000?/0n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/000=0ol0;0000Il1O`6O0Gl1W`5o0Il1O`6O0Gl1
W`5o0Il1O`0051lH7a@O61lD7aPO51lH7a@O61lD7aPO51l0060OI1mP7f@OH1mT7f0OI1mP7f@OH1mT
7f0O0003O1801G`CO19l4W`BO1<00g`B00=l4g`B000027`00goo00<007ooOol037oo000=Ool00`00
Oomoo`09O0000`00OT1n@00;OT003`00Hn1Wh6?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn0000035n001A_P
5n0Gh1OP6n000aOP00<Kh1OP00003@?/00<000?o0ol02`?o00<0006O0Il0306O00@001@O51lH7`<D
7`0561lD7a@O51lH7`0251l03`00I1mP7f@OH1mT7f0OI1mP7f@OH1mT7f0OI1l0000=O1800`00O01l
0006O003Ool00`00Oomoo`0<Ool00003Ool00000008000=oo`8000=oo`03001oogoo00Yl0003001n
@7iP00=n@005OV1n@7i0OT1nH002OT00:`00In1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn0001_P5n0K
h1OP6n0Gh1_P5n0Kh1OP6n0Gh1_P0003k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/3k00000h3o`0Z0001
O`6O0Gl1W`5o0Il1O`6O0Gl1W`5o0Il1O`0051lH7a@O61lD7aPO51lH7a@O61lD7aPO51l0060OI1mP
7f@OH1mT7f0OI1mP7f@OH1mT7`000W`B00El4g`BO19l4W`C00=l4P05O1=l4W`BO1800008O003Ool0
0`00Oomoo`0<Ool00003Ool007oo00Aoo`04001oogoo0002Ool00`00Oomoo`0:O0000`00OT1n@00;
OT004000In1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?PIn0001_P0aOP00DKh1OP5n0Gh1_P00<Gh0036n00
00?/00`3k0030003o`?o00/3o`030001W`6O00/1W`03000H7a@O008D7`0561lD7a@O51lH7`0351l0
3aPO001T7f0OI1mP7f@OH1mT7f0OI1mP7f@OH1l0000=O1800`00O01l0007O003Ool00`00Oomoo`0<
Ool0009oo`03001oogoo009oo`04001oogoo0002Ool200000gooO01l0009O0001000OT1n@7iP0gi0
035nH7i0OT1n@7iPOT0006?PIn1Sh6OPHn1Wh6?PIn1Sh6OPHn1Wh6?P000Gh1_P5n0Kh1OP6n0Gh1_P
5n0Kh1OP6n0Gh0000n/3k0?[0n`3j`?/0n/3k0?[0n`3j`?/0n/0000=0ol0:`000Gl1W`5o0Il1O`6O
0Gl1W`5o0Il1O`6O0Gl001@O61lD7aPO51lH7a@O61lD7aPO51lH7a@O001P7f@OH1mT7f0OI1mP7f@O
H1mT7f0OI1l007`C00=l4P05O1=l4W`BO19l4`03O1800g`C001l0008O002Ool2000>Ool000=oo`05
001oogooOol00002Ool01000Oomoo`00UWoo00<007ooOol037oo00001Goo001oogoo00000Woo00@0
07ooOol0009oo`04001oogoo0004Ool00`00Oomoo`02Ool00`00Oomoo`02Ool00`00Oomoo`02Ool0
1@00Oomoogoo000017oo00<007ooOol00Woo00<007ooOol00Woo00<007ooOol00Woo00<007ooOol0
0Woo00<007ooOol00Woo00D007ooOomoo`0000Aoo`03001oogoo009oo`03001oogoo009oo`03001o
ogoo009oo`03001oogoo009oo`05001oogooOol00004Ool00`00Oomoo`02Ool00`00Oomoo`02Ool0
0`00Oomoo`02Ool00`00Oomoo`02Ool01@00Oomoogoo000017oo00<007ooOol00Woo00<007ooOol0
0Woo00<007ooOol00Woo00<007ooOol00Woo00D007ooOomoo`00009oo`03001oogoo00aoo`000Woo
0P0017oo0P000gooV0003Woo002cOol00;=oo`00/goo002cOol00;=oo`00/goo002cOol00;=oo`00
37oo0P00YGoo000;Ool01000Oomoo`00Y7oo000;Ool00`00Oomoo`2UOol000Yoo`D00:Aoo`002goo
00<007ooOol0YGoo002cOol00;=oo`00/goo002cOol00;=oo`00/goo0000\
\>"],
  ImageRangeCache->{{{0, 178.625}, {182.625, 0}} -> {-3.69406, -2.03794, \
0.0413349, 0.137783}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["8.6.2 Perturbed Korteweg-de Vries Equation", "Subsection",
  CellTags->"Disturbed Korteweg de Vries Equation"],

Cell["\<\
One of the frequently discussed equations in soliton theory is the \
Korteweg-de Vries equation (KdV)\
\>", "Text",
  CellTags->{"soliton theory", "Korteweg-de Vries equation"}],

Cell[BoxData[
    \(TraditionalForm\`u\_t + \ u\ u\_x + \ u\_\(x, x, x\) = 0. \)], 
  "NumberedEquation"],

Cell[TextData[{
  "The KdV equation is one of the rare equations which is solvable and \
possesses an infinite number of integrals of motion. The equation first \
derived by ",
  ButtonBox["Korteweg and de Vries ",
    ButtonData:>{"Literat.nb", "KortewegDeVries-1895"},
    ButtonStyle->"Hyperlink"],
  "in 1895 describes shallow water waves in narrow channels. Korteweg and de \
Vries showed that periodic solutions, which they called cnoidal waves, could \
be found in closed form and without further approximation. Our interest here \
is the approximation aspect of the fluid dynamics. It is well known that the \
KdV equation is an approximated equation in a certain limit incorporating the \
effects of dispersion and surface tension which stabilizes a wave. If now we \
incorporate terms in the equation which are actually present in nature but \
are dropped in equation (8.24), we may gain information on the influence of \
such terms. For example, let us extend the KdV equation by a dispersive term \
",
  Cell[BoxData[
      \(TraditionalForm\`u\_xx\)]],
  " that ",
  ButtonBox["Burgers [1948]",
    ButtonData:>{"Literat.nb", "Burgers-1948"},
    ButtonStyle->"Hyperlink"],
  " used in his turbulent theory. In the following, we examine the equation"
}], "Text",
  CellTags->{
  "infinite number of integrals of motion", "shallow water waves", 
    "cnoidal waves", "fluid dynamics", "turbulent theory"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(KdVepsilon\  = \ 
      \[PartialD]\_t u[x, t]\  + \ u[x, t] \[PartialD]\_x u[x, t]\  + \ 
          \[PartialD]\_{x, 3}u[x, t] + 
          \[Epsilon] \[PartialD]\_{x, 2}u[x, t] == 0; KdVepsilon // LTF\)], 
  "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            \(u\_t + u\ u\_x + \[Epsilon]\ u\_\(x, x\) + u\_\(x, x, x\) == 0
              \)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[
"where \[Epsilon] is a small parameter measuring the influence of \
second-order dispersion. For short we call this equation the KdV-\[Epsilon] \
equation. The approximate symmetries of this equation follow by"], "Text",
  CellTags->{"second-order dispersion", "KdV-\[Epsilon] equation"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(KdVinfinitesimals\  = \ 
      ApproximateSymmetries[KdVepsilon, {u}, {x, t}, {\[Epsilon]}, 
        \[Epsilon], SubstitutionRules \[Rule] {\[PartialD]\_t u[x, t]}]; 
    KdVinfinitesimals // LTF\)], "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(\[Phi]\_1 == k2 + \((k5 - 2\ k7\ u)\)\ \[Epsilon]\)},
          {
            \(\[Xi]\_1 == 
              k3 + k2\ t + \((k6 + k5\ t + k7\ x)\)\ \[Epsilon]\)},
          {\(\[Xi]\_2 == k1 + \((k4 + 3\ k7\ t)\)\ \[Epsilon]\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Subscript[ \[Phi], 1], 
          Plus[ k2, 
            Times[ 
              Plus[ k5, 
                Times[ -2, k7, u]], \[Epsilon]]]], 
        Equal[ 
          Subscript[ \[Xi], 1], 
          Plus[ k3, 
            Times[ k2, t], 
            Times[ 
              Plus[ k6, 
                Times[ k5, t], 
                Times[ k7, x]], \[Epsilon]]]], 
        Equal[ 
          Subscript[ \[Xi], 2], 
          Plus[ k1, 
            Times[ 
              Plus[ k4, 
                Times[ 3, k7, t]], \[Epsilon]]]]}]]], "Output"]
}, Open  ]],

Cell["\<\
representing a seven-dimensional approximate symmetry group. The \
generating vector fields for this model read\
\>", "Text",
  CellTags->"approximate symmetry group"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Map[
        \((Fold[Plus, 0, 
                Map[Fold[NonCommutativeMultiply, 1, #]&, 
                    Transpose[{
                        Flatten[#], {\*"\"\<\!\(\[PartialD]\_x\)\>\"", 
                          \*"\"\<\!\(\[PartialD]\_t\)\>\"", 
                          \*"\"\<\!\(\[PartialD]\_u\)\>\""}}]] /. 
                  \ _**0**_ \[Rule] 0]\  /. \ 1**a_**b_ \[Rule] a**b)\)&, 
        \({{\(xi[1]\)[x, t, u], \(xi[2]\)[x, t, u]}, {
                \(phi[1]\)[x, t, u]}} /. KdVinfinitesimals\) /. 
          \((Map[\((Thread[{k1, k2, k3, k4, k5, k6, k7} \[Rule] #])\)&, 
              Permutations[{1, 0, 0, 0, 0, 0, 0}]])\)] // TableForm\)], 
  "Input"],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {\(1**"\!\(\[PartialD]\_t\)"\)},
          {\(1**"\!\(\[PartialD]\_u\)" + t**"\!\(\[PartialD]\_x\)"\)},
          {\(1**"\!\(\[PartialD]\_x\)"\)},
          {\(\[Epsilon]**"\!\(\[PartialD]\_t\)"\)},
          {
            \(\[Epsilon]**"\!\(\[PartialD]\_u\)" + 
              \((t\ \[Epsilon])\)**"\!\(\[PartialD]\_x\)"\)},
          {\(\[Epsilon]**"\!\(\[PartialD]\_x\)"\)},
          {
            \(\((3\ t\ \[Epsilon])\)**"\!\(\[PartialD]\_t\)" + 
              \((\(-2\)\ u\ \[Epsilon])\)**"\!\(\[PartialD]\_u\)" + 
              \((x\ \[Epsilon])\)**"\!\(\[PartialD]\_x\)"\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        NonCommutativeMultiply[ 1, "\!\(\[PartialD]\_t\)"], 
        Plus[ 
          NonCommutativeMultiply[ 1, "\!\(\[PartialD]\_u\)"], 
          NonCommutativeMultiply[ t, "\!\(\[PartialD]\_x\)"]], 
        NonCommutativeMultiply[ 1, "\!\(\[PartialD]\_x\)"], 
        NonCommutativeMultiply[ \[Epsilon], "\!\(\[PartialD]\_t\)"], 
        Plus[ 
          NonCommutativeMultiply[ \[Epsilon], "\!\(\[PartialD]\_u\)"], 
          NonCommutativeMultiply[ 
            Times[ t, \[Epsilon]], "\!\(\[PartialD]\_x\)"]], 
        NonCommutativeMultiply[ \[Epsilon], "\!\(\[PartialD]\_x\)"], 
        Plus[ 
          NonCommutativeMultiply[ 
            Times[ 3, t, \[Epsilon]], "\!\(\[PartialD]\_t\)"], 
          NonCommutativeMultiply[ 
            Times[ -2, u, \[Epsilon]], "\!\(\[PartialD]\_u\)"], 
          NonCommutativeMultiply[ 
            Times[ x, \[Epsilon]], "\!\(\[PartialD]\_x\)"]]}]]], "Output"]
}, Open  ]],

Cell["The coefficients of these vector fields are given by", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(casesKdV = 
      \({{\(xi[1]\)[x, t, u], \(xi[2]\)[x, t, u]}, {\(phi[1]\)[x, t, u]}} /. 
          KdVinfinitesimals\) /. 
        \((Map[\((Thread[{k1, k2, k3, k4, k5, k6, k7} \[Rule] #])\)&, 
            Permutations[{1, 0, 0, 0, 0, 0, 0}]])\)\)], "Input"],

Cell[BoxData[
    \({{{0, 1}, {0}}, {{t, 0}, {1}}, {{1, 0}, {0}}, {{0, \[Epsilon]}, {0}}, {{
          t\ \[Epsilon], 0}, {\[Epsilon]}}, {{\[Epsilon], 0}, {0}}, {{
          x\ \[Epsilon], 3\ t\ \[Epsilon]}, {\(-2\)\ u\ \[Epsilon]}}}\)], 
  "Output"]
}, Open  ]],

Cell[TextData[
"The group of seven vector fields contains symmetries like translations and \
scalings depending on the perturbation parameter \[Epsilon]. Let us choose a \
linear combination of three subgroups to reduce the original KdV-\[Epsilon] \
equation. The following line creates the similarity reduction in connection \
with the first, third, and fifth sub-groups."], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(redKdV\  = \ 
      LieReduction[KdVepsilon\ , {u}, {x, t}, 
          casesKdV\[LeftDoubleBracket]1, 1\[RightDoubleBracket] + 
            c\ casesKdV\[LeftDoubleBracket]3, 1\[RightDoubleBracket] + 
            casesKdV\[LeftDoubleBracket]5, 1\[RightDoubleBracket], 
          casesKdV\[LeftDoubleBracket]1, 2\[RightDoubleBracket] + 
            c\ casesKdV\[LeftDoubleBracket]3, 2\[RightDoubleBracket] + 
            casesKdV\[LeftDoubleBracket]5, 2\[RightDoubleBracket]] // 
        Simplify; 
    \(redKdV /. zeta1 \[Rule] \[Zeta]\_1\  // Flatten\) // LTF\)], "Input"],

Cell[BoxData[
    TagBox[GridBox[{
          {
            RowBox[{
              RowBox[{\(\(-c\)\ t\), "+", "x", "-", 
                StyleBox[\(\(t\^2\ \[Epsilon]\)\/2\),
                  ScriptLevel->0], "-", \(\[Zeta]\_1\)}], "==", "0"}]},
          {\(\(-c\) + u - t\ \[Epsilon] - F\_1 == 0\)},
          {
            \(\[Epsilon] + F\_1\ \((F\_1)\)\_\(\[Zeta]\_1\) + 
                \[Epsilon]\ \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1\) + 
                \((F\_1)\)\_\(\[Zeta]\_1, \[Zeta]\_1, \[Zeta]\_1\) == 0\)}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      (TableForm[ #]&)]], "Output"]
}, Open  ]],

Cell[TextData[{
  "The result is a similarity representation of the solution depending \
linearly on \[Epsilon]. The similarity variable ",
  Cell[BoxData[
      \(TraditionalForm\`\[Zeta]\_1 = x - ct - \[Epsilon]t\^2/2\)]],
  " also depends linearly on \[Epsilon]. The determining equation of the \
similarity function ",
  Cell[BoxData[
      \(TraditionalForm\`F\_1\)]],
  " also shows a linear dependence on \[Epsilon]. All these linear \
dependencies on \[Epsilon] will result into a non-linear dependence on \
\[Epsilon] of the solution. Thus, let us examine the determining equation \
without the \[Epsilon] terms. The reduced KdV-\[Epsilon] equation is  \
integrated by two quadratures to the form"
}], "Text",
  CellTags->{
  "similarity representation", "similarity variable", "determining equation", 
    "reduced KdV-\[Epsilon] equation"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(b1\  = \ 
      Integrate[
        \[PartialD]\_zeta1 F1[zeta1] 
          \((\ \ Integrate[
              redKdV\[LeftDoubleBracket]3, 1, 1\[RightDoubleBracket] /. 
                  \[Epsilon] \[Rule] 0 // Expand, zeta1])\), zeta1]\)], 
  "Input"],

Cell[BoxData[
    RowBox[{\(F1[zeta1]\^3\/6\), "+", 
      RowBox[{\(1\/2\), " ", 
        SuperscriptBox[
          RowBox[{
            SuperscriptBox["F1", "\[Prime]",
              MultilineFunction->None], "[", "zeta1", "]"}], "2"]}]}]], 
  "Output"]
}, Open  ]],

Cell[TextData[{
  "This result can be treated in two different ways. First, let us assume \
that the expression equals a constant ",
  Cell[BoxData[
      \(TraditionalForm\`K\)]],
  ". For this case, we find by ",
  StyleBox["DSolve[]", "MmaText"]
}], "Text",
  CellTags->"DSolve[]"],

Cell[CellGroupData[{

Cell[BoxData[
    \(case1\  = \ DSolve[b1 == K, F1, zeta1]\)], "Input"],

Cell[BoxData[
    \(Solve::"tdep" \( : \ \) 
      "The equations appear to involve transcendental functions of the \
variables in an essentially non-algebraic way."\)], "Output"],

Cell[BoxData[
    \(Solve::"tdep" \( : \ \) 
      "The equations appear to involve transcendental functions of the \
variables in an essentially non-algebraic way."\)], "Output",
  GeneratedCell->False,
  CellAutoOverwrite->False],

Cell[BoxData[
    \(Solve::"tdep" \( : \ \) 
      "The equations appear to involve transcendental functions of the \
variables in an essentially non-algebraic way."\)], "Output",
  GeneratedCell->False,
  CellAutoOverwrite->False],

Cell[BoxData[
    \(General::"stop" \( : \ \) "\<Further output of \!\(Solve::\>" tdep 
      "\) will be suppressed during this calculation."\)], "Output",
  GeneratedCell->False,
  CellAutoOverwrite->False],

Cell[BoxData[
    \({Solve[
        C[1] - \(1\/\@\(\(-F1\^3\) + 6\ K\)\) 
              \((2\ I\ 2\^\(1/3\)\ 3\^\(7/12\)\ 
                  \@\(\((\(-1\))\)\^\(5/6\)\ 
                      \((\(-1\) + F1\/\(6\^\(1/3\)\ K\^\(1/3\)\))\)\)\ 
                  \@\(1 + F1\^2\/\(6\^\(2/3\)\ K\^\(2/3\)\) + 
                      F1\/\(6\^\(1/3\)\ K\^\(1/3\)\)\)\ K\^\(1/3\)\ 
                  EllipticF[
                    ArcSin[\@\(\(-\((\(-1\))\)\^\(5/6\)\) - 
                            \(I\ F1\)\/\(6\^\(1/3\)\ K\^\(1/3\)\)\)\/3
                          \^\(1/4\)], \((\(-1\))\)\^\(1/3\)])\) == #1, F1], 
      Solve[C[1] + 
            \(1\/\@\(\(-F1\^3\) + 6\ K\)\) 
              \((2\ I\ 2\^\(1/3\)\ 3\^\(7/12\)\ 
                  \@\(\((\(-1\))\)\^\(5/6\)\ 
                      \((\(-1\) + F1\/\(6\^\(1/3\)\ K\^\(1/3\)\))\)\)\ 
                  \@\(1 + F1\^2\/\(6\^\(2/3\)\ K\^\(2/3\)\) + 
                      F1\/\(6\^\(1/3\)\ K\^\(1/3\)\)\)\ K\^\(1/3\)\ 
                  EllipticF[
                    ArcSin[\@\(\(-\((\(-1\))\)\^\(5/6\)\) - 
                            \(I\ F1\)\/\(6\^\(1/3\)\ K\^\(1/3\)\)\)\/3
                          \^\(1/4\)], \((\(-1\))\)\^\(1/3\)])\) == #1, F1]}
      \)], "Output"]
}, Open  ]],

Cell[TextData[{
  "that the solution is given by elliptic functions. Since ",
  StyleBox["Solve[]", "MmaText"],
  " is unable to invert the expression, we only get the solution in an \
implicit form. The second case we can examine is when ",
  Cell[BoxData[
      \(TraditionalForm\`K = 0\)]],
  ". In this case, the solution is given by"
}], "Text",
  CellTags->{"elliptic functions", "Solve[]"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(case2\  = \ DSolve[b1 == 0, F1, zeta1]\)], "Input"],

Cell[BoxData[
    \({{F1 \[Rule] \((\(-\(12\/\((#1 - C[1])\)\^2\)\)&)\)}}\)], "Output"]
}, Open  ]],

Cell["\<\
The similarity representation of the solution follows from this \
relation by\
\>", "Text",
  CellTags->"similarity representation"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol\  = \ 
      Solve[redKdV\[LeftDoubleBracket]2\[RightDoubleBracket]\  /. \ 
          case2\[LeftDoubleBracket]1\[RightDoubleBracket], u]\)], "Input"],

Cell[BoxData[
    \({{u \[Rule] 
          c + t\ \[Epsilon] - 
            12\/\((\(-c\)\ t + x - \(t\^2\ \[Epsilon]\)\/2 - C[1])\)\^2}}\)], 
  "Output"]
}, Open  ]],

Cell[TextData[{
  "Since the solution should depend linearly on \[Epsilon], we have to expand \
the result up to first order in \[Epsilon] around ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon] = 0\)]],
  ":"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(solution\  = \ Series[u\  /. sol, {\[Epsilon], 0, 1}]\)], "Input"],

Cell[BoxData[
    RowBox[{"{", 
      InterpretationBox[
        RowBox[{
        \((c - 12\/\((\(-c\)\ t + x - C[1])\)\^2)\), "+", 
          \(\((t - \(12\ t\^2\)\/\((\(-c\)\ t + x - C[1])\)\^3)\)\ 
            \[Epsilon]\), "+", 
          InterpretationBox[\(O[\[Epsilon]]\^2\),
            SeriesData[ \[Epsilon], 0, {}, 0, 2, 1]]}],
        SeriesData[ \[Epsilon], 0, {
          Plus[ c, 
            Times[ -12, 
              Power[ 
                Plus[ 
                  Times[ -1, c, t], x, 
                  Times[ -1, 
                    C[ 1]]], -2]]], 
          Plus[ t, 
            Times[ -12, 
              Power[ t, 2], 
              Power[ 
                Plus[ 
                  Times[ -1, c, t], x, 
                  Times[ -1, 
                    C[ 1]]], -3]]]}, 0, 2, 1]], "}"}]], "Output"]
}, Open  ]],

Cell[TextData[{
  "This expression gives us an approximate solution of the KdV-\[Epsilon] \
equation. We can check relation (",
  ButtonBox["8.11",
    ButtonData:>"eq-11",
    ButtonStyle->"Hyperlink"],
  ") by inserting the above solution into the original equation. In the first \
step, we convert the solution into a pure function representation:"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(srule\  = \ 
      u \[Rule] 
        Apply[Function, {{x, t}, 
            solution\[LeftDoubleBracket]1\[RightDoubleBracket] // Normal}]
          \)], "Input"],

Cell[BoxData[
    \(u \[Rule] 
      Function[{x, t}, 
        c + \[Epsilon]\ 
            \((t - \(12\ t\^2\)\/\((\(-c\)\ t + x - C[1])\)\^3)\) - 
          12\/\((\(-c\)\ t + x - C[1])\)\^2]\)], "Output"]
}, Open  ]],

Cell["\<\
Inserting the solution into the original equation demonstrates that \
the equation is satisfied in first-order approximation:\
\>", "Text",
  CellTags->"first-order approximation"],

Cell[CellGroupData[{

Cell[BoxData[
    \(KdVepsilon /. srule // Simplify\)], "Input"],

Cell[BoxData[
    \(\[Epsilon]\ 
        \((1 + \(432\ t\^4\ \[Epsilon]\)\/\((c\ t - x + C[1])\)\^7 + 
            \(144\ t\^2\ \[Epsilon]\)\/\((c\ t - x + C[1])\)\^5 - 
            72\/\((c\ t - x + C[1])\)\^4 + 
            \(36\ t\^3\ \[Epsilon]\)\/\((c\ t - x + C[1])\)\^4)\) == 0\)], 
  "Output"]
}, Open  ]],

Cell[TextData[{
  "The two examples presented are a small representation of the huge amount \
of equations depending on a small perturbation parameter. We note that the \
derivation of an approximate solution is not unique. As the reader has \
noticed in the solution step, there is a great flexibility in choosing the \
stage of approximation. However, at the end of the calculation, we have to \
satisfy relation (",
  ButtonBox["8.11",
    ButtonData:>"eq-11",
    ButtonStyle->"Hyperlink"],
  ") defining the order of approximation."
}], "Text"],

Cell[TextData[ButtonBox["Next notebook",
  ButtonData:>{"symmet75.nb", None},
  ButtonStyle->"Hyperlink"]], "Text",
  CellFrame->{{0, 0}, {0, 0.5}}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 1024}, {0, 695}},
ScreenStyleEnvironment->"Working",
WindowToolbars->"EditBar",
Evaluator->"Local",
WindowSize->{665, 668},
WindowMargins->{{15, Automatic}, {Automatic, 0}},
PrintingCopies->1,
PrintingStartingPageNumber->404,
PrintingPageRange->{402, 402},
PageHeaders->{{Cell[ 
        TextData[ {
          StyleBox[ 
            CounterBox[ "Page"], "PageNumber"], "     ", 
          StyleBox[ 
          "Approximate Symmetries of Partial Differential Equations", 
            FontSlant -> "Italic"]}], "Header"], Inherited, None}, {
    None, Inherited, Cell[ 
        TextData[ {
          StyleBox[ 
            CounterBox[ 
            "Section", CounterFunction :> (
              Part[ {"Introduction", "Approximations", 
                "One-Parameter Approximation Group", 
                "Approximate Group Generator", 
                "The Determining Equations and an Algorithm of Calculation", 
                "Examples"}, #]&)], FontSlant -> "Italic"], "     ", 
          StyleBox[ 
            CounterBox[ "Page"], "PageNumber"]}], "Header"]}},
PageHeaderLines->{False, False},
PrintingOptions->{"FirstPageHeader"->False,
"FirstPageFooter"->False,
"FacingPages"->True,
"FirstPageFace"->Right},
PrivateNotebookOptions->{"ColorPalette"->{RGBColor, -1}},
ShowCellLabel->True,
RenderingOptions->{"ObjectDithering"->True,
"RasterDithering"->False},
Magnification->1,
StyleDefinitions -> "BookStyles.nb"
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "Approximate Symmetries of Partial Differential Equations"->{
    Cell[1739, 51, 421, 7, 94, "ChapterLine",
      Evaluatable->False,
      CounterAssignments->{{"Title", 7}, {"SubTitle", 0}, {
          "SubSubTitle", 0}, {"Section", 0}, {"SubSection", 0}, {
          "SubSubSection", 0}, {"NumberedEquation", 0}, {
          "NumberedFigure", 0}, {"NumberedTable", 0}},
      CellTags->"Approximate Symmetries of Partial Differential Equations"]},
  "Introduction"->{
    Cell[2594, 73, 332, 9, 43, "Section",
      CounterAssignments->{{"Title", 8}, {"SubTitle", 0}, {
          "SubSubTitle", 0}, {"Section", 0}, {"SubSection", 0}, {
          "SubSubSection", 0}, {"NumberedEquation", 0}, {
          "NumberedFigure", 0}, {"NumberedTable", 0}},
      CellTags->"Introduction"]},
  "approximate symmetries"->{
    Cell[2929, 84, 1407, 25, 180, "Text",
      Evaluatable->False,
      CellTags->{
      "approximate symmetries", "extension of Lie's theory", 
        "small perturbation", "approximate group", "Lie point symmetries"}],
    Cell[4339, 111, 778, 14, 95, "Text",
      Evaluatable->False,
      CellTags->{
      "Lie's theory", "approximate calculus", "critical parameter", 
        "approximate transformation groups", "approximate symmetries"}],
    Cell[19817, 623, 602, 13, 61, "Text",
      CellTags->{
      "classical Lie theory", "approximate group analysis", 
        "algorithm of approximate group analysis", 
        "approximate symmetries"}]},
  "extension of Lie's theory"->{
    Cell[2929, 84, 1407, 25, 180, "Text",
      Evaluatable->False,
      CellTags->{
      "approximate symmetries", "extension of Lie's theory", 
        "small perturbation", "approximate group", "Lie point symmetries"}]},
  "small perturbation"->{
    Cell[2929, 84, 1407, 25, 180, "Text",
      Evaluatable->False,
      CellTags->{
      "approximate symmetries", "extension of Lie's theory", 
        "small perturbation", "approximate group", "Lie point symmetries"}]},
  "approximate group"->{
    Cell[2929, 84, 1407, 25, 180, "Text",
      Evaluatable->False,
      CellTags->{
      "approximate symmetries", "extension of Lie's theory", 
        "small perturbation", "approximate group", "Lie point symmetries"}],
    Cell[21725, 679, 418, 11, 44, "Text",
      CellTags->{"approximate vector field", "approximate group"}]},
  "Lie point symmetries"->{
    Cell[2929, 84, 1407, 25, 180, "Text",
      Evaluatable->False,
      CellTags->{
      "approximate symmetries", "extension of Lie's theory", 
        "small perturbation", "approximate group", "Lie point symmetries"}],
    Cell[18752, 587, 771, 21, 78, "Text",
      Evaluatable->False,
      CellTags->{"Lie point symmetries", "Taylor expansion"}]},
  "Lie's theory"->{
    Cell[4339, 111, 778, 14, 95, "Text",
      Evaluatable->False,
      CellTags->{
      "Lie's theory", "approximate calculus", "critical parameter", 
        "approximate transformation groups", "approximate symmetries"}]},
  "approximate calculus"->{
    Cell[4339, 111, 778, 14, 95, "Text",
      Evaluatable->False,
      CellTags->{
      "Lie's theory", "approximate calculus", "critical parameter", 
        "approximate transformation groups", "approximate symmetries"}]},
  "critical parameter"->{
    Cell[4339, 111, 778, 14, 95, "Text",
      Evaluatable->False,
      CellTags->{
      "Lie's theory", "approximate calculus", "critical parameter", 
        "approximate transformation groups", "approximate symmetries"}]},
  "approximate transformation groups"->{
    Cell[4339, 111, 778, 14, 95, "Text",
      Evaluatable->False,
      CellTags->{
      "Lie's theory", "approximate calculus", "critical parameter", 
        "approximate transformation groups", "approximate symmetries"}]},
  "approximation"->{
    Cell[5247, 137, 1162, 29, 114, "Text",
      Evaluatable->False,
      CellTags->{"approximation", "infinitesimal small functions"}]},
  "infinitesimal small functions"->{
    Cell[5247, 137, 1162, 29, 114, "Text",
      Evaluatable->False,
      CellTags->{"approximation", "infinitesimal small functions"}]},
  "Definition: Approximation"->{
    Cell[6724, 180, 88, 1, 33, "Definition",
      CellTags->"Definition: Approximation"]},
  "One-Parameter Approximation Group"->{
    Cell[7803, 228, 206, 8, 43, "Section",
      Evaluatable->False,
      CellTags->"One-Parameter Approximation Group"]},
  "one-parameter approximation group"->{
    Cell[8012, 238, 1046, 34, 62, "Text",
      Evaluatable->False,
      CellTags->{
      "one-parameter approximation group", "approximate transformation"}]},
  "approximate transformation"->{
    Cell[8012, 238, 1046, 34, 62, "Text",
      Evaluatable->False,
      CellTags->{
      "one-parameter approximation group", "approximate transformation"}]},
  "eq-3"->{
    Cell[9061, 274, 323, 8, 40, "NumberedEquation",
      CellTags->"eq-3"]},
  "group parameter"->{
    Cell[9387, 284, 1398, 38, 61, "Text",
      Evaluatable->False,
      CellTags->{"group parameter", "perturbation parameter"}]},
  "perturbation parameter"->{
    Cell[9387, 284, 1398, 38, 61, "Text",
      Evaluatable->False,
      CellTags->{"group parameter", "perturbation parameter"}]},
  "approximate identity element"->{
    Cell[10912, 328, 202, 6, 27, "Text",
      Evaluatable->False,
      CellTags->"approximate identity element"]},
  "local one-parameter approximate transformation group"->{
    Cell[11839, 362, 256, 5, 27, "Text",
      Evaluatable->False,
      CellTags->"local one-parameter approximate transformation group"]},
  "first-order approximation"->{
    Cell[12487, 385, 851, 30, 61, "Text",
      Evaluatable->False,
      CellTags->"first-order approximation"],
    Cell[283311, 7309, 190, 4, 44, "Text",
      CellTags->"first-order approximation"]},
  "approximate association relation"->{
    Cell[14885, 473, 887, 20, 95, "Text",
      Evaluatable->False,
      CellTags->"approximate association relation"]},
  "group generator"->{
    Cell[15775, 495, 558, 11, 61, "Text",
      Evaluatable->False,
      CellTags->{
      "group generator", "Lie's equation", "infinitesimal transformations"}],
    Cell[16567, 521, 414, 10, 44, "Text",
      Evaluatable->False,
      CellTags->{
      "Lie point group", "group generator", "vector field", 
        "vector field of an approximate group"}]},
  "Lie's equation"->{
    Cell[15775, 495, 558, 11, 61, "Text",
      Evaluatable->False,
      CellTags->{
      "group generator", "Lie's equation", 
        "infinitesimal transformations"}]},
  "infinitesimal transformations"->{
    Cell[15775, 495, 558, 11, 61, "Text",
      Evaluatable->False,
      CellTags->{
      "group generator", "Lie's equation", 
        "infinitesimal transformations"}]},
  "Approximate group generator"->{
    Cell[16370, 511, 194, 8, 43, "Section",
      Evaluatable->False,
      CellTags->"Approximate group generator"]},
  "Lie point group"->{
    Cell[16567, 521, 414, 10, 44, "Text",
      Evaluatable->False,
      CellTags->{
      "Lie point group", "group generator", "vector field", 
        "vector field of an approximate group"}]},
  "vector field"->{
    Cell[16567, 521, 414, 10, 44, "Text",
      Evaluatable->False,
      CellTags->{
      "Lie point group", "group generator", "vector field", 
        "vector field of an approximate group"}],
    Cell[61391, 1756, 241, 5, 44, "Text",
      CellTags->{"finite symmetry group", "vector field"}]},
  "vector field of an approximate group"->{
    Cell[16567, 521, 414, 10, 44, "Text",
      Evaluatable->False,
      CellTags->{
      "Lie point group", "group generator", "vector field", 
        "vector field of an approximate group"}]},
  "eq-7"->{
    Cell[16984, 533, 208, 6, 48, "NumberedEquation",
      CellTags->"eq-7"]},
  "eq-8"->{
    Cell[17271, 545, 271, 6, 23, "NumberedEquation",
      CellTags->"eq-8"]},
  "eq-9"->{
    Cell[17831, 562, 471, 11, 35, "NumberedEquation",
      CellTags->"eq-9"]},
  "approximate vector field"->{
    Cell[18305, 575, 149, 3, 27, "Text",
      Evaluatable->False,
      CellTags->"approximate vector field"],
    Cell[21725, 679, 418, 11, 44, "Text",
      CellTags->{"approximate vector field", "approximate group"}]},
  "Taylor expansion"->{
    Cell[18752, 587, 771, 21, 78, "Text",
      Evaluatable->False,
      CellTags->{"Lie point symmetries", "Taylor expansion"}]},
  "The Determining Equations and an Algorithm of Calculation"->{
    Cell[19560, 613, 254, 8, 43, "Section",
      Evaluatable->False,
      CellTags->
      "The Determining Equations and an Algorithm of Calculation"]},
  "classical Lie theory"->{
    Cell[19817, 623, 602, 13, 61, "Text",
      CellTags->{
      "classical Lie theory", "approximate group analysis", 
        "algorithm of approximate group analysis", 
        "approximate symmetries"}]},
  "approximate group analysis"->{
    Cell[19817, 623, 602, 13, 61, "Text",
      CellTags->{
      "classical Lie theory", "approximate group analysis", 
        "algorithm of approximate group analysis", 
        "approximate symmetries"}]},
  "algorithm of approximate group analysis"->{
    Cell[19817, 623, 602, 13, 61, "Text",
      CellTags->{
      "classical Lie theory", "approximate group analysis", 
        "algorithm of approximate group analysis", 
        "approximate symmetries"}]},
  "eq-11"->{
    Cell[20797, 650, 454, 10, 24, "NumberedEquation",
      CellTags->"eq-11"]},
  "approximate invariant"->{
    Cell[21254, 662, 157, 3, 27, "Text",
      CellTags->"approximate invariant"]},
  "determining equation for approximate symmetries"->{
    Cell[22403, 699, 661, 12, 95, "Text",
      CellTags->"determining equation for approximate symmetries"]},
  "algorithm to calculate first-order approximate symmetries"->{
    Cell[24583, 761, 366, 8, 44, "Text",
      CellTags->
      "algorithm to calculate first-order approximate symmetries"]},
  "Theorem: First-order approximations"->{
    Cell[24952, 771, 105, 1, 33, "Theorem",
      CellTags->"Theorem: First-order approximations"]},
  "auxiliary factor"->{
    Cell[25881, 800, 446, 13, 44, "Text",
      CellTags->"auxiliary factor"]},
  "algorithm of first-order approximation"->{
    Cell[26330, 815, 232, 5, 44, "Text",
      CellTags->"algorithm of first-order approximation"]},
  "deficiency"->{
    Cell[27176, 841, 545, 16, 35, "ListItem",
      CellTags->"deficiency"]},
  "prolongation formula"->{
    Cell[28821, 893, 693, 16, 80, "Text",
      CellTags->{
      "prolongation formula", "classical Lie algorithm", 
        "ApproximateSymmetries[]"}]},
  "classical Lie algorithm"->{
    Cell[28821, 893, 693, 16, 80, "Text",
      CellTags->{
      "prolongation formula", "classical Lie algorithm", 
        "ApproximateSymmetries[]"}]},
  "ApproximateSymmetries[]"->{
    Cell[28821, 893, 693, 16, 80, "Text",
      CellTags->{
      "prolongation formula", "classical Lie algorithm", 
        "ApproximateSymmetries[]"}],
    Cell[35377, 1117, 689, 13, 99, "Text",
      CellTags->{
      "isentropic exponent", "hydraulic-friction coefficient", 
        "ApproximateSymmetries[]"}]},
  "Examples"->{
    Cell[29551, 914, 108, 6, 43, "Section",
      CellTags->"Examples"]},
  "computer algebra"->{
    Cell[29662, 922, 404, 9, 61, "Text",
      CellTags->"computer algebra"]},
  "Isentropic Liquid"->{
    Cell[30091, 935, 78, 1, 41, "Subsection",
      CellTags->"Isentropic Liquid"]},
  "liquid in a pipe"->{
    Cell[30172, 938, 416, 11, 44, "Text",
      CellTags->"liquid in a pipe"]},
  "hydraulic-friction coefficient"->{
    Cell[31110, 969, 420, 13, 44, "Text",
      CellTags->"hydraulic-friction coefficient"],
    Cell[35377, 1117, 689, 13, 99, "Text",
      CellTags->{
      "isentropic exponent", "hydraulic-friction coefficient", 
        "ApproximateSymmetries[]"}]},
  "pressure"->{
    Cell[34541, 1085, 90, 1, 27, "Text",
      CellTags->"pressure"]},
  "isentropic exponent"->{
    Cell[35377, 1117, 689, 13, 99, "Text",
      CellTags->{
      "isentropic exponent", "hydraulic-friction coefficient", 
        "ApproximateSymmetries[]"}],
    Cell[53577, 1551, 692, 18, 78, "Text",
      CellTags->"isentropic exponent"]},
  "similarity representation"->{
    Cell[41565, 1278, 314, 5, 61, "Text",
      CellTags->"similarity representation"],
    Cell[202895, 5336, 221, 6, 27, "Text",
      CellTags->"similarity representation"],
    Cell[276313, 7096, 851, 17, 95, "Text",
      CellTags->{
      "similarity representation", "similarity variable", 
        "determining equation", "reduced KdV-\[Epsilon] equation"}],
    Cell[280820, 7222, 142, 4, 27, "Text",
      CellTags->"similarity representation"]},
  "approximation order"->{
    Cell[43300, 1322, 694, 16, 61, "Text",
      CellTags->"approximation order"]},
  "InverseFunction[]"->{
    Cell[57577, 1639, 345, 13, 29, "Text",
      CellTags->"InverseFunction[]"]},
  "isentropic fluid"->{
    Cell[57925, 1654, 540, 11, 78, "Text",
      CellTags->"isentropic fluid"]},
  "finite symmetry group"->{
    Cell[61391, 1756, 241, 5, 44, "Text",
      CellTags->{"finite symmetry group", "vector field"}]},
  "similarity variable"->{
    Cell[63651, 1820, 767, 21, 61, "Text",
      CellTags->{
      "similarity variable", "similarity solution", "determining equation"}],
    Cell[276313, 7096, 851, 17, 95, "Text",
      CellTags->{
      "similarity representation", "similarity variable", 
        "determining equation", "reduced KdV-\[Epsilon] equation"}]},
  "similarity solution"->{
    Cell[63651, 1820, 767, 21, 61, "Text",
      CellTags->{
      "similarity variable", "similarity solution", 
        "determining equation"}]},
  "determining equation"->{
    Cell[63651, 1820, 767, 21, 61, "Text",
      CellTags->{
      "similarity variable", "similarity solution", "determining equation"}],
    Cell[276313, 7096, 851, 17, 95, "Text",
      CellTags->{
      "similarity representation", "similarity variable", 
        "determining equation", "reduced KdV-\[Epsilon] equation"}]},
  "Lagrange coordinates"->{
    Cell[64695, 1855, 338, 5, 61, "Text",
      CellTags->"Lagrange coordinates"],
    Cell[204204, 5383, 312, 12, 27, "Text",
      CellTags->"Lagrange coordinates"]},
  "isentropic motion"->{
    Cell[195618, 5124, 352, 10, 44, "Text",
      CellTags->"isentropic motion"]},
  "infinitesimals"->{
    Cell[198440, 5219, 306, 5, 44, "Text",
      CellTags->{
      "infinitesimals", "isentropic model", "generating vector field"}]},
  "isentropic model"->{
    Cell[198440, 5219, 306, 5, 44, "Text",
      CellTags->{
      "infinitesimals", "isentropic model", "generating vector field"}]},
  "generating vector field"->{
    Cell[198440, 5219, 306, 5, 44, "Text",
      CellTags->{
      "infinitesimals", "isentropic model", "generating vector field"}]},
  "NonCommutativeMultiply[]"->{
    Cell[200302, 5263, 331, 8, 46, "Text",
      CellTags->"NonCommutativeMultiply[]"]},
  "contour plot"->{
    Cell[204817, 5409, 443, 12, 61, "Text",
      CellTags->"contour plot"]},
  "Disturbed Korteweg de Vries Equation"->{
    Cell[267349, 6861, 116, 1, 41, "Subsection",
      CellTags->"Disturbed Korteweg de Vries Equation"]},
  "soliton theory"->{
    Cell[267468, 6864, 187, 4, 27, "Text",
      CellTags->{"soliton theory", "Korteweg-de Vries equation"}]},
  "Korteweg-de Vries equation"->{
    Cell[267468, 6864, 187, 4, 27, "Text",
      CellTags->{"soliton theory", "Korteweg-de Vries equation"}]},
  "infinite number of integrals of motion"->{
    Cell[267766, 6874, 1417, 27, 163, "Text",
      CellTags->{
      "infinite number of integrals of motion", "shallow water waves", 
        "cnoidal waves", "fluid dynamics", "turbulent theory"}]},
  "shallow water waves"->{
    Cell[267766, 6874, 1417, 27, 163, "Text",
      CellTags->{
      "infinite number of integrals of motion", "shallow water waves", 
        "cnoidal waves", "fluid dynamics", "turbulent theory"}]},
  "cnoidal waves"->{
    Cell[267766, 6874, 1417, 27, 163, "Text",
      CellTags->{
      "infinite number of integrals of motion", "shallow water waves", 
        "cnoidal waves", "fluid dynamics", "turbulent theory"}]},
  "fluid dynamics"->{
    Cell[267766, 6874, 1417, 27, 163, "Text",
      CellTags->{
      "infinite number of integrals of motion", "shallow water waves", 
        "cnoidal waves", "fluid dynamics", "turbulent theory"}]},
  "turbulent theory"->{
    Cell[267766, 6874, 1417, 27, 163, "Text",
      CellTags->{
      "infinite number of integrals of motion", "shallow water waves", 
        "cnoidal waves", "fluid dynamics", "turbulent theory"}]},
  "second-order dispersion"->{
    Cell[269773, 6925, 303, 4, 44, "Text",
      CellTags->{"second-order dispersion", "KdV-\[Epsilon] equation"}]},
  "KdV-\[Epsilon] equation"->{
    Cell[269773, 6925, 303, 4, 44, "Text",
      CellTags->{"second-order dispersion", "KdV-\[Epsilon] equation"}]},
  "approximate symmetry group"->{
    Cell[271352, 6974, 177, 4, 27, "Text",
      CellTags->"approximate symmetry group"]},
  "reduced KdV-\[Epsilon] equation"->{
    Cell[276313, 7096, 851, 17, 95, "Text",
      CellTags->{
      "similarity representation", "similarity variable", 
        "determining equation", "reduced KdV-\[Epsilon] equation"}]},
  "DSolve[]"->{
    Cell[277731, 7136, 284, 8, 46, "Text",
      CellTags->"DSolve[]"]},
  "elliptic functions"->{
    Cell[280222, 7202, 397, 9, 46, "Text",
      CellTags->{"elliptic functions", "Solve[]"}]},
  "Solve[]"->{
    Cell[280222, 7202, 397, 9, 46, "Text",
      CellTags->{"elliptic functions", "Solve[]"}]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"Approximate Symmetries of Partial Differential Equations", 286629, 7402},
  {"Introduction", 287038, 7410},
  {"approximate symmetries", 287383, 7417},
  {"extension of Lie's theory", 288078, 7433},
  {"small perturbation", 288330, 7439},
  {"approximate group", 288581, 7445},
  {"Lie point symmetries", 288945, 7453},
  {"Lie's theory", 289322, 7462},
  {"approximate calculus", 289578, 7468},
  {"critical parameter", 289832, 7474},
  {"approximate transformation groups", 290101, 7480},
  {"approximation", 290350, 7486},
  {"infinitesimal small functions", 290526, 7490},
  {"Definition: Approximation", 290698, 7494},
  {"One-Parameter Approximation Group", 290831, 7497},
  {"one-parameter approximation group", 290996, 7501},
  {"approximate transformation", 291192, 7506},
  {"eq-3", 291366, 7511},
  {"group parameter", 291467, 7514},
  {"perturbation parameter", 291630, 7518},
  {"approximate identity element", 291799, 7522},
  {"local one-parameter approximate transformation group", 291976, 7526},
  {"first-order approximation", 292150, 7530},
  {"approximate association relation", 292394, 7536},
  {"group generator", 292539, 7540},
  {"Lie's equation", 292923, 7550},
  {"infinitesimal transformations", 293134, 7556},
  {"Approximate group generator", 293343, 7562},
  {"Lie point group", 293485, 7566},
  {"vector field", 293703, 7572},
  {"vector field of an approximate group", 294047, 7580},
  {"eq-7", 294257, 7586},
  {"eq-8", 294348, 7589},
  {"eq-9", 294439, 7592},
  {"approximate vector field", 294551, 7595},
  {"Taylor expansion", 294798, 7601},
  {"The Determining Equations and an Algorithm of Calculation", 294995, \
7605},
  {"classical Lie theory", 295179, 7610},
  {"approximate group analysis", 295423, 7616},
  {"algorithm of approximate group analysis", 295680, 7622},
  {"eq-11", 295903, 7628},
  {"approximate invariant", 296013, 7631},
  {"determining equation for approximate symmetries", 296152, 7634},
  {"algorithm to calculate first-order approximate symmetries", 296328, \
7637},
  {"Theorem: First-order approximations", 296498, 7641},
  {"auxiliary factor", 296623, 7644},
  {"algorithm of first-order approximation", 296749, 7647},
  {"deficiency", 296868, 7650},
  {"prolongation formula", 296974, 7653},
  {"classical Lie algorithm", 297161, 7658},
  {"ApproximateSymmetries[]", 297348, 7663},
  {"Examples", 297682, 7672},
  {"computer algebra", 297780, 7675},
  {"Isentropic Liquid", 297884, 7678},
  {"liquid in a pipe", 297993, 7681},
  {"hydraulic-friction coefficient", 298111, 7684},
  {"pressure", 298383, 7691},
  {"isentropic exponent", 298481, 7694},
  {"similarity representation", 298760, 7701},
  {"approximation order", 299244, 7712},
  {"InverseFunction[]", 299353, 7715},
  {"isentropic fluid", 299459, 7718},
  {"finite symmetry group", 299569, 7721},
  {"similarity variable", 299699, 7724},
  {"similarity solution", 300056, 7732},
  {"determining equation", 300233, 7737},
  {"Lagrange coordinates", 300591, 7745},
  {"isentropic motion", 300785, 7750},
  {"infinitesimals", 300890, 7753},
  {"isentropic model", 301049, 7757},
  {"generating vector field", 301215, 7761},
  {"NonCommutativeMultiply[]", 301382, 7765},
  {"contour plot", 301491, 7768},
  {"Disturbed Korteweg de Vries Equation", 301613, 7771},
  {"soliton theory", 301742, 7774},
  {"Korteweg-de Vries equation", 301887, 7777},
  {"infinite number of integrals of motion", 302044, 7780},
  {"shallow water waves", 302273, 7785},
  {"cnoidal waves", 302496, 7790},
  {"fluid dynamics", 302720, 7795},
  {"turbulent theory", 302946, 7800},
  {"second-order dispersion", 303179, 7805},
  {"KdV-\[Epsilon] equation", 303327, 7808},
  {"approximate symmetry group", 303478, 7811},
  {"reduced KdV-\[Epsilon] equation", 303608, 7814},
  {"DSolve[]", 303815, 7819},
  {"elliptic functions", 303914, 7822},
  {"Solve[]", 304025, 7825}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 421, 7, 94, "ChapterLine",
  Evaluatable->False,
  CounterAssignments->{{"Title", 7}, {"SubTitle", 0}, {"SubSubTitle", 0}, {
      "Section", 0}, {"SubSection", 0}, {"SubSubSection", 0}, {
      "NumberedEquation", 0}, {"NumberedFigure", 0}, {"NumberedTable", 0}},
  CellTags->"Approximate Symmetries of Partial Differential Equations"],
Cell[2163, 60, 406, 9, 184, "Title",
  Evaluatable->False,
  CounterAssignments->{{"Title", 7}, {"SubTitle", 0}, {"SubSubTitle", 0}, {
      "Section", 0}, {"SubSection", 0}, {"SubSubSection", 0}, {
      "NumberedEquation", 0}, {"NumberedFigure", 0}, {"NumberedTable", 0}}],

Cell[CellGroupData[{
Cell[2594, 73, 332, 9, 43, "Section",
  CounterAssignments->{{"Title", 8}, {"SubTitle", 0}, {"SubSubTitle", 0}, {
      "Section", 0}, {"SubSection", 0}, {"SubSubSection", 0}, {
      "NumberedEquation", 0}, {"NumberedFigure", 0}, {"NumberedTable", 0}},
  CellTags->"Introduction"],
Cell[2929, 84, 1407, 25, 180, "Text",
  Evaluatable->False,
  CellTags->{
  "approximate symmetries", "extension of Lie's theory", "small perturbation",
     "approximate group", "Lie point symmetries"}],
Cell[4339, 111, 778, 14, 95, "Text",
  Evaluatable->False,
  CellTags->{
  "Lie's theory", "approximate calculus", "critical parameter", 
    "approximate transformation groups", "approximate symmetries"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[5154, 130, 90, 5, 43, "Section"],
Cell[5247, 137, 1162, 29, 114, "Text",
  Evaluatable->False,
  CellTags->{"approximation", "infinitesimal small functions"}],
Cell[6412, 168, 165, 3, 36, "NumberedEquation"],
Cell[6580, 173, 141, 5, 27, "Text",
  Evaluatable->False],
Cell[6724, 180, 88, 1, 33, "Definition",
  CellTags->"Definition: Approximation"],
Cell[6815, 183, 451, 19, 27, "Text",
  Evaluatable->False],
Cell[7269, 204, 131, 3, 22, "NumberedEquation"],
Cell[7403, 209, 189, 7, 27, "Text",
  Evaluatable->False],
Cell[7595, 218, 171, 5, 27, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[7803, 228, 206, 8, 43, "Section",
  Evaluatable->False,
  CellTags->"One-Parameter Approximation Group"],
Cell[8012, 238, 1046, 34, 62, "Text",
  Evaluatable->False,
  CellTags->{
  "one-parameter approximation group", "approximate transformation"}],
Cell[9061, 274, 323, 8, 40, "NumberedEquation",
  CellTags->"eq-3"],
Cell[9387, 284, 1398, 38, 61, "Text",
  Evaluatable->False,
  CellTags->{"group parameter", "perturbation parameter"}],
Cell[10788, 324, 121, 2, 22, "NumberedEquation"],
Cell[10912, 328, 202, 6, 27, "Text",
  Evaluatable->False,
  CellTags->"approximate identity element"],
Cell[11117, 336, 286, 9, 20, "NumberedEquation",
  Evaluatable->False],
Cell[11406, 347, 430, 13, 44, "Text",
  Evaluatable->False],
Cell[11839, 362, 256, 5, 27, "Text",
  Evaluatable->False,
  CellTags->"local one-parameter approximate transformation group"],
Cell[12098, 369, 189, 3, 20, "NumberedEquation"],
Cell[12290, 374, 163, 7, 27, "Text",
  Evaluatable->False],
Cell[12456, 383, 28, 0, 29, "Example"],
Cell[12487, 385, 851, 30, 61, "Text",
  Evaluatable->False,
  CellTags->"first-order approximation"],
Cell[13341, 417, 143, 3, 34, "Input"],
Cell[13487, 422, 160, 3, 34, "Input"],
Cell[13650, 427, 116, 2, 27, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[13791, 433, 111, 2, 23, "Input"],
Cell[13905, 437, 78, 1, 34, "Output"]
}, Open  ]],
Cell[13998, 441, 486, 16, 44, "Text",
  Evaluatable->False],

Cell[CellGroupData[{
Cell[14509, 461, 142, 3, 23, "Input"],
Cell[14654, 466, 216, 4, 34, "Output"]
}, Open  ]],
Cell[14885, 473, 887, 20, 95, "Text",
  Evaluatable->False,
  CellTags->"approximate association relation"],
Cell[15775, 495, 558, 11, 61, "Text",
  Evaluatable->False,
  CellTags->{
  "group generator", "Lie's equation", "infinitesimal transformations"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[16370, 511, 194, 8, 43, "Section",
  Evaluatable->False,
  CellTags->"Approximate group generator"],
Cell[16567, 521, 414, 10, 44, "Text",
  Evaluatable->False,
  CellTags->{
  "Lie point group", "group generator", "vector field", 
    "vector field of an approximate group"}],
Cell[16984, 533, 208, 6, 48, "NumberedEquation",
  CellTags->"eq-7"],
Cell[17195, 541, 73, 2, 27, "Text",
  Evaluatable->False],
Cell[17271, 545, 271, 6, 23, "NumberedEquation",
  CellTags->"eq-8"],
Cell[17545, 553, 283, 7, 44, "Text",
  Evaluatable->False],
Cell[17831, 562, 471, 11, 35, "NumberedEquation",
  CellTags->"eq-9"],
Cell[18305, 575, 149, 3, 27, "Text",
  Evaluatable->False,
  CellTags->"approximate vector field"],
Cell[18457, 580, 292, 5, 37, "NumberedEquation"],
Cell[18752, 587, 771, 21, 78, "Text",
  Evaluatable->False,
  CellTags->{"Lie point symmetries", "Taylor expansion"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[19560, 613, 254, 8, 43, "Section",
  Evaluatable->False,
  CellTags->"The Determining Equations and an Algorithm of Calculation"],
Cell[19817, 623, 602, 13, 61, "Text",
  CellTags->{
  "classical Lie theory", "approximate group analysis", 
    "algorithm of approximate group analysis", "approximate symmetries"}],
Cell[20422, 638, 372, 10, 44, "Text"],
Cell[20797, 650, 454, 10, 24, "NumberedEquation",
  CellTags->"eq-11"],
Cell[21254, 662, 157, 3, 27, "Text",
  CellTags->"approximate invariant"],
Cell[21414, 667, 151, 3, 22, "NumberedEquation"],
Cell[21568, 672, 154, 5, 27, "Text"],
Cell[21725, 679, 418, 11, 44, "Text",
  CellTags->{"approximate vector field", "approximate group"}],
Cell[22146, 692, 254, 5, 23, "NumberedEquation"],
Cell[22403, 699, 661, 12, 95, "Text",
  CellTags->"determining equation for approximate symmetries"],
Cell[23067, 713, 224, 6, 44, "Text"],
Cell[23294, 721, 171, 4, 22, "NumberedEquation"],
Cell[23468, 727, 19, 0, 27, "Text"],
Cell[23490, 729, 335, 7, 37, "NumberedEquation"],
Cell[23828, 738, 100, 3, 27, "Text"],
Cell[23931, 743, 649, 16, 26, "NumberedEquation"],
Cell[24583, 761, 366, 8, 44, "Text",
  CellTags->"algorithm to calculate first-order approximate symmetries"],
Cell[24952, 771, 105, 1, 33, "Theorem",
  CellTags->"Theorem: First-order approximations"],
Cell[25060, 774, 148, 3, 44, "Text"],
Cell[25211, 779, 330, 8, 24, "NumberedEquation"],
Cell[25544, 789, 20, 0, 27, "Text"],
Cell[25567, 791, 311, 7, 24, "NumberedEquation"],
Cell[25881, 800, 446, 13, 44, "Text",
  CellTags->"auxiliary factor"],
Cell[26330, 815, 232, 5, 44, "Text",
  CellTags->"algorithm of first-order approximation"],
Cell[26565, 822, 284, 7, 35, "ListItem"],
Cell[26852, 831, 321, 8, 26, "NumberedEquation"],
Cell[27176, 841, 545, 16, 35, "ListItem",
  CellTags->"deficiency"],
Cell[27724, 859, 461, 10, 35, "NumberedEquation"],
Cell[28188, 871, 128, 4, 20, "ListItem"],
Cell[28319, 877, 401, 11, 26, "NumberedEquation"],
Cell[28723, 890, 95, 1, 20, "ListItem"],
Cell[28821, 893, 693, 16, 80, "Text",
  CellTags->{
  "prolongation formula", "classical Lie algorithm", 
    "ApproximateSymmetries[]"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[29551, 914, 108, 6, 43, "Section",
  CellTags->"Examples"],
Cell[29662, 922, 404, 9, 61, "Text",
  CellTags->"computer algebra"],

Cell[CellGroupData[{
Cell[30091, 935, 78, 1, 41, "Subsection",
  CellTags->"Isentropic Liquid"],
Cell[30172, 938, 416, 11, 44, "Text",
  CellTags->"liquid in a pipe"],
Cell[30591, 951, 106, 2, 23, "NumberedEquation"],
Cell[30700, 955, 20, 0, 27, "Text"],
Cell[30723, 957, 384, 10, 22, "NumberedEquation"],
Cell[31110, 969, 420, 13, 44, "Text",
  CellTags->"hydraulic-friction coefficient"],

Cell[CellGroupData[{
Cell[31555, 986, 243, 4, 53, "Input"],
Cell[31801, 992, 430, 13, 46, "Output"]
}, Open  ]],
Cell[32246, 1008, 26, 0, 27, "Text"],
Cell[32275, 1010, 93, 1, 23, "Input"],
Cell[32371, 1013, 39, 0, 27, "Text"],

Cell[CellGroupData[{
Cell[32435, 1017, 86, 1, 23, "Input"],
Cell[32524, 1020, 290, 9, 34, "Output"]
}, Open  ]],
Cell[32829, 1032, 433, 14, 44, "Text"],

Cell[CellGroupData[{
Cell[33287, 1050, 963, 22, 55, "Input"],
Cell[34253, 1074, 273, 8, 23, "Output"]
}, Open  ]],
Cell[34541, 1085, 90, 1, 27, "Text",
  CellTags->"pressure"],
Cell[34634, 1088, 141, 4, 37, "Input"],
Cell[34778, 1094, 85, 1, 27, "Text"],

Cell[CellGroupData[{
Cell[34888, 1099, 99, 2, 23, "Input"],
Cell[34990, 1103, 372, 11, 24, "Output"]
}, Open  ]],
Cell[35377, 1117, 689, 13, 99, "Text",
  CellTags->{
  "isentropic exponent", "hydraulic-friction coefficient", 
    "ApproximateSymmetries[]"}],

Cell[CellGroupData[{
Cell[36091, 1134, 175, 3, 39, "Input"],
Cell[36269, 1139, 3727, 102, 90, "Output"]
}, Open  ]],
Cell[40011, 1244, 596, 9, 97, "Text"],

Cell[CellGroupData[{
Cell[40632, 1257, 303, 6, 55, "Input"],
Cell[40938, 1265, 612, 10, 93, "Output"]
}, Open  ]],
Cell[41565, 1278, 314, 5, 61, "Text",
  CellTags->"similarity representation"],

Cell[CellGroupData[{
Cell[41904, 1287, 287, 5, 39, "Input"],
Cell[42194, 1294, 1091, 25, 66, "Output"]
}, Open  ]],
Cell[43300, 1322, 694, 16, 61, "Text",
  CellTags->"approximation order"],

Cell[CellGroupData[{
Cell[44019, 1342, 194, 4, 23, "Input"],
Cell[44216, 1348, 197, 4, 37, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[44450, 1357, 138, 4, 23, "Input"],
Cell[44591, 1363, 249, 6, 37, "Output"]
}, Open  ]],
Cell[44855, 1372, 833, 19, 112, "Text"],

Cell[CellGroupData[{
Cell[45713, 1395, 169, 3, 23, "Input"],
Cell[45885, 1400, 603, 14, 40, "Output"]
}, Open  ]],
Cell[46503, 1417, 165, 5, 27, "Text"],

Cell[CellGroupData[{
Cell[46693, 1426, 67, 1, 23, "Input"],
Cell[46763, 1429, 6799, 119, 636, "Output"]
}, Open  ]],
Cell[53577, 1551, 692, 18, 78, "Text",
  CellTags->"isentropic exponent"],

Cell[CellGroupData[{
Cell[54294, 1573, 142, 3, 23, "Input"],
Cell[54439, 1578, 3123, 58, 332, "Output",
  PageBreakBelow->True]
}, Open  ]],
Cell[57577, 1639, 345, 13, 29, "Text",
  CellTags->"InverseFunction[]"],
Cell[57925, 1654, 540, 11, 78, "Text",
  CellTags->"isentropic fluid"],

Cell[CellGroupData[{
Cell[58490, 1669, 273, 6, 39, "Input"],
Cell[58766, 1677, 2610, 76, 100, "Output"]
}, Open  ]],
Cell[61391, 1756, 241, 5, 44, "Text",
  CellTags->{"finite symmetry group", "vector field"}],

Cell[CellGroupData[{
Cell[61657, 1765, 307, 6, 55, "Input"],
Cell[61967, 1773, 481, 7, 90, "Output"]
}, Open  ]],
Cell[62463, 1783, 429, 11, 28, "Text"],

Cell[CellGroupData[{
Cell[62917, 1798, 320, 5, 55, "Input"],
Cell[63240, 1805, 396, 12, 46, "Output"]
}, Open  ]],
Cell[63651, 1820, 767, 21, 61, "Text",
  CellTags->{
  "similarity variable", "similarity solution", "determining equation"}],

Cell[CellGroupData[{
Cell[64443, 1845, 121, 3, 23, "Input"],
Cell[64567, 1850, 113, 2, 41, "Output"]
}, Open  ]],
Cell[64695, 1855, 338, 5, 61, "Text",
  CellTags->"Lagrange coordinates"],

Cell[CellGroupData[{
Cell[65058, 1864, 147, 4, 23, "Input"],
Cell[65208, 1870, 139, 3, 37, "Output"]
}, Open  ]],
Cell[65362, 1876, 127, 3, 27, "Text"],

Cell[CellGroupData[{
Cell[65514, 1883, 280, 5, 39, "Input"],
Cell[65797, 1890, 129806, 3231, 166, 100407, 2863, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]],
Cell[195618, 5124, 352, 10, 44, "Text",
  CellTags->"isentropic motion"],

Cell[CellGroupData[{
Cell[195995, 5138, 88, 1, 23, "Input"],
Cell[196086, 5141, 533, 15, 36, "Output"]
}, Open  ]],
Cell[196634, 5159, 92, 3, 27, "Text"],

Cell[CellGroupData[{
Cell[196751, 5166, 220, 4, 39, "Input"],
Cell[196974, 5172, 1451, 44, 58, "Output"]
}, Open  ]],
Cell[198440, 5219, 306, 5, 44, "Text",
  CellTags->{
  "infinitesimals", "isentropic model", "generating vector field"}],

Cell[CellGroupData[{
Cell[198771, 5228, 696, 13, 87, "Input"],
Cell[199470, 5243, 817, 17, 120, "Output"]
}, Open  ]],
Cell[200302, 5263, 331, 8, 46, "Text",
  CellTags->"NonCommutativeMultiply[]"],

Cell[CellGroupData[{
Cell[200658, 5275, 285, 5, 39, "Input"],
Cell[200946, 5282, 329, 5, 59, "Output"]
}, Open  ]],
Cell[201290, 5290, 178, 4, 44, "Text"],

Cell[CellGroupData[{
Cell[201493, 5298, 417, 7, 55, "Input"],
Cell[201913, 5307, 967, 26, 90, "Output"]
}, Open  ]],
Cell[202895, 5336, 221, 6, 27, "Text",
  CellTags->"similarity representation"],

Cell[CellGroupData[{
Cell[203141, 5346, 84, 1, 34, "Input"],
Cell[203228, 5349, 80, 1, 34, "Output"]
}, Open  ]],
Cell[203323, 5353, 616, 17, 61, "Text"],

Cell[CellGroupData[{
Cell[203964, 5374, 151, 3, 23, "Input"],
Cell[204118, 5379, 71, 1, 23, "Output"]
}, Open  ]],
Cell[204204, 5383, 312, 12, 27, "Text",
  CellTags->"Lagrange coordinates"],

Cell[CellGroupData[{
Cell[204541, 5399, 168, 4, 23, "Input"],
Cell[204712, 5405, 90, 1, 23, "Output"]
}, Open  ]],
Cell[204817, 5409, 443, 12, 61, "Text",
  CellTags->"contour plot"],

Cell[CellGroupData[{
Cell[205285, 5425, 285, 5, 39, "Input"],
Cell[205573, 5432, 61727, 1423, 192, 10874, 791, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[267349, 6861, 116, 1, 41, "Subsection",
  CellTags->"Disturbed Korteweg de Vries Equation"],
Cell[267468, 6864, 187, 4, 27, "Text",
  CellTags->{"soliton theory", "Korteweg-de Vries equation"}],
Cell[267658, 6870, 105, 2, 23, "NumberedEquation"],
Cell[267766, 6874, 1417, 27, 163, "Text",
  CellTags->{
  "infinite number of integrals of motion", "shallow water waves", 
    "cnoidal waves", "fluid dynamics", "turbulent theory"}],

Cell[CellGroupData[{
Cell[269208, 6905, 238, 5, 39, "Input"],
Cell[269449, 6912, 309, 10, 23, "Output"]
}, Open  ]],
Cell[269773, 6925, 303, 4, 44, "Text",
  CellTags->{"second-order dispersion", "KdV-\[Epsilon] equation"}],

Cell[CellGroupData[{
Cell[270101, 6933, 228, 4, 39, "Input"],
Cell[270332, 6939, 1005, 32, 44, "Output"]
}, Open  ]],
Cell[271352, 6974, 177, 4, 27, "Text",
  CellTags->"approximate symmetry group"],

Cell[CellGroupData[{
Cell[271554, 6982, 692, 13, 87, "Input"],
Cell[272249, 6997, 1701, 37, 92, "Output"]
}, Open  ]],
Cell[273965, 7037, 68, 0, 27, "Text"],

Cell[CellGroupData[{
Cell[274058, 7041, 279, 5, 39, "Input"],
Cell[274340, 7048, 250, 4, 38, "Output"]
}, Open  ]],
Cell[274605, 7055, 383, 5, 78, "Text"],

Cell[CellGroupData[{
Cell[275013, 7064, 592, 10, 71, "Input"],
Cell[275608, 7076, 690, 17, 60, "Output"]
}, Open  ]],
Cell[276313, 7096, 851, 17, 95, "Text",
  CellTags->{
  "similarity representation", "similarity variable", "determining equation", 
    "reduced KdV-\[Epsilon] equation"}],

Cell[CellGroupData[{
Cell[277189, 7117, 269, 7, 39, "Input"],
Cell[277461, 7126, 255, 7, 36, "Output"]
}, Open  ]],
Cell[277731, 7136, 284, 8, 46, "Text",
  CellTags->"DSolve[]"],

Cell[CellGroupData[{
Cell[278040, 7148, 71, 1, 23, "Input"],
Cell[278114, 7151, 179, 3, 38, "Output"],
Cell[278296, 7156, 231, 5, 38, "Output"],
Cell[278530, 7163, 231, 5, 38, "Output"],
Cell[278764, 7170, 208, 4, 38, "Output"],
Cell[278975, 7176, 1232, 23, 212, "Output"]
}, Open  ]],
Cell[280222, 7202, 397, 9, 46, "Text",
  CellTags->{"elliptic functions", "Solve[]"}],

Cell[CellGroupData[{
Cell[280644, 7215, 71, 1, 23, "Input"],
Cell[280718, 7218, 87, 1, 41, "Output"]
}, Open  ]],
Cell[280820, 7222, 142, 4, 27, "Text",
  CellTags->"similarity representation"],

Cell[CellGroupData[{
Cell[280987, 7230, 174, 3, 23, "Input"],
Cell[281164, 7235, 154, 4, 43, "Output"]
}, Open  ]],
Cell[281333, 7242, 227, 6, 27, "Text"],

Cell[CellGroupData[{
Cell[281585, 7252, 86, 1, 23, "Input"],
Cell[281674, 7255, 827, 24, 39, "Output"]
}, Open  ]],
Cell[282516, 7282, 363, 8, 61, "Text"],

Cell[CellGroupData[{
Cell[282904, 7294, 182, 5, 23, "Input"],
Cell[283089, 7301, 207, 5, 39, "Output"]
}, Open  ]],
Cell[283311, 7309, 190, 4, 44, "Text",
  CellTags->"first-order approximation"],

Cell[CellGroupData[{
Cell[283526, 7317, 64, 1, 23, "Input"],
Cell[283593, 7320, 301, 6, 39, "Output"]
}, Open  ]],
Cell[283909, 7329, 549, 11, 78, "Text"],
Cell[284461, 7342, 148, 3, 35, "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

