/*
 * Decompiled with CFR 0.152.
 */
package com.google.cm.diagnostic.tools.tinkexample;

import com.beust.jcommander.DynamicParameter;
import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import com.beust.jcommander.ParameterException;
import com.google.crypto.tink.Aead;
import com.google.crypto.tink.BinaryKeysetWriter;
import com.google.crypto.tink.KeysetHandle;
import com.google.crypto.tink.aead.AeadConfig;
import com.google.crypto.tink.aead.LegacyKmsAeadParameters;
import com.google.crypto.tink.aead.XChaCha20Poly1305Parameters;
import com.google.crypto.tink.integration.gcpkms.GcpKmsClient;
import com.google.crypto.tink.proto.EncryptedKeyset;
import com.google.protobuf.ByteString;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Pattern;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVPrinter;
import org.apache.commons.csv.CSVRecord;

public final class TinkExampleApplication {
    private static final byte[] ASSOCIATED_DATA = new byte[0];
    private static final CSVFormat CSV_READ_FORMAT = CSVFormat.newFormat(',').builder().setRecordSeparator('\n').setHeader(new String[0]).setSkipHeaderRecord(true).build();
    private static final CSVFormat CSV_WRITE_FORMAT_TEMPLATE = CSVFormat.newFormat(',').builder().setRecordSeparator('\n').build();
    private static final Base64.Encoder BASE64_ENCODER = Base64.getEncoder();
    private static final MessageDigest SHA256_HASHER = TinkExampleApplication.getSha256Hasher();
    private static final Pattern PREFIX_PATTERN = Pattern.compile("^(?:mr|mrs|ms|dr)\\.\\s+");
    private static final Pattern SUFFIX_PATTERN = Pattern.compile("\\s+(?:jr\\.|sr\\.|2nd|3rd|ii|iii|iv|v|vi|cpa|dc|dds|vm|jd|md|phd)$");
    private static final Pattern NON_DIGITS_PATTERN = Pattern.compile("[^0-9]");
    private static final Pattern WHITESPACE_PATTERN = Pattern.compile("\\s");

    private TinkExampleApplication() {
    }

    public static void main(String[] args) throws IOException, GeneralSecurityException {
        TinkExampleApplicationArguments cliArgs = TinkExampleApplicationArguments.parseMainArgs(args);
        String kekUri = cliArgs.kekUri;
        Map<String, String> columnTypes = cliArgs.columnTypes;
        Path inputFilePath = Path.of(cliArgs.inputFilePath, new String[0]);
        Path outputFilePath = Path.of(cliArgs.outputFilePath, new String[0]);
        Optional<String> credentialsFilePath = Optional.ofNullable(cliArgs.credentialFilePath);
        AeadConfig.register();
        GcpKmsClient.register(Optional.of(kekUri), credentialsFilePath);
        KeysetHandle kekHandle = KeysetHandle.generateNew(LegacyKmsAeadParameters.create(kekUri));
        Aead kekAead = kekHandle.getPrimitive(Aead.class);
        KeysetHandle dekHandle = KeysetHandle.generateNew(XChaCha20Poly1305Parameters.create(XChaCha20Poly1305Parameters.Variant.TINK));
        Aead dekAead = dekHandle.getPrimitive(Aead.class);
        ByteArrayOutputStream dekStream = new ByteArrayOutputStream();
        dekHandle.write(BinaryKeysetWriter.withOutputStream(dekStream), kekAead);
        ByteString dekBytes = EncryptedKeyset.parseFrom(dekStream.toByteArray()).getEncryptedKeyset();
        String encryptedDek = BASE64_ENCODER.encodeToString(dekBytes.toByteArray());
        try (BufferedReader reader = Files.newBufferedReader(inputFilePath);
             CSVParser parser = CSV_READ_FORMAT.parse(reader);
             BufferedWriter writer = Files.newBufferedWriter(outputFilePath, new OpenOption[0]);
             CSVPrinter printer = TinkExampleApplication.getCsvWriteFormat(parser.getHeaderNames()).print(writer);){
            try {
                for (CSVRecord record : parser) {
                    if (!record.isConsistent()) {
                        throw new IllegalArgumentException("Inconsistent CSV record cannot be encrypted.");
                    }
                    for (String column : parser.getHeaderNames()) {
                        String text = record.get(column);
                        String dataType = Objects.requireNonNullElse(columnTypes.get(column), "unknown");
                        printer.print(TinkExampleApplication.processColumn(text, dataType, dekAead));
                    }
                    printer.print(encryptedDek);
                    printer.print(kekUri);
                    printer.println();
                }
            }
            catch (IOException | RuntimeException | GeneralSecurityException ex) {
                throw new IOException("Encryption failed on line " + parser.getCurrentLineNumber(), ex);
            }
        }
    }

    private static MessageDigest getSha256Hasher() {
        try {
            return MessageDigest.getInstance("SHA-256");
        }
        catch (NoSuchAlgorithmException ex) {
            throw new RuntimeException(ex);
        }
    }

    private static CSVFormat getCsvWriteFormat(List<String> headerNames) {
        String[] headerRecord = new String[headerNames.size() + 2];
        headerNames.toArray(headerRecord);
        headerRecord[headerRecord.length - 2] = "encrypted_dek";
        headerRecord[headerRecord.length - 1] = "kek_uri";
        return CSV_WRITE_FORMAT_TEMPLATE.builder().setHeader(headerRecord).build();
    }

    private static String processColumn(String input, String dataType, Aead dekAead) throws GeneralSecurityException {
        if (input.isEmpty()) {
            return input;
        }
        switch (dataType.toLowerCase(Locale.US)) {
            case "email": {
                return TinkExampleApplication.encrypt(dekAead, TinkExampleApplication.hash(TinkExampleApplication.normalizeEmail(input)));
            }
            case "phone": {
                return TinkExampleApplication.encrypt(dekAead, TinkExampleApplication.hash(TinkExampleApplication.normalizePhone(input)));
            }
            case "first_name": {
                return TinkExampleApplication.encrypt(dekAead, TinkExampleApplication.hash(TinkExampleApplication.normalizeFirstName(input)));
            }
            case "last_name": {
                return TinkExampleApplication.encrypt(dekAead, TinkExampleApplication.hash(TinkExampleApplication.normalizeLastName(input)));
            }
        }
        return input;
    }

    private static String normalizeEmail(String input) {
        return WHITESPACE_PATTERN.matcher(input.toLowerCase(Locale.US)).replaceAll("");
    }

    private static String normalizeFirstName(String input) {
        return PREFIX_PATTERN.matcher(input.strip().toLowerCase(Locale.US)).replaceAll("");
    }

    private static String normalizeLastName(String input) {
        return SUFFIX_PATTERN.matcher(input.strip().toLowerCase(Locale.US)).replaceAll("");
    }

    private static String normalizePhone(String input) {
        return "+" + NON_DIGITS_PATTERN.matcher(input).replaceAll("");
    }

    private static String hash(String text) {
        return BASE64_ENCODER.encodeToString(SHA256_HASHER.digest(text.getBytes(StandardCharsets.UTF_8)));
    }

    private static String encrypt(Aead dekAead, String text) throws GeneralSecurityException {
        return BASE64_ENCODER.encodeToString(dekAead.encrypt(text.getBytes(StandardCharsets.UTF_8), ASSOCIATED_DATA));
    }

    private static class TinkExampleApplicationArguments {
        @Parameter(names={"-i", "--input-file"}, required=true, description="Input CSV file. Must contain a header record. Values must be cleartext.")
        String inputFilePath;
        @Parameter(names={"-o", "--output-file"}, required=true, description="Output CSV file.")
        String outputFilePath;
        @Parameter(names={"-u", "--kek-uri"}, required=true, description="GCP KMS KEK URI in the format \"gcp-kms://projects/{{project}}/locations/{{location}}/keyRings/{{keyring}}/cryptoKeys/{{key}}\".")
        String kekUri;
        @DynamicParameter(names={"-t", "--column-type"}, description="This flag requires two arguments separated with an equal sign: first the name of a column, then its data type. Use a separate flag for each pair. Example: -t Email=email -t Phone=phone Valid types: email, phone, first_name, last_name, country_code, and zip_code. Includes by default one column for each type, where each name matches the type.")
        Map<String, String> columnTypes = new HashMap<String, String>(Map.ofEntries(Map.entry("email", "email"), Map.entry("phone", "phone"), Map.entry("first_name", "first_name"), Map.entry("last_name", "last_name"), Map.entry("country_code", "country_code"), Map.entry("zip_code", "zip_code")));
        @Parameter(names={"-c", "--gcp-credentials-file"}, description="Path to JSON file containing GCP credentials authorized for KMS encryption requests. Defaults to GCP Application Default Credentials path. See: https://cloud.google.com/docs/authentication/provide-credentials-adc")
        String credentialFilePath;

        private TinkExampleApplicationArguments() {
        }

        static TinkExampleApplicationArguments parseMainArgs(String[] args) {
            TinkExampleApplicationArguments cliArgs = new TinkExampleApplicationArguments();
            try {
                JCommander.newBuilder().addObject(cliArgs).build().parse(args);
            }
            catch (ParameterException ex) {
                ex.getJCommander().usage();
                throw ex;
            }
            return cliArgs;
        }
    }
}

