<?php
/**
 * AccessiBe Component
 *
 * @copyright   Copyright (C) 2023 - present, AccessiBe Ltd. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Uri\Uri;

// Load component configuration
require_once __DIR__ . '/config.php';

// Load Composer autoloader if available
$composerAutoload = dirname(__DIR__) . '/vendor/autoload.php';
if (file_exists($composerAutoload)) {
    require_once $composerAutoload;
}

// Sanitize domain name - remove www, http/https, ports for local development
function sanitizeDomain($domain) {
    // Remove protocol if present
    $domain = preg_replace('/^https?:\/\//i', '', $domain);
    
    // Remove www prefix
    $domain = preg_replace('/^www\./i', '', $domain);
    
    // Remove trailing slash
    $domain = rtrim($domain, '/');
    
    // For local development, keep ports but clean format
    if (strpos($domain, 'localhost') === 0 || strpos($domain, '127.0.0.1') === 0) {
        return $domain;
    }
    
    // Remove port for production domains
    $domain = preg_replace('/:.*$/', '', $domain);
    
    return strtolower(trim($domain));
}

function get_current_domain() {
    return sanitizeDomain(Uri::getInstance()->getHost());
} 

// Get all domains from database (from menu items and site config)
function getAllDomains() {
    $domains = [];
    $db = Factory::getDbo();
    
    // Add current domain
    $current_domain = get_current_domain();
    $domains[] = $current_domain;
    
    try {
        // Get domains from menu items (external URLs)
        $query = $db->getQuery(true)
            ->select('DISTINCT link')
            ->from('#__menu')
            ->where('type = ' . $db->quote('url'))
            ->where('published = 1')
            ->where('link LIKE ' . $db->quote('http%'));
        
        $db->setQuery($query);
        $menuUrls = $db->loadColumn();
        
        if ($menuUrls) {
            foreach ($menuUrls as $url) {
                $domain = sanitizeDomain(parse_url($url, PHP_URL_HOST));
                if (!empty($domain) && !in_array($domain, $domains)) {
                    $domains[] = $domain;
                }
            }
        }
        
        // Get domains from component configuration (if any custom domains stored)
        $params = ComponentHelper::getParams('com_accessibe');
        $customDomains = json_decode($params->get('custom_domains', '[]'), true);
        if (is_array($customDomains)) {
            foreach ($customDomains as $domain) {
                $sanitized = sanitizeDomain($domain);
                if (!empty($sanitized) && !in_array($sanitized, $domains)) {
                    $domains[] = $sanitized;
                }
            }
        }
        
    } catch (Exception $e) {
        // Fallback if database queries fail
        error_log('Accessibe: Error fetching domains - ' . $e->getMessage());
    }
    
    // Add common local development domains
    $localDomains = [
        //'example.com'
    ];
    
    foreach ($localDomains as $domain) {
        $sanitized = sanitizeDomain($domain);
        if (!in_array($sanitized, $domains)) {
            $domains[] = $sanitized;
        }
    }

    $unique_domains = array_unique(array_filter($domains));

    $domain_list = array_map(function($domain) {
      return [
          'domain' => $domain,
          'siteId' => $domain
      ];
    }, $unique_domains);
        
    return $domain_list;
}

// Initialize merchant details for Accessibe
function getMerchantDetails() {
    $app = Factory::getApplication();
    $user = $app->getIdentity();
    $config = Factory::getConfig();
    $current_domain = get_current_domain();
    
    // Get component version
    $component = ComponentHelper::getComponent('com_accessibe');
    $accessibe_version = '1.0.0'; // Default version
    if ($component && isset($component->params)) {
        $params = json_decode($component->params);
        if (isset($params->version)) {
            $accessibe_version = $params->version;
        }
    }
    
    // Get current domain widget status
    $componentParams = ComponentHelper::getParams('com_accessibe');
    $domainSettings = json_decode($componentParams->get('domain_settings', '{}'), true);
    $widgetStatus = isset($domainSettings[$current_domain]) ? 
        (bool)$domainSettings[$current_domain]['widgetStatus'] : true;
    
    $detail = array(
        'source' => 'Joomla',
        'storeId' => $current_domain,
        'userId' => $user ? $user->id : 0,
        'email' => $user ? $user->email : '',
        'fullName' => $user ? $user->name : 'Guest',
        'isSignup' => false,
        //'licenseId' => $componentParams->get('license_id', ''),
        'widgetStatus' => $widgetStatus,
        'wrapperApp' => array(
            'name' => 'Joomla',
            'version' => $accessibe_version
        ),
        'joomlaVersion' => JVERSION,
        'siteUrl' => Uri::root(),
        'adminUrl' => Uri::root() . 'administrator/',
        'domains' => getAllDomains()
    );
    
    return $detail;
}

// Handle form submission
$app = Factory::getApplication();
$input = $app->input;


// Handle AJAX requests
if ($input->get('task') == 'ajax') {
    $action = $input->get('action');
    $dataParam = $input->post->get('data', '{}', 'string');
    $data = json_decode($dataParam, true);
    $domain = $input->get('domain', $data['domain'] ?? get_current_domain());

    $response = ['success' => false, 'message' => ''];
    
    try {
        $params = ComponentHelper::getParams('com_accessibe');
        $domainSettings = json_decode($params->get('domain_settings', '{}'), true);
        if (!is_array($domainSettings)) {
            $domainSettings = [];
        }
        $db = Factory::getDbo();
        switch ($action) {
            case 'enable_script':
                $domainSettings[$domain] = ['widgetStatus' => 1];
                $response['message'] = 'Script enabled for ' . $domain;
                break;
                
            case 'disable_script':
                $domainSettings[$domain] = ['widgetStatus' => 0];
                $response['message'] = 'Script disabled for ' . $domain;
                break;
                
            case 'get_merchant_details':
                $response['data'] = getMerchantDetails();
                $response['message'] = 'Merchant details retrieved successfully';
                break;
                
            case 'get_domains':
                $response['data'] = getAllDomains();
                $response['message'] = 'Domain list retrieved successfully';
                break;
                
            case 'add_verification_page':
                // Decode the JSON data
                $verificationData = $data;
                
                // Handle JSON decode errors
                if (json_last_error() !== JSON_ERROR_NONE) {
                    error_log('Accessibe: JSON decode error: ' . json_last_error_msg());
                    throw new Exception('Invalid JSON data: ' . json_last_error_msg());
                }
                
                $token = $verificationData['token'] ?? '';
                $fileName = 'accessibe_verification.txt';
                
                $result = array(
                    'type' => 'FILE_URL',
                    'path' => '',
                    'domain' => get_current_domain(),
                );
                
                if (!empty($token)) {
                    // Try to write to site root first
                    $rootPath = JPATH_ROOT . '/' . $fileName;
                    if (file_put_contents($rootPath, $token)) {
                        $result['path'] = Uri::root() . $fileName . '?v=' . time();
                    } else {
                        // Fallback to tmp directory
                        $tmpPath = JPATH_ROOT . '/tmp/' . $fileName;
                        if (file_put_contents($tmpPath, $token)) {
                            $result['path'] = Uri::root() . 'tmp/' . $fileName . '?v=' . time();
                        } else {
                            throw new Exception('Unable to create verification file');
                        }
                    }
                }
                
                $response['data'] = $result;
                $response['message'] = 'Verification file created successfully';
                break;
                
            case 'login':
                $loginData = $data;
                
                if (json_last_error() !== JSON_ERROR_NONE) {
                    throw new Exception('Invalid JSON data: ' . json_last_error_msg());
                }
                
                // Store login data in component params
                $currentParams = $params->toArray();
                $currentParams['login_data'] = $dataParam;
                
                $query = $db->getQuery(true)
                    ->update('#__extensions')
                    ->set('params = ' . $db->quote(json_encode($currentParams)))
                    ->where('element = ' . $db->quote('com_accessibe'))
                    ->where('type = ' . $db->quote('component'));
                
                $db->setQuery($query);
                $db->execute();
                
                // Track identity with Mixpanel
                if (class_exists('JoomlaMixpanelHandler') && isset($loginData['userId'])) {
                    try {
                        // Get saved UUID from component params
                        $user_metadata_json = $params->get('user_metadata', '{}');
                        $user_metadata = json_decode($user_metadata_json, true);
                        $device_uuid = $user_metadata['uuid'] ?? '';
                        
                        if (!empty($device_uuid)) {
                            $mixpanelHandler = JoomlaMixpanelHandler::getInstance();
                            $mixpanelHandler->trackEvent('userIdentified', ['$device_id' => $device_uuid, '$user_id' => $loginData['userId']]);
                        }
                    } catch (Exception $e) {
                        error_log('ACCESSIBE: Failed to track login identity: ' . $e->getMessage());
                    }
                }
                
                $response['data'] = $loginData;
                $response['message'] = 'Login data processed successfully';
                break;
                
            case 'signup':
                $signupData = $data;
                
                if (json_last_error() !== JSON_ERROR_NONE) {
                    throw new Exception('Invalid JSON data: ' . json_last_error_msg());
                }
                
                // Store signup data in component params
                $currentParams = $params->toArray();
                $currentParams['signup_data'] = $dataParam;
                
                $query = $db->getQuery(true)
                    ->update('#__extensions')
                    ->set('params = ' . $db->quote(json_encode($currentParams)))
                    ->where('element = ' . $db->quote('com_accessibe'))
                    ->where('type = ' . $db->quote('component'));
                
                $db->setQuery($query);
                $db->execute();
                
                // Track identity with Mixpanel
                if (class_exists('JoomlaMixpanelHandler') && isset($signupData['userId'])) {
                    try {
                        // Get saved UUID from component params
                        $user_metadata_json = $params->get('user_metadata', '{}');
                        $user_metadata = json_decode($user_metadata_json, true);
                        $device_uuid = $user_metadata['uuid'] ?? '';
                        
                        if (!empty($device_uuid)) {
                            $mixpanelHandler = JoomlaMixpanelHandler::getInstance();
                            $mixpanelHandler->trackEvent('userIdentified', ['$device_id' => $device_uuid, '$user_id' => $signupData['userId']]);
                        }
                    } catch (Exception $e) {
                        error_log('ACCESSIBE: Failed to track signup identity: ' . $e->getMessage());
                    }
                }
                
                $response['data'] = $signupData;
                $response['message'] = 'Signup data processed successfully';
                break;
                
            case 'acsb-user-loaded':
                $userData = $data; // contains keys acsbUser and signedDetails
                if (json_last_error() !== JSON_ERROR_NONE) {
                    throw new Exception('Invalid JSON data: ' . json_last_error_msg());
                }
                
                // Store user loaded data in component params
                $currentParams = $params->toArray();
                $currentParams['acsb_user_data'] = $dataParam;
                
                $query = $db->getQuery(true)
                    ->update('#__extensions')
                    ->set('params = ' . $db->quote(json_encode($currentParams)))
                    ->where('element = ' . $db->quote('com_accessibe'))
                    ->where('type = ' . $db->quote('component'));
                
                $db->setQuery($query);
                $db->execute();
                
                $response['data'] = $userData;
                $response['message'] = 'User data loaded successfully';
                break;
                
            case 'logout':
                $response['message'] = 'Logout processed successfully';
                break;
                
            case 'license-trial':
                $licenseData = $data;
                
                if (json_last_error() !== JSON_ERROR_NONE) {
                    throw new Exception('Invalid JSON data: ' . json_last_error_msg());
                }
                
                // Get existing licenses dictionary
                $currentParams = $params->toArray();
                $licenses = json_decode($currentParams['licenses'] ?? '{}', true);
                if (!is_array($licenses)) {
                    $licenses = [];
                }
                
                // Save license with licenseId as key
                if (isset($licenseData['licenseId'])) {
                    $licenses[$licenseData['licenseId']] = $licenseData;
                }
                
                $currentParams['licenses'] = json_encode($licenses);
                
                $query = $db->getQuery(true)
                    ->update('#__extensions')
                    ->set('params = ' . $db->quote(json_encode($currentParams)))
                    ->where('element = ' . $db->quote('com_accessibe'))
                    ->where('type = ' . $db->quote('component'));
                
                $db->setQuery($query);
                $db->execute();
                
                $response['data'] = $licenseData;
                $response['message'] = 'License trial data processed successfully';
                break;
                
            default:
                throw new Exception('Invalid action');
        }
        
        // Ensure domain has default settings if not present
        if (!isset($domainSettings[$domain]) || !isset($domainSettings[$domain]['widgetStatus'])) {
          if (!isset($domainSettings[$domain])) {
            $domainSettings[$domain] = [];
          }
          $domainSettings[$domain]['widgetStatus'] = 1; // Default enabled
        }
        // Save to database
        $params->set('domain_settings', json_encode($domainSettings));
        $query = $db->getQuery(true)
            ->update('#__extensions')
            ->set('params = ' . $db->quote($params->toString()))
            ->where('element = ' . $db->quote('com_accessibe'))
            ->where('type = ' . $db->quote('component'));
        
        $db->setQuery($query);
        $db->execute();
        
        $response['success'] = true;
        
    } catch (Exception $e) {
        $response['message'] = $e->getMessage();
    }
    
    header('Content-Type: application/json');
    echo json_encode($response);
    exit;
}

HTMLHelper::_('bootstrap.framework');
?>
<div class="acsb-wrap">
    <style>
        .acsb-wrap {
            height: calc(100vh - 150px);
        }
        @media screen and (max-width: 767px) {
            .acsb-wrap {
                height: calc(100vh - 200px);
            }
        }
        @media screen and (max-width: 575px) {
            .acsb-wrap {
                height: calc(100vh - 140px);
            }
        }
    </style>
    <?php
      $iframe_src = ACCESSIBE_UNIVERSAL_URL . (!empty($_SERVER['QUERY_STRING']) ? '?' . htmlspecialchars($_SERVER['QUERY_STRING'], ENT_QUOTES, 'UTF-8') : '');
    ?>
    <iframe 
        width="100%" 
        height="100%" 
        frameborder="0"
        style="border: none; display: block;"
        id='accessibe-universal-iframe'
        src='<?php echo htmlspecialchars($iframe_src, ENT_QUOTES, 'UTF-8'); ?>'>
    </iframe>
</div>

<script>
// Accessibe Universal Dashboard Communication
const ACSB_UNI_IFRAME_ID = 'accessibe-universal-iframe';
const AcsbStore = {
  isIframeReady: false,
  merchantData: null,
  jQueryReady: true,
};

const API = {
  sendMerchantDetails: () => {
    API.sendDataToIframe('merchantDetails', AcsbStore.merchantData);
  },
  syncMerchantDetails: async () => {
    if (AcsbStore.jQueryReady && AcsbStore.isIframeReady) {
      const detail = await API.proxy.fetchMerchantDetails();
      Logger.log("fetchMerchantDetails", detail);
      API.setMerchant(detail);
      API.sendMerchantDetails();
    }
  },
  setRedirectUrl: (url) => {
    if (AcsbStore.jQueryReady && AcsbStore.isIframeReady) {
      API.sendDataToIframe('redirect-url', url);
    }
  },
  setMerchant: (data) => {
    AcsbStore.merchantData = data;
    Logger.log("setting merchant data", data);
  },
  sendDomainVerificationRequest: (type, path, domain) => {
    API.sendDataToIframe('domain-verification-request', { type, path, domain });
  },
  sendDataToIframe: (eventName, data) => {
    document.getElementById(ACSB_UNI_IFRAME_ID).contentWindow.postMessage({ eventName, data }, '*');
  },
  ajax: {
    get: async (action) => {
      try {
        const url = `index.php?option=com_accessibe&task=ajax&action=${action}`;
        const response = await fetch(url, {
          method: 'GET',
          headers: {
            'X-Requested-With': 'XMLHttpRequest'
          }
        });
        const data = await response.json();
        return data;
      } catch (error) {
        Logger.error('AJAX GET error:', error);
        return null;
      }
    },
    post: async (action, payload) => {
      const url = `index.php?option=com_accessibe&task=ajax&action=${action}`;
      try {
        const response = await fetch(url, {
          method: 'POST',
          headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Content-Type': 'application/x-www-form-urlencoded'
          },
          body: 'data=' + encodeURIComponent(JSON.stringify(payload))
        });
        const data = await response.json();
        return data;
      } catch (error) {
        Logger.error('AJAX POST error:', error);
        return null;
      }
    },
  },
  proxy: {
    login: async (data) => {
      const response = await API.ajax.post('login', data);
      return response.data;
    },
    signup: async (data) => {
      const response = await API.ajax.post('signup', data);
      return response.data;
    },
    acsbUserLoaded: async (data) => {
      const response = await API.ajax.post('acsb-user-loaded', data);
      return response.data;
    },
    fetchMerchantDetails: async () => {
      const response = await API.ajax.get('get_merchant_details');
      return response.data;
    },
    fetchDomainList: async (domains) => {
      const response = await API.ajax.get('get_domains');
      return response.data;
    },
    sendLicenseData: async (data) => {
      const response = await API.ajax.post('license-trial', data);
      return response.data;
    },
    sendAddVerificationPageRequest: async (data) => {
      const response = await API.ajax.post('add_verification_page', data);
      return response.data;
    },
    logout: async () => {
      const response = await API.ajax.get('logout', {});
      return response.data;
    },
    injectScript: async () => {
      const response = await API.ajax.get('enable_script', { domain: window.location.hostname});
    },
    removeScript: async () => {
      const response = await API.ajax.get('disable_script', { domain: window.location.hostname});
    },
  }
};

window.addEventListener('message', async (event) => {
  if (event.data.eventName) {
    Logger.log(event.data.eventName, event.data);
  }
  switch (event.data.eventName) {
    case 'iframe-ready':
      AcsbStore.isIframeReady = true;
      await API.syncMerchantDetails();
      API.setRedirectUrl(window.location.href);
      break;
    case 'signup': {
      const res = await API.proxy.signup(event.data.data);
      Logger.debug("signup response", res);
      //API.setMerchant(res.data);
      await API.syncMerchantDetails();
      break;
    }
    case 'login': {
      const res = await API.proxy.login(event.data.data);
      Logger.debug("login response", res);
      //API.setMerchant(res.data);
      await API.syncMerchantDetails();
      break;
    }
    case 'acsb-user-loaded': {
      const res = await API.proxy.acsbUserLoaded(event.data.data);
      Logger.debug("acsb-user-loaded response", res);
      //API.setMerchant(res);
      await API.syncMerchantDetails();
      break;
    }
    case 'logout':
      await API.proxy.logout();
      await API.syncMerchantDetails()
      break;
    case 'license-trial': {
      const res = await API.proxy.sendLicenseData({
        licenseId: event.data.data.licenseId,
        accountId: event.data.data.accountId,
        domain: event.data.data.domain,
        widgetConfig: event.data.data.widgetConfig,
        siteId: event.data.data.siteId,
        isNewLicenseTrial: event.data.data.isNewLicenseTrial,
      });
      Logger.debug("license-trial response", res);
      //API.setMerchant(res);
      await API.syncMerchantDetails();
      break;
    }
    case 'domain-list':
      const response = await API.proxy.fetchDomainList(event.data.data.domains);
      API.sendDataToIframe('domains-fetched', response);
      break;
    case 'add-script':
      await API.proxy.injectScript();
      break;
    case 'remove-script':
      await API.proxy.removeScript();
      break;
    case 'redirect-to-url':
      const { add_redirect_url_with_key: redirectUrlKey, url: targetUrl } = event.data.data;
      const urlToRedirect = new URL(targetUrl);
      if (redirectUrlKey) {
        urlToRedirect.searchParams.set(redirectUrlKey, window.location.href);
      }
      window.location.href = urlToRedirect;
      break;
    case 'remove-query-param':
      const paramToremove = event.data.data.param;
      const url = new URL(window.location.href);
      url.searchParams.delete(paramToremove);
      window.history.replaceState({}, document.title, url);
      break;
    case 'create-domain-ownership-request':
      const res = await API.proxy.sendAddVerificationPageRequest(event.data.data);

      API.sendDomainVerificationRequest(res.type, res.path, res.domain);
      Logger.debug("verify-ownership response", res);
      break;
    default:
      break;
  }
});

const isDebugMode = localStorage.getItem('debug') === 'true';

const Logger = {
  log(message, ...optionalParams) {
    if (isDebugMode) {
      console.log(`.[INFO] ${message}`, ...optionalParams);
    }
  },

  error(message, ...optionalParams) {
    if (isDebugMode) {
      console.error(`.[ERROR] ${message}`, ...optionalParams);
    }
  },

  debug(message, ...optionalParams) {
    if (isDebugMode) {
      console.debug(`.[DEBUG] ${message}`, ...optionalParams);
    }
  },
};

// Initialize when page loads
document.addEventListener('DOMContentLoaded', function() {
  Logger.log('Accessibe Joomla admin interface loaded');
  API.setRedirectUrl(window.location.href);
});
</script>