<?php
/**
 * AccessiBe Component Installer Script
 *
 * @copyright   Copyright (C) 2023 - present, AccessiBe Ltd. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Component\ComponentHelper;

// Load component configuration
require_once __DIR__ . '/config.php';

// Load Composer autoloader if available
$composerAutoload = __DIR__ . '/vendor/autoload.php';
if (file_exists($composerAutoload)) {
    require_once $composerAutoload;
}

// Load mixpanel handler
require_once __DIR__ . '/mixpanel.php';

class com_accessibeInstallerScript
{
    public function install($parent)
    {
        error_log('ACCESSIBE COMPONENT: Installed');
        $this->setDefaultParams();
        $this->track_installation();        
        $this->showInstallMessage('install');
        return true;
    }
    
    public function uninstall($parent) 
    {
        error_log('ACCESSIBE COMPONENT: Uninstalled');
        $this->track_uninstall();
        $this->cleanupFiles();
        return true;
    }
    
    public function update($parent)
    {
        error_log('ACCESSIBE COMPONENT: Updated');
        $this->track_upgrade();
        $this->showInstallMessage('update');
        return true;
    }

    /**
     * Get current domain
     */
    private function get_current_domain()
    {
        $domain = $this->removeWWW($_SERVER['HTTP_HOST'] ?? Uri::getInstance()->getHost());
        return $domain;
    }

    /**
     * Remove www from domain
     */
    private function removeWWW($domain) 
    {
        return preg_replace("/^www\./", "", $domain);
    }

    /**
     * Generate UUID for tracking
     */
    private function generateUUID()
    {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }

    /**
     * Track installation
     */
    public function track_installation()
    {
        try {
            $app = Factory::getApplication();
            $user = $app->getIdentity();
            $db = Factory::getDbo();
            
            // Generate UUID and save user metadata
            $uuid = $this->generateUUID();
            $user_metadata = [
                'uuid' => $uuid,
                'install_date' => time(),
                'joomla_version' => JVERSION,
                'php_version' => PHP_VERSION
            ];
            
            // Save to component params
            $params = ComponentHelper::getParams('com_accessibe');
            $currentParams = $params->toArray();
            $currentParams['user_metadata'] = json_encode($user_metadata);
            
            $query = $db->getQuery(true)
                ->update('#__extensions')
                ->set('params = ' . $db->quote(json_encode($currentParams)))
                ->where('element = ' . $db->quote('com_accessibe'))
                ->where('type = ' . $db->quote('component'));
            
            $db->setQuery($query);
            $db->execute();
            
            // Track with Mixpanel
            if (class_exists('JoomlaMixpanelHandler')) {
                $mixpanelHandler = JoomlaMixpanelHandler::getInstance();
                $mixpanelHandler->trackEvent('pluginInstalled', [
                    '$device_id' => $uuid,
                    'primaryDomain' => $this->get_current_domain(),
                    'joomla_id' => $user ? $user->id : 0,
                    'joomla_email' => $user ? $user->email : '',
                    'joomla_username' => $user ? $user->username : '',
                    'joomla_name' => $user ? $user->name : '',
                    'joomla_version' => JVERSION,
                    'php_version' => PHP_VERSION
                ]);
            }
            
            error_log("ACCESSIBE: Installation tracked successfully");
            
        } catch (Exception $e) {
            error_log("ACCESSIBE: Failed to track installation: " . $e->getMessage());
        }
    }

    /**
     * Track upgrade/update
     */
    public function track_upgrade()
    {
        try {
            $app = Factory::getApplication();
            $user = $app->getIdentity();
            $params = ComponentHelper::getParams('com_accessibe');
            $db = Factory::getDbo();
            
            // Get current version from params
            $previousVersion = $params->get('version', 'unknown');
            $newVersion = defined('ACCESSIBE_VERSION') ? ACCESSIBE_VERSION : '1.0.0';
            
            // Skip tracking if no previous version or same version
            if ($previousVersion === 'unknown' || $previousVersion === $newVersion) {
                error_log("ACCESSIBE: Skipping upgrade tracking - previousVersion: {$previousVersion}, newVersion: {$newVersion}");
                return;
            }
            
            // Get stored user metadata
            $user_metadata_json = $params->get('user_metadata', '{}');
            $user_metadata = json_decode($user_metadata_json, true);
            $uuid = $user_metadata['uuid'] ?? '';
            
            if (class_exists('JoomlaMixpanelHandler') && !empty($uuid)) {
                $mixpanelHandler = JoomlaMixpanelHandler::getInstance();
                $mixpanelHandler->trackEvent('pluginUpgraded', [
                    '$device_id' => $uuid,
                    'primaryDomain' => $this->get_current_domain(),
                    'joomla_id' => $user ? $user->id : 0,
                    'joomla_email' => $user ? $user->email : '',
                    'joomla_username' => $user ? $user->username : '',
                    'joomla_name' => $user ? $user->name : '',
                    'joomla_version' => JVERSION,
                    'php_version' => PHP_VERSION,
                    'previousVersion' => $previousVersion,
                    'newVersion' => $newVersion
                ]);
            }
            
            // Update version in params
            $currentParams = $params->toArray();
            $currentParams['version'] = $newVersion;
            
            $query = $db->getQuery(true)
                ->update('#__extensions')
                ->set('params = ' . $db->quote(json_encode($currentParams)))
                ->where('element = ' . $db->quote('com_accessibe'))
                ->where('type = ' . $db->quote('component'));
            
            $db->setQuery($query);
            $db->execute();
            
            error_log("ACCESSIBE: Upgrade tracked successfully - from {$previousVersion} to {$newVersion}");
            
        } catch (Exception $e) {
            error_log("ACCESSIBE: Failed to track upgrade: " . $e->getMessage());
        }
    }

    /**
     * Track uninstall
     */
    public function track_uninstall()
    {
        try {
            $app = Factory::getApplication();
            $user = $app->getIdentity();
            $params = ComponentHelper::getParams('com_accessibe');
            
            // Get stored user metadata
            $user_metadata_json = $params->get('user_metadata', '{}');
            $user_metadata = json_decode($user_metadata_json, true);
            $uuid = $user_metadata['uuid'] ?? '';
            
            if (class_exists('JoomlaMixpanelHandler') && !empty($uuid)) {
                $mixpanelHandler = JoomlaMixpanelHandler::getInstance();
                
                $event_data = [
                    '$device_id' => $uuid,
                    'primaryDomain' => $this->get_current_domain(),
                    'joomla_id' => $user ? $user->id : 0,
                    'joomla_email' => $user ? $user->email : '',
                    'joomla_username' => $user ? $user->username : '',
                    'joomla_name' => $user ? $user->name : '',
                    'joomla_version' => JVERSION,
                    'php_version' => PHP_VERSION,
                    'version' => $params->get('version', 'unknown')
                ];
                
                $mixpanelHandler->trackEvent('pluginUninstalled', $event_data);
            }
            
            error_log("ACCESSIBE: Uninstall tracked successfully");
            
        } catch (Exception $e) {
            error_log("ACCESSIBE: Failed to track uninstall: " . $e->getMessage());
        }
    }

    /**
     * Track deactivation/uninstall
     */
    public function track_deactivation($extra_data = array())
    {
        try {
            $app = Factory::getApplication();
            $user = $app->getIdentity();
            $params = ComponentHelper::getParams('com_accessibe');
            
            // Get stored user metadata
            $user_metadata_json = $params->get('user_metadata', '{}');
            $user_metadata = json_decode($user_metadata_json, true);
            $uuid = $user_metadata['uuid'] ?? '';
            
            if (class_exists('JoomlaMixpanelHandler') && !empty($uuid)) {
                $mixpanelHandler = JoomlaMixpanelHandler::getInstance();
                
                $event_data = array_merge([
                    '$device_id' => $uuid,
                    'primaryDomain' => $this->get_current_domain(),
                    'joomla_id' => $user ? $user->id : 0,
                    'joomla_email' => $user ? $user->email : '',
                    'joomla_username' => $user ? $user->username : '',
                    'joomla_name' => $user ? $user->name : '',
                    'joomla_version' => JVERSION,
                    'php_version' => PHP_VERSION
                ], $extra_data);
                
                $mixpanelHandler->trackEvent('pluginDeactivated', $event_data);
            }
            
            error_log("ACCESSIBE: Deactivation tracked successfully");
            
        } catch (Exception $e) {
            error_log("ACCESSIBE: Failed to track deactivation: " . $e->getMessage());
        }
    }
    
    /**
     * Track when plugin/component is manually disabled (not uninstalled)
     */
    public static function trackManualDisable($deactivation_type = 'manual_disable')
    {
        try {
            $params = ComponentHelper::getParams('com_accessibe');
            $user_metadata_json = $params->get('user_metadata', '{}');
            $user_metadata = json_decode($user_metadata_json, true);
            $uuid = $user_metadata['uuid'] ?? '';
            
            if (class_exists('JoomlaMixpanelHandler') && !empty($uuid)) {
                $app = Factory::getApplication();
                $user = $app->getIdentity();
                
                $mixpanelHandler = JoomlaMixpanelHandler::getInstance();
                $mixpanelHandler->trackEvent('pluginDeactivated', [
                    '$device_id' => $uuid,
                    'primaryDomain' => $_SERVER['HTTP_HOST'] ?? '',
                    'joomla_id' => $user ? $user->id : 0,
                    'joomla_email' => $user ? $user->email : '',
                    'deactivation_type' => $deactivation_type,
                    'timestamp' => date('Y-m-d H:i:s')
                ]);
            }
            
        } catch (Exception $e) {
            error_log("ACCESSIBE: Failed to track manual disable: " . $e->getMessage());
        }
    }

    /**
     * Set default component parameters
     */
    private function setDefaultParams()
    {
        try {
            $db = Factory::getDbo();
            
            $defaultParams = [
                'license_id' => '',
                'domain_settings' => '{}',
                'custom_domains' => '[]',
                'version' => ACCESSIBE_VERSION,
                'installed_date' => date('Y-m-d H:i:s')
            ];
            
            $params = json_encode($defaultParams);
            
            $query = $db->getQuery(true)
                ->update('#__extensions')
                ->set('params = ' . $db->quote($params))
                ->where('element = ' . $db->quote('com_accessibe'))
                ->where('type = ' . $db->quote('component'));
            
            $db->setQuery($query);
            $db->execute();
            
            error_log("ACCESSIBE: Default parameters set");
            
        } catch (Exception $e) {
            error_log("ACCESSIBE: Failed to set default parameters: " . $e->getMessage());
        }
    }

    /**
     * Clean up files on uninstall
     */
    private function cleanupFiles()
    {
        try {
            // Remove verification files
            $filesToRemove = [
                JPATH_ROOT . '/accessibe_verification.txt',
                JPATH_ROOT . '/tmp/accessibe_verification.txt'
            ];
            
            foreach ($filesToRemove as $file) {
                if (file_exists($file)) {
                    @unlink($file);
                    error_log("ACCESSIBE: Removed file: $file");
                }
            }
            
        } catch (Exception $e) {
            error_log("ACCESSIBE: Failed to cleanup files: " . $e->getMessage());
        }
    }

    /**
     * Show installation message
     */
    private function showInstallMessage($event)
    {
        $messages = [
            'install' => [
                'title' => 'Accessibe Component Installed Successfully!',
                'message' => 'The Accessibe accessibility widget component has been installed.',
                'class' => 'alert-success'
            ],
            'update' => [
                'title' => 'Accessibe Component Updated Successfully!',
                'message' => 'The Accessibe component has been updated to the latest version.',
                'class' => 'alert-info'
            ]
        ];

        if (isset($messages[$event])) {
            $msg = $messages[$event];
            echo '<div class="alert ' . $msg['class'] . '">';
            echo '<h3>' . $msg['title'] . '</h3>';
            echo '<p>' . $msg['message'] . '</p>';
            if ($event === 'install') {
                echo '<p><a href="index.php?option=com_accessibe" class="btn btn-primary">Configure Accessibe</a></p>';
            }
            echo '</div>';
        }
    }
}