<?php
/**
 * AccessiBe Plugin
 *
 * @copyright   Copyright (C) 2023 - present, AccessiBe Ltd. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Factory;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Uri\Uri;

class PlgSystemAccessibe extends CMSPlugin
{
    /**
     * Track installation event
     */
    public function onExtensionAfterInstall($installer, $eid)
    {
        // Simple logging without accessing installer methods that might fail
        error_log('ACCESSIBE: Plugin installed - Extension ID: ' . $eid);
        return true;
    }
    
    /**
     * Track uninstallation event
     */
    public function onExtensionBeforeUninstall($installer)
    {
        // Simple logging without accessing installer methods that might fail
        error_log('ACCESSIBE: Plugin being uninstalled');
        return true;
    }
    
    public function onBeforeCompileHead()
    {
        // Force write to error log
        error_log('=== ACCESSIBE PLUGIN: onBeforeCompileHead CALLED ===');
        
        $app = Factory::getApplication();
        error_log('ACCESSIBE: App client: ' . $app->getName());
        
        // Only on frontend
        if (!$app->isClient('site')) {
            error_log('ACCESSIBE: Not on site, exiting');
            return;
        }

        error_log('ACCESSIBE: On site frontend, continuing...');

        // Get component settings
        try {
            $params = ComponentHelper::getParams('com_accessibe');
            error_log('ACCESSIBE: Got component params: ' . $params->toString());
            
            // Get current domain
            $uri = \Joomla\CMS\Uri\Uri::getInstance();
            $currentDomain = $this->get_current_domain();
            
            // Get domain-specific settings
            $domainSettings = json_decode($params->get('domain_settings', '{}'), true);
            if (!is_array($domainSettings)) {
                $domainSettings = [];
            }
        
            $enabled = isset($domainSettings[$currentDomain]) ? (int)$domainSettings[$currentDomain]['widgetStatus'] : 0;
            $url = 'https://acsbapp.com/apps/app/dist/js/app.js';
            
            error_log('ACCESSIBE: Domain=' . $currentDomain . ', Enabled=' . $enabled . ', URL=' . $url);
            
            if ($enabled) {
                // Add BigCommerce-style inline script
                $scriptName = 'accessibe-joomla-loader';
                $script = "<script id='{$scriptName}'> (function(){ var s = document.createElement('script'); var h = document.querySelector('head') || document.body; s.src = 'https://acsbapp.com/apps/app/dist/js/app.js'; s.setAttribute('data-source', 'Joomla'); s.async = true; s.onload = function(){ acsbJS.init({}); }; h.appendChild(s); })(); </script>";
                
                $doc = Factory::getDocument();
                $doc->addCustomTag($script);
                // Also force HTML comment for testing
            } else {
                error_log('ACCESSIBE: Script NOT added for domain ' . $currentDomain . ' - Enabled=' . $enabled);
            }
        } catch (Exception $e) {
            error_log('ACCESSIBE ERROR: ' . $e->getMessage());
            echo '<!-- ACCESSIBE ERROR: ' . $e->getMessage() . ' -->';
        }
    }

    // Sanitize domain name - remove www, http/https, ports for local development
    function sanitizeDomain($domain) {
        // Remove protocol if present
        $domain = preg_replace('/^https?:\/\//i', '', $domain);
        
        // Remove www prefix
        $domain = preg_replace('/^www\./i', '', $domain);
        
        // Remove trailing slash
        $domain = rtrim($domain, '/');
        
        // For local development, keep ports but clean format
        if (strpos($domain, 'localhost') === 0 || strpos($domain, '127.0.0.1') === 0) {
            return $domain;
        }
        
        // Remove port for production domains
        $domain = preg_replace('/:.*$/', '', $domain);
        
        return strtolower(trim($domain));
    }

    function get_current_domain() {
        return $this->sanitizeDomain(\Joomla\CMS\Uri\Uri::getInstance()->getHost());
    } 
}