<?php
/**
 * AccessiBe Plugin Installer
 *
 * @copyright   Copyright (C) 2023 - present, AccessiBe Ltd. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;

class PlgSystemAccessibeInstallerScript
{
    /**
     * Called after successful installation
     */
    public function install($parent)
    {
        $this->enablePlugin();
        $this->disableConflictingPlugins();
        error_log('ACCESSIBE: Installation completed and plugin enabled');
        return true;
    }
    
    /**
     * Called after successful update
     */
    public function update($parent)
    {
        $this->enablePlugin();
        $this->disableConflictingPlugins();
        error_log('ACCESSIBE: Update completed and plugin enabled');
        return true;
    }
    
    /**
     * Called before uninstallation
     */
    public function uninstall($parent)
    {
        error_log('ACCESSIBE: Plugin uninstalled');
        return true;
    }
    
    /**
     * Enable the plugin after installation
     */
    private function enablePlugin()
    {
        try {
            $db = Factory::getDbo();
            $query = $db->getQuery(true)
                ->update($db->quoteName('#__extensions'))
                ->set($db->quoteName('enabled') . ' = 1')
                ->where($db->quoteName('element') . ' = ' . $db->quote('accessibe'))
                ->where($db->quoteName('folder') . ' = ' . $db->quote('system'))
                ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'));
            
            $db->setQuery($query);
            $result = $db->execute();
            
            if ($result) {
                error_log('ACCESSIBE: Plugin enabled successfully');
            } else {
                error_log('ACCESSIBE: Failed to enable plugin');
            }
        } catch (Exception $e) {
            error_log('ACCESSIBE: Error enabling plugin - ' . $e->getMessage());
        }
    }

    private function disableConflictingPlugins()
    {
        $conflictingPlugins = [
            'httpheaders' // Joomla's HTTP Headers plugin blocks OAuth with X-Frame-Options
        ];
        
        try {
            $db = Factory::getDbo();
            
            foreach ($conflictingPlugins as $pluginElement) {
                // Check if plugin exists and is enabled
                $checkQuery = $db->getQuery(true)
                    ->select($db->quoteName(['name', 'enabled']))
                    ->from($db->quoteName('#__extensions'))
                    ->where($db->quoteName('element') . ' = ' . $db->quote($pluginElement))
                    ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'));
                
                $db->setQuery($checkQuery);
                $plugin = $db->loadObject();
                
                if ($plugin && $plugin->enabled) {
                    // Store original state for potential restoration
                    error_log('ACCESSIBE: Found conflicting plugin "' . $plugin->name . '" - disabling for OAuth compatibility');
                    
                    // Disable the conflicting plugin
                    $disableQuery = $db->getQuery(true)
                        ->update($db->quoteName('#__extensions'))
                        ->set($db->quoteName('enabled') . ' = 0')
                        ->where($db->quoteName('element') . ' = ' . $db->quote($pluginElement))
                        ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'));
                    
                    $db->setQuery($disableQuery);
                    $result = $db->execute();
                    
                    if ($result) {
                        error_log('ACCESSIBE: Successfully disabled conflicting plugin: ' . $plugin->name);
                    } else {
                        error_log('ACCESSIBE: Failed to disable conflicting plugin: ' . $plugin->name);
                    }
                } elseif ($plugin) {
                    error_log('ACCESSIBE: Plugin "' . $plugin->name . '" already disabled');
                } else {
                    error_log('ACCESSIBE: Plugin "' . $pluginElement . '" not found - no conflict');
                }
            }
        } catch (Exception $e) {
            error_log('ACCESSIBE: Error managing conflicting plugins - ' . $e->getMessage());
        }
    }
}