CREATE TABLE netdevs(
	ip		VARCHAR(256)
			PRIMARY KEY,

	dev_type	INT4
			NOT NULL,

	name		VARCHAR(256)
			NOT NULL,

	community	VARCHAR(256)
			NOT NULL
);

CREATE TABLE interfaces(
	netdev_ip	VARCHAR(256)
			NOT NULL
			REFERENCES netdevs(ip)
				ON DELETE CASCADE
				ON UPDATE CASCADE,

	ifindex		INT4
			NOT NULL,

	ifinternal	INT4,

	description	VARCHAR(256),

	alias		VARCHAR(256),

	has_netdev	INT4,

	PRIMARY KEY(netdev_ip, ifindex)
);

CREATE TABLE macs(
	mac		VARCHAR(256)
			NOT NULL,

	netdev_ip	VARCHAR(256)
			NOT NULL,

	ifindex		INT4
			NOT NULL,

	PRIMARY KEY (mac,netdev_ip),

	FOREIGN KEY (netdev_ip, ifindex)
		REFERENCES interfaces(netdev_ip, ifindex)
			ON DELETE CASCADE
			ON UPDATE CASCADE
);

CREATE TABLE ips(
	ip		VARCHAR(256)
			NOT NULL,

	mac		VARCHAR(256)
			NOT NULL,

	netdev_ip	VARCHAR(256)
			NOT NULL,

	ifindex		INT4
			NOT NULL,

	PRIMARY KEY(ip,mac,netdev_ip),

	FOREIGN KEY(netdev_ip,ifindex)
		REFERENCES interfaces(netdev_ip, ifindex)
			ON DELETE CASCADE
			ON UPDATE CASCADE
);

CREATE TABLE vendors(
	prefix		VARCHAR(50)
			PRIMARY KEY
			NOT NULL,

	name		VARCHAR(256)
);

CREATE VIEW view_macs AS
	SELECT	macs.mac		AS mac,
		macs.netdev_ip		AS sw_ip,
		macs.ifindex		AS sw_ifindex,
		netdevs.name		AS sw_name,
		interfaces.description	AS sw_ifdescr,
		interfaces.alias	AS sw_ifalias

		FROM macs, interfaces, netdevs

		WHERE	macs.netdev_ip = netdevs.ip AND
			macs.netdev_ip = interfaces.netdev_ip AND
			macs.ifindex = interfaces.ifindex;

CREATE VIEW view_ips AS
	SELECT	ips.ip			AS ip,
		ips.mac			AS mac,
		netdevs.name		AS router_name,
		netdevs.ip		AS router_ip,
		interfaces.ifindex	AS router_ifindex,
		interfaces.description	AS router_ifdescr,
		interfaces.alias	AS router_ifalias

		FROM ips, netdevs, interfaces

		WHERE	ips.netdev_ip = netdevs.ip AND
			interfaces.netdev_ip = netdevs.ip AND
			ips.ifindex = interfaces.ifindex;

CREATE VIEW view_all_pre AS
	SELECT	i.ip			AS ip,
		mac			AS mac,
		i.router_ip		AS router_ip,
		i.router_name		AS router_name,
		i.router_ifindex	AS router_ifindex,
		i.router_ifdescr	AS router_ifdescr,
		i.router_ifalias	AS router_ifalias,
		m.sw_ip			AS sw_ip,
		m.sw_name		AS sw_name,
		m.sw_ifindex		AS sw_ifindex,
		m.sw_ifdescr		AS sw_ifdescr,
		m.sw_ifalias		AS sw_ifalias

		FROM view_ips i FULL JOIN view_macs m
			USING(mac);

CREATE VIEW view_all AS
	SELECT ip, mac, router_ip, router_name, router_ifindex, router_ifdescr,
		router_ifalias, sw_ip, sw_name, sw_ifindex, sw_ifdescr, sw_ifalias,
		vendors.name AS mac_name
		FROM view_all_pre LEFT OUTER JOIN vendors
			ON vendors.prefix = substr(mac,1,8);

CREATE SEQUENCE seq_history;

CREATE TABLE history(
	id		INT4
			PRIMARY KEY
			DEFAULT NEXTVAL('seq_history'),

	ts		INT8
			NOT NULL
			DEFAULT EXTRACT(EPOCH FROM CURRENT_TIMESTAMP),

	last_ts		INT8
			NOT NULL
			DEFAULT EXTRACT(EPOCH FROM CURRENT_TIMESTAMP),

	ip		VARCHAR(256)
			NOT NULL,

	mac		VARCHAR(256)
			NOT NULL,

	router_ip	VARCHAR(256),
	router_name	VARCHAR(256),
	router_ifindex	INT4,
	router_ifdescr	VARCHAR(256),
	router_ifalias	VARCHAR(256),
	sw_ip		VARCHAR(256),
	sw_name		VARCHAR(256),
	sw_ifindex	INT4,
	sw_ifdescr	VARCHAR(256),
	sw_ifalias	VARCHAR(256)

--	PRIMARY KEY(ts,ip,mac)
);

CREATE TABLE history_state(
	hist_id		INT4
			PRIMARY KEY
			REFERENCES history(id)
				ON DELETE CASCADE
				ON UPDATE CASCADE,

	last_ignore	INT8,

	description	TEXT
);

CREATE VIEW view_history AS
	SELECT id, ts, last_ts, ip, mac, router_ip, router_name, router_ifindex,
		router_ifdescr, router_ifalias, sw_ip, sw_name, sw_ifindex, sw_ifdescr,
		sw_ifalias, vendors.name AS mac_name
		FROM history LEFT OUTER JOIN vendors
			ON vendors.prefix = substr(mac,1,8);

CREATE SEQUENCE seq_history_ignore;
CREATE TABLE history_ignore(
	id		INT4
			PRIMARY KEY
			DEFAULT NEXTVAL('seq_history_ignore'),

	ip		VARCHAR(256),

	mac		VARCHAR(256),

	router_name	VARCHAR(256),

	router_ifindex	VARCHAR(256),

	sw_name		VARCHAR(256),

	sw_ifindex	VARCHAR(256)
);

CREATE VIEW view_history_not_ignored AS
	SELECT * FROM view_history h 
		WHERE NOT EXISTS
			( SELECT * FROM history_ignore i
				WHERE ( i.router_name IS NULL OR i.router_name='' OR i.router_name=h.router_name ) AND
					( i.router_ifindex IS NULL OR i.router_ifindex='' OR i.router_ifindex=h.router_ifindex ) AND
					( i.sw_name IS NULL OR i.sw_name='' OR i.sw_name=h.sw_name ) AND
					( i.sw_ifindex IS NULL OR i.sw_ifindex='' OR i.sw_ifindex=h.sw_ifindex ) AND
					( i.ip IS NULL OR i.ip='' OR i.ip=substring(h.ip from 1 for octet_length(i.ip))) AND
					( i.mac IS NULL OR i.mac='' OR i.mac=substring(h.mac from 1 for octet_length(i.mac))) );

-- Same as view_history_not_ignore but don't include
-- entries with state ignored
CREATE VIEW view_history_not_ignore_state AS
	SELECT * FROM view_history_not_ignored h LEFT JOIN history_state hs
			ON h.id=hs.hist_id
		WHERE NOT EXISTS 
			( SELECT * FROM history_state hs
				WHERE h.id = hs.hist_id AND
					h.last_ts <= hs.last_ignore );

CREATE VIEW view_stated AS
	SELECT * FROM view_history_not_ignored h JOIN history_state hs
			ON  h.id=hs.hist_id;

CREATE TABLE	reg_persons (
	id		VARCHAR(256)
			PRIMARY KEY,

	fname		VARCHAR(256)
			NOT NULL,

	sname		VARCHAR(256)
			NOT NULL,

	description	TEXT
);

CREATE TABLE	reg_ips (
	ip		VARCHAR(256)
			PRIMARY KEY,

	reged_macs	INTEGER
			DEFAULT '0',

	personid	VARCHAR(256)
			REFERENCES reg_persons(id)
				ON UPDATE CASCADE
				ON DELETE SET NULL,

	shortdesc	VARCHAR(256),

	description	TEXT
);

CREATE TABLE	reg_macs (
	mac		VARCHAR(256)
			PRIMARY KEY,

	reged_ips	INTEGER
			DEFAULT '0',

	personid	VARCHAR(256)
			REFERENCES reg_persons(id)
				ON UPDATE CASCADE
				ON DELETE SET NULL,

	shortdesc	VARCHAR(256),

	description	TEXT
);

CREATE TABLE	reg_ip2mac (
	ip		VARCHAR(256)
--			REFERENCES reg_ips(ip)
--				ON DELETE CASCADE
--				ON UPDATE CASCADE
			NOT NULL,

	mac		VARCHAR(256)
--			REFERENCES reg_macs(mac)
--				ON DELETE CASCADE
--				ON UPDATE CASCADE
			NOT NULL,

	PRIMARY KEY (ip,mac)
);

--
-- ip,mac
--   +--- not exist -> INSERT
--   |
--   +--- exist
--          +--- old sw_ip is NULL -> UPDATE (delete & insert)
--          |
--          +--- old sw_ip is NOT NULL
--                 +-- new sw_ip is NULL -> do nothing
--                 |
--                 +-- new sw_ip is NOT NULL
--                       +-- old sw_ip == new sw_ip AND
--                       |   old sw_ifindex == new sw_ifindex -> UPDATE last_ts
--                       |
--                       +-- old sw_ip != new sw_ip OR
--                           old sw_ifindex != new sw_ifindex -> INSERT
--
--
--
--
--
--
--
--

CREATE FUNCTION set_history()
	RETURNS INT4
	AS '
-- Delete old entries with unknown switch ip that are
-- now known
		DELETE FROM history
			WHERE history.sw_ip IS NULL AND
				EXISTS ( SELECT * FROM view_all v
						WHERE history.ip = v.ip AND
							history.mac = v.mac AND
							v.sw_ip IS NOT NULL );
-- Instert New entries
		INSERT INTO history(ip,mac,router_ip,router_name,router_ifindex,router_ifdescr,router_ifalias,sw_ip,
				sw_name,sw_ifindex,sw_ifdescr,sw_ifalias)
		SELECT ip,mac,router_ip,router_name,router_ifindex,router_ifdescr,router_ifalias,
			sw_ip,sw_name,sw_ifindex,sw_ifdescr,sw_ifalias
			FROM view_all v
			WHERE ip IS NOT NULL AND
				mac IS NOT NULL AND
				( NOT EXISTS (
					SELECT * FROM history h
						WHERE h.ip = v.ip AND
							h.mac = v.mac ) OR

				( EXISTS ( SELECT * FROM history h
						WHERE h.ip = v.ip AND
							h.mac = v.mac AND
							h.sw_ip IS NOT NULL AND
							( h.sw_ip <> v.sw_ip OR
							h.sw_ifindex <> v.sw_ifindex ) AND
							v.sw_ip IS NOT NULL ) AND
					NOT EXISTS ( SELECT * FROM history h
							WHERE h.ip = v.ip AND
								h.mac = v.mac AND
								h.sw_ip IS NOT NULL AND
								h.sw_ip = v.sw_ip AND
								h.sw_ifindex = v.sw_ifindex AND
								v.sw_ip IS NOT NULL ) ) );
-- Update existing entries
		UPDATE history SET last_ts = EXTRACT(EPOCH FROM CURRENT_TIMESTAMP)
			WHERE history.sw_ip IS NOT NULL AND
				EXISTS ( SELECT * FROM view_all v
						WHERE v.ip = history.ip AND
							v.mac = history.mac AND
							v.sw_ip = history.sw_ip AND
							v.sw_ifindex = history.sw_ifindex );

		SELECT 1;
	'
	LANGUAGE SQL;

-- Remove data that are older than a week
CREATE FUNCTION expire_history()
	RETURNS INT4
	AS '
		DELETE FROM history
			WHERE last_ts < ( EXTRACT(EPOCH FROM CURRENT_TIMESTAMP) - 604800 );

		SELECT 1;
	'
	LANGUAGE SQL;

-- Remove reg_ip2mac entries that have no reged ip and no reged mac
CREATE OR REPLACE FUNCTION fix_ip2mac_ip_del()
	RETURNS OPAQUE
	AS '
		BEGIN
		DELETE FROM reg_ip2mac
			WHERE	reg_ip2mac.ip=OLD.ip AND
				NOT EXISTS (
					SELECT * FROM reg_macs m
						WHERE reg_ip2mac.mac=m.mac );

		RETURN OLD;
		END
	'
	LANGUAGE 'plpgsql';
CREATE OR REPLACE FUNCTION fix_ip2mac_mac_del()
	RETURNS OPAQUE
	AS '
		BEGIN
		DELETE FROM reg_ip2mac
			WHERE	reg_ip2mac.mac=OLD.mac AND
				NOT EXISTS (
					SELECT * FROM reg_ips i
						WHERE reg_ip2mac.ip=i.ip );

		RETURN OLD;
		END
	'
	LANGUAGE 'plpgsql';

-- Generic fix - not called by a trigger
CREATE OR REPLACE FUNCTION fix_ip2mac()
	RETURNS INTEGER
	AS '
		BEGIN
		DELETE FROM reg_ip2mac
			WHERE	NOT EXISTS (
					SELECT * FROM reg_ips i
						WHERE reg_ip2mac.ip=i.ip ) AND
				NOT EXISTS (
					SELECT * FROM reg_macs m
						WHERE reg_ip2mac.mac=m.mac );

		RETURN 1;
		END
	'
	LANGUAGE 'plpgsql';


-- When updating reg_ips or reg_macs also update reg_ip2mac
CREATE OR REPLACE FUNCTION fix_ip2mac_ip_upd()
	RETURNS OPAQUE
	AS '
		BEGIN
		UPDATE reg_ip2mac
			SET ip=NEW.ip
			WHERE ip=OLD.ip;

		RETURN NEW;
		END
	'
	LANGUAGE 'plpgsql';
CREATE OR REPLACE FUNCTION fix_ip2mac_mac_upd()
	RETURNS OPAQUE
	AS '
		BEGIN
		UPDATE reg_ip2mac
			SET mac=NEW.mac
			WHERE mac=OLD.mac;

		RETURN NEW;
		END
	'
	LANGUAGE 'plpgsql';


-- Auto run the above function
CREATE TRIGGER trig_fix_del AFTER DELETE
	ON reg_ips FOR EACH ROW
	EXECUTE PROCEDURE fix_ip2mac_ip_del();

CREATE TRIGGER trig_fix_del AFTER DELETE
	ON reg_macs FOR EACH ROW
	EXECUTE PROCEDURE fix_ip2mac_mac_del();

CREATE TRIGGER trig_fix_upd AFTER UPDATE
	ON reg_ips FOR EACH ROW
	EXECUTE PROCEDURE fix_ip2mac_ip_upd();
	
CREATE TRIGGER trig_fix_upd AFTER UPDATE
	ON reg_macs FOR EACH ROW
	EXECUTE PROCEDURE fix_ip2mac_mac_upd();

--CREATE VIEW view_hist AS
--	SELECT ip, mac, sw_ip, sw_ifindex, ts, last_ts FROM history;

-- position change
CREATE VIEW view_hist_ip_mac_cnt AS
	SELECT ip, mac, count(*) FROM
		( SELECT ip, mac, last_ts FROM view_history_not_ignored
			GROUP BY ip, mac, last_ts ) lala
		GROUP BY ip,mac;
-- mac change
CREATE VIEW view_hist_ip_cnt AS
	SELECT ip, count(*) FROM
			( SELECT DISTINCT ip, mac
				FROM view_history_not_ignored ) koko
		GROUP BY ip;

-- ip change
CREATE VIEW view_hist_mac_cnt AS
	SELECT mac, count(*) FROM
			( SELECT DISTINCT ip, mac
				FROM view_history_not_ignored ) koko
		GROUP BY mac;

-- position change (without ignored)
CREATE VIEW view_hist_ip_mac_cnt_state AS
	SELECT ip, mac, count(*) FROM
		( SELECT ip, mac, last_ts FROM view_history_not_ignore_state
			GROUP BY ip, mac, last_ts ) lala
		GROUP BY ip,mac;
-- mac change (without ignored)
CREATE VIEW view_hist_ip_cnt_state AS
	SELECT ip, count(*) FROM
			( SELECT DISTINCT ip, mac
				FROM view_history_not_ignore_state ) koko
		GROUP BY ip;

-- ip change (without ignored)
CREATE VIEW view_hist_mac_cnt_state AS
	SELECT mac, count(*) FROM
			( SELECT DISTINCT ip, mac
				FROM view_history_not_ignore_state ) koko
		GROUP BY mac;

-- view_all with ip and mac registries
CREATE VIEW view_all_reg AS
	SELECT ip, a.mac, router_ip, router_name, router_ifindex, router_ifdescr,
		router_ifalias, sw_ip, sw_name, sw_ifindex, sw_ifdescr, sw_ifalias,
		mac_name, reg_ip_ip, ip_personid, reged_macs, ip_shortdesc,
		reg_macs.mac AS reg_mac_mac, reg_macs.personid AS mac_personid,
		reged_ips, reg_macs.shortdesc AS mac_shortdesc FROM
			( SELECT view_all.ip, mac, router_ip, router_name, router_ifindex, router_ifdescr,
				router_ifalias, sw_ip, sw_name, sw_ifindex, sw_ifdescr, sw_ifalias,
				mac_name, personid AS ip_personid, reged_macs, shortdesc AS ip_shortdesc,
				reg_ips.ip AS reg_ip_ip
					FROM view_all LEFT JOIN reg_ips
						ON view_all.ip=reg_ips.ip ) AS a LEFT JOIN reg_macs
							ON a.mac=reg_macs.mac;

CREATE VIEW view_history_reg AS
	SELECT id, ts, last_ts, ip, a.mac, router_ip, router_name, router_ifindex,
		router_ifdescr, router_ifalias, sw_ip, sw_name, sw_ifindex, sw_ifdescr,
		sw_ifalias, mac_name, reg_ip_ip, ip_personid, reged_macs, ip_shortdesc,
		reg_macs.mac AS reg_mac_mac, reg_macs.personid AS mac_personid,
		reged_ips, reg_macs.shortdesc AS mac_shortdesc FROM
			( SELECT id, ts, last_ts, view_history.ip, mac, router_ip, router_name, router_ifindex,
				router_ifdescr, router_ifalias, sw_ip, sw_name, sw_ifindex, sw_ifdescr,
				sw_ifalias, mac_name, personid AS ip_personid, reged_macs,
				shortdesc AS ip_shortdesc, reg_ips.ip AS reg_ip_ip
					FROM view_history LEFT JOIN reg_ips
						ON view_history.ip=reg_ips.ip ) AS a LEFT JOIN reg_macs
							ON a.mac=reg_macs.mac;

