/****************************************/
/*					*/
/****************************************/
/* Start      : 12/10/2002		*/
/* Version    : 1.00			*/
/* Last Update: 12/10/2002		*/
/****************************************/

/*
  Copyright (c) 2002 Stefanos Harhalakis

  This file is part of netinfo.

  netinfo is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  netinfo is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with netinfo; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "snmp.h"

#include "v/misc/keytable2.cc"
#include "v/misc/nkeytable.cc"

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#include "conf.h"
#include "db.h"


#define	GET_IFDESCR_MIB		"IF-MIB::ifDescr"
#define GET_IFALIAS_MIB		"mib-2.31.1.1.1.18"	// if alias
#define GET_IFINTERNAL_MIB	"mib-2.17.1.4.1.2"
#define GET_TPFDBSTATUS_MIB	"mib-2.17.4.3.1.3"	// mac-address type (learned etc..)
#define GET_TPFDBPORT_MIB	"mib-2.17.4.3.1.2"	// mac-address port

#define GET_IFNETTOMEDIA_MIB	"IP-MIB::ipNetToMediaPhysAddress"	// ip to mac
#define GET_IFPHYSADDRESS_MIB	"IF-MIB::ifPhysAddress"	// interfaces MACs

KEYTABLE<DEVICE_DATA *>	devices;
STRLIST			netdev_macs;		// switch mac addresses
STRLIST			router_macs;		// router interface mac addresses

void	snmp_init_device_data()
{
	DEVICE_DATA	*data;
	NETDEV		*pndev;
	LONGLINE	*pll;
	
	netdevs.Rewind();
	while ((pll=netdevs.Get_Next())!=NULL)
	{
		pndev=netdevs[*pll];
		printf("%s:%s:%d\n",pll->p, pndev->community.p, pndev->type);
		
		data=new DEVICE_DATA;
		switch(pndev->type)
		{
			case 2:
				data->type=l2_switch;
				data->l2_switch.status=0;
				break;
			case 1:
				data->type=router;
				data->router.status=0;
				break;
				
		}
		data->addr = *pll;
		data->community=pndev->community;
			
		devices.Add(pll->p, data);
	}
}

int snmp_response_l2_switch(int operation, struct snmp_session *sp, int reqid,
			struct snmp_pdu *pdu, void *magic)
{
	DEVICE_DATA	*pdata = (DEVICE_DATA *)magic;
	snmp_pdu	*req;
	variable_list	*vars;
	int		done=0;

	if (pdata->l2_switch.status>=100)
		return(0);
	
//	printf("Response ");
	if (operation != NETSNMP_CALLBACK_OP_RECEIVED_MESSAGE)
	{
		pdata->l2_switch.status=-1;
	} else if (pdu==NULL)
	{
		pdata->l2_switch.status=-1;
	} else if (pdu->errstat!=SNMP_ERR_NOERROR)
	{
		pdata->l2_switch.status=-1;
	} else
	{
		for (vars=pdu->variables ; ( vars && !done) ; vars=vars->next_variable)
		{
			if (vars->name_length < pdata->root_oid_len ||
				memcmp(vars->name, pdata->root_oid, pdata->root_oid_len * sizeof(oid)))
			{
				printf("%s(%d): end of data (%d-%d)\n",
								pdata->addr.p, 
								pdata->l2_switch.status,
								vars->name_length,
								pdata->root_oid_len);
				pdata->l2_switch.status++;
				done=1;
			} else if (snmp_oid_compare(pdata->cur_oid, pdata->cur_oid_len,
					vars->name, vars->name_length) >=0)
			{
				printf("%s(%d): OID not increasing!\n", pdata->addr.p, pdata->l2_switch.status);
				pdata->l2_switch.status++;
				done=1;
			} else
			{
				switch(pdata->l2_switch.status)
				{
					case 1:
						if (vars->type == ASN_OCTET_STR)
						{
							int		n;
							char		*p;
							L2S_IFACE	iface;
							
							n=pdata->root_oid_len;

							p=(char *)malloc(vars->val_len+1);
							memcpy(p,vars->val.string,vars->val_len);
							p[vars->val_len]=0;

							printf("Interface: %d - %s\n", vars->name[pdata->root_oid_len], p);

							// Add the interface
							iface.internal=-1;
//							iface.ifindex=pdata->cur_oid[pdata->root_oid_len]+1;
							iface.ifindex=vars->name[pdata->root_oid_len];
							iface.descr=p;
							iface.has_netdev=0;
							pdata->l2_switch.ifaces.Add(iface.ifindex,iface);

							free(p);
						} else
						{
							printf("(1)Returned non-string\n");
							print_variable(vars->name, vars->name_length, vars);
							pdata->l2_switch.status=-1;
							done=1;
						}
						break;
					case 3:
						if (vars->type == ASN_OCTET_STR)
						{
							int		n;
							char		*p;
							int		ifindex;
							L2S_IFACE	*piface;

							ifindex=vars->name[pdata->root_oid_len];
							piface=pdata->l2_switch.ifaces[ifindex];

							if (piface==NULL)
							{
								printf("Ignoring unknown interface: index=%d\n",ifindex);
							} else
							{
//								print_variable(vars->name, vars->name_length, vars);
								p=(char *)malloc(vars->val_len+1);
								memcpy(p,vars->val.string,vars->val_len);
								p[vars->val_len]=0;
								piface->alias=p;
								printf("Interface-alias: %d - %s\n", vars->name[pdata->root_oid_len], p);
								free(p);
							}
						} else
						{
							printf("(1)Returned non-string\n");
							print_variable(vars->name, vars->name_length, vars);
							pdata->l2_switch.status=-1;
							done=1;
						}
						break;
					case 5:
						if (vars->type == ASN_INTEGER)
						{
							int		n;
							L2S_IFACE	*piface;
							long		ifindex, ifinternal;

							ifindex=*vars->val.integer;	// TODO: Check for NULL
//							ifinternal=pdata->cur_oid[pdata->root_oid_len]+1;
							ifinternal=vars->name[pdata->root_oid_len];
							
							piface=pdata->l2_switch.ifaces[ifindex];
							if (piface==NULL)
							{
								printf("Ignoring unknown interface: index=%d  internal=%d\n",ifindex, ifinternal);
							} else
							{
								if (piface->internal!=-1)
								{
									printf("Ignoring duplicate entry for interface: index=%d  internal=%d\n",ifindex, ifinternal);
								} else
								{
//									print_variable(vars->name, vars->name_length, vars);
									printf("Interface: index=%d  internal=%d\n",ifindex, ifinternal);
									piface->internal=ifinternal;
								}
							}
						} else
						{
							printf("(3)Returned non-integer\n");
							print_variable(vars->name, vars->name_length, vars);
							pdata->l2_switch.status=-1;
							done=1;
						}
						break;
					case 7:
						if (vars->type == ASN_INTEGER)
						{
							int		n;
							long		type;
							char		tmp[100];

							type=*vars->val.integer;	// TODO: Check for NULL
							sprintf(tmp,"%02X:%02X:%02X:%02X:%02X:%02X",
									vars->name[pdata->root_oid_len],
									vars->name[pdata->root_oid_len+1],
									vars->name[pdata->root_oid_len+2],
									vars->name[pdata->root_oid_len+3],
									vars->name[pdata->root_oid_len+4],
									vars->name[pdata->root_oid_len+5]);
									
							printf("MAC: %s  type: %d\n", tmp, type);
							pdata->l2_switch.macs_tmp.Add(tmp,type);
						} else
						{
							printf("(5)Returned non-integer\n");
							print_variable(vars->name, vars->name_length, vars);
							pdata->l2_switch.status=-1;
							done=1;
						}
						break;
					case 9:
						if (vars->type == ASN_INTEGER)
						{
							int		n,*pn;
							int		type;
							int		int_port,port;
							char		tmp[100];

							int_port=*vars->val.integer;	// TODO: Check for NULL
							sprintf(tmp,"%02X:%02X:%02X:%02X:%02X:%02X",
									vars->name[pdata->root_oid_len],
									vars->name[pdata->root_oid_len+1],
									vars->name[pdata->root_oid_len+2],
									vars->name[pdata->root_oid_len+3],
									vars->name[pdata->root_oid_len+4],
									vars->name[pdata->root_oid_len+5]);
							
							pn=pdata->l2_switch.macs_tmp[tmp];
							if (pn==NULL)
								type=-1;
							else
								type=*pn;
								
							port=-1;
							if (type==3)	// learned
							{
								L2S_IFACE	*piface;
								
								piface=pdata->l2_switch.get_internal(int_port);
								
								if (piface!=NULL)
								{
									port=piface->ifindex;
									piface->macs.Add(tmp);
								}
							} else if (type==4)	// self
							{
								netdev_macs.Add(tmp);
							}
							
							printf("MAC: %s  internal-port: %d  port: %d  type: %d\n", tmp, int_port, port, type);
						} else
						{
							printf("(7)Returned non-integer\n");
							print_variable(vars->name, vars->name_length, vars);
							pdata->l2_switch.status=-1;
							done=1;
						}
						break;
					case 11:
						if (vars->type == ASN_OCTET_STR)
						{
							char		tmp[100];
							unsigned char	*p;

							if (vars->val_len==6)
							{
								p=(unsigned char *)malloc(vars->val_len+1);
								memcpy(p,vars->val.string,vars->val_len);
								p[vars->val_len]=0;

								sprintf(tmp,"%02X:%02X:%02X:%02X:%02X:%02X",
										p[0],p[1],p[2],p[3],p[4],p[5]);

								printf("Switch phys address: %s\n", tmp);
								netdev_macs.Add(tmp);

								free(p);
							}
						} else
						{
							printf("(11)Returned non-string\n");
							print_variable(vars->name, vars->name_length, vars);
							pdata->l2_switch.status=-1;
							done=1;
						}
						break;
				}
				memcpy((char *)pdata->cur_oid, (char *)vars->name, vars->name_length * sizeof(oid));
				pdata->cur_oid_len=vars->name_length;
				if ((vars->type == SNMP_ENDOFMIBVIEW) ||
					(vars->type == SNMP_NOSUCHOBJECT) ||
					(vars->type == SNMP_NOSUCHINSTANCE))
				{
					pdata->l2_switch.status++;
					done=1;
				} else
				{
					req=snmp_pdu_create(SNMP_MSG_GETNEXT);
					snmp_add_null_var(req, pdata->cur_oid, pdata->cur_oid_len);

					if (!snmp_send(pdata->psession, req))
					{
						snmp_perror("snmp_send");
						snmp_free_pdu(req);
						pdata->l2_switch.status=-1;
						done=1;
					}
				}
			}
		}
	}
	return(1);
}

int snmp_response_router(int operation, struct snmp_session *sp, int reqid,
			struct snmp_pdu *pdu, void *magic)
{
	DEVICE_DATA	*pdata = (DEVICE_DATA *)magic;
	snmp_pdu	*req;
	variable_list	*vars;
	int		done=0;

	if (pdata->router.status>=100)
		return(0);
		
	if (operation != NETSNMP_CALLBACK_OP_RECEIVED_MESSAGE)
	{
		pdata->router.status=-1;
	} else if (pdu==NULL)
	{
		pdata->router.status=-1;
	} else if (pdu->errstat!=SNMP_ERR_NOERROR)
	{
		pdata->router.status=-1;
	} else
	{
		for (vars=pdu->variables ; ( vars && !done ) ; vars=vars->next_variable)
		{
			if (vars->name_length < pdata->root_oid_len ||
				memcmp(vars->name, pdata->root_oid, pdata->root_oid_len * sizeof(oid)))
			{
				printf("%s(%d): end of data (%d-%d)\n",
								pdata->addr.p, 
								pdata->router.status,
								vars->name_length,
								pdata->root_oid_len);
				pdata->router.status++;
			} else if (snmp_oid_compare(pdata->cur_oid, pdata->cur_oid_len,
					vars->name, vars->name_length) >=0)
			{
				printf("%s(%d): OID not increasing!\n", pdata->addr.p, pdata->l2_switch.status);
				pdata->l2_switch.status++;
				done=1;
			} else
			{
				switch(pdata->router.status)
				{
					case 1:
						if (vars->type == ASN_OCTET_STR)
						{
							int		n;
							char		*p;
							ROUTER_IFACE	iface;
							
							n=pdata->root_oid_len;

							p=(char *)malloc(vars->val_len+1);
							memcpy(p,vars->val.string,vars->val_len);
							p[vars->val_len]=0;

							printf("Interface: %d - %s\n", vars->name[pdata->root_oid_len], p);

							// Add the interface
							iface.ifindex=vars->name[pdata->root_oid_len];
							iface.descr=p;
							pdata->router.ifaces.Add(iface.ifindex,iface);

							free(p);
						} else
						{
							printf("(1)Returned non-string\n");
							print_variable(vars->name, vars->name_length, vars);
							pdata->router.status=-1;
							done=1;
						}
						break;
					case 3:
						if (vars->type == ASN_OCTET_STR)
						{
							int		n;
							char		*p;
							ROUTER_IFACE	*piface;
							int		ifindex;
							
							ifindex=vars->name[pdata->root_oid_len];
							
							piface=pdata->router.ifaces[ifindex];
							if (piface==NULL)
							{
								printf("Ignoring unknown interface: index=%d\n",ifindex);
							} else
							{
								p=(char *)malloc(vars->val_len+1);
								memcpy(p,vars->val.string,vars->val_len);
								p[vars->val_len]=0;

								printf("Interface-alias: %d - %s\n", vars->name[pdata->root_oid_len], p);
								piface->alias=p;

								free(p);
							}
						} else
						{
							printf("(1)Returned non-string\n");
							print_variable(vars->name, vars->name_length, vars);
							pdata->router.status=-1;
							done=1;
						}
						break;
					case 5:
						if (vars->type == ASN_OCTET_STR)
						{
							int		n;
							ROUTER_IFACE	*piface;
							long		ifindex, ifinternal;
							char		tmp[100];
							char		tmp2[100];

							ifindex=vars->name[pdata->root_oid_len];
							
							sprintf(tmp,"%d.%d.%d.%d",
									vars->name[pdata->root_oid_len+1],
									vars->name[pdata->root_oid_len+2],
									vars->name[pdata->root_oid_len+3],
									vars->name[pdata->root_oid_len+4]);
							
							piface=pdata->router.ifaces[ifindex];
//								print_variable(vars->name, vars->name_length, vars);
							if (piface==NULL)
							{
								print_variable(vars->name, vars->name_length, vars);
								printf("Ignoring unknown interface: index=%d  ip=%s\n",ifindex, tmp);
							} else
							{
								STRLIST		*plist,list;
								
								sprintf(tmp2,"%02X:%02X:%02X:%02X:%02X:%02X",
									vars->val.string[0],
									vars->val.string[1],
									vars->val.string[2],
									vars->val.string[3],
									vars->val.string[4],
									vars->val.string[5]);
								
								printf("Interface: index=%d  ip=%s  mac=%s\n",ifindex, tmp, tmp2);
								//piface->ips.Add(tmp2,tmp);
								plist=piface->ips[tmp2];
								if (plist==NULL)
								{
									piface->ips.Add(tmp2,list);
									plist=piface->ips[tmp2];
									plist->SetUnique(1);
								}
								plist->Add(tmp);
							}
						} else
						{
							printf("(5)Returned non-integer\n");
							print_variable(vars->name, vars->name_length, vars);
							pdata->router.status=-1;
							done=1;
						}
						break;
					case 7:
						if (vars->type == ASN_OCTET_STR)
						{
							char		tmp[100];
							unsigned char	*p;

							if (vars->val_len==6)
							{
								p=(unsigned char *)malloc(vars->val_len+1);
								memcpy(p,vars->val.string,vars->val_len);
								p[vars->val_len]=0;

								sprintf(tmp,"%02X:%02X:%02X:%02X:%02X:%02X",
										p[0],p[1],p[2],p[3],p[4],p[5]);

								printf("Router phys address: %s\n", tmp);
								router_macs.Add(tmp);

								free(p);
							}
						} else
						{
							printf("(7)Returned non-string\n");
							print_variable(vars->name, vars->name_length, vars);
							pdata->router.status=-1;
							done=1;
						}
						break;
				}
				memcpy((char *)pdata->cur_oid, (char *)vars->name, vars->name_length * sizeof(oid));
				pdata->cur_oid_len=vars->name_length;
				if ((vars->type == SNMP_ENDOFMIBVIEW) ||
					(vars->type == SNMP_NOSUCHOBJECT) ||
					(vars->type == SNMP_NOSUCHINSTANCE))
				{
					pdata->router.status++;
					done=1;
				} else
				{
					req=snmp_pdu_create(SNMP_MSG_GETNEXT);
					snmp_add_null_var(req, pdata->cur_oid, pdata->cur_oid_len);

					if (!snmp_send(pdata->psession, req))
					{
						snmp_perror("snmp_send");
						snmp_free_pdu(req);
						pdata->router.status=-1;
						done=1;
					}
				}
			}
		}
	}
	return(1);
}


void	snmp_prepare_device_data()
{
	DEVICE_DATA	*pdata;
	LONGLINE	*pll;
	
	devices.Rewind();
	
	while ((pll=devices.Get_Next())!=NULL)
	{
		pdata=*(devices[*pll]);
		
		snmp_sess_init(&pdata->session);
		pdata->session.version=SNMP_VERSION_2c;
		pdata->session.peername=pdata->addr.p;
		pdata->session.community=(u_char *)pdata->community.p;
		pdata->session.community_len=pdata->community.len;
		
		switch(pdata->type)
		{
			case l2_switch:
				pdata->session.callback=snmp_response_l2_switch;
				pdata->session.callback_magic=pdata;
		
				pdata->psession=snmp_open(&pdata->session);
				if (pdata->psession==NULL)
				{
					pdata->l2_switch.status=-1;
					snmp_perror("snmp_open");
				}
				break;
			case router:
				pdata->session.callback=snmp_response_router;
				pdata->session.callback_magic=pdata;
		
				pdata->psession=snmp_open(&pdata->session);
				if (pdata->psession==NULL)
				{
					pdata->router.status=-1;
					snmp_perror("snmp_open");
				}
		}
	}
}

/*
	Check if there are pending requests
	1: Nope.. all done
	0: Still waiting
*/
int	snmp_devices_done()
{
	DEVICE_DATA	*pdata;
	LONGLINE	*pll;
	
	devices.Rewind();
	while ((pll=devices.Get_Next())!=NULL)
	{
		pdata=*(devices[*pll]);
		switch(pdata->type)
		{
			case l2_switch:
				if (pdata->l2_switch.status>=0 && pdata->l2_switch.status<100)
				return(0);
				break;
			case router:
				if (pdata->router.status>=0 && pdata->router.status<100)
				return(0);
				break;
		}
	}
	
	return(1);
}

/*
	Send requests
*/
void	snmp_send_requests()
{
	DEVICE_DATA	*pdata;
	LONGLINE	*pll;
	snmp_pdu	*pdu;
	oid		*ret;
	char		*mib;
	
	devices.Rewind();
	while ((pll=devices.Get_Next())!=NULL)
	{
		pdata=*(devices[*pll]);
		switch(pdata->type)
		{
			case l2_switch:
				switch(pdata->l2_switch.status)
				{
					case 0:
					case 2:
					case 4:
					case 6:
					case 8:
					case 10:
						pdu=snmp_pdu_create(SNMP_MSG_GETNEXT);

						pdata->root_oid_len=MAX_OID_LEN;
						switch(pdata->l2_switch.status)
						{
							case 0: mib=GET_IFDESCR_MIB; break;
							case 2: mib=GET_IFALIAS_MIB; break;
							case 4: mib=GET_IFINTERNAL_MIB; break;
							case 6: mib=GET_TPFDBSTATUS_MIB; break;
							case 8: mib=GET_TPFDBPORT_MIB; break;
							case 10: mib=GET_IFPHYSADDRESS_MIB; break;
						}
						ret=snmp_parse_oid(mib, 
							pdata->root_oid,
							&pdata->root_oid_len);
						if (!ret)
						{
							snmp_perror("snmp_parse_oid");
							pdata->l2_switch.status=-1;
							break;
						}
						memcpy(pdata->cur_oid, pdata->root_oid, sizeof(oid) * pdata->root_oid_len);
						pdata->cur_oid_len=pdata->root_oid_len;

						snmp_add_null_var(pdu, pdata->cur_oid, pdata->cur_oid_len);
						pdata->l2_switch.status++;
						
						if (!snmp_send(pdata->psession, pdu))
						{
							snmp_perror("snmp_send");
							snmp_free_pdu(pdu);
							pdata->l2_switch.status=-1;
						}
						
						break;
					case 12:
						pdata->l2_switch.status=100;
						break;
				}
				if (pdata->l2_switch.status<0 || pdata->l2_switch.status>=12)
				{
					snmp_close(pdata->psession);
				}
				break;
			case router:
				switch(pdata->router.status)
				{
					case 0:
					case 2:
					case 4:
					case 6:
						pdu=snmp_pdu_create(SNMP_MSG_GETNEXT);

						pdata->root_oid_len=MAX_OID_LEN;
						switch(pdata->router.status)
						{
							case 0: mib=GET_IFDESCR_MIB; break;
							case 2: mib=GET_IFALIAS_MIB; break;
							case 4: mib=GET_IFNETTOMEDIA_MIB; break;
							case 6: mib=GET_IFPHYSADDRESS_MIB; break;
						}
						ret=snmp_parse_oid(mib, 
							pdata->root_oid,
							&pdata->root_oid_len);
						if (!ret)
						{
							snmp_perror("snmp_parse_oid");
							pdata->router.status=-1;
							break;
						}
						memcpy(pdata->cur_oid, pdata->root_oid, sizeof(oid) * pdata->root_oid_len);
						pdata->cur_oid_len=pdata->root_oid_len;

						snmp_add_null_var(pdu, pdata->cur_oid, pdata->cur_oid_len);
						pdata->router.status++;
						
						if (!snmp_send(pdata->psession, pdu))
						{
							snmp_perror("snmp_send");
							snmp_free_pdu(pdu);
							pdata->router.status=-1;
						}
						
						break;
					case 8:
						pdata->router.status=100;
						break;
				}
				if (pdata->router.status<0 || pdata->router.status>=8)
				{
					snmp_close(pdata->psession);
				}
				break;
		}
			
	}
}

void	snmp_do_select()
{
	fd_set		fdr;
	int		fds;
	int		block;
	struct timeval	tv;
	int		ret;
	
	snmp_send_requests();
	
	while (!snmp_devices_done())
	{
		block=1;
		fds=0;
		FD_ZERO(&fdr);

		snmp_select_info(&fds, &fdr, &tv, &block);
		ret=select(fds, &fdr, NULL, NULL, block ? NULL : &tv);
		if (ret)
			snmp_read(&fdr);
		else
			snmp_timeout();
		snmp_send_requests();
	}
}

void	snmp_doit()
{
	snmp_init_device_data();
	
	init_snmp("V-SNMP");
	snmp_prepare_device_data();
	

	snmp_do_select();	
}

/*
	Detect and mark switch ports that have other
	network devices connected 
*/
void	detect_netdevs()
{
	DEVICE_DATA	*pdata;
	LONGLINE	*pll,*pmac;
	
	devices.Rewind();
	
	while ((pll=devices.Get_Next())!=NULL)
	{
		pdata=*(devices[*pll]);
		printf("Checking %s:\n",pll->p);
		if (pdata->type==l2_switch)
		{
			L2S_IFACE	*piface;
			u_32		key;
			
			pdata->l2_switch.ifaces.Rewind();
			
			while ((key=pdata->l2_switch.ifaces.Get_Next())!=NK_INVALID)
			{
				piface=pdata->l2_switch.ifaces[key];
				piface->macs.Rewind();
				while (((pmac=piface->macs.Get_Next())!=NULL) &&
						(!piface->has_netdev))
				{
					if ((netdev_macs.Exist(*pmac)) || (router_macs.Exist(*pmac)))
					{
						piface->has_netdev=1;
						printf("Interface %s has netdev\n",piface->descr.p);
					}
				}
			}
		}
	}
}

void	create_db()
{
	DEVICE_DATA	*pdev;
	LONGLINE	*pll;
	
	clear_tables();
	
	devices.Rewind();
	while ((pll=devices.Get_Next())!=NULL)
	{
		pdev=*(devices[*pll]);
		
		switch(pdev->type)
		{
			case l2_switch:
				{
					L2S_IFACE	*piface;
					u_32		key;
					LONGLINE	*pmac;
					
					pdev->l2_switch.ifaces.Rewind();
					while ((key=pdev->l2_switch.ifaces.Get_Next())!=NK_INVALID)
					{
						piface=pdev->l2_switch.ifaces[key];
						add_interface(pdev->addr.p, piface->ifindex, piface->descr.p, piface->alias.p, piface->internal, piface->has_netdev);
						
						if (!piface->has_netdev)
						{
							piface->macs.Rewind();
							while ((pmac=piface->macs.Get_Next())!=NULL)
							{
								add_mac(pdev->addr.p, piface->ifindex, pmac->p);
							}
						}
					}
				}
				break;
			case router:
				{
					ROUTER_IFACE	*piface;
					u_32		key;
					LONGLINE	*pmac,*pip;
					STRLIST		*plist;
					
					pdev->router.ifaces.Rewind();
					while ((key=pdev->router.ifaces.Get_Next())!=NK_INVALID)
					{
						piface=pdev->router.ifaces[key];
						add_interface(pdev->addr.p, piface->ifindex, piface->descr.p, piface->alias.p);
						
						piface->ips.Rewind();
//						while ((pmac=piface->ips.Get_Next())!=NULL)
						while ((pmac=piface->ips.Get_Next())!=NULL)
						{
							if (!router_macs.Exist(*pmac))	// Don't add router IPs
							{
								plist=piface->ips[*pmac];
								plist->Rewind();
								while ((pip=plist->Get_Next())!=NULL)
								{
	//								pip=piface->ips[*pmac];
									add_ips(pip->p, pmac->p, pdev->addr.p, piface->ifindex);
								}
							}
						}
					}
				}
				break;
		}		
	}
}

int snmp_main()
{
	snmp_doit();
	detect_netdevs();
	create_db();
}

