--
-- Entries older than 30 days are deleted
--
-- Entries older than 5 hours are assumed not-current when needed
--
-- Interfaces that don't have netdevs for more than 7 days are not
-- netdev interfaces
--

--------------------------------
-- Update the history table
--
CREATE OR REPLACE FUNCTION set_history()
	RETURNS INTEGER
	AS '
--
-- update existing entries
--
		UPDATE history
			SET ts = 
				( SELECT mac_ts
					FROM view_current v
					WHERE history.ip=v.ip AND
						history.mac=v.mac AND
						history.ifid=v.ifid )
			WHERE EXISTS
				( SELECT mac_ts
					FROM view_current v
					WHERE history.ip=v.ip AND
						history.mac=v.mac AND
						history.ifid=v.ifid
					LIMIT 1 );

--
-- insert new entries
--
		INSERT INTO history(ifid,ip,mac,ts)
		SELECT ifid,ip,mac,mac_ts
			FROM view_current v
			WHERE
				ip IS NOT NULL AND
				mac IS NOT NULL AND
				ifid IS NOT NULL AND
-- on interfaces where there is no netdev ??
--				last_had_netdev=0 AND
-- Ignore VLANs - What about internal addresses ?
--				lower(ifdescr) NOT LIKE \'vlan%\' AND
-- entries that dont exist
				NOT EXISTS
					( SELECT * FROM history h
						WHERE h.ifid=v.ifid AND
							h.ip=v.ip AND
							h.mac=v.mac );
				
		SELECT 1;
	'
	LANGUAGE SQL;

--------------------------------
-- expire entries
--	* from history
--	* macs
--	* ips
--	* log
--
CREATE OR REPLACE FUNCTION do_expire()
	RETURNS INTEGER
	AS '
--
-- Delete entries older than 30 days
--
		DELETE FROM history
			WHERE (EXTRACT(EPOCH FROM CURRENT_TIMESTAMP) - ts) > ( 86400 * 30 );
		DELETE FROM macs
			WHERE (EXTRACT(EPOCH FROM CURRENT_TIMESTAMP) - ts) > ( 86400 * 30 );
		DELETE FROM allmacs
			WHERE (EXTRACT(EPOCH FROM CURRENT_TIMESTAMP) - ts) > ( 86400 * 30 );
		DELETE FROM ips
			WHERE (EXTRACT(EPOCH FROM CURRENT_TIMESTAMP) - ts) > ( 86400 * 30 );
		DELETE FROM interfaces
			WHERE (EXTRACT(EPOCH FROM CURRENT_TIMESTAMP) - ts) > ( 86400 * 30 );

--
-- Delete unlocated ips after 1 day
--
		DELETE FROM ips
			WHERE NOT EXISTS ( SELECT mac FROM macs WHERE macs.mac=ips.mac) AND
				(EXTRACT(EPOCH FROM CURRENT_TIMESTAMP) - ts) > ( 86400 * 1 );

--
-- Delete history entries older than 5 days that have similar newer entries
--
		DELETE FROM history
			WHERE (EXTRACT(EPOCH FROM CURRENT_TIMESTAMP) - ts) > ( 86400 * 5 )  AND
				EXISTS ( SELECT mac FROM history h2
						WHERE h2.mac=history.mac AND
							h2.ts > history.ts);

--
-- Reset interfaces that dont have netdevs any more
--
		UPDATE interfaces
			SET last_had_netdev=0
			WHERE (EXTRACT(EPOCH FROM CURRENT_TIMESTAMP) - last_had_netdev) > ( 86400 * 7 );

--
-- Delete log entries older than 15 days
--
		DELETE FROM log
			WHERE (EXTRACT(EPOCH FROM CURRENT_TIMESTAMP) - ts) > ( 86400 * 15 );

		SELECT 1;
	'
	LANGUAGE SQL;

--------------------------------
-- Shorten interface description
--
CREATE OR REPLACE FUNCTION short_ifdescr(VARCHAR(128))
	RETURNS VARCHAR(64)
	AS '
		SELECT
			CASE
				WHEN ( $1 LIKE \'Fast%/%/%\' ) OR
					( $1 LIKE \'Gi%/%/%\' ) OR
					( $1 LIKE \'Serial%/%/%\' ) THEN
					substring($1 FROM 1 FOR 2) ||
					substring($1 FROM \'[0-9]*/[0-9]*/[0-9]*$\')
				WHEN ( $1 LIKE \'Fast%\' ) OR
					( $1 LIKE \'Gi%\' ) OR
					( $1 LIKE \'Serial%\' ) THEN
					substring($1 FROM 1 FOR 2) ||
					substring($1 FROM \'[0-9]*/[0-9]*$\')
				WHEN	( $1 LIKE \'Eth%\' ) THEN
					\'Eth\' ||
					substring($1 FROM \'[0-9]*$\')
				WHEN	( $1 LIKE \'Loopback%\' ) THEN
					\'Lo\' ||
					substring($1 FROM \'[0-9]*$\')
				WHEN	( $1 LIKE \'Tunnel%\' ) THEN
					\'Tun\' ||
					substring($1 FROM \'[0-9]*$\')
				ELSE $1
			END;
	'
	LANGUAGE SQL;

CREATE OR REPLACE FUNCTION clear_autolock()
	RETURNS INTEGER
	AS '
-- Log interfaces that will not be autolocked any more
	INSERT INTO log(logtype, severity, msg)
		SELECT 2, 3, \'Autolock disabled for \' ||
			netdev_name || \' - \' || short_ifdescr(ifdescr)
			FROM view_interfaces v
			WHERE (v.if_flags & X\'100\'::INTEGER)>0 AND
				NOT EXISTS ( SELECT *
						FROM blacklist b, macs m
						WHERE b.mac=m.mac AND
							m.ifid=v.ifid);

-- Remove autolock from interfaces with no blacklisted MACs
	UPDATE interfaces
		SET flags = ( flags & ~(X\'100\'::INTEGER ))
		WHERE (flags & X\'100\'::INTEGER)>0 AND
			NOT EXISTS ( SELECT *
					FROM blacklist b, macs m
					WHERE b.mac=m.mac AND
						m.ifid=interfaces.id);

	SELECT 1;
	'
	LANGUAGE SQL;

--------------------------------
-- Set the is_locked flags (auto locking)
-- to interfaces that have do_locking
-- and dont have restricted MAC access
-- and there exist a violation
--
-- Also clear the locking flag from interfaces
-- that have changed
--
CREATE OR REPLACE FUNCTION set_autolock()
	RETURNS INTEGER
	AS '
-- First log the changes
	INSERT INTO log(logtype, severity, msg)
		SELECT 2, 5, \'Locking activated for \' ||
			netdev_name || \' - \' || short_ifdescr(ifdescr)
			FROM view_interfaces
-- Log only those that are not already auto-locked
			WHERE (if_flags & X\'100\'::INTEGER) = 0 AND
				( if_flags & X\'20\'::INTEGER) = X\'20\'::INTEGER AND
				( if_flags & X\'1\'::INTEGER) = 0 AND
				EXISTS ( SELECT * FROM blacklist b
					LEFT JOIN macs m USING(mac)
					WHERE m.ifid=view_interfaces.ifid LIMIT 1);

	UPDATE interfaces
		SET flags = ( flags | X\'100\'::INTEGER )
-- do_locking
		WHERE ( flags & X\'20\'::INTEGER) = X\'20\'::INTEGER AND
-- no assoc_macs
			( flags & X\'1\'::INTEGER) = 0 AND
-- violation
			EXISTS ( SELECT * FROM blacklist b
					LEFT JOIN macs m USING(mac)
					WHERE m.ifid=interfaces.id LIMIT 1);

-- Log changed interfaces
	INSERT INTO log(logtype, severity, msg)
		SELECT 3, 3, \'Interface \' || netdev_name ||
			\' - \' || ifdescr || \' changed\'
			FROM view_interfaces_info
			WHERE is_changed=1;

-- Log interfaces that will no more be locked
	INSERT INTO log(logtype, severity, msg)
		SELECT 3, 4, \'Interface \' || netdev_name || \' - \' ||
			short_ifdescr(ifdescr) || \' will not be locked any more.\'
			FROM view_interfaces
			WHERE ( if_flags & X\'40\'::INTEGER ) > 0 AND
				( if_flags & X\'20\'::INTEGER ) > 0;

-- Clear do_locking (0x20) from changed (0x40) interfaces that had do_locking
	UPDATE interfaces
		SET flags = (flags & ~(X\'20\'::INTEGER))
		WHERE ( flags & X\'40\'::INTEGER ) > 0 AND
			( flags & X\'20\'::INTEGER ) > 0;

-- Clear changed flag (0x40)
	UPDATE interfaces
		SET flags = (flags & ~(X\'40\'::INTEGER))
		WHERE (flags & (X\'40\'::INTEGER)) >0;

	SELECT 1;
	'
	LANGUAGE SQL;

CREATE TABLE ip_nets (
	net		CIDR,
	reged_ips	INTEGER,
	current_ips	INTEGER,
	all_ips		INTEGER
);

CREATE OR REPLACE FUNCTION sum_ips(INTEGER)
	RETURNS SETOF ip_nets
	AS '
		SELECT network(set_masklen(i.ip,$1)) AS net,
			COUNT((SELECT 1 WHERE EXISTS (SELECT ip FROM subnets s WHERE s.ip>>=i.ip)))::INTEGER,
			COUNT((SELECT 1 WHERE (EXTRACT(EPOCH FROM CURRENT_TIMESTAMP) - ts) < ( 86400 * 5 )))::INTEGER,
			COUNT(DISTINCT i.ip)::INTEGER

			FROM ips i

			GROUP BY net;
	'
	LANGUAGE SQL;

CREATE OR REPLACE FUNCTION policy_reason(INTEGER)
	RETURNS VARCHAR(64)
	AS '
		SELECT
			CASE
				WHEN ($1 = 1) THEN \'This MAC should not use this IP\'
				WHEN ($1 = 2) THEN \'This MAC should not exist on this interface\'
				WHEN ($1 = 3) THEN \'This IP should not be used by this MAC\'
				WHEN ($1 = 4) THEN \'This IP should not exist on this interface\'
				WHEN ($1 = 5) THEN \'This interface should not serve this MAC\'
				WHEN ($1 = 6) THEN \'This interface should not serve this IP\'
				WHEN ($1 = 7) THEN \'This interface should not serve this MAC (hunted)\'
				WHEN ($1 = 8) THEN \'This interface should not serve this IP (hunted)\'
				WHEN ($1 = 9) THEN \'This subnet should not be used by this MAC\'
				WHEN ($1 = 10) THEN \'This subnet should not exist on this interface\'
			END
	'
	LANGUAGE SQL;

CREATE OR REPLACE FUNCTION policy_reason_short(INTEGER)
	RETURNS VARCHAR(64)
	AS '
		SELECT
			CASE
				WHEN ($1 = 1) THEN \'Bad IP for this MAC\'
				WHEN ($1 = 2) THEN \'Bad Interface for this MAC\'
				WHEN ($1 = 3) THEN \'Bad MAC for this IP\'
				WHEN ($1 = 4) THEN \'Bad Interface for this IP\'
				WHEN ($1 = 5) THEN \'Bad MAC on this Interface\'
				WHEN ($1 = 6) THEN \'Bad IP on this Interface\'
				WHEN ($1 = 7) THEN \'Bad MAC on this Interface (hunted)\'
				WHEN ($1 = 8) THEN \'Bad IP on this Interface (hunted)\'
				WHEN ($1 = 9) THEN \'Bad MAC for this subnet\'
				WHEN ($1 = 10) THEN \'Bad Interface for this subnet\'
			END
	'
	LANGUAGE SQL;


--
-- Insert policy violators in the blacklist
--
CREATE OR REPLACE FUNCTION set_blacklist()
	RETURNS INTEGER
	AS '
		INSERT INTO log(logtype, severity, msg)
			SELECT 3, 5, \'Added \' || mac::TEXT || \' to blacklist. Reason: "\' ||
				policy_reason(vtype) || \'"\'
				FROM ( SELECT DISTINCT mac FROM view_policy_violators
					EXCEPT SELECT DISTINCT mac FROM blacklist ) a
				LEFT JOIN view_policy_violators USING(mac);

		INSERT INTO blacklist(mac, flags, shortdesc, comments)
			SELECT mac, 1, policy_reason_short(vtype), policy_reason(vtype)
				FROM ( SELECT DISTINCT mac FROM view_policy_violators
					EXCEPT SELECT DISTINCT mac FROM blacklist ) a
				LEFT JOIN view_policy_violators USING(mac);

		SELECT 1;
	'
	LANGUAGE SQL;

--
-- Filter out IPs that don't belong to know subnets
--
CREATE OR REPLACE FUNCTION filter_ips()
	RETURNS INTEGER
	AS '
		DELETE FROM ips
			WHERE NOT EXISTS
				( SELECT ip FROM subnets
					WHERE subnets.ip>>=ips.ip );

		SELECT 1;
	'
	LANGUAGE SQL;

--
-- Log new interfaces and clear their 'new' status
--
CREATE OR REPLACE FUNCTION log_new_interfaces()
	RETURNS INTEGER
	AS '
-- Log new interfaces
		INSERT INTO log(logtype, severity, msg)
			SELECT 3, 3, \'Found new interface \' ||
				netdev_name || \' - \' || ifdescr
			FROM view_interfaces_info
			WHERE is_added=1;

-- Clear the added flag (0x80)
		UPDATE interfaces
			SET flags=(flags & ~(X\'80\'::INTEGER) )
			WHERE ( flags & X\'80\'::INTEGER) > 0;

		SELECT 1;
	'
	LANGUAGE SQL;

--
-- Log new subnets and clear their 'new' status
--
CREATE OR REPLACE FUNCTION log_new_subnets()
	RETURNS INTEGER
	AS '
-- Log new subnets
		INSERT INTO log(logtype, severity, msg)
			SELECT 4, 3, \'Found new subnet: \' || ip::TEXT
			FROM subnets
			WHERE ( flags & X\'080\'::INTEGER) > 0;

-- Clear the added flag (0x80)
		UPDATE subnets
			SET flags=(flags & ~(X\'80\'::INTEGER) )
			WHERE ( flags & X\'80\'::INTEGER ) > 0;

		SELECT 1;
	'
	LANGUAGE SQL;

--
-- Set ml_last_update
--
CREATE OR REPLACE FUNCTION ml_set_upd(INET)
	RETURNS INTEGER
	AS '
		UPDATE netdevs
			SET ml_last_update = (EXTRACT (EPOCH FROM CURRENT_TIMESTAMP))
			WHERE ip = $1;

		SELECT 1;
	'
	LANGUAGE SQL;

--
-- Set ml_last_try
--
CREATE OR REPLACE FUNCTION ml_set_try(INET)
	RETURNS INTEGER
	AS '
		UPDATE netdevs
			SET ml_last_try = (EXTRACT (EPOCH FROM CURRENT_TIMESTAMP))
			WHERE ip = $1;

		SELECT 1;
	'
	LANGUAGE SQL;

