CREATE SEQUENCE seq_netdevs;
CREATE TABLE netdevs (
	id		INTEGER
			PRIMARY KEY
			DEFAULT NEXTVAL('seq_netdevs'),

	ip		INET
			UNIQUE
			NOT NULL,

	name		VARCHAR(64),

	community	VARCHAR(64)
			NOT NULL,

	ts		INTEGER
			DEFAULT 0
			NOT NULL,

	ml_uname	VARCHAR(64),

	ml_t_pass	VARCHAR(64),

	ml_en_pass	VARCHAR(64),

	ml_last_update	INTEGER,

	ml_last_try	INTEGER,

-- flags & 0x07: type (for mac-locking)
--	0x01:	Router
--	0x02:	Switch
--
-- flags & 0x08:
--	0x08:	Locking enabled
--
	flags		INTEGER
) WITHOUT OIDS;

CREATE SEQUENCE seq_interfaces;
CREATE TABLE interfaces(
	id		INTEGER
			PRIMARY KEY
			DEFAULT NEXTVAL('seq_interfaces'),

	netdev_id	INTEGER
			REFERENCES netdevs(id)
				ON DELETE CASCADE
				ON UPDATE CASCADE,

	description	VARCHAR(128)
			NOT NULL,

	alias		VARCHAR(128),

	ifindex		INTEGER
			NOT NULL,

	last_had_netdev	INTEGER
			DEFAULT '0',

	ts		INTEGER
			NOT NULL,

-- flags & 0x01:
--	0x01:	Associated MACs only
--
-- flags & 0x02:
--	0x02:	Associated IPs only
--
-- flags & 0x10:
--	0x10:	Routed interface
--
-- flags & 0x20:
--	0x20:	Do locking
--
-- flags & 0x40:
--	0x40:	Changed
--
-- flags & 0x80:
--	0x80:	Added
--
-- flags & 0x100:
--	0x100:	Locked - Add access lists
--
-- flags & 0x200:
--	0x200:	Hunt
--
	flags		INTEGER
			DEFAULT '0',

	comments	TEXT,

	shortdesc	TEXT,

	UNIQUE(netdev_id,description)
) WITHOUT OIDS;

CREATE TABLE macs(
	mac		MACADDR
			PRIMARY KEY,

	ifid		INTEGER
			REFERENCES interfaces(id)
				ON DELETE CASCADE
				ON UPDATE CASCADE,

	ts		INTEGER
			NOT NULL,

-- flags & 0x01:
--	0x01:	Associated IPs only
--
-- flags & 0x02:
--	0x02:	Associated Interfaces only
--
-- flags & 0x80:
--	0x80:	Added -- just like interfaces
--
-- flags & 0x100:
--	0x100:	Internal mac
--
	flags		INTEGER,

	shortdesc	VARCHAR(64),

	description	TEXT

) WITHOUT OIDS;

CREATE TABLE ips(
	ip		INET
			PRIMARY KEY
			NOT NULL,

	mac		MACADDR
-- Unlocated macs are no problem. They will be deleted after a while
--			REFERENCES macs(mac)
--				ON DELETE CASCADE
--				ON UPDATE CASCADE
			NOT NULL,

	ts		INTEGER
			NOT NULL,

-- flags & 0x01:
--	0x01:	Associated MACs only
--
-- flags & 0x02:
--	0x02:	Associated Interfaces only
--
	flags		INTEGER
			DEFAULT 0,

	shortdesc	VARCHAR(64),

	description	TEXT
) WITHOUT OIDS;

CREATE TABLE allmacs(
	ifid		INTEGER
			REFERENCES interfaces(id)
				ON DELETE CASCADE
				ON UPDATE CASCADE,

	mac		MACADDR
			NOT NULL,

	ts		INTEGER
			NOT NULL,

	PRIMARY KEY(ifid,mac)
) WITHOUT OIDS;

CREATE TABLE vendors(
	prefix		MACADDR
			PRIMARY KEY,

	name		VARCHAR(64)
) WITHOUT OIDS;

CREATE TABLE history(
	ifid		INTEGER
			REFERENCES interfaces(id)
				ON DELETE CASCADE
				ON UPDATE CASCADE,

	ip		INET,

	mac		MACADDR,

-- Last time this entry was found
	ts		INTEGER,

-- flags & 0x03:
--	0x01:	Ignored once
--	0x02:	Ignored forever
--
	flags		INTEGER,

-- The last time this entry was ignored
	ignore_ts	INTEGER,

	shortdesc	VARCHAR(64),

	comments	TEXT,

	PRIMARY KEY (ifid,ip,mac)
) WITHOUT OIDS;

--------------------------------
-- Subnet registry
--

CREATE TABLE subnets(
	ip		INET
			PRIMARY KEY,

-- flags & 0x01:
--	0x01:	Associated MACs only
--
-- flags & 0x02:
--	0x02:	Associated Interfaces only
--
-- flags & 0x10:
--	0x10:	Auto-discovered
--
-- flags & 0x80:
--	0x80:	Added
	flags		INTEGER,

	shortdesc	VARCHAR(64),

	description	TEXT
) WITHOUT OIDS;

----------------------------------
-- Monitor filters
--

CREATE SEQUENCE seq_monitor_filters;
CREATE TABLE monitor_filters(
	id		INTEGER
			PRIMARY KEY
			DEFAULT NEXTVAL('seq_monitor_filters'),

	ip_from		INET,

	ip_to		INET,

	ip_mask		CIDR,

	mac		MACADDR,

	shortdesc	VARCHAR(128),

	netdev_id	INTEGER
			REFERENCES netdevs(id)
				ON DELETE CASCADE
				ON UPDATE CASCADE,

	ifid		INTEGER
			REFERENCES interfaces(id)
				ON DELETE CASCADE
				ON UPDATE CASCADE
) WITHOUT OIDS;

----------------------------------
-- Associations
--

CREATE TABLE assoc_ip2mac(
	ip		INET,

	mac		MACADDR
			REFERENCES macs(mac)
				ON DELETE CASCADE
				ON UPDATE CASCADE,

	PRIMARY KEY(ip,mac)
) WITHOUT OIDS;

CREATE TABLE assoc_ip2iface(
	ip		INET,

	ifid		INTEGER
			REFERENCES interfaces(id)
				ON DELETE CASCADE
				ON UPDATE CASCADE,

	PRIMARY KEY(ifid,ip)
) WITHOUT OIDS;

CREATE TABLE assoc_mac2iface(
	mac		MACADDR
			REFERENCES macs(mac)
				ON DELETE CASCADE
				ON UPDATE CASCADE,

	ifid		INTEGER
			REFERENCES interfaces(id)
				ON DELETE CASCADE
				ON UPDATE CASCADE,

	PRIMARY KEY(mac,ifid)
) WITHOUT OIDS;

----------------------------------
-- Config
--

CREATE TABLE config(
	name		VARCHAR(64)
			PRIMARY KEY,

	value		VARCHAR(256)
) WITHOUT OIDS;

----------------------------------
-- Resolver cache
--

CREATE TABLE res_cache(
	ip		INET
			PRIMARY KEY
			REFERENCES ips(ip)
				ON DELETE CASCADE
				ON UPDATE CASCADE,

	hostname	VARCHAR(256),

	ts		INTEGER
) WITHOUT OIDS;

----------------------------------
-- Blacklist
--

CREATE TABLE blacklist(
	mac		MACADDR
			PRIMARY KEY
			REFERENCES macs(mac)
				ON DELETE CASCADE
				ON UPDATE CASCADE,

	ts		INTEGER
			NOT NULL
			DEFAULT EXTRACT(EPOCH FROM CURRENT_TIMESTAMP),

	shortdesc	VARCHAR(64),

	comments	TEXT,

--
--  flags & 0x01
--	0x01	auto-added
--
	flags		INTEGER
			DEFAULT 0

) WITHOUT OIDS;

----------------------------------
-- Log
--

CREATE SEQUENCE seq_log;
CREATE TABLE log(
	id		INTEGER
			PRIMARY KEY
			DEFAULT NEXTVAL('seq_log'),

-- 1: Maclock script
-- 2: Interface status
-- 3: Policy
-- 4: Data
	logtype		INTEGER,

-- 1: Debug
-- 2: Info
-- 3: Notice
-- 4: Warning
-- 5: Error/Alert
	severity	INTEGER,

	ts		INTEGER
			NOT NULL
			DEFAULT EXTRACT(EPOCH FROM CURRENT_TIMESTAMP),

	msg		TEXT
) WITHOUT OIDS;

