/*
  Copyright (c) 2002,2003,2004 Stefanos Harhalakis

  This file is part of netinfo.

  netinfo is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  netinfo is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with netinfo; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id: snmp.cc,v 1.9 2004/06/08 14:28:46 v13 Exp $
*/

#include "snmp.h"

#include "db.h"

#include "v/misc/keytable2.cc"
#include "v/misc/nkeytable.cc"
#include "v/socket/ipaddr.h"

#define	DO_GETNEXT	1
#define	DO_GET		2

struct vMIB
{
	const char	*mib;
	u_8		type;	// get or getnext
};

vMIB	vmib[]={
	{"IF-MIB::ifDescr",				DO_GETNEXT},	// 0 - iface description
	{"IF-MIB::ifAlias",				DO_GETNEXT},	// 1 - iface alias
	{"mib-2.17.1.4.1.2",				DO_GETNEXT},	// 2 - internal iface number
	{"mib-2.17.4.3.1.3",				DO_GETNEXT},	// 3 - mac-address type (learned etc..)
	{"mib-2.17.4.3.1.2",				DO_GETNEXT},	// 4 - mac-address port
	{"IP-MIB::ipNetToMediaPhysAddress",		DO_GETNEXT},	// 5 - ARP (ip to mac)
	{"IF-MIB::ifPhysAddress",			DO_GETNEXT},	// 6 - interfaces MACs
	{"IP-MIB::ipAdEntIfIndex",			DO_GETNEXT},	// 7 - interfaces IPs
	{"IF-MIB::ifType",				DO_GETNEXT},	// 8 - IANA interface type
	{"IP-MIB::ipAdEntNetMask",			DO_GETNEXT},	// 9 - Subnet masks
	{"enterprises.9.9.46.1.3.1.1.3.1",		DO_GETNEXT},	// 10 - VLAN types
	{NULL,0}
};

KEYTABLE<DEVICE_DATA *>	devices;	// key is the IP

void	snmp_init_query(NETDEV *pnetdev, DEVICE_DATA *pdev=NULL, int vlan=1);

/*
	type:
		1: ifDescr
		2: ifAlias
		3: ARP
		4: iface MACs
*/
void	parse_string(DEVICE_DATA *pdev, variable_list *vars, int type)
{
	char		*p;
	unsigned char	*p2;
	int		n;


	printf("String for %s - type=%d\n",pdev->pnetdev->ip.p, type);
	if (vars->type == ASN_OCTET_STR)
	{
		p=(char *)malloc(vars->val_len+1);
		memcpy(p,vars->val.string, vars->val_len);
		p[vars->val_len]=0;
		p2=(unsigned char *)p;

		n=vars->name[pdev->root_oid_len];

		if (type==1)
		{
			IFACE		iface;

			iface.ifindex=n;
			iface.descr=p;
			pdev->pnetdev->ifaces.Add(n,iface);
			printf("ifDescr: %s %d %s\n",pdev->pnetdev->ip.p,iface.ifindex, iface.descr.p);
		} else if (type==2)
		{
			IFACE	*piface;
			int	ifindex;

			ifindex=vars->name[pdev->root_oid_len];

			piface=pdev->pnetdev->ifaces[ifindex];
			if (piface==NULL)
			{
				printf("Ignoring data(string) for unknown interface %d (type=%d)\n", ifindex, type);
			} else
			{
//				printf("if-string: ip:%s  index:%d  type:%d  string:%s\n",pdev->pnetdev->ip.p, piface->index, type, p);
				printf("ifAlias: %s %s %s\n",pdev->pnetdev->ip.p, piface->descr.p, p);
				piface->alias=p;
			}

		} else if (type==3)	// ARP
		{
			char		tmp[100];
			char		tmp2[100];
			STRLIST		*plist;
			int		ifindex;
			IFACE		*piface;

			ifindex=vars->name[pdev->root_oid_len];

			sprintf(tmp,"%lu.%lu.%lu.%lu",
				vars->name[pdev->root_oid_len+1],
				vars->name[pdev->root_oid_len+2],
				vars->name[pdev->root_oid_len+3],
				vars->name[pdev->root_oid_len+4]);
			sprintf(tmp2,"%02X:%02X:%02X:%02X:%02X:%02X",
				p2[0],p2[1],p2[2],p2[3],p2[4],p2[5]);
			printf("ARP: ip=%s  mac=%s\n", tmp, tmp2);

			plist=arp[tmp2];
			if (plist==NULL)
			{
				STRLIST	list;

				arp.Add(tmp2,list);
				plist=arp[tmp2];
				plist->SetUnique(1);
			}
			plist->Add(tmp);

			piface=pdev->pnetdev->ifaces[ifindex];
			if (piface!=NULL)
			{
				piface->arpmacs.Add(tmp2);	// consider this a learned MAC
				printf("MAC (from arp): %s  port: %s\n", tmp2, piface->descr.p);
			}
		} else if (type==4)	// iface MACs
		{
			char	tmp[100];
			int	n;
			IFACE	*piface;

			n=vars->name[pdev->root_oid_len];

			sprintf(tmp,"%02X:%02X:%02X:%02X:%02X:%02X",
				p2[0],p2[1],p2[2],p2[3],p2[4],p2[5]);

			pdev->pnetdev->dev_macs.Add(tmp);
			netdev_macs.Add(tmp);

			piface=pdev->pnetdev->ifaces[n];
			// Add this MAC to the appropriate interface internal MACs
			if (piface!=NULL)
			{
				printf("Switch phys address: %s  Interface: %s\n", tmp, piface->descr.p);
				piface->internal_macs.Add(tmp);
			} else
			{
				printf("Switch phys address: %s for unknown interface\n", tmp);
			}
		}

		free(p);
	}
}

/*
	type:
		1: internal iface number
		2: mac-address type
		3: mac-address port
		4: iface IPs
		5: iface type
		6: subnet masks
*/
void	parse_number(DEVICE_DATA *pdev, variable_list *vars, int type)
{
	int	n,ifindex,*pn,port;
	u_32	sz;
	IFACE	*piface;
	char	tmp[100];

	printf("Number for %s - type=%d\n",pdev->pnetdev->ip.p, type);
	if (vars->type == ASN_COUNTER || vars->type == ASN_GAUGE ||
		vars->type == ASN_TIMETICKS || vars->type == ASN_INTEGER ||
		vars->type == ASN_IPADDRESS)
	{
		sz=*vars->val.integer;

		if (type==1)	// Internal iface number
		{
//			ifindex=vars->name[pdev->root_oid_len];
			n=vars->name[pdev->root_oid_len];
			ifindex=sz;
			piface=pdev->pnetdev->ifaces[ifindex];
			if (piface==NULL)
			{
				printf("Ignoring data(number) for unknown interface %d (type=%d)\n", ifindex, type);
			} else
			{
				if (piface->internal!=-1)
					printf("Ignoring duplicate entry for interface: index=%d  internal=%d\n",ifindex, n);
				else
					piface->internal=n;
			}
		} else if (type==5) // IANA interface type
		{
			ifindex=vars->name[pdev->root_oid_len];
			piface=pdev->pnetdev->ifaces[ifindex];
			if (piface==NULL)
			{
				printf("Ignoring data(number) for unknown interface %d (type=%d)\n", ifindex, type);
			} else
			{
				piface->iftype=sz;
			}
		} else if (type==2 || type==3) // mac-address type/port
		{
			sprintf(tmp,"%02X:%02X:%02X:%02X:%02X:%02X",
					(unsigned int)vars->name[pdev->root_oid_len],
					(unsigned int)vars->name[pdev->root_oid_len+1],
					(unsigned int)vars->name[pdev->root_oid_len+2],
					(unsigned int)vars->name[pdev->root_oid_len+3],
					(unsigned int)vars->name[pdev->root_oid_len+4],
					(unsigned int)vars->name[pdev->root_oid_len+5]);

			if (type==2)	// mac address type
			{
				printf("MAC: %s  type: %d\n",tmp, (int)sz);
/*				if (sz==4) // self
				{
					pdev->pnetdev->dev_macs.Add(tmp);
					netdev_macs.Add(tmp);
				} else
				{
					pdev->pnetdev->macs_tmp.Add(tmp,sz);
				}*/
				pdev->pnetdev->macs_tmp.Add(tmp,sz);
				if (sz==4) // self
				{
					pdev->pnetdev->dev_macs.Add(tmp);
					netdev_macs.Add(tmp);
				}

			} else if (type==3)	// mac address port
			{
				int	mactype;

				pn=pdev->pnetdev->macs_tmp[tmp];
				if (pn==NULL)
					mactype=-1;
				else
					mactype=*pn;
				port=-1;
//				if (mactype==3 || mactype==5)	// 3: learned, 5: security
				if (mactype==3 || mactype==4 || mactype==5)	// 3: learned, 4: self, 5: security
				{
					piface=pdev->pnetdev->get_internal(sz);
					if (piface!=NULL)
					{
						port=piface->ifindex;
						if (mactype==4)
							piface->internal_macs.Add(tmp);
						else
							piface->macs.Add(tmp);
					}
				}
				printf("MAC: %s  internal-port: %d  port: %d  type: %d\n", tmp, (int)sz, port, mactype);
			}
		} else if (type==4)	// IFace IPs
		{
			sprintf(tmp,"%lu.%lu.%lu.%lu",
				vars->name[pdev->root_oid_len],
				vars->name[pdev->root_oid_len+1],
				vars->name[pdev->root_oid_len+2],
				vars->name[pdev->root_oid_len+3]);

			piface=pdev->pnetdev->ifaces[sz];
			if (piface==NULL)
			{
				printf("IP address for unknown interface. IP: %s  ifindex: %d\n",tmp,(int)sz);
			} else
			{
				IPADDR	ipaddr;
				SUBNET	sub;

				printf("Routed interface: ifindex: %d  IP: %s\n",(int)sz,tmp);
    				piface->is_routed=1;

				ipaddr=tmp;
				sub.set(ipaddr.ipaddr.s_addr,0xFFFFFFFF);
				sub.piface=piface;
				subnets.Add(tmp,sub);

			}
		} else if (type==6)	// subnet masks
		{
			IPADDR	ipaddr;
			SUBNET	*psub;

			sprintf(tmp,"%lu.%lu.%lu.%lu",
				vars->name[pdev->root_oid_len],
				vars->name[pdev->root_oid_len+1],
				vars->name[pdev->root_oid_len+2],
				vars->name[pdev->root_oid_len+3]);

			psub=subnets[tmp];
			ipaddr=ntohl(sz);
			if (psub==NULL)
			{
				printf("Subnet mask %s for unknown ip address %s\n",ipaddr.hostname, tmp);
			} else
			{
				psub->set_mask(ipaddr.ipaddr.s_addr);
				ipaddr=ntohl(psub->net);
				printf("Subnet mask for %s (%s): %d bits\n",tmp, ipaddr.hostname, (int)psub->mask);

				subnets.Del(tmp);	// Remove the old one (the ip address)
				subnets.Add(ipaddr.hostname, *psub);	// And add correct one (the network)
			}
		} else if (type==7)	// Vlan types
		{
			int	vlan;

			/*
			 * For each vlan with type==ethernet(1)
			 * except vlan 1
			 * start a new scan
			 */
			vlan=vars->name[pdev->root_oid_len];
			printf("Vlan: %d (type: %d)\n", vlan, (int) sz);
			if (sz==1 && vlan!=1)
			{
				snmp_init_query(pdev->pnetdev, pdev, vlan);
			}
		}
	}
}

/*
	Send a GETNEXT request.
	Return: >0 ok
		0 error
*/
int	snmp_send_getnext(DEVICE_DATA *pdev)
{
	snmp_pdu	*pdu;
	int		ret=1;

	pdu=snmp_pdu_create(SNMP_MSG_GETNEXT);
	snmp_add_null_var(pdu,pdev->cur_oid, pdev->cur_oid_len);

	if (!snmp_send(pdev->psession, pdu))
	{
		snmp_perror("snmp_send");
		snmp_free_pdu(pdu);
		pdev->status=-1;
		ret=0;
	}

	return(ret);
}

/*
	Send a GET request.
	Return: >0 ok
		0 error
*/
int	snmp_send_get(DEVICE_DATA *pdev)
{
	snmp_pdu	*pdu;
	int		ret=1;

	pdu=snmp_pdu_create(SNMP_MSG_GET);
	snmp_add_null_var(pdu,pdev->cur_oid, pdev->cur_oid_len);

	if (!snmp_send(pdev->psession, pdu))
	{
		snmp_perror("snmp_send");
		snmp_free_pdu(pdu);
		pdev->status=-1;
		ret=0;
	}

	return(ret);
}

int	f_callback(int operation, struct snmp_session *sp, int reqid,
			struct snmp_pdu *pdu, void *magic)
{
	DEVICE_DATA	*pdev = (DEVICE_DATA *)magic;
	variable_list	*vars;
	int		done=0;

	printf("----- callback ------\n");
	printf("done=%d pdev->status=%d\n",done,pdev->status);

	if (pdev->status>=100 || pdev->status<0)
		return(0);

	if (operation != NETSNMP_CALLBACK_OP_RECEIVED_MESSAGE ||
		pdu==NULL ||
		pdu->errstat!=SNMP_ERR_NOERROR)
	{
		pdev->status=-1;
		return(1);
	}

	/*
		done:
			0: nothing
			1: the getnext is over (oid mismatch)
			2: oid not increasing
			3: the getnext is over (no more data)
			4: it was a get request and data came
	*/
	for (vars=pdu->variables ; (vars && (done==0 || done==1 || done==2)) ; vars=vars->next_variable)
	{
/*---*/		print_variable(vars->name, vars->name_length, vars);

		if (vars->name_length < pdev->root_oid_len ||
			memcmp(vars->name, pdev->root_oid, pdev->root_oid_len * sizeof(oid)))
		{
			printf("%s(%d): done\n",pdev->pnetdev->ip.p,pdev->status);
			done=1;
//			pdev->status++;
		} else if ((snmp_oid_compare(pdev->cur_oid, pdev->cur_oid_len,
				vars->name, vars->name_length) >=0) && 
				(vmib[(pdev->status-1)/2].type==DO_GETNEXT))	// only check on getnext requests
		{
			printf("%s(%d): OID not increasing!\n", pdev->pnetdev->ip.p, pdev->status);
//			pdev->status++;
			done=2;
		} else if ((vars->type == SNMP_ENDOFMIBVIEW) ||
			(vars->type == SNMP_NOSUCHOBJECT) ||
			(vars->type == SNMP_NOSUCHINSTANCE))
		{
			done=3;
		} else
		{
			switch(pdev->status)
			{
				case 1:	// iface descr
					parse_string(pdev, vars, 1); break;
				case 3:	// iface alias
					parse_string(pdev, vars, 2); break;
				case 5:	// internal iface number
					parse_number(pdev, vars, 1); break;
				case 7:	// mac-addrss type
					parse_number(pdev, vars, 2); break;
				case 9:	// mac-addrss port
					parse_number(pdev, vars, 3); break;
				case 11:	// ARP
					parse_string(pdev, vars, 3); break;
				case 13:	// iface MACs
					parse_string(pdev, vars, 4); break;
				case 15:	// iface IPs
					parse_number(pdev, vars, 4); break;
				case 17:	// IANA interface type
					parse_number(pdev, vars, 5); break;
				case 19:	// Subnet mask
					parse_number(pdev, vars, 6); break;
				case 21:	// VLAN type
					parse_number(pdev, vars, 7); break;
			}
			memcpy((char*)pdev->cur_oid, (char *)vars->name, vars->name_length*sizeof(oid));
			pdev->cur_oid_len=vars->name_length;
		}

	}
	if (done==1 || done==2 || done==3 || done==4)
	{
		pdev->status++;
	} else
	{
		if (!snmp_send_getnext(pdev))
			pdev->status=-1;
	}
//	if (done==0 && pdev->status!=-1)
//		if (!snmp_send_getnext(pdev))
//			pdev->status=-1;
	printf("done=%d pdev->status=%d\n",done,pdev->status);
	printf("----- end of callback ------\n\n");

	return(1);
}

/*
 * Initialize a new snmp scan for a device (or a vlan).
 * Add the new DEVICE_DATA structure to the list of the devices that
 * are being scanned.
 */
void	snmp_init_query(NETDEV *pnetdev, DEVICE_DATA *parent, int vlan)
{
	DEVICE_DATA	*pdev;
	LONGLINE	community, key;

	pdev=new DEVICE_DATA;

	key=pnetdev->ip;
	key+="@";
	key+=(long)vlan;
	devices.Add(key, pdev);

	pdev->pnetdev=pnetdev;
	community=pdev->pnetdev->community;

	/*
	 * Check if this is a vlan.
	 * Don't probe vlans for ifDescr and ifAlias..
	 * Use the appopriate community name.
	 */
	if (parent!=NULL)
	{
		pdev->parent=parent;
		pdev->status=4;
		community+="@";
		community+=(long)vlan;
	} else
	{
		pdev->parent=NULL;
		pdev->status=0;
	}

	printf("init for: %s %s\n", pnetdev->ip.p, community.p);

	snmp_sess_init(&pdev->session);
	pdev->session.version=SNMP_VERSION_1;
	pdev->session.peername=pdev->pnetdev->ip.p;
	pdev->session.community=(u_char *)community.p;
	pdev->session.community_len=community.len;
	pdev->session.callback=f_callback;
	pdev->session.callback_magic=pdev;

	pdev->psession=snmp_open(&pdev->session);
	if (pdev->psession==NULL)
	{
		pdev->status=-1;
		snmp_perror("snmp_open");
	}
}

/*
 * Initalize queries for all known devices
 */
void	snmp_init_queries()
{
	NETDEV		*pnetdev;
	LONGLINE	*pll;

	netdevs.Rewind();
	while ((pll=netdevs.Get_Next())!=NULL)
	{
		pnetdev=netdevs[*pll];

		snmp_init_query(pnetdev);
	}
}

/*
 * Send new snmp requests
 * Called at the begging and at each loop to send new requests for data
 * when needed.
 */
void	snmp_send_requests()
{
	DEVICE_DATA	*pdev;
	LONGLINE	*pll;
	oid		*ret;
	const char	*mib;
	int		ok,idx;

	devices.Rewind();
	while ((pll=devices.Get_Next())!=NULL)
	{
		pdev=*(devices[*pll]);
		if (pdev->status>=0 && pdev->status<100)
		{
			if (pdev->status%2 == 0)
			{
				idx=pdev->status/2;

				ok=0;

				while (!ok)	// find next request to send
				{
					if (vmib[idx].mib==NULL)	// end of requests
					{
						ok=1;
					} else
					{
						// Don't try to get VLANs from vlans
						if (pdev->parent!=NULL && idx==10)
							idx++;
						else
							ok=2;
						printf("parent: %p  idx: %d  ok: %d\n",
								pdev->parent, idx, ok);
					}
				}
				if (ok==1)
				{
					pdev->status=100;	// the end
				} else
				{
					mib=vmib[idx].mib;

					pdev->root_oid_len=MAX_OID_LEN;

					printf("mib=%s\n",mib);
					ret=snmp_parse_oid(mib,
						pdev->root_oid,
						&pdev->root_oid_len);
					if (!ret)
					{
						snmp_perror("snmp_parse_oid");
						pdev->status=-1;
					} else
					{

						memcpy(pdev->cur_oid, pdev->root_oid, sizeof(oid) * pdev->root_oid_len);
						pdev->cur_oid_len=pdev->root_oid_len;

						pdev->status++;

						switch(vmib[idx].type)
						{
							case DO_GETNEXT:
								snmp_send_getnext(pdev);
								break;
							case DO_GET:
								snmp_send_get(pdev);
								break;
						}
					}
				}
			}
		}

		if (pdev->status<0 || pdev->status==100)
		{
			snmp_close(pdev->psession);
		}
	}
}

/*
 * Check if all snmp queries are done and there is nothing more to do.
 */
int	snmp_devices_done()
{
	DEVICE_DATA	*pdev;
	LONGLINE	*pll;

	devices.Rewind();
	while ((pll=devices.Get_Next())!=NULL)
	{
		pdev=*(devices[*pll]);
		if (pdev->status>=0 && pdev->status<100)
			return(0);
	}

	return(1);
}

void	snmp_do_select()
{
	fd_set		fdr;
	int		fds;
	int		block;
	struct timeval	tv;
	int		ret;

	snmp_send_requests();

	while (!snmp_devices_done())
	{
		block=1;
		fds=0;
		FD_ZERO(&fdr);

		snmp_select_info(&fds, &fdr, &tv, &block);
		ret=select(fds, &fdr, NULL, NULL, block ? NULL : &tv);
		if (ret)
			snmp_read(&fdr);
		else
			snmp_timeout();
		snmp_send_requests();
	}
}

void	snmp_doit()
{
	init_snmp("V-SNMP");

	snmp_init_queries();
	snmp_do_select();
}

/*
 * Check all interfaces from all devices and find those that have a monitored
 * device connected to them
 */
void	detect_netdevs()
{
	NETDEV		*pnetdev;
	DEVICE_DATA	*pdev;
	LONGLINE	*pll, *pmac;
	IFACE		*piface;
	u_32		key;

	printf("---------- Netdev check ----------\n");

	devices.Rewind();
	while ((pll=devices.Get_Next())!=NULL)
	{
		//pdev=netdevs[*pll];
		pdev=*(devices[*pll]);
		pnetdev=pdev->pnetdev;
		printf("Checking %s:\n",pll->p);

		pnetdev->ifaces.Rewind();
		while ((key=pnetdev->ifaces.Get_Next())!=NK_INVALID)
		{
			piface=pnetdev->ifaces[key];
			if (piface->iftype!=53 &&	// Don't check for netdevs in VLANs
				strncasecmp(piface->descr.p,"VLAN",4)	// some switches report VLANs as Ethernet
									// and not as type 53 (propVirtual)
				)
			{
				piface->macs.Rewind();

				while ((pmac=piface->macs.Get_Next())!=NULL && !piface->has_netdev)
				{
					if (netdev_macs.Exist(*pmac) && ! pnetdev->dev_macs.Exist(*pmac))
					{
						piface->has_netdev=1;
						printf("Interface %s has netdev %s\n", piface->descr.p, pmac->p);
					}
				}

				piface->arpmacs.Rewind();

				while ((pmac=piface->arpmacs.Get_Next())!=NULL && !piface->has_netdev)
				{
					if (netdev_macs.Exist(*pmac) && ! pnetdev->dev_macs.Exist(*pmac))
					{
						piface->has_netdev=1;
						printf("Interface %s has netdev %s (from arp)\n", piface->descr.p, pmac->p);
					}
				}
			}
		}
	}

	printf("---------- End of netdev check ----------\n");
}

/*
 * Scan all devices and find devices that have at least two interfaces
 * with the same name. 
 * Set has_same_ifdescr on those devices.
 * Change interface descriptions on those devices (append ifindex)
 */
void	check_same_ifdescr()
{
	NETDEV		*pnetdev;
	IFACE		*piface;
	LONGLINE	*pll1;
	LONGLINE	ll;
	STRLIST		sl;
	u_32		key;
	int		done;

	printf("---------- Same IfDescr check ----------\n");

	netdevs.Rewind();
	while ((pll1=netdevs.Get_Next())!=NULL)	// for each device
	{
		sl.Clear();	// this will hold interface names

		pnetdev=netdevs[*pll1];
		pnetdev->ifaces.Rewind();
		done=0;

		while (((key=pnetdev->ifaces.Get_Next())!=NK_INVALID) && !done)	// scan each interface
		{
			piface=pnetdev->ifaces[key];
			if (sl.Exist(piface->descr))	// if there is a same ifdescr
			{
				pnetdev->has_same_ifdescr=1;	// mark this device
				done=1;				// and go for the next device
				printf("Found same ifdescr for device %s (%s) : %s\n",
						pnetdev->name.p, pnetdev->ip.p, piface->descr.p);
			} else sl.Add(piface->descr);	// else append this to the list
		}
	}

	// Now change interface descriptions on those devices
	netdevs.Rewind();
	while ((pll1=netdevs.Get_Next())!=NULL)
	{
		pnetdev=netdevs[*pll1];
		if (pnetdev->has_same_ifdescr)
		{
			pnetdev->ifaces.Rewind();
			while ((key=pnetdev->ifaces.Get_Next())!=NK_INVALID)
			{
				piface=pnetdev->ifaces[key];
				ll=piface->descr;
				ll+="-";
				ll+=(long)piface->ifindex;
				printf("%s(%s): Changed %s -> %s\n",
						pnetdev->name.p, pnetdev->ip.p,
						piface->descr.p, ll.p);
				piface->descr=ll;
			}
		}
	}

	printf("---------- End of same IfDescr check ----------\n");
}

void	update_db()
{
	NETDEV		*pnetdev;
	IFACE		*piface;
	LONGLINE	*pll1;
	u_32		key;
	STRLIST		added_macs;

	added_macs.SetUnique(1);

	db_begin_update();

	netdevs.Rewind();
	while ((pll1=netdevs.Get_Next())!=NULL)	// For each monitored device
	{
		pnetdev=netdevs[*pll1];

		update_netdev(pnetdev->id);

		pnetdev->ifaces.Rewind();

		while ((key=pnetdev->ifaces.Get_Next())!=NK_INVALID)	// For each interface
		{
			piface=pnetdev->ifaces[key];

			// Only process ethernet interfaces
			if ((piface->iftype==6 ||	// ethernetCsmacd
				piface->iftype==26 ||	// ethernet3Mbit
				piface->iftype==62 ||	// Fast Ethernet 100BaseT
				piface->iftype==69 ||	// Fast Ethernet 100BaseFX
				piface->iftype==117 ||	// Gigabit Ethernet
				piface->iftype==53 )	// propVirtual (VLANs)
				)
//				&& strncasecmp(piface->descr.p,"VLAN",4))	// ignore VLANs
			{
				STRLIST	empty;

				// Update interface description etc and set has_netdev if required
				printf("Updating %s -> %s (has_netdev=%d)\n",
					pnetdev->ip.p, piface->descr.p, piface->has_netdev);

				update_iface(pnetdev->id, piface);

				// Insert internal macs no matter if it is a netdev interface
//				update_macs(pnetdev->id, piface->descr.p, piface->internal_macs,
//					empty, 0);
				update_macs(pnetdev->id, piface->descr.p, piface->internal_macs,
					empty, 2);

				// Insert other macs too
				update_macs(pnetdev->id, piface->descr.p, piface->macs,
					pnetdev->dev_macs, piface->has_netdev);

				// Insert ARP macs on non-VLANs
				if (piface->iftype!=53 && strncasecmp(piface->descr.p, "VLAN", 4))
				{
					update_macs(pnetdev->id, piface->descr.p, piface->arpmacs,
						pnetdev->dev_macs, piface->has_netdev);
				}
/*				if (piface->iftype!=53 && strncasecmp(piface->descr.p,"VLAN",4))
				{
					update_macs(pnetdev->id, piface->descr.p, piface->macs,
						pnetdev->dev_macs, piface->has_netdev);
				} else	// insert internal macs to VLAN
				{
//					STRLIST	empty;

					update_macs(pnetdev->id, piface->descr.p, pnetdev->dev_macs,
						empty, 0);
				}*/
			} else
			{
				printf("Ignored interface %s -> %s (type=%d)\n",
					pnetdev->ip.p, piface->descr.p, piface->iftype);
				piface->is_ignored=1;
			}
		}
	}
	update_ips(arp);
	if (config.discover_subnets) insert_subnets();
	db_end_update();
//	update_res_cache();
	do_final();
}

