/*
  Copyright (c) 2002,2003 Stefanos Harhalakis

  This file is part of commonlib.

  netinfo is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  netinfo is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with netinfo; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id: basedb.cc,v 1.15 2004/05/27 17:01:55 v13 Exp $
*/

/*
	02 Mar 2003:
			Added EscapeBytea, UnescapeBytea
	26 Oct 2003:
  			Added GetIsNull
  			Added some missing tests for pgdb==NUL
			Added EscapeString
 */
#include "basedb.h"
#include "config.h"

#include <stdio.h>

#include "v/mem/longline.h"

/*
	Generic db functions
*/

BASEDB::BASEDB()
{
	pgdb=NULL;
	do_debug_stdout=0;
	res=NULL;
}


BASEDB::~BASEDB()
{
	Clear();
}

BASEDB::BASEDB(const BASEDB &db)
{
	this->pgdb=db.pgdb;
	this->est=db.est;
	this->res=NULL;		// this may prevent duplicate PQclear()
	this->do_debug_stdout=db.do_debug_stdout;
}

void	BASEDB::Finish()
{
	if (pgdb!=NULL)
	{
		PQfinish(pgdb);
		pgdb=NULL;
	}
}

int	BASEDB::Connect(const char *path, const char *dbname, const char *user, const char *pass)
{
	LONGLINE	ll;
	int		ret=1;

	ll="";
	if ((dbname!=NULL) && (dbname[0]!=0))
	{
		ll+="dbname=";
		ll+=dbname;
		ll+=" ";
	}
	if ((user!=NULL) && (user[0]!=0))
	{
		ll+="user=";
		ll+=user;
		ll+=" ";
	}
	if (pass!=NULL)
	{
		ll+="password=";
		ll+=pass;
		ll+=" ";
	}
	ll--;
/*	if (pgdb!=NULL) delete(pgdb);
	pgdb=new PgDatabase(ll.p);*/
	if (pgdb!=NULL) Disconnect();

	pgdb=PQconnectdb(ll.p);
//	if ((pgdb!=NULL) && (pgdb->Status()==CONNECTION_BAD))
	if (PQstatus(pgdb)!=CONNECTION_OK)
	{
		Finish();
		ret=0;
	}

	return(ret);
}

void	BASEDB::Clear()
{
	if (res==NULL) return;

	PQclear(res);

	res=NULL;
}

int	BASEDB::Exec(char *cmd)
{
	int		ret;

	if (do_debug_stdout) printf("Exec: %s\n",cmd);
	if (pgdb==NULL)
	{
		ret=-1;
  	} else
	{
		Clear();	// Just to be sure

		res=PQexec(pgdb, cmd);
		est=PQresultStatus(res);
//		est=pgdb->Exec(cmd);

		if (est==PGRES_COMMAND_OK) ret=1;
		else if (est==PGRES_TUPLES_OK) ret=Tuples();
		else ret=0;
	}
	if (do_debug_stdout && ret<=0)
		printf("ret=%d  Error=%s\n", ret, PQerrorMessage(pgdb));

	if (do_debug_stdout) fflush(stdout);

	return(ret);
}

#ifdef HAVE_PQEXECPARAMS
int 	BASEDB::ExecParams (char *cmd, int nparams, const char * const * params,
		const int *plengths, const int *pformats, int res_format)
{
	int		ret;

	if (do_debug_stdout) printf("Exec: %s\n", cmd);

	if (pgdb==NULL)
	{
		ret=-1;
	} else
	{
		Clear();

		res=PQexecParams(pgdb, cmd, nparams, NULL, params, plengths, pformats, res_format);
		est=PQresultStatus(res);

		if (est==PGRES_COMMAND_OK) ret=1;
		else if (est==PGRES_TUPLES_OK) ret=Tuples();
		else ret=0;

	}

	if (do_debug_stdout && ret<=0)
		printf("ret=%d  Error=%s\n", ret, PQerrorMessage(pgdb));

	if (do_debug_stdout) fflush(stdout);


	return(ret);
}
#else
int 	BASEDB::ExecParams (char *cmd, int nparams, const char * const * params,
		const int *plengths, const int *pformats, int res_formats)
{
	return(-1); // Stub
}
#endif

void	BASEDB::Disconnect()
{
	if (pgdb!=NULL)
	{
		Clear();
		Finish();
	}
}

int	BASEDB::Commit()
{
	return(Exec("COMMIT WORK"));
}

const char *BASEDB::GetValue(int tup_num, int field_num)
{
	const char	*p;

	if (pgdb==NULL) return(NULL);

	p=PQgetvalue(res, tup_num, field_num);

	return(p);
}

const char *BASEDB::GetValue(int tup_num, const char *field_name)
{
	int	field_num;

	field_num=PQfnumber(res, field_name);

	return(GetValue(tup_num, field_num));
}

int	BASEDB::GetIsNull(int tup_num, int field_num)
{
	int	ret;

	if (pgdb==NULL) return(-1);

	ret=( PQgetisnull(res, tup_num, field_num) ? 1 : 0 );

	return(ret);
}

int	BASEDB::GetIsNull(int tup_num, const char *field_name)
{
	int	field_num;

	field_num=PQfnumber(res, field_name);

	return(GetIsNull(tup_num, field_num));
}

int	BASEDB::Tuples()
{
	int	ret;

	if (pgdb==NULL) return(-1);

	ret=PQntuples(res);

	return(ret);
}

int	BASEDB::CmdTuples()
{
	int 	    ret;
	const char	*p;

	if (pgdb==NULL) return(-1);

	p=PQcmdTuples(res);

	ret=( p == NULL ? 0 : strtoll(p, NULL, 10) );

	return(ret);
}

int	BASEDB::CallFunction(const char *func)
{
	LONGLINE	ll;
	int		ret;

	ll="SELECT ";
	ll+=func;
	ll+="()";

	ret=Exec(ll.p);

	return(ret);
}

int	BASEDB::DeclareCursor(const char *cursor, const char *select)
{
	LONGLINE	ll;
	int		ret;

	ll="DECLARE ";
	ll+=cursor;
	ll+=" CURSOR FOR ";
	ll+=select;

	ret=Exec(ll.p);

	return(ret);
}

int	BASEDB::Begin()
{
	LONGLINE	ll;
	int		ret;

	ll="BEGIN";

	ret=Exec(ll.p);

	return(ret);
}

int	BASEDB::End()
{
	LONGLINE	ll;
	int		ret;

	ll="END";

	ret=Exec(ll.p);

	return(ret);
}

int	BASEDB::Fetch(const char *cursor)
{
	LONGLINE	ll;
	int		ret;

	ll="FETCH FROM ";
	ll+=cursor;

	ret=Exec(ll.p);

	return(ret);
}

int	BASEDB::LockTable(const char *tName)
{
	LONGLINE	ll;
	int		ret;

	ll="LOCK TABLE ";
	ll+=tName;
	ll+=" IN ACCESS EXCLUSIVE MODE";

	ret=Exec(ll.p);
	
	return(ret);
}

int	BASEDB::LockTableWrite(const char *tName)
{
	LONGLINE	ll;
	int		ret;

	ll="LOCK TABLE ";
	ll+=tName;
	ll+=" IN SHARE MODE";

	ret=Exec(ll.p);
	
	return(ret);
}

int	BASEDB::ReindexTable(const char *tName)
{
	LONGLINE	ll;
	int		ret;

	ll="REINDEX TABLE ";
	ll+=tName;
	ll+=" FORCE";

	Begin();
	LockTableWrite(tName);

	ret=Exec(ll.p);

	End();

	return(ret);
}

INT8	BASEDB::GetSeqCurrent(const char *sName)
{
	LONGLINE	ll;
	INT8		ret;
	const char	*p;

	ll="SELECT CURRVAL('";
	ll+=sName;
	ll+="')";

	if (Exec(ll.p))
	{
		p=GetValue(0,0);
		if (p)
		{
#if (SIZEOF_UNSIGNED_LONG_LONG == 8)
			ret=strtoll(p, NULL, 10);
#else
			ret=strtol(p, NULL, 10);
#endif
		} else ret=-1;
	}

	return(ret);
}

INT8	BASEDB::GetSeqNext(const char *sName)
{
	LONGLINE	ll;
	INT8		ret;
	const char	*p;

	ll="SELECT NEXTVAL('";
	ll+=sName;
	ll+="')";

	if (Exec(ll.p))
	{
		p=GetValue(0,0);
		if (p)
		{
#if (SIZEOF_UNSIGNED_LONG_LONG == 8)
			ret=strtoll(p, NULL, 10);
#else
			ret=strtol(p, NULL, 10);
#endif
		} else ret=-1;
	}

	return(ret);
}

LONGLINE BASEDB::EscapeString(const char *str)
{
	LONGLINE	ret;
	size_t		len, slen;
	char		*p;

	slen=strlen(str);

	p=(char *)malloc(slen*2 + 1);

	len=PQescapeString(p, str, slen);

	ret=p;
	free(p);

	return(ret);
}

/*
 * Escape a binary buffer. Return the escaped string in a LONGLINE
 */
LONGLINE BASEDB::EscapeBytea(unsigned char *buf, size_t len)
{
	unsigned char	*p;
	size_t		dlen;
	LONGLINE	ll;

	p=PQescapeBytea(buf, len, &dlen);

	ll=(char *)p;

#ifdef HAVE_PQFREEMEM
	if (p!=NULL) PQfreemem(p);
#else
	if (p!=NULL) free(p);
#endif

	return(ll);
}

/*
 * Unescape a binary buffer. Return a buffer that should be free()ed. Store the
 * size of the buffer in ret_len
 */
unsigned char *BASEDB::UnescapeBytea(unsigned char *buf, size_t *ret_len)
{
	unsigned char	*p, *ret;
	size_t		dlen;

	p=PQunescapeBytea(buf, &dlen);

	if (p!=NULL)
	{
		ret=(unsigned char *)malloc(dlen);
		memcpy(ret, p, dlen);
#ifdef HAVE_PQFREEMEM
		PQfreemem(p);
#else
		free(p);
#endif
	} else
	{
		dlen=0;
		ret=(unsigned char *)strdup("");
	}

	if (ret_len!=NULL) *ret_len=dlen;

	return(ret);
}

/*
 * Check if a value returned by a query is true
 */
int	BASEDB::isTrue(const char *value)
{
	if (!strcasecmp(value,"t") ||
		!strcasecmp(value,"true"))
		return(1);

	return(0);
}

/*
 * Same as above for false
 */
int	BASEDB::isFalse(const char *value)
{
	if (!strcasecmp(value,"f") ||
		!strcasecmp(value,"false"))
		return(1);

	return(0);
}

/*
 * Same as above but also fetch the result
 * This will destroy the pointer to the last result (!)
 */
int	BASEDB::isTrue(int tup_num, int field_num)
{
	const char	*p;

	p=GetValue(tup_num, field_num);
	if (p==NULL) return(0);

	return(isTrue(p));
}

int	BASEDB::isFalse(int tup_num, int field_num)
{
	const char	*p;

	p=GetValue(tup_num, field_num);
	if (p==NULL) return(0);

	return(isFalse(p));
}

