/*
  Copyright (c) 2002,2003,2004 Stefanos Harhalakis

  This file is part of netinfo.

  netinfo is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  netinfo is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with netinfo; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id: db.cc,v 1.11 2004/06/07 19:28:34 v13 Exp $
*/

#define _DB_CC
#include "db.h"
#include "conf.h"

#include "v/mem/longline.h"
#include "v/misc/keytable2.cc"
#include "v/misc/nkeytable.cc"
#include "v/socket/ipaddr.h"

#include <time.h>

#define TBL_NETDEVS	"netdevs"
#define TBL_IFACES	"interfaces"
#define	TBL_MACS	"macs"
#define TBL_ALLMACS	"allmacs"
#define TBL_IPS		"ips"
#define TBL_RES_CACHE	"res_cache"

#define MAINTENANCE_INTERVAL	43000		// seconds between two vaccum/reindex etc
#define FULLVACUUM_INTERVAL	7*24*3600	// VACUUM FULL once a week

//#define RESOLVER_EXPIRE		12 * 3600	// expire resolver cache every 12h

#define	FL_ROUTED	0x10
#define FL_CHANGED	0x40
#define FL_ADDED	0x80

#define FL_MAC_INTERNAL	0x100

BASEDB	db;
time_t	ts=time(NULL);

int	connect_db()
{
	int	ret;
	
	ret=db.Connect(NULL,conf.dbname.p, conf.dbuser.p, conf.dbpass.p);

	return(ret);
}

void	close_db()
{
	db.Disconnect();
}

LONGLINE escape_string(const char *str)
{
	LONGLINE	ret;
	const char	*p;

	p=str;
	
	while (*p)
	{
		if (*p=='\'' || *p=='\\' || *p=='%')
			ret+='\\';
		ret+=*p;
		p++;
	}

	return(ret);
}

#define	CURS	"curs1"
int	get_netdevs()
{
	char		tmp[1000];
	const char	*p;
	NETDEV		ndev;
	LONGLINE	ip;
	int		ret;
	int		done=0;

	sprintf(tmp, "SELECT ip, community, name, id FROM %s",
		TBL_NETDEVS);

	netdevs.ClearTable();

	db.Begin();
	ret=db.DeclareCursor(CURS,tmp);
	if (ret<=0)
		done=1;
	while (!done && db.Fetch(CURS)>0)
	{
		p=db.GetValue(0,0);
		if (p!=NULL)
		{
			ip=p;

			ndev.ip=ip;

			p=db.GetValue(0,1);
			ndev.community=p;

			p=db.GetValue(0,2);
			ndev.name=p;

			p=db.GetValue(0,3);
			ndev.id=atoi(p);

			netdevs.Add(ip,ndev);
		} else done=1;
	}
	db.End();

	return(netdevs.count);
}

void	get_res_cache()
{
	char		tmp[1000];
	const char	*p;
	LONGLINE	ip;
	LONGLINE	hname;
	int		ret;
	int		done=0;

	sprintf(tmp, "SELECT ip, hostname FROM %s "
			"WHERE (EXTRACT(EPOCH FROM CURRENT_TIMESTAMP) - ts) < %u",
		TBL_RES_CACHE, config.resolver_expire_time);

	res_cache.ClearTable();

	db.Begin();
	ret=db.DeclareCursor(CURS,tmp);
	if (ret<=0)
		done=1;
	while (!done && db.Fetch(CURS)>0)
	{
		p=db.GetValue(0,0);
		if (p!=NULL)
		{
			ip=p;

			p=db.GetValue(0,1);
			hname=p;

			res_cache.Add(ip,hname);
		} else done=1;
	}
	db.End();
}

#undef CURS


/*void	update_iface(int netdev_id, const char *ifdescr,
			const char *ifalias, int ifindex, int has_netdev, int is_routed)*/
void	update_iface(int netdev_id, IFACE *piface)
{
	const char	*p;
	char		buf[1000];
	int		ret;
	int		flags,n,id=0;
	LONGLINE	ll1, esc_alias;
	time_t		last_had_netdev;

	esc_alias=escape_string(piface->alias.p);

	// Check if entry exist
	sprintf(buf,"SELECT last_had_netdev, flags, alias, ifindex, id FROM %s WHERE "
			"netdev_id='%d' AND description='%s'",
			TBL_IFACES, netdev_id, piface->descr.p);
	ret=db.Exec(buf);
	if (ret>0)	// exist
	{
		p=db.GetValue(0,0); if (p!=NULL) last_had_netdev=atoi(p); else last_had_netdev=0;
		p=db.GetValue(0,1); if (p!=NULL) flags=atoi(p); else flags=0;
		ll1=db.GetValue(0,2);
		p=db.GetValue(0,3); if (p!=NULL) n=atoi(p); else n=0;
		p=db.GetValue(0,4); if (p!=NULL) id=atoi(p);	// p cannot be NULL

		if (ll1!=piface->alias || n!=piface->ifindex ||
			(last_had_netdev==0 && piface->has_netdev==1) ) flags |= FL_CHANGED;

		flags &= ~(FL_ROUTED);

		if (piface->is_routed) flags|=FL_ROUTED;

		if (piface->has_netdev) last_had_netdev=ts;
		else if (last_had_netdev) piface->has_netdev=1;

		// update it
		sprintf(buf,"UPDATE %s SET alias='%s', ifindex='%d', "
				"last_had_netdev='%d', ts='%d', flags='%d' "
				"WHERE id='%d'",
				TBL_IFACES, esc_alias.p, piface->ifindex,
				(int)last_had_netdev, (int)ts, flags,
				id);
		db.Exec(buf);
	} else	// doesn't exist
	{
		// delete entries with same ifindex (if any)
		sprintf(buf,"DELETE FROM %s WHERE "
				"netdev_id='%d' AND ifindex='%d'",
				TBL_IFACES, netdev_id, piface->ifindex);

		db.Exec(buf);

		if (piface->has_netdev) last_had_netdev=ts; else last_had_netdev=0;

		flags=FL_ADDED;

		if (piface->is_routed) flags |= FL_ROUTED;

		// insert new entry
		sprintf(buf,"INSERT INTO %s(netdev_id,description,alias,"
				"ifindex, last_had_netdev, ts, flags) VALUES"
				"('%d','%s','%s','%d','%d','%d','%u')",
				TBL_IFACES, netdev_id, piface->descr.p, esc_alias.p,
				piface->ifindex, (int)last_had_netdev, (int)ts, flags);
		db.Exec(buf);
	}
}

void	update_netdev	(int netdev_id)
{
	char		buf[1000];

	sprintf(buf,"UPDATE %s SET ts='%d' WHERE id='%d'",
			TBL_NETDEVS, (int)ts, netdev_id);
	db.Exec(buf);
}

void	update_mac(int ifid, const char *mac, int is_internal=0)
{
	char	buf[1000];
	int	ret;
	int	fl;
	
//	sprintf(buf,"UPDATE %s SET ts='%d' WHERE mac='%s'",
//			TBL_MACS, (int)ts, mac);
	// Update location changes
	if (!is_internal)
	{
		sprintf(buf,"UPDATE %s SET ifid='%d', ts='%d', flags=(flags & ~(X'100'::INTEGER)) WHERE mac='%s'",
				TBL_MACS, ifid, (int)ts, mac);
	} else
	{
		sprintf(buf,"UPDATE %s SET ifid='%d', ts='%d', flags=(flags | X'100'::INTEGER) WHERE mac='%s'",
				TBL_MACS, ifid, (int)ts, mac);
	}
	ret=db.Exec(buf);
	if (db.CmdTuples()==0)
	{
		fl=FL_ADDED;
		if (is_internal) fl|=FL_MAC_INTERNAL;
		sprintf(buf,"INSERT INTO %s(mac, ifid, ts, flags) "
				"VALUES('%s','%d','%d','%d') ",
				TBL_MACS,
				mac, ifid, (int)ts, fl);
		db.Exec(buf);
	}
}

void	update_allmac(int ifid, const char *mac)
{
	char	buf[1000];
	int	ret;

	sprintf(buf,"UPDATE %s SET ts='%d' WHERE mac='%s' AND ifid='%d'",
			TBL_ALLMACS, (int)ts, mac, ifid);
	ret=db.Exec(buf);
	if (db.CmdTuples()==0)
	{
		sprintf(buf,"INSERT INTO %s(mac, ifid, ts) "
				"VALUES('%s','%d','%d') ",
				TBL_ALLMACS,
				mac, ifid, (int)ts);
		db.Exec(buf);
	}
}

int	get_ifid(int netdev_id, const char *ifdescr)
{
	char		buf[1000];
	int		ifid, ret;
	const char	*p;

	sprintf(buf,"SELECT id FROM %s WHERE netdev_id='%d' AND description='%s'",
			TBL_IFACES, netdev_id, ifdescr);

	ret=db.Exec(buf);
	if (ret>0)
	{
		p=db.GetValue(0,0);
		ifid=atoi(p);
	} else ifid=-1;

	return(ifid);
}

// netdev_if:
//	1: Only update TBL_ALLMACS
//	0: Update TBL_ALLMACS and TBL_MACS
//	2: Like 0 but consider them as internal MACs
void	update_macs	(int netdev_id, const char *ifdescr, STRLIST &macs,
			STRLIST &exclude_macs, int netdev_if)
{
	int		ifid;
	LONGLINE	*pll;

	ifid=get_ifid(netdev_id, ifdescr);

	if (ifid==-1) return;

	macs.Rewind();
	while ((pll=macs.Get_Next())!=NULL)
	{
		if (!exclude_macs.Exist(pll->p))
		{
			update_allmac(ifid, pll->p);
			if (!netdev_if) update_mac(ifid, pll->p);
			else if (netdev_if==2) update_mac(ifid, pll->p, 1);
		}
	}
}

void	update_ip	(const char *mac, const char *ip)
{
	char	buf[1000];
	int	ret;

	sprintf(buf,"UPDATE %s SET ts='%d', mac='%s' WHERE ip='%s'",
			TBL_IPS, (int)ts, mac, ip);

	ret=db.Exec(buf);

	if (db.CmdTuples()==0)
	{
		sprintf(buf,"INSERT INTO %s(ip,mac,ts) VALUES('%s','%s','%d')",
				TBL_IPS, ip, mac, (int)ts);
		db.Exec(buf);
	}
}

void	update_ips	(KEYTABLE<STRLIST> &arp)
{
	STRLIST		*plist;
	LONGLINE	*pll,*pll2;

	arp.Rewind();
	while ((pll=arp.Get_Next())!=NULL)
	{
		plist=arp[*pll];

		plist->Rewind();
		while ((pll2=plist->Get_Next())!=NULL)
		{
			// pll: MAC
			// pll2: IP

			update_ip(pll->p, pll2->p);
		}
	}
}

void	update_res_cache_one(const char *ip, const char *hname)
{
	char	buf[1000];
	int	ret;

	sprintf(buf,"UPDATE %s SET ts='%d', hostname='%s' WHERE ip='%s'",
			TBL_RES_CACHE, (int)ts, hname, ip);

	ret=db.Exec(buf);

	if (db.CmdTuples()==0)
	{
		// Check if there is an ip entry for this ip.
		// IPs that were filtered out as strange IPs or were expired
		// used to cause referential integrity errors (25 Oct 2003)
		sprintf(buf,"SELECT ip FROM %s WHERE ip='%s'",
				TBL_IPS, ip);
		ret=db.Exec(buf);
	
		if (ret>0)
		{
			sprintf(buf,"INSERT INTO %s(ip,hostname,ts) VALUES('%s','%s','%d')",
					TBL_RES_CACHE, ip, hname, (int)ts);
			db.Exec(buf);
		}
	}
}

void	update_res_cache()
{
	LONGLINE	*pll1,*pll2;

	res_new.Rewind();

	while((pll1=res_new.Get_Next())!=NULL)
	{
		pll2=res_new[*pll1];

		update_res_cache_one(pll1->p, pll2->p);
	}
}

void	insert_subnet(u_32 net, u_8 mask)
{
	IPADDR	ipaddr;
	char	tmp[100], buf[1000];
	int	ret;

	ipaddr=ntohl(net);

	sprintf(tmp,"%s/%d", ipaddr.hostname, (int)mask);
	sprintf(buf,"SELECT ip FROM subnets WHERE ip='%s'",tmp);

	ret=db.Exec(buf);
	if (!ret)	// not exist
	{
		sprintf(buf,"INSERT INTO subnets(ip,flags,description)"
				" VALUES('%s','%d','auto discovered')",
				tmp, 0x10 | 0x80);	// auto-discovered, added
		db.Exec(buf);
	}
}

void	insert_subnets()
{
	SUBNET		*psub;
	LONGLINE	*pll;

	subnets.Rewind();
	while ((pll=subnets.Get_Next())!=NULL)
	{
		psub=subnets[*pll];

		if (!psub->piface->is_ignored && psub->mask<=config.subnet_max_mask)
			insert_subnet(psub->net, psub->mask);
	}
}


void	set_last_run()
{
	char	buf[1000];

	sprintf(buf,"DELETE FROM config WHERE name='last_run'");
	db.Exec(buf);
	sprintf(buf,"INSERT INTO config(name,value) VALUES ('last_run','%d')",
			(int)time(NULL));
	db.Exec(buf);
}

// Get a string config entry
LONGLINE get_config_str(const char *name)
{
	char		buf[1000];
	int		ret;
	LONGLINE	ll;

	sprintf(buf,"SELECT value FROM config WHERE name='%s'", name);

	ret=db.Exec(buf);
	if (ret>0)
	{
		ll=db.GetValue(0,0);
	}

	return(ll);
}

// Get a long config entry (-1 on error)
long get_config_long(const char *name)
{
	LONGLINE	ll;
	long		l;

	ll=get_config_str(name);
	if (ll!="")
	{
		l=strtoul(ll.p,NULL,10);
	} else l=-1;

	return(l);
}

void	get_config()
{
	long		l;
	time_t		t;
	LONGLINE	ll;

	l=get_config_long("last_maintenance");
	if (l<0) t=0; else t=(time_t)l;
	config.last_maintenance=t;

	l=get_config_long("last_vacuumfull");
	if (l<0) t=0; else t=(time_t)l;
	config.last_vacuumfull=t;

	ll=get_config_str("discover_subnets");
	if (ll=="" || !strcasecmp(ll.p,"yes")) l=1;
	else l=0;
	config.discover_subnets=l;

	ll=get_config_str("filter_out_strange_ips");
	if (ll=="" || strcasecmp(ll.p,"yes")) l=0;
	else l=1;
	config.filter_out_strange_ips=l;

	l=get_config_long("subnet_max_mask");
	if (l<0) l=30;
	config.subnet_max_mask=l;

	l=get_config_long("resolver_expire_time");
	if (l<0) l=43200;
	config.resolver_expire_time=l;
}

void	set_last_maintenance()
{
	char	buf[1000];

	sprintf(buf,"DELETE FROM config WHERE name='last_maintenance'");
	db.Exec(buf);
	sprintf(buf,"INSERT INTO CONFIG(name,value) VALUES('last_maintenance','%d')",
			(int)time(NULL));
	db.Exec(buf);
}

void    set_last_vacuumfull()
{
	char	buf[1000];

	sprintf(buf,"DELETE FROM config WHERE name='last_vacuumfull'");
	db.Exec(buf);
	sprintf(buf,"INSERT INTO CONFIG(name,value) VALUES('last_vacuumfull','%d')",
			(int)time(NULL));
	db.Exec(buf);
}

void	do_reindex()
{
	db.ReindexTable("allmacs");
	db.ReindexTable("assoc_ip2mac");
	db.ReindexTable("assoc_ip2iface");
	db.ReindexTable("assoc_mac2iface");
	db.ReindexTable("blacklist");
	db.ReindexTable("config");
	db.ReindexTable("history");
	db.ReindexTable("interfaces");
	db.ReindexTable("ip_nets");
	db.ReindexTable("ips");
	db.ReindexTable("macs");
	db.ReindexTable("monitor_filters");
	db.ReindexTable("netdevs");
	db.ReindexTable("res_cache");
	db.ReindexTable("subnets");
/*	db.Exec("REINDEX TABLE allmacs FORCE");
	db.Exec("REINDEX TABLE assoc_ip2mac FORCE");
	db.Exec("REINDEX TABLE assoc_ip2iface FORCE");
	db.Exec("REINDEX TABLE assoc_mac2iface FORCE");
	db.Exec("REINDEX TABLE blacklist FORCE");
	db.Exec("REINDEX TABLE config FORCE");
	db.Exec("REINDEX TABLE history FORCE");
	db.Exec("REINDEX TABLE interfaces FORCE");
	db.Exec("REINDEX TABLE ip_nets FORCE");
	db.Exec("REINDEX TABLE ips FORCE");
	db.Exec("REINDEX TABLE macs FORCE");
	db.Exec("REINDEX TABLE monitor_filters FORCE");
	db.Exec("REINDEX TABLE netdevs FORCE");
	db.Exec("REINDEX TABLE res_cache FORCE");
	db.Exec("REINDEX TABLE subnets FORCE");*/
//	db.Exec("REINDEX TABLE vendors FORCE");
}

void	do_final()
{
	time_t	t;
	int	to_vacuumfull=0;
	int	to_maintenance=0;

	db.Begin();
	db.LockTableWrite("config");

	set_last_run();

	// Check if it is time for VACUUM FULL
	t=time(NULL)-config.last_vacuumfull;
	if (t>FULLVACUUM_INTERVAL)
	{
		to_vacuumfull=1;
		set_last_vacuumfull();
	}

	// Check if it is time for maintenance
	t=time(NULL)-config.last_maintenance;
	if (t>MAINTENANCE_INTERVAL)
	{
		to_maintenance=1;
		set_last_maintenance(); // Set this first to avoid dupplicate-invocation problems and release the lock
	}
	db.End();

	db.Exec("SELECT log_new_interfaces()");
	db.Exec("SELECT log_new_subnets()");
	db.Exec("SELECT set_history()");
	db.Exec("SELECT clear_autolock()");
	db.Exec("SELECT set_blacklist()");
	db.Exec("SELECT set_autolock()");

	if (config.filter_out_strange_ips)
		db.Exec("SELECT filter_ips()");

	if (to_maintenance)
	{
		db.Exec("SELECT do_expire()");
		do_reindex();

		if (!to_vacuumfull)
			db.Exec("VACUUM ANALYZE");
	}

	if (to_vacuumfull)
	{
		db.Exec("VACUUM FULL ANALYZE");
	}
}

void	db_begin_update()
{
	db.Begin();
	db.LockTableWrite("config");
	db.LockTableWrite("netdevs");
}

void	db_end_update()
{
	db.End();
}
