/*
  Copyright (c) 2003 Stefanos Harhalakis

  This file is part of netmap.

  netmap is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  netmap is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with netmap; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#define _DB_CC
#include "db.h"
#include "conf.h"

#include "v/mem/longline.h"
#include "v/misc/keytable2.cc"
#include "v/misc/nkeytable.cc"

#include <time.h>

#define TBL_NETDEVS	"netdevs"
#define TBL_IFACES	"interfaces"
#define	TBL_IFACCT	"ifacct"
#define TBL_MEMACCT	"memacct"
#define TBL_CPUACCT	"cpuacct"

#define MAINTENANCE_INTERVAL	43000	// seconds between two vaccum/reindex etc
#define FULLVACUUM_INTERVAL	7*24*3600	// VACUUM FULL once a week

int	intervals[]={1,150,1050,4200,54750, 27450};

BASEDB	db;

int	connect_db()
{
	int	ret;
	
	ret=db.Connect(NULL,conf.dbname.p, conf.dbuser.p, conf.dbpass.p);
	
	return(ret);
}

void	close_db()
{
	db.Disconnect();
}

LONGLINE escape_string(const char *str)
{
	LONGLINE        ret;
	const char      *p;

	p=str;

	while (*p)
	{
		if (*p=='\'' || *p=='\\' || *p=='%')
			ret+='\\';
		ret+=*p;
		p++;
	}

	return(ret);
}


#define	CURS	"curs1"
int	get_netdevs()
{
	char		tmp[1000];
	const char	*p;
	NETDEV		ndev;
	LONGLINE	ip;
	int		ret;
	int		done=0;
	u_32		u1;
	
	sprintf(tmp, "SELECT ip, community, name, flags FROM %s",
		TBL_NETDEVS);
		
	netdevs.ClearTable();
	
	db.Begin();
	ret=db.DeclareCursor(CURS,tmp);
	if (ret<=0)
		done=1;
	while (!done && db.Fetch(CURS)>0)
	{
		p=db.GetValue(0,0);
		if (p!=NULL)
		{
			ip=p;
		
			ndev.ip=ip;
				
			p=db.GetValue(0,1);
			ndev.community=p;
			
			p=db.GetValue(0,2);
			ndev.name=p;
			
			p=db.GetValue(0,3);
			if (p==NULL) u1=0; else u1=strtoul(p,NULL,10);
			ndev.flags=u1;
			
			ndev.uptime=0;
			ndev.cpu1=ndev.cpu2=ndev.cpu3=0;
			ndev.mem_used=ndev.mem_free=0;
			
			netdevs.Add(ip,ndev);
		} else done=1;
	}
	db.End();
	
	return(netdevs.count);
}
#undef CURS

//void	update_iface(const char *netdev_ip, const char *ifdescr,
//			const char *ifalias, int ifindex, u_32 cur_in, u_32 cur_out, u_32 speed,
//			time_t ts)
void	update_iface(const char *netdev_ip, IFACE *piface)
{
	char		buf[1000];
	int		ret;
	LONGLINE	last_in, last_out, last_ts, esc_alias;
	
	// Check if entry exist
	sprintf(buf,"SELECT cur_in, cur_out, cur_ts FROM %s WHERE "
			"netdev_ip='%s' AND description='%s'",
			TBL_IFACES, netdev_ip, piface->descr.p);

	esc_alias=escape_string(piface->alias.p);
			
	ret=db.Exec(buf);
	if (ret>0)	// exist
	{
		last_in=db.GetValue(0,0);
		last_out=db.GetValue(0,1);
		last_ts=db.GetValue(0,2);
		
		piface->last_bytes_in=strtoul(last_in.p, NULL, 10);
		piface->last_bytes_out=strtoul(last_out.p, NULL, 10);
		piface->last_ts=strtoul(last_ts.p, NULL, 10);
		
		// update it
		sprintf(buf,"UPDATE %s SET ifindex='%d', alias='%s', "
				"cur_in='%u', cur_out='%u', cur_ts='%u', "
				"last_in='%s', last_out='%s', last_ts='%s', "
				"speed='%u' "
				"WHERE netdev_ip='%s' AND description='%s'",
				TBL_IFACES, piface->index, esc_alias.p,
				piface->bytes_in, piface->bytes_out, piface->ts,
				last_in.p, last_out.p, last_ts.p,
				piface->speed,
				netdev_ip, piface->descr.p);

		db.Exec(buf);
	} else	// doesn't exist
	{
		piface->last_bytes_in=piface->bytes_in;
		piface->last_bytes_out=piface->bytes_out;
		piface->last_ts=piface->ts-10;
		// delete entries with same ifindex (if any)
		sprintf(buf,"DELETE FROM %s WHERE "
				"netdev_ip='%s' AND ifindex='%d'",
				TBL_IFACES, netdev_ip, piface->index);

		db.Exec(buf);
		
		// insert new entry
		sprintf(buf,"INSERT INTO %s(netdev_ip,description,alias,"
				"ifindex,cur_in,cur_out,cur_ts, speed) VALUES"
				"('%s','%s','%s','%d','%u','%u','%u','%u')",
				TBL_IFACES,netdev_ip,piface->descr.p,
				esc_alias.p, piface->index,
				piface->bytes_in,piface->bytes_out,
				piface->ts,piface->speed);
		db.Exec(buf);
	}
}

void	update_ifacct_oneslot(const char *netdev_ip, const char *ifdescr,
			u_32 bps_in, u_32 bps_out,
			u_32 actype, time_t dbts)
{
	char		buf[1000];
	int		ret;
	u_32		last_bps_in, last_bps_out;
	u_32		new_bps_in, new_bps_out;
	u_32		ocount;
	LONGLINE	ll;

	// Check if entry exist
	sprintf(buf,"SELECT bps_in, bps_out, count FROM %s WHERE "
			"netdev_ip='%s' AND ifdescr='%s' AND "
			"actype='%d' AND ts='%u'",
			TBL_IFACCT, 
			netdev_ip, ifdescr, 
			actype, dbts);

	ret=db.Exec(buf);
	if (ret>0)	// exist
	{
		if (bps_in!=BPS_INVALID && bps_out!=BPS_INVALID)
		{
			ll=db.GetValue(0,0); last_bps_in=strtoul(ll.p, NULL, 10);
			ll=db.GetValue(0,1); last_bps_out=strtoul(ll.p, NULL, 10);
			ll=db.GetValue(0,2); ocount=strtoul(ll.p, NULL, 10);

			new_bps_in=(u_32)((1.0*last_bps_in*ocount)+bps_in)/(ocount+1);
			new_bps_out=(u_32)((1.0*last_bps_out*ocount)+bps_out)/(ocount+1);

			ocount++;
			sprintf(buf,"UPDATE %s SET "
					"bps_in='%u', bps_out='%u', count='%u' "
					"WHERE netdev_ip='%s' AND ifdescr='%s' AND "
					"actype='%u' AND ts='%u'",
					TBL_IFACCT,
					new_bps_in, new_bps_out, ocount,
					netdev_ip, ifdescr, 
					actype, dbts);
		}
	} else	// doesn't exist
	{
		if (bps_in==BPS_INVALID) bps_in=0;
		if (bps_out==BPS_INVALID) bps_out=0;

		// insert new entry
		sprintf(buf,"INSERT INTO %s(netdev_ip, ifdescr, actype, ts,"
				"bps_in, bps_out, count) VALUES"
				"('%s','%s','%u','%u',"
				"'%u', '%u', '%u')",
				TBL_IFACCT,
				netdev_ip, ifdescr, actype, dbts,
				bps_in, bps_out, 1);
	}
	db.Exec(buf);

}

/*
 * max: The max bandwidth of this interface (to calculate invalid values)
 */
//void	update_ifacct(const char *netdev_ip, const char *ifdescr,
//			u_32 cur_in, u_32 cur_out,
//			u_32 last_in, u_32 last_out,
//			u_32 bps_max,
//			time_t last_ts, time_t cur_ts)
void	update_ifacct(const char *netdev_ip, IFACE *piface)
{
	u_32		bps_in, bps_out;
	int		n;
	time_t		cur_dbts, last_dbts;

	if (piface->ts==0)
		return;

	bps_in=(piface->bytes_in - piface->last_bytes_in)*8/(piface->ts - piface->last_ts);
	bps_out=(piface->bytes_out - piface->last_bytes_out)*8/(piface->ts - piface->last_ts);

	if (bps_in>piface->speed) bps_in=BPS_INVALID;
	if (bps_out>piface->speed) bps_out=BPS_INVALID;
	for (n=1;n<=5;n++)
	{
		int	upd_type;	// 1: one slot, 2: filling
		cur_dbts=piface->ts/intervals[n];
		cur_dbts*=intervals[n];

		last_dbts=piface->last_ts/intervals[n];
		last_dbts*=intervals[n];

		if (cur_dbts==last_dbts)	// same slot
		{
			upd_type=1;
		} else if (cur_dbts==last_dbts+1)	// new 'next' slot
		{
			upd_type=1;
		} else if (cur_dbts>last_dbts)	// slot filling required
		{
			if ((cur_dbts-last_dbts)>(intervals[n]*10))
				upd_type=1;
			else
				upd_type=2;
		} else upd_type=0;

		printf("upd_type=%d  bps_in=%d  bps_out=%d  cur_dbts=%d  last_dbts=%d n=%d\n",
				upd_type, bps_in, bps_out, cur_dbts, last_dbts, n);
		if (upd_type==1)
		{
			update_ifacct_oneslot(netdev_ip, piface->descr.p, bps_in, bps_out, n, cur_dbts);
		} else if (upd_type==2)
		{
			update_ifacct_oneslot(netdev_ip, piface->descr.p, bps_in, bps_out, n, cur_dbts);
//			for (m=last_dbts+1; m<=cur_dbts; m+=intervals[n])
//				update_ifacct_oneslot(netdev_ip, piface->descr.p, bps_in, bps_out, n, m);
		}
	}
}

void	update_netdev	(const char *netdev_name, u_32 uptime, u_32 cpu1, u_32 cpu2, u_32 cpu3,
			u_32 memused, u_32 memfree)
{
	char		buf[1000];
	
	sprintf(buf,"INSERT INTO netdev_data(netdev_name,uptime,cpu1,cpu2,cpu3,memfree,memused) "
			"VALUES ('%s','%u','%u','%u','%u','%u','%u')",
			netdev_name, uptime, cpu1, cpu2, cpu3, memfree, memused);
	db.Exec(buf);
}

void	update_memacct	(const char *netdev_ip, u_32 memused)
{
	char		buf[1000];
	time_t		ts,dbts;
	int		n,ret;
	LONGLINE	ll;
	u_32		omem,ocount,nmem;

	ts=time(NULL);

	
	for (n=1;n<=5;n++)
	{
		dbts=ts/intervals[n];
		dbts*=intervals[n];
		sprintf(buf,"SELECT mem, count FROM %s WHERE "
				"netdev_ip='%s' AND actype='%d' AND ts='%d'",
				TBL_MEMACCT,
				netdev_ip,n,dbts);

		ret=db.Exec(buf);
		if (ret>0)
		{
			ll=db.GetValue(0,0); omem=strtoul(ll.p,NULL,10);
			ll=db.GetValue(0,1); ocount=strtoul(ll.p,NULL,10);
			nmem=(u_32)((1.0*omem*ocount)+memused)/(ocount+1);
			ocount++;

			sprintf(buf,"UPDATE %s SET mem='%u', count='%d' WHERE "
					"netdev_ip='%s' AND actype='%d' AND ts='%d'",
					TBL_MEMACCT, nmem, ocount,
					netdev_ip, n, dbts);
		} else
		{
			sprintf(buf,"INSERT INTO %s(netdev_ip,actype,ts,mem,count) "
					"VALUES('%s','%d','%d','%u','%d')",
					TBL_MEMACCT,
					netdev_ip, n, dbts, memused, 1);
		}
		db.Exec(buf);
	}
}

void	update_cpuacct	(const char *netdev_ip, u_32 cpu)
{
	char		buf[1000];
	time_t		ts,dbts;
	int		n,ret;
	LONGLINE	ll;
	u_32		ocpu,ocount,ncpu;

	ts=time(NULL);

	
	for (n=1;n<=5;n++)
	{
		dbts=ts/intervals[n];
		dbts*=intervals[n];
		sprintf(buf,"SELECT cpu, count FROM %s WHERE "
				"netdev_ip='%s' AND actype='%d' AND ts='%d'",
				TBL_CPUACCT,
				netdev_ip,n,dbts);

		ret=db.Exec(buf);
		if (ret>0)
		{
			ll=db.GetValue(0,0); ocpu=strtoul(ll.p,NULL,10);
			ll=db.GetValue(0,1); ocount=strtoul(ll.p,NULL,10);
			ncpu=(u_32)((1.0*ocpu*ocount)+cpu)/(ocount+1);
			ocount++;

			sprintf(buf,"UPDATE %s SET cpu='%d', count='%d' WHERE "
					"netdev_ip='%s' AND actype='%d' AND ts='%d'",
					TBL_CPUACCT, ncpu, ocount,
					netdev_ip, n, dbts);
		} else
		{
			sprintf(buf,"INSERT INTO %s(netdev_ip,actype,ts,cpu,count) "
					"VALUES('%s','%d','%d','%d','%d')",
					TBL_CPUACCT,
					netdev_ip, n, dbts, cpu, 1);
		}
		db.Exec(buf);
	}
}

void	set_last_run()
{
	char	buf[1000];

	sprintf(buf,"DELETE FROM config WHERE name='last_run'");
	db.Exec(buf);
	sprintf(buf,"INSERT INTO config(name,value) VALUES ('last_run','%d')",
			time(NULL));
	db.Exec(buf);
}

time_t	get_last_maintenance()
{
	char		buf[1000];
	int		ret;
	LONGLINE	ll;
	time_t		t;

	sprintf(buf,"SELECT value FROM config WHERE name='last_maintenance'");
	ret=db.Exec(buf);
	if (ret>0)
	{
		ll=db.GetValue(0,0);
		t=strtoul(ll.p,NULL,10);
	} else t=0;

	return(t);
}

time_t	get_last_vacuumfull()
{
	char		buf[1000];
	int		ret;
	LONGLINE	ll;
	time_t		t;

	sprintf(buf,"SELECT value FROM config WHERE name='last_vacuumfull'");
	ret=db.Exec(buf);
	if (ret>0)
	{
		ll=db.GetValue(0,0);
		t=strtoul(ll.p,NULL,10);
	} else t=0;

	return(t);
}

void	set_last_maintenance()
{
	char	buf[1000];

	sprintf(buf,"DELETE FROM config WHERE name='last_maintenance'");
	db.Exec(buf);
	sprintf(buf,"INSERT INTO CONFIG(name,value) VALUES('last_maintenance','%d')",
			time(NULL));
	db.Exec(buf);
}

void	set_last_vacuumfull()
{
	char	buf[1000];

	sprintf(buf,"DELETE FROM config WHERE name='last_vacuumfull'");
	db.Exec(buf);
	sprintf(buf,"INSERT INTO CONFIG(name,value) VALUES('last_vacuumfull','%d')",
			time(NULL));
	db.Exec(buf);
}

void	do_rotate()
{
	db.Exec("SELECT do_rotate()");
}

void	do_reindex()
{
	db.ReindexTable("config");
	db.ReindexTable("cpuacct");
	db.ReindexTable("datasources");
	db.ReindexTable("ifacct");
	db.ReindexTable("interfaces");
	db.ReindexTable("map_config");
	db.ReindexTable("maps");
	db.ReindexTable("memacct");
	db.ReindexTable("netdev_data");
	db.ReindexTable("netdevs");
	db.ReindexTable("points");
/*	db.Exec("REINDEX TABLE config FORCE");
	db.Exec("REINDEX TABLE cpuacct FORCE");
	db.Exec("REINDEX TABLE datasources FORCE");
	db.Exec("REINDEX TABLE ifacct FORCE");
	db.Exec("REINDEX TABLE interfaces FORCE");
	db.Exec("REINDEX TABLE map_config FORCE");
	db.Exec("REINDEX TABLE maps FORCE");
	db.Exec("REINDEX TABLE memacct FORCE");
	db.Exec("REINDEX TABLE netdev_data FORCE");
	db.Exec("REINDEX TABLE netdevs FORCE");
	db.Exec("REINDEX TABLE points FORCE");*/
}

void	do_final()
{
	time_t	t;
	int	to_vacuumfull=0;
	int	to_maintenance=0;

	db.Begin();
	db.LockTableWrite("config");	// This should be a very short lock, just to avoid dupplicate-invocation problems

	set_last_run();

	// Check if it is time for VACUUM FULL
	t=time(NULL)-get_last_vacuumfull();
	if (t>FULLVACUUM_INTERVAL)
	{
		to_vacuumfull=1;
		set_last_vacuumfull();
	}

	// Check if it is time for maintenance
	t=time(NULL)-get_last_maintenance();
	if (t>MAINTENANCE_INTERVAL)
	{
		to_maintenance=1;
		set_last_maintenance();	// Set this first to avoid dupplicate-invocation problems and release the lock
	}
	db.End();

	if (to_maintenance)
	{
		db.Exec("SELECT do_acct_expire()");
		db.Exec("DELETE FROM interfaces WHERE"
			"(EXTRACT(EPOCH FROM CURRENT_TIMESTAMP)-cur_ts)>432000");	// delete old interfaces
//		db.Exec("DELETE FROM ifacct WHERE"
//			"(EXTRACT(EPOCH FROM CURRENT_TIMESTAMP)-ts)>90000");	// delete entries older than a day
		do_reindex();

		if (!to_vacuumfull)	// Dont do this if VACUUM FULL is going to be done
			db.Exec("VACUUM ANALYZE");
	}

	if (to_vacuumfull)
	{
		db.Exec("VACUUM FULL ANALYZE");
	}
}

void	db_begin_update()
{
	db.Begin();
	db.LockTableWrite("config");
	db.LockTableWrite("netdevs");
	db.LockTableWrite("interfaces");
	db.LockTableWrite("cpuacct");
	db.LockTableWrite("ifacct");
	db.LockTableWrite("memacct");
	db.LockTableWrite("netdev_data");
}

void	db_end_update()
{
	db.End();
}

